/*
**      $VER: rgfx 4.0 (16.08.2012)
**
**      Specs and structure definitions for the IFF-RGFX file format -
**
**      (C) Copyright 1997-2012 Andreas R. Kleinert
**      Freeware. All Rights Reserved.
*/

#ifndef RGFX_H
#define RGFX_H

#define RGFX_VERSION 3

#ifndef EXEC_TYPES_H
#include <exec/types.h>
#endif /* EXEC_TYPES_H */

#ifndef LIBARIES_IFFPARSE_H
#include <libraries/iffparse.h>
#endif /* LIBARIES_IFFPARSE_H */


/* *************************************************** */
/* *                                                 * */
/* * IFF/RGFX specific definitions and declarations  * */
/* *                                                 * */
/* *************************************************** */

#define ID_FORM  MAKE_ID('F','O','R','M')  /* if not already defined */
#define ID_RGFX  MAKE_ID('R','G','F','X')  /* IFF-RGFX               */

/* A typical structure of a RGFX file (recommended) :

     FORM-RGFX

       RGHD
       RSCM
       RCOL
       RBOD
*/


/*******************************************************
   RGHD - RawGfx Bitmap Header (mandatory)

   ** This one replaces BMHD

*/

#define ID_RGHD  MAKE_ID('R','G','H','D')

struct RGHD
{
 ULONG rgfx_LeftEdge;     /* (see BMHD)                               */
 ULONG rgfx_TopEdge;      /* (see BMHD)                               */
 ULONG rgfx_Width;        /* (see BMHD)                               */
 ULONG rgfx_Height;       /* (see BMHD)                               */
 ULONG rgfx_PageWidth;    /* (see BMHD)                               */
 ULONG rgfx_PageHeight;   /* (see BMHD)                               */

 ULONG rgfx_Depth;        /* 1-8         for RMBT_BYTEPLANAR8,
                             1-8         for RMBT_BYTECHUNKY8,
                              15         for RMBT_2BYTERGB15
                              15         for RMBT_2BYTERGB16
                              24         for RMBT_3BYTERGB24
                              24         for RMBT_4BYTERGB32
                              48         for RMBT_6BYTERGB48
                              48         for RMBT_8BYTERGB64
							  96         for RMBT_12BYTERGB96
                              96         for RMBT_16BYTERGB128
						  */
 ULONG rgfx_PixelBits;    /* 1-8         for RMBT_BYTEPLANAR8,
                               8         for RMBT_BYTECHUNKY8,
                              15         for RMBT_2BYTERGB15
                              16         for RMBT_2BYTERGB16
                              24         for RMBT_3BYTERGB24
                              32         for RMBT_4BYTERGB32
                              48         for RMBT_6BYTERGB48
                              64         for RMBT_8BYTERGB64
							  96         for RMBT_12BYTERGB96
                              128        for RMBT_16BYTERGB128
					      */
 ULONG rgfx_BytesPerLine; /* (width+7)/8 for RMBT_BYTEPLANAR8,
                              width      for RMBT_BYTECHUNKY8,
                              width*2    for RMBT_2BYTERGB15
                              width*2    for RMBT_2BYTERGB16
                              width*3    for RMBT_3BYTERGB24
                              width*4    for RMBT_4BYTERGB32
                              width*6    for RMBT_6BYTERGB48
                              width*8    for RMBT_8BYTERGB64
							  width*12   for RMBT_12BYTERGB96
                              width*16   for RMBT_16BYTERGB128
						  */
 ULONG rgfx_Compression;  /* RCMP_ type flag                          */
 ULONG rgfx_XAspect;      /* (see BMHD)                               */
 ULONG rgfx_YAspect;      /* (see BMHD)                               */
 ULONG rgfx_BitMapType;   /* RBMT_ type flag                          */
};

 /* if you encounter unknown depth/pixelbits/bytesperline combinations,
    then do reject these (future expansions).
  */

#define RCMT_NOCOMPRESSION  (0L)
#define RCMT_XPK            (1L)     /* any XPK-packer                           */
#define RCMT_ZIP            (2L)     /* libzip (LZ77/ZIP) compression            */

#define RMBT_BYTEPLANAR8    (0L)     /*  unaligned planar 8 bit bitmap           */
#define RMBT_BYTECHUNKY8    (1<<0L)  /*  unaligned chunky 8 bit bitmap           */
#define RMBT_3BYTERGB24     (1<<1L)  /*  3-byte 24  bit  RGB triples             */
#define RMBT_4BYTERGB32     (1<<2L)  /*  4-byte 32  bit ARGB quadruples          */
#define RMBT_2BYTERGB15     (1<<4L)  /*  2-byte 15  bit  RGB (x+3x5 bit integer) */
#define RMBT_2BYTERGB16     (1<<5L)  /*  2-byte 16  bit ARGB (1+3x5 bit integer) */
#define RMBT_6BYTERGB48     (1<<6L)  /*  6-byte 48  bit  RGB (3x 16 bit integer) */
#define RMBT_8BYTERGB64     (1<<7L)  /*  8-byte 64  bit ARGB (4x 16 bit integer) */
#define RMBT_12BYTERGB96    (1<<8L)  /* 12-byte 96  bit  RGB (3x 32 bit float)   */
#define RMBT_16BYTERGB128   (1<<9L)  /* 16-byte 128 bit ARGB (4x 32 bit float)   */

#define RMBT_ALPHA          (1<<30L) /* set if A is meaningful                   */
#define RMBT_ALPHAINV       (1<<31L) /* set if A is meaningful but               */
                                     /* inversed (A = 255 - alpha)               */

/* Meaning of "A" (alpha information)
   ----------------------------------

   Definition of the alpha word/byte/bit is the same as in PNG:

   0x0000 / 0x00 / 0 / 0.0f means "fully transparent"
   0xFFFF / 0xFF / 1 / 1.0f means "fully opaque"

   Note, that the content of the alpha value will be zero also
   in case RMBT_ALPHA/ALPHAINV is not set - but this should
   not be interpreted as transparency: These flags
   must be checked prior to interpretation of the alpha value.
*/

/* ***************************************************** */


/*******************************************************
   RGXY - RawGfx Aspect Info (optional)

   ** This one replaces BMHD's XAspect/YAspect

*/

#define ID_RGXY  MAKE_ID('R','G','X','Y')

struct RGXY
{
 ULONG  rgfx_XAspect;      /* (see BMHD)                               */
 ULONG  rgfx_YAspect;      /* (see BMHD)                               */
 ULONG  rgfx_HasDouble;    /* does a valid number follow? (TRUE/FALSE) */
 DOUBLE rgfx_XYRatio;      /* := x/y, IEEE754 (mathieedoubbas)         */
  LONG  rgfx_VGAMode;      /* see below                                */
};

/* As VGA mode, you can also specifiy a Linux LILO VGA mode,
   like "788" (800x600, 16 bit color) to simplify the viewmode
   selection. The reason for this is, that AmigaOS RTG
   screenmodes are monitor-dependent and cannot be
   exchanged between different systems - they are only
   valid for local usage (see also RSCM chunk).

   Use the following table for PC/BIOS/Linux modes:

   VGA   | 640x480 800x600 1024x768 1280x1024
   ------+-------------------------------------
   256   |     257     259      261       263
   15bit |     272     275      278       281
   16bit |     273     276      279       282
   24bit |     274     277      280       283

   VESA  | 640x480 800x600 1024x768 1280x1024 1600x1200
  -------+---------------------------------------------
   256   |     769     771      773       775       796
   15bit |     784     787      790       793       797
   16bit |     785     788      791       794       798
   24bit |     786     789      792       795       799

   Note: Take care, that the selected viewmode matches
         RBMT_BitMapType!

   Negative values are reserved or defined as below:
*/

#define RGFX_VGAMODE_DEFAULT        (0L)  /* do not use!     */

#define RGFX_VGAMODE_INVALID        (-1L) /* use, if default */
#define RGFX_VGAMODE_AMIGA_VIEWMODE RGFX_VGAMODE_INVALID

#define RGFX_VGAMODE_PAL       (-2L)      /*  720x576        */
#define RGFX_VGAMODE_NTSC      (-3L)      /*  720x480        */
#define RGFX_VGAMODE_HDTV_720  (-4L)      /* 1280x720        */
#define RGFX_VGAMODE_HDTV_1080 (-5L)      /* 1920x1080       */
#define RGFX_VGAMODE_HDTV_1125 (-6L)      /* 1920x1125       */
#define RGFX_VGAMODE_HDTV_1250 (-7L)      /* 1920x1250       */

/* the TV modes do not assume any depth/pixelbits      */
/* since they are not implemented in this form on      */
/* today's LCD monitors anyway                         */

/*******************************************************/


/*******************************************************
   RSCM - View/ScreenMode (mandatory, processing optional)

   ** This one replaces CAMG.

   The default settings are:

   rscm_ViewMode:  default screenmode
   rscm_LocalVM_0: INVALID_ID           (obsolete local RTG)
   rscm_LovalVM_1: INVALID_ID           (obsolete local RTG)


   Note: Since HAM modes only can be identified by their
         ID (or DIPF) you have to make sure, that
         rcsm_ViewMode is OR'ed with HAM_KEY for these
         (same for EHB and EXTRAHALFBRITE_KEY).

         Specific RTG ViewModes will lose their meaning,
         as soon as graphics are transferred between different
         systems, which is why the two LocalVM entries are
         considered obsolete.

         Always set the obsolete entries to 0xFFFFFFFF
         and avoid interpreting them.
*/

#define ID_RSCM  MAKE_ID('R','S','C','M')

struct RSCM
{
 ULONG rscm_ViewMode;  /* 32 Bit AGA Viewmode ID  */
 ULONG rscm_LocalVM_0; /* only valid on computer  */
 ULONG rscm_LocalVM_1; /* where first saved       */
};
/* ***************************************************** */

/*******************************************************
   RDIP - AmigaOS DIPF information (optional)

   ** This one complements RSCM.

   AmigaOS DIPF information for rscm_ViewMode
   or rscm_LocalVM_0/1 can be put here.
*/

#define ID_RDIP  MAKE_ID('R','D','I','P')

struct RDIP
{
 ULONG rscm_NativeDIPF; /* 32 Bit DIPF information */
};
/* ***************************************************** */


/*******************************************************
   RCOL - RawGfx Colormap (optional for > 8 bit)

   ** This one replaces CMAP.

      Required with RMBT_BYTEPLANAR8 and RMBT_BYTECHUNKY8,
      and optionally allowed for the other modes as a
      recommended dithering destination colormap.

      Stored are 256 byte triples in RGB format. Note, that
      full-range values (0..255) have to be stored.

      Sample:   rcol_Colors[0][0]   = 0..255 red   value, color #0 (1st)
                rcol_Colors[0][1]   = 0..255 green value, color #0
                rcol_Colors[0][2]   = 0..255 blue  value, color #0
                ...
                rcol_Colors[255][0] = 0..255 red   value, color #255 (256th)
                rcol_Colors[255][1] = 0..255 green value, color #255
                rcol_Colors[255][2] = 0..255 blue  value, color #255


      Unused entries must be filled with zeroes.


   Allowed boolean values for "rcol_TransColor":  TRUE (1L) and FALSE (0L)

   Note: With EHB mode, which e.g. uses 64 colors with only a 32 color
         palette, you always should store the full color range within
         the RCOL structure: if someone is going to display EHB pictures
         on an RTG screen, he won't have to calculate the 'missing'
         colors by himself.

  Note:  In case dithering of 16/32 bit graphics is done, the alpha
         information gets lost! In this case, the transparent color
         from below can be used. For example, if the only transparent
         color was BLACK (0,0,0) the effect surely will be the same
         if you put the palette index of BLACK as rcol_TransColorNum.
*/

#define ID_RCOL  MAKE_ID('R','C','O','L')

struct RCOL
{
 ULONG rcol_TransColor;      /* boolean: is there a transparent color ?  */
 ULONG rcol_TransColorNum;   /* if yes: pick one from below (index)      */
 UBYTE rcol_Colors[256][3];
};
/* ***************************************************** */

/*******************************************************
   RFLG - Additional flags (optional)

   ** This one complements RRGH, RCOL and RBOD..

   Holds information which allows lossless back
   transformations to previous formats or similar things.  
*/

#define ID_RFLG  MAKE_ID('R','F','L','G')

struct RFLG
{
 ULONG rflg_Flags;     /* as defined below */
 ULONG rflg_MoreFlags; /* reserved; 0L yet */
};

#define RFLG_RESERVED_01	(1<<0)  /* never set                         */
#define RFLG_RESERVED_02	(1<<1)  /* never set                         */
#define RFLG_RESERVED_03	(1<<2)  /* never set                         */
/* The following is kind of a meta information flag  and may help to     */
/* either do a lossless "to gray" conversion or to apply grayscale ICC   */
/* profiles to something that looks like a colored graphics at first.    */
/* As RGFX does not know grayscale graphics as such, this flag serves    */
/* that purpose.                                                         */
#define RFLG_FROM_GRAY_SRC  (1<<3)  /* created from 8/16 bit gray source */
                                    /* so R==G==B                        */
/* The following is kind of a meta information flag  and may help to     */
/* do a lossless "to 8 bit" back conversion.                             */
#define RFLG_FROM_8BIT_SRC  (1<<4)  /* created from 8 bit source,        */
							 	    /* so (R,G,B)&0xFF00 == ... & 0x00FF */
/* The following is kind of a meta information flag  and may help to     */
/* do a lossless "to 4 bit" back conversion.                             */
#define RFLG_FROM_4BIT_SRC  (1<<5)  /* created from 4 bit source,        */
							 	    /* so (R,G,B)&0xF0 == ... & 0x0F     */
/* The following is kind of a meta information flag  and may help to     */
/* do a lossless "to 8 bit" back conversion.                             */ 
#define RFLG_8BIT_ALPHA_SRC (1<<6)  /* 16/32 bit alpha created from      */
                                    /* 8 bit alpha source                */
/* The following is kind of a meta information flag  and may help to     */
/* do a lossless "to 16 bit" back conversion.                            */ 
#define RFLG_16BIT_ALPHA_SRC (1<<7) /* 32 bit alpha created from         */
                                    /* 16 bit alpha source               */

/* ***************************************************** */

/*******************************************************
   RTRN - Multi-Color Transparency (optional)

   ** This one complements RCOL.

      Only to be used in case there is MORE THAN ONE
      of the colors from RCOL to be made transparent
      and if finegrained transparency levels are
      required.

      In this case, put one of the fully transparent
      colors to RCOL and set the rcol_TransColor flag.

      Within RTRN again indicate for each color
      the exact level of transparency.

      Stored are 256 bytes (one for each color from
      the palette). The exact meaning of each
      entry is defined by the rtrn_Flags - same
      options as with the RBMT_ alpha flags.
*/

#define ID_RTRN  MAKE_ID('R','T','R','N')

struct RTRN
{
 ULONG rtrn_TransFlag;       /* RTRN_ALPHA / ...INV ?    */
 UBYTE rtrn_Alpha[256];      /* 0x00 .. 0xFF             */
};

#define RTRN_ALPHA     (0L) /* default                   */
#define RTRN_ALPHAINV  (1L) /* inverse (255-alpha)       */

/* Note: more flags may be defined in the future -
         check for equality                              */

/* ***************************************************** */



/*******************************************************
   RBOD - RawGfx Bitmap Body (mandatory)

   ** This one replaces BODY.
*/

#define ID_RBOD  MAKE_ID('R','B','O','D')

/* May look like:

struct RBOD /* ZIP compression */
{
 ULONG rbod_UCSize;     ** uncompressed size
 UBYTE rbod_BitMap[];
};

struct RBOD /* XPK compression */
{
 UBYTE rbod_XPK[3];     ** containing 'XPK'
 UBYTE rbod_BitMap[];
};

   Or simply:

struct RBOD /* no compression */
{
 UBYTE rbod_BitMap[];
};

 This has not been defined as a union structure here, since some
 compilers might add unwanted pad bytes to the structure.

 Therfore in your code you should reference its chunk
 data as UBYTE array, only.

 ******************************************************* */


/*******************************************************
   RGFX - Other chunks

   - NAME, AUTH, ANNO and (C) chunks are allowed
   - as well as EXIF, IPTC, XMP0, XMP1

******************************************************** */

#endif /* RGFX_H */
