/* Simple example and test program to demonstrate use of template Listview
   class */

#define MUIPP_DEBUG         // Turn debugging mode on for invalid use of classes
//#define MUIPP_NOINLINES       // No inlines makes code compile quicker but the resulting
                            // executable is larger and slower. Best to use this
                            // option when developing and turn off for final release.
#define MUIPP_TEMPLATES     // Allows use of MUI template classes

// This is the main C++ header file

#include <cstdlib>

#include <libraries/mui_mcc.hpp>
#include <proto/exec.h>

#define DEBUG 1
#include <aros/debug.h>

#include "class/Person.h"


// Hook function to display Person object in listview

static void hookFunctionDisplayPerson(struct Hook *hook, char **column, Person *person) {
    column[0] = person->name;
}

static APTR hookFunctionConstructPerson(struct Hook *hook, APTR pool, Person *person) {
    Person *newPerson = new Person(person->name);
    return newPerson;
}

static void hookFunctionDestructPerson(struct Hook *hook, APTR pool, Person *person) {
    if (person) {
        delete person;
    }
}

static struct Hook hookDisplay = {{NULL, NULL}, NULL, NULL, NULL};
static struct Hook hookConstruk = {{NULL, NULL}, NULL, NULL, NULL};
static struct Hook hookDestruct = {{NULL, NULL}, NULL, NULL, NULL};

// Although the listview is passed by value it is actually passed by reference
// always. This is because the class is only a wrapper to the BOOPSI class
// and only has one attribute - the BOOPSI object pointer. Hence, it is the
// equivalent of passing an Object * on the stack.

void PrintPersonList(CTMUI_Listview<Person> list) {
    // You can use Length() or Entries() to get the length of a list

    int numPeople = list.Length();

    printf("Number of people in list = %d\n", numPeople);

    // You can treat Listviews just like arrays!!

    for (int i = 0; i < numPeople; i++) {
        printf("%d %s\n", i, list[i].name);
    }
}

int main (void) {
    hookDisplay.h_Entry = (APTR)HookEntry;
    hookDisplay.h_SubEntry = (APTR)hookFunctionDisplayPerson;
    hookConstruk.h_Entry = (APTR)(APTR)HookEntry;
    hookConstruk.h_SubEntry = (APTR)hookFunctionConstructPerson;
    hookDestruct.h_Entry = (APTR)HookEntry;
    hookDestruct.h_Entry = (APTR)hookFunctionDestructPerson;

    // Declare template Listview of type Person to display list of people
    // NOTE: This does no initialization, it's just a declaration.

    CTMUI_Listview<Person> list;

    CMUI_Window window;

    // Create Application object. I am not using any shortcuts here to create
    // the objects. I actually prefer the layout like this than when using
    // shortcuts. If you prefer the old way of creating objects by using the
    // shortcuts then you can still do this. See the shortcuts.cpp example
    // for details as some shortcuts have had to change name so as not to clash
    // with class member functions.

    D(bug("Create app"));
    
    CMUI_Application app
    (
        MUIA_Application_Title,         "TListview",
        MUIA_Application_Author,        "Nicholas Allen, Piotr Hoppe",
        MUIA_Application_Base,          "TEST",
        MUIA_Application_Copyright,     "AllenSoft",
        MUIA_Application_Description,   "Test Program For Template Listview class",
        MUIA_Application_Version,       "$VER: Test 1.0 (15.12.2015)",
        SubWindow, window = CMUI_Window
        (
            MUIA_Window_Title, "Test Program For Template Listview class",
            MUIA_Window_ID, 10,
            WindowContents, CMUI_VGroup
            (
                Child, list = CTMUI_Listview<Person>
                (
                    MUIA_Listview_List, CMUI_List
                    (
                        MUIA_List_ConstructHook, (APTR)&hookConstruk,
                        MUIA_List_DisplayHook, (APTR)&hookDisplay,
                        MUIA_List_DestructHook, (APTR)&hookDestruct,
                        InputListFrame,
                        TAG_DONE
                    ),
                    MUIA_CycleChain, 1,
                    MUIA_ShortHelp, "Listview created using templates!!",
                    TAG_DONE
                ),
                TAG_DONE
            ),
            TAG_DONE
        ),
        TAG_DONE        // Don't forget these if you're not using shortcuts!
    );

    D(bug(" -> DONE\n"));
    
    // Any MUI object created as a C++ class can be tested for validity by
    // calling its IsValid() method. This method just checks that the
    // BOOPSI object pointer is not NULL.

    if (!app.IsValid()) {
        printf("Could not create application!\n");
        return 10;
    }

    D(bug("Insert some new people into the listview"));

    list.InsertBottom(new Person ("Nick"));
    list.InsertBottom(new Person ("Dom"));
    list.InsertBottom(new Person ("Mart"));
    list.InsertBottom(new Person ("Nicky"));

    D(bug(" -> DONE\n"));
    
    // This only copies 4 bytes onto stack!! It is the same as passing a
    // BOOPSI Object *

    PrintPersonList (list);

    // Setup close window notification.
    // Because Notify() is a variable args method we have to pass sva as the
    // first parameter. Failing to do this will result in an error at
    // COMPILE time so there won't be any weird crashes by forgetting to do
    // this.

    D(bug("Open window"));
    
    window.Notify(sva, MUIA_Window_CloseRequest, TRUE,
                  *app, 2, MUIM_Application_ReturnID, MUIV_Application_ReturnID_Quit);

    window.SetOpen(TRUE);

    D(bug(" -> DONE\n"));

    ULONG sigs = 0;
    BOOL running = TRUE;

    while (running) {
        switch (app.NewInput(&sigs)) {
            case MUIV_Application_ReturnID_Quit:
                running = FALSE;
                break;
        }

        if (sigs) {
            sigs = Wait(sigs | SIGBREAKF_CTRL_C);
            if (sigs & SIGBREAKF_CTRL_C) {
                break;
            }
        }
    }

    // This disposes of the application and all windows and objects in the
    // windows.

    app.Dispose();

    return 0;
}
