#!/usr/bin/python
# -*- coding: UTF-8 -*-

#    PyAGLaunch Guide Creator (c) 2016 Žarko Živanov

#    PyAGLaunch is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function

import os,re,time,binascii

VERSION = "1.1"



#########################################################################
#                 PyAGLauncher Guide Creator SETTINGS
#                - change them to suit your needs... -
#########################################################################

# You'll need WBRun package from aminet: http://aminet.net/package/util/cli/WBRun_fix

# PC game/demo directory paths relative to script
#    Demos  - path to Demos directory
#    Games  - path to Games
#    AGA    - path to AGA Games directory
#    Guides - path to directory for guide generation
Demos  = "Demos_WHDLoad"
Games  = "Games_WHDLoad"
AGA    = "Games_WHDLoad_AGA"
Guides = "Guides"

# Amiga game/demo paths and names
#    DemosDrivePath - path to Demos directory on the Amiga
#    GamesDrivePath - path to Games directory on the Amiga
#    AGADrivePath   - path to AGA Games directory on the Amiga
#    LauncherPath   - path to guides directory on the Amiga
#    WBRunPath      - path to wbrun command on the Amiga
#    MainLauncher   - name for main guide launcher
#    LauncherPrefix - prefix for launcher files (may be empty)
DemosDrivePath = "Games:"
GamesDrivePath = "Games:"
AGADrivePath   = "Games:"
LauncherPath   = "agl:guides"
WBRunPath      = "c:wbrun"
MainLauncher   = "AG-Launcher"
LauncherPrefix = "AGL-"

# guide generation preferences
#    PreferredLanguages - space delimited list of languages for games, in order of preference
#                         if no version exists for preffered language, existing language version
#                         will be included
#                         examples: "En"  "En De" "Fr It En", or "" - all languages
#    StrictLanguage     - do not include other languages, even if game exists only in other language
#    TranslateToEnglish - should non-english games with english title be translated to english
#    PreferredTVSystem  - one of: "" "PAL" "NTSC"
#    PreferredRelease   - one of: "" "Release" - regular game, "Demo" - Demo/Preview/Prerelease version
#    PreferredMemType   - one of: "" "Fast" "Slow"
#    PreferredFileType  - one of: "" "Files" "Image"
#    PreferredDiskNum   - NOT CURRENTLY SUPPORTED! examples: "1D" "3D 2D 1D"
#    PreferredMemSize   - NOT CURRENTLY SUPPORTED! examples: "512k" "1M 512k"
#    ShowOnlyCD32       - if true, do not add AGA suffix in AGA game list, only CD32 if needed
#    OnlyFirstPrefedred - if multiple versions of game exist, list only first that matches preferences
PreferredLanguages = "En"
StrictLanguage     = False
TranslateToEnglish = True
PreferredTVSystem  = "PAL"
PreferredRelease   = "Release"
PreferredMemType   = ""
PreferredFileType  = ""
PreferredDiskNum   = ""
PreferredMemSize   = ""
ShowOnlyCD32       = True
OnlyFirstPrefedred = False

# list layout options
#    GameColumns     - number of columns in game list
#    ColumnPadding   - minimal number of spaces between columns in game list
#    RowsPadding     - number of empty lines between rows in game list
#    LetterColumns   - number of columns in main (letter) list
#    LetterPadding   - minimal number of spaces between columns in main (letter) list
#    LetterRPadding  - number of empty lines between rows in main (letter) list
#    WindowCharWidth - width of guide window in chars; 76 is for standard Amiga resolution
#                      if you have an Amiga with higher supported resolution, you may have
#                      a bigger launcher window with more columns
GameColumns     = 3
ColumnPadding   = 1
RowsPadding     = 0
LetterColumns   = 19
LetterPadding   = 2
LetterRPadding  = 0
WindowCharWidth = 76

# custom name expansion
#    - for certain titles where automatic expansion doesn't work good enough
#    - first element should be game info file name
#    - second element should be how you want to see it in list (automatic shortening
#      will be applied as needed, so you don't need to worry about length)
ExpandAs = {
    "1000ccTurbo" : "1000cc Turbo",
    "3001OConnorsFight" : "3001 O'Connor's Fight",
    "4x4OffRoadRacing" : "4x4 OffRoad Racing",
    "AotGSAFP27b6" : "Attack of the Green Smelly Aliens",
    "AV8BHarrierAssault" : "AV8B Harrier Assault",
    "BlackGoldreLINE" : "Black Gold reLINE",
    "ChaosStrikesBack&DeUtil" : "Chaos Strikes Back Util De",   # wrong naming?
    "ChaosStrikesBack&EnUtil" : "Chaos Strikes Back Util",      # wrong naming?
    "ChaosStrikesBack&FrUtil" : "Chaos Strikes Back Util Fr",   # wrong naming?
    "CJsElephantAntics" : "CJ's Elephant Antics",
    "DrakkhenFilesDe" : "Drakkhen De Files",    # wrong naming?
    "DrakkhenFilesFr" : "Drakkhen Fr Files",    # wrong naming?
    "DrakkhenImageDe" : "Drakkhen De Image",    # wrong naming?
    "DrakkhenImageFr" : "Drakkhen Fr Image",    # wrong naming?
    "ETsRugbyLeague" : "ET's Rugby League",
    "EnemyDe(EasyPlay)" : "Enemy (EasyPlay) De",        # wrong naming?
    "EnemyTOVEasyPlay" : "Enemy TOV (EasyPlay)",        # wrong naming?
    "EnemyTOVEasyPlayDe" : "Enemy TOV (EasyPlay) De",   # wrong naming?
    "GrandPrix5002Fr" : "Grand Prix 500 2 Fr",
    "KickOff2FWDe" : "Kick Off 2 Final Whistle De",     # wrong naming?
    "Kingpin1Mb" : "King Pin 1Mb",                      # wrong naming?
    "KnightsOfCrystallionDe" : "Knights Of The Crystallion De",     # wrong naming?
    "LeadingLapMPVCUAmiga" : "Leading Lap MPV CU Amiga",
    "LeonardoDOS" : "Leonardo (DOS)",
    "LeonardoHighDensity" : "Leonardo (HighDensity)",
    "LoomDeKick13" : "Loom Kick 13 De",     # wrong naming?
    "LoomDeKick31" : "Loom Kick 31 De",     # wrong naming?
    "LoomEsKick13" : "Loom Kick 13 Es",     # wrong naming?
    "LoomEsKick31" : "Loom Kick 31 Es",     # wrong naming?
    "LoomFrKick13" : "Loom Kick 13 Fr",     # wrong naming?
    "LoomFrKick31" : "Loom Kick 31 Fr",     # wrong naming?
    "MiG29Fulcrum" : "MiG 29 Fulcrum",
    "MiG29MSuperFulcrum" : "MiG 29M Super Fulcrum",
    "Mig29SovietFighter" : "MiG 29 Soviet Fighter",     # wrong naming?
    "Nam19651975" : "Nam 1965 1975",
    "NewYearLemmings199192" : "New Year Lemmings 91 92",    # wrong naming?
    "OsWALDOfTheIceFloes" : "OsWALD Of The Ice Floes",
    "OverlordDemoRowan" : "Overlord Rowan Demo",            # wrong naming?
    "PeterBeardsleysIFCracked" : "Peter Beardsleys IF (Cracked)",
    "ProjectXSE" : "Project X SE",
    "ProjectXSEBTTR" : "Project X SE BTTR",
    "ProjectXv2" : "Project X v2",
    "QuestForTheTimeBird" : "Quest For Time Bird",                  # wrong naming - does it have The?
    "RickDangerous25" : "Rick Dangerous 2.5",
    "RoadBlastersCracked" : "Road Blasters (Cracked)",
    "SensibleSoccer9293" : "Sensible Soccer 92 93",
    "SensibleWorldOfSoccer9596" : "Sensible World Of Soccer 95 96",
    "SensibleWorldOfSoccer9697" : "Sensible World Of Soccer 96 97",
    "SimCity12" : "Sim City 1.2",
    "SuperOsWALDDk" : "Super OsWALD Dk",
    "SuperfrogDemoAmigaAction" : "Superfrog Amiga Action Demo", # wrong naming?
    "SuperfrogDemoCUAmiga" : "Superfrog CU Amiga Demo",         # wrong naming?
    "SuperfrogDemoTheOne" : "Superfrog The One Demo",           # wrong naming?
    "T-Zer0Demo" : "T-Zer0 Demo",
    "TipOffDeEsFrIt" : "Tip Off (MultiLang)",
    "Virocop2Meg" : "Virocop 2M",           # wrong naming?
    "WalkerCrunched" : "Walker",            # wrong naming (original is crunched)?
    "WalkerCrunchedNoSpeech" : "Walker No Speech",
    "WITUSAIsCarmenSandiego" : "WI USA Is Carmen Sandiego",
    "WITWIsCarmenSandiego" : "WI TW Is Carmen Sandiego",
    "XIt" : "X-It",
    "Xenon2" : "Xenon 2 CDTV",      # wrong naming?
    "XmasLemmingsDemo1991" : "Xmas Lemmings 1991 Demo",     # wrong naming?
    "XmasLemmingsDemo1992" : "Xmas Lemmings 1992 Demo",     # wrong naming?
    "ACSYSDemoAGA" : "AC SYS Demo AGA",
    "AlienBreed3DDemoLatest" : "Alien Breed 3D Latest Demo AGA",     # wrong naming?
    "AlienBreed3DDemoPlayAGA" : "Alien Breed 3D Play Demo AGA",         # wrong naming?
    "AlienBreed3DDemoRollAGA" : "Alien Breed 3D Roll Demo AGA",         # wrong naming?
    "T-Zer0DemoAGA" : "T-Zer0 Demo AGA",
    "T-TZer0DemoNoMovieAGA" : "T-Zer0 No Movie Demo AGA",
    "VirtualKarting2AGACD" : "Virtual Karting 2 CD AGA",
        "LISTEND" : "LISTEND"
}

# translations
#    - just for grouping same game under the same name
Translations = {
    "AigleDOrFr" : "Golden Eagle Fr",
    "CartonRougeFr" : "Anstoss Fr",
    "CroisiereCadavreFr" : "Cruise For A Corpse Fr",
    "DefenseurCouronne2CD32Fr" : "Defender Of The Crown 2 CD32 Fr",
    "DenverJeFaisDesCouleursFr" : "Denver Je Decouvre Les Couleurs Fr",
    "ManoirDeMortvielleFr" : "Mortville Manor Fr",
    "MickeyJeuDeMemoireFr" : "Mickeys Memory Challenge Fr",
    "MickeyPuzzlesAnimesFr" : "Mickeys Jigsaw Puzzles Fr",
    "NilDieuVivantFr" : "Day Of The Pharaoh Fr",
    "PrinceDePerseFr" : "Prince Of Persia Fr",
    "QueteLOiseauDuTempFrFiles" : "Quest For Time Bird Fr Files",
    "QueteLOiseauDuTempFrImage" : "Quest For Time Bird Fr Image",
    "VoyageursDuTempsFr" : "Future Wars Fr",

    "DefensorDeLaCorona2CD32Es" : "Defender Of The Crown 2 CD32 Es",
    "LaAventuraEspacialEs" : "Aventura Espacial Es",
    "LaAventuraOriginalEs" : "Aventura Original Es",
    "LaDiosaDeCozumelEs" : "Diosa De Cozumel Es",
    "LosTemplosSagradosEs" : "Templos Sagrados Es",

    "ClouDe" : "Clue De",                       # wrong naming?
    "ClouDe&Profidisk" : "Clue ProfiDisk De",   # wrong naming?
    "FluchtAusColditzDe" : "Escape From Colditz De",
    "FussballTotalDe" : "Football Glory De",
    "LandsitzVonMortvilleDe" : "Mortville Manor De",
    "PatrizierDe" : "Patrician De",
    "RuesselsheimDe" : "Detroit De",
    "SchwarzeAugeDe" : "Realms Of Arkania De",
    "SiedlerDe" : "Settlers De",
    "ChristophKolumbusAGADe" : "Voyages Of Discovery AGA De",
    "ClouDeAGA" : "Clue AGA De",                            # wrong naming?
    "ClouDeAGA&Profidisk" : "Clue & ProfiDisk AGA De",      # wrong naming?
    "RuesselsheimDeAGA" : "Detroit AGA De",                 # wrong naming?
    "SeelenturmAGADe" : "Tower Of Souls AGA De",
    "VerteidigerKrone2CD32De" : "Defender Of The Crown 2 CD32 De",
    "LISTEND" : "LISTEND"
}



#########################################################################
#                PyAGLauncher Guide Creator variables
#########################################################################

# language suffixes
Languages = "En De Fr It Se Pl Cz Cs Es Dk Fi Gr"
LangList = Languages.split()

# headers and footers
Title = 'WHDLoad Amiga Guide Launcher'
Database = 'WHDLoad HyperText Game Launcher'
ListHeader = [
['@database %s' % Database,False],
['@node Main "%s"' % Title,False],
['%s: #03# directory [ #01# ]' % Title,True],
["",False],
['Click title to play!',True],
["",False]
]

ListFooter = [
["",False],
['Name suffixes: /1D,2D-1 Disk,2 Disk   /F-Fast   /N-NTSC',True],
['/Prv-Demo/Preview   /Fls,Img-Files,Image   /De,Fr,It,...-Language',True],
["",False],
['PyAGLaunch Guide Creator %s (c) 2016 Zarko Zivanov' % VERSION,False],
['This guide is compiled on #02#',False],
['@TOC %s/%s.guide/main' % (LauncherPath,MainLauncher),False],
['@endnode',False]
]

MainHeader = [
['@database %s' % Database,False],
['@WORDWRAP',False],
['@node Main "%s"' % Title,False],
['%s made with PyAGLaunch %s' % (Title,VERSION),True],
["",False],
["",False]
]

MainFooter = [
["",False],
['Many thanks to all the people involved in WHDLoad development!',True],
['Also thanks to Bloodwych and Zetr0 for Amiga Guide Launcher file layout',True],
["",False],
['Guides created using PyAGLaunch Guide Creator %s (c) 2016 Zarko Zivanov' % VERSION,False],
['This guide is compiled on #02#',False],
['@endnode',False]
]

# regular expressions for name splitting
rslave = re.compile(r"slave", flags=re.IGNORECASE)
rspl1 = re.compile(r"([a-z])([A-Z])")
rspl2 = re.compile(r"([0-9])([a-zA-Z])")
rspl3 = re.compile(r"([a-zA-Z])([0-9(])")
rspl4 = re.compile(r"([a-zA-Z0-9+])&([a-zA-Z0-9])")
rspl5 = re.compile(r"([A-Z]+)([A-Z][a-z]+)")
rspl6 = re.compile(r"(AGA)(NTSC)")
rdisk1 = re.compile(r"([0-9]) Disk$", flags=re.IGNORECASE)
rdisk2 = re.compile(r"Disk([0-9])$", flags=re.IGNORECASE)
rjoin1 = re.compile(r"([0-9]) (st\b|nd\b|rd\b|th\b|d \b)", flags=re.IGNORECASE)
rjoin2 = re.compile(r"(\b[A-Z]) ([0-9])", flags=re.IGNORECASE)
rjoin3 = re.compile(r"([0-9]) ([A-Z]\b)", flags=re.IGNORECASE)
rjoin4 = re.compile(r"([a-z]) (s\b)")
rjoin5 = re.compile(r"(2|3)(1|2) (M\b|MB\b)", flags=re.IGNORECASE)
rfiles = re.compile(r"(Files|Image)$", flags=re.IGNORECASE)
rmem = re.compile(r"([0-9]{1,3}) ?(k\b|kb|m\b|mb)", flags=re.IGNORECASE)
rntsc = re.compile(r"\bNTSC\b", flags=re.IGNORECASE)
rfast = re.compile(r"(Fast|Slow|Chip)$", flags=re.IGNORECASE)
raga = re.compile(r"(\baga\b|\bcd 32\b|\bcd 32 aga\b|\bagacd 32\b)", flags=re.IGNORECASE)
rdemo = re.compile(r'(Demo|Demos|Preview|Prv|Prerelease)$')

# preferred options (defined later)
plang = []
pntsc = []
pfast = []
pfile = []

CurrentTime = time.strftime("%d.%m.%Y, %H:%M")

InfoFileData = """
e31000010000000000000000003000140004000100010024e1800000000000000000000000000000
0000000000000001040000217b880000000080000000800000000000000000000000000010000000
0000003000140002000126c803000000000000000000000100000000000300000000000300000000
00030003800000030007c0000003000ee0000003001c7000000300383800000300701c00000300ff
fe01ffc3007ffe00000300000001f8030000000000030000007ffff30000000000030000007fff03
0000000000030000000000037ffffffffffffffffffffffed55555555554dffffffd5554d0000005
5554d00300055554d00780055554d00cc0055554d0186005fffcd0303005fffcd0601805fffcd0ff
fc04003cd0000005fffcd000000407fcdffffffdfffcd5555780000cd55557fffffcd555578000fc
d55557fffffcd55557fffffc8000000000000000000a6d756c74697669657700"""



#########################################################################
#                 PyAGLauncher Guide Creator CODE
#########################################################################

# shortens name to given number of chars by shortening the words
def shortenName(name,length):
    if len(name) <= length: return name
    debug = False #name[:13] == "Chaos Strikes"
    if "-" in name:
        i = name.index("-")
        name1 = name[:i]
        name2 = name[i:]
    else:
        name1 = name
        name2 = ""
    if debug: print("%s [%s][%s] %d" % (name,name1,name2,length))
    rsplit = re.compile(r"(\b[A-Z][a-z]{3,})")
    splitname1 = rsplit.split(name1)
    splitname2 = rsplit.split(name2)
    j = 1
    newname = name
    newname2 = name2
    for i in reversed(range(1,len(splitname1),2)):
        unshorten = splitname1[i]
        splitname1[i] = splitname1[i][0] + "."
        if splitname1[i+1] != "" and splitname1[i+1][0] == " ":
            splitname1[i+1] = splitname1[i+1][1:]
        newname1 = "".join(splitname1)
        newname = newname1 + newname2
        newlen = len(newname)
        if debug: print("1 [%s][%s]" % (newname1,newname2))
        if newlen == length: break
        elif newlen < length:
            splitname1[i] = unshorten[:length-newlen+1] + "."
            newname1 = "".join(splitname1)
            newname = newname1 + newname2
            if debug: print("2 [%s][%s]" % (newname1,newname2))
            break
        if j < len(splitname2):
            unshorten = splitname2[j]
            splitname2[j] = splitname2[j][0] + "."
            if splitname2[j+1] != "" and splitname2[j+1][0] == " ":
                splitname2[j+1] = splitname2[j+1][1:]
            newname2 = "".join(splitname2)
            newname = newname1 + newname2
            newlen = len(newname)
            if debug: print("3 [%s][%s]" % (newname1,newname2))
            if newlen == length: break
            elif newlen < length:
                if debug: print("4 [%s]" % (unshorten), newlen, splitname2[j])
                splitname2[j] = unshorten[:length-newlen+1] + "."
                if debug: print("4 [%s]" % (unshorten), splitname2[j])
                newname2 = "".join(splitname2)
                newname = newname1 + newname2
                if debug: print("5 [%s][%s]" % (newname1,newname2))
                break
        j += 2
    if debug: print("[%s][%s]" % (newname1,newname2))
    return newname

# expands condensed game/demo name, extracts various information
def expandName(name, isAGA=False, isGame=True):
    lang = ""
    ntsc = False
    mem = ""
    disk = ""
    fast = False
    aga = ""
    files = False
    demo = False
    #debug = name[:10] == "Xenon2"
    debug = False
    if name in ExpandAs:
        name = ExpandAs[name]
        if raga.search(name):
            aga = raga.search(name).group(1)
            name = raga.sub(r"",name)
            name = name.strip()
            name = ' '.join(name.split())
    else:
        name = rspl1.sub(r"\1 \2",name)
        name = rspl2.sub(r"\1 \2",name)
        name = rspl3.sub(r"\1 \2",name)
        name = rspl4.sub(r"\1 & \2",name)
        name = rspl5.sub(r"\1 \2",name)
        name = rspl6.sub(r"\1 \2",name)
        if debug: print(name)
        if raga.search(name):
            aga = raga.search(name).group(1)
            name = raga.sub(r"",name)
            name = name.strip()
            name = ' '.join(name.split())
        name = rjoin1.sub(r"\1\2",name)
        if debug: print(name)
        name = rjoin2.sub(r"\1\2",name)
        if debug: print(name)
        name = rjoin3.sub(r"\1\2",name)
        if debug: print(name)
        name = rjoin4.sub(r"\1'\2",name)
        if debug: print(name)
        name = rjoin5.sub(r"\1 \2\3",name)
        if debug: print(name)
    if rntsc.search(name):
        name = rntsc.sub(r"",name)
        ntsc = True
        name = name.strip()
        name = ' '.join(name.split())
    if rfiles.search(name):
        files = rfiles.search(name).group(1)
        files = files == "Files"
        name = rfiles.sub(r"",name)
        name = name.strip()
        name = ' '.join(name.split())
    if debug: print(name)
    if rdisk1.search(name):
        disk = rdisk1.search(name).group(1)
        name = rdisk1.sub(r"",name)
        name = name.strip()
        name = ' '.join(name.split())
    if debug: print(name)
    if rmem.search(name):
        mem = ""
        for m in rmem.findall(name):
            m2 = "".join(m)
            m2 = m2.upper()
            if m2[-1:] == "B": m2 = m2[:-1]
            if mem == "":
                mem = m2
            else:
                mem += "/" + m2
        name = rmem.sub(r"", name)
        name = name.strip()
        name = ' '.join(name.split())
    if rdemo.search(name) and isGame:
        name = rdemo.sub(r"",name)
        demo = True
        name = name.strip()
        name = ' '.join(name.split())
    if debug: print(name)
    # Language and "Fast/Slow" are sometimes in reverese order, check twice
    if rfast.search(name):
        fast = rfast.search(name).group(1)
        name = rfast.sub(r"",name)
        fast = fast == "Fast"
        name = name.strip()
        name = ' '.join(name.split())
    if debug: print(name)
    if (name[-2:] in LangList) and (name[-3:-2] == " ") and isGame:
        lang = name[-2:]
        name = name[:-2]
        name = name.strip()
        name = ' '.join(name.split())
    if rfast.search(name):
        fast = rfast.search(name).group(1)
        name = rfast.sub(r"",name)
        fast = fast == "Fast"
    if debug: print(name)
    if (name[-2:] in LangList) and (name[-3:-2] == " ") and isGame:
        lang = name[-2:]
        name = name[:-2]
    name = name.strip()
    name = ' '.join(name.split())
    if "32" in aga: aga = "CD32"
    if isAGA and (aga == ""): aga = "AGA"
    return name.strip(),lang,ntsc,mem,disk,fast,aga,files,demo

# scans directory for info launchers and creates a dictionary
def scanDir(directory, isAGA=False, isGame=True):
    infodict = {}
    for root, dirs, files in os.walk(directory):
        moreinfo = 0
        for name in files:
            if name[-4:] == "info":
                iname = os.path.join(root,name)
                info = open(iname,"r").read()
                if rslave.search(info):
                    moreinfo += 1
        for name in files:
            if name[-4:] == "info":
                iname = os.path.join(root,name)
                info = open(iname,"r").read()
                if rslave.search(info):
                    rname1 = name[:-5]
                    rname2 = ""
                    lang2 = ""
                    ntsc2 = False
                    mem2 = ""
                    disk2 = ""
                    fast2 = False
                    aga2 = ""
                    files2 = ""
                    demo2 = False
                    if moreinfo > 1:
                        rname2,lang2,ntsc2,mem2,disk2,fast2,aga2,files2,demo2 = expandName(os.path.basename(root),isAGA,isGame)
                        rname2 += "-"
                    rname1,lang1,ntsc1,mem1,disk1,fast1,aga1,files1,demo1 = expandName(rname1,isAGA,isGame)
                    if (lang2 != "") and (lang1 == ""): lang1 = lang2
                    if lang1 == "": lang1 = "En" # assume english if not specified
                    if (mem2 != "") and (mem1 == ""): mem1 = mem2
                    if (disk2 != "") and (disk1 == ""): disk1 = disk2
                    if (aga2 != "") and (aga1 == ""): aga1 = aga2
                    if (files2 != "") and (files1 == ""): files1 = files2
                    ntsc1 = ntsc1 or ntsc2
                    fast1 = fast1 or fast2
                    demo1 = demo1 or demo2
                    rname = rname2 + rname1
                    if rname in infodict:
                        infodict[rname].append((root,name[:-5],mem1,disk1,ntsc1,fast1,lang1,aga1,files1,demo1))
                    else:
                        infodict[rname] = [(root,name[:-5],mem1,disk1,ntsc1,fast1,lang1,aga1,files1,demo1)]
                    #if name[:5] == "Dogfi":print("Found:",rname, name)
    return infodict

# adds game/demo qualifier to the string
def addQualifier(qlist,qnew):
    if qnew == "":
        return qlist
    else:
        return qlist+"/"+qnew

# generates list of games/demos from dictionary, according to preferences
def generateList(gdict):
    glist = []
    # go through all versions of game/demo
    for name in gdict:
        #print("NAME:",name,"VERSIONS:",len(gdict[name]),[i[5] for i in gdict[name]])
        vmemcfg = [v[2] for v in gdict[name]]
        vdiskcfg = [v[3] for v in gdict[name]]
        vntsc = [v[4] for v in gdict[name]]
        vfast = [v[5] for v in gdict[name]]
        vlang = [v[6] for v in gdict[name]]
        vaga = [v[7] for v in gdict[name]]
        vfiles = [v[8] for v in gdict[name]]
        vdemo = [v[9] for v in gdict[name]]

        # check available languages
        llist1 = []

        for l in plang:
            if l in vlang: llist1.extend( [i for i,x in enumerate(vlang) if x == l] )
        if llist1 == []:
            if StrictLanguage:
                continue
            else:
                llist1 = range(len(vlang))
        #print("LANG:",[vlang[i] for i in llist1],llist1)

        # check video system
        llist2 = []
        for i in llist1:
            if vntsc[i] in pntsc: llist2.append(i)
        if llist2 == []: llist2 = [i for i in llist1]
        #print("NTSC:",[vntsc[i] for i in llist2],llist2)

        # check memory type
        llist3 = []
        for i in llist2:
            if vfast[i] in pfast: llist3.append(i)
        if llist3 == []: llist3 = [i for i in llist2]
        #print("FAST:",[vfast[i] for i in llist3],llist3)

        # check files type
        llist4 = []
        for i in llist3:
            if vfiles[i] in pfiles: llist4.append(i)
        if llist4 == []: llist4 = [i for i in llist3]
        #print("FILES:",[vfiles[i] for i in llist4],llist4)

        # check if demo/preview
        llist5 = []
        for i in llist4:
            if vdemo[i] in pdemo: llist5.append(i)
        if llist5 == []: llist5 = [i for i in llist4]
        #print("DEMO:",[vdemo[i] for i in llist5],llist5)

        # check differences in versions
        dmemcfg = False
        lmemcfg = vmemcfg[llist5[0]]
        ddiskcfg = False
        dntsc = False
        lntsc = vntsc[llist5[0]]
        dfast = False
        lfast = vfast[llist5[0]]
        dfiles = False
        lfiles = vfiles[llist5[0]]
        ddemo = False
        ldemo = vdemo[llist5[0]]
        for i in llist5:
            if lmemcfg != vmemcfg[i]: dmemcfg = True
            if vdiskcfg[i] != "": ddiskcfg = True
            if lntsc != vntsc[i]: dntsc = True
            if lfast != vfast[i]: dfast = True
            if lfiles != vfiles[i]: dfiles = True
            if vdemo[i]: ddemo = True
            lmemcfg = vmemcfg[i]
            ldiskcfg = vdiskcfg[i]
            lntsc = vntsc[i]
            lfast = vfast[i]
            lfiles = vfiles[i]
            ldemo = vdemo[i]

        # put versions in list
        for i in llist5:
            lname = name
            qualifiers = ""
            if dmemcfg and vmemcfg[i] != "": qualifiers = addQualifier(qualifiers,vmemcfg[i])
            if ddiskcfg and vdiskcfg[i] != "": qualifiers = addQualifier(qualifiers,vdiskcfg[i] + "D")
            if dntsc and vntsc[i]: qualifiers = addQualifier(qualifiers,"N")
            if dfast and vfast[i]: qualifiers = addQualifier(qualifiers,"F")
            if ddemo and vdemo[i]: qualifiers = addQualifier(qualifiers,"Prv")
            if dfiles: qualifiers = addQualifier(qualifiers,"Fls" if vfiles[i] else "Img")
            if ShowOnlyCD32:
                if (vaga[i] == "CD32"): lname = lname + " " + "CD32"
            else:
                if vaga[i] != "": lname = lname + " " + vaga[i]
            if vlang[i] != "En": qualifiers = addQualifier(qualifiers,vlang[i])
            lname += qualifiers
            glist.append( (lname, os.path.join(gdict[name][i][0], gdict[name][i][1])) )
            if OnlyFirstPrefedred: break
    return sorted(glist, key=lambda x: x[0].lower())

# splits list of games/demos by directories (0-9,A,B,...)
def splitByDir(glist):
    gsplit = {}
    for name in sorted(glist, key = lambda x: x[0].lower()):
        directory = name[0][0]
        if (directory >= "0") and (directory <= "9"): directory = "0-9"
        if directory in gsplit:
            gsplit[directory].append(name)
        else:
            gsplit[directory] = [name]
    return gsplit

# generates a table of links
# width - width of guide window in chars
# colums - numbr of columns for links
# padding - number of spaces between the columns
# prefix - prefix for links
# system - name of program for SYSTEM links
def generateLinks(glist,width,columns,padding,rpadding,prefix,system=None):
    length = len(glist)
    rows = (length + columns - 1) / columns
    txt = ""
    colwidth = (width - padding * (columns-1)) / columns
    bpadding = (width - colwidth*columns - padding * (columns-1)) / 2
    for r in range(rows):
        line = " " * bpadding
        for c in range(columns):
            i = c * rows + r
            item = "@{"
            if i < length:
                name = shortenName(glist[i][0],colwidth)
                item += '"%s"' % name
                if system:
                    item += ' SYSTEM "%s %s%s"' % (system,prefix,glist[i][1])
                else:
                    item += " LINK %s%s" % (prefix,glist[i][1])
                item += "}"
                if c > 0: line += " " * padding
                line += item
                if len(name) < colwidth: line += " " * (colwidth - len(name))
        txt += line + "\n"
        txt += "\n" * rpadding
    return txt

def centerText(txt, width=WindowCharWidth):
    l = len(txt)
    if l < width:
        return (" " * ( (width-l)/2)) + txt
    else:
        return txt

def generateHF(lines,name="",directory=""):
    txt = ""
    for line in lines:
        ltxt = line[0]
        ltxt = ltxt.replace("#01#",directory)
        ltxt = ltxt.replace("#02#",CurrentTime)
        ltxt = ltxt.replace("#03#",name)
        if line[1]: ltxt = centerText(ltxt)
        txt += ltxt + "\n"
    return txt

def generateListFiles(directory,name,prefix,drivepath,isAGA,isGame):
    print("Scanning info files for %s ..." % name)
    gdict = scanDir(directory,isAGA,isGame)
    print("Generating list entries for %s\n" % name)
    glist = generateList(gdict)
    gsplit = splitByDir(glist)
    dirs = sorted(gsplit, key=lambda x: x[0].lower())
    ilist = []
    for l in dirs:
        txt = generateLinks(gsplit[l],WindowCharWidth,GameColumns,ColumnPadding,RowsPadding,drivepath,WBRunPath)
        header = generateHF(ListHeader,name,l)
        footer = generateHF(ListFooter,name,l)
        gname = "%s%s-%s.guide" % (LauncherPrefix,prefix,l)
        guideFName = os.path.join(Guides,gname)
        guide = open(guideFName, "w")
        guide.write(header)
        guide.write(txt)
        guide.write(footer)
        guide.close()
        os.chmod(guideFName,0o664)
        if len(l) == 1:
            lname = " " + l + " "
        else:
            lname = l
        ilist.append( (lname,"/%s/main" % gname) )
    txt = "Select %s directory:\n" % name
    txt += generateLinks(ilist,WindowCharWidth,LetterColumns,LetterPadding,LetterRPadding,LauncherPath)
    return txt

# create directory for guide generation, if missing
if not os.path.isdir(Guides):
    os.makedirs(Guides,0o775)

# set variables for preferences
if PreferredLanguages == "":
    plang = LangList
else:
    plang = PreferredLanguages.split()

if PreferredTVSystem == "PAL":
    pntsc = [False]
elif PreferredTVSystem == "NTSC":
    pntsc = [True]
else:
    pntsc = [False,True]

if PreferredMemType == "Slow":
    pfast = [False]
elif PreferredMemType == "Fast":
    pfast = [True]
else:
    pfast = [False,True]

if PreferredFileType == "Files":
    pfiles = [True]
elif PreferredFileType == "Image":
    pfiles = [False]
else:
    pfiles = [False,True]

if PreferredRelease == "Release":
    pdemo = [False]
elif PreferredRelease == "Demo":
    pdemo = [True]
else:
    pdemo = [False,True]

if TranslateToEnglish:
    ExpandAs.update(Translations)

# generate game/demo lists
print("Scanning could take a 1-2 minutes, please be patient\n")
txt = generateListFiles(Games,"ECS Games","Games",GamesDrivePath,False,True)
txt += "\n" + generateListFiles(AGA,"AGA Games","AGA",GamesDrivePath,True,True)
ShowOnlyCD32 = False
txt += "\n" + generateListFiles(Demos,"Demos","Demos",GamesDrivePath,False,False)

# generate main guide file
print("Genrating main launcher file...")
header = generateHF(MainHeader)
footer = generateHF(MainFooter)
gname = "%s.guide" % MainLauncher
guide = open( os.path.join(Guides,gname), "w" )
guide.write(header)
guide.write(txt)
guide.write(footer)
guide.close()

# generate info file
InfoFileData = InfoFileData.replace("\n","")
infoFName = os.path.join(Guides,gname+".info")
info = open(infoFName, "w")
info.write(binascii.unhexlify(InfoFileData))
info.close
os.chmod(infoFName,0o664)
print("All done!")

# history

# 1.0 - initial version

# 1.1
#   - many adjustments to name expansion
#   - few code errors solved
#   - added custom name expansion and translations from other languages to english
#     for games that exist in more languages
#   - added several new options: PreferredRelease, StrictLanguage, TranslateToEnglish
#   - program now displays what it is doing

