
#include <stdlib.h>
#include <stdio.h>

#include "console.h"


// Amiga includes.
#include <proto/exec.h>
#include <proto/dos.h>

#include <sys/stat.h>

 
#define __PHYSICSFS_INTERNAL__
#include "physfs_internal.h"

/* Handy macros for checking what kind of object a ExAllData's 
   ed_Type describes;  ExAll() */

#define EAD_IS_FILE(ead)    ((ead)->ed_Type <  0)

#define EAD_IS_DRAWER(ead)  ((ead)->ed_Type >= 0 && \
                             (ead)->ed_Type != ST_SOFTLINK)

#define EAD_IS_LINK(ead)    ((ead)->ed_Type == ST_SOFTLINK || \
                             (ead)->ed_Type == ST_LINKDIR || \
                             (ead)->ed_Type == ST_LINKFILE)

#define EAD_IS_SOFTLINK(ead) ((ead)->ed_Type == ST_SOFTLINK)

#define EAD_IS_LINKDIR(ead)  ((ead)->ed_Type == ST_LINKDIR)

#define FIB_IS_FILE(fib)      ((fib)->fib_DirEntryType <  0)

#define FIB_IS_DRAWER(fib)    ((fib)->fib_DirEntryType >= 0 && \
                               (fib)->fib_DirEntryType != ST_SOFTLINK)

#define FIB_IS_LINK(fib)      ((fib)->fib_DirEntryType == ST_SOFTLINK || \
                               (fib)->fib_DirEntryType == ST_LINKDIR || \
                               (fib)->fib_DirEntryType == ST_LINKFILE)

#define FIB_IS_SOFTLINK(fib)  ((fib)->fib_DirEntryType == ST_SOFTLINK)

#define FIB_IS_LINKDIR(fib)   ((fib)->fib_DirEntryType == ST_LINKDIR)


// 16K ExAllData buffer.
#define FILE_DATA_BUFFER 16384  

typedef struct
{
    //BPTR    lock;
    FILE    *handle;
    //int     isDirectory;
} AmigaFile;

const char *__PHYSFS_platformDirSeparator = "/";









/* -ansi and -pedantic flags prevent use of strcasecmp() on Linux. */
int __PHYSFS_platformStricmp(const char *x, const char *y)
{
    int ux, uy;

    do
    {
        ux = toupper((int) *x);
        uy = toupper((int) *y);
        if (ux != uy)
            return((ux > uy) ? 1 : -1);
        x++;
        y++;
    } while ((ux) && (uy));

    return(0);
} /* __PHYSFS_platformStricmp */


int __PHYSFS_platformStrnicmp(const char *x, const char *y, PHYSFS_uint32 len)
{
    int ux, uy;

    if (!len)
        return(0);

    do
    {
        ux = toupper((int) *x);
        uy = toupper((int) *y);
        if (ux != uy)
            return((ux > uy) ? 1 : -1);
        x++;
        y++;
        len--;
    } while ((ux) && (uy) && (len));

    return(0);
} /* __PHYSFS_platformStrnicmp */

int __PHYSFS_platformExists(const char *fname)
{
    struct stat statbuf;   
    
    return (stat(fname, &statbuf) == 0);
} /* __PHYSFS_platformExists */


int __PHYSFS_platformIsSymLink(const char *fname)
{
#if (defined __PHYSFS_NO_SYMLINKS__)
    return(0);
#else
    struct stat statbuf;
    BAIL_IF_MACRO(lstat(fname, &statbuf) == -1, ERR_IS_INITIALIZED, 0);
    return( (S_ISLNK(statbuf.st_mode)) ? 1 : 0 );
#endif
} /* __PHYSFS_platformIsSymlink */

int __PHYSFS_platformIsDirectory(const char *fname)
{
    struct stat statbuf;
    BAIL_IF_MACRO(stat(fname, &statbuf) == -1, ERR_IS_INITIALIZED, 0);
    return( (S_ISDIR(statbuf.st_mode)) ? 1 : 0 );  
} /* __PHYSFS_platformIsDirectory */

char *__PHYSFS_platformCvtToDependent(const char *prepend,
                                      const char *dirName,
                                      const char *append)
{
    
    int len = ((prepend) ? strlen(prepend) : 0) +
              ((append) ? strlen(append) : 0) +
              strlen(dirName) + 1;
              
    char *retval = (char *) allocator.Malloc(len);
    

    BAIL_IF_MACRO(retval == NULL, ERR_OUT_OF_MEMORY, NULL);

    /* platform-independent notation is Unix-style already.  :)  */

    if (prepend)
        strcpy(retval, prepend);
    else
        retval[0] = '\0';

    strcat(retval, dirName);

    if (append)
        strcat(retval, append);
        

    return(retval);
} /* __PHYSFS_platformCvtToDependent */



void __PHYSFS_platformEnumerateFiles(const char *dirname,
                                     int omitSymLinks,
                                     PHYSFS_EnumFilesCallback callback,
                                     const char *origdir,
                                     void *callbackdata)                                     
{
    int _bIsDirectory = 0;
    int _bIsValid = 0;


    struct FileInfoBlock *fib = (struct FileInfoBlock *)AllocDosObject(DOS_FIB, TAG_END);
    if (!fib) {
		__PHYSFS_setError(ERR_OUT_OF_MEMORY);
		return;
	}

	// Check whether the node exists and if it is a directory
	BPTR _pFileLock = Lock(dirname, SHARED_LOCK);
	if (_pFileLock) {
		if (Examine(_pFileLock, fib) != DOSFALSE) {
			if (FIB_IS_DRAWER(fib)) {
				_bIsDirectory = 1;
				_bIsValid = (_pFileLock != 0);

				// Add a trailing slash if it is needed
				/*const char c = _sPath.lastChar();
				if (c != '/' && c != ':')
					_sPath += '/';*/
			}
		}
		
		
		if ((!_bIsValid) || (!_bIsDirectory)) {
            con_printf(CON_DEBUG, "Node does not exist or is not a directory\n");
            UnLock(_pFileLock);
            FreeDosObject(DOS_FIB, fib);
            return;
        }
    }	
		
		
    struct ExAllControl *eac = (struct ExAllControl *)AllocDosObject(DOS_EXALLCONTROL, TAG_END);
    if (eac) {
		struct ExAllData *data = (struct ExAllData *)AllocVec(FILE_DATA_BUFFER, MEMF_ANY);
		if (data) {
			BOOL bExMore;
			eac->eac_LastKey = 0;
			do {
				// Examine directory
				bExMore = ExAll(_pFileLock, data, FILE_DATA_BUFFER, ED_TYPE, eac);

				LONG error = IoErr();
				if (!bExMore && error != ERROR_NO_MORE_ENTRIES)
					break; // Abnormal failure

				if (eac->eac_Entries ==	0)
					continue; // Normal failure, no entries

				struct ExAllData *ead = data;
				do {
					if (EAD_IS_FILE(ead)) {
						//BPTR lock = Lock(filePath, SHARED_LOCK);
						//if (lock) {
							callback(callbackdata, origdir, (char*)ead->ed_Name);
							//UnLock(lock);
						//}
					}
					ead = ead->ed_Next;
				} while (ead);
			} while (bExMore);

			FreeVec(data);
		}

		FreeDosObject(DOS_EXALLCONTROL, eac);
	}		
	

	UnLock(_pFileLock);
	FreeDosObject(DOS_FIB, fib);

} /* __PHYSFS_platformEnumerateFiles */




int __PHYSFS_platformMkDir(const char *path)
{
    int rc;
    rc = mkdir(path, S_IRWXU);
    BAIL_IF_MACRO(rc == -1, ERR_IS_INITIALIZED, 0);
    return(1);
} /* __PHYSFS_platformMkDir */










static void *doOpen(const char *filename, const char *mode)
{   
    AmigaFile *retval;
    FILE *file = NULL;
    
         
    // This is a file so we can open it.
    file = fopen(filename, mode);
    
    
    BAIL_IF_MACRO(file == NULL, ERR_BAD_FILENAME, NULL);     
 
   
    retval = (AmigaFile *) allocator.Malloc(sizeof(AmigaFile));
    if (retval == NULL)
    {
        /*if (isDirectory) {
            UnLock(lock);
        } else */{
            fclose(file);
        }
        
        BAIL_MACRO(ERR_OUT_OF_MEMORY, NULL);
    }
    
        
    // Note that this if this is a directory it is currently locked.
    // Note that file will be NULL if this is a directory.    
    retval->handle = file;
    //retval->lock = lock;
    //retval->isDirectory = isDirectory;
    return(retval);
} /* doOpen */

void *__PHYSFS_platformOpenRead(const char *filename)
{
    return(doOpen(filename, "rb"));
} /* __PHYSFS_platformOpenRead */


void *__PHYSFS_platformOpenWrite(const char *filename)
{        
    return(doOpen(filename, "wb"));
} /* __PHYSFS_platformOpenWrite */


void *__PHYSFS_platformOpenAppend(const char *filename)
{
    return(doOpen(filename, "ab"));
} /* __PHYSFS_platformOpenAppend */

PHYSFS_sint64 __PHYSFS_platformRead(void *opaque, void *buffer,
                                    PHYSFS_uint32 size, PHYSFS_uint32 count)
{
    PHYSFS_sint64 retval;
    FILE *file = ((AmigaFile *) opaque)->handle;

    BAIL_IF_MACRO(file == NULL, ERR_IO_ERROR, retval);
    
    retval = (PHYSFS_sint64) fread(buffer, size, count, file);
    
    BAIL_IF_MACRO(retval != count, ERR_IO_ERROR, retval);

    return retval;
} /* __PHYSFS_platformRead */

PHYSFS_sint64 __PHYSFS_platformWrite(void *opaque, const void *buffer,
                                     PHYSFS_uint32 size, PHYSFS_uint32 count)
{
    PHYSFS_sint64 retval;
    FILE *file = ((AmigaFile *) opaque)->handle;

    retval = (PHYSFS_sint64) fwrite(buffer, size, count, file);
    BAIL_IF_MACRO(retval != count, ERR_IO_ERROR, -1);
    BAIL_IF_MACRO(retval == 0, ERR_IO_ERROR, -1);

    return retval;
} /* __PHYSFS_platformWrite */

int __PHYSFS_platformSeek(void *opaque, PHYSFS_uint64 offset)
{
    FILE *file = ((AmigaFile *) opaque)->handle;
    
    BAIL_IF_MACRO(fseek(file, (int) offset, SEEK_SET), ERR_IO_ERROR, -1);
    
    return 1;  /* No error occured */
} /* __PHYSFS_platformSeek */

PHYSFS_sint64 __PHYSFS_platformTell(void *opaque)
{
    PHYSFS_sint64 retval;
    FILE *file = ((AmigaFile *) opaque)->handle;

    retval = (PHYSFS_sint64) ftell(file);
    BAIL_IF_MACRO(retval == -1, ERR_IO_ERROR, 0);

    return retval;
} /* __PHYSFS_platformTell */

PHYSFS_sint64 __PHYSFS_platformFileLength(void *opaque)
{
    PHYSFS_sint64 retval;
    FILE *file = ((AmigaFile *) opaque)->handle;
    long currentPosition = 0;
    
    currentPosition = ftell(file);
    BAIL_IF_MACRO(currentPosition == -1, ERR_IO_ERROR, -1);
        
    fseek(file, 0, SEEK_END); // seek to end of file
    retval = ftell(file); // get file position
    BAIL_IF_MACRO(retval == -1, ERR_IO_ERROR, -1);
    fseek(file, 0, SEEK_SET); // seek back to beginning of file
    
    // seek back to the original position.
    fseek(file, currentPosition, SEEK_CUR);
    
    return retval;
} /* __PHYSFS_platformFileLength */

int __PHYSFS_platformEOF(void *opaque)
{
    PHYSFS_sint64 filePosition;
    int retval = 0;

    /* Get the current position in the file */
    if ((filePosition = __PHYSFS_platformTell(opaque)) != 0)
    {
        /* Non-zero if EOF is equal to the file length */
        retval = filePosition == __PHYSFS_platformFileLength(opaque);
    } /* if */

    return(retval);
} /* __PHYSFS_platformEOF */
 

int __PHYSFS_platformFlush(void *opaque)
{
    FILE *file = ((AmigaFile *) opaque)->handle;

    BAIL_IF_MACRO(fflush(file) != 0, ERR_IO_ERROR, 0);

    return 1;
} /* __PHYSFS_platformFlush */

int __PHYSFS_platformClose(void *opaque)
{
    AmigaFile *amigaFile = (AmigaFile *) opaque;

    if (amigaFile->handle) {
        BAIL_IF_MACRO(fclose(amigaFile->handle) != 0, ERR_IO_ERROR, 0);
        amigaFile->handle = NULL;    
    }
    
    allocator.Free(opaque);

    return 1;
} /* __PHYSFS_platformClose */

int __PHYSFS_platformDelete(const char *path)
{    
    BAIL_IF_MACRO(remove(path) != 0, ERR_IO_ERROR, 0);

    return 1;
} /* __PHYSFS_platformDelete */

PHYSFS_sint64 __PHYSFS_platformGetLastModTime(const char *fname)
{
    struct stat statbuf;
    
    BAIL_IF_MACRO(stat(fname, &statbuf) < 0, ERR_IO_ERROR, -1);

    return statbuf.st_mtime;
} /* __PHYSFS_platformGetLastModTime */







static char baseDirectory[PATH_MAX];


char *__PHYSFS_platformGetUserDir(void)
{
    return baseDirectory;
} /* __PHYSFS_platformGetUserDir */

char *__PHYSFS_platformGetUserName(void)
{
    Error("ERR_NOT_IMPLEMENTED + __PHYSFS_platformGetUserName");
    
    return NULL;    
} /* __PHYSFS_platformGetUserName */

char *__PHYSFS_platformCalcBaseDir(const char *argv0)
{    
    GetCurrentDirName(baseDirectory, PATH_MAX);   

    return(baseDirectory);
} /* __PHYSFS_platformCalcBaseDir */

char *__PHYSFS_platformCurrentDir(void)
{
    char *retval = (char *) allocator.Malloc(PATH_MAX);
    BAIL_IF_MACRO(retval == NULL, ERR_OUT_OF_MEMORY, NULL);
    
    GetCurrentDirName(retval, PATH_MAX);   

    return(retval);
} /* __PHYSFS_platformCurrentDir */

char *__PHYSFS_platformRealPath(const char *path)
{
    char *retval = (char *) allocator.Malloc(strlen(path) + 1);
    
    strcpy(retval,path);
    return(retval); 
} /* __PHYSFS_platformRealPath */




/* Stub version for platforms without CD-ROM support. */
void __PHYSFS_platformDetectAvailableCDs(PHYSFS_StringCallback cb, void *data)
{
} /* __PHYSFS_platformDetectAvailableCDs */



















int __PHYSFS_platformInit(void)
{
    return(1);  /* always succeed. */
} /* __PHYSFS_platformInit */


int __PHYSFS_platformDeinit(void)
{
    return(1);  /* always succeed. */
} /* __PHYSFS_platformDeinit */





int __PHYSFS_platformSetDefaultAllocator(PHYSFS_Allocator *a)
{
    return(0);  /* just use malloc() and friends. */
} /* __PHYSFS_platformSetDefaultAllocator */









// No Mutex support.
void *__PHYSFS_platformGetThreadID(void) { return((void *) 0x0001); }
void *__PHYSFS_platformCreateMutex(void) { return((void *) 0x0001); }
void __PHYSFS_platformDestroyMutex(void *mutex) {}
int __PHYSFS_platformGrabMutex(void *mutex) { return(1); }
void __PHYSFS_platformReleaseMutex(void *mutex) {}


















int __PHYSFS_platformAllocatorInit(void)
{
    return(1);  /* always succeeds. */
} /* __PHYSFS_platformAllocatorInit */


void __PHYSFS_platformAllocatorDeinit(void)
{
    /* no-op */
} /* __PHYSFS_platformAllocatorInit */

void *__PHYSFS_platformAllocatorMalloc(size_t s)
{
    #undef malloc
    return(malloc(s));
} /* __PHYSFS_platformMalloc */


void *__PHYSFS_platformAllocatorRealloc(void *ptr, size_t s)
{
    #undef realloc
    return(realloc(ptr, s));
} /* __PHYSFS_platformRealloc */


void __PHYSFS_platformAllocatorFree(void *ptr)
{
    #undef free
    free(ptr);
} /* __PHYSFS_platformAllocatorFree */
