/*
 * Wolfenstein: Enemy Territory GPL Source Code
 * Copyright (C) 1999-2010 id Software LLC, a ZeniMax Media company.
 *
 * ET: Legacy
 * Copyright (C) 2012 Jan Simek <mail@etlegacy.com>
 *
 * This file is part of ET: Legacy - http://www.etlegacy.com
 *
 * ET: Legacy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ET: Legacy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ET: Legacy. If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, Wolfenstein: Enemy Territory GPL Source Code is also
 * subject to certain additional terms. You should have received a copy
 * of these additional terms immediately following the terms and conditions
 * of the GNU General Public License which accompanied the source code.
 * If not, please request a copy in writing from id Software at the address below.
 *
 * id Software LLC, c/o ZeniMax Media Inc., Suite 120, Rockville, Maryland 20850 USA.
 *
 * @file files.c
 * @brief handle based filesystem for Quake III Arena
 */

#include "q_shared.h"
#include "qcommon.h"
#include "unzip.h"
#ifdef _WIN32
#define realpath(N, R) _fullpath((R), (N), _MAX_PATH)
#elif defined (__MORPHOS__)
#define realpath(N, R) strdup(N)
#endif // _WIN32

/*
=============================================================================
ET: Legacy (QUAKE3) FILESYSTEM

FIXME/Note: This docu needs an update - the following lines aren't fully valid for ET: Legacy

All of ET: Legacy'S (Quake's) data access is through a hierarchical file system, but the contents of
the file system can be transparently merged from several sources.

A "qpath" is a reference to game file data.  MAX_ZPATH is 256 characters, which must include
a terminating zero. "..", "\\", and ":" are explicitly illegal in qpaths to prevent any
references outside the quake directory system.

The "base path" is the path to the directory holding all the game directories and usually
the executable.  It defaults to ".", but can be overridden with a "+set fs_basepath c:\etlegacy"
command line to allow code debugging in a different directory (make sure it contains an install)
Basepath cannot be modified at all after startup and is usually readonly.

The "home path" is the path used for all write access. On win32 systems we have had "base path"
== "home path", but this no longer the case for ET: Legacy - it requires "base path"
!= "home path" Any files that are created (demos, screenshots, etc) will be created relative
to the base path, so home path should usually be writable.
"home path" points to ~/.etlwolf or c:\<USER>\My Documets\etlwolf by default.

The user can also install custom mods and content in "home path", so it should be searched
along with "home path" and "base path" for game content. Mods downloads from online server games
are installed into "home path". These downloads install client files only. A full mod server
and client install should be done in "base path". This ensures the install of the game or mods
are never overwritten by any downloads.

The "base game" is the directory under the paths where data comes from by default, and
can be any path if set.

The "current game" may be the same as the base game, or it may be the name of another
directory under the paths that should be searched for files before looking in the base game.
This is the basis for addons.

Clients automatically set the game directory after receiving a gamestate from a server,
so only servers need to worry about +set fs_game.

No other directories outside of the base game and current game will ever be referenced by
filesystem functions.

To save disk space and speed loading, directory trees can be collapsed into zip files.
The files use a ".pk3" extension to prevent users from unzipping them accidentally, but
otherwise the are simply normal uncompressed zip files.  A game directory can have multiple
zip files of the form "pak0.pk3", "pak1.pk3", etc.  Zip files are searched in decending order
from the highest number to the lowest, and will always take precedence over the filesystem.
This allows a pk3 distributed as a patch to override all existing data.

Because we will have updated executables freely available online, there is no point to
trying to restrict demo / oem versions of the game with code changes.  Demo / oem versions
should be exactly the same executables as release versions, but with different data that
automatically restricts where game media can come from to prevent add-ons from working.

File search order: when FS_FOpenFileRead gets called it will go through the fs_searchpaths
structure and stop on the first successful hit. fs_searchpaths is built with successive
calls to FS_AddGameDirectory

Additionaly, we search in several subdirectories:
current game is the current mode
base game is a variable to allow mods based on other mods
(such as etmain + missionpack content combination in a mod for instance)
BASEGAME is the hardcoded base game ("etmain")

e.g. the qpath "sound/newstuff/test.wav" would be searched for in the following places:

home path + current game's zip files
home path + current game's directory
base path + current game's zip files
base path + current game's directory
cd path + current game's zip files
cd path + current game's directory

home path + base game's zip file
home path + base game's directory
base path + base game's zip file
base path + base game's directory
cd path + base game's zip file
cd path + base game's directory

home path + BASEGAME's zip file
home path + BASEGAME's directory
base path + BASEGAME's zip file
base path + BASEGAME's directory
cd path + BASEGAME's zip file
cd path + BASEGAME's directory

server download, to be written to home path + current game's directory


The filesystem can be safely shutdown and reinitialized with different
basedir / cddir / game combinations, but all other subsystems that rely on it
(sound, video) must also be forced to restart.

Because the same files are loaded by both the clip model (CM_) and renderer (TR_)
subsystems, a simple single-file caching scheme is used.  The CM_ subsystems will
load the file with a request to cache.  Only one file will be kept cached at a time,
so any models that are going to be referenced by both subsystems should alternate
between the CM_ load function and the ref load function.

TODO: A qpath that starts with a leading slash will always refer to the base game, even if another
game is currently active.  This allows character models, skins, and sounds to be downloaded
to a common directory no matter which game is active.

How to prevent downloading zip files?
Pass pk3 file names in systeminfo, and download before FS_Restart()?

Aborting a download disconnects the client from the server.

How to mark files as downloadable?  Commercial add-ons won't be downloadable.

Non-commercial downloads will want to download the entire zip file.
the game would have to be reset to actually read the zip in

Auto-update information

Path separators

Casing

  separate server gamedir and client gamedir, so if the user starts
  a local game after having connected to a network game, it won't stick
  with the network game.

  allow menu options for game selection?

Read / write config to floppy option.

Different version coexistance?

When building a pak file, make sure a etconfig.cfg isn't present in it,
or configs will never get loaded from disk!

  todo:

  downloading (outside fs?)
  game directory passing and restarting
  explain MAX_OSPATH and MAX_QPATH
=============================================================================
*/

#define MAX_ZPATH           256
#define MAX_SEARCH_PATHS    4096
#define MAX_FILEHASH_SIZE   1024

typedef struct fileInPack_s
{
	char *name;                         // name of the file
	unsigned long pos;                  // file info position in zip
	unsigned long len;                  // uncompress file size
	struct  fileInPack_s *next;         // next file in the hash
} fileInPack_t;

typedef struct
{
	char pakPathname[MAX_OSPATH];               // c:\etlegacy\etmain
	char pakFilename[MAX_OSPATH];               // c:\etlegacy\etmain\pak0.pk3
	char pakBasename[MAX_OSPATH];               // pak0
	char pakGamename[MAX_OSPATH];               // etmain
	unzFile handle;                             // handle to zip file
	int checksum;                               // regular checksum
	int pure_checksum;                          // checksum for pure
	int numfiles;                               // number of files in pk3
	int referenced;                             // referenced file flags
	int hashSize;                               // hash table size (power of 2)
	fileInPack_t **hashTable;                   // hash table
	fileInPack_t *buildBuffer;                  // buffer with the filenames etc.
} pack_t;

typedef struct
{
	char path[MAX_OSPATH];          // c:\etlegacy
	char fullpath[MAX_OSPATH];      // c:\etlegacy\etmain
	char gamedir[MAX_OSPATH];       // etmain
} directory_t;

typedef struct searchpath_s
{
	struct searchpath_s *next;

	pack_t *pack;           // only one of pack / dir will be non NULL
	directory_t *dir;
} searchpath_t;

char                fs_gamedir[MAX_OSPATH]; // this will be a single file name with no separators
static cvar_t       *fs_debug;
static cvar_t       *fs_homepath;
static cvar_t       *fs_basepath;
static cvar_t       *fs_basegame;
static cvar_t       *fs_gamedirvar;
static searchpath_t *fs_searchpaths;
static int          fs_readCount;           // total bytes read
static int          fs_loadCount;           // total files read
static int          fs_loadStack;           // total files in memory
static int          fs_packFiles = 0;       // total number of files in packs

static int fs_checksumFeed;

typedef union qfile_gus
{
	FILE *o;
	unzFile z;
} qfile_gut;

typedef struct qfile_us
{
	qfile_gut file;
	qboolean unique;
} qfile_ut;

typedef struct
{
	qfile_ut handleFiles;
	qboolean handleSync;
	int baseOffset;
	int fileSize;
	int zipFilePos;
	qboolean zipFile;
	char name[MAX_ZPATH];
} fileHandleData_t;

static fileHandleData_t fsh[MAX_FILE_HANDLES];

// wether we did a reorder on the current search path when joining the server
static qboolean fs_reordered;

// never load anything from pk3 files that are not present at the server when pure
// ex: when fs_numServerPaks != 0, FS_FOpenFileRead won't load anything outside of pk3 except .cfg .menu .game .dat
static int  fs_numServerPaks = 0;
static int  fs_serverPaks[MAX_SEARCH_PATHS];                    // checksums
static char *fs_serverPakNames[MAX_SEARCH_PATHS];               // pk3 names

// only used for autodownload, to make sure the client has at least
// all the pk3 files that are referenced at the server side
static int  fs_numServerReferencedPaks;
static int  fs_serverReferencedPaks[MAX_SEARCH_PATHS];              // checksums
static char *fs_serverReferencedPakNames[MAX_SEARCH_PATHS];         // pk3 names

// last valid game folder used
char lastValidBase[MAX_OSPATH];
char lastValidGame[MAX_OSPATH];

#ifdef FS_MISSING
FILE *missingFiles = NULL;
#endif

/* C99 defines __func__ */
#ifndef __func__
#   define __func__ "(unknown)"
#endif

qboolean FS_Initialized(void)
{
	return (fs_searchpaths != NULL);
}

qboolean FS_PakIsPure(pack_t *pack)
{
	if (fs_numServerPaks)
	{
		int i;

		for (i = 0 ; i < fs_numServerPaks ; i++)
		{
			// FIXME: also use hashed file names
			// NOTE TTimo: a pk3 with same checksum but different name would be validated too
			//   I don't see this as allowing for any exploit, it would only happen if the client does manips of it's file names 'not a bug'
			if (pack->checksum == fs_serverPaks[i])
			{
				return qtrue;       // on the approved list
			}
		}
		return qfalse;  // not on the pure server pak list
	}
	return qtrue;
}

/**
 * @brief return load stack
 */
int FS_LoadStack(void)
{
	return fs_loadStack;
}

/**
 * @brief return a hash value for the filename
 */
static long FS_HashFileName(const char *fname, int hashSize)
{
	int  i    = 0;
	long hash = 0;
	char letter;

	while (fname[i] != '\0')
	{
		letter = tolower(fname[i]);
		if (letter == '.')
		{
			break;                              // don't include extension
		}
		if (letter == '\\')
		{
			letter = '/';                       // damn path names
		}
		if (letter == PATH_SEP)
		{
			letter = '/';                       // damn path names

		}
		hash += (long)(letter) * (i + 119);
		i++;
	}
	hash  = (hash ^ (hash >> 10) ^ (hash >> 20));
	hash &= (hashSize - 1);

	return hash;
}

static fileHandle_t FS_HandleForFile(void)
{
	int i;

	for (i = 1 ; i < MAX_FILE_HANDLES ; i++)
	{
		if (fsh[i].handleFiles.file.o == NULL)
		{
			return i;
		}
	}
	Com_Error(ERR_DROP, "FS_HandleForFile: none free");
	return 0;
}

static FILE *FS_FileForHandle(fileHandle_t f)
{
	if (f < 1 || f >= MAX_FILE_HANDLES)
	{
		Com_Error(ERR_DROP, "FS_FileForHandle: %d out of range", f);
	}
	if (fsh[f].zipFile == qtrue)
	{
		Com_Error(ERR_DROP, "FS_FileForHandle: can't get FILE on zip file");
	}
	if (!fsh[f].handleFiles.file.o)
	{
		Com_Error(ERR_DROP, "FS_FileForHandle: NULL");
	}

	return fsh[f].handleFiles.file.o;
}

void FS_ForceFlush(fileHandle_t f)
{
	FILE *file;

	file = FS_FileForHandle(f);
	setvbuf(file, NULL, _IONBF, 0);
}

long FS_fplength(FILE *h)
{
	long pos;
	long end;

	pos = ftell(h);
	fseek(h, 0, SEEK_END);
	end = ftell(h);
	fseek(h, pos, SEEK_SET);

	return end;
}

/**
 * If this is called on a non-unique FILE (from a pak file), it will return
 * the size of the pak file, not the expected size of the file.
 */
long FS_filelength(fileHandle_t f)
{
	FILE *h;

	h = FS_FileForHandle(f);

	if (h == NULL)
	{
		return -1;
	}
	else
	{
		return FS_fplength(h);
	}
}

/**
 * @brief Fix things up differently for win/unix/mac
 */
static void FS_ReplaceSeparators(char *path)
{
	char     *s;
	qboolean lastCharWasSep = qfalse;

	for (s = path ; *s ; s++)
	{
		if (*s == '/' || *s == '\\')
		{
			if (!lastCharWasSep)
			{
				*s             = PATH_SEP;
				lastCharWasSep = qtrue;
			}
			else
			{
				memmove(s, s + 1, strlen(s));
			}
		}
		else
		{
			lastCharWasSep = qfalse;
		}
	}
}

/**
 * @brief Qpath may have either forward or backwards slashes
 */
char *FS_BuildOSPath(const char *base, const char *game, const char *qpath)
{
	char        temp[MAX_OSPATH];
	static char ospath[2][MAX_OSPATH];
	static int  toggle;

	toggle ^= 1;        // flip-flop to allow two returns without clash

	if (!game || !game[0])
	{
		game = fs_gamedir;
	}

	Com_sprintf(temp, sizeof(temp), "/%s/%s", game, qpath);
	FS_ReplaceSeparators(temp);
	Com_sprintf(ospath[toggle], sizeof(ospath[0]), "%s%s", base, temp);

	return ospath[toggle];
}

/**
 * @brief Creates any directories needed to store the given filename
 */
qboolean FS_CreatePath(char *OSPath)
{
	char *ofs;
	char path[MAX_OSPATH];

	// make absolutely sure that it can't back up the path
	if (strstr(OSPath, "..") || strstr(OSPath, "::"))
	{
		Com_Printf("WARNING: refusing to create relative path \"%s\"\n", OSPath);
		return qtrue;
	}

	Q_strncpyz(path, OSPath, sizeof(path));
	FS_ReplaceSeparators(path);

	// Skip creation of the root directory as it will always be there
	ofs = strchr(path, PATH_SEP);
	ofs++;

	for (; ofs != NULL && *ofs ; ofs++)
	{
		if (*ofs == PATH_SEP)
		{
			// create the directory
			*ofs = 0;
			if (!Sys_Mkdir(path))
			{
				Com_Error(ERR_FATAL, "FS_CreatePath: failed to create path \"%s\"",
				          path);
			}
			*ofs = PATH_SEP;
		}
	}

	return qfalse;
}

/**
 * @brief Copy a fully specified file from one place to another
 */
void FS_CopyFile(char *fromOSPath, char *toOSPath)
{
	FILE   *f;
	size_t len;
	byte   *buf;

	if ((!fromOSPath || fromOSPath[0] == '\0') || (!toOSPath || toOSPath[0] == '\0'))
	{
		Com_Printf(S_COLOR_YELLOW "WARNING: cannot copy files. Empty path passed to FS_CopyFile\n");
		return;
	}

	Com_Printf("Copying %s to %s\n", fromOSPath, toOSPath);

	if (strstr(fromOSPath, "journal.dat") || strstr(fromOSPath, "journaldata.dat"))
	{
		Com_Printf("Ignoring journal files\n");
		return;
	}

	f = Sys_FOpen(fromOSPath, "rb");
	if (!f)
	{
		return;
	}
	fseek(f, 0, SEEK_END);
	len = ftell(f);
	fseek(f, 0, SEEK_SET);

	buf = malloc(len);
	if (!buf)
	{
		Com_Error(ERR_FATAL, "FS_CopyFile: unable to allocate buffer");
	}
	if (fread(buf, 1, len, f) != len)
	{
		Com_Error(ERR_FATAL, "FS_CopyFile: Short read");
	}
	fclose(f);

	if (FS_CreatePath(toOSPath))
	{
		free(buf);
		return;
	}

	f = Sys_FOpen(toOSPath, "wb");
	if (!f)
	{
		free(buf);
		return;
	}
	if (fwrite(buf, 1, len, f) != len)
	{
		Com_Error(ERR_FATAL, "FS_CopyFile: short write");
	}
	fclose(f);
	free(buf);
}

/**
 * @brief ERR_FATAL if trying to manipulate a file with the platform library extension
 */
static void FS_CheckFilenameIsNotExecutable(const char *filename, const char *function)
{
	// Check if the filename ends with the library extension
	if (COM_CompareExtension(filename, DLL_EXT))
	{
		Com_Error(ERR_FATAL, "%s: Not allowed to manipulate '%s' due "
		                     "to %s extension", function, filename, DLL_EXT);
	}
}

void FS_Remove(const char *osPath)
{
	FS_CheckFilenameIsNotExecutable(osPath, __func__);

	remove(osPath);
}

void FS_HomeRemove(const char *homePath)
{
	FS_CheckFilenameIsNotExecutable(homePath, __func__);

	remove(FS_BuildOSPath(fs_homepath->string,
	                      fs_gamedir, homePath));
}

/**
 * @brief Tests if path and file exists
 */
qboolean FS_FileInPathExists(const char *testpath)
{
	FILE *filep;

	filep = Sys_FOpen(testpath, "rb");

	if (filep)
	{
		fclose(filep);
		return qtrue;
	}

	return qfalse;
}

/**
 * @brief Tests if the file exists in the current gamedir
 *
 * DOES NOT search the paths. This is to determine if opening a file to
 * write (which always goes into the current gamedir) will cause any overwrites.
 * @note this goes with FS_FOpenFileWrite for opening the file afterwards
 */
qboolean FS_FileExists(const char *file)
{
	if (!file)
	{
		Com_Printf("ERROR FS_FileExists: null filename\n");
		return qfalse;
	}

	if (file[0] == '\0')
	{
		Com_Printf("WARNING FS_FileExists: empty filename\n");
		return qfalse;
	}

	return FS_FileInPathExists(FS_BuildOSPath(fs_homepath->string, fs_gamedir, file));
}

/**
 * @brief Tests if the file exists
 */
qboolean FS_SV_FileExists(const char *file)
{
	char *testpath;

	testpath                       = FS_BuildOSPath(fs_homepath->string, file, "");
	testpath[strlen(testpath) - 1] = '\0';

	return FS_FileInPathExists(testpath);
}

fileHandle_t FS_SV_FOpenFileWrite(const char *filename)
{
	char         *ospath;
	fileHandle_t f;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_SV_FOpenFileWrite: Filesystem call made without initialization");
	}

	ospath                     = FS_BuildOSPath(fs_homepath->string, filename, "");
	ospath[strlen(ospath) - 1] = '\0';

	f              = FS_HandleForFile();
	fsh[f].zipFile = qfalse;

	if (fs_debug->integer)
	{
		Com_Printf("FS_SV_FOpenFileWrite: %s\n", ospath);
	}

	FS_CheckFilenameIsNotExecutable(ospath, __func__);

	if (FS_CreatePath(ospath))
	{
		return 0;
	}

	Com_DPrintf("writing to: %s\n", ospath);
	fsh[f].handleFiles.file.o = Sys_FOpen(ospath, "wb");

	Q_strncpyz(fsh[f].name, filename, sizeof(fsh[f].name));

	fsh[f].handleSync = qfalse;
	if (!fsh[f].handleFiles.file.o)
	{
		f = 0;
	}

	return f;
}

/**
 * @brief search for a file somewhere below the home path, base path or cd path
 * we search in that order, matching FS_SV_FOpenFileRead order
 */
long FS_SV_FOpenFileRead(const char *filename, fileHandle_t *fp)
{
	char         *ospath;
	fileHandle_t f = 0;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_SV_FOpenFileRead: Filesystem call made without initialization");
	}

	f              = FS_HandleForFile();
	fsh[f].zipFile = qfalse;

	Q_strncpyz(fsh[f].name, filename, sizeof(fsh[f].name));

	// search homepath
	ospath = FS_BuildOSPath(fs_homepath->string, filename, "");
	// remove trailing slash
	ospath[strlen(ospath) - 1] = '\0';

	if (fs_debug->integer)
	{
		Com_Printf("FS_SV_FOpenFileRead (fs_homepath): %s\n", ospath);
	}

	fsh[f].handleFiles.file.o = Sys_FOpen(ospath, "rb");
	fsh[f].handleSync         = qfalse;
	if (!fsh[f].handleFiles.file.o)
	{
		// If fs_homepath == fs_basepath, don't bother (dedicated server)
		// FIXME: use FS_PathCmp() (slashes, low/big letters ... trailing slash)
		if (Q_stricmp(fs_homepath->string, fs_basepath->string))
		{
			// search basepath
			ospath                     = FS_BuildOSPath(fs_basepath->string, filename, "");
			ospath[strlen(ospath) - 1] = '\0';

			if (fs_debug->integer)
			{
				Com_Printf("FS_SV_FOpenFileRead (fs_basepath): %s\n", ospath);
			}

			fsh[f].handleFiles.file.o = Sys_FOpen(ospath, "rb");
			fsh[f].handleSync         = qfalse;
		}
	}

	// File not found
	if (!fsh[f].handleFiles.file.o)
	{
		f = 0;
	}

	*fp = f;
	if (f)
	{
		return FS_filelength(f);
	}

	return -1;
}

void FS_SV_Rename(const char *from, const char *to)
{
	char *from_ospath, *to_ospath;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_SV_Rename: Filesystem call made without initialization");
	}

	from_ospath                          = FS_BuildOSPath(fs_homepath->string, from, "");
	to_ospath                            = FS_BuildOSPath(fs_homepath->string, to, "");
	from_ospath[strlen(from_ospath) - 1] = '\0';
	to_ospath[strlen(to_ospath) - 1]     = '\0';

	if (fs_debug->integer)
	{
		Com_Printf("FS_SV_Rename: %s --> %s\n", from_ospath, to_ospath);
	}
	FS_CheckFilenameIsNotExecutable(to_ospath, __func__);

	if (rename(from_ospath, to_ospath))
	{
		// Failed, try copying it and deleting the original
		FS_CopyFile(from_ospath, to_ospath);
		FS_Remove(from_ospath);
	}
}

/*
===========
FS_Rename
===========
*/
void FS_Rename(const char *from, const char *to)
{
	char *from_ospath, *to_ospath;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_Rename: Filesystem call made without initialization");
	}

	from_ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, from);
	to_ospath   = FS_BuildOSPath(fs_homepath->string, fs_gamedir, to);

	if (fs_debug->integer)
	{
		Com_Printf("FS_Rename: %s --> %s\n", from_ospath, to_ospath);
	}
	FS_CheckFilenameIsNotExecutable(to_ospath, __func__);

	if (rename(from_ospath, to_ospath))
	{
		// Failed, try copying it and deleting the original
		FS_CopyFile(from_ospath, to_ospath);
		FS_Remove(from_ospath);
	}
}

/*
==============
FS_FCloseFile

If the FILE pointer is an open pak file, leave it open.

For some reason, other dll's can't just cal fclose()
on files returned by FS_FOpenFile...
==============
*/
void FS_FCloseFile(fileHandle_t f)
{
	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FCloseFile: Filesystem call made without initialization");
	}

	if (fsh[f].zipFile == qtrue)
	{
		unzCloseCurrentFile(fsh[f].handleFiles.file.z);
		if (fsh[f].handleFiles.unique)
		{
			unzClose(fsh[f].handleFiles.file.z);
		}
		Com_Memset(&fsh[f], 0, sizeof(fsh[f]));
		return;
	}

	// we didn't find it as a pak, so close it as a unique file
	if (fsh[f].handleFiles.file.o)
	{
		fclose(fsh[f].handleFiles.file.o);
	}
	Com_Memset(&fsh[f], 0, sizeof(fsh[f]));
}

/*
===========
FS_FOpenFileWrite
===========
*/
fileHandle_t FS_FOpenFileWrite(const char *filename)
{
	char         *ospath;
	fileHandle_t f;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FOpenFileWrite: Filesystem call made without initialization");
	}

	f              = FS_HandleForFile();
	fsh[f].zipFile = qfalse;

	ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, filename);

	if (fs_debug->integer)
	{
		Com_Printf("FS_FOpenFileWrite: %s\n", ospath);
	}

	FS_CheckFilenameIsNotExecutable(ospath, __func__);

	if (FS_CreatePath(ospath))
	{
		return 0;
	}

	fsh[f].handleFiles.file.o = Sys_FOpen(ospath, "wb");

	Q_strncpyz(fsh[f].name, filename, sizeof(fsh[f].name));

	fsh[f].handleSync = qfalse;
	if (!fsh[f].handleFiles.file.o)
	{
		f = 0;
	}
	return f;
}

/*
===========
FS_FOpenFileWrite
===========
*/
fileHandle_t FS_FOpenDLLWrite(const char *filename)
{
	char         *ospath;
	fileHandle_t f;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FOpenDLLWrite: Filesystem call made without initialization\n");
	}

	f              = FS_HandleForFile();
	fsh[f].zipFile = qfalse;

	ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, filename);

	if (fs_debug->integer)
	{
		Com_Printf("FS_FOpenDLLWrite: %s\n", ospath);
	}

	if (FS_CreatePath(ospath))
	{
		return 0;
	}

	fsh[f].handleFiles.file.o = Sys_FOpen(ospath, "wb");

	Q_strncpyz(fsh[f].name, filename, sizeof(fsh[f].name));

	fsh[f].handleSync = qfalse;
	if (!fsh[f].handleFiles.file.o)
	{
		f = 0;
	}
	return f;
}

/*
===========
FS_FOpenFileAppend
===========
*/
fileHandle_t FS_FOpenFileAppend(const char *filename)
{
	char         *ospath;
	fileHandle_t f;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FOpenFileAppend: Filesystem call made without initialization");
	}

	f              = FS_HandleForFile();
	fsh[f].zipFile = qfalse;

	Q_strncpyz(fsh[f].name, filename, sizeof(fsh[f].name));

	ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, filename);

	if (fs_debug->integer)
	{
		Com_Printf("FS_FOpenFileAppend: %s\n", ospath);
	}

	FS_CheckFilenameIsNotExecutable(ospath, __func__);

	if (FS_CreatePath(ospath))
	{
		return 0;
	}

	fsh[f].handleFiles.file.o = Sys_FOpen(ospath, "ab");
	fsh[f].handleSync         = qfalse;
	if (!fsh[f].handleFiles.file.o)
	{
		f = 0;
	}
	return f;
}

/*
===========
FS_FilenameCompare

Ignore case and seprator char distinctions
===========
*/
qboolean FS_FilenameCompare(const char *s1, const char *s2)
{
	int c1, c2;

	do
	{
		c1 = *s1++;
		c2 = *s2++;

		if (c1 >= 'a' && c1 <= 'z')
		{
			c1 -= ('a' - 'A');
		}
		if (c2 >= 'a' && c2 <= 'z')
		{
			c2 -= ('a' - 'A');
		}

		if (c1 == '\\' || c1 == ':')
		{
			c1 = '/';
		}
		if (c2 == '\\' || c2 == ':')
		{
			c2 = '/';
		}

		if (c1 != c2)
		{
			return qtrue;      // strings not equal
		}
	}
	while (c1);

	return qfalse;       // strings are equal
}

/*
===========
FS_IsExt

Return qtrue if ext matches file extension filename
===========
*/
qboolean FS_IsExt(const char *filename, const char *ext, int namelen)
{
	int extlen;

	extlen = strlen(ext);

	if (extlen > namelen)
	{
		return qfalse;
	}

	filename += namelen - extlen;

	return !Q_stricmp(filename, ext);
}

/*
===========
FS_IsDemoExt

Return qtrue if filename has a demo extension
===========
*/
qboolean FS_IsDemoExt(const char *filename, int namelen)
{
	char *ext_test;

	ext_test = strrchr(filename, '.');
	if (ext_test && !Q_stricmpn(ext_test + 1, DEMOEXT, ARRAY_LEN(DEMOEXT) - 1))
	{
		int index;
		int protocol = atoi(ext_test + ARRAY_LEN(DEMOEXT));

		if (protocol == PROTOCOL_VERSION /*com_protocol->integer*/)
		{
			return qtrue;
		}

		for (index = 0; demo_protocols[index]; index++)
		{
			if (demo_protocols[index] == protocol)
			{
				return qtrue;
			}
		}
	}

	return qfalse;
}

extern qboolean com_fullyInitialized;

/**
 * @brief Finds the file in the search path.
 * Used for streaming data out of either a separate file or a ZIP file.
 *
 * @param[in]   filename to be opened
 * @param[out]  file set to open FILE pointer
 * @param[in]   uniqueFILE
 * @returns filesize or qboolean indicating if file exists when FILE pointer param is NULL
 */
static int fs_filter_flag = 0;

long FS_FOpenFileReadDir(const char *filename, searchpath_t *search, fileHandle_t *file, qboolean uniqueFILE, qboolean unpure)
{
	long         hash;
	pack_t       *pak;
	fileInPack_t *pakFile;
	directory_t  *dir;
	char         *netpath;
	FILE         *filep;
	int          len;

	if (filename == NULL)
	{
		Com_Error(ERR_FATAL, "FS_FOpenFileReadDir: NULL 'filename' parameter passed");
	}

	// qpaths are not supposed to have a leading slash
	if (filename[0] == '/' || filename[0] == '\\')
	{
		filename++;
	}

	// make absolutely sure that it can't back up the path.
	// The searchpaths do guarantee that something will always
	// be prepended, so we don't need to worry about "c:" or "//limbo"
	if (strstr(filename, "..") || strstr(filename, "::"))
	{
		if (file == NULL)
		{
			return qfalse;
		}

		*file = 0;
		return -1;
	}

	// make sure the q3key file is only readable by the quake3.exe at initialization
	// any other time the key should only be accessed in memory using the provided functions
	if (com_fullyInitialized && strstr(filename, "etkey"))
	{
		if (file == NULL)
		{
			return qfalse;
		}

		*file = 0;
		return -1;
	}

	if (file == NULL)
	{
		// just wants to see if file is there

		// is the element a pak file?
		if (search->pack)
		{
			hash = FS_HashFileName(filename, search->pack->hashSize);

			if (search->pack->hashTable[hash])
			{
				// look through all the pak file elements
				pak     = search->pack;
				pakFile = pak->hashTable[hash];

				do
				{
					// case and separator insensitive comparisons
					if (!FS_FilenameCompare(pakFile->name, filename))
					{
						// found it!
						if (pakFile->len)
						{
							return pakFile->len;
						}
						else
						{
							// It's not nice, but legacy code depends
							// on positive value if file exists no matter
							// what size
							return 1;
						}
					}

					pakFile = pakFile->next;
				}
				while (pakFile != NULL);
			}
		}
		else if (search->dir)
		{
			dir = search->dir;

			netpath = FS_BuildOSPath(dir->path, dir->gamedir, filename);
			filep   = Sys_FOpen(netpath, "rb");

			if (filep)
			{
				len = FS_fplength(filep);
				fclose(filep);

				if (len)
				{
					return len;
				}
				else
				{
					return 1;
				}
			}
		}

		return 0;
	}

	*file                         = FS_HandleForFile();
	fsh[*file].handleFiles.unique = uniqueFILE;

	// is the element a pak file?
	if (search->pack)
	{
		hash = FS_HashFileName(filename, search->pack->hashSize);

		if (search->pack->hashTable[hash])
		{
			// disregard if it doesn't match one of the allowed pure pak files
			if (!unpure && !FS_PakIsPure(search->pack))
			{
				*file = 0;
				return -1;
			}

			// look through all the pak file elements
			pak     = search->pack;
			pakFile = pak->hashTable[hash];

			do
			{
				// case and separator insensitive comparisons
				if (!FS_FilenameCompare(pakFile->name, filename))
				{
					// found it!

					// mark the pak as having been referenced and mark specifics on cgame and ui
					// shaders, txt, arena files  by themselves do not count as a reference as
					// these are loaded from all pk3s
					// from every pk3 file..
					len = strlen(filename);

					if (!(pak->referenced & FS_GENERAL_REF))
					{
						if (!FS_IsExt(filename, ".shader", len) &&
						    !FS_IsExt(filename, ".txt", len) &&
						    !FS_IsExt(filename, ".cfg", len) &&
						    !FS_IsExt(filename, ".config", len) &&
						    !FS_IsExt(filename, ".bot", len) &&
						    !FS_IsExt(filename, ".arena", len) &&
						    !FS_IsExt(filename, ".menu", len) && // FIXME: remove from ETL ? *.bot files are obsolete
						    Q_stricmp(filename, Sys_GetDLLName("qagame")) != 0 &&
						    !strstr(filename, "levelshots"))
						{
							pak->referenced |= FS_GENERAL_REF;
						}
					}

					// for OS client/server interoperability, we expect binaries for .so and .dll to be in the same pk3
					// so that when we reference the DLL files on any platform, this covers everyone else

#ifdef FEATURE_PUREHACK
					// CLIENT and SERVER
					if (!(pak->referenced & FS_QAGAME_REF) && !Q_stricmp(filename, "qagame_mp_x86.dll"))
					{
						Com_DPrintf("PureHack: %s QAGAME referenced!\n", pak->pakBasename);
						pak->referenced |= FS_QAGAME_REF;
					}
					if (!(pak->referenced & FS_CGAME_REF) && !Q_stricmp(filename, "cgame_mp_x86.dll"))
					{
						Com_DPrintf("PureHack: %s CGAME referenced!\n", pak->pakBasename);
						pak->referenced |= FS_CGAME_REF;
					}
					if (!(pak->referenced & FS_UI_REF) && !Q_stricmp(filename, "ui_mp_x86.dll"))
					{
						Com_DPrintf("PureHack: %s UI referenced!\n", pak->pakBasename);
						pak->referenced |= FS_UI_REF;
					}
#else
					// qagame dll
					if (!(pak->referenced & FS_QAGAME_REF) && !Q_stricmp(filename, Sys_GetDLLName("qagame")))
					{
						pak->referenced |= FS_QAGAME_REF;
					}
					// cgame dll
					if (!(pak->referenced & FS_CGAME_REF) && !Q_stricmp(filename, Sys_GetDLLName("cgame")))
					{
						pak->referenced |= FS_CGAME_REF;
					}
					// ui dll
					if (!(pak->referenced & FS_UI_REF) && !Q_stricmp(filename, Sys_GetDLLName("ui")))
					{
						pak->referenced |= FS_UI_REF;
					}
#endif

					if (uniqueFILE)
					{
						// open a new file on the pakfile
						fsh[*file].handleFiles.file.z = unzOpen(pak->pakFilename);

						if (fsh[*file].handleFiles.file.z == NULL)
						{
							Com_Error(ERR_FATAL, "FS_FOpenFileReadDir: Couldn't open %s", pak->pakFilename);
						}
					}
					else
					{
						fsh[*file].handleFiles.file.z = pak->handle;
					}

					Q_strncpyz(fsh[*file].name, filename, sizeof(fsh[*file].name));
					fsh[*file].zipFile = qtrue;

					// set the file position in the zip file (also sets the current file info)
					unzSetOffset(fsh[*file].handleFiles.file.z, pakFile->pos);

					// open the file in the zip
					unzOpenCurrentFile(fsh[*file].handleFiles.file.z);
					fsh[*file].zipFilePos = pakFile->pos;

					if (fs_debug->integer)
					{
						Com_Printf("FS_FOpenFileRead: %s (found in '%s')\n",
						           filename, pak->pakFilename);
					}

					return pakFile->len;
				}

				pakFile = pakFile->next;
			}
			while (pakFile != NULL);
		}
	}
	else if (search->dir)
	{
		// check a file in the directory tree

		// if we are running restricted, the only files we
		// will allow to come from the directory are .cfg files
		len = strlen(filename);
		// FIXME TTimo I'm not sure about the fs_numServerPaks test
		// if you are using FS_ReadFile to find out if a file exists,
		//   this test can make the search fail although the file is in the directory
		// I had the problem on https://zerowing.idsoftware.com/bugzilla/show_bug.cgi?id=8
		// turned out I used FS_FileExists instead
		if (!unpure && fs_numServerPaks)
		{
			if (!FS_IsExt(filename, ".cfg", len) &&      // for config files
			    !FS_IsExt(filename, ".menu", len) &&    // menu files
			    !FS_IsExt(filename, ".game", len) &&    // menu files
			    !FS_IsExt(filename, ".dat", len) &&     // for journal files
			    !FS_IsDemoExt(filename, len)           // demos
#ifdef FEATURE_PUREHACK
				// SERVER
				&& Q_stricmp(filename, Sys_GetDLLName("qagame")) 
				&& Q_stricmp(filename, Sys_GetDLLName("cgame"))
				&& Q_stricmp(filename, Sys_GetDLLName("ui"))
#endif
				)
			{
				*file = 0;
				return -1;
			}
		}

		dir = search->dir;

		netpath = FS_BuildOSPath(dir->path, dir->gamedir, filename);
		filep   = Sys_FOpen(netpath, "rb");

		if (filep == NULL)
		{
			*file = 0;
			return -1;
		}

		Q_strncpyz(fsh[*file].name, filename, sizeof(fsh[*file].name));
		fsh[*file].zipFile = qfalse;

		if (fs_debug->integer)
		{
			Com_Printf("FS_FOpenFileRead: %s (found in '%s/%s')\n", filename,
			           dir->path, dir->gamedir);
		}

		fsh[*file].handleFiles.file.o = filep;
		return FS_fplength(filep);
	}

	return -1;
}

/**
 * @brief Finds the file in the search path.
 * @returns filesize and an open FILE pointer.
 *
 * Used for streaming data out of either a separate file or a ZIP file.
 */
long FS_FOpenFileRead(const char *filename, fileHandle_t *file, qboolean uniqueFILE)
{
	searchpath_t *search;
	long         len;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FOpenFileRead: Filesystem call made without initialization");
	}

	for (search = fs_searchpaths; search; search = search->next)
	{
		if (search->pack && (fs_filter_flag & FS_EXCLUDE_PK3))
		{
			continue;
		}
		if (search->dir && (fs_filter_flag & FS_EXCLUDE_DIR))
		{
			continue;
		}
		len = FS_FOpenFileReadDir(filename, search, file, uniqueFILE, qfalse);

		if (file == NULL)
		{
			if (len > 0)
			{
				return len;
			}
		}
		else
		{
			if (len >= 0 && *file)
			{
				return len;
			}
		}
	}

#ifdef FS_MISSING
	if (missingFiles)
	{
		fprintf(missingFiles, "%s\n", filename);
	}
#endif

	if (file)
	{
		Com_DPrintf(S_COLOR_RED "ERROR: Can't find %s\n", filename);

		*file = 0;
		return -1;
	}
	else
	{
		// When file is NULL, we're querying the existance of the file
		// If we've got here, it doesn't exist
		return 0;
	}
}

long FS_FOpenFileRead_Filtered(const char *qpath, fileHandle_t *file, qboolean uniqueFILE, int filter_flag)
{
	long ret;

	fs_filter_flag = filter_flag;
	ret            = FS_FOpenFileRead(qpath, file, uniqueFILE);
	fs_filter_flag = 0;

	return ret;
}

// relevant to client only
#if !defined(DEDICATED)
/*
==================
FS_CL_ExtractFromPakFile

Extracts the latest file from a pak file.

Compares packed file against extracted file. If no differences, does not copy.
This is necessary for exe/dlls which may or may not be locked.

NOTE:
  fullpath gives the full OS path to the dll that will potentially be loaded
    on win32 it's always in fs_basepath/<fs_game>/
    on linux it can be in fs_homepath/<fs_game>/ or fs_basepath/<fs_game>/
  the dll is extracted to fs_homepath (== fs_basepath on win32) if needed

  the return value doesn't tell wether file was extracted or not, it just says wether it's ok to continue
  (i.e. either the right file was extracted successfully, or it was already present)

  cvar_lastVersion is the optional name of a CVAR_ARCHIVE used to store the wolf version for the last extracted .so
==================
*/
qboolean FS_CL_ExtractFromPakFile(const char *base, const char *gamedir, const char *filename)
{
	int           srcLength;
	unsigned char *srcData;
	unsigned char *destData;
	qboolean      needToCopy = qtrue;
	FILE          *destHandle;
	char          *fn;

	fn = FS_BuildOSPath(base, gamedir, filename);

	// read in compressed file
	srcLength = FS_ReadFile(filename, (void **)&srcData);

	// if its not in the pak, we bail
	if (srcLength == -1)
	{
		return qfalse;
	}

	// read in local file
	destHandle = Sys_FOpen(fn, "rb");

	// if we have a local file, we need to compare the two
	if (destHandle)
	{
		int destLength;

		fseek(destHandle, 0, SEEK_END);
		destLength = ftell(destHandle);
		fseek(destHandle, 0, SEEK_SET);

		if (destLength > 0)
		{
			destData = (unsigned char *)Z_Malloc(destLength);

			fread(destData, destLength, 1, destHandle);

			// compare files
			if (destLength == srcLength)
			{
				int i;

				for (i = 0; i < destLength; i++)
				{
					if (destData[i] != srcData[i])
					{
						break;
					}
				}

				if (i == destLength)
				{
					needToCopy = qfalse;
				}
			}

			Z_Free(destData);   // TTimo
		}

		fclose(destHandle);
	}

	// write file
	if (needToCopy)
	{
		fileHandle_t f;

		f = FS_FOpenDLLWrite(filename);
		if (!f)
		{
			Com_Printf("Failed to open %s\n", filename);
			return qfalse;
		}

		FS_Write(srcData, srcLength, f);

		FS_FCloseFile(f);
	}

	FS_FreeFile(srcData);
	return qtrue;
}
#endif

/*
==============
FS_AllowDeletion
==============
*/
qboolean FS_AllowDeletion(char *filename)
{
	// for safety, only allow deletion from the save, profiles and demo directory
	if (Q_strncmp(filename, "save/", 5) != 0 &&
	    Q_strncmp(filename, "profiles/", 9) != 0 &&
	    Q_strncmp(filename, "demos/", 6) != 0)
	{
		return qfalse;
	}

	return qtrue;
}

/*
==============
FS_DeleteDir
==============
*/
int FS_DeleteDir(char *dirname, qboolean nonEmpty, qboolean recursive)
{
	char *ospath;
	char **pFiles = NULL;
	int  i, nFiles = 0;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_DeleteDir: Filesystem call made without initialization");
	}

	if (!dirname || dirname[0] == 0)
	{
		return 0;
	}

	if (!FS_AllowDeletion(dirname))
	{
		return 0;
	}

	if (recursive)
	{
		ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, dirname);
		pFiles = Sys_ListFiles(ospath, "/", NULL, &nFiles, qfalse);
		for (i = 0; i < nFiles; i++)
		{
			char temp[MAX_OSPATH];

			if (!Q_stricmp(pFiles[i], "..") || !Q_stricmp(pFiles[i], "."))
			{
				continue;
			}

			Com_sprintf(temp, sizeof(temp), "%s/%s", dirname, pFiles[i]);

			if (!FS_DeleteDir(temp, nonEmpty, recursive))
			{
				return 0;
			}
		}
		Sys_FreeFileList(pFiles);
	}

	if (nonEmpty)
	{
		ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, dirname);
		pFiles = Sys_ListFiles(ospath, NULL, NULL, &nFiles, qfalse);
		for (i = 0; i < nFiles; i++)
		{
			ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, va("%s/%s", dirname, pFiles[i]));

			if (remove(ospath) == -1)        // failure
			{
				return 0;
			}
		}
		Sys_FreeFileList(pFiles);
	}

	ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, dirname);

	if (Q_rmdir(ospath) == 0)
	{
		return 1;
	}

	return 0;
}

/*
==============
FS_OSStatFile
Test an file given OS path:
returns -1 if not found
returns 1 if directory
returns 0 otherwise
==============
*/
#ifdef WIN32
int FS_OSStatFile(char *ospath)
{
	struct _stat stat;

	if (_stat(ospath, &stat) == -1)
	{
		return -1;
	}
	if (stat.st_mode & _S_IFDIR)
	{
		return 1;
	}
	return 0;
}
#else
int FS_OSStatFile(char *ospath)
{
	struct stat stat_buf;

	if (stat(ospath, &stat_buf) == -1)
	{
		return -1;
	}
	if (S_ISDIR(stat_buf.st_mode))
	{
		return 1;
	}
	return 0;
}
#endif

/**
 * @brief Removes file in the current fs_gamedir in homepath
 */
int FS_Delete(char *filename)
{
	char *ospath;
	int  stat;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_Delete: Filesystem call made without initialization");
	}

	if (!filename || filename[0] == 0)
	{
		return 0;
	}

	if (!FS_AllowDeletion(filename))
	{
		return 0;
	}

	ospath = FS_BuildOSPath(fs_homepath->string, fs_gamedir, filename);

	FS_CheckFilenameIsNotExecutable(ospath, __func__);

	stat = FS_OSStatFile(ospath);
	if (stat == -1)
	{
		return 0;
	}

	if (stat == 1)
	{
		return(FS_DeleteDir(filename, qtrue, qtrue));
	}
	else
	{
		if (remove(ospath) != -1)        // success
		{
			return 1;
		}
	}

	return 0;
}

/*
=================
FS_Read

Properly handles partial reads
=================
*/
int FS_Read2(void *buffer, int len, fileHandle_t f)
{
	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_Read2: Filesystem call made without initialization");
	}

	if (!f)
	{
		return 0;
	}

	return FS_Read(buffer, len, f);
}

int FS_Read(void *buffer, int len, fileHandle_t f)
{
	byte *buf;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_Read: Filesystem call made without initialization");
	}

	if (!f)
	{
		return 0;
	}

	buf           = (byte *)buffer;
	fs_readCount += len;

	if (fsh[f].zipFile == qfalse)
	{
		int read, block;
		int remaining = len;
		int tries     = 0;

		while (remaining)
		{
			block = remaining;
			read  = fread(buf, 1, block, fsh[f].handleFiles.file.o);
			if (read == 0)
			{
				// we might have been trying to read from a CD, which
				// sometimes returns a 0 read on windows
				if (!tries)
				{
					tries = 1;
				}
				else
				{
					return len - remaining;   //Com_Error (ERR_FATAL, "FS_Read: 0 bytes read");
				}
			}

			if (read == -1)
			{
				Com_Error(ERR_FATAL, "FS_Read: -1 bytes read");
			}

			remaining -= read;
			buf       += read;
		}
		return len;
	}
	else
	{
		return unzReadCurrentFile(fsh[f].handleFiles.file.z, buffer, len);
	}
}

/*
=================
FS_Write

Properly handles partial writes
=================
*/
int FS_Write(const void *buffer, int len, fileHandle_t h)
{
	int  block, remaining;
	int  written;
	byte *buf;
	int  tries;
	FILE *f;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_Write: Filesystem call made without initialization");
	}

	if (!h)
	{
		return 0;
	}

	f   = FS_FileForHandle(h);
	buf = (byte *)buffer;

	remaining = len;
	tries     = 0;
	while (remaining)
	{
		block   = remaining;
		written = fwrite(buf, 1, block, f);
		if (written == 0)
		{
			if (!tries)
			{
				tries = 1;
			}
			else
			{
				Com_Printf("FS_Write: 0 bytes written (%d attempted)\n", block);
				return 0;
			}
		}

		if (written < 0)
		{
			Com_Printf("FS_Write: %d bytes written (%d attempted)\n", written, block);
			return 0;
		}

		remaining -= written;
		buf       += written;
	}
	if (fsh[h].handleSync)
	{
		fflush(f);
	}
	return len;
}

void QDECL FS_Printf(fileHandle_t h, const char *fmt, ...)
{
	va_list argptr;
	char    msg[MAXPRINTMSG];

	va_start(argptr, fmt);
	Q_vsnprintf(msg, sizeof(msg), fmt, argptr);
	va_end(argptr);

	FS_Write(msg, strlen(msg), h);
}

#define PK3_SEEK_BUFFER_SIZE 65536

/*
=================
FS_Seek
=================
*/
int FS_Seek(fileHandle_t f, long offset, int origin)
{
	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_Seek: Filesystem call made without initialization");
		return -1;
	}

	if (fsh[f].zipFile == qtrue)
	{
		// FIXME: this is incomplete and really, really
		// crappy (but better than what was here before)
		byte buffer[PK3_SEEK_BUFFER_SIZE];
		int  remainder = offset;

		if (offset < 0 || origin == FS_SEEK_END)
		{
			Com_Error(ERR_FATAL, "FS_Seek: Negative offsets and FS_SEEK_END not implemented on pk3 file contents");
			return -1;
		}

		switch (origin)
		{
		case FS_SEEK_SET:
			unzSetOffset(fsh[f].handleFiles.file.z, fsh[f].zipFilePos);
			unzOpenCurrentFile(fsh[f].handleFiles.file.z);
		// fall through
		case FS_SEEK_CUR:
			while (remainder > PK3_SEEK_BUFFER_SIZE)
			{
				FS_Read(buffer, PK3_SEEK_BUFFER_SIZE, f);
				remainder -= PK3_SEEK_BUFFER_SIZE;
			}
			FS_Read(buffer, remainder, f);
			return offset;
		default:
			Com_Error(ERR_FATAL, "FS_Seek: Bad zip origin");
			return -1;
		}
	}
	else
	{
		int  _origin;
		FILE *file;

		file = FS_FileForHandle(f);

		switch (origin)
		{
		case FS_SEEK_CUR:
			_origin = SEEK_CUR;
			break;
		case FS_SEEK_END:
			_origin = SEEK_END;
			break;
		case FS_SEEK_SET:
			_origin = SEEK_SET;
			break;
		default:
			_origin = SEEK_CUR;
			Com_Error(ERR_FATAL, "FS_Seek: Bad origin");
			break;
		}

		return fseek(file, offset, _origin);
	}
}

/*
======================================================================================
CONVENIENCE FUNCTIONS FOR ENTIRE FILES
======================================================================================
*/

int FS_FileIsInPAK(const char *filename, int *pChecksum)
{
	searchpath_t *search;
	pack_t       *pak;
	fileInPack_t *pakFile;
	long         hash = 0;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FileIsInPAK: Filesystem call made without initialization");
	}

	if (!filename)
	{
		Com_Error(ERR_FATAL, "FS_FileIsInPAK: NULL 'filename' parameter passed");
	}

	// qpaths are not supposed to have a leading slash
	if (filename[0] == '/' || filename[0] == '\\')
	{
		filename++;
	}

	// make absolutely sure that it can't back up the path.
	// The searchpaths do guarantee that something will always
	// be prepended, so we don't need to worry about "c:" or "//limbo"
	if (strstr(filename, "..") || strstr(filename, "::"))
	{
		return -1;
	}

	// search through the path, one element at a time
	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (search->pack)
		{
			hash = FS_HashFileName(filename, search->pack->hashSize);
		}
		// is the element a pak file?
		if (search->pack && search->pack->hashTable[hash])
		{
			// disregard if it doesn't match one of the allowed pure pak files
			if (!FS_PakIsPure(search->pack))
			{
				continue;
			}

			// look through all the pak file elements
			pak     = search->pack;
			pakFile = pak->hashTable[hash];
			do
			{
				// case and separator insensitive comparisons
				if (!FS_FilenameCompare(pakFile->name, filename))
				{
					if (pChecksum)
					{
						*pChecksum = pak->pure_checksum;
					}
					return 1;
				}
				pakFile = pakFile->next;
			}
			while (pakFile != NULL);
		}
	}
	return -1;
}

/**
 * @brief Open a file relative to the ET:L search path.
 * A null buffer will just return the file length without loading.
 */
int FS_ReadFile(const char *qpath, void **buffer)
{
	fileHandle_t h;
	byte         *buf;
	qboolean     isConfig;
	int          len;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_ReadFile: Filesystem call made without initialization");
	}

	if (!qpath || !qpath[0])
	{
		Com_Error(ERR_FATAL, "FS_ReadFile: empty name");
	}

	buf = NULL; // quiet compiler warning

	// if this is a .cfg file and we are playing back a journal, read
	// it from the journal file
	if (strstr(qpath, ".cfg"))
	{
		isConfig = qtrue;
		if (com_journal && com_journal->integer == 2)
		{
			int r;

			Com_DPrintf("Loading %s from journal file.\n", qpath);
			r = FS_Read(&len, sizeof(len), com_journalDataFile);
			if (r != sizeof(len))
			{
				if (buffer != NULL)
				{
					*buffer = NULL;
				}
				return -1;
			}
			// if the file didn't exist when the journal was created
			if (!len)
			{
				if (buffer == NULL)
				{
					return 1;           // hack for old journal files
				}
				*buffer = NULL;
				return -1;
			}
			if (buffer == NULL)
			{
				return len;
			}

			buf     = Hunk_AllocateTempMemory(len + 1);
			*buffer = buf;

			r = FS_Read(buf, len, com_journalDataFile);
			if (r != len)
			{
				Com_Error(ERR_FATAL, "FS_ReadFile: Read from journalDataFile failed");
			}

			fs_loadCount++;
			fs_loadStack++;

			// guarantee that it will have a trailing 0 for string operations
			buf[len] = 0;

			return len;
		}
	}
	else
	{
		isConfig = qfalse;
	}

	// look for it in the filesystem or pack files
	len = FS_FOpenFileRead(qpath, &h, qfalse);
	if (h == 0)
	{
		if (buffer)
		{
			*buffer = NULL;
		}
		// if we are journalling and it is a config file, write a zero to the journal file
		if (isConfig && com_journal && com_journal->integer == 1)
		{
			Com_DPrintf("Writing zero for %s to journal file.\n", qpath);
			len = 0;
			FS_Write(&len, sizeof(len), com_journalDataFile);
			FS_Flush(com_journalDataFile);
		}
		return -1;
	}

	if (!buffer)
	{
		if (isConfig && com_journal && com_journal->integer == 1)
		{
			Com_DPrintf("Writing len for %s to journal file.\n", qpath);
			FS_Write(&len, sizeof(len), com_journalDataFile);
			FS_Flush(com_journalDataFile);
		}
		FS_FCloseFile(h);
		return len;
	}

	fs_loadCount++;
	fs_loadStack++;

	buf     = Hunk_AllocateTempMemory(len + 1);
	*buffer = buf;

	FS_Read(buf, len, h);

	// guarantee that it will have a trailing 0 for string operations
	buf[len] = 0;
	FS_FCloseFile(h);

	// if we are journalling and it is a config file, write it to the journal file
	if (isConfig && com_journal && com_journal->integer == 1)
	{
		Com_DPrintf("Writing %s to journal file.\n", qpath);
		FS_Write(&len, sizeof(len), com_journalDataFile);
		FS_Write(buf, len, com_journalDataFile);
		FS_Flush(com_journalDataFile);
	}
	return len;
}

/*
=============
FS_FreeFile
=============
*/
void FS_FreeFile(void *buffer)
{
	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FreeFile: Filesystem call made without initialization");
	}
	if (!buffer)
	{
		Com_Error(ERR_FATAL, "FS_FreeFile: NULL parameter");
	}
	fs_loadStack--;

	Hunk_FreeTempMemory(buffer);

	// if all of our temp files are free, clear all of our space
	if (fs_loadStack == 0)
	{
		Hunk_ClearTempMemory();
	}
}

/*
============
FS_WriteFile

Filename are reletive to the quake search path
============
*/
void FS_WriteFile(const char *qpath, const void *buffer, int size)
{
	fileHandle_t f;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_WriteFile: Filesystem call made without initialization");
	}

	if (!qpath || !buffer)
	{
		Com_Error(ERR_FATAL, "FS_WriteFile: NULL parameter");
	}

	f = FS_FOpenFileWrite(qpath);
	if (!f)
	{
		Com_Printf("Failed to open %s\n", qpath);
		return;
	}

	FS_Write(buffer, size, f);

	FS_FCloseFile(f);
}

/*
==========================================================================
ZIP FILE LOADING
==========================================================================
*/

/*
=================
FS_LoadZipFile

Creates a new pak_t in the search chain for the contents
of a zip file.
=================
*/
static pack_t *FS_LoadZipFile(const char *zipfile, const char *basename)
{
	fileInPack_t    *buildBuffer;
	pack_t          *pack;
	unzFile         uf;
	int             err;
	unz_global_info gi;
	char            filename_inzip[MAX_ZPATH];
	unz_file_info   file_info;
	unsigned int    i, len;
	long            hash;
	int             fs_numHeaderLongs = 0;
	int             *fs_headerLongs;
	char            *namePtr;

	uf  = unzOpen(zipfile);
	err = unzGetGlobalInfo(uf, &gi);

	if (err != UNZ_OK)
	{
		return NULL;
	}

	len = 0;
	unzGoToFirstFile(uf);
	for (i = 0; i < gi.number_entry; i++)
	{
		err = unzGetCurrentFileInfo(uf, &file_info, filename_inzip, sizeof(filename_inzip), NULL, 0, NULL, 0);
		if (err != UNZ_OK)
		{
			break;
		}
		len += strlen(filename_inzip) + 1;
		unzGoToNextFile(uf);
	}

	buildBuffer                         = Z_Malloc((gi.number_entry * sizeof(fileInPack_t)) + len);
	namePtr                             = ((char *) buildBuffer) + gi.number_entry * sizeof(fileInPack_t);
	fs_headerLongs                      = Z_Malloc((gi.number_entry + 1) * sizeof(int));
	fs_headerLongs[fs_numHeaderLongs++] = LittleLong(fs_checksumFeed);

	// get the hash table size from the number of files in the zip
	// because lots of custom pk3 files have less than 32 or 64 files
	for (i = 1; i <= MAX_FILEHASH_SIZE; i <<= 1)
	{
		if (i > gi.number_entry)
		{
			break;
		}
	}

	pack            = Z_Malloc(sizeof(pack_t) + i * sizeof(fileInPack_t *));
	pack->hashSize  = i;
	pack->hashTable = ( fileInPack_t ** )(((char *) pack) + sizeof(pack_t));
	for (i = 0; i < pack->hashSize; i++)
	{
		pack->hashTable[i] = NULL;
	}

	Q_strncpyz(pack->pakFilename, zipfile, sizeof(pack->pakFilename));
	Q_strncpyz(pack->pakBasename, basename, sizeof(pack->pakBasename));

	// strip .pk3 if needed
	if (strlen(pack->pakBasename) > 4 && !Q_stricmp(pack->pakBasename + strlen(pack->pakBasename) - 4, ".pk3"))
	{
		pack->pakBasename[strlen(pack->pakBasename) - 4] = 0;
	}

	pack->handle   = uf;
	pack->numfiles = gi.number_entry;
	unzGoToFirstFile(uf);

	for (i = 0; i < gi.number_entry; i++)
	{
		err = unzGetCurrentFileInfo(uf, &file_info, filename_inzip, sizeof(filename_inzip), NULL, 0, NULL, 0);
		if (err != UNZ_OK)
		{
			break;
		}
		if (file_info.uncompressed_size > 0)
		{
			fs_headerLongs[fs_numHeaderLongs++] = LittleLong(file_info.crc);
		}
		Q_strlwr(filename_inzip);
		hash                = FS_HashFileName(filename_inzip, pack->hashSize);
		buildBuffer[i].name = namePtr;
		strcpy(buildBuffer[i].name, filename_inzip);
		namePtr += strlen(filename_inzip) + 1;
		// store the file position in the zip
		buildBuffer[i].pos    = unzGetOffset(uf);
		buildBuffer[i].len    = file_info.uncompressed_size;
		buildBuffer[i].next   = pack->hashTable[hash];
		pack->hashTable[hash] = &buildBuffer[i];
		unzGoToNextFile(uf);
	}

	pack->checksum      = Com_BlockChecksum(&fs_headerLongs[1], sizeof(*fs_headerLongs) * (fs_numHeaderLongs - 1));
	pack->pure_checksum = Com_BlockChecksum(fs_headerLongs, sizeof(*fs_headerLongs) * fs_numHeaderLongs);
	pack->checksum      = LittleLong(pack->checksum);
	pack->pure_checksum = LittleLong(pack->pure_checksum);

	Z_Free(fs_headerLongs);

	pack->buildBuffer = buildBuffer;
	return pack;
}

/*
=================
FS_FreePak

Frees a pak structure and releases all associated resources
=================
*/

static void FS_FreePak(pack_t *thepak)
{
	unzClose(thepak->handle);
	Z_Free(thepak->buildBuffer);
	Z_Free(thepak);
}

/*
=================
FS_GetZipChecksum

Compares whether the given pak file matches a referenced checksum
=================
*/
qboolean FS_CompareZipChecksum(const char *zipfile)
{
	pack_t *thepak;
	int    index, checksum;

	thepak = FS_LoadZipFile(zipfile, "");

	if (!thepak)
	{
		return qfalse;
	}

	checksum = thepak->checksum;
	FS_FreePak(thepak);

	for (index = 0; index < fs_numServerReferencedPaks; index++)
	{
		if (checksum == fs_serverReferencedPaks[index])
		{
			return qtrue;
		}
	}

	return qfalse;
}

/*
=================================================================================
DIRECTORY SCANNING FUNCTIONS
=================================================================================
*/

static int FS_ReturnPath(const char *zname, char *zpath, int *depth)
{
	int len = 0, at = 0, newdep = 0;

	zpath[0] = 0;

	while (zname[at] != 0)
	{
		if (zname[at] == '/' || zname[at] == '\\')
		{
			len = at;
			newdep++;
		}
		at++;
	}
	strcpy(zpath, zname);
	zpath[len] = 0;
	*depth     = newdep;

	return len;
}

/*
==================
FS_AddFileToList
==================
*/
static int FS_AddFileToList(char *name, char *list[MAX_FOUND_FILES], int nfiles)
{
	int i;

	if (nfiles == MAX_FOUND_FILES - 1)
	{
		return nfiles;
	}
	for (i = 0 ; i < nfiles ; i++)
	{
		if (!Q_stricmp(name, list[i]))
		{
			return nfiles;      // allready in list
		}
	}
	list[nfiles] = CopyString(name);
	nfiles++;

	return nfiles;
}

/*
===============
FS_ListFilteredFiles

Returns a uniqued list of files that match the given criteria
from all search paths
===============
*/
char **FS_ListFilteredFiles(const char *path, const char *extension, char *filter, int *numfiles, qboolean allowNonPureFilesOnDisk)
{
	int          nfiles;
	char         **listCopy;
	char         *list[MAX_FOUND_FILES];
	searchpath_t *search;
	int          i;
	int          pathLength;
	int          extensionLength;
	int          length, pathDepth, temp;
	pack_t       *pak;
	fileInPack_t *buildBuffer;
	char         zpath[MAX_ZPATH];
	char         *name;
	int          zpathLen, depth;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_ListFilteredFiles: Filesystem call made without initialization");
	}

	if (!path)
	{
		*numfiles = 0;
		return NULL;
	}
	if (!extension)
	{
		extension = "";
	}

	pathLength = strlen(path);
	if (path[pathLength - 1] == '\\' || path[pathLength - 1] == '/')
	{
		pathLength--;
	}
	extensionLength = strlen(extension);
	nfiles          = 0;
	FS_ReturnPath(path, zpath, &pathDepth);

	// search through the path, one element at a time, adding to list
	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (search->pack)
		{
			// If we are pure, don't search for files on paks that
			// aren't on the pure list
			if (!FS_PakIsPure(search->pack))
			{
				continue;
			}

			// look through all the pak file elements
			pak         = search->pack;
			buildBuffer = pak->buildBuffer;
			for (i = 0; i < pak->numfiles; i++)
			{
				// check for directory match
				name = buildBuffer[i].name;

				if (filter)
				{
					// case insensitive
					if (!Com_FilterPath(filter, name, qfalse))
					{
						continue;
					}
					// unique the match
					nfiles = FS_AddFileToList(name, list, nfiles);
				}
				else
				{
					zpathLen = FS_ReturnPath(name, zpath, &depth);

					if ((depth - pathDepth) > 2 || pathLength > zpathLen || Q_stricmpn(name, path, pathLength))
					{
						continue;
					}

					// check for extension match
					length = strlen(name);
					if (length < extensionLength)
					{
						continue;
					}

					if (Q_stricmp(name + length - extensionLength, extension))
					{
						continue;
					}
					// unique the match

					temp = pathLength;
					if (pathLength)
					{
						temp++;     // include the '/'
					}
					nfiles = FS_AddFileToList(name + temp, list, nfiles);
				}
			}
		}
		else if (search->dir)     // scan for files in the filesystem
		{
			// don't scan directories for files if we are pure or restricted
			if (fs_numServerPaks && !allowNonPureFilesOnDisk)
			{
				continue;
			}
			else
			{
				char *netpath;
				int  numSysFiles;
				char **sysFiles;
				char *name;

				netpath  = FS_BuildOSPath(search->dir->path, search->dir->gamedir, path);
				sysFiles = Sys_ListFiles(netpath, extension, filter, &numSysFiles, qfalse);
				for (i = 0 ; i < numSysFiles ; i++)
				{
					// unique the match
					name   = sysFiles[i];
					nfiles = FS_AddFileToList(name, list, nfiles);
				}
				Sys_FreeFileList(sysFiles);
			}
		}
	}

	// return a copy of the list
	*numfiles = nfiles;

	if (!nfiles)
	{
		return NULL;
	}

	listCopy = Z_Malloc((nfiles + 1) * sizeof(*listCopy));
	for (i = 0 ; i < nfiles ; i++)
	{
		listCopy[i] = list[i];
	}
	listCopy[i] = NULL;

	return listCopy;
}

/*
=================
FS_ListFiles
=================
*/
char **FS_ListFiles(const char *path, const char *extension, int *numfiles)
{
	return FS_ListFilteredFiles(path, extension, NULL, numfiles, qfalse);
}

/*
=================
FS_FreeFileList
=================
*/
void FS_FreeFileList(char **list)
{
	int i;

	if (!fs_searchpaths)
	{
		Com_Error(ERR_FATAL, "FS_FreeFileList: Filesystem call made without initialization");
	}

	if (!list)
	{
		return;
	}

	for (i = 0 ; list[i] ; i++)
	{
		Z_Free(list[i]);
	}

	Z_Free(list);
}

/*
================
FS_GetFileList
================
*/
int FS_GetFileList(const char *path, const char *extension, char *listbuf, int bufsize)
{
	int  nFiles   = 0, i, nTotal = 0, nLen;
	char **pFiles = NULL;

	*listbuf = 0;

	if (Q_stricmp(path, "$modlist") == 0)
	{
		return FS_GetModList(listbuf, bufsize);
	}

	pFiles = FS_ListFiles(path, extension, &nFiles);

	for (i = 0; i < nFiles; i++)
	{
		nLen = strlen(pFiles[i]) + 1;
		if (nTotal + nLen + 1 < bufsize)
		{
			strcpy(listbuf, pFiles[i]);
			listbuf += nLen;
			nTotal  += nLen;
		}
		else
		{
			nFiles = i;
			break;
		}
	}

	FS_FreeFileList(pFiles);

	return nFiles;
}

/*
=======================
Sys_ConcatenateFileLists

mkv: Naive implementation. Concatenates three lists into a
     new list, and frees the old lists from the heap.
bk001129 - from cvs1.17 (mkv)

FIXME those two should move to common.c next to Sys_ListFiles
=======================
 */
static unsigned int Sys_CountFileList(char **list)
{
	int i = 0;

	if (list)
	{
		while (*list)
		{
			list++;
			i++;
		}
	}
	return i;
}

static char **Sys_ConcatenateFileLists(char **list0, char **list1, char **list2)
{
	int  totalLength = 0;
	char **cat       = NULL, **dst, **src;

	totalLength += Sys_CountFileList(list0);
	totalLength += Sys_CountFileList(list1);
	totalLength += Sys_CountFileList(list2);

	/* Create new list. */
	dst = cat = Z_Malloc((totalLength + 1) * sizeof(char *));

	/* Copy over lists. */
	if (list0)
	{
		for (src = list0; *src; src++, dst++)
			*dst = *src;
	}
	if (list1)
	{
		for (src = list1; *src; src++, dst++)
			*dst = *src;
	}
	if (list2)
	{
		for (src = list2; *src; src++, dst++)
			*dst = *src;
	}

	// Terminate the list
	*dst = NULL;

	// Free our old lists.
	// NOTE: not freeing their content, it's been merged in dst and still being used
	if (list0)
	{
		Z_Free(list0);
	}
	if (list1)
	{
		Z_Free(list1);
	}
	if (list2)
	{
		Z_Free(list2);
	}

	return cat;
}

/*
================
FS_GetModList

Returns a list of mod directory names
A mod directory is a peer to etmain with a pk3 in it
The directories are searched in base path and home path
================
*/
int FS_GetModList(char *listbuf, int bufsize)
{
	int          nMods    = 0, i, j, nTotal = 0, nLen, nPaks, nPotential, nDescLen;
	char         **pFiles = NULL;
	char         **pPaks  = NULL;
	char         *name, *path;
	char         descPath[MAX_OSPATH];
	fileHandle_t descHandle;
	int          dummy;
	char         **pFiles0 = NULL;
	char         **pFiles1 = NULL;
	char         **pFiles2 = NULL;
	qboolean     bDrop     = qfalse;

	*listbuf = 0;

	pFiles0 = Sys_ListFiles(fs_homepath->string, NULL, NULL, &dummy, qtrue);
	pFiles1 = Sys_ListFiles(fs_basepath->string, NULL, NULL, &dummy, qtrue);

	// we searched for mods in the three paths
	// it is likely that we have duplicate names now, which we will cleanup below
	pFiles     = Sys_ConcatenateFileLists(pFiles0, pFiles1, pFiles2);
	nPotential = Sys_CountFileList(pFiles);

	for (i = 0 ; i < nPotential ; i++)
	{
		name = pFiles[i];
		// NOTE: cleaner would involve more changes
		// ignore duplicate mod directories
		if (i != 0)
		{
			bDrop = qfalse;
			for (j = 0; j < i; j++)
			{
				if (Q_stricmp(pFiles[j], name) == 0)
				{
					// this one can be dropped
					bDrop = qtrue;
					break;
				}
			}
		}
		if (bDrop)
		{
			continue;
		}

		// we drop "etmain" "." and ".."
		if (Q_stricmp(name, BASEGAME) && Q_stricmpn(name, ".", 1))
		{
			// now we need to find some .pk3 files to validate the mod
			// NOTE: (actually I'm not sure why .. what if it's a mod under developement with no .pk3?)
			// we didn't keep the information when we merged the directory names, as to what OS Path it was found under
			//   so it could be in base path, cd path or home path
			//   we will try each three of them here (yes, it's a bit messy)
			// NOTE: what about dropping the current loaded mod as well?
			path  = FS_BuildOSPath(fs_basepath->string, name, "");
			nPaks = 0;
			pPaks = Sys_ListFiles(path, ".pk3", NULL, &nPaks, qfalse);
			Sys_FreeFileList(pPaks);   // we only use Sys_ListFiles to check wether .pk3 files are present

			/* try on home path */
			if (nPaks <= 0)
			{
				path  = FS_BuildOSPath(fs_homepath->string, name, "");
				nPaks = 0;
				pPaks = Sys_ListFiles(path, ".pk3", NULL, &nPaks, qfalse);
				Sys_FreeFileList(pPaks);
			}

			if (nPaks > 0)
			{
				nLen = strlen(name) + 1;
				// nLen is the length of the mod path
				// we need to see if there is a description available
				descPath[0] = '\0';
				strcpy(descPath, name);
				strcat(descPath, "/description.txt");
				nDescLen = FS_SV_FOpenFileRead(descPath, &descHandle);
				if (nDescLen > 0 && descHandle)
				{
					FILE *file;

					file = FS_FileForHandle(descHandle);
					Com_Memset(descPath, 0, sizeof(descPath));
					nDescLen = fread(descPath, 1, 48, file);
					if (nDescLen >= 0)
					{
						descPath[nDescLen] = '\0';
					}
					FS_FCloseFile(descHandle);
				}
				else
				{
					strcpy(descPath, name);
				}
				nDescLen = strlen(descPath) + 1;

				if (nTotal + nLen + 1 + nDescLen + 1 < bufsize)
				{
					strcpy(listbuf, name);
					listbuf += nLen;
					strcpy(listbuf, descPath);
					listbuf += nDescLen;
					nTotal  += nLen + nDescLen;
					nMods++;
				}
				else
				{
					break;
				}
			}
		}
	}
	Sys_FreeFileList(pFiles);

	return nMods;
}

//============================================================================

/*
================
FS_Dir_f
================
*/
void FS_Dir_f(void)
{
	char *path;
	char *extension;
	char **dirnames;
	int  ndirs;
	int  i;

	if (Cmd_Argc() < 2 || Cmd_Argc() > 3)
	{
		Com_Printf("usage: dir <directory> [extension]\n");
		return;
	}

	if (Cmd_Argc() == 2)
	{
		path      = Cmd_Argv(1);
		extension = "";
	}
	else
	{
		path      = Cmd_Argv(1);
		extension = Cmd_Argv(2);
	}

	Com_Printf("Directory of %s %s\n", path, extension);
	Com_Printf("---------------\n");

	dirnames = FS_ListFiles(path, extension, &ndirs);

	for (i = 0; i < ndirs; i++)
	{
		Com_Printf("%s\n", dirnames[i]);
	}
	FS_FreeFileList(dirnames);
}

/*
===========
FS_ConvertPath
===========
*/
void FS_ConvertPath(char *s)
{
	while (*s)
	{
		if (*s == '\\' || *s == ':')
		{
			*s = '/';
		}
		s++;
	}
}

/*
===========
FS_PathCmp

Ignore case and separator char distinctions

FS_PathCmp doesn't deal with trailing slashes! See FS_IsSamePath
/aa/bb - /aa/bb/ returns not equal
===========
*/
int FS_PathCmp(const char *s1, const char *s2)
{
	int c1, c2;

	do
	{
		c1 = *s1++;
		c2 = *s2++;

		if (c1 >= 'a' && c1 <= 'z')
		{
			c1 -= ('a' - 'A');
		}
		if (c2 >= 'a' && c2 <= 'z')
		{
			c2 -= ('a' - 'A');
		}

		if (c1 == '\\' || c1 == ':')
		{
			c1 = '/';
		}
		if (c2 == '\\' || c2 == ':')
		{
			c2 = '/';
		}

		if (c1 < c2)
		{
			return -1;      // strings not equal
		}
		if (c1 > c2)
		{
			return 1;
		}
	}
	while (c1);

	return 0;       // strings are equal
}

/**
 * FS_IsSamePath
 *
 * @brief Similar to FS_PathCmp but based on the real path name comparison.
 *        It deals with trailing slashes and relative pathes.
 *
 * Important: Don't use this with quake game pathes - only real filesystem pathes!
 *
 * @param[in] path A
 * @param[in] path B
 */
qboolean FS_IsSamePath(const char *s1, const char *s2)
{
	char *res1, *res2;

	res1 = realpath(s1, NULL);
	res2 = realpath(s2, NULL);

	// realpath() returns NULL if there are issues with the file
	// so the function returns true (only) if there are no errors and pathes are equal
	if (res1 && res2 && !Q_stricmp(res1, res2))
	{
		free(res1);
		free(res2);
		return qtrue;
	}

	free(res1);
	free(res2);
	return qfalse;
}

/*
================
FS_SortFileList
================
*/
void FS_SortFileList(char **filelist, int numfiles)
{
	int  i, j, k, numsortedfiles;
	char **sortedlist;

	sortedlist     = Z_Malloc((numfiles + 1) * sizeof(*sortedlist));
	sortedlist[0]  = NULL;
	numsortedfiles = 0;
	for (i = 0; i < numfiles; i++)
	{
		for (j = 0; j < numsortedfiles; j++)
		{
			if (FS_PathCmp(filelist[i], sortedlist[j]) < 0)
			{
				break;
			}
		}
		for (k = numsortedfiles; k > j; k--)
		{
			sortedlist[k] = sortedlist[k - 1];
		}
		sortedlist[j] = filelist[i];
		numsortedfiles++;
	}
	Com_Memcpy(filelist, sortedlist, numfiles * sizeof(*filelist));
	Z_Free(sortedlist);
}

/*
================
FS_NewDir_f
================
*/
void FS_NewDir_f(void)
{
	char *filter;
	char **dirnames;
	int  ndirs;
	int  i;

	if (Cmd_Argc() < 2)
	{
		Com_Printf("usage: fdir <filter>\n");
		Com_Printf("example: fdir *q3dm*.bsp\n");
		return;
	}

	filter = Cmd_Argv(1);

	Com_Printf("---------------\n");

	dirnames = FS_ListFilteredFiles("", "", filter, &ndirs, qfalse);

	FS_SortFileList(dirnames, ndirs);

	for (i = 0; i < ndirs; i++)
	{
		FS_ConvertPath(dirnames[i]);
		Com_Printf("%s\n", dirnames[i]);
	}
	Com_Printf("%d files listed\n", ndirs);
	FS_FreeFileList(dirnames);
}

/*
============
FS_Path_f
============
*/
void FS_Path_f(void)
{
	searchpath_t *s;
	int          i;

	Com_Printf("Current search path:\n");
	for (s = fs_searchpaths; s; s = s->next)
	{
		if (s->pack)
		{
			//Com_Printf( "%s %X (%i files)\n", s->pack->pakFilename, s->pack->checksum, s->pack->numfiles );
			Com_Printf("    %s (%i files)\n", s->pack->pakFilename, s->pack->numfiles);
			if (fs_numServerPaks)
			{
				if (!FS_PakIsPure(s->pack))
				{
					Com_Printf("        not on the pure list\n");
				}
				else
				{
					Com_Printf("        on the pure list\n");
				}
			}
		}
		else
		{
			Com_Printf("    %s%c%s\n", s->dir->path, PATH_SEP, s->dir->gamedir);
		}
	}

	Com_Printf("\n");
	for (i = 1 ; i < MAX_FILE_HANDLES ; i++)
	{
		if (fsh[i].handleFiles.file.o)
		{
			Com_Printf("handle %i: %s\n", i, fsh[i].name);
		}
	}
}

/**
 * @brief Simulates the 'touch' unix command
 */
void FS_TouchFile_f(void)
{
	fileHandle_t f;

	if (Cmd_Argc() != 2)
	{
		Com_Printf("Usage: touchFile <file>\n");
		return;
	}

	FS_FOpenFileRead(Cmd_Argv(1), &f, qfalse);
	if (f)
	{
		FS_FCloseFile(f);
	}
}

/*
============
FS_Which
============
*/

qboolean FS_Which(const char *filename, void *searchPath)
{
	searchpath_t *search = (searchpath_t *)searchPath;

	if (FS_FOpenFileReadDir(filename, search, NULL, qfalse, qfalse) > 0)
	{
		if (search->pack)
		{
			Com_Printf("File \"%s\" found in \"%s\"\n", filename, search->pack->pakFilename);
			return qtrue;
		}
		else if (search->dir)
		{
			Com_Printf("File \"%s\" found at \"%s\"\n", filename, search->dir->fullpath);
			return qtrue;
		}
	}

	return qfalse;
}

/*
============
FS_Which_f
============
*/
void FS_Which_f(void)
{
	searchpath_t *search;
	char         *filename = Cmd_Argv(1);

	if (!filename[0])
	{
		Com_Printf("Usage: which <file>\n");
		return;
	}

	// qpaths are not supposed to have a leading slash
	if (filename[0] == '/' || filename[0] == '\\')
	{
		filename++;
	}

	// just wants to see if file is there
	for (search = fs_searchpaths; search; search = search->next)
	{
		if (FS_Which(filename, search))
		{
			return;
		}
	}

	Com_Printf("File not found: \"%s\"\n", filename);
}

//===========================================================================

static int QDECL paksort(const void *a, const void *b)
{
	char *aa, *bb;

	aa = *(char **)a;
	bb = *(char **)b;

	return FS_PathCmp(aa, bb);
}

/*
================
FS_AddGameDirectory

Sets fs_gamedir, adds the directory to the head of the path,
then loads the zip headers
================
*/
void FS_AddGameDirectory(const char *path, const char *dir)
{
	searchpath_t *sp;
	searchpath_t *search;
	pack_t       *pak;
	char         curpath[MAX_OSPATH + 1], *pakfile;
	int          numfiles;
	char         **pakfiles;
	int          pakfilesi;
	char         **pakfilestmp;
	int          numdirs;
	char         **pakdirs;
	int          pakdirsi;
	char         **pakdirstmp;
	int          pakwhich;
	int          len;

	// Unique
	for (sp = fs_searchpaths ; sp ; sp = sp->next)
	{
		if (sp->dir && !Q_stricmp(sp->dir->path, path) && !Q_stricmp(sp->dir->gamedir, dir))
		{
			return;         // we've already got this one
		}
	}

	Q_strncpyz(fs_gamedir, dir, sizeof(fs_gamedir));

	// find all pak files in this directory
	Q_strncpyz(curpath, FS_BuildOSPath(path, dir, ""), sizeof(curpath));
	curpath[strlen(curpath) - 1] = '\0';    // strip the trailing slash

	// Get .pk3 files
	pakfiles = Sys_ListFiles(curpath, ".pk3", NULL, &numfiles, qfalse);

	qsort(pakfiles, numfiles, sizeof(char *), paksort);

	if (fs_numServerPaks)
	{
		numdirs = 0;
		pakdirs = NULL;
	}
	else
	{
		// Get top level directories (we'll filter them later since the Sys_ListFiles filtering is terrible)
		pakdirs = Sys_ListFiles(curpath, "/", NULL, &numdirs, qfalse);

		qsort(pakdirs, numdirs, sizeof(char *), paksort);
	}

	pakfilesi = 0;
	pakdirsi  = 0;

	while ((pakfilesi < numfiles) || (pakdirsi < numdirs))
	{
		// Check if a pakfile or pakdir comes next
		if (pakfilesi >= numfiles)
		{
			// We've used all the pakfiles, it must be a pakdir.
			pakwhich = 0;
		}
		else if (pakdirsi >= numdirs)
		{
			// We've used all the pakdirs, it must be a pakfile.
			pakwhich = 1;
		}
		else
		{
			// Could be either, compare to see which name comes first
			// Need tmp variables for appropriate indirection for paksort()
			pakfilestmp = &pakfiles[pakfilesi];
			pakdirstmp  = &pakdirs[pakdirsi];
			pakwhich    = (paksort(pakfilestmp, pakdirstmp) < 0);
		}
		if (pakwhich)
		{
			// The next .pk3 file is before the next .pk3dir
			pakfile = FS_BuildOSPath(path, dir, pakfiles[pakfilesi]);
			if ((pak = FS_LoadZipFile(pakfile, pakfiles[pakfilesi])) == 0)
			{
				// This isn't a .pk3! Next!
				pakfilesi++;
				continue;
			}

			Q_strncpyz(pak->pakPathname, curpath, sizeof(pak->pakPathname));
			// store the game name for downloading
			Q_strncpyz(pak->pakGamename, dir, sizeof(pak->pakGamename));

			fs_packFiles += pak->numfiles;

			search         = Z_Malloc(sizeof(searchpath_t));
			search->pack   = pak;
			search->next   = fs_searchpaths;
			fs_searchpaths = search;

			pakfilesi++;
		}
		else
		{
			// The next .pk3dir is before the next .pk3 file
			// But wait, this could be any directory, we're filtering to only ending with ".pk3dir" here.
			len = strlen(pakdirs[pakdirsi]);
			if (!FS_IsExt(pakdirs[pakdirsi], ".pk3dir", len))
			{
				// This isn't a .pk3dir! Next!
				pakdirsi++;
				continue;
			}

			pakfile = FS_BuildOSPath(path, dir, pakdirs[pakdirsi]);

			// add the directory to the search path
			search      = Z_Malloc(sizeof(searchpath_t));
			search->dir = Z_Malloc(sizeof(*search->dir));

			Q_strncpyz(search->dir->path, curpath, sizeof(search->dir->path));  // c:\etlegacy\etmain
			Q_strncpyz(search->dir->fullpath, pakfile, sizeof(search->dir->fullpath));  // c:\etlegacy\etmain\mypak.pk3dir
			Q_strncpyz(search->dir->gamedir, pakdirs[pakdirsi], sizeof(search->dir->gamedir)); // mypak.pk3dir

			search->next   = fs_searchpaths;
			fs_searchpaths = search;

			pakdirsi++;
		}
	}

	// done
	Sys_FreeFileList(pakfiles);
	Sys_FreeFileList(pakdirs);

	//
	// add the directory to the search path
	//
	search      = Z_Malloc(sizeof(searchpath_t));
	search->dir = Z_Malloc(sizeof(*search->dir));

	Q_strncpyz(search->dir->path, path, sizeof(search->dir->path));
	Q_strncpyz(search->dir->fullpath, curpath, sizeof(search->dir->fullpath));
	Q_strncpyz(search->dir->gamedir, dir, sizeof(search->dir->gamedir));

	search->next   = fs_searchpaths;
	fs_searchpaths = search;
}

/*
================
FS_idPak
================
*/
qboolean FS_idPak(char *pak, char *base)
{
	int i;

	// ET:L does not need original mp_bin
	//if (!FS_FilenameCompare(pak, va("%s/mp_bin", base)))
	//{
	//	return qtrue;
	//}

	for (i = 0; i < NUM_ID_PAKS; i++)
	{
		if (!FS_FilenameCompare(pak, va("%s/pak%d", base, i)))
		{
			break;
		}
	}
	if (i < NUM_ID_PAKS)
	{
		return qtrue;
	}
	return qfalse;
}

typedef struct
{
	char pakname[MAX_QPATH];
	qboolean ok;
} officialpak_t;

/*
================
FS_VerifyOfficialPaks
================
*/
qboolean FS_VerifyOfficialPaks(void)
{
	int           i, j;
	searchpath_t  *sp;
	int           numOfficialPaksOnServer = 0;
	int           numOfficialPaksLocal    = 0;
	officialpak_t officialpaks[64];

	if (!fs_numServerPaks)
	{
		return qtrue;
	}

	for (i = 0; i < fs_numServerPaks; i++)
	{
		if (FS_idPak(fs_serverPakNames[i], BASEGAME))
		{
			Q_strncpyz(officialpaks[numOfficialPaksOnServer].pakname, fs_serverPakNames[i], sizeof(officialpaks[i].pakname));
			officialpaks[numOfficialPaksOnServer].ok = qfalse;
			numOfficialPaksOnServer++;
		}
	}

	for (i = 0; i < fs_numServerPaks; i++)
	{
		for (sp = fs_searchpaths ; sp ; sp = sp->next)
		{
			if (sp->pack && sp->pack->checksum == fs_serverPaks[i])
			{
				char packPath[MAX_QPATH];

				Com_sprintf(packPath, sizeof(packPath), "%s/%s", sp->pack->pakGamename, sp->pack->pakBasename);

				if (FS_idPak(packPath, BASEGAME))
				{
					for (j = 0; j < numOfficialPaksOnServer; j++)
					{
						if (!Q_stricmp(packPath, officialpaks[j].pakname))
						{
							officialpaks[j].ok = qtrue;
							numOfficialPaksLocal++;
						}
					}
				}
				break;
			}
		}
	}

	if (numOfficialPaksOnServer != numOfficialPaksLocal)
	{
		for (i = 0; i < numOfficialPaksOnServer; i++)
		{
			if (officialpaks[i].ok != qtrue)
			{
				Com_Printf("ERROR: Missing/corrupt official pak file %s\n", officialpaks[i].pakname);
			}
		}
		return qfalse;
	}
	else
	{
		return qtrue;
	}
}

/*
================
FS_CheckDirTraversal

Check whether the string contains stuff like "../" to prevent directory traversal bugs
and return qtrue if it does.
================
*/
qboolean FS_CheckDirTraversal(const char *checkdir)
{
	if (strstr(checkdir, "../") || strstr(checkdir, "..\\"))
	{
		return qtrue;
	}

	return qfalse;
}

/*
================
FS_ComparePaks

----------------
dlstring == qtrue

Returns a list of pak files that we should download from the server. They all get stored
in the current gamedir and an FS_Restart will be fired up after we download them all.

The string is the format:

@remotename@localname [repeat]

static int      fs_numServerReferencedPaks;
static int      fs_serverReferencedPaks[MAX_SEARCH_PATHS];
static char     *fs_serverReferencedPakNames[MAX_SEARCH_PATHS];

----------------
dlstring == qfalse

we are not interested in a download string format, we want something human-readable
(this is used for diagnostics while connecting to a pure server)
================
*/
qboolean CL_WWWBadChecksum(const char *pakname);
qboolean FS_ComparePaks(char *neededpaks, int len, qboolean dlstring)
{
	searchpath_t *sp;
	qboolean     havepak;
	char         *origpos = neededpaks;
	int          i;

	if (!fs_numServerReferencedPaks)
	{
		return qfalse; // Server didn't send any pack information along
	}

	*neededpaks = 0;

	for (i = 0 ; i < fs_numServerReferencedPaks ; i++)
	{
		// Ok, see if we have this pak file
		havepak = qfalse;

		// never autodownload any of the id paks
		if (FS_idPak(fs_serverReferencedPakNames[i], BASEGAME))
		{
			continue;
		}

		// Make sure the server cannot make us write to non-quake3 directories.
		if (FS_CheckDirTraversal(fs_serverReferencedPakNames[i]))
		{
			Com_Printf("WARNING: Invalid download name %s\n", fs_serverReferencedPakNames[i]);
			continue;
		}

		for (sp = fs_searchpaths ; sp ; sp = sp->next)
		{
			if (sp->pack && sp->pack->checksum == fs_serverReferencedPaks[i])
			{
				havepak = qtrue; // This is it!
				break;
			}
		}

		if (!havepak && fs_serverReferencedPakNames[i] && *fs_serverReferencedPakNames[i])
		{
			// Don't got it

			if (dlstring)
			{
				// We need this to make sure we won't hit the end of the buffer or the server could
				// overwrite non-pk3 files on clients by writing so much crap into neededpaks that
				// Q_strcat cuts off the .pk3 extension.

				origpos += strlen(origpos);

				// Remote name
				Q_strcat(neededpaks, len, "@");
				Q_strcat(neededpaks, len, fs_serverReferencedPakNames[i]);
				Q_strcat(neededpaks, len, ".pk3");

				// Local name
				Q_strcat(neededpaks, len, "@");
				// Do we have one with the same name?
				if (FS_SV_FileExists(va("%s.pk3", fs_serverReferencedPakNames[i])))
				{
					char st[MAX_ZPATH];
					// We already have one called this, we need to download it to another name
					// Make something up with the checksum in it
					Com_sprintf(st, sizeof(st), "%s.%08x.pk3", fs_serverReferencedPakNames[i], fs_serverReferencedPaks[i]);
					Q_strcat(neededpaks, len, st);
				}
				else
				{
					Q_strcat(neededpaks, len, fs_serverReferencedPakNames[i]);
					Q_strcat(neededpaks, len, ".pk3");
				}

				// Find out whether it might have overflowed the buffer and don't add this file to the
				// list if that is the case.
				if (strlen(origpos) + (origpos - neededpaks) >= len - 1)
				{
					*origpos = '\0';
					break;
				}
			}
			else
			{
				Q_strcat(neededpaks, len, fs_serverReferencedPakNames[i]);
				Q_strcat(neededpaks, len, ".pk3");
				// Do we have one with the same name?
				if (FS_SV_FileExists(va("%s.pk3", fs_serverReferencedPakNames[i])))
				{
					Q_strcat(neededpaks, len, " (local file exists with wrong checksum)");
#ifndef DEDICATED
					// let the client subsystem track bad download redirects (dl file with wrong checksums)
					// this is a bit ugly but the only other solution would have been callback passing..
					if (CL_WWWBadChecksum(va("%s.pk3", fs_serverReferencedPakNames[i])))
					{
						// remove a potentially malicious download file
						// (this is also intended to avoid expansion of the pk3 into a file with different checksum .. messes up wwwdl chkfail)
						char *rmv = FS_BuildOSPath(fs_homepath->string, va("%s.pk3", fs_serverReferencedPakNames[i]), "");

						rmv[strlen(rmv) - 1] = '\0';
						FS_Remove(rmv);
					}
#endif
				}
				Q_strcat(neededpaks, len, "\n");
			}
		}
	}

	if (*neededpaks)
	{
		Com_Printf("Need paks: %s\n", neededpaks);
		return qtrue;
	}

	return qfalse; // We have them all
}

/*
================
FS_Shutdown

Frees all resources and closes all files
================
*/
void FS_Shutdown(qboolean closemfp)
{
	searchpath_t *p, *next;
	int          i;

	for (i = 0; i < MAX_FILE_HANDLES; i++)
	{
		if (fsh[i].fileSize)
		{
			FS_FCloseFile(i);
		}
	}

	// free everything
	for (p = fs_searchpaths ; p ; p = next)
	{
		next = p->next;

		if (p->pack)
		{
			FS_FreePak(p->pack);
		}
		if (p->dir)
		{
			Z_Free(p->dir);
		}
		Z_Free(p);
	}

	// any FS_ calls will now be an error until reinitialized
	fs_searchpaths = NULL;

	Cmd_RemoveCommand("path");
	Cmd_RemoveCommand("dir");
	Cmd_RemoveCommand("fdir");
	Cmd_RemoveCommand("touchFile");
	Cmd_RemoveCommand("which");

#ifdef FS_MISSING
	if (closemfp)
	{
		fclose(missingFiles);
	}
#endif
}

/*
================
FS_ReorderPurePaks
NOTE: the reordering that happens here is not reflected in the cvars (\cvarlist *pak*)
  this can lead to misleading situations
================
*/
static void FS_ReorderPurePaks(void)
{
	searchpath_t *s;
	int          i;
	searchpath_t **p_insert_index, // for linked list reordering
	**p_previous;     // when doing the scan

	fs_reordered = qfalse;

	// only relevant when connected to pure server
	if (!fs_numServerPaks)
	{
		return;
	}

	p_insert_index = &fs_searchpaths; // we insert in order at the beginning of the list
	for (i = 0 ; i < fs_numServerPaks ; i++)
	{
		p_previous = p_insert_index; // track the pointer-to-current-item
		for (s = *p_insert_index; s; s = s->next)     // the part of the list before p_insert_index has been sorted already
		{
			if (s->pack && fs_serverPaks[i] == s->pack->checksum)
			{
				fs_reordered = qtrue;
				// move this element to the insert list
				*p_previous     = s->next;
				s->next         = *p_insert_index;
				*p_insert_index = s;
				// increment insert list
				p_insert_index = &s->next;
				break; // iterate to next server pack
			}
			p_previous = &s->next;
		}
	}
}

/*
================
FS_Startup
================
*/
static void FS_Startup(const char *gameName)
{
	const char *homePath;

	Com_Printf("----- FS_Startup -----\n");

	fs_packFiles = 0;

	fs_debug    = Cvar_Get("fs_debug", "0", 0);
	fs_basepath = Cvar_Get("fs_basepath", Sys_DefaultInstallPath(), CVAR_INIT | CVAR_PROTECTED);
	fs_basegame = Cvar_Get("fs_basegame", "", CVAR_INIT | CVAR_PROTECTED);

	homePath = Sys_DefaultHomePath();

	if (!homePath || !homePath[0])
	{
#ifdef DEDICATED
		homePath = fs_basepath->string;
#else
		Com_Error(ERR_FATAL, "FS_Startup: Default home path is empty.");
#endif
	}

	fs_homepath = Cvar_Get("fs_homepath", homePath, CVAR_INIT);

	fs_gamedirvar = Cvar_Get("fs_game", "", CVAR_INIT | CVAR_SYSTEMINFO);

	// add search path elements in reverse priority order
	if (fs_basepath->string[0])
	{
		FS_AddGameDirectory(fs_basepath->string, gameName);
	}

	// fs_homepath is somewhat particular to *nix systems, only add if relevant
	// - update: now fs_homepath is used for all system
	// NOTE: same filtering below for mods and basegame
	if (fs_basepath->string[0] && Q_stricmp(fs_homepath->string, fs_basepath->string))
	{
		FS_AddGameDirectory(fs_homepath->string, gameName);
	}

	// check for additional base game so mods can be based upon other mods
	if (fs_basegame->string[0] && !Q_stricmp(gameName, BASEGAME) && Q_stricmp(fs_basegame->string, gameName))
	{
		if (fs_basepath->string[0])
		{
			FS_AddGameDirectory(fs_basepath->string, fs_basegame->string);
		}
		if (fs_homepath->string[0] && Q_stricmp(fs_homepath->string, fs_basepath->string))
		{
			FS_AddGameDirectory(fs_homepath->string, fs_basegame->string);
		}
	}

	// check for additional game folder for mods
	if (fs_gamedirvar->string[0] && !Q_stricmp(gameName, BASEGAME) && Q_stricmp(fs_gamedirvar->string, gameName))
	{
		if (fs_basepath->string[0])
		{
			FS_AddGameDirectory(fs_basepath->string, fs_gamedirvar->string);
		}
		if (fs_homepath->string[0] && Q_stricmp(fs_homepath->string, fs_basepath->string))
		{
			FS_AddGameDirectory(fs_homepath->string, fs_gamedirvar->string);
		}
	}

	// add our commands
	Cmd_AddCommand("path", FS_Path_f);
	Cmd_AddCommand("dir", FS_Dir_f);
	Cmd_AddCommand("fdir", FS_NewDir_f);
	Cmd_AddCommand("touchFile", FS_TouchFile_f);
	Cmd_AddCommand("which", FS_Which_f);

	// reorder the pure pk3 files according to server order
	FS_ReorderPurePaks();

	// print the current search paths
	FS_Path_f();

	fs_gamedirvar->modified = qfalse; // We just loaded, it's not modified

	Com_Printf("----------------------\n");

#ifdef FS_MISSING
	if (missingFiles == NULL)
	{
		missingFiles = Sys_FOpen("\\missing.txt", "ab");
	}
#endif
	Com_Printf("%d files in pk3 files\n", fs_packFiles);

#ifdef DEDICATED
	// don't start if base == home, so downloads won't overwrite original files
	//if (FS_PathCmp(fs_homepath->string, fs_basepath->string) == 0)
	if (FS_IsSamePath(fs_homepath->string, fs_basepath->string))
	{
		Com_Error(ERR_FATAL, "FS_Startup: fs_homepath and fs_basepath are identical - set different paths!");
		// NOTE: if both are same - crashlog.txt is written into (fs_homepath)
	}
#endif
}

/*
=====================
FS_LoadedPakChecksums

Returns a space separated string containing the checksums of all loaded pk3 files.
Servers with sv_pure set will get this string and pass it to clients.
=====================
*/
const char *FS_LoadedPakChecksums(void)
{
	static char  info[BIG_INFO_STRING];
	searchpath_t *search;
	int          len;

	info[0] = 0;

	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (!search->pack)
		{
			continue;
		}

		Q_strcat(info, sizeof(info), va("%i ", search->pack->checksum));
	}

	// remove last space char
	len = strlen(info);
	if (len > 1) // foolproof
	{
		info[len - 1] = 0;
	}

	return info;
}

/*
=====================
FS_LoadedPakNames

Returns a space separated string containing the names of all loaded pk3 files.
Servers with sv_pure set will get this string and pass it to clients.
=====================
*/
const char *FS_LoadedPakNames(void)
{
	static char  info[BIG_INFO_STRING];
	searchpath_t *search;

	info[0] = 0;

	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (!search->pack)
		{
			continue;
		}

		if (*info)
		{
			Q_strcat(info, sizeof(info), " ");
		}

		// full path ...
		Q_strcat(info, sizeof(info), search->pack->pakGamename);
		Q_strcat(info, sizeof(info), "/");
		Q_strcat(info, sizeof(info), search->pack->pakBasename);
	}

	return info;
}

/*
=====================
FS_LoadedPakPureChecksums

Returns a space separated string containing the pure checksums of all loaded pk3 files.
Servers with sv_pure use these checksums to compare with the checksums the clients send
back to the server.
=====================
*/
const char *FS_LoadedPakPureChecksums(void)
{
	static char  info[BIG_INFO_STRING];
	searchpath_t *search;
	int          len;

	info[0] = 0;

	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (!search->pack)
		{
			continue;
		}

		Q_strcat(info, sizeof(info), va("%i ", search->pack->pure_checksum));
	}

	// remove last space char
	len = strlen(info);
	if (len > 1) // foolproof
	{
		info[len - 1] = 0;
	}

	// only comment out when you need a new pure checksums string
	//Com_DPrintf("FS_LoadPakPureChecksums: %s\n", info);

	return info;
}

/*
=====================
FS_ReferencedPakChecksums

Returns a space separated string containing the checksums of all referenced pk3 files.
The server will send this to the clients so they can check which files should be auto-downloaded.
=====================
*/
const char *FS_ReferencedPakChecksums(void)
{
	static char  info[BIG_INFO_STRING];
	searchpath_t *search;
	int          len;

	info[0] = 0;

	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (search->pack)
		{
			if (search->pack->referenced || Q_stricmpn(search->pack->pakGamename, BASEGAME, strlen(BASEGAME)))
			{
				Q_strcat(info, sizeof(info), va("%i ", search->pack->checksum));
			}
		}
	}

	// remove last space char
	len = strlen(info);
	if (len > 1) // foolproof
	{
		info[len - 1] = 0;
	}

	return info;
}

/*
=====================
FS_ReferencedPakNames

Returns a space separated string containing the names of all referenced pk3 files.
The server will send this to the clients so they can check which files should be auto-downloaded.
=====================
*/
const char *FS_ReferencedPakNames(void)
{
	static char  info[BIG_INFO_STRING];
	searchpath_t *search;
	int          len;

	info[0] = 0;

	// we want to return ALL pk3's from the fs_game path
	// and referenced one's from etmain
	for (search = fs_searchpaths ; search ; search = search->next)
	{
		// is the element a pak file?
		if (search->pack)
		{
			if (search->pack->referenced || Q_stricmpn(search->pack->pakGamename, BASEGAME, strlen(BASEGAME)))
			{
				Q_strcat(info, sizeof(info), search->pack->pakGamename);
				Q_strcat(info, sizeof(info), "/");
				Q_strcat(info, sizeof(info), search->pack->pakBasename);
				Q_strcat(info, sizeof(info), " ");
			}
		}
	}

	// remove last space char
	len = strlen(info);
	if (len > 1) // foolproof
	{
		info[len - 1] = 0;
	}

	return info;
}

/*
=====================
FS_ReferencedPakPureChecksums

Returns a space separated string containing the pure checksums of all referenced pk3 files.
Servers with sv_pure set will get this string back from clients for pure validation

The string has a specific order, "cgame ui @ ref1 ref2 ref3 ..."

this function is only used by the client to build the string sent back to server,
it's useless in dedicated
=====================
*/
const char *FS_ReferencedPakPureChecksums(void)
{
	static char  info[BIG_INFO_STRING];
	searchpath_t *search;
	int          nFlags, numPaks = 0, checksum = fs_checksumFeed, len;

	info[0] = 0;

	for (nFlags = FS_CGAME_REF; nFlags; nFlags = nFlags >> 1)
	{
		if (nFlags & FS_GENERAL_REF)
		{
			// add a delimter between must haves and general refs
			//Q_strcat(info, sizeof(info), "@ ");
			info[strlen(info) + 1] = '\0';
			info[strlen(info) + 2] = '\0';
			info[strlen(info)]     = '@';
			info[strlen(info)]     = ' ';
		}
		for (search = fs_searchpaths ; search ; search = search->next)
		{
			// is the element a pak file and has it been referenced based on flag?
			if (search->pack && (search->pack->referenced & nFlags))
			{
				Q_strcat(info, sizeof(info), va("%i ", search->pack->pure_checksum));
				if (nFlags & (FS_CGAME_REF | FS_UI_REF))
				{
					break;
				}
				checksum ^= search->pack->pure_checksum;
				numPaks++;
			}
		}
	}
	// last checksum is the encoded number of referenced pk3s
	checksum ^= numPaks;
	Q_strcat(info, sizeof(info), va("%i ", checksum));

	// remove last space char
	len = strlen(info);
	if (len > 1) // foolproof
	{
		info[len - 1] = 0;
	}

	return info;
}

/*
=====================
FS_ClearPakReferences
=====================
*/
void FS_ClearPakReferences(int flags)
{
	searchpath_t *search;

	if (!flags)
	{
		flags = -1;
	}
	for (search = fs_searchpaths; search; search = search->next)
	{
		// is the element a pak file and has it been referenced?
		if (search->pack)
		{
			search->pack->referenced &= ~flags;
		}
	}
}

/**
 * @brief If the string is empty, all data sources will be allowed.
 * If not empty, only pk3 files that match one of the space separated checksums
 * will be checked for files, with the exception of .cfg and .dat files.
 */
void FS_PureServerSetLoadedPaks(const char *pakSums, const char *pakNames)
{
	int i, c, d;

	Cmd_TokenizeString(pakSums);

	c = Cmd_Argc();
	if (c > MAX_SEARCH_PATHS)
	{
		c = MAX_SEARCH_PATHS;
	}

	fs_numServerPaks = c;

	for (i = 0 ; i < c ; i++)
	{
		fs_serverPaks[i] = atoi(Cmd_Argv(i));
	}

	if (fs_numServerPaks)
	{
		Com_DPrintf("Connected to a pure server.\n");
	}
	else
	{
		if (fs_reordered)
		{
			// force a restart to make sure the search order will be correct
			Com_DPrintf("FS search reorder is required\n");
			FS_Restart(fs_checksumFeed);
			return;
		}
	}

	for (i = 0 ; i < c ; i++)
	{
		if (fs_serverPakNames[i])
		{
			Z_Free(fs_serverPakNames[i]);
		}
		fs_serverPakNames[i] = NULL;
	}

	if (pakNames && *pakNames)
	{
		Cmd_TokenizeString(pakNames);

		d = Cmd_Argc();
		if (d > MAX_SEARCH_PATHS)
		{
			d = MAX_SEARCH_PATHS;
		}

		for (i = 0 ; i < d ; i++)
		{
			fs_serverPakNames[i] = CopyString(Cmd_Argv(i));
		}
	}
}

/**
 * @brief The checksums and names of the pk3 files referenced at the server are sent to
 * the client and stored here. The client will use these checksums to see if any
 * pk3 files need to be auto-downloaded.
 */
void FS_PureServerSetReferencedPaks(const char *pakSums, const char *pakNames)
{
	int i, c, d = 0;

	Cmd_TokenizeString(pakSums);

	c = Cmd_Argc();
	if (c > MAX_SEARCH_PATHS)
	{
		c = MAX_SEARCH_PATHS;
	}

	fs_numServerReferencedPaks = c;

	for (i = 0 ; i < c ; i++)
	{
		fs_serverReferencedPaks[i] = atoi(Cmd_Argv(i));
	}

	for (i = 0 ; i < ARRAY_LEN(fs_serverReferencedPakNames); i++)
	{
		if (fs_serverReferencedPakNames[i])
		{
			Z_Free(fs_serverReferencedPakNames[i]);
		}

		fs_serverReferencedPakNames[i] = NULL;
	}

	if (pakNames && *pakNames)
	{
		Cmd_TokenizeString(pakNames);

		d = Cmd_Argc();
		if (d > MAX_SEARCH_PATHS)
		{
			d = MAX_SEARCH_PATHS;
		}

		for (i = 0 ; i < d ; i++)
		{
			fs_serverReferencedPakNames[i] = CopyString(Cmd_Argv(i));
		}
	}
}

/**
 * @brief Prints file system info
 */
void FS_Fileinfo_f(void)
{
	Com_Printf("fs_gamedir %s\n", fs_gamedir);
	Com_Printf("fs_debug %s\n", fs_debug->string);

	Com_Printf("fs_homepath %s\n", fs_homepath->string);
	Com_Printf("fs_basepath %s\n", fs_basepath->string);
	Com_Printf("fs_basegame %s\n", fs_basegame->string);
	Com_Printf("fs_gamedirvar %s\n", fs_gamedirvar->string);

	// static searchpath_t *fs_searchpaths;

	Com_Printf("Total megs read %i\n", fs_readCount / (1024 * 1024)); // total at runtime, isn't reset
	Com_Printf("Total files read %i\n", fs_loadCount);                // total at runtime, isn't reset
	Com_Printf("Total files in memory %i\n", fs_loadStack);           // should be 0 in game
	Com_Printf("Total number of files in packs %i\n", fs_packFiles);
}

/**
 * @brief Called only at initial startup, not when the filesystem
 * is resetting due to a game change
 */
void FS_InitFilesystem(void)
{
	cvar_t *tmp_fs_game;

	// allow command line params to override our defaults
	// we have to specially handle this, because normal command
	// line variable sets don't happen until after the filesystem
	// has already been initialized
	Com_StartupVariable("fs_basepath");
	Com_StartupVariable("fs_homepath");
	Com_StartupVariable("fs_game");

	// ET: Legacy start
	// if fs_game is not specified, set 'legacy' mod as default fs_game
	// this 'optimization' grants us 2.60b compatibilty w/o deeper changes and users
	// don't have to set fs_game param to run latest mod code
	tmp_fs_game = Cvar_Get("fs_game", "", 0);
	if (!strcmp(tmp_fs_game->string, ""))
	{
		Cvar_Set("fs_game", DEFAULT_MODGAME);
		tmp_fs_game         = Cvar_Get("fs_game", "", 0);
		tmp_fs_game->flags |= CVAR_USER_CREATED; // deal as startup var

		Com_Printf("Info: fs_game now defaults to '%s' mod instead of 'etmain'\n", tmp_fs_game->string);
	}

	// try to start up normally
	FS_Startup(BASEGAME);

	// if we can't find default.cfg, assume that the paths are
	// busted and error out now, rather than getting an unreadable
	// graphics screen when the font fails to load
	// - we want the nice error message here as well
	// FIXME: temporary fix - default.cfg is now part of pak3.pk3, add a smarter check
	if (FS_ReadFile("fonts/ariblk_0_16.tga", NULL) <= 0)
	{
		Com_Error(ERR_FATAL, "FS_InitFilesystem: Couldn't load base data files!\nVerify your installation and make sure genuine ET files\n- pak0.pk3\n- pak1.pk3\n- pak2.pk3\nare located in 'etmain' folder of fs_basepath: %s", fs_basepath->string);
	}

	Q_strncpyz(lastValidBase, fs_basepath->string, sizeof(lastValidBase));
	Q_strncpyz(lastValidGame, fs_gamedirvar->string, sizeof(lastValidGame));

	if (fs_debug->integer)
	{
		Cmd_AddCommand("fileinfo", FS_Fileinfo_f);
	}
}

void FS_Restart(int checksumFeed)
{
	// free anything we currently have loaded
	FS_Shutdown(qfalse);

	// set the checksum feed
	fs_checksumFeed = checksumFeed;

	// clear pak references
	FS_ClearPakReferences(0);

	// try to start up normally
	FS_Startup(BASEGAME);

	// if we can't find default.cfg, assume that the paths are
	// busted and error out now, rather than getting an unreadable
	// graphics screen when the font fails to load
	if (FS_ReadFile("default.cfg", NULL) <= 0
#ifdef PANDORA
	    // also ensure default_pandora.cfg
	    && FS_ReadFile(CONFIG_NAME_DEFAULT, NULL) <= 0
#endif
	    )
	{
		// this might happen when connecting to a pure server not using BASEGAME/pak0.pk3
		// (for instance a TA demo server)
		if (lastValidBase[0])
		{
			FS_PureServerSetLoadedPaks("", "");
			Cvar_Set("fs_basepath", lastValidBase);
			Cvar_Set("fs_game", lastValidGame);
			lastValidBase[0] = '\0';
			lastValidGame[0] = '\0';
			FS_Restart(checksumFeed);
			Com_Error(ERR_DROP, "Invalid game folder");

			return;
		}
		// added some verbosity, 'couldn't load default.cfg' confuses the hell out of users
#ifdef PANDORA
		Com_Error(ERR_FATAL, "FS_Restart: Couldn't load default.cfg and default_pandora.cfg - I am missing essential files - verify your installation?");
#else
		Com_Error(ERR_FATAL, "FS_Restart: Couldn't load default.cfg - I am missing essential files - verify your installation?");
#endif
	}

	// new check before safeMode
	if (Q_stricmp(fs_gamedirvar->string, lastValidGame))
	{
		// skip the etconfig.cfg if "safe" is on the command line
		if (!Com_SafeMode())
		{
			char *cl_profileStr = Cvar_VariableString("cl_profile");

			if (cl_profileStr[0])
			{
				// check existing pid file and make sure it's ok
				if (!Com_CheckProfile())
				{
#ifdef NDEBUG
					Com_Printf("^3WARNING: profile.pid found for profile '%s' - system settings will revert to defaults\n", cl_profileStr);
					// set crashed state
					Cbuf_AddText("set com_crashed 1\n");
#endif
				}

				// write a new one
				if (!Sys_WritePIDFile())
				{
					Com_Printf("^3WARNING: couldn't write %s\n", com_pidfile->string);
				}

				// exec the config
				Cbuf_AddText(va("exec profiles/%s/%s\n", cl_profileStr, CONFIG_NAME));

			}
			else
			{
				Cbuf_AddText(va("exec %s\n", CONFIG_NAME));
			}
		}
	}

	Q_strncpyz(lastValidBase, fs_basepath->string, sizeof(lastValidBase));
	Q_strncpyz(lastValidGame, fs_gamedirvar->string, sizeof(lastValidGame));
}

/**
 * @brief Restart if necessary.
 * @todo This doesn't catch all cases where an FS_Restart is necessary
 */
qboolean FS_ConditionalRestart(int checksumFeed)
{
	if (fs_gamedirvar->modified || checksumFeed != fs_checksumFeed)
	{
		FS_Restart(checksumFeed);

		return qtrue;
	}

	return qfalse;
}

/**
 * @brief Handle based file calls for virtual machines
 */
int FS_FOpenFileByMode(const char *qpath, fileHandle_t *f, fsMode_t mode)
{
	int      r;
	qboolean sync = qfalse;

	switch (mode)
	{
	case FS_READ:
		r = FS_FOpenFileRead(qpath, f, qtrue);
		break;
	case FS_WRITE:
		*f = FS_FOpenFileWrite(qpath);
		r  = 0;
		if (*f == 0)
		{
			r = -1;
		}
		break;
	case FS_APPEND_SYNC:
		sync = qtrue;
	// fall through
	case FS_APPEND:
		*f = FS_FOpenFileAppend(qpath);
		r  = 0;
		if (*f == 0)
		{
			r = -1;
		}
		break;
	default:
		Com_Error(ERR_FATAL, "FS_FOpenFileByMode: bad mode");

		return -1;
	}

	if (!f)
	{
		return r;
	}

	if (*f)
	{
		if (fsh[*f].zipFile == qtrue)
		{
			fsh[*f].baseOffset = unztell(fsh[*f].handleFiles.file.z);
		}
		else
		{
			fsh[*f].baseOffset = ftell(fsh[*f].handleFiles.file.o);
		}
		fsh[*f].fileSize = r;
	}
	fsh[*f].handleSync = sync;

	return r;
}

int FS_FTell(fileHandle_t f)
{
	int pos;

	if (fsh[f].zipFile == qtrue)
	{
		pos = unztell(fsh[f].handleFiles.file.z);
	}
	else
	{
		pos = ftell(fsh[f].handleFiles.file.o);
	}

	return pos;
}

void FS_Flush(fileHandle_t f)
{
	fflush(fsh[f].handleFiles.file.o);
}

void FS_FilenameCompletion(const char *dir, const char *ext,
                           qboolean stripExt, void (*callback)(const char *s), qboolean allowNonPureFilesOnDisk)
{
	char **filenames;
	int  nfiles;
	int  i;
	char filename[MAX_STRING_CHARS];

	filenames = FS_ListFilteredFiles(dir, ext, NULL, &nfiles, allowNonPureFilesOnDisk);

	FS_SortFileList(filenames, nfiles);

	for (i = 0; i < nfiles; i++)
	{
		FS_ConvertPath(filenames[i]);
		Q_strncpyz(filename, filenames[i], MAX_STRING_CHARS);

		if (stripExt)
		{
			COM_StripExtension(filename, filename, sizeof(filename));
		}

		callback(filename);
	}
	FS_FreeFileList(filenames);
}

// CVE-2006-2082
// compared requested pak against the names as we built them in FS_ReferencedPakNames
qboolean FS_VerifyPak(const char *pak)
{
	char         teststring[BIG_INFO_STRING];
	searchpath_t *search;

	for (search = fs_searchpaths ; search ; search = search->next)
	{
		if (search->pack)
		{
			Q_strncpyz(teststring, search->pack->pakGamename, sizeof(teststring));
			Q_strcat(teststring, sizeof(teststring), "/");
			Q_strcat(teststring, sizeof(teststring), search->pack->pakBasename);
			Q_strcat(teststring, sizeof(teststring), ".pk3");
			if (!Q_stricmp(teststring, pak))
			{
				return qtrue;
			}
		}
	}

	return qfalse;
}

/**
 * @brief Extracts zipped file into the current gamedir
 * @param[in] filename to extract
 * @param[in] qboolean whether to inform if unzipping fails
 * @retval qtrue    if successfully extracted
 * @retval qfalse   if extraction failed
 */
qboolean FS_Unzip(char *filename, qboolean quiet)
{
	char            zipPath[MAX_OSPATH];
	unzFile         zipFile;
	unz_global_info zipInfo;
	int             err, i;
	void            *buf = NULL;

	Com_sprintf(zipPath, sizeof(zipPath), "%s", filename);
	zipFile = unzOpen(zipPath);

	if (!zipFile)
	{
		if (!quiet || fs_debug->integer)
		{
			Com_Printf(S_COLOR_RED "ERROR: not a zip file (%s).\n", zipPath);
		}
		return qfalse;
	}

	err = unzGetGlobalInfo(zipFile, &zipInfo);

	if (err != UNZ_OK)
	{
		if (!quiet || fs_debug->integer)
		{
			Com_Printf(S_COLOR_RED "FS_Unzip: unable to unzip file (%s).\n", zipPath);
		}
		return qfalse;
	}

	err = unzGoToFirstFile(zipFile);

	for (i = 0; i < zipInfo.number_entry; i++)
	{
		unz_file_info file_info;
		char          newFileName[MAX_OSPATH];
		char          newFilePath[MAX_OSPATH];
		FILE          *newFile;

		err = unzGetCurrentFileInfo(zipFile, &file_info, newFileName, sizeof(newFileName), NULL, 0, NULL, 0);

		Com_sprintf(newFilePath, sizeof(newFilePath), "%s%c%s%c%s", fs_homepath->string, PATH_SEP, fs_gamedir, PATH_SEP, newFileName);

		if (newFilePath[strlen(newFilePath) - 1] == PATH_SEP)
		{
			if (fs_debug->integer)
			{
				Com_Printf("FS_Unzip: Creating directory %s...\n", newFilePath);
			}

			if (FS_CreatePath(newFilePath))
			{
				Com_Printf(S_COLOR_RED "ERROR: Unable to create directory (%s).\n", newFilePath);
				return qfalse;
			}
		}
		else
		{
			if (fs_debug->integer)
			{
				Com_Printf("FS_Unzip: Extracting %s...\n", newFilePath);
			}

			newFile = Sys_FOpen(newFilePath, "wb");
			if (!newFile)
			{
				break;
			}

			if (buf)
			{
				free(buf);
			}

			buf = malloc(file_info.uncompressed_size);

			if (!buf)
			{
				break;
			}

			err = unzOpenCurrentFile(zipFile);
			if (err)
			{
				break;
			}

			err = unzReadCurrentFile(zipFile, buf, file_info.uncompressed_size);
			if ((err < 0) || (err != file_info.uncompressed_size))
			{
				break;
			}

			err = fwrite(buf, file_info.uncompressed_size, 1, newFile);
			if (err != 1)
			{
				break;
			}

			fclose(newFile);

			err = unzCloseCurrentFile(zipFile);
		}

		err = unzGoToNextFile(zipFile);
		if (err)
		{
			break;
		}
	}

	if (buf)
	{
		free(buf);
	}

	return qtrue;
}
