/*
 * Copyright (C) 2012 Szilard Biro
 * Copyright (C) 1997-2001 Id Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * =======================================================================
 *
 * This file implements an OpenGL context via AROS Mesa
 *
 * =======================================================================
 */

#include "../refresh/header/local.h"
#include "../unix/header/glwindow.h"

#include <cybergraphx/cybergraphics.h>
#include <proto/exec.h>
#include <proto/intuition.h>
#include <proto/graphics.h>
#include <proto/cybergraphics.h>

#include <GL/gl.h>
#include <GL/arosmesa.h>

struct Window *g_pWindow;
struct Screen *g_pScreen;
void *g_pCursor;
AROSMesaContext g_Context;

glwstate_t glw_state;
qboolean have_stencil = false;

/*
 * Initialzes the OpenGL context
 */
int
GLimp_Init(void)
{
	return true;
}

/*
 * Sets the hardware gamma
 */
void
UpdateHardwareGamma(void)
{
}

/*
 * Initializes the OpenGL window
 */
static qboolean
GLimp_InitGraphics(qboolean fullscreen)
{
	int stencil_bits;
	cvar_t *vid_xpos;
	cvar_t *vid_ypos;	
	ULONG wflgs;
	Tag screentag;
	static char title[24];

	GLimp_Shutdown();
	
	vid_xpos = ri.Cvar_Get("vid_xpos", "0", CVAR_ARCHIVE);
	vid_ypos = ri.Cvar_Get("vid_ypos", "0", CVAR_ARCHIVE);

	if (fullscreen)
	{
	    ULONG ModeID;

	    ModeID = BestCModeIDTags(
			CYBRBIDTG_Depth, 8,
			CYBRBIDTG_NominalWidth, vid.width,
			CYBRBIDTG_NominalHeight, vid.height,
			TAG_DONE);

	    g_pScreen = OpenScreenTags(NULL,
			ModeID != INVALID_ID ? SA_DisplayID : TAG_IGNORE, ModeID,
			SA_Width,		vid.width,
			SA_Height,		vid.height,
			SA_Depth,		8,
			SA_Quiet,		TRUE,
			SA_ShowTitle,	FALSE,			
			TAG_DONE);
	}

#ifdef __AROS__
	screentag = g_pScreen ? WA_CustomScreen : TAG_IGNORE;
#else
	screentag = g_pScreen ? WA_PubScreen : TAG_IGNORE;
#endif

	wflgs = WFLG_RMBTRAP | WFLG_REPORTMOUSE | WFLG_ACTIVATE;

	if (g_pScreen)
		wflgs |= WFLG_BORDERLESS;
	else
		wflgs |= WFLG_CLOSEGADGET | WFLG_DRAGBAR | WFLG_DEPTHGADGET;

	snprintf(title, sizeof(title), "Yamagi Quake II %s", VERSION);

	g_pWindow = OpenWindowTags(NULL,
	    g_pScreen ? TAG_IGNORE : WA_Left, vid_xpos->value,
	    g_pScreen ? TAG_IGNORE : WA_Top, vid_ypos->value,
	    WA_Flags,			wflgs,
	    WA_InnerWidth,		vid.width,
	    WA_InnerHeight,		vid.height,
	    screentag,			g_pScreen,
	    WA_Title,			title,
		WA_IDCMP,			/*IDCMP_RAWKEY | IDCMP_MOUSEBUTTONS | */IDCMP_CLOSEWINDOW/*|IDCMP_MOUSEMOVE|IDCMP_DELTAMOVE*/,
		TAG_DONE);
		
	if (!g_pWindow)
		ri.Sys_Error(ERR_FATAL, "Couldn't create window");
		
	/* Create the window */
	ri.Vid_NewWindow(vid.width, vid.height);

	g_Context = AROSMesaCreateContextTags(
        AMA_Window, g_pWindow,
        AMA_Left, g_pScreen ? 0 : g_pWindow->BorderLeft,
        AMA_Top, g_pScreen ? 0 : g_pWindow->BorderTop,
        AMA_Width, vid.width,
        AMA_Height, vid.height,
        g_pScreen ? AMA_Screen : TAG_IGNORE, g_pScreen,
        AMA_DoubleBuf, GL_TRUE,
        AMA_RGBMode, GL_TRUE,
        AMA_NoAccum, GL_TRUE,
        TAG_DONE);
		
	if (!g_Context)
	{
		GLimp_Shutdown();
		ri.Sys_Error (ERR_FATAL, "Couldn't create GL context");
	}

	AROSMesaMakeCurrent(g_Context);
		
	/* Initialize the stencil buffer */		
	AROSMesaGetConfig(g_Context, GL_STENCIL_BITS, &stencil_bits);
	ri.Con_Printf(PRINT_ALL, "Got %d bits of stencil.\n", stencil_bits);

	if (stencil_bits >= 1)
	{
		have_stencil = true;
	}

	/* Initialize hardware gamma */
	gl_state.hwgamma = false;

	return true;
}

/*
 * Swaps the buffers to show the new frame
 */
void
GLimp_EndFrame(void)
{
	AROSMesaSwapBuffers(g_Context);
}

/*
 * Changes the video mode
 */
int
GLimp_SetMode(int *pwidth, int *pheight, int mode, qboolean fullscreen)
{
	ri.Con_Printf(PRINT_ALL, "setting mode %d:", mode);

	/* mode -1 is not in the vid mode table - so we keep the values in pwidth
	   and pheight and don't even try to look up the mode info */
	if ((mode != -1) && !ri.Vid_GetModeInfo(pwidth, pheight, mode))
	{
		ri.Con_Printf(PRINT_ALL, " invalid mode\n");
		return rserr_invalid_mode;
	}

	ri.Con_Printf(PRINT_ALL, " %d %d\n", *pwidth, *pheight);

	if (!GLimp_InitGraphics(fullscreen))
	{
		return rserr_invalid_mode;
	}

	return rserr_ok;
}

/*
 * Shuts the GL render backend down
 */
void
GLimp_Shutdown(void)
{
    if (g_Context)
    {
        AROSMesaDestroyContext(g_Context);
		g_Context = 0;
    }
	
    if (g_pWindow)
	{
		CloseWindow(g_pWindow);
		g_pWindow = 0;
	}

    if (g_pScreen)
	{
		CloseScreen(g_pScreen);
	    g_pScreen = 0;
	}

	gl_state.hwgamma = false;
}

void *
qwglGetProcAddress(char *symbol)
{
	return AROSMesaGetProcAddress(symbol);
}

