/*
 * Copyright (C) 2012 Szilard Biro
 * Copyright (C) 1997-2001 Id Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * =======================================================================
 *
 * This file implements the software refresher via CyberGraphX
 *
 * =======================================================================
 */

#include "../ref_soft/r_local.h"

#include <cybergraphx/cybergraphics.h>
#include <proto/exec.h>
#include <proto/intuition.h>
#include <proto/graphics.h>
#include <proto/cybergraphics.h>

#ifdef __MORPHOS__
struct IntuitionBase *IntuitionBase;
struct GfxBase *GfxBase;
struct Library *CyberGfxBase;
#define BIGENDIAN
#endif

#ifdef __AROS__
#include <aros/cpu.h>
#if AROS_BIGENDIAN
#define BIGENDIAN
#endif
#endif

struct Window *g_pWindow;
struct Screen *g_pScreen;
	
char pal[256*4];

/*
 * This routine is responsible for initializing the implementation
 * specific stuff in a software rendering subsystem.
 */
int
SWimp_Init(void *hInstance, void *wndProc)
{
	return true;
}

/*
 * This initializes the software refresh's implementation specific
 * graphics subsystem.
 */
static qboolean
SWimp_InitGraphics( qboolean fullscreen )
{
	cvar_t *vid_xpos;
	cvar_t *vid_ypos;
	ULONG wflgs;
	Tag screentag;
	static char title[24];
	
	SWimp_Shutdown();

	vid_xpos = ri.Cvar_Get("vid_xpos", "0", CVAR_ARCHIVE);
	vid_ypos = ri.Cvar_Get("vid_ypos", "0", CVAR_ARCHIVE);

#ifdef __MORPHOS__
	GfxBase = (struct GfxBase *)OpenLibrary("graphics.library", 50L);

	if (!GfxBase)
	{
		SWimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open graphics.library");
	}
	
	CyberGfxBase = OpenLibrary("cybergraphics.library", 50L);
	
	if (!CyberGfxBase)
	{
		SWimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open cybergraphics.library");
	}
	
	IntuitionBase = (struct IntuitionBase *)OpenLibrary("intuition.library", 50L);	
	
	if (!IntuitionBase)
	{
		SWimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open intuition.library");
	}
#endif

	if (fullscreen)
	{
	    ULONG ModeID;

	    ModeID = BestCModeIDTags(
			CYBRBIDTG_Depth, 8,
			CYBRBIDTG_NominalWidth, vid.width,
			CYBRBIDTG_NominalHeight, vid.height,
			TAG_DONE);

	    g_pScreen = OpenScreenTags(NULL,
			ModeID != INVALID_ID ? SA_DisplayID : TAG_IGNORE, ModeID,
			SA_Width,		vid.width,
			SA_Height,		vid.height,
			SA_Depth,		8,
			SA_Quiet,		TRUE,
			SA_ShowTitle,	FALSE,
			TAG_DONE);
	}

#ifdef __AROS__
	screentag = g_pScreen ? WA_CustomScreen : TAG_IGNORE;
#else
	screentag = g_pScreen ? WA_PubScreen : TAG_IGNORE;
#endif

	wflgs = WFLG_RMBTRAP | WFLG_REPORTMOUSE | WFLG_ACTIVATE;

	if (g_pScreen)
		wflgs |= WFLG_BORDERLESS;
	else
		wflgs |= WFLG_CLOSEGADGET | WFLG_DRAGBAR | WFLG_DEPTHGADGET;

	snprintf(title, sizeof(title), "Yamagi Quake II %s", VERSION);

	g_pWindow = OpenWindowTags(NULL,
	    g_pScreen ? TAG_IGNORE : WA_Left, vid_xpos->value,
	    g_pScreen ? TAG_IGNORE : WA_Top, vid_ypos->value,
	    WA_Flags,			wflgs,
	    WA_InnerWidth,		vid.width,
	    WA_InnerHeight,		vid.height,
	    screentag,			g_pScreen,
	    WA_Title,			title,
		WA_IDCMP,			/*IDCMP_RAWKEY | IDCMP_MOUSEBUTTONS | */IDCMP_CLOSEWINDOW/*|IDCMP_MOUSEMOVE|IDCMP_DELTAMOVE*/,
		TAG_DONE);
		
	if (!g_pWindow)
	{
		SWimp_Shutdown();
		ri.Sys_Error (ERR_FATAL, "Couldn't create window");
	}

	/* Create the window */
	ri.Vid_NewWindow (vid.width, vid.height);	

	vid.buffer = AllocVec(vid.width * vid.height, MEMF_ANY);
	if (!vid.buffer)
	{
		SWimp_Shutdown();
		ri.Sys_Error (ERR_FATAL, "Couldn't create window");
	}	
	vid.rowbytes = vid.width;

	return true;
}

/*
 * This does an implementation specific copy from the backbuffer to the
 * front buffer.
 */
void
SWimp_EndFrame (void)
{
	if (g_pScreen)
	{
		WritePixelArray(vid.buffer,
			0,
			0,
			vid.rowbytes,
			g_pWindow->RPort,
			0,
			0,
			vid.width,
			vid.height,
			RECTFMT_LUT8);
	}
	else
	{
		WriteLUTPixelArray(vid.buffer,
			0,
			0,
			vid.rowbytes,
			g_pWindow->RPort,
			pal,
			g_pWindow->BorderLeft,
			g_pWindow->BorderTop,
			vid.width,
			vid.height,
			CTABFMT_XRGB8);
	}	
}

/*
 * Changes the video mode
 */
rserr_t
SWimp_SetMode(int *pwidth, int *pheight, int mode, qboolean fullscreen)
{
	ri.Con_Printf(PRINT_ALL, "setting mode %d:", mode);

	/* mode -1 is not in the vid mode table - so we keep the values in pwidth
	   and pheight and don't even try to look up the mode info */
	if ((mode != -1) && !ri.Vid_GetModeInfo(pwidth, pheight, mode))
	{
		ri.Con_Printf(PRINT_ALL, " invalid mode\n");
		return rserr_invalid_mode;
	}

	ri.Con_Printf(PRINT_ALL, " %d %d\n", *pwidth, *pheight);

	if (!SWimp_InitGraphics(fullscreen))
	{
		return rserr_invalid_mode;
	}

	R_GammaCorrectAndSetPalette((const unsigned char *) d_8to24table);
	
	return rserr_ok;
}

/*
 * System specific palette setting routine.
 */
void
SWimp_SetPalette(const unsigned char *palette)
{
	ULONG spal[1+(256*3)+1];
	int i;
	
	if (!palette)
	{
		palette = (const unsigned char *) sw_state.currentpalette;
	}
 
	if (g_pScreen)
	{
		spal[0] = 256<<16;

		for(i=0; i<256; i++)
		{
			spal[1+(i*3)] = ((unsigned int)palette[i*4])<<24;
			spal[2+(i*3)] = ((unsigned int)palette[i*4+1])<<24;
			spal[3+(i*3)] = ((unsigned int)palette[i*4+2])<<24;
		}

		spal[1+(3*256)] = 0;

		LoadRGB32(&g_pScreen->ViewPort, spal);
	}

	for(i=0;i<256;i++)
	{

#ifdef BIGENDIAN
		pal[i*4+0] = 0;
		pal[i*4+1] = palette[i*4+0];
		pal[i*4+2] = palette[i*4+1];
		pal[i*4+3] = palette[i*4+2];
#else
		pal[i*4+3] = 0;
		pal[i*4+2] = palette[i*4+0];
		pal[i*4+1] = palette[i*4+1];
		pal[i*4] = palette[i*4+2];
#endif
	}	
}

/*
 * System specific graphics subsystem shutdown routine.
 */
void
SWimp_Shutdown(void)
{
	if (vid.buffer)
	{
		FreeVec(vid.buffer);
		vid.buffer = 0;
	}
	
    if (g_pWindow)
	{
		// save the window coordinates if not running in fullscreen mode
		if (!g_pScreen)
		{
			/*cvar_t *vid_xpos;
			cvar_t *vid_ypos;

			vid_xpos = ri.Cvar_Get("vid_xpos", "0", CVAR_ARCHIVE);
			vid_ypos = ri.Cvar_Get("vid_ypos", "0", CVAR_ARCHIVE);
			
			ri.Con_Printf(PRINT_ALL,"old coordinates: %d %d\n", vid_xpos->value, vid_ypos->value);			
			
			ri.Cvar_SetValue("vid_xpos", g_pWindow->LeftEdge);
			ri.Cvar_SetValue("vid_ypos", g_pWindow->TopEdge);

			vid_xpos = ri.Cvar_Get("vid_xpos", "0", CVAR_ARCHIVE);
			vid_ypos = ri.Cvar_Get("vid_ypos", "0", CVAR_ARCHIVE);
			
			ri.Con_Printf(PRINT_ALL,"New coordinates: %d %d -> %d %d\n", g_pWindow->LeftEdge, g_pWindow->TopEdge, vid_xpos->value, vid_ypos->value);*/
		}
		CloseWindow(g_pWindow);
		g_pWindow = 0;
	}

    if (g_pScreen)
	{
		CloseScreen(g_pScreen);
	    g_pScreen = 0;
	}
#ifdef __MORPHOS__
	if (IntuitionBase)
	{
		CloseLibrary((struct Library *) IntuitionBase);
		IntuitionBase = 0;
	}
	
	if (GfxBase)
	{
		CloseLibrary((struct Library *) GfxBase);
		GfxBase = 0;
	}
	
	if (CyberGfxBase)
	{
		CloseLibrary(CyberGfxBase);
		CyberGfxBase = 0;
	}
#endif	
}

/*
 * SWimp_AppActivate
 */
void
SWimp_AppActivate( qboolean active )
{
}
