/*
 * Copyright (C) 2012 Szilard Biro
 * Copyright (C) 1997-2001 Id Software, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * =======================================================================
 *
 * This file implements an OpenGL context via TinyGL
 *
 * =======================================================================
 */

#include "../refresh/header/local.h"
#include "../unix/header/glwindow.h"

#include <cybergraphx/cybergraphics.h>
#include <proto/exec.h>
#include <proto/intuition.h>
#include <proto/graphics.h>
#include <proto/cybergraphics.h>
#include <intuition/intuitionbase.h>

#include <GL/gl.h>
#include <tgl/gl.h>
//#include <tgl/gla.h>
#include <proto/tinygl.h>

#ifndef SA_GammaControl
#define SA_GammaControl (SA_Dummy + 123)
#endif

#ifndef SA_3DSupport
#define SA_3DSupport (SA_Dummy + 127)
#endif

#ifndef SA_GammaRed
#define SA_GammaRed (SA_Dummy + 124)
#endif

#ifndef SA_GammaBlue
#define SA_GammaBlue (SA_Dummy + 125)
#endif

#ifndef SA_GammaGreen
#define SA_GammaGreen (SA_Dummy + 126)
#endif

struct Window *g_pWindow;
struct Screen *g_pScreen;
void *g_pCursor;

struct Library *TinyGLBase;
struct IntuitionBase *IntuitionBase;
struct GfxBase *GfxBase;
struct Library *CyberGfxBase;

GLContext *__tglContext;
int contextinit;

glwstate_t glw_state;
qboolean have_stencil = false;

unsigned char gammatable/*[3]*/[256];

/*
 * Initialzes the OpenGL context
 */
int
GLimp_Init(void)
{
	contextinit = 0;
	return true;
}

/* this comes from PowerSDL */
static void CalculateGammaRamp(float gamma, unsigned char *ramp)
{
	int i;
	int value;

	/* 0.0 gamma is all black */
	if ( gamma <= 0.0f ) {
		for ( i=0; i<256; ++i ) {
			ramp[i] = 0;
		}
		return;
	}

	/* 1.0 gamma is identity */
	if ( gamma == 1.0f ) {
		for ( i=0; i<256; ++i ) {
			ramp[i] = (i << 8) | i;
		}
		return;
	}
	
	/* Calculate a real gamma ramp */
    gamma = 1.0f / gamma;
	for ( i=0; i<256; ++i )
	{
		value = (int)(pow((double)i/256.0, gamma)*255.0+0.5);
		if ( value > 255 )
		{
			value = 255;
		}
		ramp[i] = value;
	}
}

/*
 * Sets the hardware gamma
 */
void
UpdateHardwareGamma(void)
{
	CalculateGammaRamp(vid_gamma->value, gammatable);
	
	SetAttrs(g_pScreen,
	SA_GammaRed, gammatable,
	SA_GammaGreen, gammatable,
	SA_GammaBlue, gammatable,
	TAG_DONE);
}

//void GLimp_Shutdown(void);

/*
 * Initializes the OpenGL window
 */
static qboolean
GLimp_InitGraphics(qboolean fullscreen)
{
	int stencil_bits;
	cvar_t *vid_xpos;
	cvar_t *vid_ypos;	
	ULONG wflgs;
	Tag screentag;
	static char title[24];

	GLimp_Shutdown();
	
	vid_xpos = ri.Cvar_Get("vid_xpos", "0", CVAR_ARCHIVE);
	vid_ypos = ri.Cvar_Get("vid_ypos", "0", CVAR_ARCHIVE);

	GfxBase = (struct GfxBase *)OpenLibrary("graphics.library", 50L);

	if (!GfxBase)
	{
		GLimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open graphics.library");
	}
	
	CyberGfxBase = OpenLibrary("cybergraphics.library", 50L);
	
	if (!CyberGfxBase)
	{
		GLimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open cybergraphics.library");
	}
	
	IntuitionBase = (struct IntuitionBase *)OpenLibrary("intuition.library", 50L);	
	
	if (!IntuitionBase)
	{
		GLimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open intuition.library");
	}
	
	TinyGLBase = OpenLibrary("tinygl.library", 0);

	if (!TinyGLBase)
	{
		GLimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Can't open tinygl.library");
	}

	if (fullscreen)
	{
	    ULONG ModeID;

	    ModeID = BestCModeIDTags(
			CYBRBIDTG_Depth, 8,
			CYBRBIDTG_NominalWidth, vid.width,
			CYBRBIDTG_NominalHeight, vid.height,
			TAG_DONE);

	    g_pScreen = OpenScreenTags(NULL,
			ModeID != INVALID_ID ? SA_DisplayID : TAG_IGNORE, ModeID,
			SA_Width,			vid.width,
			SA_Height,			vid.height,
			SA_Depth,			8,
			SA_Quiet,			TRUE,
			SA_ShowTitle,		FALSE,
			SA_GammaControl,	TRUE,
			SA_3DSupport,		TRUE,
			TAG_DONE);
	}

	screentag = g_pScreen ? WA_PubScreen : TAG_IGNORE;

	wflgs = WFLG_RMBTRAP | WFLG_REPORTMOUSE | WFLG_ACTIVATE;

	if (g_pScreen)
		wflgs |= WFLG_BORDERLESS;
	else
		wflgs |= WFLG_CLOSEGADGET | WFLG_DRAGBAR | WFLG_DEPTHGADGET;

	snprintf(title, sizeof(title), "Yamagi Quake II %s", VERSION);

	g_pWindow = OpenWindowTags(NULL,
	    g_pScreen ? TAG_IGNORE : WA_Left, vid_xpos->value,
	    g_pScreen ? TAG_IGNORE : WA_Top, vid_ypos->value,
	    WA_Flags,			wflgs,
	    WA_InnerWidth,		vid.width,
	    WA_InnerHeight,		vid.height,
	    screentag,			g_pScreen,
	    WA_Title,			title,
		WA_IDCMP,			/*IDCMP_RAWKEY | IDCMP_MOUSEBUTTONS | */IDCMP_CLOSEWINDOW/*|IDCMP_MOUSEMOVE|IDCMP_DELTAMOVE*/,
		TAG_DONE);
		
	if (!g_pWindow)
	{
		GLimp_Shutdown();
		ri.Sys_Error(ERR_FATAL, "Couldn't create window");
	}
		
	/* Create the window */
	ri.Vid_NewWindow(vid.width, vid.height);

	__tglContext = GLInit();
	
	if (!__tglContext)
	{
		GLimp_Shutdown();
		ri.Sys_Error (ERR_FATAL, "Couldn't create GL context");
	}

	if (g_pScreen && !(TinyGLBase->lib_Version == 0 && TinyGLBase->lib_Revision < 4))
	{
		contextinit = glAInitializeContextScreen(g_pScreen);
	}
	else
	{
		contextinit = glAInitializeContextWindowed(g_pWindow);
	}
	
	if (!contextinit)
	{
		GLimp_Shutdown();
		ri.Sys_Error (ERR_FATAL, "Couldn't initialize GL context");
	}
	
	/* Initialize the stencil buffer */		
	glGetIntegerv(GL_STENCIL_BITS, &stencil_bits);
	ri.Con_Printf(PRINT_ALL, "Got %d bits of stencil.\n", stencil_bits);

	if (stencil_bits >= 1)
	{
		have_stencil = true;
	}

	/* Initialize hardware gamma */
	if (IntuitionBase->LibNode.lib_Version > 50 || (IntuitionBase->LibNode.lib_Version == 50 && IntuitionBase->LibNode.lib_Revision >= 74))
		gl_state.hwgamma = true;
	else
		gl_state.hwgamma = false;
		
	if (gl_state.hwgamma)
		UpdateHardwareGamma();
	
	return true;
}

/*
 * Swaps the buffers to show the new frame
 */
void
GLimp_EndFrame(void)
{
//	glASwapBuffers();
	GLASwapBuffers(__tglContext);
}

/*
 * Changes the video mode
 */
int
GLimp_SetMode(int *pwidth, int *pheight, int mode, qboolean fullscreen)
{
	ri.Con_Printf(PRINT_ALL, "setting mode %d:", mode);

	/* mode -1 is not in the vid mode table - so we keep the values in pwidth
	   and pheight and don't even try to look up the mode info */
	if ((mode != -1) && !ri.Vid_GetModeInfo(pwidth, pheight, mode))
	{
		ri.Con_Printf(PRINT_ALL, " invalid mode\n");
		return rserr_invalid_mode;
	}

	ri.Con_Printf(PRINT_ALL, " %d %d\n", *pwidth, *pheight);

	if (!GLimp_InitGraphics(fullscreen))
	{
		return rserr_invalid_mode;
	}
	
	return rserr_ok;
}

/*
 * Shuts the GL render backend down
 */
void
GLimp_Shutdown(void)
{
	if (contextinit)
	{
		if (g_pScreen && !(TinyGLBase->lib_Version == 0 && TinyGLBase->lib_Revision < 4))
			glADestroyContextScreen();
		else
			glADestroyContextWindowed();
		contextinit = 0;
	}
			
    if (__tglContext)
    {
		GLClose(__tglContext);        
		__tglContext = 0;
    }
	
    if (g_pWindow)
	{
		CloseWindow(g_pWindow);
		g_pWindow = 0;
	}

    if (g_pScreen)
	{
		CloseScreen(g_pScreen);
	    g_pScreen = 0;
	}

	if (TinyGLBase)
	{
		CloseLibrary(TinyGLBase);
		TinyGLBase = 0;
	}

	if (IntuitionBase)
	{
		CloseLibrary((struct Library *) IntuitionBase);
		IntuitionBase = 0;
	}
	
	if (GfxBase)
	{
		CloseLibrary((struct Library *) GfxBase);
		GfxBase = 0;
	}
	
	if (CyberGfxBase)
	{
		CloseLibrary(CyberGfxBase);
		CyberGfxBase = 0;
	}
	
	gl_state.hwgamma = false;
}

void myglLockArraysEXT(GLint first, GLint count)
{
	GLLockArraysEXT(__tglContext, first, count);
}

void myglUnlockArraysEXT(void)
{
	GLUnlockArraysEXT(__tglContext);
}

void myglPointParameterfEXT(GLenum pname, GLfloat param)
{
	GLPointParameterfEXT(__tglContext, pname, param);
}

void myglPointParameterfvEXT(GLenum pname, GLfloat *params)
{
	GLPointParameterfvEXT(__tglContext, pname, params);
}

void myglMultiTexCoord2fARB(GLenum unit, GLfloat s, GLfloat t)
{
	GLMultiTexCoord2fARB(__tglContext, unit, s, t);
}

void myglActiveTextureARB(GLenum unit)
{
	GLActiveTextureARB(__tglContext, unit);
}

void myglClientActiveTextureARB(GLenum unit)
{
	GLClientActiveTextureARB(__tglContext, unit);
}

void *
qwglGetProcAddress(char *symbol)
{
	/* compiled vertex array */
	if (!strcmp(symbol, "glLockArraysEXT"))
		return (void *)myglLockArraysEXT;
	else if (!strcmp(symbol, "glUnlockArraysEXT"))
		return (void *)myglUnlockArraysEXT;

	/* point parameters */
	else if (!strcmp(symbol, "glPointParameterfEXT"))
		return (void *)myglPointParameterfEXT;
	else if (!strcmp(symbol, "glUnlockArraysEXT"))
		return (void *)myglUnlockArraysEXT;		
		
	/* ARB multitexture */
	else if (!strcmp(symbol, "glMultiTexCoord2fARB"))
		return (void *)myglMultiTexCoord2fARB;
	else if (!strcmp(symbol, "glActiveTextureARB"))
		return (void *)myglActiveTextureARB;
	else if (!strcmp(symbol, "glClientActiveTextureARB"))
		return (void *)myglClientActiveTextureARB;
		
	/* SGIS multitexture */
	/*else if (!strcmp(symbol, "glMTexCoord2fSGIS"))
		return (void *)myglMTexCoord2fSGIS;
	else if (!strcmp(symbol, "glActiveTextureARB"))
		return (void *)myglActiveTextureARB;*/

	return 0;
}

