// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package logger

import (
	"bytes"
)

type Level int

const (
	_             = iota
	Message Level = iota
	Trace   Level = iota
	Debug   Level = iota
	Info    Level = iota
	Warn    Level = iota
	Error   Level = iota
	Fatal   Level = iota
)

var toString = map[Level]string{
	Message: "message",
	Trace:   "trace",
	Debug:   "debug",
	Info:    "info",
	Warn:    "warn",
	Error:   "error",
	Fatal:   "fatal",
}

func (l Level) String() string {
	s, ok := toString[l]
	if ok {
		return s
	}
	return "undefined"
}

type Fun func(string, ...any)

type CaptureInterface interface {
	Interface
	GetBuffer() *bytes.Buffer
	String() string
	Reset()
}

type MessageInterface interface {
	Message(string, ...any)
	Trace(string, ...any)
	Debug(string, ...any)
	Info(string, ...any)
	Warn(string, ...any)
	Error(string, ...any)
	Fatal(string, ...any)
	Log(skip int, level Level, message string, args ...any)
}

type ManageInterface interface {
	SetLevel(level Level)
	GetLevel() Level
}

type Interface interface {
	MessageInterface
	ManageInterface
}
