package util

import (
	"bytes"
	"testing"
)

// We might need them later :p
var (
	EFI_GLOBAL_GUID              = EFIGUID{0x8be4df61, 0x93ca, 0x11d2, [8]uint8{0xaa, 0x0d, 0x00, 0xe0, 0x98, 0x03, 0x2b, 0x8c}}
	PARTITION_SYSTEM_GUID        = EFIGUID{0xC12A7328, 0xF81F, 0x11d2, [8]uint8{0xBA, 0x4B, 0x00, 0xA0, 0xC9, 0x3E, 0xC9, 0x3B}}
	LEGACY_MBR_PARTITION_GUID    = EFIGUID{0x024DEE41, 0x33E7, 0x11d3, [8]uint8{0x9D, 0x69, 0x00, 0x08, 0xC7, 0x81, 0xF3, 0x9F}}
	PARTITION_MSFT_RESERVED_GUID = EFIGUID{0xE3C9E316, 0x0B5C, 0x4DB8, [8]uint8{0x81, 0x7D, 0xF9, 0x2D, 0xF0, 0x02, 0x15, 0xAE}}
	PARTITION_BASIC_DATA_GUID    = EFIGUID{0xEBD0A0A2, 0xB9E5, 0x4433, [8]uint8{0x87, 0xC0, 0x68, 0xB6, 0xB7, 0x26, 0x99, 0xC7}}
	PARTITION_LINUX_RAID_GUID    = EFIGUID{0xa19d880f, 0x05fc, 0x4d3b, [8]uint8{0xA0, 0x06, 0x74, 0x3f, 0x0f, 0x84, 0x91, 0x1e}}
	PARTITION_LINUX_SWAP_GUID    = EFIGUID{0x0657fd6d, 0xa4ab, 0x43c4, [8]uint8{0x84, 0xE5, 0x09, 0x33, 0xc8, 0x4b, 0x4f, 0x4f}}
	PARTITION_LINUX_LVM_GUID     = EFIGUID{0xe6d6d379, 0xf507, 0x44c2, [8]uint8{0xa2, 0x3c, 0x23, 0x8f, 0x2a, 0x3d, 0xf9, 0x28}}
)

var TestData = []struct {
	GUID  EFIGUID
	Bytes []byte
}{
	{
		GUID:  EFI_GLOBAL_GUID,
		Bytes: []byte{0x8b, 0xe4, 0xdf, 0x61, 0x93, 0xca, 0x11, 0xd2, 0xaa, 0xd, 0x0, 0xe0, 0x98, 0x3, 0x2b, 0x8c},
	},
	{
		GUID:  PARTITION_SYSTEM_GUID,
		Bytes: []byte{0xc1, 0x2a, 0x73, 0x28, 0xf8, 0x1f, 0x11, 0xd2, 0xba, 0x4b, 0x0, 0xa0, 0xc9, 0x3e, 0xc9, 0x3b},
	},
	{
		GUID:  LEGACY_MBR_PARTITION_GUID,
		Bytes: []byte{0x2, 0x4d, 0xee, 0x41, 0x33, 0xe7, 0x11, 0xd3, 0x9d, 0x69, 0x0, 0x8, 0xc7, 0x81, 0xf3, 0x9f},
	},
	{
		GUID:  PARTITION_MSFT_RESERVED_GUID,
		Bytes: []byte{0xe3, 0xc9, 0xe3, 0x16, 0xb, 0x5c, 0x4d, 0xb8, 0x81, 0x7d, 0xf9, 0x2d, 0xf0, 0x2, 0x15, 0xae},
	},

	{
		GUID:  PARTITION_BASIC_DATA_GUID,
		Bytes: []byte{0xeb, 0xd0, 0xa0, 0xa2, 0xb9, 0xe5, 0x44, 0x33, 0x87, 0xc0, 0x68, 0xb6, 0xb7, 0x26, 0x99, 0xc7},
	},
	{
		GUID:  PARTITION_LINUX_RAID_GUID,
		Bytes: []byte{0xa1, 0x9d, 0x88, 0xf, 0x5, 0xfc, 0x4d, 0x3b, 0xa0, 0x6, 0x74, 0x3f, 0xf, 0x84, 0x91, 0x1e},
	},
	{
		GUID:  PARTITION_LINUX_SWAP_GUID,
		Bytes: []byte{0x6, 0x57, 0xfd, 0x6d, 0xa4, 0xab, 0x43, 0xc4, 0x84, 0xe5, 0x9, 0x33, 0xc8, 0x4b, 0x4f, 0x4f},
	},
	{
		GUID:  PARTITION_LINUX_LVM_GUID,
		Bytes: []byte{0xe6, 0xd6, 0xd3, 0x79, 0xf5, 0x7, 0x44, 0xc2, 0xa2, 0x3c, 0x23, 0x8f, 0x2a, 0x3d, 0xf9, 0x28},
	},
}

func TestParseGUIDToBytes(t *testing.T) {
	for _, data := range TestData {
		b := GUIDToBytes(&data.GUID)
		if r := bytes.Equal(b, data.Bytes); !r {
			t.Errorf("Expected true, got false")
		}
	}
}

func TestParseBytesToGuid(t *testing.T) {
	for _, data := range TestData {
		g := BytesToGUID(data.Bytes)
		if g.Format() != data.GUID.Format() {
			t.Errorf("GUID doesn't match: Got %s, wanted %s", g.Format(), data.GUID.Format())
		}
	}
}

func TestParseFormatRoundTrip(t *testing.T) {
	input := "8be4df61-93ca-11d2-aa0d-00e098032b8c"
	guid := StringToGUID(input)

	if guid.Format() != input {
		t.Errorf("GUID failed round trip: Got %s, wanted %s", guid.Format(), input)
	}
}

func TestParseFormatRoundTripWithLeadingZero(t *testing.T) {
	input := "4a67b082-0a4c-41cf-b6c7-440b29bb8c4f"
	guid := StringToGUID(input)

	if guid.Format() != input {
		t.Errorf("GUID failed round trip: Got %s, wanted %s", guid.Format(), input)
	}
}
