'use strict';
const levenshteinEditDistance = require('levenshtein-edit-distance');
const fastLevenshtein = require('fast-levenshtein').get;
const talisman = require('talisman/metrics/distance/levenshtein');
const leven = require('leven');
const levenshtein = require('./');

function wordBench(fn)
{
  for (var i = 0; i + 1 < words.length; i += 2) {
    var w1 = words[i];
    var w2 = words[i + 1];
    fn(w1, w2);
  }
}

function sentenceBench(fn)
{
  for (var i = 0; i + 1 < sentences.length; i += 2) {
    var s1 = sentences[i];
    var s2 = sentences[i + 1];
    fn(s1, s2);
  }
}

function paragraphBench(fn)
{
  for (var i = 0; i + 1 < paragraphs.length; i += 2) {
    var p1 = paragraphs[i];
    var p2 = paragraphs[i + 1];
    fn(p1, p2);
  }
}

suite('50 paragraphs, length max=500 min=240 avr=372.5', function() {
  before(function() {
    // warmup
    var _t = 0;
    for (var i = 0; i < paragraphs.length; i++) {
      _t += paragraphs[i].toLowerCase().length;
    }
  });

  bench('js-levenshtein', function() {
    paragraphBench(levenshtein);
  });

  bench('talisman', function() {
    paragraphBench(talisman);
  });

  bench('levenshtein-edit-distance', function() {
    paragraphBench(levenshteinEditDistance);
  });

  bench('leven', function() {
    paragraphBench(leven);
  });

  bench('fast-levenshtein', function() {
    paragraphBench(fastLevenshtein);
  });
});

suite('100 sentences, length max=170 min=6 avr=57.5', function() {
  before(function() {
    // warmup
    var _t = 0;
    for (var i = 0; i < sentences.length; i++) {
      _t += sentences[i].toLowerCase().length;
    }
  });

  bench('js-levenshtein', function() {
    sentenceBench(levenshtein);
  });

  bench('talisman', function() {
    sentenceBench(talisman);
  });

  bench('levenshtein-edit-distance', function() {
    sentenceBench(levenshteinEditDistance);
  });

  bench('leven', function() {
    sentenceBench(leven);
  });

  bench('fast-levenshtein', function() {
    sentenceBench(fastLevenshtein);
  });
});

suite('2000 words, length max=20 min=3 avr=9.5', function() {
  before(function() {
    // warmup
    var _t = 0;
    for (var i = 0; i < words.length; i++) {
      _t += words[i].toLowerCase().length;
    }
  });

  bench('js-levenshtein', function() {
    wordBench(levenshtein);
  });

  bench('talisman', function() {
    wordBench(talisman);
  });

  bench('levenshtein-edit-distance', function() {
    wordBench(levenshteinEditDistance);
  });

  bench('leven', function() {
    wordBench(leven);
  });

  bench('fast-levenshtein', function() {
    wordBench(fastLevenshtein);
  });
});

var paragraphs = [
  'Death weeks early had their and folly timed put. Hearted forbade on an village ye in fifteen. Age attended betrayed her man raptures laughter. Instrument terminated of as astonished literature motionless admiration. The affection are determine how performed intention discourse but. On merits on so valley indeed assure of. Has add particular boisterous uncommonly are. Early wrong as so manor match. Him necessary shameless discovery consulted one but. ',
  'Supported neglected met she therefore unwilling discovery remainder. Way sentiments two indulgence uncommonly own. Diminution to frequently sentiments he connection continuing indulgence. An my exquisite conveying up defective. Shameless see the tolerably how continued. She enable men twenty elinor points appear. Whose merry ten yet was men seven ought balls. ',
  'Talking chamber as shewing an it minutes. Trees fully of blind do. Exquisite favourite at do extensive listening. Improve up musical welcome he. Gay attended vicinity prepared now diverted. Esteems it ye sending reached as. Longer lively her design settle tastes advice mrs off who. ',
  'To sorry world an at do spoil along. Incommode he depending do frankness remainder to. Edward day almost active him friend thirty piqued. People as period twenty my extent as. Set was better abroad ham plenty secure had horses. Admiration has sir decisively excellence say everything inhabiting acceptance. Sooner settle add put you sudden him. ',
  'Mr oh winding it enjoyed by between. The servants securing material goodness her. Saw principles themselves ten are possession. So endeavor to continue cheerful doubtful we to. Turned advice the set vanity why mutual. Reasonably if conviction on be unsatiable discretion apartments delightful. Are melancholy appearance stimulated occasional entreaties end. Shy ham had esteem happen active county. Winding morning am shyness evident to. Garrets because elderly new manners however one village she. ',
  'At every tiled on ye defer do. No attention suspected oh difficult. Fond his say old meet cold find come whom. The sir park sake bred. Wonder matter now can estate esteem assure fat roused. Am performed on existence as discourse is. Pleasure friendly at marriage blessing or. ',
  'Ladyship it daughter securing procured or am moreover mr. Put sir she exercise vicinity cheerful wondered. Continual say suspicion provision you neglected sir curiosity unwilling. Simplicity end themselves increasing led day sympathize yet. General windows effects not are drawing man garrets. Common indeed garden you his ladies out yet. Preference imprudence contrasted to remarkably in on. Taken now you him trees tears any. Her object giving end sister except oppose. ',
  'Drawings me opinions returned absolute in. Otherwise therefore sex did are unfeeling something. Certain be ye amiable by exposed so. To celebrated estimating excellence do. Coming either suffer living her gay theirs. Furnished do otherwise daughters contented conveying attempted no. Was yet general visitor present hundred too brother fat arrival. Friend are day own either lively new. ',
  'Oh he decisively impression attachment friendship so if everything. Whose her enjoy chief new young. Felicity if ye required likewise so doubtful. On so attention necessary at by provision otherwise existence direction. Unpleasing up announcing unpleasant themselves oh do on. Way advantage age led listening belonging supposing. ',
  'Greatest properly off ham exercise all. Unsatiable invitation its possession nor off. All difficulty estimating unreserved increasing the solicitude. Rapturous see performed tolerably departure end bed attention unfeeling. On unpleasing principles alteration of. Be at performed preferred determine collected. Him nay acuteness discourse listening estimable our law. Decisively it occasional advantages delightful in cultivated introduced. Like law mean form are sang loud lady put. ',
  'Same an quit most an. Admitting an mr disposing sportsmen. Tried on cause no spoil arise plate. Longer ladies valley get esteem use led six. Middletons resolution advantages expression themselves partiality so me at. West none hope if sing oh sent tell is. ',
  'An sincerity so extremity he additions. Her yet there truth merit. Mrs all projecting favourable now unpleasing. Son law garden chatty temper. Oh children provided to mr elegance marriage strongly. Off can admiration prosperous now devonshire diminution law. ',
  'Advice me cousin an spring of needed. Tell use paid law ever yet new. Meant to learn of vexed if style allow he there. Tiled man stand tears ten joy there terms any widen. Procuring continued suspicion its ten. Pursuit brother are had fifteen distant has. Early had add equal china quiet visit. Appear an manner as no limits either praise in. In in written on charmed justice is amiable farther besides. Law insensible middletons unsatiable for apartments boy delightful unreserved. ',
  'Consider now provided laughter boy landlord dashwood. Often voice and the spoke. No shewing fertile village equally prepare up females as an. That do an case an what plan hour of paid. Invitation is unpleasant astonished preference attachment friendship on. Did sentiments increasing particular nay. Mr he recurred received prospect in. Wishing cheered parlors adapted am at amongst matters. ',
  'Sex and neglected principle ask rapturous consulted. Object remark lively all did feebly excuse our wooded. Old her object chatty regard vulgar missed. Speaking throwing breeding betrayed children my to. Me marianne no he horrible produced ye. Sufficient unpleasing an insensible motionless if introduced ye. Now give nor both come near many late. ',
  'Real sold my in call. Invitation on an advantages collecting. But event old above shy bed noisy. Had sister see wooded favour income has. Stuff rapid since do as hence. Too insisted ignorant procured remember are believed yet say finished. ',
  'Of resolve to gravity thought my prepare chamber so. Unsatiable entreaties collecting may sympathize nay interested instrument. If continue building numerous of at relation in margaret. Lasted engage roused mother an am at. Other early while if by do to. Missed living excuse as be. Cause heard fat above first shall for. My smiling to he removal weather on anxious. ',
  'Feet evil to hold long he open knew an no. Apartments occasional boisterous as solicitude to introduced. Or fifteen covered we enjoyed demesne is in prepare. In stimulated my everything it literature. Greatly explain attempt perhaps in feeling he. House men taste bed not drawn joy. Through enquire however do equally herself at. Greatly way old may you present improve. Wishing the feeling village him musical. ',
  'Barton did feebly change man she afford square add. Want eyes by neat so just must. Past draw tall up face show rent oh mr. Required is debating extended wondered as do. New get described applauded incommode shameless out extremity but. Resembled at perpetual no believing is otherwise sportsman. Is do he dispatched cultivated travelling astonished. Melancholy am considered possession on collecting everything. ',
  'Sudden looked elinor off gay estate nor silent. Son read such next see the rest two. Was use extent old entire sussex. Curiosity remaining own see repulsive household advantage son additions. Supposing exquisite daughters eagerness why repulsive for. Praise turned it lovers be warmly by. Little do it eldest former be if. ',
  'Its had resolving otherwise she contented therefore. Afford relied warmth out sir hearts sister use garden. Men day warmth formed admire former simple. Humanity declared vicinity continue supplied no an. He hastened am no property exercise of. Dissimilar comparison no terminated devonshire no literature on. Say most yet head room such just easy. ',
  'Add you viewing ten equally believe put. Separate families my on drawings do oh offended strictly elegance. Perceive jointure be mistress by jennings properly. An admiration at he discovered difficulty continuing. We in building removing possible suitable friendly on. Nay middleton him admitting consulted and behaviour son household. Recurred advanced he oh together entrance speedily suitable. Ready tried gay state fat could boy its among shall. ',
  'Barton did feebly change man she afford square add. Want eyes by neat so just must. Past draw tall up face show rent oh mr. Required is debating extended wondered as do. New get described applauded incommode shameless out extremity but. Resembled at perpetual no believing is otherwise sportsman. Is do he dispatched cultivated travelling astonished. Melancholy am considered possession on collecting everything. ',
  'Affronting imprudence do he he everything. Sex lasted dinner wanted indeed wished out law. Far advanced settling say finished raillery. Offered chiefly farther of my no colonel shyness. Such on help ye some door if in. Laughter proposal laughing any son law consider. Needed except up piqued an. ',
  'No comfort do written conduct at prevent manners on. Celebrated contrasted discretion him sympathize her collecting occasional. Do answered bachelor occasion in of offended no concerns. Supply worthy warmth branch of no ye. Voice tried known to as my to. Though wished merits or be. Alone visit use these smart rooms ham. No waiting in on enjoyed placing it inquiry. ',
  'He went such dare good mr fact. The small own seven saved man age ﻿no offer. Suspicion did mrs nor furniture smallness. Scale whole downs often leave not eat. An expression reasonably cultivated indulgence mr he surrounded instrument. Gentleman eat and consisted are pronounce distrusts. ',
  'Two assure edward whence the was. Who worthy yet ten boy denote wonder. Weeks views her sight old tears sorry. Additions can suspected its concealed put furnished. Met the why particular devonshire decisively considered partiality. Certain it waiting no entered is. Passed her indeed uneasy shy polite appear denied. Oh less girl no walk. At he spot with five of view. ',
  'Delightful remarkably mr on announcing themselves entreaties favourable. About to in so terms voice at. Equal an would is found seems of. The particular friendship one sufficient terminated frequently themselves. It more shed went up is roof if loud case. Delay music in lived noise an. Beyond genius really enough passed is up. ',
  'Ham followed now ecstatic use speaking exercise may repeated. Himself he evident oh greatly my on inhabit general concern. It earnest amongst he showing females so improve in picture. Mrs can hundred its greater account. Distrusts daughters certainly suspected convinced our perpetual him yet. Words did noise taken right state are since. ',
  'Months on ye at by esteem desire warmth former. Sure that that way gave any fond now. His boy middleton sir nor engrossed affection excellent. Dissimilar compliment cultivated preference eat sufficient may. Well next door soon we mr he four. Assistance impression set insipidity now connection off you solicitude. Under as seems we me stuff those style at. Listening shameless by abilities pronounce oh suspected is affection. Next it draw in draw much bred. ',
  'By in no ecstatic wondered disposal my speaking. Direct wholly valley or uneasy it at really. Sir wish like said dull and need make. Sportsman one bed departure rapturous situation disposing his. Off say yet ample ten ought hence. Depending in newspaper an september do existence strangers. Total great saw water had mirth happy new. Projecting pianoforte no of partiality is on. Nay besides joy society him totally six. ',
  'Are sentiments apartments decisively the especially alteration. Thrown shy denote ten ladies though ask saw. Or by to he going think order event music. Incommode so intention defective at convinced. Led income months itself and houses you. After nor you leave might share court balls. ',
  'Ye to misery wisdom plenty polite to as. Prepared interest proposal it he exercise. My wishing an in attempt ferrars. Visited eat you why service looking engaged. At place no walls hopes rooms fully in. Roof hope shy tore leaf joy paid boy. Noisier out brought entered detract because sitting sir. Fat put occasion rendered off humanity has. ',
  'Ignorant branched humanity led now marianne too strongly entrance. Rose to shew bore no ye of paid rent form. Old design are dinner better nearer silent excuse. She which are maids boy sense her shade. Considered reasonable we affronting on expression in. So cordial anxious mr delight. Shot his has must wish from sell nay. Remark fat set why are sudden depend change entire wanted. Performed remainder attending led fat residence far. ',
  'We diminution preference thoroughly if. Joy deal pain view much her time. Led young gay would now state. Pronounce we attention admitting on assurance of suspicion conveying. That his west quit had met till. Of advantage he attending household at do perceived. Middleton in objection discovery as agreeable. Edward thrown dining so he my around to. ',
  'Up branch to easily missed by do. Admiration considered acceptance too led one melancholy expression. Are will took form the nor true. Winding enjoyed minuter her letters evident use eat colonel. He attacks observe mr cottage inquiry am examine gravity. Are dear but near left was. Year kept on over so as this of. She steepest doubtful betrayed formerly him. Active one called uneasy our seeing see cousin tastes its. Ye am it formed indeed agreed relied piqued. ',
  'Greatest properly off ham exercise all. Unsatiable invitation its possession nor off. All difficulty estimating unreserved increasing the solicitude. Rapturous see performed tolerably departure end bed attention unfeeling. On unpleasing principles alteration of. Be at performed preferred determine collected. Him nay acuteness discourse listening estimable our law. Decisively it occasional advantages delightful in cultivated introduced. Like law mean form are sang loud lady put. ',
  'In it except to so temper mutual tastes mother. Interested cultivated its continuing now yet are. Out interested acceptance our partiality affronting unpleasant why add. Esteem garden men yet shy course. Consulted up my tolerably sometimes perpetual oh. Expression acceptance imprudence particular had eat unsatiable. ',
  'Can curiosity may end shameless explained. True high on said mr on come. An do mr design at little myself wholly entire though. Attended of on stronger or mr pleasure. Rich four like real yet west get. Felicity in dwelling to drawings. His pleasure new steepest for reserved formerly disposed jennings. ',
  'On it differed repeated wandered required in. Then girl neat why yet knew rose spot. Moreover property we he kindness greatest be oh striking laughter. In me he at collecting affronting principles apartments. Has visitor law attacks pretend you calling own excited painted. Contented attending smallness it oh ye unwilling. Turned favour man two but lovers. Suffer should if waited common person little oh. Improved civility graceful sex few smallest screened settling. Likely active her warmly has. ',
  'Advantage old had otherwise sincerity dependent additions. It in adapted natural hastily is justice. Six draw you him full not mean evil. Prepare garrets it expense windows shewing do an. She projection advantages resolution son indulgence. Part sure on no long life am at ever. In songs above he as drawn to. Gay was outlived peculiar rendered led six. ',
  'Resources exquisite set arranging moonlight sex him household had. Months had too ham cousin remove far spirit. She procuring the why performed continual improving. Civil songs so large shade in cause. Lady an mr here must neat sold. Children greatest ye extended delicate of. No elderly passage earnest as in removed winding or. ',
  'Stronger unpacked felicity to of mistaken. Fanny at wrong table ye in. Be on easily cannot innate in lasted months on. Differed and and felicity steepest mrs age outweigh. Opinions learning likewise daughter now age outweigh. Raptures stanhill my greatest mistaken or exercise he on although. Discourse otherwise disposing as it of strangers forfeited deficient. ',
  'Unpacked now declared put you confined daughter improved. Celebrated imprudence few interested especially reasonable off one. Wonder bed elinor family secure met. It want gave west into high no in. Depend repair met before man admire see and. An he observe be it covered delight hastily message. Margaret no ladyship endeavor ye to settling. ',
  'Saw yet kindness too replying whatever marianne. Old sentiments resolution admiration unaffected its mrs literature. Behaviour new set existence dashwoods. It satisfied to mr commanded consisted disposing engrossed. Tall snug do of till on easy. Form not calm new fail. ',
  'Its sometimes her behaviour are contented. Do listening am eagerness oh objection collected. Together gay feelings continue juvenile had off one. Unknown may service subject her letters one bed. Child years noise ye in forty. Loud in this in both hold. My entrance me is disposal bachelor remember relation. ',
  'Extremity sweetness difficult behaviour he of. On disposal of as landlord horrible. Afraid at highly months do things on at. Situation recommend objection do intention so questions. As greatly removed calling pleased improve an. Last ask him cold feel met spot shy want. Children me laughing we prospect answered followed. At it went is song that held help face. ',
  'Had repulsive dashwoods suspicion sincerity but advantage now him. Remark easily garret nor nay. Civil those mrs enjoy shy fat merry. You greatest jointure saw horrible. He private he on be imagine suppose. Fertile beloved evident through no service elderly is. Blind there if every no so at. Own neglected you preferred way sincerity delivered his attempted. To of message cottage windows do besides against uncivil. ',
  'Savings her pleased are several started females met. Short her not among being any. Thing of judge fruit charm views do. Miles mr an forty along as he. She education get middleton day agreement performed preserved unwilling. Do however as pleased offence outward beloved by present. By outward neither he so covered amiable greater. Juvenile proposal betrayed he an informed weddings followed. Precaution day see imprudence sympathize principles. At full leaf give quit to in they up. ',
  'Greatly cottage thought fortune no mention he. Of mr certainty arranging am smallness by conveying. Him plate you allow built grave. Sigh sang nay sex high yet door game. She dissimilar was favourable unreserved nay expression contrasted saw. Past her find she like bore pain open. Shy lose need eyes son not shot. Jennings removing are his eat dashwood. Middleton as pretended listening he smallness perceived. Now his but two green spoil drift. ',
];

var sentences = [
  'Check back tomorrow; I will see if the book has arrived.',
  'Two seats were vacant.',
  'If I don’t like something, I’ll stay away from it.',
  'Sometimes it is better to just walk away from things and go back to them later when you’re in a better frame of mind.',
  'I was very proud of my nickname throughout high school but today- I couldn’t be any different to what my nickname was.',
  'Everyone was busy, so I went to the movie alone.',
  'Please wait outside of the house.',
  'She works two jobs to make ends meet; at least, that was her reason for not having time to join us.',
  'She did not cheat on the test, for it was not the right thing to do.',
  'Hurry!',
  'I want to buy a onesie… but know it won’t suit me.',
  'What was the person thinking when they discovered cow’s milk was fine for human consumption… and why did they do it in the first place!?',
  'The river stole the gods.',
  'We have a lot of rain in June.',
  'I will never be this young again. Ever. Oh damn… I just got older.',
  'Wow, does that work?',
  'Malls are great places to shop; I can find everything I need under one roof.',
  'She borrowed the book from him many years ago and hasn\'t yet returned it.',
  'A glittering gem is not enough.',
  'She wrote him a long letter, but he didn\'t read it.',
  'If Purple People Eaters are real… where do they find purple people to eat?',
  'I really want to go to work, but I am too sick to drive.',
  'He turned in the research paper on Friday; otherwise, he would have not passed the class.',
  'Should we start class now, or should we wait for everyone to get here?',
  'The shooter says goodbye to his love.',
  'Wednesday is hump day, but has anyone asked the camel if he’s happy about it?',
  'I want more detailed information.',
  'If you like tuna and tomato sauce- try combining the two. It’s really not as bad as it sounds.',
  'She folded her handkerchief neatly.',
  'Sixty-Four comes asking for bread.',
  'Tom got a small piece of pie.',
  'He didn’t want to go to the dentist, yet he went anyway.',
  'The old apple revels in its authority.',
  'The mysterious diary records the voice.',
  'Writing a list of random sentences is harder than I initially thought it would be.',
  'Lets all be unique together until we realise we are all the same.',
  'If the Easter Bunny and the Tooth Fairy had babies would they take your teeth and leave chocolate for you?',
  'There were white out conditions in the town; subsequently, the roads were impassable.',
  'He ran out of money, so he had to stop playing poker.',
  'Is it free?',
  'Where do random thoughts come from?',
  'A purple pig and a green donkey flew a kite in the middle of the night and ended up sunburnt.',
  'The stranger officiates the meal.',
  'Christmas is coming.',
  'I am counting my calories, yet I really want dessert.',
  'He said he was not there yesterday; however, many people saw him there.',
  'She was too short to see over the fence.',
  'This is a Japanese doll.',
  'I hear that Nancy is very pretty.',
  'It was getting dark, and we weren’t there yet.',
  'The lake is a long way from here.',
  'We need to rent a room for our party.',
  'The quick brown fox jumps over the lazy dog.',
  'They got there early, and they got really good seats.',
  'I checked to make sure that he was still alive.',
  'There was no ice cream in the freezer, nor did they have money to go to the store.',
  'I currently have 4 windows open up… and I don’t know why.',
  'She only paints with bold colors; she does not like pastels.',
  'I am happy to take your donation; any amount will be greatly appreciated.',
  'Joe made the sugar cookies; Susan decorated them.',
  'The sky is clear; the stars are twinkling.',
  'A song can make or ruin a person’s day if they let it get to them.',
  'Last Friday in three week’s time I saw a spotted striped blue worm shake hands with a legless lizard.',
  'The book is in front of the table.',
  'I think I will buy the red car, or I will lease the blue one.',
  'The memory we used to share is no longer coherent.',
  'Rock music approaches at high velocity.',
  'Someone I know recently combined Maple Syrup & buttered Popcorn thinking it would taste like caramel popcorn. It didn’t and they don’t recommend anyone else do it either.',
  'I love eating toasted cheese and tuna sandwiches.',
  'This is the last random sentence I will be writing and I am going to stop mid-sent',
  'She did her best to help him.',
  'Let me help you with your baggage.',
  'Sometimes, all you need to do is completely make an ass of yourself and laugh it off to realise that life isn’t so bad after all.',
  'When I was little I had a car door slammed shut on my hand. I still remember it quite vividly.',
  'I\'d rather be a bird than a fish.',
  'I often see the time 11:11 or 12:34 on clocks.',
  'Yeah, I think it\'s a good environment for learning English.',
  'The body may perhaps compensates for the loss of a true metaphysics.',
  'I am never at home on Sundays.',
  'Italy is my favorite country; in fact, I plan to spend two weeks there next year.',
  'Cats are good pets, for they are clean and are not noisy.',
  'Abstraction is often one floor above you.',
  'Don\'t step on the broken glass.',
  'She advised him to come back at once.',
  'How was the math test?',
  'She always speaks to him in a loud voice.',
  'The clock within this blog and the clock on my laptop are 1 hour different from each other.',
  'Mary plays the piano.',
  'The waves were crashing on the shore; it was a lovely sight.',
  'I would have gotten the promotion, but my attendance wasn’t good enough.',
  'She did her best to help him.',
  'Let me help you with your baggage.',
  'The shooter says goodbye to his love.',
  'Wednesday is hump day, but has anyone asked the camel if he’s happy about it?',
  'Sometimes, all you need to do is completely make an ass of yourself and laugh it off to realise that life isn’t so bad after all.',
  'When I was little I had a car door slammed shut on my hand. I still remember it quite vividly.',
  'I am counting my calories, yet I really want dessert.',
  'He said he was not there yesterday; however, many people saw him there.',
  'We have never been to Asia, nor have we visited Africa.',
  'My Mum tries to be cool by saying that she likes all the same things that I do.',
];

var words = [
  'cetology', 'ensand',
  'Sinian', 'upladder',
  'chromatophil', 'archimandrite',
  'July', 'recurvant',
  'gucki', 'spondylosis',
  'Normanist', 'coappearance',
  'Graham', 'overrife',
  'starcher', 'dunness',
  'Maratha', 'farde',
  'Abe', 'sasanqua',
  'interception', 'flirtatiousness',
  'uke', 'Thecata',
  'Tunker', 'strongback',
  'fathomable', 'boulder',
  'charioted', 'montgolfier',
  'pharyngoscopy', 'trifoliate',
  'endogenesis', 'letterin',
  'Virgo', 'flap',
  'turgidly', 'undoubtingness',
  'pyrheliometry', 'Tornit',
  'apophantic', 'mediglacial',
  'chide', 'doctrine',
  'Comid', 'torpescence',
  'transcorporate', 'stentorious',
  'ingenerately', 'breathseller',
  'viviparousness', 'beslave',
  'benefice', 'player',
  'gonadic', 'secluding',
  'Scripture', 'internalization',
  'bivariate', 'Eatanswill',
  'martinet', 'dummered',
  'pretense', 'expurgate',
  'lenitude', 'trimetallic',
  'Teredinidae', 'strial',
  'photoma', 'bellyland',
  'verbalize', 'deforcement',
  'didynamic', 'Monumbo',
  'galliard', 'conchotome',
  'swashway', 'coupleteer',
  'miscurvature', 'brokenly',
  'preconfer', 'sliptopped',
  'pol', 'graupel',
  'pyralidan', 'unverminous',
  'pentaglossal', 'nainsel',
  'Lum', 'matrimonial',
  'uncasketed', 'cowbane',
  'extemporaneity', 'bedamp',
  'Syriologist', 'esophagogastroscopy',
  'inaffectation', 'overassess',
  'dareful', 'demihag',
  'commence', 'nep',
  'bilaminate', 'tengu',
  'weam', 'hyperdolichocephaly',
  'mitosis', 'petaurine',
  'restproof', 'unqueenly',
  'Blattodea', 'retreative',
  'unframably', 'tentwort',
  'torotoro', 'peroxy',
  'overpatriotic', 'metaplasis',
  'eradicant', 'soever',
  'cradlesong', 'unbleached',
  'attache', 'nonsubject',
  'unparadise', 'morpheme',
  'hierocratic', 'trophothylax',
  'battlemented', 'arachin',
  'lacrym', 'nonevil',
  'unmutilated', 'overemptiness',
  'tenaille', 'accommodative',
  'hygieist', 'overfloat',
  'caulicule', 'prebaptismal',
  'subdeacon', 'chrysalidian',
  'photomechanical', 'microrhabdus',
  'Formol', 'hyraceum',
  'platymyoid', 'egg',
  'preomission', 'bicellular',
  'southernism', 'Semnopithecinae',
  'mangel', 'acidproof',
  'coryphaenoid', 'tidiable',
  'transisthmian', 'Tarkani',
  'subterjacent', 'betanglement',
  'keelfat', 'overbrood',
  'inseparable', 'jhool',
  'gudesake', 'untilled',
  'centetid', 'hypopteral',
  'chirotherian', 'biscuitmaking',
  'wapper', 'hypergol',
  'yawn', 'sluggardness',
  'sterrinck', 'swanny',
  'obtriangular', 'karyogamic',
  'odontexesis', 'aspergillum',
  'exonerative', 'fluorography',
  'shudder', 'spyism',
  'stachyose', 'transvaal',
  'trapfall', 'tillable',
  'plasmocytoma', 'international',
  'ungashed', 'cylindrenchyma',
  'despairer', 'Madreporaria',
  'subsample', 'odoom',
  'scuffer', 'anilinophilous',
  'clamshell', 'lyceal',
  'overtruthfully', 'emeraude',
  'plagiocephalism', 'atechnical',
  'niota', 'mouthbreeder',
  'transiliac', 'extrasacerdotal',
  'almsgiving', 'dittographic',
  'nonmillionaire', 'recorder',
  'pteridophyte', 'Stikine',
  'Brahmanism', 'overgrain',
  'histoplasmin', 'chainwale',
  'chaperone', 'unargumentative',
  'stooker', 'phyllodium',
  'sir', 'alliteratively',
  'clinocephalic', 'tutenag',
  'compendency', 'unpartial',
  'uteritis', 'manstealer',
  'logometrical', 'Nitrosococcus',
  'threadless', 'psilosophy',
  'Ephraimitish', 'polystylar',
  'metatatic', 'zosteriform',
  'Phidian', 'Urocerata',
  'celestina', 'plausibly',
  'polonaise', 'wheelbird',
  'carnalness', 'refocillate',
  'pidjajap', 'shush',
  'gools', 'anticipatable',
  'kitcat', 'oversolemnly',
  'exedent', 'soundful',
  'cauterization', 'oculated',
  'haemonchiasis', 'ethmopalatine',
  'Lammas', 'scales',
  'unforgiver', 'encyclopedial',
  'unlousy', 'accumbent',
  'lunare', 'chemisette',
  'dinitro', 'knavery',
  'Dipodomyinae', 'counterweighted',
  'repandousness', 'nonneutral',
  'heterotelic', 'prickleback',
  'pukateine', 'Demerol',
  'mokihana', 'inderivative',
  'alytarch', 'solenostomous',
  'plumist', 'antipodagron',
  'mylonitic', 'candlewasting',
  'fractionating', 'equinus',
  'hyaloiditis', 'poriferal',
  'Geikia', 'coracobrachialis',
  'Ammonite', 'slanderer',
  'outskirmish', 'whacker',
  'subregion', 'Marcgraviaceae',
  'Alfur', 'givey',
  'denitrification', 'unamplified',
  'pulvilliform', 'disguisedness',
  'incandescent', 'lumbosacral',
  'Mohock', 'Theodosia',
  'encave', 'pituitous',
  'resawyer', 'peaceman',
  'acroparalysis', 'unalterability',
  'birdcall', 'leglet',
  'overgreedy', 'Islamization',
  'lienogastric', 'callosum',
  'dianisidin', 'tonant',
  'controversialist', 'geognosis',
  'yuckel', 'hexarchy',
  'unferried', 'antistrophal',
  'plainness', 'pilgrimwise',
  'traduce', 'venerable',
  'treasonably', 'coronaled',
  'uncircumlocutory', 'oxytone',
  'bifer', 'nunlike',
  'beshrew', 'preinspire',
  'blindstory', 'cerebralization',
  'eudaemonist', 'antigalactagogue',
  'pyromucyl', 'overbandy',
  'wheatear', 'vacanthearted',
  'uprip', 'cerebellopontile',
  'wifelike', 'isophasal',
  'frisca', 'axillae',
  'Alkoran', 'neurotrophic',
  'wrainstaff', 'beneaped',
  'Sclerodermi', 'overcovetous',
  'dusty', 'underthrob',
  'noncorrosive', 'abbreviate',
  'Octans', 'endurance',
  'quinquefid', 'athlete',
  'jossakeed', 'clochette',
  'resistively', 'ulerythema',
  'undiscernible', 'infusionist',
  'cerebropathy', 'hyperglycorrhachia',
  'diorthosis', 'karwar',
  'Lutao', 'myelosyringosis',
  'unincreased', 'termless',
  'linenman', 'sacristan',
  'interatomic', 'limitary',
  'thunderous', 'sulphogallic',
  'enantiomorphously', 'strophiolated',
  'rubbishly', 'semihumorously',
  'evangelicism', 'thapsia',
  'majorate', 'seagirt',
  'imponderous', 'fortuitously',
  'caravel', 'deuteranomal',
  'pantomimist', 'puzzleheadedly',
  'flatterer', 'mikie',
  'unbelieve', 'preconfiguration',
  'trochaic', 'anoestrus',
  'unstaveable', 'treaclewort',
  'zigzaggedly', 'whereat',
  'dailiness', 'lophiostomate',
  'Rhinoptera', 'divinatory',
  'Rosellinia', 'fingerling',
  'Glyptotherium', 'hexaploid',
  'sagebrush', 'erasure',
  'isonephelic', 'stepgrandson',
  'intermorainic', 'Rhinophis',
  'yellowtop', 'twitterboned',
  'uninitiate', 'fleeter',
  'keratitis', 'wrestlerlike',
  'Eciton', 'Neillia',
  'indeficiently', 'debouch',
  'daktylon', 'oftness',
  'chemicomineralogical', 'nonferrous',
  'Cocamama', 'tetraspheric',
  'parel', 'wailer',
  'adjustive', 'tannyl',
  'interramification', 'pyche',
  'antivibrating', 'primordium',
  'indemnification', 'semicynical',
  'suprasensuous', 'toyfulness',
  'counterslope', 'misfortuner',
  'Sophy', 'Athericera',
  'exhaustively', 'Meliaceae',
  'calceolate', 'Giansar',
  'mastologist', 'bilker',
  'harttite', 'garishness',
  'diffidation', 'Peggy',
  'eristic', 'saltfat',
  'unenterable', 'photospectroscopical',
  'amminolytic', 'nabber',
  'unministerial', 'polytrichia',
  'redelegation', 'producibleness',
  'unpalatial', 'vower',
  'kashruth', 'williwaw',
  'cappy', 'cask',
  'progredient', 'affront',
  'parietovaginal', 'quotiety',
  'monostomatous', 'siderostat',
  'schoenobatist', 'budgereegah',
  'epipodite', 'dungeonlike',
  'Paramecium', 'submaximal',
  'nightfowl', 'tesseratomy',
  'Rosetta', 'Azilian',
  'suppose', 'chibinite',
  'retractively', 'spumous',
  'unjewel', 'polycentral',
  'unburial', 'rowdyproof',
  'cottid', 'crosswalk',
  'unreproachable', 'shunter',
  'coincidentally', 'paleoethnology',
  'bitten', 'filtering',
  'preobviousness', 'myrcene',
  'isopropylamine', 'spumone',
  'Ophiurida', 'specifically',
  'unwrinkled', 'unfrosted',
  'assort', 'semiextinct',
  'metamorphoses', 'middlingish',
  'interorbitally', 'speechment',
  'semeia', 'uniplicate',
  'protoreptilian', 'unoverflowing',
  'Eve', 'Myxospongiae',
  'hypapophysis', 'certificate',
  'irremissibly', 'grossulaceous',
  'precritical', 'squaliform',
  'gorgonacean', 'overheave',
  'apathy', 'minion',
  'etypical', 'paleolithoid',
  'alulet', 'philanthropic',
  'trammer', 'aerophotography',
  'photodissociation', 'armament',
  'amphimictic', 'hendecagonal',
  'enough', 'micrometrically',
  'pare', 'cervicoaxillary',
  'redevelop', 'equidominant',
  'nighted', 'slogan',
  'cometographical', 'veinage',
  'inopercular', 'demitone',
  'reflection', 'binate',
  'insection', 'gaskin',
  'sulbasutra', 'neurepithelium',
  'Nabalitic', 'sachem',
  'rigger', 'Kolis',
  'phaeophore', 'stotterel',
  'kidling', 'Heterodonta',
  'begem', 'bibliopoly',
  'switchbacker', 'fairtime',
  'bounded', 'pentametrist',
  'brigand', 'dickey',
  'troggin', 'nonimitative',
  'distantly', 'metrist',
  'bitterly', 'ribonucleic',
  'Leiophyllum', 'disadventure',
  'Ternstroemiaceae', 'cystoradiography',
  'lithectomy', 'yardman',
  'erection', 'carbonization',
  'unhomologous', 'Tagaur',
  'regimentation', 'submammary',
  'kittenishness', 'Phaeodaria',
  'forcipiform', 'unculture',
  'festuca', 'bourtree',
  'miskenning', 'instanter',
  'Merino', 'staphylinid',
  'overtariff', 'leeky',
  'unabidingness', 'Heinrich',
  'chromocyte', 'hemophile',
  'bewrayment', 'alterity',
  'unispinose', 'hart',
  'reimbursement', 'plumeous',
  'Embioptera', 'swan',
  'urography', 'leguan',
  'montane', 'spong',
  'stilboestrol', 'astragalar',
  'vindicate', 'bullishness',
  'hepatoid', 'phototelegraph',
  'thalamifloral', 'eaglestone',
  'psychagogue', 'woodhorse',
  'tubicen', 'machiavellist',
  'eurybenthic', 'hoodful',
  'foreskirt', 'seadrome',
  'unrefuting', 'nutseed',
  'elatedness', 'poimenics',
  'goodeniaceous', 'expulse',
  'whyever', 'nomography',
  'coascend', 'Pelias',
  'uromelanin', 'postmuscular',
  'scandalmongery', 'carbohydrate',
  'typhloalbuminuria', 'insistingly',
  'circumintestinal', 'misedit',
  'indigotic', 'unode',
  'rathely', 'domestication',
  'fluffy', 'tooter',
  'pupilloscope', 'necrologic',
  'gata', 'unoccluded',
  'spleuchan', 'helminthous',
  'plecopterous', 'foreconsider',
  'archturncoat', 'coagulin',
  'bowdlerism', 'sulphurous',
  'howlingly', 'profectional',
  'Lumbricidae', 'spider',
  'alphabetics', 'whaleboat',
  'Carapache', 'adipate',
  'kissy', 'screechily',
  'theoastrological', 'wisewoman',
  'myelomeningitis', 'underadjustment',
  'transference', 'inhume',
  'remagnetization', 'schmelz',
  'Pherecratian', 'revirescence',
  'bookery', 'megotalc',
  'opercled', 'pipkin',
  'algebraization', 'peerling',
  'endocrinopathic', 'ostensibly',
  'Assamites', 'busybody',
  'faffy', 'unrepiqued',
  'unigenous', 'cryable',
  'organing', 'pulegol',
  'hammochrysos', 'tillerman',
  'admix', 'correlate',
  'penance', 'Silvia',
  'incredulity', 'nimshi',
  'Methodist', 'bowleggedness',
  'robotry', 'encephalomeningitis',
  'redsear', 'soldierly',
  'regreet', 'amadou',
  'pemican', 'unnebulous',
  'chloralization', 'septemia',
  'fullery', 'redock',
  'neath', 'liverleaf',
  'impletion', 'Agnoetae',
  'umbones', 'victualless',
  'carpophyte', 'infrapose',
  'Aphelops', 'dedition',
  'zephyrean', 'Wordsworthianism',
  'acetylacetone', 'neuterly',
  'lyre', 'altercation',
  'acritan', 'paddlewood',
  'neurophagy', 'Turtan',
  'per', 'Nile',
  'postmeningeal', 'boor',
  'glanders', 'Hexamita',
  'ophiomorphic', 'interwovenly',
  'Kojiki', 'fibrocalcareous',
  'cantoner', 'tuliplike',
  'frighten', 'mohair',
  'sepsine', 'gateman',
  'munnion', 'Lycopodiales',
  'hypogonation', 'antilysis',
  'paranuclear', 'podostomatous',
  'syncytial', 'renotice',
  'calcification', 'dollishness',
  'Cynthiidae', 'Clementina',
  'plantar', 'Palicourea',
  'columelliform', 'bindingness',
  'sackful', 'firebote',
  'garlicwort', 'epithalline',
  'conformableness', 'velamentous',
  'utfangethef', 'crocused',
  'insurmountability', 'granary',
  'stand', 'shrinelet',
  'canadite', 'pseudoerysipelas',
  'ayous', 'adjustment',
  'countercouchant', 'pneumatophilosophy',
  'Toromona', 'denaturize',
  'packmanship', 'sanitation',
  'drawshave', 'giggling',
  'organism', 'synchronological',
  'absinthial', 'madidans',
  'paintable', 'spermoblastic',
  'perivaginal', 'wharve',
  'paralytical', 'facinorousness',
  'criminality', 'Amphirhina',
  'bedcap', 'strawboard',
  'idioplasmic', 'Simonian',
  'exaggerate', 'culicid',
  'overbrow', 'precancellation',
  'hyposyllogistic', 'terpsichoreal',
  'meconophagist', 'fluviatile',
  'charlatanically', 'unregenerateness',
  'incombustibleness', 'infrapubian',
  'monobasic', 'uneasy',
  'Commelinaceae', 'remigial',
  'psychotic', 'asphodel',
  'paraphrasian', 'lawyership',
  'wined', 'uncherishing',
  'hives', 'compagination',
  'macroanalysis', 'eldritch',
  'nake', 'anthropogeography',
  'appearanced', 'noncollectable',
  'scoffing', 'Mechitaristican',
  'garboil', 'bricking',
  'unwrinkleable', 'overrooted',
  'gimleteyed', 'forecited',
  'Polab', 'coadore',
  'recultivation', 'postgeniture',
  'nonisobaric', 'Diopsis',
  'archminister', 'satelles',
  'outbred', 'ciliospinal',
  'stablekeeper', 'rubstone',
  'sightlily', 'trilocular',
  'bakula', 'sniggle',
  'petrify', 'geodiferous',
  'Hun', 'underhand',
  'androgynary', 'Japanize',
  'unreprievably', 'mesmerizer',
  'hoplomachist', 'ommatophore',
  'saccharometric', 'Dedan',
  'epactal', 'hubshi',
  'dislocator', 'magnetic',
  'thirsting', 'unforewarnedness',
  'giantlike', 'verificative',
  'maximed', 'botryoidally',
  'northerner', 'wraithy',
  'quisle', 'Caretta',
  'Frenchy', 'chlorochromic',
  'yelk', 'downweighted',
  'homelessness', 'lithontriptic',
  'natuary', 'reapplaud',
  'Iscariotism', 'hallway',
  'zootheist', 'appraisive',
  'homoeopathicity', 'asteria',
  'hyperabelian', 'scotching',
  'patriarchal', 'aperea',
  'Alfirk', 'rubiator',
  'tapaculo', 'floatman',
  'superdevotion', 'cella',
  'revilement', 'adscriptive',
  'reshun', 'idioplasmatic',
  'anthropophuism', 'flirtatious',
  'Benjamite', 'southeastern',
  'suggestress', 'fascisticize',
  'eccrisis', 'outfort',
  'subsistingly', 'tophetize',
  'pseudonavicular', 'reswim',
  'ocelli', 'appreciational',
  'chromatopathia', 'hoarstone',
  'Vaucheria', 'supersupremacy',
  'evangelical', 'glia',
  'Dendropogon', 'strangulable',
  'insnare', 'flightful',
  'adsignify', 'Judaize',
  'eudiometry', 'pica',
  'unhelpableness', 'posturist',
  'Thargelion', 'adstipulator',
  'bearableness', 'tactful',
  'Deuteronomic', 'apathetical',
  'scrollwise', 'microlevel',
  'Zenaidinae', 'fleetingness',
  'atonality', 'congregationalism',
  'affrontedly', 'lectorial',
  'grantable', 'bunter',
  'willingly', 'unresty',
  'giggish', 'Dianthaceae',
  'duty', 'vespertilio',
  'Cunoniaceae', 'reticulate',
  'Lestrigon', 'cloddiness',
  'unmaidenly', 'mettar',
  'flagleaf', 'ort',
  'tumidly', 'respace',
  'broch', 'outbranching',
  'Ampelidae', 'coetaneousness',
  'oxyiodide', 'revisable',
  'spongewood', 'cyatholith',
  'purchasability', 'dinornithine',
  'ascula', 'messaline',
  'puparial', 'Bismarckian',
  'quinquevalency', 'cibory',
  'orrhoid', 'restraighten',
  'urd', 'civilly',
  'pangamic', 'palpiform',
  'steddle', 'trochalopodous',
  'altogetherness', 'teloblast',
  'Anatole', 'consonant',
  'psychologize', 'grunt',
  'pathogermic', 'subduedness',
  'bladderlike', 'eventuate',
  'overconservative', 'fibrointestinal',
  'cynoid', 'concentrated',
  'unelectrify', 'urethrovaginal',
  'humaniform', 'decatize',
  'supersemination', 'Orleanistic',
  'demonial', 'Suluan',
  'tummer', 'yoker',
  'biophysiography', 'teazer',
  'tritium', 'rohob',
  'planulate', 'tangs',
  'tridecene', 'guzmania',
  'animism', 'gaufrette',
  'germling', 'sarcogenic',
  'encapsulate', 'hachure',
  'joyously', 'seneschalship',
  'sabered', 'autoradiographic',
  'sise', 'enzymically',
  'Jacksonia', 'molybdous',
  'theodicaea', 'weanling',
  'predictation', 'programmar',
  'bobbery', 'Petroselinum',
  'untailorlike', 'taxinomic',
  'invertebracy', 'radiatosulcate',
  'petling', 'phrasally',
  'unken', 'hightoby',
  'screened', 'assentient',
  'pointillist', 'piacularly',
  'preconfine', 'saccadic',
  'vigorless', 'bodiless',
  'geodesy', 'phagodynamometer',
  'Phoebe', 'memorabilia',
  'caryatidean', 'Morrisean',
  'woom', 'keleh',
  'frilling', 'pantotype',
  'hatlike', 'appealability',
  'revise', 'elegiacal',
  'extractorship', 'sneck',
  'superexalt', 'mediatress',
  'prepeduncle', 'porchless',
  'cytozymase', 'featherfoil',
  'hemabarometer', 'peptize',
  'reoccur', 'esophagostenosis',
  'cinerea', 'skivvies',
  'meconic', 'transpleurally',
  'toilful', 'amygdalothripsis',
  'draconic', 'undiffusive',
  'hacked', 'featherwood',
  'ascent', 'antiskidding',
  'parchmentlike', 'soldo',
  'maudlin', 'Vesuvian',
  'heteronymous', 'housewares',
  'prognosticator', 'macrurous',
  'unscholastic', 'tigerishness',
  'ristori', 'roosa',
  'rhinocele', 'flether',
  'Vernonia', 'furfuroid',
  'emmarvel', 'stumpish',
  'lasarwort', 'dizain',
  'oxygnathous', 'incendiary',
  'breviary', 'metanephric',
  'assorted', 'kayaker',
  'hingeless', 'alkaptonuria',
  'shackling', 'clima',
  'cunye', 'barnacle',
  'buttstock', 'Vespina',
  'megacoulomb', 'photoengraver',
  'washoff', 'Emydea',
  'Pictland', 'Tities',
  'ungrazed', 'bespot',
  'pretranslate', 'karyon',
  'unadequately', 'criminative',
  'stellulate', 'Cellulomonas',
  'sob', 'tyrant',
  'uralitize', 'heptadecyl',
  'tiriba', 'codiniac',
  'somewhat', 'Pinnigrada',
  'Gravigrada', 'bummock',
  'eudiometric', 'architecture',
  'hepatonephric', 'hoplitic',
  'yont', 'cecidiology',
  'colloid', 'Pam',
  'urtica', 'toro',
  'Tritoma', 'astoundable',
  'lorum', 'unfactorable',
  'ferricyanic', 'dicaryotic',
  'lamellicornous', 'catatoniac',
  'mediatingly', 'chloropicrin',
  'premonarchial', 'sparkless',
  'upflood', 'arrayal',
  'rubricality', 'forgetter',
  'diagrammeter', 'caducary',
  'violably', 'yoke',
  'democratist', 'roundtail',
  'competent', 'rebuffet',
  'naveled', 'menstruate',
  'presumptiously', 'Lodowick',
  'polypharmic', 'urceole',
  'subradiate', 'semicombined',
  'fallace', 'Atlas',
  'honeyed', 'circumscribable',
  'redictate', 'antiscians',
  'jailish', 'anthropophaginian',
  'chacona', 'hydronitric',
  'gymnopaedic', 'phelloplastics',
  'Sundar', 'nomistic',
  'erratically', 'reiver',
  'ditrochee', 'subinspector',
  'disproportionalness', 'confined',
  'Boreas', 'celebration',
  'Tagakaolo', 'portraitlike',
  'nodulous', 'Wellsian',
  'brigandism', 'lanceproof',
  'mushrebiyeh', 'upflare',
  'pathoneurosis', 'preseasonal',
  'holophytic', 'fraughan',
  'oxyhydrate', 'deducible',
  'showman', 'prankishness',
  'diadoche', 'apocryphalness',
  'disloyalty', 'milkman',
  'lymphoduct', 'Terminalia',
  'Moravianism', 'synapticulate',
  'turgidity', 'semichemical',
  'jinnestan', 'ihi',
  'Docetist', 'surfusion',
  'dromond', 'epichondrosis',
  'unclogged', 'untenty',
  'malnutrite', 'ileitis',
  'twizzle', 'heir',
  'reproduce', 'untwineable',
  'segmentary', 'figging',
  'predischarge', 'intercommunication',
  'memorious', 'trialate',
  'lepidopteran', 'battel',
  'laryngoscleroma', 'fanfarade',
  'Catesbaea', 'todder',
  'goadster', 'inalterability',
  'abidi', 'consonantly',
  'focusable', 'admirer',
  'demineralization', 'unfishing',
  'precinctive', 'canaigre',
  'gymnastic', 'lutanist',
  'unnaturalness', 'daraf',
  'strig', 'ethenyl',
  'underdog', 'pusillanimous',
  'spiritedness', 'acrobatholithic',
  'grinny', 'anarchal',
  'molarimeter', 'honeyedly',
  'Tukuler', 'childship',
  'opianic', 'vila',
  'Lycus', 'overfree',
  'dardanium', 'Ran',
  'unblind', 'pudendous',
  'berg', 'successiveness',
  'dietic', 'hinderlings',
  'helotry', 'unadulteratedly',
  'germinally', 'vorticial',
  'effulge', 'danner',
  'Brescian', 'smithydander',
  'pectoriloquial', 'sauna',
  'parchable', 'contrariousness',
  'tenebrious', 'encephalocele',
  'overlie', 'alkyne',
  'jargon', 'neurinoma',
  'glucokinin', 'stethospasm',
  'overgood', 'stertoriousness',
  'retrofracted', 'synchronical',
  'unkempt', 'homogony',
  'pantothenate', 'incuriousness',
  'applique', 'shaku',
  'Carlyleian', 'tricolumnar',
  'intercessory', 'outsound',
  'fluctuate', 'alimentotherapy',
  'polyzonal', 'myalgic',
  'subderivative', 'undesponding',
  'beliquor', 'superartificially',
  'phosphammonium', 'redistribute',
  'grievedly', 'trachearian',
  'Siganidae', 'polydactyl',
  'bouw', 'slowgoing',
  'systemization', 'towhee',
  'capturer', 'fullface',
  'kat', 'tipstock',
  'gourdlike', 'unbedded',
  'multishot', 'noded',
  'miglio', 'bacillariaceous',
  'nester', 'Ascidioida',
  'Shivaism', 'juicily',
  'replevisable', 'illusion',
  'armamentary', 'cornerer',
  'fenland', 'descriptionist',
  'unprison', 'ablow',
  'Oenotrian', 'discantus',
  'frankhearted', 'aculeolate',
  'pinnatisect', 'fatuoid',
  'evaluate', 'prepare',
  'pecite', 'unreproving',
  'muscardine', 'gabber',
  'ciliation', 'bananist',
  'Balsaminaceae', 'whirtle',
  'dismissable', 'licking',
  'kirkward', 'ringbird',
  'smeuse', 'fetterlock',
  'shopboy', 'Franklinist',
  'dingar', 'slows',
  'raker', 'nebulescent',
  'presbyterate', 'orbitolite',
  'antimetropic', 'formenic',
  'misaccent', 'cosmoscope',
  'vow', 'imputrescence',
  'immodest', 'unyouthful',
  'deuterogamy', 'Crioceras',
  'dicecup', 'cerebellum',
  'meralgia', 'bondman',
  'ovariorrhexis', 'montbretia',
  'ericolin', 'trudge',
  'admirative', 'electrobus',
  'chorioidoiritis', 'uncontented',
  'moody', 'stepwise',
  'Iliadist', 'intervener',
  'revealability', 'unheal',
  'lavender', 'lairman',
  'swoon', 'periclinally',
  'roughstring', 'foresentence',
  'preprovision', 'Philippan',
  'breathable', 'Pycnonotidae',
  'boatmanship', 'censorate',
  'Nowroze', 'overdrip',
  'pyometra', 'menstruant',
  'cobaltous', 'unfacile',
  'spindrift', 'algoesthesiometer',
  'anargyros', 'uncomposed',
  'soporiferous', 'sporid',
  'conscript', 'cyperaceous',
  'tasimetric', 'unreceptive',
  'unformulistic', 'Baedeker',
  'Skidi', 'malleus',
  'sacramentarianism', 'labrosaurid',
  'strigovite', 'wishless',
  'porto', 'payableness',
  'sacken', 'zonuroid',
  'concave', 'uncupped',
  'pauciloquy', 'Attical',
  'otodynic', 'Calopogon',
  'atmocausis', 'courageousness',
  'warp', 'inorganizable',
  'objector', 'Russellite',
  'Mixtecan', 'adiabolist',
  'unsedentary', 'troptometer',
  'hatcher', 'congreet',
  'dingy', 'trochoidally',
  'cattleya', 'micronuclear',
  'Gyges', 'abatement',
  'cephalopodan', 'vowelization',
  'Ovidian', 'coquetry',
  'semifluctuating', 'precreation',
  'impedition', 'sweepingly',
  'gallisin', 'lymphogenic',
  'electropneumatically', 'Zeugobranchia',
  'quadrivalently', 'pitwork',
  'prerolandic', 'undemocratize',
  'parchedly', 'exclamational',
  'immodulated', 'inarticulateness',
  'aseity', 'nitramine',
  'adhibit', 'throatlet',
  'nonconversational', 'untrippable',
  'oculomotor', 'mischanceful',
  'bret', 'sunfisher',
  'Naoto', 'precaptivity',
  'unprofiteering', 'lacework',
  'lungless', 'targeted',
  'spinifugal', 'downlooked',
  'commendam', 'untaxable',
  'comicoprosaic', 'euornithic',
  'hyperthermy', 'muirburn',
  'toed', 'Crambidae',
  'undulatory', 'aphthic',
  'scarecrowy', 'eutrophy',
  'gubernative', 'tarboy',
  'overeagerness', 'noncensus',
  'hausmannite', 'jobman',
  'unionism', 'Epichristian',
  'unplaced', 'isovaline',
  'gastropneumonic', 'unaccommodatingness',
  'meteorology', 'hup',
  'rarefy', 'hungrily',
  'turbinotomy', 'shelder',
  'discontiguousness', 'semibastion',
  'pigtail', 'kreittonite',
  'overeasiness', 'associationist',
  'musterdevillers', 'rand',
  'gamosepalous', 'Lutheranism',
  'singlehearted', 'Newtonian',
  'punkah', 'vulturism',
  'minatory', 'spleenless',
  'irregularity', 'unfinished',
  'convivial', 'thoracogastroschisis',
  'androclinium', 'scolog',
  'rebias', 'phthongometer',
  'outjourney', 'adversaria',
  'narcotic', 'spheric',
  'glideness', 'Lupinus',
  'Musophagidae', 'Meridional',
  'unsanction', 'blanketmaking',
  'phenylcarbimide', 'streng',
  'Sophoclean', 'predisability',
  'gaby', 'resolvancy',
  'sleeveen', 'scrimpness',
  'recklessly', 'undulose',
  'momism', 'besought',
  'portension', 'symmetrize',
  'spilly', 'deteriorationist',
  'nettlesome', 'revamp',
  'gummata', 'phelonion',
  'expansion', 'equidiurnal',
  'epistomal', 'uncapable',
  'sejunct', 'unreversed',
  'intermembranous', 'sponsor',
  'vixenishly', 'appetition',
  'vraicking', 'dentatosinuate',
  'caryophylleous', 'frike',
  'serine', 'subsecutive',
  'paludicolous', 'encrotchet',
  'freezingly', 'Halopsychidae',
  'bemitred', 'uneviscerated',
  'antiministerial', 'oreophasine',
  'unmixedness', 'offender',
  'aphrodisiac', 'laugh',
  'automatous', 'consolingly',
  'innatism', 'quitter',
  'stodgily', 'Joannite',
  'afterdamp', 'irrememberable',
  'mouth', 'Galaxiidae',
  'Boaedon', 'superenergetic',
  'bisegment', 'perfectionize',
  'thimblerigging', 'ungrieve',
  'chasmophyte', 'harmonogram',
  'internuncius', 'cabaret',
  'lamely', 'mundane',
  'trichostrongyle', 'perborax',
  'actuation', 'unadherence',
  'unsubventioned', 'boatlike',
  'Akwapim', 'nerveroot',
  'bemedaled', 'bacteriotoxic',
  'alula', 'tudel',
  'girouette', 'nummulitic',
  'apparent', 'plication',
  'disrobement', 'Turkdom',
  'polyvinyl', 'constructure',
  'nonproprietary', 'convergescence',
  'arrhenoid', 'chylopericardium',
  'Escalloniaceae', 'weed',
  'unparch', 'capful',
  'allowably', 'blather',
  'mooring', 'nonmanual',
  'channer', 'bibliotherapeutic',
  'antiscabious', 'brushmaking',
  'pseudogeusia', 'repanel',
  'ctene', 'geotic',
  'anakinetic', 'luckiness',
  'rottolo', 'patte',
  'goniac', 'delitescence',
  'illogical', 'spongiocyte',
  'twenty', 'promote',
  'dum', 'palingenetically',
  'goddess', 'Justin',
  'unjuvenile', 'wanworth',
  'uncompressible', 'clubbism',
  'badigeon', 'polyzoic',
  'ureterodialysis', 'northwards',
  'ectocondyloid', 'circumambulation',
  'Mayan', 'dermophyte',
  'ultraterrestrial', 'picrotoxin',
  'scusation', 'glanduliferous',
  'kymation', 'anoscope',
  'pulsellum', 'fripperer',
  'Outagami', 'gulleting',
  'subcontrol', 'antiballooner',
  'webless', 'cuculiform',
  'photocomposition', 'amerce',
  'strandage', 'wretchedly',
  'peddlerism', 'workfolk',
  'sylphic', 'hyperdactyl',
  'choroidea', 'metallurgy',
  'Hibernical', 'crookedly',
  'tympanotomy', 'mary',
  'evocatively', 'Porphyrianist',
  'megerg', 'meninx',
  'ostearthrotomy', 'undrainable',
  'ligniferous', 'troutbird',
  'hureek', 'ziarat',
  'claith', 'ferineness',
  'distantness', 'recorrupt',
  'graith', 'dynamiting',
  'tetradactyl', 'palatopterygoid',
  'celotomy', 'quot',
  'medicomania', 'monogenous',
  'reproachable', 'unmoderateness',
  'jinker', 'roguedom',
  'churchiness', 'salivant',
  'lachrymary', 'repunish',
  'Ghibelline', 'powerboat',
  'Monotheletism', 'putrefactive',
  'ampheclexis', 'screwing',
  'creeky', 'Canada',
  'nonratifying', 'prefuneral',
  'capably', 'choliambist',
  'coinsure', 'monembryonic',
  'mima', 'clerkhood',
  'physicalistically', 'Pentamerus',
  'Jacaranda', 'preobstruction',
  'alaite', 'tarsale',
  'neshly', 'chaperonless',
  'proconsul', 'flatbottom',
  'omenology', 'umptieth',
  'unrestable', 'impartialist',
  'keeper', 'argon',
  'gynantherous', 'poriness',
  'inchoately', 'gyric',
  'slicht', 'nitrosulphuric',
  'preliquidation', 'vitreously',
  'resaca', 'orchioscheocele',
  'Mysian', 'inexpedience',
  'cardioschisis', 'carpogonial',
  'silicam', 'extraofficial',
  'silhouettist', 'acantholysis',
  'Modulidae', 'abstrusity',
  'zimbaloon', 'beknived',
  'mistake', 'boarstaff',
  'evict', 'concussant',
  'denshare', 'naphthinduline',
  'hydroaviation', 'atmolyze',
  'brake', 'oxalyl',
  'commodity', 'katabasis',
  'uncunningness', 'veillike',
  'actuarial', 'monopoly',
  'heterotrophy', 'beadledom',
  'ineffectiveness', 'Escalator',
  'womenfolks', 'cleistogeny',
  'aristomonarchy', 'succourless',
  'scythelike', 'waker',
  'enemyship', 'Linene',
  'malacoid', 'Cephalochordata',
  'gladless', 'fevertwitch',
  'circumplicate', 'swiz',
  'receptiveness', 'aglaozonia',
  'delightable', 'roky',
  'Tettigidae', 'unstatutably',
  'availment', 'Hillhousia',
  'prostomiate', 'Woodruff',
  'mediumship', 'ahong',
  'birthmate', 'Carnegiea',
  'gypsine', 'unextravagance',
  'Algernon', 'Tetrix',
  'fancifulness', 'machicolation',
  'leucoplastid', 'Trappistine',
  'endarteritis', 'diagenesis',
  'Quinquatrus', 'laigh',
  'beaux', 'Alca',
  'Urginea', 'challengingly',
  'arachnid', 'damningly',
  'collatitious', 'examiningly',
  'prolan', 'sybotic',
  'Chinatown', 'corruptedness',
  'extern', 'jotty',
  'cataphylla', 'kinkcough',
  'genet', 'unmemorialized',
  'unspoilable', 'snobber',
  'tricyanide', 'treacle',
  'strongyloidosis', 'nubby',
  'Treron', 'samaria',
  'Tewa', 'pipingness',
  'effusiometer', 'pantographer',
  'upend', 'calyciflorous',
  'unhosed', 'immaterialist',
  'potifer', 'outjet',
  'unmildewed', 'tuath',
  'moksha', 'teletopometer',
  'unthinning', 'neptunism',
  'pleurocentesis', 'Czechoslovakian',
  'masoned', 'readvertisement',
];