/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "upm_checkconfig.h"

UpmCheckConfig::UpmCheckConfig(QObject *parent) : QObject(parent) {}

QVariant UpmCheckConfig::checkConfig(
        const QString &configName, QVariant defaultValue, QVariant::Type dataType,
        const QStringList &valueRange, bool continuousValue)
{
    QVariant gsettingsValue;
    QString systemConfigValue;
    QVariant tempValue;

    systemConfigValue = UpmControlDBus::self()->getUpmSystemConfig(configName);
    if (systemConfigValue.isNull()) {
        tempValue = QString();
        qDebug() << "get" << configName << "error";
    } else {
        if (false == checkSystemConfigValue(systemConfigValue, valueRange, continuousValue)) {
            tempValue = QString();
        } else {
            if (QVariant::String == dataType) {
                tempValue = systemConfigValue;
            } else if (QVariant::Int == dataType) {
                tempValue = systemConfigValue.toInt();
            } else if (QVariant::Bool == dataType) {
                "true" == systemConfigValue ? tempValue = true : tempValue = false;
            }
        }
        qDebug() << "system config:" << configName
                 << "value:" << tempValue;
    }

    if (true == UpmGsettings::self()->getGsettingsConfig(configName, gsettingsValue)) {
        if (true == checkGsettingsConfigValue(gsettingsValue, dataType, valueRange, continuousValue)) {
            if (tempValue.isNull()) {
                tempValue = gsettingsValue;
                if (QVariant::String == dataType) {
                    UpmControlDBus::self()->setUpmSystemConfig(configName, tempValue.toString());
                } else if (QVariant::Int == dataType) {
                    UpmControlDBus::self()->setUpmSystemConfig(configName, QString::number(tempValue.toInt()));
                } else if (QVariant::Bool == dataType) {
                    true == tempValue ? UpmControlDBus::self()->setUpmSystemConfig(configName, "true") :
                                        UpmControlDBus::self()->setUpmSystemConfig(configName, "false");
                }
                qDebug() << "sync gsettings config:" << configName
                         << "action:" << tempValue;
            } else {
                UpmGsettings::self()->setGsettingsConfig(configName, tempValue);
                qDebug() << "sync sytstem config:" << configName
                         << "action:" << tempValue;
            }
        } else {
            if (!tempValue.isNull()) {
                UpmGsettings::self()->setGsettingsConfig(configName, tempValue);
                qDebug() << "sync sytstem config:" << configName
                         << "action:" << tempValue;
            }
        }
    }

    if (tempValue.isNull()) {
        tempValue = defaultValue;
        UpmGsettings::self()->setGsettingsConfig(configName, tempValue);
        if (QVariant::String == dataType) {
            UpmControlDBus::self()->setUpmSystemConfig(configName, tempValue.toString());
        } else if (QVariant::Int == dataType) {
            UpmControlDBus::self()->setUpmSystemConfig(configName, QString::number(tempValue.toInt()));
        } else if (QVariant::Bool == dataType) {
            true == tempValue ? UpmControlDBus::self()->setUpmSystemConfig(configName, "true") :
                                UpmControlDBus::self()->setUpmSystemConfig(configName, "false");
        }
        qDebug() << "use default value:"
                 << defaultValue;
    }
    return tempValue;
}

bool UpmCheckConfig::checkSystemConfigValue(
        const QString &configValue, const QStringList &valueRange, bool continuousValue)
{
    qDebug() << "check system config value:" << configValue;
    if (UPM_POWER_CONFIG_ERROR == configValue) {
        return false;
    }

    qDebug() << "config value range:" << valueRange;
    if (true == continuousValue) {
        qDebug() << "config value is continuous:" << continuousValue;
        int value = configValue.toInt();
        int minValue = valueRange.at(0).toInt();
        int maxValue = valueRange.at(1).toInt();
        if (value < minValue || value > maxValue) {
            return false;
        } else {
            return true;
        }
    }
    return valueRange.contains(configValue);
}

bool UpmCheckConfig::checkGsettingsConfigValue(
        QVariant &configValue, QVariant::Type dataType,
        const QStringList &valueRange, bool continuousValue)
{
    qDebug() << "check gsettings config value:" << configValue;
    QString tempValue;

    if (QVariant::String == dataType) {
        tempValue = configValue.toString();
    } else if (QVariant::Int == dataType) {
        tempValue = QString::number(configValue.toInt());
    } else if (QVariant::Bool == dataType) {
        if (true == configValue.toBool()) {
            tempValue = "true";
        } else {
            tempValue = "false";
        }
    }

    qDebug() << "config value range:" << valueRange;
    if (true == continuousValue) {
        qDebug() << "config value is continuous:" << continuousValue;
        int value = tempValue.toInt();
        int minValue = valueRange.at(0).toInt();
        int maxValue = valueRange.at(1).toInt();
        if (value < minValue || value > maxValue) {
            return false;
        } else {
            return true;
        }
    }
    return valueRange.contains(tempValue);
}


