/***************************************************************************
 *                                                                         *
 *                                                                         *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "codonmodel.h"
#include <assert.h>
#include "opturtree.h"
#include "model.h"

// codon list
CodonAAMap codon_list[NUM_CODON];

// codon trans info
CodonTransInfo codon_trans[NUM_CODON][NUM_CODON];

// index from 64 to 61
int codon_index[64];

bool getCpG_Status(char *codon) {
	return (codon[0] == 'C' && codon[1] == 'G') ||
		(codon[1] == 'C' && codon[2] == 'G');
}

/**
	build the 61 codons list
*/
void buildCodonInfo() {
	int count = 0;
	// build the codon list
	for (int i = 0; i < 64; i++) {
		codon_index[i] = count;
		if (codon_aa_map[i].amino_acid != STOP_CODON) {
			codon_list[count] = codon_aa_map[i];
			codon_list[count].CpG_status = getCpG_Status(codon_list[count].name);
			count++;
		}
	}
	assert(count == NUM_CODON);

	CodonTransInfo cinfo;

	memset(codon_trans, 0, sizeof(codon_trans));

	// build the codon transfer info matrix
	for (int count1 = 0; count1 < NUM_CODON; count1 ++)
		for (int count2 = count1 + 1; count2 < NUM_CODON; count2++) {
			int num_differ = 0;
			for (int cpos = 0; cpos < CODON_LENGTH; cpos ++)
				if (codon_list[count1].name[cpos] != codon_list[count2].name[cpos]) {
					num_differ ++;
					if (num_differ > 1) { // break of more than 1 codon positions are different
						break;
					}
					cinfo.source = codon_list[count1].name[cpos];
					cinfo.dest = codon_list[count2].name[cpos];
					cinfo.position = cpos+1;
				}
			if (num_differ > 1)
				continue;
			cinfo.is_transition = false;
			if ((cinfo.source == 'A' && cinfo.dest == 'G') ||
			        (cinfo.source == 'G' && cinfo.dest == 'A') ||
			        (cinfo.source == 'C' && cinfo.dest == 'T') ||
			        (cinfo.source == 'T' && cinfo.dest == 'C') )
				cinfo.is_transition = true;
			
			// synonymous if the coded amino acid is the same
			cinfo.is_synonymous = (codon_list[count1].amino_acid == codon_list[count2].amino_acid);

			bool source_CpG = codon_list[count1].CpG_status;
			bool dest_CpG = codon_list[count2].CpG_status;
			if (source_CpG == dest_CpG)
				cinfo.CpG_status = CPG_UNCHANGED;
			else if (dest_CpG)
				cinfo.CpG_status = CPG_GAINED;
			else 
				cinfo.CpG_status = CPG_LOST;
			
			codon_trans[count1][count2] = cinfo;
			codon_trans[count2][count1] = cinfo;
			if (cinfo.CpG_status == CPG_GAINED)
				codon_trans[count2][count1].CpG_status = CPG_LOST;
			else if (cinfo.CpG_status == CPG_LOST)
				codon_trans[count2][count1].CpG_status = CPG_GAINED;
		}
}

inline int getGranthamDistance(int amino1, int amino2) {
	if (amino1 == amino2)
		return 0;
	if (amino1 < amino2) {
		int temp = amino1;
		amino1 = amino2;
		amino2 = temp;
	}
	return grantham_half_matrix[((amino1-1) * amino1) / 2 + amino2 ];
}

CodonModel::CodonModel() : BaseM() {
	v_parm = 40.0;
}


/**
	all things are inited here
*/
void CodonModel::init () {

	/*
	if (mymodel.getTsTvRatioType() != ESTIMATE) {
		tsTvRatio_ = mymodel.getTsTvRatio();
	}
	*/
	memset(unitRateMat_, 0, sizeof(unitRateMat_));
	calcRateParameters ();
	tranprobmat(nState_);
} /* tranprobmat */


void CodonModel::reInit () {
	calcRateParameters ();
	tranprobmat(nState_);
}


/**
	get V parameter (for Codon Goldman Yang model 1994)
*/
double CodonModel::getVParm () {
	return v_parm;
}

/**
	set v parameter (for Codon Goldman Yang model 1994)
*/
void CodonModel::setVParm (const double vparm) {
	v_parm = vparm;
}


/**
		calculate the rate parameters (not rate matrix)
*/
void CodonModel::calcRateParameters() {
	double sum = 0;

	for (int i = 0; i < NUM_CODON; i++)
		for (int j = i+1; j < NUM_CODON; j++) {
			if (codon_trans[i][j].source == 0) { // more than 2 different codon positions
				continue;
			}
			//unitRateMat_[i][j]
			int amino1 = codon_list[i].amino_acid;
			int amino2 = codon_list[j].amino_acid;
			int grant_dist = getGranthamDistance(amino1, amino2);
			if (codon_trans[i][j].is_transition)
				unitRateMat_[i][j] = tsTvRatio_ * exp(-grant_dist / v_parm);
			else
				unitRateMat_[i][j] = exp(-grant_dist / v_parm);
			unitRateMat_[j][i] = unitRateMat_[i][j];
			sum += unitRateMat_[i][j];
		}
	/*
	for (int i = 0; i < NUM_CODON; i++)
		for (int j = i+1; j < NUM_CODON; j++) {
			if (codon_trans[i][j].source == 0) { // more than 2 different codon positions
				continue;
			}
			unitRateMat_[i][j] /= sum;
			unitRateMat_[j][i] /= sum;
		}
	*/
}


double CodonModel::cmpNegLogLi (double value) {
	if (isActPam_ == TS_TV_RATIO)
		tsTvRatio_ = value;
	else
		v_parm = value;

	reInit ();

	opt_urtree.cmpLiNd ();
	//opt_urtree.opt (MAX_IT_UR_BR_PAM_OPT);
	double logLi_ = opt_urtree.getLogLi ();
	return -logLi_;
}


//--------------------------------------------------------------------
//optimize the tsTvRatio basing Brent method
bool CodonModel::optPam () {
	double fx_, error_;
	double oriTsTvRatio_ = tsTvRatio_;
	double ori_vparm = v_parm;
			//  double logLi_ = opt_urtree.getLogLi ();
	if (isMasterProc())
		std::cout <<"Optimizing Goldman & Yang 1994 codon-based model parameters ..." << endl;
	Brent::turnOnOptedPam ();
	double logLi_ = opt_urtree.cmpLogLi ();
	//  std::cout <<"Log likelihood: " << logLi_ << endl;


	if (mymodel.getTsTvRatioType() == ESTIMATE) {
		isActPam_ = TS_TV_RATIO;
		//double oriTsTvRatio_ = tsTvRatio_;
		if (isMasterProc())
			std::cout <<"   Optimizing transition/transversion ratio..." << endl;
		tsTvRatio_ = Brent::optOneDim (MIN_TS_TV_RATIO, tsTvRatio_, MAX_TS_TV_RATIO,
		                              EPS_MODEL_PAM_ERROR, &fx_, &error_);
		reInit ();
	} 
	//end of optimizing tsTvRatio

	if (true) {
		isActPam_ = V_PARM;
		//double ori_vparm = v_parm;
		if (isMasterProc())
			std::cout <<"   Optimizing V parameter..." << endl;
		v_parm = Brent::optOneDim (MIN_V_PARM, v_parm, MAX_V_PARM,
		                              EPS_MODEL_PAM_ERROR, &fx_, &error_);
		reInit ();

	}//end of optimizing pyPuRatio

	//      logLi_ = opt_urtree.cmpLogLi ();
	//     std::cout <<"Log likelihood: " << logLi_ << endl;
	//------------------------------/------------------------------/





	opt_urtree.cmpLiNd ();
	logLi_ = opt_urtree.getLogLi ();

	std::cout.precision (10);
	//  std::cout << "Log likelihood: " << logLi_ << endl;

	if (isMasterProc()) {

		std::cout.precision (5);
		std::cout << "Transition/transversion ratio =  " << tsTvRatio_/2.0 << endl;
		std::cout << "V parameter                   =  " << v_parm << endl;
	}

	Brent::turnOffOptedPam ();
	return (fabs (tsTvRatio_ - oriTsTvRatio_) > EPS_MODEL_PAM_ERROR ||
			fabs (v_parm - ori_vparm) > EPS_MODEL_PAM_ERROR );
}



CodonModel::~CodonModel() {}
