/***************************************************************************
 *                                                                         *
 *                                                                         *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "simanneal.h"
#include "rannum.h"
#include <math.h>
#include <stdarg.h>
#include "vec.h"
//#define NRANSI
//#include "nrutil.h"
#define NR_END 1
#define FREE_ARG char*

#define GET_PSUM \
					for (n=1;n<=ndim;n++) {\
					for (sum=0.0,m=1;m<=mpts;m++) sum += p[m][n];\
					psum[n]=sum;}


#define IA 16807
#define IM 2147483647
#define AM (1.0/IM)
#define IQ 127773
#define IR 2836
#define NTAB 32
#define NDIV (1+(IM-1)/NTAB)
#define EPS 1.2e-7
#define RNMX (1.0-EPS)

double ran1(long *idum) {
	int j;
	long k;
	static long iy=0;
	static long iv[NTAB];
	double temp;

	if (*idum <= 0 || !iy) {
		if (-(*idum) < 1) *idum=1;
		else *idum = -(*idum);
		for (j=NTAB+7;j>=0;j--) {
			k=(*idum)/IQ;
			*idum=IA*(*idum-k*IQ)-IR*k;
			if (*idum < 0) *idum += IM;
			if (j < NTAB) iv[j] = *idum;
		}
		iy=iv[0];
	}
	k=(*idum)/IQ;
	*idum=IA*(*idum-k*IQ)-IR*k;
	if (*idum < 0) *idum += IM;
	j=iy/NDIV;
	iy=iv[j];
	iv[j] = *idum;
	if ((temp=AM*iy) > RNMX) return RNMX;
	else return temp;
}
#undef IA
#undef IM
#undef AM
#undef IQ
#undef IR
#undef NTAB
#undef NDIV
#undef EPS
#undef RNMX


long idum = 123456;
double tt;

void nrerror(const char error_text[])
/* Numerical Recipes standard error handler */
{
	fprintf(stderr,"Numerical Recipes run-time error...\n");
	fprintf(stderr,"%s\n",error_text);
	fprintf(stderr,"...now exiting to system...\n");
	exit(1);
}

double *vector(long nl, long nh)
/* allocate a double vector with subscript range v[nl..nh] */
{
	double *v;

	v=(double *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(double)));
	if (!v) nrerror("allocation failure in vector()");
	return v-nl+NR_END;
}

double **matrix(long nrl, long nrh, long ncl, long nch)
/* allocate a double matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	double **m;

	/* allocate pointers to rows */
	m=(double **) malloc((size_t)((nrow+NR_END)*sizeof(double*)));
	if (!m) nrerror("allocation failure 1 in matrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(double)));
	if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}


void free_vector(double *v, long nl, long nh)
/* free a double vector allocated with vector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}

void free_matrix(double **m, long nrl, long nrh, long ncl, long nch)
/* free a double matrix allocated by dmatrix() */
{
	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}


SimAnneal::SimAnneal() {
	total_move = 100;
	cur_move = 0;
	increment_move = 10;
}

void fixBound(double x[], double lower[], double upper[], int n) {
	for (int i = 1; i <= n; i++) {
		if (x[i] < lower[i])
			x[i] = lower[i];
		else if (x[i] > upper[i])
			x[i] = upper[i];
	}
}


double SimAnneal::amotsa(double **p, double y[], double psum[], int ndim, double pb[],
                         double *yb, int ihi, double *yhi, double fac, double lower[], double upper[]) {
	//double ran1(long *idum);
	int j;
	double fac1,fac2,yflu,ytry, *ptry;

	ptry=vector(1,ndim);
	fac1=(1.0-fac)/ndim;
	fac2=fac1-fac;
	for (j=1;j<=ndim;j++)
		ptry[j]=psum[j]*fac1-p[ihi][j]*fac2;
	fixBound(ptry, lower, upper, ndim);
	checkRange(ptry);
	ytry=targetFunk(ptry);
	if (ytry <= *yb) {
		for (j=1;j<=ndim;j++) pb[j]=ptry[j];
		*yb=ytry;
	}
	yflu=ytry-tt*log(ran1(&idum));
	if (yflu < *yhi) {
		y[ihi]=ytry;
		*yhi=yflu;
		for (j=1;j<=ndim;j++) {
			psum[j] += ptry[j]-p[ihi][j];
			p[ihi][j]=ptry[j];
		}
	}
	free_vector(ptry,1,ndim);
	return yflu;
}


void SimAnneal::amebsa(double **p, double y[], int ndim, double pb[], double *yb, double ftol,
                       int *iter, double temptr, double lower[], double upper[]) {
	/*
		double amotsa(double **p, double y[], double psum[], int ndim, double pb[],
			double *yb, double (*funk)(double []), int ihi, double *yhi, double fac);
		double ran1(long *idum);
		*/
	int i,ihi,ilo,j,m,n,mpts=ndim+1;
	double rtol,sum,swap,yhi,ylo,ynhi,ysave,yt,ytry, *psum;

	psum = vector(1, ndim);
	tt = -temptr;
	GET_PSUM
	for (;;) {
		ilo=1;
		ihi=2;
		ynhi=ylo=y[1]+tt*log(ran1(&idum));
		yhi=y[2]+tt*log(ran1(&idum));
		if (ylo > yhi) {
			ihi=1;
			ilo=2;
			ynhi=yhi;
			yhi=ylo;
			ylo=ynhi;
		}
		for (i=3;i<=mpts;i++) {
			yt=y[i]+tt*log(ran1(&idum));
			if (yt <= ylo) {
				ilo=i;
				ylo=yt;
			}
			if (yt > yhi) {
				ynhi=yhi;
				ihi=i;
				yhi=yt;
			} else if (yt > ynhi) {
				ynhi=yt;
			}
		}
		rtol=2.0*fabs(yhi-ylo)/(fabs(yhi)+fabs(ylo));
		if (rtol < ftol || *iter < 0) {
			swap=y[1];
			y[1]=y[ilo];
			y[ilo]=swap;
			for (n=1;n<=ndim;n++) {
				swap=p[1][n];
				p[1][n]=p[ilo][n];
				p[ilo][n]=swap;
			}
			break;
		}
		*iter -= 2;
		ytry=amotsa(p,y,psum,ndim,pb,yb,ihi,&yhi,-1.0, lower, upper);
		if (ytry <= ylo) {
			ytry=amotsa(p,y,psum,ndim,pb,yb,ihi,&yhi,2.0, lower, upper);
		} else if (ytry >= ynhi) {
			ysave=yhi;
			ytry=amotsa(p,y,psum,ndim,pb,yb,ihi,&yhi,0.5, lower, upper);
			if (ytry >= ysave) {
				for (i=1;i<=mpts;i++) {
					if (i != ilo) {
						for (j=1;j<=ndim;j++) {
							psum[j]=0.5*(p[i][j]+p[ilo][j]);
							p[i][j]=psum[j];
						}
						fixBound(p[i], lower, upper, ndim);
						fixBound(psum, lower, upper, ndim);
						checkRange(p[i]);
						checkRange(psum);
						y[i]=targetFunk(psum);
					}
				}
				*iter -= ndim;
				GET_PSUM
			}
		} else ++(*iter);
	}
	free_vector(psum,1,ndim);
}
#undef GET_PSUM
//#undef NRANSI

double SimAnneal::simulatedAnnealing(int ndim, double *variables, double *lower_bound, double *upper_bound) {
	idum = -RanNum::getRanInt(100000);
	double **simplex = new double*[ndim+2];
	double *simplex_funk = new double[ndim+2];
	double best_val = INFINITIVE;
	int iterations;

	int cnt;
	// initialize the simplex
	for (cnt = 1; cnt <= ndim+1; cnt++) {
		simplex[cnt] = new double[ndim+1];
		memmove(simplex[cnt], variables, (ndim+1) * sizeof(double));
		if (cnt > 1) {
			double diff = (upper_bound[cnt-1] - lower_bound[cnt-1]) / 8.0;
			if (diff > 1.0) 
				diff = 1.0;
			simplex[cnt][cnt-1] = variables[cnt-1] + diff;
			if (simplex[cnt][cnt-1] < lower_bound[cnt-1])
				simplex[cnt][cnt-1] = lower_bound[cnt-1];
			if (simplex[cnt][cnt-1] > upper_bound[cnt-1])
				simplex[cnt][cnt-1] = upper_bound[cnt-1];
		}
		checkRange(simplex[cnt]);
		simplex_funk[cnt] = targetFunk(simplex[cnt]);
	}
	//best_val = simplex_funk[1];

	// start the simulated annealing algorithm
	initSchedule();
	do {
		iterations = 50;
		amebsa(simplex, simplex_funk, ndim, variables, &best_val, EPS_MODEL_PAM_ERROR,
		       &iterations, temperature, lower_bound, upper_bound);
		coolingSchedule();
		cout << cur_move << ": ";
		for (cnt=1; cnt <= ndim; cnt++)
			cout << variables[cnt] << "  ";
		cout << best_val << endl;
	} while (!stopAnnealing());

	// delete the memory
	for (cnt=1; cnt <= ndim+1; cnt++)
		delete simplex[cnt];
	delete simplex_funk;
	delete simplex;
	return best_val;
}

void SimAnneal::initSchedule() {
	cur_move = 0;
	total_move = 10;
	increment_move = 1;
	init_temperature = 10;
	temperature = init_temperature;

}


/**
	return the new temperature
	@param temperature (IN OUT) the temperature 
*/
void SimAnneal::coolingSchedule() {
	cur_move ++;
	if (cur_move % increment_move == 0) {
		temperature = init_temperature * (1-cur_move/total_move);
	}
}

bool SimAnneal::stopAnnealing() {
	return cur_move == total_move;
}

SimAnneal::~SimAnneal() {}


/**
	the approximated derivative function
	@param x the input vector x
	@param dfx the derivative at x
	@return the function value at x
*/
double SimAnneal::derivativeFunk(double x[], double dfx[]) {
	/*
	if (!checkRange(x))
		return INFINITIVE;
	*/
	double fx = targetFunk(x);
	int ndim = getNDim();
	double h, temp;
	for (int dim = 1; dim <= ndim; dim++ ){
		temp = x[dim];
		h = ERROR_X * fabs(temp);
		if (h == 0.0) h = ERROR_X;
		x[dim] = temp + h;
		h = x[dim] - temp;
		dfx[dim] = (targetFunk(x) - fx) / h;
		x[dim] = temp;
	}
	return fx;
}
