 /* --------------------------------------------------------------- 
    xhkeys. Jul 2002
    Copyright (C) 2002,  Michael Glickman  <wmalms@yahoo.com>
    License: GPL
    --------------------------------------------------------------- */
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "xhkeys_conf.h"

#ifdef USE_DMALLOC
#include <dmalloc.h>
#endif

#ifdef PLUGIN_SUPPORT
#include <dlfcn.h>
#include "xhkeys_plugin.h"
#endif

#include "xhkeys.h"

const char *cancelledText = "Cancelled";
const char *timeoutText = "Cancelled by timeout";
const char *timeoutBye = "\nQuit on time-out. Bye\n";

// xhconf.c
extern Display *dpy;
extern Window wnd;
extern int keyTimeout, btnTimeout;
extern int count;
extern const char *resFileName;

// resources.c
extern const char *TargetSelCodes;
extern CmdDescr *Commands;
extern int CommandCount;
extern int maxCodes;

#ifdef PLUGIN_SUPPORT
// resources.c
extern PluginData *Plugins;
extern int PluginCount;
#endif

// Currently used only with xhkconfig (xkccmnds.c)
// 1 - has parameteres, 2 - ignores type modifier
const int SpecFuncFlags[] =
{
    1,			// SPEC_WNDROTATE, 
    3,			// SPEC_WNDKILL, 
#ifdef PLUGIN_SUPPORT    
    0,			// SPEC_KILLPLUGIN,	
#endif    
#ifdef OSD_SUPPORT
    2,			// SPEC_VERSION,	
#endif
    0			// SPEC_QUIT
};

static const char *CommandTypeNames[] = {
    "Internal Function",
    "Application",
#ifdef PLUGIN_SUPPORT
    "Plugin Call", 
#endif			
    "Key Event", "Mouse Event",
    "Cancel"
};

static const int CommandTypeCnt = sizeof(CommandTypeNames) / sizeof(char *);

extern const char *CommandTypeCodes;

// common.c
extern const char *SpecCodeNames[];
extern const int SpecCodeCount;

static const char *GenerateNames[] = {"None (just raise window)", "Press", "Release",  "Both"};
static const int GenerateNamesCount = sizeof(GenerateNames)/sizeof(char *);


static const char *StackOrderNames[] = {"Focused", "Topmost", "Bottom",  "Root",  "Select By Clicking"};
static const int StackOrderNameCount = sizeof(StackOrderNames) / sizeof(char *);

static Bool DisplaySpecCommandLine(CmdDescr *cmd, char **errMsg)
{
    int type_modif = cmd->type_modifier;
    int cmdCode, flag;
    char *cmdCopy = strdup(cmd->command);
    char *cmdPtr, *argPtr;
    char *title;

    if (cmdCopy == NULL) {
	*errMsg = "Memory allocation error";
	return False;
    }

    cmdPtr = cmdCopy;

    // Get rid of comment
    argPtr = strchr(cmdPtr,'#');
    if (argPtr != NULL) *argPtr = '\0';

    switch(type_modif) 
    {
        case 0:
	    title = "<< standard >>";
	    break;

        case 2:
	    title = "<< disabled >>";
	    break;
	    
	default:
	    title = strsep((char **)&cmdPtr, ";");
	    break;	    
     } 	

     argPtr = cmdPtr;
     strsep(&argPtr, "; \t#");

     for(cmdCode = 0; cmdCode < SpecCodeCount; cmdCode++)
     {
        if (strcasecmp(cmdPtr, SpecCodeNames[cmdCode]) == 0) break;
     }

    flag = 0;	
    if (cmdCode < SpecCodeCount) flag = SpecFuncFlags[cmdCode];
     
    if ((flag & 2) == 0)
         printf ("\tOSD title: %s\n", title);
    
     printf ("\tCommand:   %s", cmdPtr);

    if ((flag & 1) != 0 && argPtr != NULL && *argPtr != '\0')
	 printf (" %s", argPtr);

    if (cmdCode >= SpecCodeCount) printf ("  << invalid >>");
    printf ("\n");

     if (cmdCopy != NULL) free(cmdCopy);
     
     return True;

}

static Bool DisplayAppCommandLine(CmdDescr *cmd, char **errMsg)
{
    int type_modif = cmd->type_modifier;
    char *cmdCopy = NULL;
    const char *cmdPtr =  cmd->command;
    char *title;
     
    switch(type_modif) 
    {
        case 0:
	    title = "<< standard >>";
	    break;

        case 2:
	    title = "<< disabled >>";
	    break;
	    
	default:
	    cmdCopy = strdup(cmdPtr);
	    if (cmdCopy == NULL)
    	        title = "<< unavailable >>";
	    else {
	        cmdPtr = cmdCopy;
		title = strsep((char **)&cmdPtr, ";");
	    }
	    break;	    
     } 	
	
     // Explicit title
     printf ("\tOSD title: %s\n", title);
     printf ("\tCommand:   %s\n", cmdPtr);
     
     if (cmdCopy != NULL) free(cmdCopy);
     
     return True;
}	

#ifdef PLUGIN_SUPPORT
Bool GetPluginOpcodeFlags(const PluginData *plData,
                         int opCode, int *opFlagsPtr)
{
    XHKEYS_PLUGIN_GETOPDATA *getopdataPtr;

    getopdataPtr = (XHKEYS_PLUGIN_GETOPDATA *) dlsym(plData->handle, "getopdata");
    if (getopdataPtr == NULL)  return False;

    return (*getopdataPtr)(opCode, NULL, opFlagsPtr);
    
}			 




Bool GetPluginInfoText(const PluginData *plData, char *text, int txtLength)
{
    int len;
    const char *name = plData->name;
    if (name == NULL) return False;
        
    strncpy(text, name, txtLength);
	    
    name = plData->shortName;
    if (name != NULL) {
        len = strlen(text);
	if (len < txtLength)
	    snprintf(text+len,  txtLength-len, "(%s)", name);
    }
	    		 
    name = plData->initString;
    if (name != NULL) {
        len = strlen(text);
	while (len < 20 && len<txtLength) text[len++] = ' ';
	if (len < txtLength)
    	    snprintf(text+len,  txtLength-len,  "  init: %s", name);
    } 
    
    return True;
}    
    

static Bool DisplayPluginCommandLine(CmdDescr *cmd, char **errMsg)
{
    PluginData *plData = NULL;
    char pluginDataText[81];
    char *pluginNameNo, *tmp;
    char *cmdCopy, *strNext;
    const char *opName, *opParms;
    int opFlags;
    int index = -1;
    Bool isCommandValid;

    strNext = cmdCopy = strdup(cmd->command);
    if (cmdCopy == NULL) {
	*errMsg = "Memory allocation error";
	return False;
    }

    pluginNameNo= strsep(&strNext, ";");	// Plugin name / line no
        
    if (cmd->type_modifier == 1)  {
	index = strtol(pluginNameNo, &tmp, 0);
	if (*tmp == '\0') {
	    printf("\tLine number: %d\n", index);
	    index = FindPluginByLineNo(index);
	}
	else 
	    printf("\tLine number: << invalid >>\n");
	
    } else {
        printf("\tName/Alias:  %s\n", pluginNameNo);
	index = FindPluginByName(pluginNameNo);
    }	

    strcpy(pluginDataText, "<< unavailable >>");
    if (index >= 0) {
        plData = Plugins+index;
	GetPluginInfoText(plData, pluginDataText, sizeof(pluginDataText));
    } // if (index >= 0)		
    printf ("\tInformation: %s\n", pluginDataText);
    
    isCommandValid = False;
            
    opName = strsep(&strNext, " \t#");
    opFlags = 0;
    
    if (opName == NULL)	
	opName = "";
    else
    if (plData != NULL)
	isCommandValid = GetPluginOpnameInfo(plData, opName, NULL, &opFlags);

    
    if (strNext != NULL && (opFlags & XHKEYS_PLUGOPTYPE_PARMSPECCHARS) == 0) {
	strNext += strspn (strNext, " \t");
	opParms = strsep(&strNext, " \t#");
    } else
	opParms = strNext;

    printf ("\tCommand:     %s", opName);
    if (opParms != NULL) printf (" %s", opParms);

    if (isCommandValid == False)
	printf (" << invalid >>");

    printf("\n");
    
    if ((opFlags & XHKEYS_PLUGOPTYPE_CANCONTINUE) != 0) 
	printf ("\tCalling:     %s\n", 
	    cmd->target_selection > 0 ? "continuously" : "once");

    free(cmdCopy);
    return True;    
}
#endif

static Bool DisplayKeyBtnEventCommandLine(CmdDescr *cmd, char **errMsg)
{	//  CT_KEYEVENT, CT_BTNEVENT
    char *ptr1, *ptr2;
    char *cmdCopy, *strNext;
    unsigned int val;
    unsigned long lval;
    unsigned int type; 

    strNext = cmdCopy = strdup(cmd->command);
    if (cmdCopy == NULL) {
	*errMsg = "Memory allocation error";
	return False;
    }
    
    type = cmd->type;
    ptr1 = strsep(&strNext, ";");

    // Select target window by ...
    switch (cmd->target_selection)
    {
	case 0:
	    // ... app name and class
	    ptr2 = ptr1;
	    ptr1 = strsep(&ptr2, ":");
	    if (ptr1 == NULL) ptr1 = "<< invalid >>";
	    if (ptr2 == NULL) ptr2 = "<< invalid >>";
	    printf ("\tWnd App Name:  %s   WndAppClass: %s\n", ptr1, ptr2);
	    break;

	case 1:
	    // ... window title
	    printf ("\tWnd Title:  '%s'\n", ptr1);
	    break;
		
	case 2:
	{
	    // ... window ID
	    lval = strtoul(ptr1, (char **)NULL, 0);	  
//	    lval = strtoul(ptr1, &ptr2, 0);	  
//	    if (*ptr2 != '\0' && strchr("# \t", *ptr) == NULL)
//	        printf ("\tWnd ID:  '%s' << invalid, need to be numeric !! >>", ptr1);		
//	    else	
		printf ("\tWnd ID:  %10ld (0x%lx)\n", lval, lval);		
	}	    	
	    break;

	case 3:
	    // ... window stacking order ("Topmost, bottom, root")
	{
	    char sCode = toupper(*ptr1);
	    const char *ordName = "Unknown";
	    const char *ordnameCurrent;
	    int j;
		
	    for (j=0; j<3; j++) {
		ordnameCurrent = StackOrderNames[j];
		if (*ordnameCurrent  == sCode) {
		    ordName = ordnameCurrent;
		    break;
		}
	    }		
	    
	    printf ("\tWnd Order:  %s\n", ordName);
	}	
	    break;
    }	

    ptr1 = strsep(&strNext, ";");
    
    if (ptr1 != NULL && type == CT_BTNEVENT)  {
	// Location of immitated mouse click 
	ptr2 = strsep(&strNext, ";");
	if (ptr2 == NULL)
	    ptr1 = NULL;
	else {
	    int x = (int) strtol(ptr1, (char **) NULL, 0);
	    int y = (int) strtol(ptr2, (char **) NULL, 0);
	    printf ("\tWnd Location (x,y):  %d, %d\n", x, y);
	    ptr1 = strsep(&strNext, ";"); 
	}
    }

    if (ptr1 != NULL) {
	// Scan code / KeySym (CT_KEYEVENT) or mouse button (CT_BTNEVENT)
	unsigned long code;
	
	ptr2 = strsep(&strNext, ";");
	val = (ptr2 == NULL) ? 0 : strtol(ptr2, (char **)NULL, 0);

	if (type == CT_KEYEVENT) {
	    switch(cmd->type_modifier)
	    {
		case 0:	    
		    // By KeySym
		    printf ("\tKeysym: %s  State %u (0x%x)\n", ptr1, val, val);
		    break;

		case 1:	    
		    // By ScanCode
		    code = strtoul(ptr1, (char **)NULL, 0);
	            printf ("\tScan code: %lu (0x%lx)  State %u (0x%x)\n", code, code, val, val);
		    break;
		
		case 2:
		    // By key code    
		    code = strtoul(ptr1, (char **)NULL, 0);	    
	            printf ("\tKeysym code: %lu (0x%lx)  State %u (0x%x)\n", code, code, val, val);
	    }
	    
	}
	else	  
//	if (type == CT_BTNEVENT) {
	{
	    code = strtoul(ptr1, (char **)NULL, 0);
	    printf ("\tMouse button: %lu   State %u (0x%x)\n", code, val, val);
	}
	  	  
	ptr1 = strsep(&strNext, ";");
    }  

    val = (ptr1 == NULL) ? 3 : strtol(ptr1, (char **)NULL, 0);
    printf ("\tGenerate: %d [%s]", val, GenerateNames[val]);

    if (ptr1 != NULL  && (ptr1 = strsep(&strNext, "")) != NULL &&
	    (val = strtol(ptr1, (char **)NULL, 0)) > 0)
		  printf ("   %d time(-s)", val);
		  
    printf ("\n");	  

    if (cmdCopy != NULL) free(cmdCopy);    
    
    return True;
}


static Bool displayCommandLine(CmdDescr *cmd, char **errMsg)
{
    KeyCode code;
    unsigned int type;
    Bool result;    	
  
    code = cmd->keycode;
    type = cmd->modifier;
    printf ("Scan code: %3u (0x%02x)  State: %3u (0x%02x)\n",
            code, code, type, type); 	  

    type = cmd->type;  
    printf ("\tOperType:  %c [%s]\n", CommandTypeCodes[type], CommandTypeNames[type]);


    switch(type)  
    {
	case CT_INTERNAL:
	    result = DisplaySpecCommandLine(cmd, errMsg);
	    break;
	    
	case CT_EXTAPP: 
	    result = DisplayAppCommandLine(cmd, errMsg);
	    break;
	    
#ifdef PLUGIN_SUPPORT  
	case CT_PLUGIN: 
	    result = DisplayPluginCommandLine(cmd, errMsg);
	    break;	
#endif  

//	case CT_KEYEVENT:
//	case CT_BTNEVENT:
	default:
	    result = DisplayKeyBtnEventCommandLine(cmd, errMsg);
    }	    

    if (result)
	printf ("\n");
    else	
	fprintf (stderr, "%s\n", *errMsg);      

    return result;  
}

static Bool displayCommandEntry(int index, char **errMsg)
{
  return displayCommandLine(&(Commands[index]), errMsg);
}

static int findCommandByCode(unsigned int scanCode, unsigned int modif)
{
  int i;
  
  for (i=0; i<CommandCount; i++)
  {
	if (Commands[i].keycode == scanCode &&
	    Commands[i].modifier == modif) return i;
  }
  return -1;
}

Bool displayCommandByCode(unsigned int scanCode, unsigned int modif, char **errMsg)
{
  int i = findCommandByCode(scanCode, modif);
  
  if (i>=0)
	return displayCommandEntry(i, errMsg);

  printf ("Scan code: %3u (0x%02x)  State: %3u (0x%02x)\n",
            scanCode, scanCode, modif, modif); 	  
  printf ("\tUnassigned\n\n");
  return True;
}

Bool displayCommand(char **errMsg)
{
  XEvent xev;
  short res;
  short cnt;
  
  printf("Type any key to see the associated command\n");
  printf ("Quits ");
  if (count > 1)  printf("after %d entries, or ", count);
  printf ("if idle within %d secs\n", keyTimeout);
  cnt = count;
  
  while((res = getScanCodeEvent(&xev, keyTimeout, errMsg)) == 1)
  {
	if (displayCommandByCode(xev.xkey.keycode, xev.xkey.state, errMsg) == False)
	  return False;
	if (cnt > 0 && --cnt==0) break;
  }	 

  if (res == 2)
	printf(timeoutBye);
  return (res != 0);	 
}

Bool displayAllCommands(char **errMsg)
{
  int i;
  
  for (i=0; i<CommandCount; i++)
  {
	if (!displayCommandEntry(i, errMsg)) return False;
  }

  return True;

}

/* -------------------------------------------------------------- */
// Processes Backspaces allright and gets rid of \n and \r 

void enterText(const char *prompt, char *answer, int len)
{
  char *sep;

  printf (prompt);
  fgets(answer, len, stdin);
  fflush(stdin);
  sep = strpbrk(answer, "\r\n");
  if (sep) *sep = '\0';
}

Bool YesNo(const char *question, Bool def_answer)
{
  char qcopy[121], reply[5];
  int  repval = -1;
  
  strcpy(qcopy, question);
  strcat(qcopy, def_answer ? "? (y, n): " : "? (n, y): ");
  
  while (repval < 0) {
	enterText(qcopy, reply, sizeof(reply));

	switch(toupper(*reply)) {
	  case 'Y':  repval = 1; break;
	  case 'N':  repval = 0; break;

	  case '\r':
	  case '\n':
	  case '\0': repval = (int)def_answer;
	}	
  }	  

  return (Bool) repval; 
}

#ifdef OSD_SUPPORT
static int GetOSDTitle(char *title, int titleLength)
{
    char answer[4];

    *title = '\0';
    
    for(;;) {
        printf ("\nOSD title types:\n");
	printf ("\t1. Standard (command line)\n");
        printf ("\t2. Custom\n");
	printf ("\t3. Disabled (no OSD message)\n");
        enterText("Select title type: ", answer, sizeof(answer));
    
	switch(answer[0])
	{
	    case '\0':
	    case '1':	
		return 0;

	    case '3':
		return 2;

	    case '2':
		do {
	            enterText("Enter OSD title: ", title, titleLength);
		    printf ("\n");
		} while(strlen(title) == 0);
		return 1;
	}
    }                

}
#endif

// Defval = -2 - only by number
static int DisplayMenu(const char *title, const char **lines, int count, int defVal)
{
    int i, j, iter;
    char answer[5], *ansnext;
    char letter, lineletter;
    const char *curLine;
    Bool selectByLetter = False;
    const char *hdrText; 
    
    if (defVal > -2) {
	hdrText = "Enter selection by number, or cap. letter: ";
	selectByLetter = True;
    } else {	
	hdrText = "Enter selection:"; 
	selectByLetter = False;
	defVal = -1;
    }	

    printf ("\n%s\n", title);
    iter = 3;
  
    while(--iter >= 0) {
	for (i=0; i<count; i++)
	  printf ("%d. %s\n", i+1, lines[i]);
  	
	enterText(hdrText, answer, sizeof(answer));
	letter = answer[0];

	if (letter == '\0') return defVal;

	if (isdigit(letter)) {	
	    // Selection by number
	    i = (int) strtol(answer, &ansnext, 10);
	    if (i<= count && *ansnext == '\0') return i-1;
	} else 
	if (selectByLetter) {  
	    // Selection by letter       
	    for (i=0; i<count; i++) {
		curLine = lines[i];
		lineletter = '\0';
		for (j=0; ; j++) {
		    lineletter = curLine[j];
		    if (lineletter == '\0' ||
		        isupper(lineletter)) break;
		}
		
		if (lineletter == toupper(letter))
		    return i;		
	    }
	}	  
    }

  return -1;  
}

Bool editSpecFunction(char *command, int sze, 
#ifdef OSD_SUPPORT		
		     short *osd_title_type_ptr,
#endif
		     char **errMsg)
{
  int fcode;
  char parms[41];
  int iter, flag;
#ifdef OSD_SUPPORT
  int osd_title_type;
  char osd_title[41];
#endif

  iter = 3;
  
  do {
	enterText("Function Code (H-help): ", command, sze);

	if (command[0] == '\0') {
	  *errMsg = (char *) cancelledText ;
	   return False;
	}
	
	if (strcasecmp(command, "H") == 0 || 
	    strcasecmp(command, "HELP") == 0) {
	   fcode = DisplayMenu("Select function", SpecCodeNames, SpecCodeCount, -2);
	   printf ("\nSelected function: '%s'\n\n", SpecCodeNames[fcode]);
	}  
	else {
	  for (fcode = 0; 
		fcode < SpecCodeCount && strcasecmp(SpecCodeNames[fcode], command) != 0;
		fcode++);
	  
	  if (fcode >= SpecCodeCount) fcode = -1;
	}	
  
	if (fcode < 0) 
	  printf ("Sorry, that was an invalid code\n");

  }	while ((--iter > 0) && (fcode < 0));  
  
  if (fcode < 0) {
    *errMsg = (char *) cancelledText ;
	return False;
  }	
  
  *parms = '\0';
 

  flag = SpecFuncFlags[fcode];

#ifdef OSD_SUPPORT
  osd_title_type = 0;
  if ((flag & 2) == 0) 
    osd_title_type = GetOSDTitle(osd_title, sizeof(osd_title));
#endif

  if ((flag & 1) != 0)
     enterText("Arguments: ", parms, sizeof(parms));

#ifdef OSD_SUPPORT
   *osd_title_type_ptr = osd_title_type;
   if (osd_title_type == 1)
      sprintf (command, "%s;%s %s", osd_title, SpecCodeNames[fcode], parms);
   else
#endif   
      sprintf (command, "%s %s", SpecCodeNames[fcode], parms);

  return True;
}

#ifdef OSD_SUPPORT		
Bool editExternalApp(char *command, int sze, 
		     short *osd_title_type_ptr,
	             char **errMsg)
{
    int osd_title_type = 0;
    Bool success = False;
    char *command1 = malloc(sze);

    if (command1 == NULL) command1 = command;

    enterText("Enter the shell command to call the application:\n", command1, sze);

    if (*command == '\0') goto HastaLaVista;
	
    if (command1 != command ) {
	char osd_title[41];
	
	if (command1 != command)
    	    osd_title_type = GetOSDTitle(osd_title, sizeof(osd_title));
    
	if (osd_title_type == 1)
	    snprintf(command, sze, "%s;%s", osd_title, command1);
	else
	    strncpy(command, command1, sze);	

    }	

    *osd_title_type_ptr = osd_title_type;   
    success = True;

HastaLaVista:
    if (command1 != command) free(command1);
    return success;	
}
#else
Bool editExternalApp(char *command, int sze, char **errMsg)
{		      
  enterText("Enter the shell command to call the application:\n", command, sze);
  return (*command != '\0') ;
}
#endif

Bool getEventTarget(char *data, int sze, short *target_sel, // short canSelectOrder, 
                   int *xptr, int *yptr, int *button, int *state, char **errMsg)
{
  Window cur, tmp;
  XClassHint xch;
  XEvent xev;
  Bool tsel_defined;
  XTextProperty xtp;
  char **list;
  int i;

/*
  char keyPressed;
  
  if (canSelectOrder) {
  	i = getMouseOrKeyEvent(&xev, btnTimeout, TargetSelCodes, &keyPressed, errMsg);
	
	if ( i == 2) {
  		// Key has been pressed
		*target_sel = 3;	
		*data = keyPressed;
		*(data+1) = '\0';
		return True;
	}			
	
  }	
  else
*/  
  	i = getMouseEvent(&xev, btnTimeout, errMsg);
             	

  
  
  if (i != 1) {
  	if (*errMsg == NULL) errMsg = (char **) timeoutText;
	return False;
  }

  //  Valid button press
  tmp = xev.xbutton.subwindow;
  if (tmp == None)
	cur = wnd;
  else {  
	cur = findClientWindow(tmp, -1);
	if (cur == None) cur = tmp;
  }  

  // res_name + ress_class myst exist and be unique
  tsel_defined = False;
  if (XGetClassHint(dpy, cur, &xch) && 
      countWindowsByWMClass(wnd, xch.res_name, xch.res_class) == 1) {
	tsel_defined = True;
	*target_sel = 0;	
	snprintf(data, sze, "%s:%s", xch.res_name, xch.res_class);
  } else
  if (XGetWMName(dpy, cur, &xtp) &&
      XTextPropertyToStringList(&xtp, &list, &i)) {
	  if (i > 0) {
		tsel_defined = True;
		*target_sel = 1;	
		strncpy(data, list[0], sze);
	  }	
  }  	 

  if (tsel_defined == False) {
	*target_sel = 2;	
	snprintf(data, sze, "0x%lx", (long)cur);		
  }	

  if (xptr != NULL && yptr != NULL)
  	  XTranslateCoordinates(dpy, xev.xbutton.root, cur,
	                               xev.xbutton.x_root,  xev.xbutton.y_root,
								   xptr, yptr, &tmp);
  if (button != NULL)
	*button = xev.xbutton.button;

  if (state != NULL)
	*state = xev.xbutton.state;
	
  return True;
}

Bool getEventType(int *type, char **errMsg)
{
  int sel;
	
  sel = DisplayMenu("Select event(-s) to generate:", 
             GenerateNames, GenerateNamesCount, GenerateNamesCount-1);
  if (sel < 0) {
    *errMsg = (char *) cancelledText ;
   return False;
 }

  *type = sel;  

  return True;
}

Bool getEventCount(int *countPtr, char **errMsg)
{
    char countStr[6], *next;
    int count;

    enterText("Select event count (1 by default)\n", countStr, sizeof(countStr));

    if (countStr[0] == '\0') {
	*countPtr = 0;
	return True;
    }	

    count = strtol(countStr, &next, 0);
    if (*next=='\0' && count > 0) {
	*countPtr = count;
	return True;  
    }	

    *errMsg = (char *) cancelledText ;
    return False;
}  

#ifdef PLUGIN_SUPPORT
char **collectPluginNames(int *countPtr)
{
    int i, l, count = 0;
    const PluginData *plData;    
    char **PluginNames;
    const char *name, *initString, *shortName;
    char *ptr;

    PluginNames = (char **)malloc(PluginCount * sizeof(char *));
    if (PluginNames == NULL) return NULL;
    
    for (i=0; i<PluginCount; i++) {
	plData = Plugins + i;
	name = plData->name;
	if (name == NULL) continue;
	l = strlen(name) + 1;		// Including tgrainig zero
	
	shortName = plData->shortName;
	if (shortName != NULL) l += strlen(shortName) + 2;
	
	initString = plData->initString;
	if (shortName != NULL) l += strlen(initString) + 5;
	
	ptr = malloc(l);
	if (ptr == NULL) continue;
	
	PluginNames[count++] = ptr;
	
	strcpy(ptr, name);

	if (shortName != NULL)
	    sprintf (ptr + strlen(ptr), "(%s)", shortName);
			
	if (initString != NULL)
	    sprintf (ptr + strlen(ptr), ":   %s", initString);
    }

    *countPtr = count;
    return PluginNames;
}

void freeCollectedNames(char **Names, int count)
{
    int i;
    
    for (i=0; i<count; i++) free(Names[i]);

    free(Names);
}

char **collectPluginOpNames(const PluginData *plData, int *countPtr)
{
    PLUGIN_HANDLE handle;
    XHKEYS_PLUGIN_GETOPDATA *getopdataPtr;
    XHKEYS_PLUGIN_GETOPCOUNT *getopcountPtr;
    char opNameTmp[XHKEYS_PLUGIN_OPNAME_LENGTH];
    int opCount, opCode, opFlags, count;
    char **OpNames;

    
    handle = plData->handle;    
    getopcountPtr = (XHKEYS_PLUGIN_GETOPCOUNT *) dlsym(handle, "getopcount");
    getopdataPtr = (XHKEYS_PLUGIN_GETOPDATA *) dlsym(handle, "getopdata");
    if (getopcountPtr == NULL || getopdataPtr == NULL)  return NULL;
    
    opCount = (*getopcountPtr)();
    if (opCount == 0) return NULL;

    OpNames = (char **)malloc(opCount * sizeof(char *));
    if (OpNames == NULL) return NULL;

    count = 0;

    for (opCode = 0; opCode < opCount; opCode++)
    {
	if ((*getopdataPtr)(opCode, opNameTmp, &opFlags)) 
	    OpNames[count++] = strdup(opNameTmp);
	
    }

    *countPtr = count;
    return OpNames;
}

Bool editPluginCall(char *command, int sze,
                  short *target_sel, short *type_modif, char **errMsg)
{		  
    int count;
    int index =-1;
    int iter = 3;
    PluginData *plData;
    char **Names;
    char opName[XHKEYS_PLUGIN_OPNAME_LENGTH];
    int opFlags;

    if (PluginCount <= 0) {
	*errMsg = "No plugins configured. Use 'xhkconf -p' to configure plugins";
        return False;
    }
    
    
    do {
	enterText("Plugin name/alias (H-help): ", command, sze);

	if (*command == '\0') {
    	    *errMsg = (char *) cancelledText ;
	    return False;
        }

	if (strcasecmp(command, "H") == 0 || 
    	    strcasecmp(command, "HELP") == 0) {

	    *type_modif = 1;
	    Names = collectPluginNames(&count);
	    if (Names != NULL) {
		index = DisplayMenu("Select plugin", (const char **)Names, count, -2);
	        freeCollectedNames(Names, count);
	    }	
	    
        }	 
	else {

	    *type_modif = 0;
	    index = FindPluginByName(command);
	    if (index < 0) 
		printf ("Sorry, that was an invalid name/alias\n");
	}	

    } while (--iter >= 0 && index < 0);    
    
    if (index < 0) {
	*errMsg = (char *) cancelledText ;
        return False;
    }

    plData = Plugins + index;

    {
	char pluginInfo[121];
	if (GetPluginInfoText(plData, pluginInfo, sizeof(pluginInfo)))
	    printf ("Selected plugin: %s\n", pluginInfo);
    }	    
    
    if (*type_modif == 1)
	snprintf(command, sze, "%d", plData->lineno);
    // otherwise command alrady has what we are after

    iter = 3;
    
    do {
	enterText("Enter operation (H-help): ", opName, sizeof(opName));
        if (*opName == '\0' || strchr("Qq", *opName) != NULL) {
	    *errMsg = (char *) cancelledText ;
	    return False;
        }


        if (strcasecmp(opName, "H") == 0 || 
    	    strcasecmp(opName, "HELP") == 0) {
	    int opCode;	        

	    Names = collectPluginOpNames(plData, &count);
	
	    if (Names == NULL) {
		*errMsg = (char *) "Plugin is invalid (No getopcount/getopdata ?)";
	        return False;
	    }
	
	    opCode = DisplayMenu("Select operation", (const char **)Names, count, -2);
	    if (opCode >= 0) strcpy(opName, Names[opCode]);
	    freeCollectedNames(Names, count);

	    if (opCode < 0) {
		*errMsg = (char *) cancelledText ;
	        return False;
	    }

	    GetPluginOpcodeFlags(plData, opCode, &opFlags);

            printf ("\nSelected operation: '%s'\n", opName);
	    break;
	    
	} else
	if (GetPluginOpnameInfo(plData, opName, NULL, &opFlags))
	     break;

	printf ("Operation name is invalid\n");
    } while (--iter >= 0);

    if (iter < 0) {
	*errMsg = (char *) cancelledText ;
        return False;
    }

    count =  strlen(command);
    snprintf(command+count, sze-count, ";%s", opName);

    count = opFlags & 3;

    if(count != XHKEYS_PLUGOPTYPE_PARMNONE) {
	char *opArgs = malloc(sze);
	
	if (opArgs == NULL) {
	    *errMsg = "Memory allocation error";
	    return False;
	}


	iter = 2;    
	do {
	    if (count == XHKEYS_PLUGOPTYPE_PARMOPTIONAL) 
		enterText("Arguments (if needed): ", opArgs, sze);
	    else {
		enterText("Argument(-s): ", opArgs, sze);
	        if (*opArgs == '\0') {
		    printf("\nArgument is required!\n");
		    continue;
		}        			
	    }    
	
	    if ((opFlags & XHKEYS_PLUGOPTYPE_PARMSPECCHARS) == 0 &&
	        strpbrk(opArgs, " \t#") != NULL) {
		printf ("\nArgument must not contain Space, Tab or #");	
		continue;
	    }	
	    
	    break;
	} while (--iter >= 0);

	if (*opArgs != '\0') {
	    count = strlen(command);
	    snprintf(command + count, sze-count, " %s", opArgs);
	}
	    
	free(opArgs);
	
        if (iter < 0) {
	    *errMsg = (char *) cancelledText ;
	    return False;
	}
    }

    *target_sel = 0;
    if ((opFlags & XHKEYS_PLUGOPTYPE_CANCONTINUE) != 0 &&
	YesNo("Call continuously", False))
	*target_sel = 1;	
	
    return True;
}
    
#endif

Bool editKeyEvent(char *command, int sze,
                  short *target_sel, short *type_modif, char **errMsg)
{
    XEvent xev;
 /*  char answ[4]; */
    int curSize, type;
    const char *targetWndName;

/*
  enterText("\nBring up the window to get key events,\ntype Y when ready: ", answ, sizeof(answ));
  if (toupper(answ[0]) != 'Y') {
	*errMsg = (char *) cancelledText;
	return False;
  }
*/  

    type = DisplayMenu("Select window to recieve key event(-s)",    
	    StackOrderNames,  StackOrderNameCount, StackOrderNameCount-1);

    if (type < (StackOrderNameCount-1)) {
    	*target_sel = 3;	
	*command = StackOrderNames[type][0];
	*(command+1) = '\0';

	targetWndName = StackOrderNames[type];
    } else {	
	// Select by click
	printf ("\nClick a window to receive key event(-s)\n");
        if (getEventTarget(command, sze, target_sel, // 1,
                      NULL, NULL, NULL, NULL, errMsg) == False) 
	    return False;
	
	targetWndName = command;
    }

    printf ("\nTarget window : %s\n", targetWndName);
	
    printf ("Press a key (key combination) to send\n");
    *type_modif = 1;			// Scan code
    if (getScanCodeEvent(&xev, keyTimeout, errMsg) != 1) {
	if (errMsg == NULL) *errMsg = (char *) timeoutText;
	return False;
    } 

    // Process code and modifier
    printf ("\nScan code;modiifier:  %d;%d\n", xev.xkey.keycode, xev.xkey.state);
    curSize = strlen(command);
    snprintf(command+curSize, sze-curSize, ";%d;%d", xev.xkey.keycode, xev.xkey.state);

  // Process code and modifier

    if (getEventType(&type, errMsg) == False) return False;
    curSize = strlen(command);
    snprintf(command+curSize, sze-curSize, ";%d", type);

    if (type > 0) {    
	if (getEventCount(&type, errMsg) == False) return False;

	if (type > 0) {
	    curSize = strlen(command);
	    snprintf(command+curSize, sze-curSize, ";%d", type);
	}
    }		

  return True;
}

Bool editButtonEvent(char *command, int sze, short *target_sel, char **errMsg)
{
    int curSize, type;
    int x, y;
    int state, button;
    char answ[4];

    enterText("\nBring up the window to get the mouse click,\ntype Y when ready: ", answ, sizeof(answ));
    if (toupper(answ[0]) != 'Y') {
	*errMsg = (char *)cancelledText ;
	return False;
    }
  
    printf ("\nPerform a mouse click to be simulated\n");
    if (getEventTarget(command, sze, target_sel, // 0,
           &x, &y, &button, &state, errMsg) == False) {
	*errMsg = (char *) cancelledText ;					  
	return False;
    }	

    printf ("\nTarget Window: %s\n", command);
	
    curSize = strlen(command);
    snprintf(command+curSize, sze-curSize, ";%d;%d;%d;%d", x, y, button, state);

    if (getEventType(&type, errMsg) == False) return False;
    curSize = strlen(command);
    snprintf(command+curSize, sze-curSize, ";%d", type);
  
    if (type > 0) {    
	if (getEventCount(&type, errMsg) == False) return False;
	if (type > 0) {
	    curSize = strlen(command);
	    snprintf(command+curSize, sze-curSize, ";%d", type);
	}	
    }	
    return True;
}

int GetFreePosition()
{
  int i;
  for (i=0; i<CommandCount && Commands[i].lineno <= (i+1); i++);
 // if (i>=maxCodes) maxCodes = i+1;

  return i;
}


Bool editCommandByCode(unsigned int scanCode, unsigned int modif, char **errMsg)
{
  int pos, iter;
  int cmd_type;
  CmdDescr cmdNew;
  char command[1024];
  Bool success;
  
  pos  = findCommandByCode(scanCode, modif);

  if (pos >= 0 && displayCommandEntry(pos, errMsg) &&
	  YesNo ("Overwrite", False) == False) {
	*errMsg = (char *) cancelledText;  return False;
  } else 	
    printf ("\nUnused scan code;modiifier:  %d;%d\n", scanCode, modif);

  success = False;
  cmdNew.keycode = scanCode;
  cmdNew.modifier = modif;
  cmdNew.type_modifier = 0;
  cmdNew.target_selection = 0;
  cmdNew.command = NULL;
    
  iter = 3;
    
  do
  {
	cmd_type =DisplayMenu("Select command type", CommandTypeNames, 
				CommandTypeCnt, CommandTypeCnt-1);
	if (cmd_type < 0) break;
  

	cmdNew.type = cmd_type;

	switch (cmd_type) 
	{
	  case CT_INTERNAL:
		success = editSpecFunction(command, sizeof(command),
#ifdef OSD_SUPPORT		
		                     &(cmdNew.type_modifier),
#endif					 
				     errMsg);
		break;

	  case  CT_EXTAPP:
		success = editExternalApp(command, sizeof(command),
#ifdef OSD_SUPPORT		
		                      &(cmdNew.type_modifier),
#endif					 
	                              errMsg);
	    break;

#ifdef PLUGIN_SUPPORT
	  case  CT_PLUGIN:
		success = editPluginCall(command, sizeof(command),
		            &(cmdNew.target_selection), &(cmdNew.type_modifier), errMsg);
		break;
#endif		

	  case  CT_KEYEVENT:
		success = editKeyEvent(command, sizeof(command),
		            &(cmdNew.target_selection), &(cmdNew.type_modifier), errMsg);
		break;

	  case  CT_BTNEVENT:
		success = editButtonEvent(command, sizeof(command),
		            &(cmdNew.target_selection), errMsg);
		break;
		
	    default:	
		goto TheEnd;	// Cancelled
	}  

	if (success) {
	  cmdNew.command = strdup(command);
	  printf ("\n");
	  success  = displayCommandLine(&cmdNew, errMsg);
	  if (success)
		success = YesNo ("Accept", True);
	}

  } while ((--iter > 0) && (success == False));

  if (success == False) goto TheEnd;

  if (pos >= 0) {
    cmdNew.lineno = Commands[pos].lineno;
    if (Commands[pos].command) free((char *)(Commands[pos].command));
  } else {
    pos = GetFreePosition();
    cmdNew.lineno = pos+1;
	CommandCount++;

    Commands = realloc(Commands, CommandCount*sizeof(CmdDescr));
    success = (Commands != NULL);
	if (success == False) goto TheEnd;
		  
	for (iter = CommandCount-1; iter > pos; iter--)
	  memcpy(&Commands[iter], &Commands[iter-1], sizeof(CmdDescr));	
  }	

  memcpy(&Commands[pos], &cmdNew, sizeof(CmdDescr));
  success = UpdateResource(resFileName, pos, errMsg);

TheEnd:
  if (success == False && cmdNew.command)
	free((char *)(cmdNew.command));
  return success;
}


Bool editCommand(char **errMsg)
{
  XEvent xev;
  short res;
  short cnt;
  
  cnt = count;

  do
  {
	printf ("----------- ADD/MODIFY hot key -------------\n");
	printf("Type a key combination you wish to add/modify\n");
	printf ("Quits ");
	if (cnt > 1)  printf("after %d entries, or ", cnt);
	printf ("if idle within %d secs\n", keyTimeout);
	res = getScanCodeEvent(&xev, keyTimeout, errMsg);
	if (res != 1)  break;
	editCommandByCode(xev.xkey.keycode, xev.xkey.state, errMsg);
	printf ("\n");
  }	while (cnt == 0 || --cnt > 0);

  if (res == 2)
	printf(timeoutBye);
  return (res != 0);	 
}


/* ---------------------------------------------------------------------- */
Bool deleteCommandByCode(unsigned int scanCode, unsigned int modif, char **errMsg)
{
  int pos;
  
  pos  = findCommandByCode(scanCode, modif);

  if (pos < 0) {
	*errMsg = "Key/key combination is not assigned";
	return False;
  }

  if (pos >= 0 && displayCommandEntry(pos, errMsg) &&
	  YesNo ("Delete", False) == False) {
	*errMsg = (char *) cancelledText;  return False;
  }	

  Commands[pos].type = CT_DELETED;

  if (UpdateResource(resFileName, pos, errMsg)) {
	int iter;

    CommandCount--;
	for (iter = pos; iter < CommandCount; iter++)
	  memcpy(&Commands[iter], &Commands[iter+1], sizeof(CmdDescr));	

	return True;
  }

  return False;
}


Bool deleteCommand(char **errMsg)
{
  XEvent xev;
  short res;
  short cnt;
  
  cnt = count;

  do
  {
	printf ("---------  DELETE  hot  key   ----------\n");
        printf("Type a key combination you wish to delete\n");
	printf ("Quits ");
	if (cnt > 1)  printf("after %d entries, or ", cnt);
	printf ("if idle within %d secs\n", keyTimeout);
	res = getScanCodeEvent(&xev, keyTimeout, errMsg);
	if (res != 1)  break;
	deleteCommandByCode(xev.xkey.keycode, xev.xkey.state, errMsg);
	printf ("\n");
  }	while (cnt == 0 || --cnt > 0);

  if (res == 2)
	printf(timeoutBye);
  return (res != 0);	 
}
