/*
 * globalsymbols.cpp
 *
 *  Created on: Feb 21, 2012
 *      Author: cedric
 */

#include "globalsymbols.h"
#include "functions.h"
#include "ginacutils.h"
#include "files.h"
#include "yamlutils.h"
#include "kinematics.h"
#include "fermat.h"
#include <string>

using std::string;

namespace Reduze {

// register type
namespace {
YAMLProxy<GlobalSymbols> dummy1;
YAMLProxy<Paths> dummy2;
YAMLProxy<GlobalOptions> dummy3;
}

// global symbols

namespace {
  GiNaC::realsymbol undefined("undefined");
}

GlobalSymbols::GlobalSymbols() : //
		Na_(GiNaC::realsymbol("Na")), /**/
		Nc_(GiNaC::realsymbol("Nc")), /**/
		d_(GiNaC::realsymbol("d")) {
	set_all();
}

/// adjoint dimension of su(N)
const GiNaC::realsymbol& GlobalSymbols::Na() const {
	return Na_;
}
/// fundamental dimension of su(N)
const GiNaC::realsymbol& GlobalSymbols::Nc() const {
	return Nc_;
}
/// coupling constants
const GiNaC::lst& GlobalSymbols::coupling_constants() const {
	return coupling_constants_;
}
/// space time dimension
const GiNaC::realsymbol& GlobalSymbols::d() const {
	return d_;
}

const GiNaC::lst& GlobalSymbols::all() const {
	return all_;
}

void GlobalSymbols::read(const YAML::Node& node) {
	verify_yaml_spec(node);
	if (node.FindValue("coupling_constants")) {
		const YAML::Node& c_node = node["coupling_constants"];
		VERIFY(c_node.Type() == YAML::NodeType::Sequence);
		std::list<std::string> symbs;
		for (YAML::Iterator s = c_node.begin(); s != c_node.end(); ++s) {
			std::string str;
			*s >> str;
			symbs.push_back(str);
		}
		coupling_constants_ = create_real_symbols(symbs);
	}
	if (node.FindValue("su_n_dimensions")) {
		const YAML::Node& su_dim_node = node["su_n_dimensions"];
		std::string na, nc;
		su_dim_node["adjoint"] >> na;
		su_dim_node["fundamental"] >> nc;
		Na_ = create_real_symbol(na);
		Nc_ = create_real_symbol(nc);
	}
	if (node.FindValue("space_time_dimension")) {
		std::string dim;
		node["space_time_dimension"] >> dim;
		d_ = create_real_symbol(dim);
	}
	set_all();
	set_external_symbols();
}

void GlobalSymbols::print(YAML::Emitter& ye) const {
	using namespace YAML;
	ye << BeginMap;
	ye << Key << "coupling_constants" << Value << Flow << coupling_constants_;
	ye << Key << "su_n_dimensions" << Value << BeginMap;
	ye << Key << "adjoint" << Value << Na_;
	ye << Key << "fundamental" << Value << Nc_ << EndMap;
	ye << Key << "space_time_dimension" << Value << d_;
	ye << EndMap;
}

void GlobalSymbols::set_external_symbols() {
	try {
		std::string file_fr = Files::instance()->get_filename_feynmanrules();
		if (is_readable_file(file_fr)) {
			std::ifstream fin(file_fr.c_str());
			YAML::Parser parser(fin);
			YAML::Node doc;
			parser.GetNextDocument(doc);
			try {
				const YAML::Node& node = doc["feynmanrules"];
				if (node.FindValue("coupling_constants")) {
					LOGXX(
							"Replacing default coupling constants in global.yaml by values from the Feynman rules file.");
					const YAML::Node& c_node = node["coupling_constants"];
					VERIFY(c_node.Type() == YAML::NodeType::Sequence);
					coupling_constants_.remove_all();
					for (YAML::Iterator s = c_node.begin(); s != c_node.end();
							++s) {
						std::string str;
						*s >> str;
						coupling_constants_.append(create_real_symbol(str));
					}
				}
				if (node.FindValue("su_n_dimensions")) {
					LOGXX(
							"Replacing default su_n_dimensions in global.yaml by values from the Feynman rules file.");
					const YAML::Node& su_dim_node = node["su_n_dimensions"];
					std::string str;
					su_dim_node["adjoint"] >> str;
					Na_ = create_real_symbol(str);
					su_dim_node["fundamental"] >> str;
					Nc_ = create_real_symbol(str);
				}
			} catch (std::exception& e) {
				throw std::runtime_error(
						std::string("Failed reading Feynman rules file:\n")
								+ e.what());
			}
		}

		if (d_.is_equal(undefined)) {
			std::string file_kin = Files::instance()->get_filename_kinematics();
			if (is_readable_file(file_kin)) {
				std::ifstream fin(file_kin.c_str());
				YAML::Parser parser(fin);
				YAML::Node doc;
				parser.GetNextDocument(doc);
				try {
					const YAML::Node& node = doc["kinematics"];
					verify_is_valid_map_or_non_empty_sequence(node);
					const YAML::Node* kin_node = &node;
					if (node.Type() == YAML::NodeType::Sequence)
						kin_node = &(*node.begin());
					if (kin_node->FindValue("dimension")) {
						LOG("Using space time dimension symbol from the kinematics file");
						std::string str;
						(*kin_node)["dimension"] >> str;
						d_ = create_real_symbol(str);
					}
				} catch (std::exception& e) {
					throw std::runtime_error(
							std::string("Failed reading kinematics file:\n")
									+ e.what());
				}
			}
		}
		if (d_.is_equal(undefined))
			ERROR("Missing space time dimension (global_options, kinematics)");
	} catch (std::exception& e) {
		ERROR(std::string("Failed initializing global symbols:\n") + e.what());
	}
	set_all();
}

void GlobalSymbols::set_all() {
	all_ = coupling_constants_;
	all_.append(Na_);
	all_.append(Nc_);
	all_.append(d_);
}

// Paths

void Paths::read(const YAML::Node& node) {
	verify_yaml_spec(node);
	if (node.FindValue("fermat"))
		node["fermat"] >> fermat_;
	string msg;
	if (fermat_ != "" && !Fermat::is_proper_executable_path(fermat_, msg))
		throw std::runtime_error(msg.c_str());
}

void Paths::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;
	os << Key << "fermat" << Value << fermat_;
	os << EndMap;
}

const std::string& Paths::fermat() const {
	return fermat_;
}

/// GlobalOptions

}
