/*  job_computediagraminterferences.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_COMPUTEDIAGRAMINTERFERENCES_H_
#define JOB_COMPUTEDIAGRAMINTERFERENCES_H_

#include "job.h"

#include <ginac/ginac.h>

namespace Reduze {

/// Job to select diagrams and to make jobs for multiplying them
class ComputeDiagramInterferences: public Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("compute_diagram_interferences");
		s.set_short_description("Computes interferences of diagrams.");
		s.set_long_description(""//
					"Job to compute interferences of a sum of diagrams with."
					" the complex conjugated of another sum of diagrams.");
		s.add_option("diagrams_file1", true, "string",
				"File name for first factor in interference.");
		s.add_option("diagrams_file2", true, "string",
				"File name for second factor in interference, to be complex conjugated.");
		s.add_option("position1", false, "integer", ""//
					"If non-positive, all diagrams in 'diagrams_file1' are"
					" considered. Otherwise the diagrams at position 'position1'"
					" is selected.");
		s.add_option("position2", false, "integer",
				"Similar to 'position1' but for 'diagrams_file2'.");
		s.add_option("output_file", true, "string",
				"Name of output file for computed linear combinations.");
		s.add_option("result_name", false, "string", "Name for the result.");
		s.add_option("polarization_orthogonal_vectors", false,
				"sequence of 2-element sequences of strings",
				"Four-vectors orthogonal to momentum of respective external "
					" field to be used for polarization sums of vector-bosons."
					" First element in each 2-element sequence is the leg index,"
					" second element the four-vector orthogonal to the field's"
					" momentum. Example: [ [1, p2], [2, p1] ] . If this option"
					" is not set, Feynman-t'Hooft polarization sums are used.");
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}
	ComputeDiagramInterferences() :
		position_1_(-1), position_2_(-1) {
		add_auto_options();
	}
	virtual ~ComputeDiagramInterferences() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;
protected:
	virtual void add_auto_options() {
		add_auto_io("diagrams_file1", diagrams_file_1_);
		add_auto_io("diagrams_file2", diagrams_file_2_);
		add_auto_io("position1", position_1_);
		add_auto_io("position2", position_2_);
		add_auto_io("output_file", output_file_);
		add_auto_io("result_name", result_name_);
	}
	virtual void print_manual_options(YAML::Emitter& os) const;
	virtual void read_manual_options(const YAML::Node&);
private:
	std::string diagrams_file_1_, diagrams_file_2_, output_file_;
	int position_1_, position_2_;
	std::string result_name_;
	std::map<int, GiNaC::ex> polarization_orthogonal_vectors_;
};

}

#endif /* JOB_COMPUTEDIAGRAMINTERFERENCES_H_ */
