/*  sector.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef SECTOR_H_
#define SECTOR_H_

#include <ostream>
#include <set>
#include <list>
#include <vector>
#include "yamlconfigurable.h"

namespace YAML {
class Node;
class Emitter;
}

namespace GiNaC {
template <template <class T, class = std::allocator<T> > class> class container;
typedef container<std::list> lst;
class ex;
}

class Propagator;

namespace Reduze {

class IntegralFamily;
class SectorMappings;
class SectorGraph;
class CanonicalLabel;
class Permutation;

/// classification of integrals whose integrands share the same denominator factors
class Sector: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("sector");
		s.set_short_description("A sector of a integral family.");
		s.set_long_description(""//
					"A sector of an integral family with n propagators {p_1, ..., p_n}"
					" is a selection of t propagators {s_1, ..., s_t} of that family."
					" A sector is represented as a 2-element sequence consisting of the name"
					" of the integral family and a identification number id."
					" The id is determined by the sum over 2^(i-1) where i is the i-th propagator"
					" of the integral family that appears in the sector."
					" For integral families with one or more loop momenta"
					" the integrals of the sector with id = 0 are all assumed to be zero."
					" See the help for \"sector_selection\" for further information."
					" A strict ordering is defined for sectors by comparing the"
					" integral families, the number of propagators and the id-numbers."
					" The ordering of the integral families is as they are declared in"
					" the file 'integralfamilies.yaml'");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	/// creates a sector from a YAML description
	/** supports the following formats:
	 ** [A, 182]
	 ** (not yet supported:
	 **  [A, [1,0,1,0,1,0,0,0]]  would be user friendly alternative) */
	Sector(const YAML::Node& node);
	/// creates a sector in an integral family with id
	Sector(const IntegralFamily* ic, int id);
	/// creates the sector from a product integral family
	Sector(const Sector& s1, const Sector& s2);
	virtual ~Sector() {
	}

	/// returns the integral family of the sector
	inline const IntegralFamily* integralfamily() const {
		return integralfamily_;
	}

	/// returns the identification number of the sector
	inline int id() const {
		return id_;
	}
	/// returns the number of different propagators
	inline int t() const {
		return t_of_id(id_);
	}

	/// returns uncrossed sector if crossed, same sector otherwise
	Sector get_uncrossed() const;

	/// returns minimal sector which is equivalent by permutation symmetries
	Sector get_equivalent() const;

	/// returns the propagators occurring in this Sector
	/** bits will be set in cutmask for cut propagators with positions
	 ** relative to the returned vector */
	std::vector<Propagator> get_propagators(int& cutmask) const;
	/// returns the propagators of this sector by the position (starting with 0) in the integral family
	/** the 'offset' is added to each key value in the map **/
	std::map<int, Propagator> get_propagators_by_pos(int& cutmask, int offset = 0) const;

	/// returns true if the sector is obviously zero, false otherwise
	/** includes test whether sector is "cut to zero" **/
	bool is_obviously_zero() const;
	/// returns true if the number of loop momenta is zero (tree level)
	bool is_obviously_non_zero() const;

	/// number of integrals of class (r,s)
	long int get_num_integrals(int r, int s) const;
	/// number of integrals of classes given by the (r,s)-ranges
	long int get_num_integrals(int r_min, int r_max, int s_min, int s_max) const;

	/// returns a string "intcat_t_id_", safe to use as filename
	std::string get_safe_string_for_filename(bool fill_digits = false) const;
	bool operator<(const Sector& other) const;
	bool operator>(const Sector& other) const {
		return other < *this;
	}
	bool operator!=(const Sector& other) const {
		return *this < other || other < *this;
	}
	bool operator==(const Sector& other) const {
		return !(*this < other || other < *this);
	}

	/// returns the U polynomial of the corner integral of this sector
	GiNaC::ex find_U_polynomial(const GiNaC::lst& xi) const;

	/// returns the U and F polynomials of the corner integral of this sector
	void find_UF_polynomials(const GiNaC::lst& xi, GiNaC::ex& u, GiNaC::ex& f) const;

	// graph construction

	/// construct all possible graphs representing the sector
	/** if no graph has been found then the bool is set to false if the sector
	 ** is trivially zero, true otherwise.
	 ** The constructed graphs are connected and self-loops contain additional
	 ** degree-2-nodes. For further processing use the cleanup functions below. **/
	void construct_graphs(std::list<SectorGraph>& graphs,
			bool& det_non_zero) const;

	/// construct all graphs representing the sector from the graphs of a subsector
	/** The input subgraphs and the constructed graphs must be and are connected with
	 ** self-loops containing additional degree-2-nodes.
	 ** For further processing use the cleanup functions below. **/
	void construct_graphs(std::list<SectorGraph>& graphs, int subsector,
			const std::list<SectorGraph>& subgraphs) const;

	/// clean up constructed graphs and return first (arbitrary) representant
	/** removes degree-2-nodes and disconnects vacuum components **/
	static SectorGraph cleanup_and_select_first_graph(const std::list<SectorGraph>& graphlist);

	/// clean up constructed graphs and return minimal representant (wrt. canonical labeling)
	/** removes degree-2-nodes and disconnects vacuum components **/
	static SectorGraph cleanup_and_select_minimal_graph(const std::list<SectorGraph>& graphlist);

	/// searches a graph for the sector, returns true on success
	/** If false is returned no graph has been found and det_non_zero is set
	 ** to true if the graph is not trivially zero, false otherwise.
	 ** If "select_minimal_graphs" is set to true then from all possible
	 ** graphs per sector the smallest w.r.t. lex.ordering is chosen
	 ** to represent the sector, otherwise an arbitrary one **/
	bool find_graph(SectorGraph& graph, bool& det_non_zero,
			bool select_minimal_graph) const;

	// input/output

	virtual void print(YAML::Emitter& os) const;
	virtual void print_mma(std::ostream&) const;
	/// reads in a sector from a YAML node
	/** supports the same formats as Sector(const YAML::Node&) and additionally:
	 ** 182 (just the id, current integral family will be used) */
	virtual void read(const YAML::Node&);
	friend std::ostream& operator<<(std::ostream&, const Sector&);
	friend class YAMLProxy<Sector> ;

	// static members

	/// get the number of integrals for n propagators, t different propagators and given r and s
	static long int get_num_integrals_static(int n, int t, int r, int s);
	/// the number of different propagators for given sector id
	static int t_of_id(int sector_id);

private:
	Sector();
	const IntegralFamily* integralfamily_;
	int id_;
	int t_;

	/// if set to true the sectors with smaller ID sort before those with greater ID
	static const bool smaller_sector_id_is_less_ = true;
};

inline void operator>>(const YAML::Node& n, Sector& s) {
	s.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const Sector& s) {
	s.print(ye);
	return ye;
}

/// selection of Sector instances
/** This selection corresponds closely to the way ReduceSectors will set
 ** up a recursive reduction, see the tutorial for more details. **/
class SectorSelection: public YAMLConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("sector_selection");
		s.set_short_description("A selection of sectors.");
		s.set_long_description(""//
					" A selection of sectors."
					" Any sector in the sector selection is automatically"
					" replaced by its simplest equivalent one as defined by the"
					" sector relations. A sector in one of the lists doesn't"
					" need the keyword \"sector\" but can be given directly as"
					" the sequence of the integral family name and the sector"
					" id. See the help for \"sector\" for further information.");
		s.add_option("select_all", false, "boolean", ""//
					"Whether to include all sectors in the selection."
				    " This will recusively select the top level sectors of"
				    " all uncrossed families.");
		s.add_option("select", false, "sequence", ""//
					"List of sectors to be included in the selection.");
		s.add_option("select_recursively", false, "sequence", ""//
					"List of sectors including recursively all"
					" sub-sectors to be included in the selection.");
		/*s.add_option("select_dependents", false, "sequence", "" //
				    "Select all sectors which depend on any sector in the"
				    " given list.");*/
		s.add_option("deselect", false, "sequence", ""//
					"List of sectors to be excluded from the selection.");
		s.add_option("deselect_recursively", false, "sequence", ""//
					"List of sectors including recursively all"
					" sub-sectors to be excluded from the selection.");
		s.add_option("deselect_independents", false, "sequence", ""//
				    "Deselect sectors which are independent of all sectors"
				    " in this list. The empty list deselects nothing.");
		s.add_option("deselect_graphless", false, "boolean", ""//
				    "Deselect sectors which don't have a graph.");
		s.add_option("t_restriction", false, "2-element sequence", ""//
					"Given the list [t_min, t_max] only sectors"
					" with t >= t_min and t <= t_max are included in the selection."
					" A negative value t_min < 0 (t_max < 0) means no"
					" restriction on t_min (t_max).");
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	SectorSelection() :
		select_all_(false), deselect_graphless_(false), t_min_(-1), t_max_(-1) {
	}
	virtual ~SectorSelection() {
	}

	virtual void print(YAML::Emitter& os) const;
	virtual void read(const YAML::Node&);

	/// adds a sector
	/** note: already deselected sectors will override selection given here */
	void add(const Sector&);

	/// adds a sector and prerequisite sectors (subsectors)
	/** note: already deselected sectors will override selection given here */
	void add_recursively(const Sector&);

	/// creates a SectorSelection which includes all given sectors incl. subsectors
	static SectorSelection find_compact_recursive_selection(const std::set<
			Sector>& sectors, bool use_uncrossed = false);

	/// finds all sectors which should be reduced prior to given minimal sector
	/** returns minimal subsectors and uncrossed versions of minimal sectors,
	 ** which are treated similar to subsectors **/
	static std::set<Sector> find_prerequisite_sectors(const Sector&,
			bool recurse);

	/// finds sectors whose reduction depend at least on one of given sectors
	static std::set<Sector> find_dependent_sectors(const std::list<Sector>&);

	/// returns all selected sectors as a set
	std::set<Sector> find_sectors() const;

public:
	bool select_all_, deselect_graphless_;
	std::list<Sector> select_, select_recursively_/*, select_dependents_*/;
	std::list<Sector> deselect_, deselect_recursively_, deselect_independents_;
	int t_min_, t_max_;
};

inline void operator>>(const YAML::Node& n, SectorSelection& s) {
	s.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const SectorSelection& s) {
	s.print(ye);
	return ye;
}

/// helper class to control parsing of uncrossed sectors
class AllowReadingCrossedSectors {
public:
	static bool value() {
		return value_;
	}
private:
	static bool value_;
	// only friend allowed to change the sector read option
	friend class UncrossedReadSectorSelection;
};

/// sector selection for uncrossed sectors
class UncrossedReadSectorSelection: public SectorSelection {
public:
	virtual void read(const YAML::Node& node);
};

// functions

/** returns a set #loopmomenta independent propagators together with
 *  the determinant of the jacobian matrix.
 *  If the determinant is zero then no independent propagators have been found
 *  Propagators without a momentum (bilinear props) are omitted
 */
std::pair<GiNaC::ex, std::set<int> > independent_propagators(
		const std::vector<Propagator>& props, const GiNaC::lst& loopmoms);

std::pair<GiNaC::ex, std::set<int> > independent_propagators(
		const std::map<int, Propagator>& props, const GiNaC::lst& loopmoms);

}

#endif /* SECTOR_H_ */
