#include "bool.h"
#ifdef HAVE_SYSLOG
# include <syslog.h>
#endif
#include <limits.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <sys/param.h>
#include <string.h>
#include "rets.h"
#include "network.h"
#include "util.h"
#include "rcfile.h"
#include "main.h"

int main(int argc, char *argv[]);

static void helpinfo(char *progname);
static void verinfo(void);
static void setnulls(void);
static void setfree(void);
static bool set_string_defaults(void);
static void set_nonstring_defaults(void);
static bool setargs(int argc, char *argv[]);
static bool remove_pidfile(void);

/* extern'ed in main.h */
struct settings set;
char *error_messages;

int main(int argc, char *argv[]) 
{
	char orig_work_dir[MAXPATHLEN];
	MYLOG_OPTIONS = MYLOG_STDERR;
	umask(022); /* -rw-r--r- */

	if (!(error_messages = xmalloc(MAX_LOG_LENGTH)))
		return EXIT_FAILURE;
	if (!(getcwd(orig_work_dir, MAXPATHLEN))) {
		notice_err("getting working directory failed");
		return EXIT_FAILURE;
	}
	setnulls();
	while (		!setargs(argc, argv) &&
#ifndef DEBUG_GDB
			!become_daemon() &&
#endif
			server()!=QSHUTDOWN) {
		notice("restarting\n");
		chdir(orig_work_dir);
		remove_pidfile();
		optind = 1; /* allow arguments to be re-evaluated */
	}
	notice("exiting\n");
	remove_pidfile();
	setfree();
	free(error_messages);
	return EXIT_SUCCESS;
}

bool remove_pidfile(void)
{
#ifndef DEBUG_GDB
	if (set.pidfile && -1==remove(set.pidfile)) {
		notice_err("removing pidfile '%s' failed", set.pidfile);
		return TRUE;
	}
#endif
	return FALSE;
}

void setnulls(void)
{
	set.authfile = NULL;
	set.onconnect = NULL;
	set.ondisconnect = NULL;
	set.pidfile = NULL;
	set.commandon = NULL;
	set.commandoff = NULL;
	set.command_logfile = NULL;
	set.logfile = NULL;
	set.external = NULL;
	set.bind_ips = NULL;
	set.hmdata = NULL;
	return;
}

void setfree(void)
{
	free(set.authfile);
	free(set.onconnect);
	free(set.ondisconnect);
	free(set.pidfile);
	free(set.commandon);
	free(set.commandoff);
	free(set.command_logfile);
	free(set.logfile);
	free(set.external);
	free(set.hmdata);
	if (set.bind_ips)
		free(set.bind_ips[0]);
	free(set.bind_ips);
	setnulls();
	return;
}

void set_nonstring_defaults(void)
{
	setfree();
	/* defaults */
	set.debug = FALSE;
	set.keep_alive_timeout = 60;
	set.port = 5540;
	set.maxcons = -1; /* unlimited */
	set.max_from_one = -1; /* unlimited */
	set.rcfile_must_exist = FALSE;
	set.disconnect_wait = -1;
	set.max_redials = -1; /* unlimited */
	set.staydropped = FALSE;
	set.waitpipe = TRUE;
	set.onconnect_wait = FALSE;
	return;
}

/* must be done *after* user settings
 * this is so if a user specifies -a ./debug.authfile (for example) we won't
 * complain when we can't see /etc/dwunauth
 *
 * These rely on the user never being able to set them to NULL.
 * (if we add a no_pid option for example, we're stuffed here)
 */
bool set_string_defaults(void)
{
	char *bindarg;
	if (!set.authfile) {
		if (realfile(&set.authfile, "/etc/dwunauth", 0))
			return TRUE;
	}
	if (!set.pidfile) {
		if (realfile(&set.pidfile, "/var/run/dwun.pid",
					MY_CREATE | MY_ALLREAD))
			return TRUE;
	}
#ifndef HAVE_SYSLOG
	if (!set.logfile) {
		if (realfile(&set.logfile, "/var/log/dwun.log", MY_CREATE))
			return TRUE;
	}
#endif
	if (!set.bind_ips) {
		if (alloc_copy(&bindarg, "127.0.0.1"))
			return TRUE;
		if (set_bind_ips(bindarg)) {
			free(bindarg);
			return TRUE;
		}
	}
	return FALSE;
}

bool setargs(int argc, char *argv[])
{
	int ret;
	char *bindarg;
	char *rcfile = NULL;
	char **pt;

	/* which calls setfree, which calls setnulls */
	set_nonstring_defaults();

	/* options from rcfile */
	for (pt=argv; *pt; ++pt) {
		if (**pt=='-' && *(*pt+1)=='r') {
			if (alloc_copy(&rcfile, *(pt+1)))
				return TRUE;
			set.rcfile_must_exist = TRUE;
			break;
		}
	}
	if (!rcfile) {
		if (alloc_copy(&rcfile, "/etc/dwunrc"))
			return TRUE;
	}
	if (get_rcfile_settings(rcfile))
		return TRUE;
	free(rcfile); rcfile = NULL;

	/* command-line settings override defaults and those from rcfile */
	while ((ret = getopt(argc, argv, "p:b:m:a:k:r:l:c:dvh"))!=EOF) {
		switch(ret) {
		case 'p': /* port to bind to */
			set.port = (short)strtoul(optarg,(char **)NULL, 10);
			if (set.port > USHRT_MAX || 0==set.port) {
				notice("Invalid port\n");
				return TRUE;
			}
			break;
	    	case 'b':
			if (set.bind_ips)
				free(set.bind_ips[0]);
			free(set.bind_ips);
			if (alloc_copy(&bindarg, optarg))
				return TRUE;
			if (set_bind_ips(bindarg)) {
				free(bindarg);
				return TRUE;
			}
			break;
    		case 'm': /* max clients allowed */
			if (mystrtoi(optarg, &set.maxcons)) {
				notice_err("invalid -m argument");
				return TRUE;
			}
			break;
    		case 'a':
			free(set.authfile);
			if (realfile(&set.authfile, optarg, 0))
				return TRUE;
			break;
		case 'k':
			if (mystrtol(optarg, &set.keep_alive_timeout)) {
				notice_err("invalid -k argument");
				return TRUE;
			}
			break;
		case 'r': /* already dealt with */
			break;
		case 'l': 
			free(set.logfile);
		 	if (realfile(&set.logfile, optarg, MY_CREATE))
				return TRUE;
			break;
		case 'c': 
			notice("warning: '-c' option is deprecated, use "
					"'commandon' in rcfile\n");
			free(set.commandon);
	 		if (alloc_copy(&set.commandon, optarg))
 				return TRUE;
			break;
		case 'd':
		     	set.debug = TRUE;
	 		break;
		case 'v': 
			verinfo();
			exit(EXIT_SUCCESS);
		case 'h': /* help */
		     	helpinfo(argv[0]);
	 		exit(EXIT_SUCCESS);
		case ':':
		     	notice("option requires an argument\n");
	 		helpinfo(argv[0]);
 			return TRUE;
		case '?':
	 		helpinfo(argv[0]);
 			return TRUE;
		default: 
			notice_debug("bug in setargs\n");
		       	break;
		}
	}
	if (set_string_defaults())
		return TRUE;
	return FALSE;
}

void verinfo(void)
{
	notice("DWUN version %s\n", VERSION);
	notice("Copyright (C) 2001 Tim Sutherland\n");
	notice("See the file COPYING in the distribution\n");
	return;
}

void helpinfo(char *progname)
{
	verinfo(); putchar('\n');
	notice("Usage: %s [OPTION] ...\n\n", progname);
	notice("-p\tport\n");
	notice("-b\tIPs to bind to\n");
	notice("-m\tmaximum clients allowed\n");
	notice("-a\tauthentication file\n");
	notice("-k\tseconds to wait for client keepalive\n");
	notice("-r\tread options from file\n");
	notice("-l\tlog to file\n");
	notice("-d\tlog debugging information\n");
	notice("-v\tversion\n");
	notice("-h\tthis help\n");
	return;
}
