/*
 *  Copyright (C) 2005-2018 Team Kodi
 *  This file is part of Kodi - https://kodi.tv
 *
 *  SPDX-License-Identifier: GPL-2.0-or-later
 *  See LICENSES/README.md for more information.
 */

#pragma once

#include "IStorageProvider.h"
#include "MediaSource.h" // for std::vector<CMediaSource>
#include "jobs/IJobCallback.h"
#include "storage/discs/IDiscDriveHandler.h"
#include "threads/CriticalSection.h"
#include "utils/DiscsUtils.h"

#include <map>
#include <memory>
#include <vector>

#include "PlatformDefs.h"

class CFileItem;

class CNetworkLocation
{
public:
  CNetworkLocation() { id = 0; }
  int id;
  std::string path;
};

class CMediaManager : public IStorageEventsCallback, public IJobCallback
{
public:
  enum class HasBlurayPlaylist : uint8_t
  {
    YES,
    NO,
    UNKNOWN
  };

  CMediaManager();

  void Initialize();
  void Stop();

  void LoadSources();
  bool SaveSources();

  void GetLocalDrives(std::vector<CMediaSource>& localDrives, bool includeQ = true);
  void GetRemovableDrives(std::vector<CMediaSource>& removableDrives);
  void GetNetworkLocations(std::vector<CMediaSource>& locations, bool autolocations = true);

  bool AddNetworkLocation(const std::string &path);
  bool HasLocation(const std::string& path) const;
  bool RemoveLocation(const std::string& path);
  bool SetLocationPath(const std::string& oldPath, const std::string& newPath);

  void AddAutoSource(const CMediaSource &share, bool bAutorun=false);
  void RemoveAutoSource(const CMediaSource &share);
  bool IsDiscInDrive(const std::string& devicePath="");
  bool IsAudio(const std::string& devicePath="");
  bool HasOpticalDrive();
  std::string TranslateDevicePath(const std::string& devicePath, bool bReturnAsDevice=false);
  DriveState GetDriveStatus(const std::string& devicePath = "");
#ifdef HAS_OPTICAL_DRIVE
  MEDIA_DETECT::CCdInfo* GetCdInfo(const std::string& devicePath="");
  bool RemoveCdInfo(const std::string& devicePath="");
  std::string GetDiskLabel(const std::string& devicePath="");
  std::string GetDiskUniqueId(const std::string& devicePath="");
  bool HasMediaBlurayPlaylist(const std::string& devicePath = "");

  /*! \brief Reset flag for removable bluray playlist status
   * This is needed as HasMediaBlurayPlaylist() is called every screen refresh when
   * the disc node is highlighted.
   * It needs to be reset whenever a disc is ejected or played (as a playlist may have been selected).
  */
  void ResetBlurayPlaylistStatus();

  /*! \brief Gets the platform disc drive handler
  * @todo this likely doesn't belong here but in some discsupport component owned by media manager
  * let's keep it here for now
  * \return The platform disc drive handler
  */
  std::shared_ptr<IDiscDriveHandler> GetDiscDriveHandler();
#endif
  std::string GetDiscPath();
  void SetHasOpticalDrive(bool bstatus);

  bool Eject(const std::string& mountpath);
  void EjectTray( const bool bEject=true, const char cDriveLetter='\0' );
  void CloseTray(const char cDriveLetter='\0');
  void ToggleTray(const char cDriveLetter='\0');

  void ProcessEvents();

  std::vector<std::string> GetDiskUsage();

  /*! \brief Callback executed when a new storage device is added
    * \sa IStorageEventsCallback
    * @param device the storage device
  */
  void OnStorageAdded(const MEDIA_DETECT::STORAGE::StorageDevice& device) override;

  /*! \brief Callback executed when a new storage device is safely removed
    * \sa IStorageEventsCallback
    * @param device the storage device
  */
  void OnStorageSafelyRemoved(const MEDIA_DETECT::STORAGE::StorageDevice& device) override;

  /*! \brief Callback executed when a new storage device is unsafely removed
    * \sa IStorageEventsCallback
    * @param device the storage device
  */
  void OnStorageUnsafelyRemoved(const MEDIA_DETECT::STORAGE::StorageDevice& device) override;

  void OnJobComplete(unsigned int jobID, bool success, CJob *job) override { }

  bool playStubFile(const CFileItem& item);

  UTILS::DISCS::DiscInfo GetDiscInfo(const std::string& mediaPath);

protected:
  std::vector<CNetworkLocation> m_locations;

  CCriticalSection m_muAutoSource, m_CritSecStorageProvider;
#ifdef HAS_OPTICAL_DRIVE
  std::map<std::string,MEDIA_DETECT::CCdInfo*> m_mapCdInfo;
#endif
  bool m_bhasoptical;
  std::string m_strFirstAvailDrive;

private:
  /*! \brief Loads the addon sources for the different supported browsable addon types
   */
  void LoadAddonSources() const;

  /*! \brief Get the addons root source for the given content type
   \param type the type of addon content desired
   \return the given CMediaSource for the addon root directory
   */
  CMediaSource GetRootAddonTypeSource(const std::string& type) const;

  /*! \brief Generate the addons source for the given content type
   \param type the type of addon content desired
   \param label the name of the addons source
   \param thumb image to use as the icon
   \return the given CMediaSource for the addon root directory
   */
  CMediaSource ComputeRootAddonTypeSource(const std::string& type,
                                          const std::string& label,
                                          const std::string& thumb) const;

  std::unique_ptr<IStorageProvider> m_platformStorage;
#ifdef HAS_OPTICAL_DRIVE
  std::shared_ptr<IDiscDriveHandler> m_platformDiscDriveHander;
#endif

  void RemoveDiscInfo(const std::string& devicePath);
  std::map<std::string, UTILS::DISCS::DiscInfo> m_mapDiscInfo;
#ifdef HAVE_LIBBLURAY
  HasBlurayPlaylist m_hasBlurayPlaylist{HasBlurayPlaylist::UNKNOWN};
#endif
};
