# This file is part of Cantera. See License.txt in the top-level directory or
# at https://cantera.org/license.txt for license and copyright information.

from collections.abc import Iterable, Sequence
from pathlib import Path
from typing import Any, Literal

from pandas import DataFrame

from ._onedim import (
    AxisymmetricFlow,
    Domain1D,
    FreeFlow,
    Inlet1D,
    Outlet1D,
    ReactingSurface1D,
    Sim1D,
    Surface1D,
    SymmetryPlane1D,
    UnstrainedFlow,
)
from ._types import (
    Array,
    ArrayLike,
    Basis,
    CompositionLike,
    LogLevel,
    RefineCriteria,
)
from .composite import Solution, SolutionArray
from .kinetics import Kinetics

class FlameBase(Sim1D):
    gas: Solution
    def __init__(
        self, domains: Iterable[Domain1D], gas: Solution, grid: ArrayLike | None = None
    ) -> None: ...
    def set_refine_criteria(  # type: ignore[override]
        self,
        ratio: float = 10.0,
        slope: float = 0.8,
        curve: float = 0.8,
        prune: float = 0.0,
    ) -> None: ...
    def get_refine_criteria(self) -> RefineCriteria: ...  # type: ignore[override]
    def set_initial_guess(
        self,
        *args: Any,
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
        **kwargs: Any,
    ) -> None: ...
    def set_profile(  # type: ignore[override]
        self, component: str | int, positions: Sequence[float], values: Sequence[float]
    ) -> None: ...
    @property
    def max_grid_points(self) -> int: ...
    @max_grid_points.setter
    def max_grid_points(self, npmax: int) -> None: ...
    @property
    def transport_model(self) -> str: ...
    @transport_model.setter
    def transport_model(self, model: str) -> None: ...
    @property
    def energy_enabled(self) -> bool: ...
    @energy_enabled.setter
    def energy_enabled(self, enable: bool) -> None: ...
    @property
    def soret_enabled(self) -> bool: ...
    @soret_enabled.setter
    def soret_enabled(self, enable: bool) -> None: ...
    @property
    def flux_gradient_basis(self) -> Basis: ...
    @flux_gradient_basis.setter
    def flux_gradient_basis(self, basis: Basis) -> None: ...
    @property
    def radiation_enabled(self) -> bool: ...
    @radiation_enabled.setter
    def radiation_enabled(self, enable: bool) -> None: ...
    @property
    def boundary_emissivities(self) -> tuple[float, float]: ...
    @boundary_emissivities.setter
    def boundary_emissivities(self, epsilon: Sequence[float]) -> None: ...
    @property
    def grid(self) -> Array: ...
    @property
    def P(self) -> float: ...
    @P.setter
    def P(self, P: float) -> None: ...
    @property
    def T(self) -> Array: ...
    @property
    def velocity(self) -> Array: ...
    @property
    def spread_rate(self) -> Array: ...
    @property
    def L(self) -> Array: ...
    @property
    def E(self) -> Array: ...
    @property
    def Uo(self) -> Array: ...
    @property
    def left_control_point_temperature(self) -> float: ...
    @left_control_point_temperature.setter
    def left_control_point_temperature(self, T: float) -> None: ...
    @property
    def left_control_point_coordinate(self) -> float: ...
    @property
    def right_control_point_temperature(self) -> float: ...
    @right_control_point_temperature.setter
    def right_control_point_temperature(self, T: float) -> None: ...
    @property
    def right_control_point_coordinate(self) -> float: ...
    def elemental_mass_fraction(self, m: str) -> Array: ...
    def elemental_mole_fraction(self, m: str) -> Array: ...
    def set_gas_state(self, point: int) -> None: ...
    def to_array(
        self, domain: Domain1D | str | int | None = None, normalize: bool = False
    ) -> SolutionArray[Solution]: ...
    def from_array(
        self, arr: SolutionArray[Solution], domain: Domain1D | str | int | None = None
    ) -> None: ...
    def to_pandas(
        self, species: Literal["X", "Y"] = "X", normalize: bool = True
    ) -> Array: ...
    @property
    def electric_field_enabled(self) -> bool: ...
    @electric_field_enabled.setter
    def electric_field_enabled(self, enable: bool) -> None: ...
    @property
    def two_point_control_enabled(self) -> bool: ...
    @two_point_control_enabled.setter
    def two_point_control_enabled(self, enable: bool) -> None: ...

    # Dynamically-added properties
    @property
    def density(self) -> Array: ...
    @property
    def density_mass(self) -> Array: ...
    @property
    def density_mole(self) -> Array: ...
    @property
    def volume_mass(self) -> Array: ...
    @property
    def volume_mole(self) -> Array: ...
    @property
    def int_energy_mole(self) -> Array: ...
    @property
    def int_energy_mass(self) -> Array: ...
    @property
    def h(self) -> Array: ...
    @property
    def enthalpy_mole(self) -> Array: ...
    @property
    def enthalpy_mass(self) -> Array: ...
    @property
    def s(self) -> Array: ...
    @property
    def entropy_mole(self) -> Array: ...
    @property
    def entropy_mass(self) -> Array: ...
    @property
    def g(self) -> Array: ...
    @property
    def gibbs_mole(self) -> Array: ...
    @property
    def gibbs_mass(self) -> Array: ...
    @property
    def cv(self) -> Array: ...
    @property
    def cv_mole(self) -> Array: ...
    @property
    def cv_mass(self) -> Array: ...
    @property
    def cp(self) -> Array: ...
    @property
    def cp_mole(self) -> Array: ...
    @property
    def cp_mass(self) -> Array: ...
    @property
    def isothermal_compressibility(self) -> Array: ...
    @property
    def thermal_expansion_coeff(self) -> Array: ...
    @property
    def sound_speed(self) -> Array: ...
    @property
    def viscosity(self) -> Array: ...
    @property
    def thermal_conductivity(self) -> Array: ...
    @property
    def heat_release_rate(self) -> Array: ...
    @property
    def mean_molecular_weight(self) -> Array: ...
    @property
    def volume(self) -> Array: ...
    @property
    def int_energy(self) -> Array: ...
    @property
    def X(self) -> Array: ...
    @property
    def Y(self) -> Array: ...
    @property
    def concentrations(self) -> Array: ...
    @property
    def partial_molar_enthalpies(self) -> Array: ...
    @property
    def partial_molar_entropies(self) -> Array: ...
    @property
    def partial_molar_int_energies(self) -> Array: ...
    @property
    def chemical_potentials(self) -> Array: ...
    @property
    def electrochemical_potentials(self) -> Array: ...
    @property
    def partial_molar_cp(self) -> Array: ...
    @property
    def partial_molar_volumes(self) -> Array: ...
    @property
    def standard_enthalpies_RT(self) -> Array: ...
    @property
    def standard_entropies_R(self) -> Array: ...
    @property
    def standard_int_energies_RT(self) -> Array: ...
    @property
    def standard_gibbs_RT(self) -> Array: ...
    @property
    def standard_cp_R(self) -> Array: ...
    @property
    def creation_rates(self) -> Array: ...
    @property
    def destruction_rates(self) -> Array: ...
    @property
    def net_production_rates(self) -> Array: ...
    @property
    def creation_rates_ddC(self) -> Array: ...
    @property
    def creation_rates_ddP(self) -> Array: ...
    @property
    def creation_rates_ddT(self) -> Array: ...
    @property
    def destruction_rates_ddC(self) -> Array: ...
    @property
    def destruction_rates_ddP(self) -> Array: ...
    @property
    def destruction_rates_ddT(self) -> Array: ...
    @property
    def net_production_rates_ddC(self) -> Array: ...
    @property
    def net_production_rates_ddP(self) -> Array: ...
    @property
    def net_production_rates_ddT(self) -> Array: ...
    @property
    def mix_diff_coeffs(self) -> Array: ...
    @property
    def mix_diff_coeffs_mass(self) -> Array: ...
    @property
    def mix_diff_coeffs_mole(self) -> Array: ...
    @property
    def thermal_diff_coeffs(self) -> Array: ...
    @property
    def activities(self) -> Array: ...
    @property
    def activity_coefficients(self) -> Array: ...
    @property
    def mobilities(self) -> Array: ...
    @property
    def species_viscosities(self) -> Array: ...
    @property
    def forward_rates_of_progress(self) -> Array: ...
    @property
    def reverse_rates_of_progress(self) -> Array: ...
    @property
    def net_rates_of_progress(self) -> Array: ...
    @property
    def equilibrium_constants(self) -> Array: ...
    @property
    def forward_rate_constants(self) -> Array: ...
    @property
    def reverse_rate_constants(self) -> Array: ...
    @property
    def delta_enthalpy(self) -> Array: ...
    @property
    def delta_gibbs(self) -> Array: ...
    @property
    def delta_entropy(self) -> Array: ...
    @property
    def delta_standard_enthalpy(self) -> Array: ...
    @property
    def delta_standard_gibbs(self) -> Array: ...
    @property
    def delta_standard_entropy(self) -> Array: ...
    @property
    def heat_production_rates(self) -> Array: ...
    @property
    def third_body_concentrations(self) -> Array: ...
    @property
    def forward_rate_constants_ddC(self) -> Array: ...
    @property
    def forward_rate_constants_ddP(self) -> Array: ...
    @property
    def forward_rate_constants_ddT(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddC(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddP(self) -> Array: ...
    @property
    def forward_rates_of_progress_ddT(self) -> Array: ...
    @property
    def net_rates_of_progress_ddC(self) -> Array: ...
    @property
    def net_rates_of_progress_ddP(self) -> Array: ...
    @property
    def net_rates_of_progress_ddT(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddC(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddP(self) -> Array: ...
    @property
    def reverse_rates_of_progress_ddT(self) -> Array: ...

class FreeFlame(FlameBase):
    inlet: Inlet1D
    outlet: Outlet1D
    flame: FreeFlow
    def __init__(
        self, gas: Solution, grid: ArrayLike | None = None, width: float | None = None
    ) -> None: ...
    def set_initial_guess(  # type: ignore[override]
        self,
        locs: ArrayLike = [0.0, 0.3, 0.5, 1.0],
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
    ) -> None: ...
    def solve(
        self,
        loglevel: LogLevel = 1,
        refine_grid: bool = True,
        auto: bool = False,
        stage: Literal[1, 2] | None = None,
    ) -> None: ...
    def get_flame_speed_reaction_sensitivities(self) -> Array: ...

class BurnerFlame(FlameBase):
    burner: Inlet1D
    outlet: Outlet1D
    flame: UnstrainedFlow
    def __init__(
        self, gas: Solution, grid: ArrayLike | None = None, width: float | None = None
    ) -> None: ...
    def set_initial_guess(  # type: ignore[override]
        self,
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
    ) -> None: ...
    def solve(
        self,
        loglevel: LogLevel = 1,
        refine_grid: bool = True,
        auto: bool = False,
        stage: Literal[1, 2] | None = None,
    ) -> None: ...

class CounterflowDiffusionFlame(FlameBase):
    fuel_inlet: Inlet1D
    oxidizer_inlet: Inlet1D
    flame: AxisymmetricFlow
    def __init__(
        self, gas: Solution, grid: ArrayLike | None = None, width: float | None = None
    ) -> None: ...
    def set_initial_guess(  # type: ignore[override]
        self,
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
    ) -> None: ...
    def extinct(self) -> bool: ...
    def solve(
        self,
        loglevel: LogLevel = 1,
        refine_grid: bool = True,
        auto: bool = False,
        stage: Literal[1, 2] | None = None,
    ) -> None: ...
    def strain_rate(
        self,
        definition: Literal[
            "mean",
            "max",
            "stoichiometric",
            "potential_flow_fuel",
            "potential_flow_oxidizer",
        ],
        fuel: CompositionLike | None = None,
        oxidizer: CompositionLike = "O2",
        stoich: float | None = None,
    ) -> float: ...
    def mixture_fraction(self, m: str | int) -> Array: ...
    @property
    def equivalence_ratio(self) -> Array: ...

class ImpingingJet(FlameBase):
    inlet: Inlet1D
    flame: AxisymmetricFlow
    surface: Surface1D | ReactingSurface1D
    def __init__(
        self,
        gas: Solution,
        grid: ArrayLike | None = None,
        width: float | None = None,
        surface: Kinetics | None = None,
    ) -> None: ...
    def set_initial_guess(  # type: ignore[override]
        self,
        products: Literal["inlet", "equil"] = "inlet",
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
    ) -> None: ...

class CounterflowPremixedFlame(FlameBase):
    reactants: Inlet1D
    products: Inlet1D
    flame: AxisymmetricFlow
    def __init__(
        self, gas: Solution, grid: ArrayLike | None = None, width: float | None = None
    ) -> None: ...
    def set_initial_guess(  # type: ignore[override]
        self,
        equilibrate: bool = True,
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
    ) -> None: ...

class CounterflowTwinPremixedFlame(FlameBase):
    reactants: Inlet1D
    flame: AxisymmetricFlow
    products: SymmetryPlane1D
    def __init__(
        self, gas: Solution, grid: ArrayLike | None = None, width: float | None = None
    ) -> None: ...
    def set_initial_guess(  # type: ignore[override]
        self,
        data: SolutionArray[Solution] | DataFrame | str | Path | None = None,
        group: str | None = None,
    ) -> None: ...
