// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0 OR ISC

#include <gtest/gtest.h>
#include <openssl/crypto.h>
#include <openssl/curve25519.h>
#include <openssl/dh.h>
#include <openssl/ec_key.h>
#include <openssl/evp.h>
#include <openssl/nid.h>
#include <openssl/rsa.h>

#include "internal.h"

extern "C" {
  void AWS_LC_fips_failure_callback(const char* message);
}

int callback_call_count = 0;

void AWS_LC_fips_failure_callback(const char* message) {
  SCOPED_TRACE(callback_call_count);
  SCOPED_TRACE(message);
  const std::map<std::string, std::vector<std::string>> kat_failure_messages = {
    {"RSA_PWCT", {"RSA keygen checks failed"}},
    {"ECDSA_PWCT", {"EC keygen checks failed"}},
    {"EDDSA_PWCT", {"Ed25519 keygen PCT failed"}},
    {"MLKEM_PWCT", {"ML-KEM keygen PCT failed", "ML-KEM self tests failed", "ML-KEM keygen PCT failed"}},
    {"MLDSA_PWCT", {"ML-DSA keygen PCT failed", "ML-DSA self tests failed", "ML-DSA keygen PCT failed"}},
    {"HMAC-SHA-256", {"HMAC-SHA-256 KAT failed.\nExpected:   365f5bd5f5ebfdc76e53a5736d732013aad3bc864bb884941646889c48eea90e\nCalculated: 853c7403937d8b6239569b184eb7993fc5f751aefcea28f2c863858e2d29c50b\n", "Integrity test failed"}},
    {"AES-CBC-encrypt", {"AES-CBC-encrypt KAT failed.\nExpected:   5646c141f413d6ff6292417a26c686bd305fb657a7d2503ac55e8e9340f210d8\nCalculated: a2be9b9cf41b6e1ddb4d65278d5dd28c02e449fca4bdff92f1a9a9cec178954c\n", "Power on self test failed"}},
    {"AES-CBC-decrypt", {"AES-CBC-decrypt KAT failed.\nExpected:   51a7a01f6b796ccd4803a141dc56a6c216b5d1d3b706b2256fa6d0d20e6f19b5\nCalculated: 85d7b98cd1599f7340ec7a00db67519185d7b98cd1599f7340ec7a00db675191\n", "Power on self test failed"}},
    {"AES-GCM-encrypt", {"AES-GCM-encrypt KAT failed.\nExpected:   877bd58d963e4be66494402f619b7e56527da45af9a6e2db1c632e97930ffbedb59e1c20b2b058da48072dbd960d34c6\nCalculated: 08b7951416b03e2ccb61c2a6e9d3d6dbd2f6f1f1b73592a261f7257f5f128369f9ecdc5bad903c25ee946273656b472f\n", "Power on self test failed"}},
    {"AES-GCM-decrypt", {"AES-GCM-decrypt KAT failed because EVP_AEAD_CTX_open failed", "Power on self test failed"}},
    {"DRBG", {"DRBG Generate KAT failed.\nExpected:   191f2b497685fd51b656bc1c7dd5dd4476a35e179b8eb8986512ca356ca06fa022e4f6d843ed4e2d9739433b57fc233f710ae0edfed5b8677a0039b26ea92597\nCalculated: 4af020970d7d770f44491cd477258a5a8f93a6496a5b98f342d1dcf5ebc374d9d21264e4627aa36678405a542bcf318c4d931da7a9012da5759a8ec58064144b\n", "CTR-DRBG failed", "Power on self test failed"}},
    {"DRBG-reseed", {"DRBG-reseed KAT failed.\nExpected:   00f205aafd116c77bc818699ca51cf80159f029e0bcd26c84b878a151addf2f3eb940b08c8c957a40b4b0f13de7c0c6aac344a9af2d083020517c9818f2a8192\nCalculated: 533a0ea1f2577e4d2b4ddcf8d2d9a73132ea6e5a4595c1b923d610798707afde1efd7e0cfed5147fe0d4ce418b61c794dfbd125dd38d3c81684114655d927cd0\n", "CTR-DRBG failed", "Power on self test failed"}},
    {"SHA-1", {"SHA-1 KAT failed.\nExpected:   941955930a582938ebf509116d1afd0f1e11e3cb\nCalculated: e129f27c5103bc5cc44bcdf0a15e160d445066ff\n", "Power on self test failed"}},
    {"SHA-256", {"SHA-256 KAT failed.\nExpected:   7fe4d5f1a1e38287d958f511c71d5e275eccd266cfb9c8c660d8921e57fd4675\nCalculated: 374708fff7719dd5979ec875d56cd2286f6d3cf7ec317a3b25632aab28ec37bb\n", "Integrity test failed"}},
    {"SHA-512", {"SHA-512 KAT failed.\nExpected:   293c94354e9883e5c278367ae51890bf35410164198d26ebe1f82f048efa8b2bc6b29d5d46765ac8b525a3ea5284476d6df4c971f33d894c3b208c5b75e8f87c\nCalculated: 0b6cbac838dfe7f47ea1bd0df00ec282fdf45510c92161072ccfb84035390c4da743d9c3b954eaa1b0f86fc9861b23cc6c8667ab232c11c686432ebb5c8c3f27\n", "Power on self test failed"}},
    {"SHA3-256", {"SHA3-256 KAT failed.\nExpected:   b87d9e4722edd3918729ded9a6d03af8256998ee088a1ae662ef4bcaff142a96\nCalculated: 61664696888a110278ff672620c85217e69aa662a83304052f1014d395f545bf\n", "Power on self test failed"}},
    {"TLS-KDF", {"TLS-KDF KAT failed.\nExpected:   e21dd6c268c757032c2cebbbb8a97de9eee6c947830abd11605dd52c47b60588\nCalculated: f6bf8fed0639cca6712ccba58f955c225fcc666ae8f55b968fd022bddd77db63\n", "Power on self test failed"}},
    {"RSA-sign", {"RSA-sign KAT failed.\nExpected:   b7ee251166d4fd87108f7c859a1d35422b0230aa8a81f750b264c015e6bd5de1038bf4d35b4a8cf17f3fcd1fe090783adeefd269e7e559bac9de537484aa33062947d60f00cae2deab6c869db2c64fb68edebe1c26462fbb05c12b79b0824ec903b31a27d0c8522e220470501dcfd66d7aa088903a496465728851e6db8e8cbe1c50c71ac08b443d8446b0a27b37c803f0012e76c10b7b0a5e62d9934b86afa844203fcae76b8d2756f603c1d55eaa6434a4f6f20b86c1d4a4a96e588c0a09caad4bd66b3bfefefb093c76db332d622844c4d543e4862ea8a026a88ce9407cf8598407bd90e0c7cb768b410fbaedc9bf1477c5db16d1db33bc20d0ec04bff8e3\nCalculated: 24e2b547c5ce89bd2943f6cf0ae6956ac6e639ceea104b5aff8940e89b57c7f5fef175cb6db081898ee6be7e4588fd1a44af573467743617d8c64c9edb9ca1b6925bc0423535d40bd9f60d5978ef1814d9344a5e5eaf5351c1b4d88cf0ec452b7f30ee14f49b61986434249d0e8ae7d0f3d96f3f0c3103cd419966bdde766c8ff0ae674ea6f0d5e2b57dd08f42f0b735c37ec85583e46ac7fa1b9ef530aa4fd34d9122004b069b2dab228f4ee1fc7e722f552994f8922eab4ee5522638a506e2fefd7fe568dc058eb2b59937c8e40b2719a7291e3bb574d5ee1580662657331e0bc4e02371a362172d642ff5b0a641a386be74870dd98833f4c5ba489b6faef8\n", "RSA self tests failed"}},
    {"RSA-verify", {"RSA-verify KAT failed", "RSA self tests failed"}},
    {"ECDSA-sign", {"ECDSA-sign signature failed.\nExpected:   6780c5fc70275e2c7061a0e7877bb174deadeb9887027f3fa83654158ba7f50c3a82d965b2a72ac5a607bc56ab3722c842d921c04936e9e25fd7b600e7d8dc80\nCalculated: 6780c5fc70275e2c7061a0e7877bb174deadeb9887027f3fa83654158ba7f50c14672fa0338e4b0376d7255bf240b99a3c40f37dc1747346de9a6aaaedb3175b\n", "ECDSA-sign KAT failed", "ECC self tests failed"}},
    {"ECDSA-verify", {"ECDSA-verify KAT failed", "ECC self tests failed"}},
    {"Z-computation", {"Z-computation failed.\nExpected:   04f1630088c5d5e90552acb6ec6876b8737f0f7234e6bb30322237b62a80e89e6e6f3602e721d231db9463b7d8190ec2c0a72f15491aa27c418faf9c40af2e4a0c\nCalculated: 0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\n", "ECC self tests failed"}},
    {"FFDH", {"FFDH self tests failed", "Power on self test failed"}},
    {"ED25519-sign", {"ED25519-sign failed.\nExpected:   a881e8d95ddbd5d14760af4ecfce4596f72e04d7eeccb9c6a193e24dd735b13c18a534c7793145469dd16f0c5e0371a3fb85063597c0924597cb427560db2a0b\nCalculated: 8608f1c9cf5070fae1f6833c868886a1e997bd3d02d200c942286d831ed78e16ce580009d05bea51d78dd4f65fb0179373d3449c7088133fd0774854cf03bb00\n", "ED25519-sign failed", "EdDSA self tests failed"}},
    {"ED25519-verify", {"ED25519-verify failed", "EdDSA self tests failed"}},
    {"ED25519ph-sign", {"ED25519ph-sign failed.\nExpected:   0b933d3f5900e3a1e53947ce9732c7014037e9c94b71cd3afb6046aa29fea9bbd81c50541064c659d0075fb38c8b420f8148682dc9f8384355105c3970d20609\nCalculated: 55cf180696924ba9ac1275ea19da4d1584f69250c479145cdbd7068ffbfadf8d5aafb666893b365dcaf66ae20bc9e813df3b3f9d3197ead79d644fc5a17dff0d\n", "EdDSA-ph self tests failed"}},
    {"ED25519ph-verify", {"ED25519ph-verify failed", "EdDSA-ph self tests failed"}},
    {"ML-KEM-keyGen-decaps", {"ML-KEM-keyGen-decaps failed.\nExpected:   000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\nCalculated: 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\n", "ML-KEM self tests failed"}},
    {"ML-KEM-keyGen-encaps", {"ML-KEM-keyGen-encaps failed.\nExpected:   0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\nCalculated: a7cc68f8d02110ca5720223b9e2a8987c8a24835a20dabcbefa430e74a85af80b9b74b74574c5e5f585459ca3610940f0b57b33344ceacccc135557b82f4968688a0168c1aa2940e5604482bf8b900a4343096446330cfee10917c0338181b7fe8d30f4816087d6299f225417f533ee40473894847bd45291367be6b1a7dee55bb21d60e3828552f4c8a6f3c54fc74cf67a614eeab002a076851879cef2218fdb3766123c2de32a269b6aa4661b69370314c004446f7258c40b2ea789f40ca023bbb1217c2c44b380c6e3194edd129d039218d9b75194a386d944acce7a9720ab026362004e95bf9229290b53613416082e82ba8a42a5ff14759f57712395706b307f7635ecee42317a48eb3b90683f3ab53d17c50f53c6cfb1c0cf59d6f2a981021428a7ac5edf13b26844d83c31d3608710e623aabb24b6c5b48baebc1b3078972589201b7a30bc09315612b067655bec403a69c89eb137c31157971098cb693ba4ae9ae40a8031cec92580bcc1b5ab3ecd1aa5f79aa2cd69249d138c8965a81c87a07eb59a4612e60658f4df028cef8af1b837e0ab0bfedb726904290d0bc41df6a67f7a4166609952439631960648e229a21f2a4d82abad3ec8135dc9bb43c703d3b33e437c9ef7bca91c3465676740125a15ad1707088b101b4273d3c4bf30181b4b2575de75ccfc13312a2a6bcebc477a9668e751629b569bfa20beca09800992c63f04b6b4a7df977a00131c2f8722e5138775235b517a709852167c1d415fdc7ad32f2aaca437e9cc6b248d9ca7c65b405e68d24e81b8688caa22b3cf5c9b147f0cc27e667a80b83ccbb4f4161a5ffd0194b1b5720e68ea0f59997f26740972d2c8124d7a6ad8327c2075a3f76b968d2aaad19c00697599e0be49fa6d65b4088b0be692dcda028095852a0d7205e4417409c0317780305a878fb582963b6953f6b8f0880b050178301d659be3a4db7e0bf2587129164178f32707d40392d85713dea82913999aa6c35aa94b3547abe40e2b0ba82b5b78319182a5a47d173176ba6fa3a4d70a8130b310743fa8aaae314381c9b7f49991f19a4ff8369638de380b5d828b7b5fcfdd91fe68efe16d4cd9eed66d65c1d2d8caf5af4a692\n", "ML-KEM self tests failed"}},
    {"ML-KEM-encapsulate-ciphertext", {"ML-KEM-encapsulate-ciphertext failed.\nExpected:   000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000\nCalculated: 431a4f1b2d2c6c00f1690bbe482541ef3d563774daff83207f96de7e5e4a59d5d936d9443ad422e645793e7a60a9b0a76cd672d20c69b82a5563df52d96f9a6cdfc56fbd4fd8d5a8afeb2a09d92ec854094794b4ed2db381f04c68439608aa9902a4d1689e2eb1e5f07a4a1c709262d7c2ff2f81f6eeaab2a86a41ba210eb1bf8e75febccd1a15b4d7a7b60257c89d00bd81d39fcb8d1ce3278102595dd652f7fb7d5584874f3327b174043b350ebd4d41fe08bd0e854d41cbb027c481da64dc6151b88dececcf022ddac2e22736c147e0773294231c0589967154c526b0b7cdd59568eeff5749a40cb100c60c6480897655d96e9f64d61684c0b3150646732c19409fe565540a31894703cf0179cae85bc8c1a5732649836e48e676405b9591b65ba25f9b489b9e5772aa1ed5a00143cb9f5449fd013457a3c13874cb58c75b52c9b6a9ae495ccb504a89cb5f145695b921632fb85b0316b30d4ad17fef0862d6b1e6ca6a611c8a6a7234b4362c5ca0ad9f7697687798cf624dc9f35fbb376e09953156532a9033709df755b46cc6d83de3a111e19a76b361e0ef14c91db8d91c6c6d9e3e46f42291fd6cbf5cfd122716fb0675698e602ab39ee98e0d8145eebaaa9374f5b3bb0df4d0fd83a40e0d25038c39e9bee01cf79c86f3086158d031d5c5e86bc7e7eb16e622505f2888213884c0b5252289b11fce5bfeebfbef0a32ceaf9c14c6250090028463db6f8d19684f541108fe934d88e7ef5cce9daebb32700b9397691a684298c9bf1b7c22d1bcec3fcacfbb17f2ed2b98b85e6a8fe2482996b5e099e9d0211cb9412614de87dc18d23613ed7f6c29cc37b727116dd901c2817938c29fcd026089336addc09eca90de9a25a6374fee86bcdd06ae3daaf0b1bc5b3b2790d4d9f759bef8ac743612a2bbf6e45de8b22efa61226625d4c39f346b844c5ebec5355866c00b726cc1640cb237c34a20a7c603d251f46e6b3b0fa71b3276835e3e9da5b9485e789614af49f1e9504db2528631fbe1cd7dbee85164e4c099a27a4583e9247d078f8830b46874c1b010bf3cd90eb0774961f239ba\n", "ML-KEM self tests failed"}},
    {"ML-KEM-encapsulate-shared-secret", {"ML-KEM-encapsulate-shared-secret failed.\nExpected:   0000000000000000000000000000000000000000000000000000000000000000\nCalculated: a772df2de250ac7d896bbb820b57f2ae05f9a412ab55baa421d4af6dac62662a\n", "ML-KEM self tests failed"}},
    {"ML-DSA-sign", {"ML-DSA-sign failed.\nExpected:   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\nCalculated: 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\n", "ML-DSA self tests failed"}},
    {"ML-DSA-verify", {"ML-DSA-verify failed", "ML-DSA self tests failed"}},
    {"HKDF-SHA-256", {"HKDF-SHA-256 KAT failed.\nExpected:   000000000000000000000000000000000000000000000000000000000000000000000000000000000000\nCalculated: ca5e6410e7a52332fe0ab3601212a7d3dbdf55a162af42a5daf38b94f24523477e880dd711508684cc21\n", "Power on self test failed"}},
    {"KBKDF", {"KBKDF-CTR-HMAC-SHA-256 KAT failed.\nExpected:   10621342bfb0fd40046c0e29f2cfdbf0\nCalculated: 606060902f7c6632bcde3a67f5818c48\n", "Power on self test failed"}},
    {"PBKDF2", {"PBKDF2 KAT failed.\nExpected:   c6ac0779e4a117c922287f5e10e7ee6ba74d8b19519b4cc738\nCalculated: e442f1807d5fc9b466badcdfd3806fed7fa50da9a6f5729117\n", "Power on self test failed"}},
    {"SSKDF", {"SSKDF_digest KAT failed.\nExpected:   5a2e26644d16222cd636a1fdb57bfaa17f94449127612bcd7be1bb39cc18f32893d3c648c16372fb6e9c63de5433b1ccdeb51bb5f15368c8a849a1e5a4efc666fd33eeb9f6728b0479f76668cfafc13a91367074def2b50e9d9a918a120210824165d596ad4f94a3236ef7cf5843282a0a57a483819f63e0cfb2081daf9ccf35c66a03e7a02d3891f45022e1c89d888aa8087e08f45babbc52062b18e6fb70c12dcb29a194d23abc351cfb3cf4f161cc775a3e711bb1502d6901f6931407a9ae868476f998d1ca4cca296a9f14752d14f47427e666289f80892a3d14a84fe343fd78d0dadbde1819aca915f7c0c024376b40cb34bae2d26e9f4552b6b1a26fa5\nCalculated: 63d19beaa2b5a2685510e4a3bd3692cec7d370f7d8eea41239a2d89439dc9643767c37c3fd892434c4f567d3c6854b57e63222bc99df63d7d02e6527b90fe57baaecf7fcb4ba278be3b2ff6f22d8435ccbc1c4d92fd89b88915d0711ad5ca81151d7c62a3d12a8315c2b3cf6f1d7b8b367687eadcb87d7c7b4e79f629df1aa35e05a8450291fdcdb744c1013bc25883fb913942ab0b77ac3a9dcd068e46af564e7deb1d2abc60d45d7f16ff985c652bc7559b8277026d238460f6133dfe6ab5641dbbcb749481c94c6e910161e54bef623cbaec3f4e096ec66bb6f3702c86dad3bd57b25d5115a972aafe8ab15fd60a30d401a6fbd421bab89095f7be374a439\n", "Power on self test failed"}}
  };

  char* broken_kat = getenv("FIPS_CALLBACK_TEST_EXPECTED_FAILURE");
  SCOPED_TRACE(broken_kat);
  if (broken_kat != nullptr) {
      auto test_config = kat_failure_messages.find(broken_kat);
      if (test_config != kat_failure_messages.end()) {
        ASSERT_LT(callback_call_count, (int)test_config->second.size());
        std::string expected_string = test_config->second[callback_call_count];
        EXPECT_STREQ(expected_string.c_str(), message);
      } else {
        FAIL() << "Failed to find expected message for FIPS_CALLBACK_TEST_POWER_ON_TEST_FAILURE=" << broken_kat;
      }
  } else {
    FAIL() << "AWS_LC_fips_failure_callback called when no KAT was expected to be broken";
  }
  callback_call_count++;

}

TEST(FIPSCallback, PowerOnSelfTests) {
  char* broken_kat = getenv("FIPS_CALLBACK_TEST_EXPECTED_FAILURE");
  SCOPED_TRACE(broken_kat);

  // Some KATs are lazy and run on first use
  bssl::UniquePtr<RSA> rsa(RSA_new());
  EXPECT_TRUE(RSA_generate_key_fips(rsa.get(), 2048, nullptr));
  bssl::UniquePtr<EC_KEY> key(EC_KEY_new_by_curve_name(NID_X9_62_prime256v1));
  EXPECT_TRUE(EC_KEY_generate_key_fips(key.get()));

  bssl::UniquePtr<DH> dh(DH_new());
  EXPECT_TRUE(DH_generate_parameters_ex(dh.get(), 64, DH_GENERATOR_5, nullptr));
  EXPECT_TRUE(DH_generate_key(dh.get()));

  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_KEM, nullptr));
  EXPECT_TRUE(ctx);
  EXPECT_TRUE(EVP_PKEY_CTX_kem_set_params(ctx.get(), NID_MLKEM512));
  EXPECT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  EXPECT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  EVP_PKEY_free(raw);

  uint8_t public_key[ED25519_PUBLIC_KEY_LEN];
  uint8_t private_key[ED25519_PRIVATE_KEY_LEN];
  ED25519_keypair(public_key, private_key);

  uint8_t message[2];
  uint8_t context[2];
  uint8_t signature[ED25519_SIGNATURE_LEN];
  EXPECT_TRUE(ED25519ph_sign(signature, message, sizeof(message), private_key, context, sizeof(context)));

  EVP_PKEY *dsa_raw = NULL;
  bssl::UniquePtr<EVP_PKEY_CTX> dsa_ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));
  ASSERT_TRUE(dsa_ctx);
  ASSERT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(dsa_ctx.get(), NID_MLDSA44));
  ASSERT_TRUE(EVP_PKEY_keygen_init(dsa_ctx.get()));
  ASSERT_TRUE(EVP_PKEY_keygen(dsa_ctx.get(), &dsa_raw));
  EVP_PKEY_free(dsa_raw);

  if (broken_kat == nullptr) {
    EXPECT_EQ(0, callback_call_count);
  } else {
    EXPECT_NE(0, callback_call_count);

  }
}

TEST(FIPSCallback, PWCT) {
  char*broken_runtime_test = getenv("FIPS_CALLBACK_TEST_EXPECTED_FAILURE");
  bssl::UniquePtr<RSA> rsa(RSA_new());
  SCOPED_TRACE(broken_runtime_test);
  if (broken_runtime_test != nullptr && strcmp(broken_runtime_test, "RSA_PWCT" ) == 0) {
    EXPECT_FALSE(RSA_generate_key_fips(rsa.get(), 2048, nullptr));
  } else {
    EXPECT_TRUE(RSA_generate_key_fips(rsa.get(), 2048, nullptr));
  }

  bssl::UniquePtr<EC_KEY> key(EC_KEY_new_by_curve_name(NID_X9_62_prime256v1));
  if (broken_runtime_test != nullptr && strcmp(broken_runtime_test, "ECDSA_PWCT" ) == 0) {
    EXPECT_FALSE(EC_KEY_generate_key_fips(key.get()));
  } else {
    EXPECT_TRUE(EC_KEY_generate_key_fips(key.get()));
  }

  bssl::UniquePtr<EVP_PKEY_CTX> ed_ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_ED25519, nullptr));
  ASSERT_TRUE(ed_ctx);
  EXPECT_TRUE(EVP_PKEY_keygen_init(ed_ctx.get()));
  EVP_PKEY *ed_key = nullptr;
  EXPECT_TRUE(EVP_PKEY_keygen(ed_ctx.get(), &ed_key));
  if (broken_runtime_test != nullptr && strcmp(broken_runtime_test, "EDDSA_PWCT" ) == 0) {
    EXPECT_FALSE(EVP_PKEY_keygen(ed_ctx.get(), &ed_key));
  } else {
    EXPECT_TRUE(EVP_PKEY_keygen(ed_ctx.get(), &ed_key));
  }
  EVP_PKEY_free(ed_key);

  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_KEM, nullptr));
  ASSERT_TRUE(ctx);
  EXPECT_TRUE(EVP_PKEY_CTX_kem_set_params(ctx.get(), NID_MLKEM512));
  EXPECT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  EVP_PKEY *raw = nullptr;
  if (broken_runtime_test != nullptr && strcmp(broken_runtime_test, "MLKEM_PWCT" ) == 0) {
    EXPECT_FALSE(EVP_PKEY_keygen(ctx.get(), &raw));
  } else {
    EXPECT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  }
  EVP_PKEY_free(raw);

  EVP_PKEY *dsa_raw = NULL;
  bssl::UniquePtr<EVP_PKEY_CTX> dsa_ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_PQDSA, nullptr));

  ASSERT_TRUE(dsa_ctx);
  EXPECT_TRUE(EVP_PKEY_CTX_pqdsa_set_params(dsa_ctx.get(), NID_MLDSA44));
  EXPECT_TRUE(EVP_PKEY_keygen_init(dsa_ctx.get()));
  if (broken_runtime_test != nullptr && strcmp(broken_runtime_test, "MLDSA_PWCT" ) == 0) {
    EXPECT_FALSE(EVP_PKEY_keygen(dsa_ctx.get(), &dsa_raw));
  } else {
    EXPECT_TRUE(EVP_PKEY_keygen(dsa_ctx.get(), &dsa_raw));
  }
  EVP_PKEY_free(dsa_raw);
}
