/*
 * GQradio
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqradio.h"
#include "preset.h"

#include "io_radio.h"
#include "ui_fileops.h"


/* there is one global preset list... */
static GList *preset_list = NULL;


/*
 *-----------------------------------------------------------------------------
 * preset data container
 *-----------------------------------------------------------------------------
 */

/* presets are opaque (only referenced by number) */

typedef struct _PresetData PresetData;
struct _PresetData {
	gchar *path;
	guint32 freq;
	gchar *description;
};

static PresetData *preset_data_new(guint32 freq, const gchar *description)
{
	PresetData *pd;

	pd = g_new0(PresetData, 1);
	pd->path = radio_freq_to_path(freq, description);
	pd->freq = freq;
	pd->description = g_strdup(description);

	return pd;
}

#if 0
static PresetData *preset_data_new_by_path(const gchar *path)
{
	PresetData *pd;

	if (!path) return NULL;

	pd = g_new0(PresetData, 1);
	pd->path = g_strdup(path);
	pd->freq = radio_path_to_freq(path);
	pd->description = radio_path_to_title(path);

	return pd;
}

static void preset_data_set_freq(PresetData *pd, guint32 freq)
{
	gchar *description;

	if (!pd) return;

	description = radio_path_to_title(pd->path);
	g_free(pd->path);
	pd->path = radio_freq_to_path(freq, description);
	g_free(description);
	pd->freq = freq;
}

static void preset_data_set_description(PresetData *pd, const gchar *description)
{
	if (!pd) return;

	g_free(pd->description);
	pd->description = g_strdup(description);
	g_free(pd->path);
	pd->path = radio_freq_to_path(pd->freq, description);
}
#endif

static void preset_data_free(PresetData *pd)
{
	if (!pd) return;

	g_free(pd->path);
	g_free(pd->description);
	g_free(pd);
}

/*
 *-----------------------------------------------------------------------------
 * preset list
 *-----------------------------------------------------------------------------
 */

void preset_list_init(gint size)
{
	gint i;

	preset_list_clear();
	g_list_free(preset_list);
	preset_list = NULL;

	for (i = 0; i < size; i++)
		{
		preset_list = g_list_prepend(preset_list, NULL);
		}
}

void preset_list_clear(void)
{
	GList *work;

	work = preset_list;
	while (work)
		{
		PresetData *pd = work->data;

		preset_data_free(pd);
		work->data = NULL;

		work = work->next;
		}
}

gint preset_list_find_free_slot(void)
{
	GList *work;
	gint i = 0;

	work = preset_list;
	while (work)
		{
		if (!work->data) return i;
		work = work->next;
		i++;
		}

	return -1;
}

gint preset_set(gint index, guint32 freq, const gchar *description)
{
	GList *work;
	PresetData *pd;

	work = g_list_nth(preset_list, index);
	if (!work) return FALSE;

	pd = work->data;
	preset_data_free(pd);

	work->data = preset_data_new(freq, description);

	return TRUE;
}

gint preset_set_by_path(gint index, const gchar *path)
{
	gint ret;
	guint32 freq;
	gchar *description;

	freq = radio_path_to_freq(path);
	description = radio_path_to_title(path);

	ret = preset_set(index, freq, description);

	g_free(description);

	return ret;
}

gint preset_clear(gint index)
{
	GList *work;
	PresetData *pd;

	work = g_list_nth(preset_list, index);
	if (!work) return FALSE;

	pd = work->data;
	preset_data_free(pd);
	work->data = NULL;

	return TRUE;
}

static PresetData *preset_index(gint index)
{
	GList *work;

	work = g_list_nth(preset_list, index);

	if (!work) return NULL;
	return work->data;
}

guint32 preset_get_freq(gint index)
{
	PresetData *pd;

	pd = preset_index(index);
	if (pd) return pd->freq;

	return 0;
}

const gchar *preset_get_description(gint index)
{
	PresetData *pd;

	pd = preset_index(index);
	if (pd) return pd->description;

	return NULL;
}

gint preset_find_freq(guint32 freq)
{
	GList *work;
	gint i = 0;

	work = preset_list;
	while (work)
		{
		PresetData *pd = work->data;
		if (pd && pd->freq == freq) return i;
		work = work->next;
		i++;
		}

	return -1;
}

gint preset_is_freq(gint index, guint32 freq)
{
	return (preset_get_freq(index)  == freq);
}

gint preset_is_set(gint index)
{
	return (preset_index(index) != NULL);
}


/*
 *-----------------------------------------------------------------------------
 * preset save/load
 *-----------------------------------------------------------------------------
 */

gint preset_list_save(const gchar *path)
{
	gchar *pathl;
	int i, c;
	GList *work;
	FILE *f;

	if (!path) return FALSE;

	pathl = path_from_utf8(path);
	f = fopen (pathl, "w");
	g_free(pathl);
	if (!f)
		{
		/* file open failed */
		printf(_("failed to open %s\n"), path);
		return FALSE;
		}

	fprintf(f, "# GQmpeg song playlist\n");
	fprintf(f, "# GQradio preset list\n");
	fprintf(f, "# created by GQradio version %s\n\n", VERSION);

	c = 0;
	i = 0;
	work = preset_list;
	while (work)
		{
		PresetData *pd = work->data;
		if (pd)
			{
			fprintf(f, "#Preset_%03d\n%s\n", i, pd->path);
			c++;
			}
		work = work->next;
		i++;
		}

	fprintf(f, "\n# contains %d presets\n", c);
	fprintf(f, "# end of list #\n");
	fclose(f);

	return TRUE;
}

gint preset_list_load(const gchar *path)
{
	gchar *pathl;
	gchar s_buf[1024];
	FILE *f;
	gint i;

	pathl = path_from_utf8(path);
	f = fopen(pathl, "r");
	g_free(pathl);
	if (!f)
		{
		return FALSE;
		}

	preset_list_clear();

	i = -1;
	while (fgets(s_buf, sizeof(s_buf), f))
		{
		gchar *s_buf_ptr;
		s_buf_ptr = s_buf;
		while (*s_buf_ptr != '\n' && *s_buf_ptr != '\0') s_buf_ptr++;
		*s_buf_ptr = '\0';

		if (s_buf[0]=='\n') continue;
		if (s_buf[0]=='#')
			{
			if (strncmp(s_buf, "#Preset_", 8) == 0)
				{
				/* pushes the index of the next preset */
				i = atoi(s_buf + 8);
				}
			}
		else
			{
			if (radio_path_is_radio(s_buf))
				{
				if (i == -1) i = preset_list_find_free_slot();
				if (i >= 0) preset_set_by_path(i, s_buf);
				i = -1;
				}
			}
		}

	fclose(f);

	return TRUE;
}


