# $Id: Hexfile.pm,v 1.2 2005/03/05 18:15:22 kissg Exp $

package Pista::Object::Hexfile;

use strict;
use warnings;
use vars qw(@ISA);
@ISA = qw(Pista::Object);

use IO::File;
use Pista::Util qw(min max);
use Pista::Object;

use Dumpvalue;
my $dumper = new Dumpvalue('arrayDepth' => 16);

sub new {
	my $pkg = shift;
	my $self = { filename => shift, mode => shift, device => shift, };
	$self->{handle} = IO::File->new($self->{filename}, $self->{mode});
	if (!$self->{handle}) {
		print "550 Cannot open file '$self->{filename}': $!\n";
		return undef;
	}
	if (!$self->{device}) {
		print "440 No device selected\n";
		return undef;
	}
	bless $self, $pkg;
}

sub _first_defined {
	my ($start, $end, $listref) = @_;
#print "start=$start, end=$end\n";
	while ($start <= $end) {
		return $start if defined $$listref[$start];
		$start++;
	}
	return undef;
}

# Write an IHEX16 file
sub write {
	my ($self, $src) = @_;
	my $handle = $self->{handle};
	$self->{ulba} = -1;

	foreach (qw(prog cal userid devid conf eeprom)) {
		next unless exists $src->{$_};
		$self->_write($src->{$_}, $self->{device}->{$_}->{width});
	}

	delete $self->{ulba};
	print $handle ":00000001FF\r\n";
	return $src;
}

sub _write {
	my ($self, $section, $width) = @_;
	my $start = my $base = $section->{start};
	my $end = $section->{end};
	my $buffer = $section->{content};
	my $handle = $self->{handle};
	my $linelen = 16/$width;
	my $boundary = ($linelen-$base%$linelen)%$linelen;

#$dumper->dumpValue([$self, $section, $width, $boundary]);
	while ($start <= $end) {
		my $idx = _first_defined($start-$base, $#{$buffer}, $buffer);
		last unless defined $idx;
		$start = $idx + $base;
#print "start=$start base=$base idx=$idx\n";

		my $highadd = (($start*$width) >> 16) & 0xffff;
		if ($highadd != $self->{ulba}) {
			$self->{ulba} = $highadd;
			my $binline = pack('CnCn', 2,0,4,$self->{ulba});
			my $sum = unpack('%8C*', $binline);
			printf($handle ":%s%.2X\r\n",
				uc(unpack('H*', $binline)), (-$sum)&0xff);
		}

		my @items = ($buffer->[$idx]);
		for ($idx++; defined($buffer->[$idx]) and
				$idx%$linelen!=$boundary; $idx++) {
			push (@items, $buffer->[$idx]);
		}
		my $binline = pack($width == 2 ? 'CnCv*' : 'CnCC*',
					($#items+1)*$width,	  # contentlen
					($start*$width) & 0xffff, # offset
					0,			  # type
					@items);		  # content
		my $sum = unpack('%8C*', $binline);		  # chksum
		printf($handle ":%s%.2X\r\n", uc(unpack('H*',$binline)),
						(-$sum)&0xff);
		$start = $idx+$base;
	}
}

sub read {
	my $self = shift;
	my %arg = (@_);
#	my $noblank = $arg{noblank};
	my $range = $arg{range};
	undef %arg;

	my $ulba = 0;
	my $sba = 0;
	my $format = undef;
	my $handle = $self->{handle};
	my @buffer;
	my ($minaddress, $maxaddress);
	my $width = $self->{device}->{addressing} eq 'byte' ? 1 : 2;

	while (<$handle>) {
		return -1 unless m/^:(		# record mark
				([\dA-F]{2})	# length
				([\dA-F]{4})	# offset
				([\dA-F]{2})	# type
				([\dA-F]*)	# content
				[\dA-F]{2}	# checksum
				)\s*$/xi; 	# optional \r
		# verify checksum
		return -2 if unpack('%8C*', pack('H*',$1));

		my $len  = unpack('C',pack('H2',$2));
		my $drlo = unpack('n',pack('H4',$3));
		my $type = unpack('C',pack('H2',$4));
		my $data = $5;

#print "len=$len, drlo=$drlo, type=$type, format=$format\n";
		if ($len and length($data) == $len*4) {
			$format = 'inhx16' unless $format;
			return -3 unless $format eq 'inhx16';
		}
		elsif (length($data) != $len*2) {
			return -4;
		}
		else {
			no warnings qw(uninitialized);
			return -5 if $format eq 'inhx16';
		}
			
		if    ($type == 0x04) {
			$ulba = unpack('n',pack('H4',$data))<<16;
			$format = 'inhx32' unless $format;
			return -6 unless $format eq 'inhx32';
		}
		elsif ($type == 0x02) {
			$sba = unpack('n',pack('H4',$data)) << 4;
		}
		elsif ($type == 0x01) {
			last;
		}
		elsif ($type == 0x00) {
			my $start = $ulba+$sba+$drlo; # INCORRECT but usable
			my $template;
			if ($format ne 'inhx16') {
				$len /= $width;
				$start /= $width;
				$template = $width == 2 ? 'v*' : 'C*';
			}
			else {
				$template = 'n*';
			}

			my $address = $start;
			for my $val (unpack($template, pack('H*', $data))) {
				my $sect = $self->{device}->qualify($address);
				next unless $sect;
				my $section = Pista::Section->new($address,
							$address, [$val]);
				if (exists $self->{$sect}) {
					$self->{$sect} = $self->{$sect}->merge($section);
				}
				else {
					$self->{$sect} = $section;
				}
			}
			continue {
				$address++;
			}
		}
	}

	return $self unless $range;

	for (qw(prog cal userid devid conf eeprom)) {
		next unless exists $self->{$_};
		if (exists $range->{$_}) {
			$self->{$_}->intersect($range->{$_}->{start},
						$range->{$_}->{end});
		}
		else {
			delete $self->{$_};
		}
	}
#$dumper->dumpValue(['hexfile',$self]);
	return $self;
}

sub erase {
	my $self = shift;

	my $tmpbuf = Pista::Object::Buffer->new();
	$self->copy($tmpbuf);
	$tmpbuf->erase(@_);
	my $newfile = Pista::Object::Hexfile->new($self->{filename}, 'w',
				$self->{device}) or return;
	$tmpbuf->copy($newfile);
	print "150 File rewritten\n";
	return $newfile;
}

1;
