// Copyright 2018 The Cockroach Authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied. See the License for the specific language governing
// permissions and limitations under the License.

package storage

import (
	"context"
	"testing"

	"github.com/cockroachdb/cockroach/pkg/config"
	"github.com/cockroachdb/cockroach/pkg/gossip"
	"github.com/cockroachdb/cockroach/pkg/keys"
	"github.com/cockroachdb/cockroach/pkg/roachpb"
	"github.com/cockroachdb/cockroach/pkg/storage/engine/enginepb"
	"github.com/cockroachdb/cockroach/pkg/storage/storagebase"
	"github.com/cockroachdb/cockroach/pkg/util/hlc"
	"github.com/cockroachdb/cockroach/pkg/util/leaktest"
	"github.com/cockroachdb/cockroach/pkg/util/stop"
	"github.com/gogo/protobuf/proto"
)

func TestMergeQueueShouldQueue(t *testing.T) {
	defer leaktest.AfterTest(t)()

	ctx := context.Background()
	testCtx := testContext{}
	stopper := stop.NewStopper()
	defer stopper.Stop(ctx)
	testCtx.Start(t, stopper)

	mq := newMergeQueue(testCtx.store, testCtx.store.DB(), testCtx.gossip)
	storagebase.MergeQueueEnabled.Override(&testCtx.store.ClusterSettings().SV, true)

	tableKey := func(i uint32) []byte {
		return keys.MakeTablePrefix(keys.MaxReservedDescID + i)
	}

	config.TestingSetZoneConfig(keys.MaxReservedDescID+1, *config.NewZoneConfig())
	config.TestingSetZoneConfig(keys.MaxReservedDescID+2, *config.NewZoneConfig())

	// Disable merges for table ID 4.
	if err := testCtx.gossip.AddInfo(
		gossip.MakeTableDisableMergesKey(keys.MaxReservedDescID+4),
		nil /* value */, 0 /* ttl */); err != nil {
		t.Fatal(err)
	}

	type testCase struct {
		startKey, endKey []byte
		minBytes         int64
		bytes            int64
		expShouldQ       bool
		expPriority      float64
	}

	testCases := []testCase{
		// The last range of table 1 should not be mergeable because table 2 exists.
		{
			startKey: tableKey(1),
			endKey:   tableKey(2),
			minBytes: 1,
		},
		{
			startKey: append(tableKey(1), 'z'),
			endKey:   tableKey(2),
			minBytes: 1,
		},

		// Unlike the last range of table 1, the last range of table 2 is mergeable
		// because there is no table that follows. (In this test, the system only
		// knows about tables on which TestingSetZoneConfig has been called.)
		{
			startKey:    tableKey(2),
			endKey:      tableKey(3),
			minBytes:    1,
			expShouldQ:  true,
			expPriority: 1,
		},
		{
			startKey:    append(tableKey(2), 'z'),
			endKey:      tableKey(3),
			minBytes:    1,
			expShouldQ:  true,
			expPriority: 1,
		},

		// The last range is never mergeable.
		{
			startKey: tableKey(3),
			endKey:   roachpb.KeyMax,
			minBytes: 1,
		},
		{
			startKey: append(tableKey(3), 'z'),
			endKey:   roachpb.KeyMax,
			minBytes: 1,
		},

		// An interior range of a table is not mergeable if it meets or exceeds the
		// minimum byte threshold.
		{
			startKey:    tableKey(1),
			endKey:      append(tableKey(1), 'a'),
			minBytes:    1024,
			bytes:       1024,
			expShouldQ:  false,
			expPriority: 0,
		},
		{
			startKey:    tableKey(1),
			endKey:      append(tableKey(1), 'a'),
			minBytes:    1024,
			bytes:       1024,
			expShouldQ:  false,
			expPriority: 0,
		},
		// Edge case: a minimum byte threshold of zero. This effectively disables
		// the threshold, as an empty range is no longer considered mergeable.
		{
			startKey:    tableKey(1),
			endKey:      append(tableKey(1), 'a'),
			minBytes:    0,
			bytes:       0,
			expShouldQ:  false,
			expPriority: 0,
		},

		// An interior range of a table is mergeable if it does not meet the minimum
		// byte threshold. Its priority is inversely related to its size.
		{
			startKey:    tableKey(1),
			endKey:      append(tableKey(1), 'a'),
			minBytes:    1024,
			bytes:       0,
			expShouldQ:  true,
			expPriority: 1,
		},
		{
			startKey:    tableKey(1),
			endKey:      append(tableKey(1), 'a'),
			minBytes:    1024,
			bytes:       768,
			expShouldQ:  true,
			expPriority: 0.25,
		},

		// Merges disabled for a table via gossip. Merges are disabled on table
		// 4. We test it both for the start key and the end key.
		{
			startKey: tableKey(3),
			endKey:   tableKey(4),
			minBytes: 1,
		},
		{
			startKey: tableKey(4),
			endKey:   append(tableKey(4), 'z'),
			minBytes: 1,
		},
		{
			startKey: tableKey(4),
			endKey:   tableKey(5),
			minBytes: 1,
		},
	}

	for _, tc := range testCases {
		t.Run("", func(t *testing.T) {
			repl := &Replica{}
			repl.mu.state.Desc = &roachpb.RangeDescriptor{StartKey: tc.startKey, EndKey: tc.endKey}
			repl.mu.state.Stats = &enginepb.MVCCStats{KeyBytes: tc.bytes}
			zoneConfig := config.DefaultZoneConfig()
			zoneConfig.RangeMinBytes = proto.Int64(tc.minBytes)
			repl.SetZoneConfig(&zoneConfig)
			shouldQ, priority := mq.shouldQueue(ctx, hlc.Timestamp{}, repl, config.NewSystemConfig())
			if tc.expShouldQ != shouldQ {
				t.Errorf("incorrect shouldQ: expected %v but got %v", tc.expShouldQ, shouldQ)
			}
			if tc.expPriority != priority {
				t.Errorf("incorrect priority: expected %v but got %v", tc.expPriority, priority)
			}
		})
	}
}
