#include <metal_stdlib>

using namespace metal;

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ triu/tril ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
template <bool upper>
inline bool triul_mask(int row, int col, int k);
template <>
inline bool triul_mask<true>(int row, int col, int k) {
  return col - row >= k;
}
template <>
inline bool triul_mask<false>(int row, int col, int k) {
  return col - row <= k;
}

template <typename IndexType>
inline IndexType compute_offs(
    constant IndexType* strides,
    constant uint* sizes,
    uint3 pos,
    int ndim) {
  auto offs = pos.x * strides[0] + pos.y * strides[1];
  if (ndim < 4) {
    return ndim == 3 ? offs + pos.z * strides[2] : offs;
  }
  auto idx = pos.z;
  for (int i = 2; i < ndim; ++i) {
    offs += strides[i] * (idx % sizes[i]);
    idx /= sizes[i];
  }
  return offs;
}

template <typename T, typename IndexType, bool upper>
kernel void triul_inplace(
    device T* self,
    constant IndexType* strides,
    constant uint* sizes,
    constant int2& k_ndim,
    uint3 pos [[thread_position_in_grid]]) {
  if (triul_mask<upper>(pos.y, pos.x, k_ndim.x)) {
    return;
  }
  auto offs = compute_offs(strides, sizes, pos, k_ndim.y);
  self[offs] = 0;
}

template <typename T, typename IndexType, bool upper>
kernel void triul(
    device T* out,
    device T* inp,
    constant IndexType* out_strides,
    constant IndexType* inp_strides,
    constant uint* sizes,
    constant int2& k_ndim,
    uint3 pos [[thread_position_in_grid]]) {
  auto out_offs = compute_offs(out_strides, sizes, pos, k_ndim.y);
  if (!triul_mask<upper>(pos.y, pos.x, k_ndim.x)) {
    out[out_offs] = 0;
    return;
  }
  auto inp_offs = compute_offs(inp_strides, sizes, pos, k_ndim.y);
  out[out_offs] = inp[inp_offs];
}

#define INSTANTIATE_TRIUL_KERNELS(DTYPE, IDX_TYPE)                         \
  template [[host_name("triu_inplace_" #IDX_TYPE "_" #DTYPE)]] kernel void \
  triul_inplace<DTYPE, IDX_TYPE, true>(                                    \
      device DTYPE * self,                                                 \
      constant IDX_TYPE * strides,                                         \
      constant uint * sizes,                                               \
      constant int2 & k_ndim,                                              \
      uint3 pos [[thread_position_in_grid]]);                              \
  template [[host_name("tril_inplace_" #IDX_TYPE "_" #DTYPE)]] kernel void \
  triul_inplace<DTYPE, IDX_TYPE, false>(                                   \
      device DTYPE * self,                                                 \
      constant IDX_TYPE * strides,                                         \
      constant uint * sizes,                                               \
      constant int2 & k_ndim,                                              \
      uint3 pos [[thread_position_in_grid]]);                              \
  template [[host_name("triu_" #IDX_TYPE "_" #DTYPE)]] kernel void         \
  triul<DTYPE, IDX_TYPE, true>(                                            \
      device DTYPE * out,                                                  \
      device DTYPE * inp,                                                  \
      constant IDX_TYPE * out_strides,                                     \
      constant IDX_TYPE * inp_strides,                                     \
      constant uint * sizes,                                               \
      constant int2 & k_ndim,                                              \
      uint3 pos [[thread_position_in_grid]]);                              \
  template [[host_name("tril_" #IDX_TYPE "_" #DTYPE)]] kernel void         \
  triul<DTYPE, IDX_TYPE, false>(                                           \
      device DTYPE * out,                                                  \
      device DTYPE * inp,                                                  \
      constant IDX_TYPE * out_strides,                                     \
      constant IDX_TYPE * inp_strides,                                     \
      constant uint * sizes,                                               \
      constant int2 & k_ndim,                                              \
      uint3 pos [[thread_position_in_grid]])

INSTANTIATE_TRIUL_KERNELS(float, int);
INSTANTIATE_TRIUL_KERNELS(half, int);
INSTANTIATE_TRIUL_KERNELS(bfloat, int);

INSTANTIATE_TRIUL_KERNELS(float2, int);
INSTANTIATE_TRIUL_KERNELS(half2, int);

INSTANTIATE_TRIUL_KERNELS(long, int);
INSTANTIATE_TRIUL_KERNELS(int, int);
INSTANTIATE_TRIUL_KERNELS(short, int);
INSTANTIATE_TRIUL_KERNELS(char, int);
INSTANTIATE_TRIUL_KERNELS(uchar, int);
INSTANTIATE_TRIUL_KERNELS(bool, int);

// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ triangle ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// To find the max integer that does not exceed the root of an int64_t variable,
// we could use a loop to test one bit at a time, which takes up to 31
// iterations. This would give the accurate result, but is relatively slow and
// is an overkill for most cases where double's precision suffice.
//
// If we directly use sqrt to calculate the root, the conversion from int64_t
// to double would lose 11 bits precision.
//
// The following solution uses sqrt directly for most cases, and would only
// special handle it if there is indeed precision loss.
inline int64_t resolve_root_int(
    int64_t b,
    int64_t cX4,
    int64_t x,
    int32_t sign) {
  int64_t bXb_cX4 = b * b - cX4;
  // precision loss could occur here when casting int64_t (63 bits
  // precision) to float (23 bits precision)
  float sr = sqrt((float)bXb_cX4);
  int64_t res = floor((-b + sign * sr) / 2);

  // have to cast double to int64_t, otherwise it would only compare up to the
  // precision of a double variable, ignoring the precision loss
  if (bXb_cX4 != (int64_t)(sr * sr)) {
    // handle precision loss by using binary search
    int64_t llsr = floor(sr);
    // Use the following math to reduce search space.
    // Suppose z is the accurate result of sqrt(bXb_cX4) without precision loss
    // let d = abs(bXb_cX4 - llsr * llsr), then we have:
    // z = sqrt(bXb_cX4) <= sqrt(llsr * llsr + d) <= llsr + sqrt(d)
    // z = sqrt(bXb_cX4) >= sqrt(llsr * llsr - d) >= llsr - sqrt(d)
    // Hence, it is sufficient to search range [llsr - sqrt(d), llsr + sqrt(d)).
    // And the true value of row would also be with in range,
    //            [res - sqrt(d), res + sqrt(d) + 1)
    // as the denominator would only reduce the precision penalty.
    int64_t diff = ceil(sqrt(abs((float)(bXb_cX4 - llsr * llsr))));
    // l never exceeds (could equal to) the target row index
    auto l = res > diff ? res - diff : 0;
    // r is always larger than the target row index
    auto r = res + diff + 1;

    // binary search for the correct answer
    x <<= 1; // the loop always compares with 2x, so do it once here
    while (l + 1 < r) {
      auto m = (l + r) >> 1;
      // for tril:
      //    b = 2f - 1, sign = 1, hence (2f + m - 1) * m / 2
      // for triu:
      //    b = -2f - 1, sign = -1, hence (2f - m + 1) * m / 2
      if (sign * (b + m) * m > x) {
        r = m;
      } else {
        l = m;
      }
    }
    res = l;
  }

  return res;
}

// f: the number of elements in the first row of the trapezoid.
// x: the index of the target coordinates ordered by row and then column.
//
// View the tril as a top trapezoid stacked on a bottom rectangle. Assume x
// corresponds to the coordinate (row, col) in the trapezoid, where the row and
// the col both start from 0, then we have:
//
//                   (f + f + row - 1) * row / 2 <= x                       [1]
//                 (f + f + row) * (row + 1) / 2  > x                       [2]
//
// Therefore, row is the maximum integer satisfying the following inequality:
//
//                       (row + 2f - 1)row <= 2x
//                  row^2 + (2f-1)row - 2x <= 0.                            [3]
//
// Based on inequality [3], we have the following coefficients for formula of
// root:
//                               a = 1
//                               b = 2f - 1
//                               c = -2x
// There are two roots, and we should use the largest integer that does not
// exceed the root on the right. Intuitively, it is because:
//  i)  the valid solution range of row is between two roots, as it is <= 0;
//  ii) as we count in more rows, the total # of elements should always
//      increase, hence so does the left-hand side row^2 + (2f-1)row - 2x.
//      Therefore, the valid range of row lies in between the nadir point and
//      the larger root on the right.
// Full proof can be derived from inequality [2]. So, we calculate the result
// coordinate as:
//
//                   row = floor((-b + sqrt(b^2 - 4c)) / 2)
//                   col = x - (f + f + row - 1) * row / 2
inline void get_coordinate_in_tril_trapezoid(
    int64_t f,
    int64_t x,
    thread int64_t& row,
    thread int64_t& col) {
  f <<= 1; // all statements use 2f, so only calculate it once here.
  auto b = f - 1;
  auto cX4 = -(x << 3); // 4 * c = 4 * (-2x) = -8x;
  row = resolve_root_int(b, cX4, x, 1);
  col = x - ((f + row - 1) * row >> 1);
}

// f: the number of elements in the first row of the bottom trapezoid.
// x: the index of the target coordinates ordered by row and then column.
//
// View the triu as a top rectangle stacked on a bottom trapezoid, where the
// trapezoid is upside down. Assume x corresponds to the coordinate (row, col)
// in the bottom trapezoid, where the row and the col start from 0, then we
// have:
//
//                   (f + f - row + 1) * row / 2 <= x                       [1]
//                 (f + f - row) * (row + 1) / 2  > x                       [2]
//
// Therefore, row is the maximum integer satisfying the following inequality:
//
//                       (-row + 2f + 1)row <= 2x
//                   row^2 - (2f+1)row + 2x >= 0.                           [3]
//
// Based on inequality [3], we have the following coefficients for formula of
// root:
//                               a = 1
//                               b = -1 - 2f
//                               c = 2x
// There are two roots, and we should use the largest integer that does not
// exceed the root on the left. Intuitively, it is because:
//  i)  the valid solution range of row is outside of the two roots, as it is <
//      > 0;
//  ii) as we count in more rows, the total # of elements should always
//      increase, hence so does the left-hand side row^2 - (2f+1)row + 2x.
//      Therefore, the valid range of row lies to the left of the smaller root
//      on the left.
// Full proof can be derived from inequality [2]. So, we calculate the result
// coordinate as:
//
//                   row = floor((-b - sqrt(b^2 - 4c)) / 2)
//                   col = x - (f + f - row + 1) * row / 2
inline void get_coordinate_in_triu_trapezoid(
    int64_t f,
    int64_t x,
    thread int64_t& row,
    thread int64_t& col) {
  f <<= 1; // all statements use 2f, so only calculate it once here.
  auto b = -1 - f;
  auto cX4 = x << 3; // 4 * c = 4 * (2x) = 8x;
  row = resolve_root_int(b, cX4, x, -1);
  col = x - ((f - row + 1) * row >> 1) + row;
}

template <typename scalar_t>
kernel void tril_indices(
    device scalar_t* tensor,
    constant int64_t& row_offset,
    constant int64_t& m_first_row,
    constant int64_t& col,
    constant int64_t& trapezoid_size,
    constant int64_t& tril_size,
    uint linear_index [[thread_position_in_grid]]) {
  int64_t r, c;
  if (linear_index < trapezoid_size) {
    // the coordinate is within the top trapezoid
    get_coordinate_in_tril_trapezoid(m_first_row, linear_index, r, c);
  } else {
    // the coordinate falls in the bottom rectangle
    auto surplus = linear_index - trapezoid_size;
    // add the height of trapezoid: m_last_row (col) - m_first_row + 1
    r = surplus / col + col - m_first_row + 1;
    c = surplus % col;
  }
  r += row_offset;

  tensor[linear_index] = r;
  tensor[linear_index + tril_size] = c;
}

template <typename scalar_t>
kernel void triu_indices(
    device scalar_t* tensor,
    constant int64_t& col_offset,
    constant int64_t& m_first_row,
    constant int64_t& col,
    constant int64_t& rectangle_size,
    constant int64_t& triu_size,
    uint linear_index [[thread_position_in_grid]]) {
  int64_t r, c;
  if (linear_index < rectangle_size) {
    // the coordinate is within the top rectangle
    r = linear_index / col;
    c = linear_index % col;
  } else {
    // the coordinate falls in the bottom trapezoid
    get_coordinate_in_triu_trapezoid(
        m_first_row, linear_index - rectangle_size, r, c);
    r += rectangle_size / col;
  }

  c += col_offset;
  tensor[linear_index] = r;
  tensor[linear_index + triu_size] = c;
}

#define INSTANTIATE_TRI_INDICES(NAME, DTYPE)                   \
  template [[host_name(#NAME "_indices_" #DTYPE)]] kernel void \
      NAME##_indices<DTYPE>(                                   \
          device DTYPE * tensor,                               \
          constant int64_t& col_offset,                        \
          constant int64_t& m_first_row,                       \
          constant int64_t& col,                               \
          constant int64_t& rectangle_size,                    \
          constant int64_t& triu_size,                         \
          uint linear_index [[thread_position_in_grid]])

INSTANTIATE_TRI_INDICES(triu, long);
INSTANTIATE_TRI_INDICES(triu, int);
INSTANTIATE_TRI_INDICES(tril, long);
INSTANTIATE_TRI_INDICES(tril, int);
