/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.service;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.Assertions;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.cluster.AckedClusterStateTaskListener;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStatePublicationEvent;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.coordination.ClusterStatePublisher;
import org.elasticsearch.cluster.coordination.FailedToCommitClusterStateException;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.ProcessClusterEventTimeoutException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.service.ClusterStateUpdateStats;
import org.elasticsearch.cluster.service.PendingClusterTask;
import org.elasticsearch.cluster.service.SourcePrioritizedRunnable;
import org.elasticsearch.cluster.service.TaskBatcher;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.text.Text;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.FutureUtils;
import org.elasticsearch.common.util.concurrent.PrioritizedEsThreadPoolExecutor;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.node.Node;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;

public class MasterService
extends AbstractLifecycleComponent {
    private static final Logger logger = LogManager.getLogger(MasterService.class);
    public static final Setting<TimeValue> MASTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING = Setting.positiveTimeSetting("cluster.service.slow_master_task_logging_threshold", TimeValue.timeValueSeconds((long)10L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> MASTER_SERVICE_STARVATION_LOGGING_THRESHOLD_SETTING = Setting.positiveTimeSetting("cluster.service.master_service_starvation_logging_threshold", TimeValue.timeValueMinutes((long)5L), Setting.Property.NodeScope);
    static final String MASTER_UPDATE_THREAD_NAME = "masterService#updateTask";
    ClusterStatePublisher clusterStatePublisher;
    private final String nodeName;
    private Supplier<ClusterState> clusterStateSupplier;
    private volatile TimeValue slowTaskLoggingThreshold;
    private final TimeValue starvationLoggingThreshold;
    protected final ThreadPool threadPool;
    private volatile PrioritizedEsThreadPoolExecutor threadPoolExecutor;
    private volatile Batcher taskBatcher;
    private final ClusterStateUpdateStatsTracker clusterStateUpdateStatsTracker = new ClusterStateUpdateStatsTracker();

    public MasterService(Settings settings, ClusterSettings clusterSettings, ThreadPool threadPool) {
        this.nodeName = Objects.requireNonNull(Node.NODE_NAME_SETTING.get(settings));
        this.slowTaskLoggingThreshold = MASTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING.get(settings);
        clusterSettings.addSettingsUpdateConsumer(MASTER_SERVICE_SLOW_TASK_LOGGING_THRESHOLD_SETTING, this::setSlowTaskLoggingThreshold);
        this.starvationLoggingThreshold = MASTER_SERVICE_STARVATION_LOGGING_THRESHOLD_SETTING.get(settings);
        this.threadPool = threadPool;
    }

    private void setSlowTaskLoggingThreshold(TimeValue slowTaskLoggingThreshold) {
        this.slowTaskLoggingThreshold = slowTaskLoggingThreshold;
    }

    public synchronized void setClusterStatePublisher(ClusterStatePublisher publisher) {
        this.clusterStatePublisher = publisher;
    }

    public synchronized void setClusterStateSupplier(Supplier<ClusterState> clusterStateSupplier) {
        this.clusterStateSupplier = clusterStateSupplier;
    }

    @Override
    protected synchronized void doStart() {
        Objects.requireNonNull(this.clusterStatePublisher, "please set a cluster state publisher before starting");
        Objects.requireNonNull(this.clusterStateSupplier, "please set a cluster state supplier before starting");
        this.threadPoolExecutor = this.createThreadPoolExecutor();
        this.taskBatcher = new Batcher(logger, this.threadPoolExecutor);
    }

    protected PrioritizedEsThreadPoolExecutor createThreadPoolExecutor() {
        return EsExecutors.newSinglePrioritizing(this.nodeName + "/" + MASTER_UPDATE_THREAD_NAME, EsExecutors.daemonThreadFactory(this.nodeName, MASTER_UPDATE_THREAD_NAME), this.threadPool.getThreadContext(), this.threadPool.scheduler(), new MasterServiceStarvationWatcher(this.starvationLoggingThreshold.getMillis(), this.threadPool::relativeTimeInMillis, () -> this.threadPoolExecutor));
    }

    public ClusterStateUpdateStats getClusterStateUpdateStats() {
        return this.clusterStateUpdateStatsTracker.getStatistics();
    }

    @Override
    protected synchronized void doStop() {
        ThreadPool.terminate(this.threadPoolExecutor, 10L, TimeUnit.SECONDS);
    }

    @Override
    protected synchronized void doClose() {
    }

    ClusterState state() {
        return this.clusterStateSupplier.get();
    }

    public static boolean isMasterUpdateThread() {
        return Thread.currentThread().getName().contains("[masterService#updateTask]");
    }

    public static boolean assertMasterUpdateThread() {
        assert (MasterService.isMasterUpdateThread()) : "not called from the master service thread";
        return true;
    }

    public static boolean assertNotMasterUpdateThread(String reason) {
        assert (!MasterService.isMasterUpdateThread()) : "Expected current thread [" + Thread.currentThread() + "] to not be the master service thread. Reason: [" + reason + "]";
        return true;
    }

    private void runTasks(TaskInputs taskInputs) {
        String summary = taskInputs.summary;
        if (!this.lifecycle.started()) {
            logger.debug("processing [{}]: ignoring, master service not started", (Object)summary);
            return;
        }
        logger.debug("executing cluster state update for [{}]", (Object)summary);
        ClusterState previousClusterState = this.state();
        if (!previousClusterState.nodes().isLocalNodeElectedMaster() && taskInputs.runOnlyWhenMaster()) {
            logger.debug("failing [{}]: local node is no longer master", (Object)summary);
            taskInputs.onNoLongerMaster();
            return;
        }
        long computationStartTime = this.threadPool.rawRelativeTimeInMillis();
        TaskOutputs taskOutputs = this.calculateTaskOutputs(taskInputs, previousClusterState);
        taskOutputs.notifyFailedTasks();
        TimeValue computationTime = this.getTimeSince(computationStartTime);
        this.logExecutionTime(computationTime, "compute cluster state update", summary);
        if (taskOutputs.clusterStateUnchanged()) {
            long notificationStartTime = this.threadPool.rawRelativeTimeInMillis();
            taskOutputs.notifySuccessfulTasksOnUnchangedClusterState();
            TimeValue executionTime = this.getTimeSince(notificationStartTime);
            this.logExecutionTime(executionTime, "notify listeners on unchanged cluster state", summary);
            this.clusterStateUpdateStatsTracker.onUnchangedClusterState(computationTime.millis(), executionTime.millis());
        } else {
            ClusterState newClusterState = taskOutputs.newClusterState;
            if (logger.isTraceEnabled()) {
                logger.trace("cluster state updated, source [{}]\n{}", (Object)summary, (Object)newClusterState);
            } else {
                logger.debug("cluster state updated, version [{}], source [{}]", (Object)newClusterState.version(), (Object)summary);
            }
            long publicationStartTime = this.threadPool.rawRelativeTimeInMillis();
            try {
                String nodesDeltaSummary;
                ClusterStatePublicationEvent clusterStatePublicationEvent = new ClusterStatePublicationEvent(summary, previousClusterState, newClusterState, computationTime.millis(), publicationStartTime);
                DiscoveryNodes.Delta nodesDelta = newClusterState.nodes().delta(previousClusterState.nodes());
                if (nodesDelta.hasChanges() && logger.isInfoEnabled() && (nodesDeltaSummary = nodesDelta.shortSummary()).length() > 0) {
                    logger.info("{}, term: {}, version: {}, delta: {}", (Object)summary, (Object)newClusterState.term(), (Object)newClusterState.version(), (Object)nodesDeltaSummary);
                }
                logger.debug("publishing cluster state version [{}]", (Object)newClusterState.version());
                this.publish(clusterStatePublicationEvent, taskOutputs);
            }
            catch (Exception e) {
                this.handleException(summary, publicationStartTime, newClusterState, e);
            }
        }
    }

    private TimeValue getTimeSince(long startTimeMillis) {
        return TimeValue.timeValueMillis((long)Math.max(0L, this.threadPool.rawRelativeTimeInMillis() - startTimeMillis));
    }

    protected void publish(ClusterStatePublicationEvent clusterStatePublicationEvent, TaskOutputs taskOutputs) {
        PlainActionFuture<Void> fut = new PlainActionFuture<Void>(){

            @Override
            protected boolean blockingAllowed() {
                return MasterService.isMasterUpdateThread() || super.blockingAllowed();
            }
        };
        this.clusterStatePublisher.publish(clusterStatePublicationEvent, (ActionListener<Void>)fut, taskOutputs.createAckListener(this.threadPool, clusterStatePublicationEvent.getNewState()));
        try {
            FutureUtils.get(fut);
            this.onPublicationSuccess(clusterStatePublicationEvent, taskOutputs);
        }
        catch (Exception e) {
            this.onPublicationFailed(clusterStatePublicationEvent, taskOutputs, e);
        }
    }

    void onPublicationSuccess(ClusterStatePublicationEvent clusterStatePublicationEvent, TaskOutputs taskOutputs) {
        long notificationStartTime = this.threadPool.rawRelativeTimeInMillis();
        taskOutputs.processedDifferentClusterState(clusterStatePublicationEvent.getOldState(), clusterStatePublicationEvent.getNewState());
        try {
            taskOutputs.clusterStatePublished(clusterStatePublicationEvent);
        }
        catch (Exception e) {
            logger.error(() -> new ParameterizedMessage("exception thrown while notifying executor of new cluster state publication [{}]", (Object)clusterStatePublicationEvent.getSummary()), (Throwable)e);
        }
        TimeValue executionTime = this.getTimeSince(notificationStartTime);
        this.logExecutionTime(executionTime, "notify listeners on successful publication of cluster state (version: " + clusterStatePublicationEvent.getNewState().version() + ", uuid: " + clusterStatePublicationEvent.getNewState().stateUUID() + ')', clusterStatePublicationEvent.getSummary());
        this.clusterStateUpdateStatsTracker.onPublicationSuccess(this.threadPool.rawRelativeTimeInMillis(), clusterStatePublicationEvent, executionTime.millis());
    }

    void onPublicationFailed(ClusterStatePublicationEvent clusterStatePublicationEvent, TaskOutputs taskOutputs, Exception exception) {
        if (exception instanceof FailedToCommitClusterStateException) {
            long notificationStartTime = this.threadPool.rawRelativeTimeInMillis();
            long version = clusterStatePublicationEvent.getNewState().version();
            logger.warn(() -> new ParameterizedMessage("failing [{}]: failed to commit cluster state version [{}]", (Object)clusterStatePublicationEvent.getSummary(), (Object)version), (Throwable)exception);
            taskOutputs.publishingFailed((FailedToCommitClusterStateException)exception);
            long notificationMillis = this.threadPool.rawRelativeTimeInMillis() - notificationStartTime;
            this.clusterStateUpdateStatsTracker.onPublicationFailure(this.threadPool.rawRelativeTimeInMillis(), clusterStatePublicationEvent, notificationMillis);
        } else {
            assert (this.publicationMayFail()) : exception;
            this.clusterStateUpdateStatsTracker.onPublicationFailure(this.threadPool.rawRelativeTimeInMillis(), clusterStatePublicationEvent, 0L);
            this.handleException(clusterStatePublicationEvent.getSummary(), clusterStatePublicationEvent.getPublicationStartTimeMillis(), clusterStatePublicationEvent.getNewState(), exception);
        }
    }

    protected boolean publicationMayFail() {
        return false;
    }

    private void handleException(String summary, long startTimeMillis, ClusterState newClusterState, Exception e) {
        TimeValue executionTime = this.getTimeSince(startTimeMillis);
        long version = newClusterState.version();
        String stateUUID = newClusterState.stateUUID();
        String fullState = newClusterState.toString();
        logger.warn((Message)new ParameterizedMessage("took [{}] and then failed to publish updated cluster state (version: {}, uuid: {}) for [{}]:\n{}", new Object[]{executionTime, version, stateUUID, summary, fullState}), (Throwable)e);
    }

    private TaskOutputs calculateTaskOutputs(TaskInputs taskInputs, ClusterState previousClusterState) {
        ClusterStateTaskExecutor.ClusterTasksResult<Object> clusterTasksResult = this.executeTasks(taskInputs, previousClusterState);
        ClusterState newClusterState = this.patchVersions(previousClusterState, clusterTasksResult);
        return new TaskOutputs(taskInputs, previousClusterState, newClusterState, this.getNonFailedTasks(taskInputs, clusterTasksResult), clusterTasksResult.executionResults);
    }

    private ClusterState patchVersions(ClusterState previousClusterState, ClusterStateTaskExecutor.ClusterTasksResult<?> executionResult) {
        ClusterState newClusterState = executionResult.resultingState;
        if (previousClusterState != newClusterState) {
            SortedMap<String, IndexAbstraction> previousIndicesLookup = newClusterState.metadata().getIndicesLookup();
            ClusterState.Builder builder = this.incrementVersion(newClusterState);
            if (previousClusterState.routingTable() != newClusterState.routingTable()) {
                builder.routingTable(newClusterState.routingTable().withIncrementedVersion());
            }
            if (previousClusterState.metadata() != newClusterState.metadata()) {
                builder.metadata(newClusterState.metadata().withIncrementedVersion());
            }
            newClusterState = builder.build();
            assert (previousIndicesLookup == newClusterState.metadata().getIndicesLookup());
        }
        return newClusterState;
    }

    public ClusterState.Builder incrementVersion(ClusterState clusterState) {
        return ClusterState.builder(clusterState).incrementVersion();
    }

    public <T extends ClusterStateTaskConfig & ClusterStateTaskExecutor<T>> void submitStateUpdateTask(String source, T updateTask) {
        this.submitStateUpdateTask(source, updateTask, updateTask, updateTask, (ClusterStateTaskListener)updateTask);
    }

    public <T> void submitStateUpdateTask(String source, T task, ClusterStateTaskConfig config, ClusterStateTaskExecutor<T> executor, ClusterStateTaskListener listener) {
        this.submitStateUpdateTasks(source, Collections.singletonMap(task, listener), config, executor);
    }

    public List<PendingClusterTask> pendingTasks() {
        return Arrays.stream(this.threadPoolExecutor.getPending()).map(pending -> {
            assert (pending.task instanceof SourcePrioritizedRunnable) : "thread pool executor should only use SourcePrioritizedRunnable instances but found: " + pending.task.getClass().getName();
            SourcePrioritizedRunnable task = (SourcePrioritizedRunnable)pending.task;
            return new PendingClusterTask(pending.insertionOrder, pending.priority, new Text(task.source()), task.getAgeInMillis(), pending.executing);
        }).collect(Collectors.toList());
    }

    public int numberOfPendingTasks() {
        return this.threadPoolExecutor.getNumberOfPendingTasks();
    }

    public TimeValue getMaxTaskWaitTime() {
        return this.threadPoolExecutor.getMaxTaskWaitTime();
    }

    private SafeClusterStateTaskListener safe(ClusterStateTaskListener listener, Supplier<ThreadContext.StoredContext> contextSupplier) {
        if (listener instanceof AckedClusterStateTaskListener) {
            return new SafeAckedClusterStateTaskListener((AckedClusterStateTaskListener)listener, contextSupplier, logger);
        }
        return new SafeClusterStateTaskListener(listener, contextSupplier, logger);
    }

    private void logExecutionTime(TimeValue executionTime, String activity, String summary) {
        if (executionTime.getMillis() > this.slowTaskLoggingThreshold.getMillis()) {
            logger.warn("took [{}/{}ms] to {} for [{}], which exceeds the warn threshold of [{}]", (Object)executionTime, (Object)executionTime.getMillis(), (Object)activity, (Object)summary, (Object)this.slowTaskLoggingThreshold);
        } else {
            logger.debug("took [{}] to {} for [{}]", (Object)executionTime, (Object)activity, (Object)summary);
        }
    }

    private ClusterStateTaskExecutor.ClusterTasksResult<Object> executeTasks(TaskInputs taskInputs, ClusterState previousClusterState) {
        ClusterStateTaskExecutor.ClusterTasksResult<Object> clusterTasksResult;
        try {
            List inputs = taskInputs.updateTasks.stream().map(tUpdateTask -> tUpdateTask.task).collect(Collectors.toList());
            clusterTasksResult = taskInputs.executor.execute(previousClusterState, inputs);
            if (previousClusterState != clusterTasksResult.resultingState && previousClusterState.nodes().isLocalNodeElectedMaster() && !clusterTasksResult.resultingState.nodes().isLocalNodeElectedMaster()) {
                throw new AssertionError((Object)"update task submitted to MasterService cannot remove master");
            }
        }
        catch (Exception e) {
            logger.trace(() -> new ParameterizedMessage("failed to execute cluster state update (on version: [{}], uuid: [{}]) for [{}]\n{}{}{}", new Object[]{previousClusterState.version(), previousClusterState.stateUUID(), taskInputs.summary, previousClusterState.nodes(), previousClusterState.routingTable(), previousClusterState.getRoutingNodes()}), (Throwable)e);
            clusterTasksResult = ClusterStateTaskExecutor.ClusterTasksResult.builder().failures(taskInputs.updateTasks.stream().map(updateTask -> updateTask.task)::iterator, e).build(previousClusterState);
        }
        assert (clusterTasksResult.executionResults != null);
        assert (clusterTasksResult.executionResults.size() == taskInputs.updateTasks.size()) : String.format(Locale.ROOT, "expected [%d] task result%s but was [%d]", taskInputs.updateTasks.size(), taskInputs.updateTasks.size() == 1 ? "" : "s", clusterTasksResult.executionResults.size());
        if (Assertions.ENABLED) {
            ClusterStateTaskExecutor.ClusterTasksResult<Object> finalClusterTasksResult = clusterTasksResult;
            taskInputs.updateTasks.forEach(updateTask -> {
                assert (finalClusterTasksResult.executionResults.containsKey(updateTask.task)) : "missing task result for " + updateTask;
            });
        }
        return clusterTasksResult;
    }

    private List<Batcher.UpdateTask> getNonFailedTasks(TaskInputs taskInputs, ClusterStateTaskExecutor.ClusterTasksResult<Object> clusterTasksResult) {
        return taskInputs.updateTasks.stream().filter(updateTask -> {
            assert (clusterTasksResult.executionResults.containsKey(updateTask.task)) : "missing " + updateTask;
            ClusterStateTaskExecutor.TaskResult taskResult = clusterTasksResult.executionResults.get(updateTask.task);
            return taskResult.isSuccess();
        }).collect(Collectors.toList());
    }

    public <T> void submitStateUpdateTasks(String source, Map<T, ClusterStateTaskListener> tasks, ClusterStateTaskConfig config, ClusterStateTaskExecutor<T> executor) {
        block9: {
            if (!this.lifecycle.started()) {
                return;
            }
            ThreadContext threadContext = this.threadPool.getThreadContext();
            Supplier<ThreadContext.StoredContext> supplier = threadContext.newRestorableContext(true);
            try (ThreadContext.StoredContext ignore = threadContext.stashContext();){
                threadContext.markAsSystemContext();
                List safeTasks = tasks.entrySet().stream().map(e -> {
                    Batcher batcher = this.taskBatcher;
                    Objects.requireNonNull(batcher);
                    return batcher.new Batcher.UpdateTask(config.priority(), source, e.getKey(), this.safe((ClusterStateTaskListener)e.getValue(), supplier), executor);
                }).collect(Collectors.toList());
                this.taskBatcher.submitTasks(safeTasks, config.timeout());
            }
            catch (EsRejectedExecutionException e2) {
                if (this.lifecycle.stoppedOrClosed()) break block9;
                throw e2;
            }
        }
    }

    private class TaskInputs {
        final String summary;
        final List<Batcher.UpdateTask> updateTasks;
        final ClusterStateTaskExecutor<Object> executor;

        TaskInputs(ClusterStateTaskExecutor<Object> executor, List<Batcher.UpdateTask> updateTasks, String summary) {
            this.summary = summary;
            this.executor = executor;
            this.updateTasks = updateTasks;
        }

        boolean runOnlyWhenMaster() {
            return this.executor.runOnlyOnMaster();
        }

        void onNoLongerMaster() {
            this.updateTasks.forEach(task -> task.listener.onNoLongerMaster(task.source()));
        }
    }

    private static class ClusterStateUpdateStatsTracker {
        private long unchangedTaskCount;
        private long publicationSuccessCount;
        private long publicationFailureCount;
        private long unchangedComputationElapsedMillis;
        private long unchangedNotificationElapsedMillis;
        private long successfulComputationElapsedMillis;
        private long successfulPublicationElapsedMillis;
        private long successfulContextConstructionElapsedMillis;
        private long successfulCommitElapsedMillis;
        private long successfulCompletionElapsedMillis;
        private long successfulMasterApplyElapsedMillis;
        private long successfulNotificationElapsedMillis;
        private long failedComputationElapsedMillis;
        private long failedPublicationElapsedMillis;
        private long failedContextConstructionElapsedMillis;
        private long failedCommitElapsedMillis;
        private long failedCompletionElapsedMillis;
        private long failedMasterApplyElapsedMillis;
        private long failedNotificationElapsedMillis;

        private ClusterStateUpdateStatsTracker() {
        }

        synchronized void onUnchangedClusterState(long computationElapsedMillis, long notificationElapsedMillis) {
            ++this.unchangedTaskCount;
            this.unchangedComputationElapsedMillis += computationElapsedMillis;
            this.unchangedNotificationElapsedMillis += notificationElapsedMillis;
        }

        synchronized void onPublicationSuccess(long currentTimeMillis, ClusterStatePublicationEvent clusterStatePublicationEvent, long notificationElapsedMillis) {
            ++this.publicationSuccessCount;
            this.successfulComputationElapsedMillis += clusterStatePublicationEvent.getComputationTimeMillis();
            this.successfulPublicationElapsedMillis += currentTimeMillis - clusterStatePublicationEvent.getPublicationStartTimeMillis();
            this.successfulContextConstructionElapsedMillis += clusterStatePublicationEvent.getPublicationContextConstructionElapsedMillis();
            this.successfulCommitElapsedMillis += clusterStatePublicationEvent.getPublicationCommitElapsedMillis();
            this.successfulCompletionElapsedMillis += clusterStatePublicationEvent.getPublicationCompletionElapsedMillis();
            this.successfulMasterApplyElapsedMillis += clusterStatePublicationEvent.getMasterApplyElapsedMillis();
            this.successfulNotificationElapsedMillis += notificationElapsedMillis;
        }

        synchronized void onPublicationFailure(long currentTimeMillis, ClusterStatePublicationEvent clusterStatePublicationEvent, long notificationMillis) {
            ++this.publicationFailureCount;
            this.failedComputationElapsedMillis += clusterStatePublicationEvent.getComputationTimeMillis();
            this.failedPublicationElapsedMillis += currentTimeMillis - clusterStatePublicationEvent.getPublicationStartTimeMillis();
            this.failedContextConstructionElapsedMillis += clusterStatePublicationEvent.maybeGetPublicationContextConstructionElapsedMillis();
            this.failedCommitElapsedMillis += clusterStatePublicationEvent.maybeGetPublicationCommitElapsedMillis();
            this.failedCompletionElapsedMillis += clusterStatePublicationEvent.maybeGetPublicationCompletionElapsedMillis();
            this.failedMasterApplyElapsedMillis += clusterStatePublicationEvent.maybeGetMasterApplyElapsedMillis();
            this.failedNotificationElapsedMillis += notificationMillis;
        }

        synchronized ClusterStateUpdateStats getStatistics() {
            return new ClusterStateUpdateStats(this.unchangedTaskCount, this.publicationSuccessCount, this.publicationFailureCount, this.unchangedComputationElapsedMillis, this.unchangedNotificationElapsedMillis, this.successfulComputationElapsedMillis, this.successfulPublicationElapsedMillis, this.successfulContextConstructionElapsedMillis, this.successfulCommitElapsedMillis, this.successfulCompletionElapsedMillis, this.successfulMasterApplyElapsedMillis, this.successfulNotificationElapsedMillis, this.failedComputationElapsedMillis, this.failedPublicationElapsedMillis, this.failedContextConstructionElapsedMillis, this.failedCommitElapsedMillis, this.failedCompletionElapsedMillis, this.failedMasterApplyElapsedMillis, this.failedNotificationElapsedMillis);
        }
    }

    class Batcher
    extends TaskBatcher {
        Batcher(Logger logger, PrioritizedEsThreadPoolExecutor threadExecutor) {
            super(logger, threadExecutor);
        }

        @Override
        protected void onTimeout(List<? extends TaskBatcher.BatchedTask> tasks, TimeValue timeout) {
            MasterService.this.threadPool.generic().execute(() -> tasks.forEach(task -> ((UpdateTask)task).listener.onFailure(task.source, new ProcessClusterEventTimeoutException(timeout, task.source))));
        }

        @Override
        protected void run(Object batchingKey, List<? extends TaskBatcher.BatchedTask> tasks, String tasksSummary) {
            ClusterStateTaskExecutor taskExecutor = (ClusterStateTaskExecutor)batchingKey;
            List<? extends TaskBatcher.BatchedTask> updateTasks = tasks;
            MasterService.this.runTasks(new TaskInputs(taskExecutor, updateTasks, tasksSummary));
        }

        class UpdateTask
        extends TaskBatcher.BatchedTask {
            final ClusterStateTaskListener listener;

            UpdateTask(Priority priority, String source, Object task, ClusterStateTaskListener listener, ClusterStateTaskExecutor<?> executor) {
                super(Batcher.this, priority, source, executor, task);
                this.listener = listener;
            }

            @Override
            public String describeTasks(List<? extends TaskBatcher.BatchedTask> tasks) {
                return ((ClusterStateTaskExecutor)this.batchingKey).describeTasks(tasks.stream().map(TaskBatcher.BatchedTask::getTask).collect(Collectors.toList()));
            }
        }
    }

    private static class MasterServiceStarvationWatcher
    implements PrioritizedEsThreadPoolExecutor.StarvationWatcher {
        private final long warnThreshold;
        private final LongSupplier nowMillisSupplier;
        private final Supplier<PrioritizedEsThreadPoolExecutor> threadPoolExecutorSupplier;
        private long lastLogMillis;
        private long nonemptySinceMillis;
        private boolean isEmpty = true;

        MasterServiceStarvationWatcher(long warnThreshold, LongSupplier nowMillisSupplier, Supplier<PrioritizedEsThreadPoolExecutor> threadPoolExecutorSupplier) {
            this.nowMillisSupplier = nowMillisSupplier;
            this.threadPoolExecutorSupplier = threadPoolExecutorSupplier;
            this.warnThreshold = warnThreshold;
        }

        @Override
        public synchronized void onEmptyQueue() {
            this.isEmpty = true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void onNonemptyQueue() {
            long nonemptyDurationMillis;
            long nowMillis = this.nowMillisSupplier.getAsLong();
            MasterServiceStarvationWatcher masterServiceStarvationWatcher = this;
            synchronized (masterServiceStarvationWatcher) {
                if (this.isEmpty) {
                    this.isEmpty = false;
                    this.nonemptySinceMillis = nowMillis;
                    this.lastLogMillis = nowMillis;
                    return;
                }
                if (nowMillis - this.lastLogMillis < this.warnThreshold) {
                    return;
                }
                this.lastLogMillis = nowMillis;
                nonemptyDurationMillis = nowMillis - this.nonemptySinceMillis;
            }
            PrioritizedEsThreadPoolExecutor threadPoolExecutor = this.threadPoolExecutorSupplier.get();
            TimeValue maxTaskWaitTime = threadPoolExecutor.getMaxTaskWaitTime();
            logger.warn("pending task queue has been nonempty for [{}/{}ms] which is longer than the warn threshold of [{}ms]; there are currently [{}] pending tasks, the oldest of which has age [{}/{}ms]", (Object)TimeValue.timeValueMillis((long)nonemptyDurationMillis), (Object)nonemptyDurationMillis, (Object)this.warnThreshold, (Object)threadPoolExecutor.getNumberOfPendingTasks(), (Object)maxTaskWaitTime, (Object)maxTaskWaitTime.millis());
        }
    }

    class TaskOutputs {
        final TaskInputs taskInputs;
        final ClusterState previousClusterState;
        final ClusterState newClusterState;
        final List<Batcher.UpdateTask> nonFailedTasks;
        final Map<Object, ClusterStateTaskExecutor.TaskResult> executionResults;

        TaskOutputs(TaskInputs taskInputs, ClusterState previousClusterState, ClusterState newClusterState, List<Batcher.UpdateTask> nonFailedTasks, Map<Object, ClusterStateTaskExecutor.TaskResult> executionResults) {
            this.taskInputs = taskInputs;
            this.previousClusterState = previousClusterState;
            this.newClusterState = newClusterState;
            this.nonFailedTasks = nonFailedTasks;
            this.executionResults = executionResults;
        }

        void publishingFailed(FailedToCommitClusterStateException t) {
            this.nonFailedTasks.forEach(task -> task.listener.onFailure(task.source(), t));
        }

        void processedDifferentClusterState(ClusterState previousClusterState, ClusterState newClusterState) {
            this.nonFailedTasks.forEach(task -> task.listener.clusterStateProcessed(task.source(), previousClusterState, newClusterState));
        }

        void clusterStatePublished(ClusterStatePublicationEvent clusterStatePublicationEvent) {
            this.taskInputs.executor.clusterStatePublished(clusterStatePublicationEvent);
        }

        ClusterStatePublisher.AckListener createAckListener(ThreadPool threadPool, ClusterState newClusterState) {
            return new DelegatingAckListener(this.nonFailedTasks.stream().filter(task -> task.listener instanceof AckedClusterStateTaskListener).map(task -> new AckCountDownListener((AckedClusterStateTaskListener)task.listener, newClusterState.version(), newClusterState.nodes(), threadPool)).collect(Collectors.toList()));
        }

        boolean clusterStateUnchanged() {
            return this.previousClusterState == this.newClusterState;
        }

        void notifyFailedTasks() {
            for (Batcher.UpdateTask updateTask : this.taskInputs.updateTasks) {
                assert (this.executionResults.containsKey(updateTask.task)) : "missing " + updateTask;
                ClusterStateTaskExecutor.TaskResult taskResult = this.executionResults.get(updateTask.task);
                if (taskResult.isSuccess()) continue;
                updateTask.listener.onFailure(updateTask.source(), taskResult.getFailure());
            }
        }

        void notifySuccessfulTasksOnUnchangedClusterState() {
            this.nonFailedTasks.forEach(task -> {
                if (task.listener instanceof AckedClusterStateTaskListener) {
                    ((AckedClusterStateTaskListener)task.listener).onAllNodesAcked(null);
                }
                task.listener.clusterStateProcessed(task.source(), this.newClusterState, this.newClusterState);
            });
        }
    }

    private static class SafeAckedClusterStateTaskListener
    extends SafeClusterStateTaskListener
    implements AckedClusterStateTaskListener {
        private final AckedClusterStateTaskListener listener;
        private final Logger logger;

        SafeAckedClusterStateTaskListener(AckedClusterStateTaskListener listener, Supplier<ThreadContext.StoredContext> context, Logger logger) {
            super(listener, context, logger);
            this.listener = listener;
            this.logger = logger;
        }

        @Override
        public boolean mustAck(DiscoveryNode discoveryNode) {
            return this.listener.mustAck(discoveryNode);
        }

        @Override
        public void onAllNodesAcked(@Nullable Exception e) {
            try (ThreadContext.StoredContext ignore = (ThreadContext.StoredContext)this.context.get();){
                this.listener.onAllNodesAcked(e);
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                this.logger.error("exception thrown by listener while notifying on all nodes acked", (Throwable)inner);
            }
        }

        @Override
        public void onAckTimeout() {
            try (ThreadContext.StoredContext ignore = (ThreadContext.StoredContext)this.context.get();){
                this.listener.onAckTimeout();
            }
            catch (Exception e) {
                this.logger.error("exception thrown by listener while notifying on ack timeout", (Throwable)e);
            }
        }

        @Override
        public TimeValue ackTimeout() {
            return this.listener.ackTimeout();
        }
    }

    private static class SafeClusterStateTaskListener
    implements ClusterStateTaskListener {
        private final ClusterStateTaskListener listener;
        protected final Supplier<ThreadContext.StoredContext> context;
        private final Logger logger;

        SafeClusterStateTaskListener(ClusterStateTaskListener listener, Supplier<ThreadContext.StoredContext> context, Logger logger) {
            this.listener = listener;
            this.context = context;
            this.logger = logger;
        }

        @Override
        public void onFailure(String source, Exception e) {
            try (ThreadContext.StoredContext ignore = this.context.get();){
                this.listener.onFailure(source, e);
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                this.logger.error(() -> new ParameterizedMessage("exception thrown by listener notifying of failure from [{}]", (Object)source), (Throwable)inner);
            }
        }

        @Override
        public void onNoLongerMaster(String source) {
            try (ThreadContext.StoredContext ignore = this.context.get();){
                this.listener.onNoLongerMaster(source);
            }
            catch (Exception e) {
                this.logger.error(() -> new ParameterizedMessage("exception thrown by listener while notifying no longer master from [{}]", (Object)source), (Throwable)e);
            }
        }

        @Override
        public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
            try (ThreadContext.StoredContext ignore = this.context.get();){
                this.listener.clusterStateProcessed(source, oldState, newState);
            }
            catch (Exception e) {
                this.logger.error(() -> new ParameterizedMessage("exception thrown by listener while notifying of cluster state processed from [{}], old cluster state:\n{}\nnew cluster state:\n{}", new Object[]{source, oldState, newState}), (Throwable)e);
            }
        }
    }

    private static class AckCountDownListener
    implements ClusterStatePublisher.AckListener {
        private static final Logger logger = LogManager.getLogger(AckCountDownListener.class);
        private final AckedClusterStateTaskListener ackedTaskListener;
        private final CountDown countDown;
        private final DiscoveryNode masterNode;
        private final ThreadPool threadPool;
        private final long clusterStateVersion;
        private volatile Scheduler.Cancellable ackTimeoutCallback;
        private Exception lastFailure;

        AckCountDownListener(AckedClusterStateTaskListener ackedTaskListener, long clusterStateVersion, DiscoveryNodes nodes, ThreadPool threadPool) {
            this.ackedTaskListener = ackedTaskListener;
            this.clusterStateVersion = clusterStateVersion;
            this.threadPool = threadPool;
            this.masterNode = nodes.getMasterNode();
            int countDown = 0;
            for (DiscoveryNode node : nodes) {
                if (!node.equals(this.masterNode) && !ackedTaskListener.mustAck(node)) continue;
                ++countDown;
            }
            logger.trace("expecting {} acknowledgements for cluster_state update (version: {})", (Object)countDown, (Object)clusterStateVersion);
            this.countDown = new CountDown(countDown + 1);
        }

        @Override
        public void onCommit(TimeValue commitTime) {
            TimeValue timeLeft;
            TimeValue ackTimeout = this.ackedTaskListener.ackTimeout();
            if (ackTimeout == null) {
                ackTimeout = TimeValue.ZERO;
            }
            if ((timeLeft = TimeValue.timeValueNanos((long)Math.max(0L, ackTimeout.nanos() - commitTime.nanos()))).nanos() == 0L) {
                this.onTimeout();
            } else if (this.countDown.countDown()) {
                this.finish();
            } else {
                this.ackTimeoutCallback = this.threadPool.schedule(this::onTimeout, timeLeft, "generic");
                if (this.countDown.isCountedDown()) {
                    this.ackTimeoutCallback.cancel();
                }
            }
        }

        @Override
        public void onNodeAck(DiscoveryNode node, @Nullable Exception e) {
            if (!node.equals(this.masterNode) && !this.ackedTaskListener.mustAck(node)) {
                return;
            }
            if (e == null) {
                logger.trace("ack received from node [{}], cluster_state update (version: {})", (Object)node, (Object)this.clusterStateVersion);
            } else {
                this.lastFailure = e;
                logger.debug(() -> new ParameterizedMessage("ack received from node [{}], cluster_state update (version: {})", (Object)node, (Object)this.clusterStateVersion), (Throwable)e);
            }
            if (this.countDown.countDown()) {
                this.finish();
            }
        }

        private void finish() {
            logger.trace("all expected nodes acknowledged cluster_state update (version: {})", (Object)this.clusterStateVersion);
            if (this.ackTimeoutCallback != null) {
                this.ackTimeoutCallback.cancel();
            }
            this.ackedTaskListener.onAllNodesAcked(this.lastFailure);
        }

        public void onTimeout() {
            if (this.countDown.fastForward()) {
                logger.trace("timeout waiting for acknowledgement for cluster_state update (version: {})", (Object)this.clusterStateVersion);
                this.ackedTaskListener.onAckTimeout();
            }
        }
    }

    private static class DelegatingAckListener
    implements ClusterStatePublisher.AckListener {
        private final List<ClusterStatePublisher.AckListener> listeners;

        private DelegatingAckListener(List<ClusterStatePublisher.AckListener> listeners) {
            this.listeners = listeners;
        }

        @Override
        public void onCommit(TimeValue commitTime) {
            for (ClusterStatePublisher.AckListener listener : this.listeners) {
                listener.onCommit(commitTime);
            }
        }

        @Override
        public void onNodeAck(DiscoveryNode node, @Nullable Exception e) {
            for (ClusterStatePublisher.AckListener listener : this.listeners) {
                listener.onNodeAck(node, e);
            }
        }
    }
}

