/*

    Euchre - a free as in freedom and as in beer version of the 
             euchre card game
  
    Copyright 2002 C Nathan Buckles (nbuckles@bigfoot.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "Card.hpp"
#include "Common.hpp"

Card::Card() : itsSuit(Card::NoSuit), itsNumber(Card::NoNumber) {}

Card::Card(Suit aSuit, Number aNumber) :
  itsSuit(aSuit), itsNumber(aNumber),
  itsName(getSuitChar(aSuit) + " " + getNumberChar(aNumber)) {}

Card::~Card() {}

Card::Suit Card::getSuit() const {
  return itsSuit;
}

Card::Suit Card::getAdjSuit() const {
  Card::Suit ret = itsSuit;
  
  if (itsNumber == Jack) {
    if (ret == getPartnerSuit(Common::getTrump())) {
      ret = Common::getTrump();
    }
  }

  return ret;
}

char Card::getSuitChar() const {
  return getSuitChar(itsSuit);
}

char Card::getSuitChar(Suit suit) {
  char c = 'X';

  switch (suit) {
  case Clubs:
    c = 'c';
    break;

  case Diamonds:
    c = 'd';
    break;

  case Hearts:
    c = 'h';
    break;

  case Spades:
    c = 's';
    break;
  }

  return c;
}

char* Card::getSuitStr() const {
  return getSuitStr(itsSuit);
}

static char* suitStrings[] = {
  "Diamonds",
  "Clubs",
  "Hearts",
  "Spades",
  "None"
};

char* Card::getSuitStr(Card::Suit suit) {
  if (suit < Diamonds || suit > Spades) {
    suit = (Suit) (Spades + 1);
  }

  return suitStrings[suit];
}

Card::Number Card::getNumber() const {
  return itsNumber;
}

char Card::getNumberChar() const {
  return getNumberChar(itsNumber);
}

char Card::getNumberChar(Number number) {
  char c = 'X';

  switch (number) {
  case Nine:
    c = '9';
    break;

  case Ten:
    c = 'T';
    break;

  case Jack:
    c = 'J';
    break;

  case Queen:
    c = 'Q';
    break;

  case King:
    c = 'K';
    break;

  case Ace:
    c = 'A';
    break;
  }

  return c;
}

int Card::isSuit(const Suit aSuit) const {
  return isSuit(aSuit, Common::getTrump());
}

int Card::isSuit(const Suit aSuit, const Suit aTrump) const {
  Suit ptrump = getPartnerSuit(aTrump);

  if (aSuit == aTrump) {
    if (itsNumber == Jack && itsSuit == ptrump) {
      return 1;
    }
  } else if (aSuit == ptrump) {
    if (itsNumber == Jack && itsSuit == ptrump) {
      return 0;
    }
  }
  
  return (itsSuit == aSuit);
}

int Card::getValue() const {
  return getValue(Common::getTrump());
}

int Card::getValue(const Suit aTrump) const {
  int ret;
  Suit trump_partner = getPartnerSuit(aTrump);

  if (itsSuit == aTrump) {
    if (itsNumber == Jack) {
      ret = JackTrumpVal;
    } else {
      ret = (itsNumber + TrumpAdj);
    }
  } else if (itsSuit == trump_partner) {
    if (itsNumber == Jack) {
      ret = JackPartVal;
    } else {
      ret = itsNumber;
    } 
  } else {
    ret = itsNumber;
  }

  if (ret < 0) {
    ret = 0;
  }
  
  return ret;
}

int Card::operator==(const Card& card) const {
  if (itsSuit == card.itsSuit &&
      itsNumber == card.itsNumber) {
    return 1;
  }

  return 0;
}

int Card::operator!=(const Card& card) const {
  return ! (*this == card);
}

Card::Suit Card::getPartnerSuit(Suit suit) {
  if (suit == Card::NoSuit) {
    return Card::NoSuit;
  }

  return (Card::Suit) ((suit + 2) % (Spades + 1));
}

const char* Card::getName() const {
  return itsName.c_str();
}
