/*

    Euchre - a free as in freedom and as in beer version of the 
             euchre card game
  
    Copyright 2002 C Nathan Buckles (nbuckles@bigfoot.com)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "Debug.hpp"
#include "Options.hpp"

#define OPTIONS_FILE_NAME ".euchrerc"
#define OPTIONS_PATH_SIZE 1024

Options* Options::itsOptions = NULL;

Options::Options() {
  itsVersion           = CURRENT_VERSION;
  itsDelayMode         = DEFAULTDELAY;
  itsTricksOnePoint    = 3;
  itsPointsForGame     = DEFAULTPOINTS;
  
  itsLevel[0]          = Options::DEFAULTLEVEL;
  itsLevel[1]          = Options::DEFAULTLEVEL;
  itsLevel[2]          = Options::DEFAULTLEVEL;
  itsLevel[3]          = Options::DEFAULTLEVEL;

  itsAgg[0]            = Options::DEFAULTAGG;
  itsAgg[1]            = Options::DEFAULTAGG;
  itsAgg[2]            = Options::DEFAULTAGG;
  itsAgg[3]            = Options::DEFAULTAGG;

  itsCheat             = DEFAULTCHEAT;
  itsAutoPlay          = 0;
  itsDelayBetweenPlays = 1;
  itsAutoDealEnd       = 0;
  itsPartnerLoner      = 1;
  itsStickTheDealer    = 0;
  itsDebug             = 0;
}

Options::~Options() { }

void Options::init() {
  /* read from the file if we can */
  load();

  /* now do other stuff so it will override the file contents */
  if (getenv("EUCHRE_DEBUG") != NULL) {
    itsDebug = 1;
  }
}

int Options::save() {
  char* dir = getenv("HOME");
  char fullpath[OPTIONS_PATH_SIZE];

  FILE *fp;

  snprintf(fullpath, OPTIONS_PATH_SIZE, "%s/%s", dir, OPTIONS_FILE_NAME);

  fp = fopen(fullpath, "w+");
  if (!fp) {
    LOG("could not open %s\n", fullpath);
    return 1;
  }

  return writeOutOptions(fp);
}

int Options::load() {
  char* dir = getenv("HOME");
  char fullpath[OPTIONS_PATH_SIZE];

  FILE *fp;

  snprintf(fullpath, OPTIONS_PATH_SIZE, "%s/%s", dir, OPTIONS_FILE_NAME);

  fp = fopen(fullpath, "r");

  if (!fp) {
    LOG("could not open %s\n", fullpath);
    return 1;
  }

  int ret = readInOptions(fp);
  if (normalizeOptions()) {
    save();
  }

  return ret;
}

int Options::writeOutOptions(FILE *fp) {
  fprintf(fp, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d ",
	  itsVersion, itsDelayMode, itsTricksOnePoint,
	  itsPointsForGame,  itsLevel[0], itsLevel[1], itsLevel[2],
	  itsLevel[3], itsAgg[0], itsAgg[1], itsAgg[2], itsAgg[3], itsCheat,
	  itsAutoPlay, itsDelayBetweenPlays, itsAutoDealEnd,
	  itsPartnerLoner, itsStickTheDealer, itsDebug);

  return 0;
}

int Options::readInOptions(FILE *fp) {
  int temp;
  fscanf(fp, "%d", &temp);

  if (temp == CURRENT_VERSION) {
    itsVersion = temp;
    fscanf(fp, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
	   &itsDelayMode, &itsTricksOnePoint,
	   &itsPointsForGame,  &itsLevel[0], &itsLevel[1], &itsLevel[2],
	   &itsLevel[3], &itsAgg[0], &itsAgg[1], &itsAgg[2], &itsAgg[3], &itsCheat,
	   &itsAutoPlay, &itsDelayBetweenPlays, &itsAutoDealEnd,
	   &itsPartnerLoner, &itsStickTheDealer, &itsDebug);
  } else if (temp == FIRST_VERSION) {
    /* this version does not have itsPartnerLoner */
    itsVersion = temp;
    fscanf(fp, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
	   &itsDelayMode, &itsTricksOnePoint,
	   &itsPointsForGame,  &itsLevel[0], &itsLevel[1], &itsLevel[2],
	   &itsLevel[3], &itsAgg[0], &itsAgg[1], &itsAgg[2], &itsAgg[3], &itsCheat,
	   &itsAutoPlay, &itsDelayBetweenPlays, &itsAutoDealEnd, &itsDebug);
  } else if (temp == SECOND_VERSION) {
    /* this version does not have itsStickTheDealer */
    itsVersion = temp;
    fscanf(fp, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
	   &itsDelayMode, &itsTricksOnePoint, &itsPointsForGame,
	   &itsLevel[0], &itsLevel[1], &itsLevel[2], &itsLevel[3],
	   &itsAgg[0], &itsAgg[1], &itsAgg[2], &itsAgg[3], &itsCheat,
	   &itsAutoPlay, &itsDelayBetweenPlays, &itsAutoDealEnd, 
	   &itsPartnerLoner, &itsDebug);
  } else {
    /* the oldest version (those with no version at all really) have
       no itsAutoDealEnd.  set invalid version so we will rewrite the
       file properly2 */
    itsDelayMode = temp;
    fscanf(fp, "%d %d %d %d %d %d %d %d %d %d %d %d %d",
	   &itsTricksOnePoint, &itsPointsForGame,
	   &itsLevel[0], &itsLevel[1], &itsLevel[2], &itsLevel[3],
	   &itsAgg[0], &itsAgg[1], &itsAgg[2], &itsAgg[3], &itsCheat,
	   &itsAutoPlay, &itsDelayBetweenPlays, &itsDebug);
  }    

  return 0;
}

int Options::normalizeOptions() {
  int ret = 0;

  if (itsVersion != CURRENT_VERSION) {
    LOG("Options::normalizeOptions - version is not current!\n");
    itsVersion = CURRENT_VERSION;
    ret        = 1;
  }
  
  if (itsDelayMode < NODELAY || itsDelayMode > PAUSE) {
    LOG("WARNING:  bad option in preferences file! seting itsDelayMode to %s\n", DEFAULTDELAY);
    itsDelayMode = DEFAULTDELAY;
    ret          = 1;
  }

  if (itsTricksOnePoint < 0 || itsTricksOnePoint > Common::CARDS_PER_HAND) {
    LOG("WARNING: bad option in preferences file! seting itsTricksOnePoint to 3\n");
    itsTricksOnePoint = 3;
    ret               = 1;
  }

  if (itsPointsForGame < MINPOINTS || itsPointsForGame > MAXPOINTS) {
    LOG("WARNING: bad option in preferences file!  setting itsPointsForGame to %d\n", DEFAULTPOINTS);
    itsPointsForGame = DEFAULTPOINTS;
    ret              = 1;
  }

  for (int i = 0; i < Common::PLAYERS_PER_GAME; i++) {
    if (itsLevel[i] < EASY || itsLevel[i] > HARD) {
      LOG("WARNING: bad option in preferences file!  setting itsLevel[%d] to %d\n", i, DEFAULTLEVEL);
      itsLevel[i] = DEFAULTLEVEL;
      ret         = 1;
    }

    if (itsAgg[i] < MINAGG || itsAgg[i] > MAXAGG) {
      LOG("WARNING: bad option in preferences file! setting itsAgg[%d] to %d\n", i, DEFAULTAGG);
      itsAgg[i] = DEFAULTAGG;
      ret       = 1;
    }
  }

  return ret;
}

Options::AILevel Options::getAILevel(Common::PlayerPosition pos) {
  if (pos < Common::NORTH || pos > Common::WEST) {
    return NOAILEVEL;
  }

  return (AILevel) itsLevel[pos];
}

Options::AIAgg Options::getAIAgg(Common::PlayerPosition pos) {
  if (pos < Common::NORTH || pos > Common::WEST) {
    return NOAIAGG;
  }

  return (AIAgg) itsAgg[pos];
}

Options* Options::get() {
  return itsOptions;
}

void Options::set(Options* opt) {
  itsOptions = opt;
}

