/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoClientException;
import com.mongodb.MongoException;
import com.mongodb.MongoIncompatibleDriverException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.ServerAddress;
import com.mongodb.UnixServerAddress;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ClusterDescription;
import com.mongodb.connection.ClusterId;
import com.mongodb.connection.ClusterSettings;
import com.mongodb.connection.ClusterType;
import com.mongodb.connection.ServerDescription;
import com.mongodb.event.ClusterClosedEvent;
import com.mongodb.event.ClusterDescriptionChangedEvent;
import com.mongodb.event.ClusterListener;
import com.mongodb.event.ClusterOpeningEvent;
import com.mongodb.internal.Locks;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.Cluster;
import com.mongodb.internal.connection.ClusterClock;
import com.mongodb.internal.connection.ClusterableServer;
import com.mongodb.internal.connection.ClusterableServerFactory;
import com.mongodb.internal.connection.EventHelper;
import com.mongodb.internal.connection.OperationContext;
import com.mongodb.internal.connection.Server;
import com.mongodb.internal.connection.ServerTuple;
import com.mongodb.internal.diagnostics.logging.Logger;
import com.mongodb.internal.diagnostics.logging.Loggers;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.logging.LogMessage;
import com.mongodb.internal.logging.StructuredLogger;
import com.mongodb.internal.selector.LatencyMinimizingServerSelector;
import com.mongodb.internal.thread.InterruptionUtil;
import com.mongodb.lang.Nullable;
import com.mongodb.selector.CompositeServerSelector;
import com.mongodb.selector.ServerSelector;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;

abstract class BaseCluster
implements Cluster {
    private static final Logger LOGGER = Loggers.getLogger("cluster");
    private static final StructuredLogger STRUCTURED_LOGGER = new StructuredLogger("cluster");
    private final ReentrantLock lock = new ReentrantLock();
    private final AtomicReference<CountDownLatch> phase = new AtomicReference<CountDownLatch>(new CountDownLatch(1));
    private final ClusterableServerFactory serverFactory;
    private final ClusterId clusterId;
    private final ClusterSettings settings;
    private final ClusterListener clusterListener;
    private final Deque<ServerSelectionRequest> waitQueue = new ConcurrentLinkedDeque<ServerSelectionRequest>();
    private final ClusterClock clusterClock = new ClusterClock();
    private Thread waitQueueHandler;
    private volatile boolean isClosed;
    private volatile ClusterDescription description;

    BaseCluster(ClusterId clusterId, ClusterSettings settings, ClusterableServerFactory serverFactory) {
        this.clusterId = Assertions.notNull("clusterId", clusterId);
        this.settings = Assertions.notNull("settings", settings);
        this.serverFactory = Assertions.notNull("serverFactory", serverFactory);
        this.clusterListener = EventListenerHelper.singleClusterListener(settings);
        this.clusterListener.clusterOpening(new ClusterOpeningEvent(clusterId));
        this.description = new ClusterDescription(settings.getMode(), ClusterType.UNKNOWN, Collections.emptyList(), settings, serverFactory.getSettings());
    }

    @Override
    public ClusterClock getClock() {
        return this.clusterClock;
    }

    @Override
    public ServerTuple selectServer(ServerSelector serverSelector, OperationContext operationContext) {
        Assertions.isTrue("open", !this.isClosed());
        try {
            long startTimeNanos;
            CountDownLatch currentPhase = this.phase.get();
            ClusterDescription curDescription = this.description;
            BaseCluster.logServerSelectionStarted(this.clusterId, operationContext, serverSelector, curDescription);
            ServerSelector compositeServerSelector = this.getCompositeServerSelector(serverSelector);
            ServerTuple serverTuple = this.selectServer(compositeServerSelector, curDescription);
            boolean selectionWaitingLogged = false;
            long curTimeNanos = startTimeNanos = System.nanoTime();
            Long maxWaitTimeNanos = this.getMaxWaitTimeNanos();
            while (true) {
                Long remainingTimeNanos;
                if (!curDescription.isCompatibleWithDriver()) {
                    throw this.createAndLogIncompatibleException(operationContext, serverSelector, curDescription);
                }
                if (serverTuple != null) {
                    BaseCluster.logServerSelectionSucceeded(this.clusterId, operationContext, serverTuple.getServerDescription().getAddress(), serverSelector, curDescription);
                    return serverTuple;
                }
                Long l = remainingTimeNanos = maxWaitTimeNanos == null ? null : Long.valueOf(maxWaitTimeNanos - (curTimeNanos - startTimeNanos));
                if (remainingTimeNanos != null && remainingTimeNanos <= 0L) {
                    throw this.createAndLogTimeoutException(operationContext, serverSelector, curDescription);
                }
                if (!selectionWaitingLogged) {
                    BaseCluster.logServerSelectionWaiting(this.clusterId, operationContext, remainingTimeNanos, serverSelector, curDescription);
                    selectionWaitingLogged = true;
                }
                this.connect();
                currentPhase.await(remainingTimeNanos == null ? this.getMinWaitTimeNanos() : Math.min(remainingTimeNanos, this.getMinWaitTimeNanos()), TimeUnit.NANOSECONDS);
                curTimeNanos = System.nanoTime();
                currentPhase = this.phase.get();
                curDescription = this.description;
                serverTuple = this.selectServer(compositeServerSelector, curDescription);
            }
        }
        catch (InterruptedException e) {
            throw InterruptionUtil.interruptAndCreateMongoInterruptedException(String.format("Interrupted while waiting for a server that matches %s", serverSelector), e);
        }
    }

    @Override
    public void selectServerAsync(ServerSelector serverSelector, OperationContext operationContext, SingleResultCallback<ServerTuple> callback) {
        Assertions.isTrue("open", !this.isClosed());
        CountDownLatch currentPhase = this.phase.get();
        ClusterDescription currentDescription = this.description;
        BaseCluster.logServerSelectionStarted(this.clusterId, operationContext, serverSelector, currentDescription);
        ServerSelectionRequest request = new ServerSelectionRequest(operationContext, serverSelector, this.getCompositeServerSelector(serverSelector), this.getMaxWaitTimeNanos(), callback);
        if (!this.handleServerSelectionRequest(request, currentPhase, currentDescription)) {
            this.notifyWaitQueueHandler(request);
        }
    }

    @Override
    public ClusterId getClusterId() {
        return this.clusterId;
    }

    @Override
    public ClusterSettings getSettings() {
        return this.settings;
    }

    public ClusterableServerFactory getServerFactory() {
        return this.serverFactory;
    }

    protected abstract void connect();

    @Override
    public void close() {
        if (!this.isClosed()) {
            this.isClosed = true;
            this.phase.get().countDown();
            this.clusterListener.clusterClosed(new ClusterClosedEvent(this.clusterId));
            this.stopWaitQueueHandler();
        }
    }

    @Override
    public boolean isClosed() {
        return this.isClosed;
    }

    protected void updateDescription(ClusterDescription newDescription) {
        this.withLock(() -> {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(String.format("Updating cluster description to  %s", newDescription.getShortDescription()));
            }
            this.description = newDescription;
            this.updatePhase();
        });
    }

    protected void fireChangeEvent(ClusterDescription newDescription, ClusterDescription previousDescription) {
        if (!EventHelper.wouldDescriptionsGenerateEquivalentEvents(newDescription, previousDescription)) {
            this.clusterListener.clusterDescriptionChanged(new ClusterDescriptionChangedEvent(this.getClusterId(), newDescription, previousDescription));
        }
    }

    @Override
    public ClusterDescription getCurrentDescription() {
        return this.description;
    }

    @Override
    public void withLock(Runnable action) {
        Locks.withInterruptibleLock((Lock)this.lock, action);
    }

    private void updatePhase() {
        this.withLock(() -> this.phase.getAndSet(new CountDownLatch(1)).countDown());
    }

    @Nullable
    private Long getMaxWaitTimeNanos() {
        if (this.settings.getServerSelectionTimeout(TimeUnit.NANOSECONDS) < 0L) {
            return null;
        }
        return this.settings.getServerSelectionTimeout(TimeUnit.NANOSECONDS);
    }

    private long getMinWaitTimeNanos() {
        return this.serverFactory.getSettings().getMinHeartbeatFrequency(TimeUnit.NANOSECONDS);
    }

    private boolean handleServerSelectionRequest(ServerSelectionRequest request, CountDownLatch currentPhase, ClusterDescription description) {
        try {
            if (currentPhase != request.phase) {
                CountDownLatch prevPhase = request.phase;
                request.phase = currentPhase;
                if (!description.isCompatibleWithDriver()) {
                    request.onResult(null, this.createAndLogIncompatibleException(request.operationContext, request.originalSelector, description));
                    return true;
                }
                ServerTuple serverTuple = this.selectServer(request.compositeSelector, description);
                if (serverTuple != null) {
                    BaseCluster.logServerSelectionSucceeded(this.clusterId, request.operationContext, serverTuple.getServerDescription().getAddress(), request.originalSelector, description);
                    request.onResult(serverTuple, null);
                    return true;
                }
                if (prevPhase == null) {
                    BaseCluster.logServerSelectionWaiting(this.clusterId, request.operationContext, request.getRemainingTime(), request.originalSelector, description);
                }
            }
            if (request.timedOut()) {
                request.onResult(null, this.createAndLogTimeoutException(request.operationContext, request.originalSelector, description));
                return true;
            }
            return false;
        }
        catch (Exception e) {
            request.onResult(null, e);
            return true;
        }
    }

    @Nullable
    private ServerTuple selectServer(ServerSelector serverSelector, ClusterDescription clusterDescription) {
        return BaseCluster.selectServer(serverSelector, clusterDescription, this::getServer);
    }

    @Nullable
    static ServerTuple selectServer(ServerSelector serverSelector, ClusterDescription clusterDescription, Function<ServerAddress, Server> serverCatalog) {
        return BaseCluster.atMostNRandom(new ArrayList<ServerDescription>(serverSelector.select(clusterDescription)), 2, serverDescription -> {
            Server server = (Server)serverCatalog.apply(serverDescription.getAddress());
            return server == null ? null : new ServerTuple(server, (ServerDescription)serverDescription);
        }).stream().min(Comparator.comparingInt(serverTuple -> serverTuple.getServer().operationCount())).orElse(null);
    }

    private static List<ServerTuple> atMostNRandom(ArrayList<ServerDescription> list, int n, Function<ServerDescription, ServerTuple> transformer) {
        ThreadLocalRandom random = ThreadLocalRandom.current();
        ArrayList<ServerTuple> result = new ArrayList<ServerTuple>(n);
        for (int i = list.size() - 1; i >= 0 && result.size() < n; --i) {
            Collections.swap(list, i, random.nextInt(i + 1));
            ServerTuple serverTuple = transformer.apply(list.get(i));
            if (serverTuple == null) continue;
            result.add(serverTuple);
        }
        return result;
    }

    private ServerSelector getCompositeServerSelector(ServerSelector serverSelector) {
        LatencyMinimizingServerSelector latencyMinimizingServerSelector = new LatencyMinimizingServerSelector(this.settings.getLocalThreshold(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
        if (this.settings.getServerSelector() == null) {
            return new CompositeServerSelector(Arrays.asList(serverSelector, latencyMinimizingServerSelector));
        }
        return new CompositeServerSelector(Arrays.asList(serverSelector, this.settings.getServerSelector(), latencyMinimizingServerSelector));
    }

    protected ClusterableServer createServer(ServerAddress serverAddress) {
        return this.serverFactory.create(this, serverAddress);
    }

    private MongoIncompatibleDriverException createAndLogIncompatibleException(OperationContext operationContext, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        MongoIncompatibleDriverException exception = this.createIncompatibleException(clusterDescription);
        BaseCluster.logServerSelectionFailed(this.clusterId, operationContext, exception, serverSelector, clusterDescription);
        return exception;
    }

    private MongoIncompatibleDriverException createIncompatibleException(ClusterDescription curDescription) {
        String message;
        ServerDescription incompatibleServer = curDescription.findServerIncompatiblyOlderThanDriver();
        if (incompatibleServer != null) {
            message = String.format("Server at %s reports wire version %d, but this version of the driver requires at least %d (MongoDB %s).", incompatibleServer.getAddress(), incompatibleServer.getMaxWireVersion(), 6, "3.6");
        } else {
            incompatibleServer = curDescription.findServerIncompatiblyNewerThanDriver();
            if (incompatibleServer != null) {
                message = String.format("Server at %s requires wire version %d, but this version of the driver only supports up to %d.", incompatibleServer.getAddress(), incompatibleServer.getMinWireVersion(), 21);
            } else {
                throw new IllegalStateException("Server can't be both older than the driver and newer.");
            }
        }
        return new MongoIncompatibleDriverException(message, curDescription);
    }

    private MongoException createAndLogTimeoutException(OperationContext operationContext, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        MongoTimeoutException exception = new MongoTimeoutException(String.format("Timed out while waiting for a server that matches %s. Client view of cluster state is %s", serverSelector, clusterDescription.getShortDescription()));
        BaseCluster.logServerSelectionFailed(this.clusterId, operationContext, exception, serverSelector, clusterDescription);
        return exception;
    }

    private void notifyWaitQueueHandler(ServerSelectionRequest request) {
        this.withLock(() -> {
            if (this.isClosed) {
                return;
            }
            this.waitQueue.add(request);
            if (this.waitQueueHandler == null) {
                this.waitQueueHandler = new Thread((Runnable)new WaitQueueHandler(), "cluster-" + this.clusterId.getValue());
                this.waitQueueHandler.setDaemon(true);
                this.waitQueueHandler.start();
            } else {
                this.updatePhase();
            }
        });
    }

    private void stopWaitQueueHandler() {
        this.withLock(() -> {
            if (this.waitQueueHandler != null) {
                this.waitQueueHandler.interrupt();
            }
        });
    }

    static void logServerSelectionStarted(ClusterId clusterId, OperationContext operationContext, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.DEBUG, "Server selection started", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationContext.getId()), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Server selection started for operation[ {}] with ID {}. Selector: {}, topology description: {}"));
        }
    }

    private static void logServerSelectionWaiting(ClusterId clusterId, OperationContext operationContext, @Nullable Long remainingTimeNanos, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.INFO, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.INFO, "Waiting for suitable server to become available", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationContext.getId()), new LogMessage.Entry(LogMessage.Entry.Name.REMAINING_TIME_MS, remainingTimeNanos == null ? null : Long.valueOf(TimeUnit.NANOSECONDS.toMillis(remainingTimeNanos))), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Waiting for server to become available for operation[ {}] with ID {}.[ Remaining time: {} ms.] Selector: {}, topology description: {}."));
        }
    }

    private static void logServerSelectionFailed(ClusterId clusterId, OperationContext operationContext, MongoException failure, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            String failureDescription = failure instanceof MongoTimeoutException ? MongoTimeoutException.class.getName() + ": Timed out while waiting for a suitable server" : failure.toString();
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.DEBUG, "Server selection failed", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationContext.getId()), new LogMessage.Entry(LogMessage.Entry.Name.FAILURE, failureDescription), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Server selection failed for operation[ {}] with ID {}. Failure: {}. Selector: {}, topology description: {}"));
        }
    }

    static void logServerSelectionSucceeded(ClusterId clusterId, OperationContext operationContext, ServerAddress serverAddress, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.DEBUG, "Server selection succeeded", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationContext.getId()), new LogMessage.Entry(LogMessage.Entry.Name.SERVER_HOST, serverAddress.getHost()), new LogMessage.Entry(LogMessage.Entry.Name.SERVER_PORT, serverAddress instanceof UnixServerAddress ? null : Integer.valueOf(serverAddress.getPort())), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Server selection succeeded for operation[ {}] with ID {}. Selected server: {}[:{}]. Selector: {}, topology description: {}"));
        }
    }

    private static final class ServerSelectionRequest {
        private final OperationContext operationContext;
        private final ServerSelector originalSelector;
        private final ServerSelector compositeSelector;
        @Nullable
        private final Long maxWaitTimeNanos;
        private final SingleResultCallback<ServerTuple> callback;
        private final long startTimeNanos = System.nanoTime();
        private CountDownLatch phase;

        ServerSelectionRequest(OperationContext operationContext, ServerSelector serverSelector, ServerSelector compositeSelector, @Nullable Long maxWaitTimeNanos, SingleResultCallback<ServerTuple> callback) {
            this.operationContext = operationContext;
            this.originalSelector = serverSelector;
            this.compositeSelector = compositeSelector;
            this.maxWaitTimeNanos = maxWaitTimeNanos;
            this.callback = callback;
        }

        void onResult(@Nullable ServerTuple serverTuple, @Nullable Throwable t) {
            try {
                this.callback.onResult(serverTuple, t);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }

        boolean timedOut() {
            Long remainingTimeNanos = this.getRemainingTime();
            return remainingTimeNanos != null && remainingTimeNanos <= 0L;
        }

        @Nullable
        Long getRemainingTime() {
            return this.maxWaitTimeNanos == null ? null : Long.valueOf(this.maxWaitTimeNanos - (System.nanoTime() - this.startTimeNanos));
        }
    }

    private final class WaitQueueHandler
    implements Runnable {
        private WaitQueueHandler() {
        }

        @Override
        public void run() {
            while (!BaseCluster.this.isClosed) {
                CountDownLatch currentPhase = (CountDownLatch)BaseCluster.this.phase.get();
                ClusterDescription curDescription = BaseCluster.this.description;
                long waitTimeNanos = Long.MAX_VALUE;
                Iterator iter = BaseCluster.this.waitQueue.iterator();
                while (iter.hasNext()) {
                    ServerSelectionRequest nextRequest = (ServerSelectionRequest)iter.next();
                    if (BaseCluster.this.handleServerSelectionRequest(nextRequest, currentPhase, curDescription)) {
                        iter.remove();
                        continue;
                    }
                    Long remainingTimeNanos = nextRequest.getRemainingTime();
                    long minWaitTimeNanos = Math.min(BaseCluster.this.getMinWaitTimeNanos(), waitTimeNanos);
                    waitTimeNanos = remainingTimeNanos == null ? minWaitTimeNanos : Math.min(remainingTimeNanos, minWaitTimeNanos);
                }
                if (waitTimeNanos < Long.MAX_VALUE) {
                    BaseCluster.this.connect();
                }
                try {
                    currentPhase.await(waitTimeNanos, TimeUnit.NANOSECONDS);
                }
                catch (InterruptedException interruptedException) {}
            }
            Iterator iter = BaseCluster.this.waitQueue.iterator();
            while (iter.hasNext()) {
                ((ServerSelectionRequest)iter.next()).onResult(null, new MongoClientException("Shutdown in progress"));
                iter.remove();
            }
        }
    }
}

