from sys import stdout
from weakref import ref as weakref_ref
from logging import (basicConfig, getLogger, StreamHandler, Formatter,
    DEBUG, INFO, WARNING, ERROR)

LOG_FILENAME = 'fusil.log'

class ApplicationLogger:
    def __init__(self, application):
        self.application = weakref_ref(application)
        self.timestamp_format = '%(asctime)s: %(message)s'

        # Choose log levels
        if application.options.debug:
            stdout_level = INFO
            file_level = DEBUG
        elif application.options.verbose:
            stdout_level = WARNING
            file_level = INFO
        elif not application.options.quiet:
            stdout_level = ERROR
            file_level = WARNING
        else:
            stdout_level = ERROR
            file_level = None

        # fusil.log file
        if not application.options.quiet:
            basicConfig(level=file_level,
                        format=self.timestamp_format,
                        filename=application.options.log_filename,
                        filemode='w')
        self.logger = getLogger()

        # Create stdout logger
        handler = StreamHandler(stdout)
        handler.setLevel(stdout_level)

        self.logger.addHandler(handler)

    def addFileHandler(self, filename, level=None):
        if level is None:
            if self.application().options.verbose:
                level = DEBUG
            else:
                level = INFO
        handler = StreamHandler(open(filename, 'w'))
        handler.setLevel(level)
        handler.setFormatter(Formatter(self.timestamp_format))
        self.logger.addHandler(handler)
        return handler

    def removeHandler(self, handler):
        handler.stream.close()
        handler.close()
        self.logger.removeHandler(handler)

    def formatMessage(self, message, sender):
        message = str(message)
        application = self.application()
        prefix = []
        if application and application.project and application.project.session:
            project = application.project
            prefix.append('[session %s]' % project.session_index)
            if application.options.verbose and project.step:
                prefix.append('[step %s]' % project.step)
        if sender is not None:
            prefix.append('[%s]' % sender.name)
        if prefix:
            message = ''.join(prefix)+' '+message
        return message

    def debug(self, message, sender):
        self.log(self.logger.debug, message, sender)

    def info(self, message, sender):
        self.log(self.logger.info, message, sender)

    def warning(self, message, sender):
        self.log(self.logger.warning, message, sender)

    def error(self, message, sender):
        self.log(self.logger.error, message, sender)

    def log(self, func, message, sender):
        message = self.formatMessage(message, sender)
        func(message)

