from os import readlink, listdir
from os.path import join as path_join
from errno import EACCES, ENOENT
from resource import getpagesize
from fusil.tools import timestampUNIX
from datetime import datetime, timedelta

class ProcError(Exception):
    pass

PROC_DIRNAME = '/proc'
PAGE_SIZE = getpagesize()

def procFilename(*args):
    return path_join(PROC_DIRNAME, *args)

def openProc(path):
    try:
        filename = procFilename(path)
        return open(filename)
    except IOError, err:
        raise ProcError("Unable to read %r: %s" % (filename, err))

def readProcessProc(pid, key):
    try:
        return openProc(path_join(str(pid), str(key))).read()
    except ProcError, error:
        raise ProcError("Process %s doesn't exist: %s" % (
            pid, error))

def readProcessStatm(pid):
    statm = readProcessProc(pid, 'statm')
    statm = [ int(item)*PAGE_SIZE for item in statm.split() ]
    return statm

def readProcessProcList(pid, key):
    data = readProcessProc(pid, key)
    if not data:
        # Empty file: empty list
        return []
    data = data.split("\0")
    if not data[-1]:
        del data[-1]
    return data

def readProcessLink(pid, key):
    try:
        filename = procFilename(str(pid), str(key))
        return readlink(filename)
    except OSError, err:
        raise ProcError("Unable to read proc link %r: %s" % (filename, err))

def readProcesses():
    """
    Iterate on process directories from /proc
    """
    for filename in listdir(PROC_DIRNAME):
        try:
            yield int(filename)
        except ValueError:
            # Filename is not an integer (eg. "stat" from /proc/stat)
            continue

def readProcessCmdline(pid, escape_stat=True):
    # Try /proc/42/cmdline
    try:
        cmdline = readProcessProcList(pid, 'cmdline')
        if cmdline:
            return cmdline
    except ProcError:
        pass

    # Try /proc/42/stat
    try:
        stat = readProcessProc(pid, 'stat')
        # "2 (ksoftirqd/0) S 1 1 1 0 ..." => extract "ksoftirqd/0"
        stat = stat.rsplit(')', 1)[0]
        stat = stat.split('(', 1)[1]
        if escape_stat:
            stat = "[%s]" % stat
        return [stat]
    except ProcError:
        return None

def searchProcessByName(process_name):
    """
    Find process identifier (PID) using its name
    (eg. "ssh" to find "/usr/bin/ssh").
    """
    suffix = '/'+process_name
    for pid in readProcesses():
        cmdline = readProcessCmdline(pid)
        if not cmdline:
            continue
        program = cmdline[0]
        if program == process_name or program.endswith(suffix):
            return pid
    raise ProcError("Unable to find process: %r" % process_name)

def getUptime():
    """
    Get system uptime: return datetime.timedelta object.
    """
    uptime = openProc('uptime').read()
    uptime = uptime.strip().split()
    uptime = float(uptime[0])
    return timedelta(seconds=uptime)

def getSystemBoot():
    """
    Get system boot date, return datetime.datetime object.
    """
    if getSystemBoot.value is None:
        for line in openProc('stat'):
            if not line.startswith("btime "):
                continue
            seconds = int(line[6:])
            btime = timestampUNIX(seconds, True)
            getSystemBoot.value = btime
        if getSystemBoot.value is None:
            raise ProcError("Unable to read system boot time!")
    return getSystemBoot.value
getSystemBoot.value = None

