from itertools import izip
from datetime import datetime, timedelta

LOCAL_TIMEZONE_OFFSET = datetime.fromtimestamp(0) - datetime.utcfromtimestamp(0)

def listDiff(old, new):
    """
    Difference of two lists item by item.

    >>> listDiff([4, 0, 3], [10, 0, 50])
    [6, 0, 47]
    """
    return [ item[1]-item[0] for item in izip(old, new) ]

# Start of UNIX timestamp (Epoch): 1st January 1970 at 00:00
UNIX_TIMESTAMP_T0 = datetime(1970, 1, 1)

def timestampUNIX(value, is_local):
    """
    Convert an UNIX (32-bit) timestamp to datetime object. Timestamp value
    is the number of seconds since the 1st January 1970 at 00:00. Maximum
    value is 2147483647: 19 january 2038 at 03:14:07.

    May raise ValueError for invalid value: value have to be in 0..2147483647.

    >>> timestampUNIX(0, False)
    datetime.datetime(1970, 1, 1, 0, 0)
    >>> timestampUNIX(1154175644.37, False)
    datetime.datetime(2006, 7, 29, 12, 20, 44, 370000)
    """
    timestamp = UNIX_TIMESTAMP_T0 + timedelta(seconds=value)
    if is_local:
        timestamp += LOCAL_TIMEZONE_OFFSET
    return timestamp

def timedeltaSeconds(delta):
    """
    Convert a datetime.timedelta() objet to a number of second
    (floatting point number).

    >>> timedeltaSeconds(timedelta(seconds=2, microseconds=40000))
    2.04
    >>> timedeltaSeconds(timedelta(minutes=1, milliseconds=250))
    60.25
    """
    return delta.microseconds / 1000000.0 + delta.seconds \
        + delta.days * 3600 * 24

def minmax(min_value, value, max_value):
    """
    Restrict value to [min_value; max_value]

    >>> minmax(-2, -3, 10)
    -2
    >>> minmax(-2, 27, 10)
    10
    >>> minmax(-2, 0, 10)
    0
    """
    return min(max(min_value, value), max_value)

