/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001-2005 National * Institute of Water and
 * Atmospheric Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include "config.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <getopt.h>
#include <gfs.h>

#include <GL/osmesa.h>
#include <GL/glu.h>

#include "gl/gfsgl.h"
#include "gl/trackball.h"

static void view_draw (GfsGlViewParams * view,
		       GfsGl2PSParams * p,
		       GfsDomain * domain,
		       GList * i,
		       guint width, guint height)
{
  GLfloat m[4][4];
  gdouble max;

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  max = gfs_gl_domain_extent (domain);
  gluPerspective (view->fov, width/(float)height, 1., 1. + 2.*max);
  glMatrixMode (GL_MODELVIEW);
	    
  glLoadIdentity ();
  glTranslatef (view->tx, view->ty, - (1. + max));
  gfs_gl_build_rotmatrix (m, view->quat);
  glMultMatrixf (&m[0][0]);
	    
  glClearColor (view->bg.r, view->bg.g, view->bg.b, 1);
  glClear (GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	    
  while (i) {
    GfsGl * gl = i->data;
    gl->format = p->format;
    gfs_gl_draw (gl);
    i = i->next;
  }
	    
  glFinish ();
}

static void read_commands (GtsFile * fp, 
			   GList ** list, 
			   GfsGlViewParams * view, 
			   GfsSimulation * sim)
{
  while (fp->type == GTS_STRING)
    if (!strcmp (fp->token->str, "View"))
      gfs_gl_view_params_read (view, fp);
    else if (!strcmp (fp->token->str, "Clear")) {
      g_list_foreach (*list, (GFunc) gts_object_destroy, NULL);
      g_list_free (*list);
      *list = NULL;
      gts_file_next_token (fp);
    }
    else if (!strcmp (fp->token->str, "Save")) {
      GfsGl2PSParams p;
      FILE * fptr;

      gts_file_next_token (fp);
      if (fp->type != GTS_STRING) {
	gts_file_error (fp, "expecting a string (filename)");
	break;
      }
      fptr = (!strcmp (fp->token->str, "stdout") ? stdout :
	      !strcmp (fp->token->str, "stderr") ? stderr :
	      fopen (fp->token->str, "w"));
      if (fptr == NULL) {
	gts_file_error (fp, "cannot open file `%s'", fp->token->str);
	break;
      }
      gts_file_next_token (fp);
      gfs_gl2ps_params_read (&p, fp);
      if (fp->type != GTS_ERROR) {
	OSMesaContext ctx;
	guint width = p.width > 0 ? p.width : 640;
	guint height = p.height > 0 ? p.height : 480;
	void * image = g_malloc (width*height*4*sizeof (GLubyte));
	    
	/* Create an RGBA-mode context for OSMesa */
#if OSMESA_MAJOR_VERSION * 100 + OSMESA_MINOR_VERSION >= 305
	/* specify Z, stencil, accum sizes */
	ctx = OSMesaCreateContextExt (OSMESA_RGBA, 16, 0, 0, NULL);
#else
	ctx = OSMesaCreateContext (OSMESA_RGBA, NULL);
#endif
	if (!ctx) {
	  fprintf (stderr, "gfsview-batch: OSMesaCreateContext failed!\n");
	  exit (1);
	}

	if (!OSMesaMakeCurrent (ctx, image, GL_UNSIGNED_BYTE, width, height)) {
	  fprintf (stderr, "gfsview-batch: OSMesaMakeCurrent failed!\n");
	  exit (1);
	}
	
	gfs_gl_init_gl ();

	switch (p.format) {
	case GFSGL_PPM_OFFSCREEN: case GFSGL_PPM_SCREEN: {
	  view_draw (view, &p, GFS_DOMAIN (sim), *list, width, height);
	  gfs_gl_write_image (fptr, image, width, height);
	  break;
	}
	case GFSGL_GNUPLOT : {
	  guint buffsize = 0;
	  gboolean done = FALSE;

	  while (!done) {
	    GfsGlFeedback * f;

	    buffsize += 2048*2048;
	    f = gfs_gl_feedback_begin (buffsize);
	    view_draw (view, &p, GFS_DOMAIN (sim), *list, width, height);
	    done = gfs_gl_feedback_end (f, fptr);
	  }
	  break;
	}
	default: {
	  GLint buffsize = 0, state = GL2PS_OVERFLOW;

	  while (state == GL2PS_OVERFLOW) {
	    buffsize += 2048*2048;
	    gl2psBeginPage ("", "GfsView",
			    NULL,
			    p.format, p.sort, p.options, 
			    GL_RGBA, 0, NULL, 
			    0, 0, 0,
			    buffsize, fptr, "");
	    gl2psLineWidth (p.lw);
	    view_draw (view, &p, GFS_DOMAIN (sim), *list, width, height);
	    state = gl2psEndPage();
	  }
	}
	}
	    
	g_free (image);
	fflush (fptr);

	OSMesaDestroyContext (ctx);
      }
      if (fptr != stdout && fptr != stderr)
	fclose (fptr);
    }
    else { /* GfsGl objects */
      GfsGl * gl;

      if ((gl = gfs_gl_new_from_file (fp))) {
	gl->p = view;
	if (sim)
	  gfs_gl_set_simulation (gl, sim);
	*list = g_list_append (*list, gl);
      }
    }
}

static GfsSimulation * read_simulation (GtsFile * fp, GList * list)
{
  GfsSimulation * sim = gfs_simulation_read (fp);
  GSList * i;
  
  if (sim == NULL)
    return NULL;
  
  i = GFS_DOMAIN (sim)->variables;
  while (i) {
    GfsVariable * v = i->data;

    gfs_domain_cell_traverse (GFS_DOMAIN (sim),
			      FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			      (FttCellTraverseFunc) v->fine_coarse, v);
    gfs_domain_bc (GFS_DOMAIN (sim), FTT_TRAVERSE_LEAFS, -1, v);
    i = i->next;
  }
  
  g_list_foreach (list, (GFunc) gfs_gl_set_simulation, sim);
  return sim;
}

static FILE * open_pipe (const gchar * fname)
{
  gchar pname[] = "/tmp/gfsview.XXXXXX";
  FILE * fptr = fopen (fname, "r");
  gchar * script;
  gint fd;

  if (!fptr)
    return NULL;
  fclose (fptr);

  fd = mkstemp (pname);
  if (fd < 0) {
    g_warning ("cannot create temporary file");
    return NULL;
  }
  close (fd);
  remove (pname);
  if (mkfifo (pname, S_IWUSR|S_IRUSR)) {
    g_warning ("cannot create named pipe: %s", strerror (errno));
    return NULL;
  }
  script = g_strconcat ("if gunzip -t ", fname, " 2> /dev/null; then "
			"  gunzip -c ", fname, " 2> /dev/null > ", pname, "; else ",
			"  cat ", fname, " > ", pname, "; fi &", NULL);
  system (script);
  g_free (script);
  fptr = fopen (pname, "r");
  unlink (pname);
  return fptr;
}

int main (int argc, char * argv[])
{
  GfsGlViewParams view;
  GfsSimulation * sim = NULL;
  GList * list = NULL;
  int c = 0;

  /* initialize gfs */
  gfs_init (&argc, &argv);

  /* initialize gfsgl */
  gfs_gl_init ();

  /* options */
  while (c != EOF) {
    static struct option long_options[] = {
      {"survive-broken-pipe", no_argument, NULL, 's'},
      {"help", no_argument, NULL, 'h'},
      {"version", no_argument, NULL, 'V'}
    };
    int option_index = 0;
    switch ((c = getopt_long (argc, argv, "hVs", long_options, &option_index))) {
    case 's':
      /* These options are ignored, they are here for command-line compatibility with
	 GfsView interactive version */
      break;
    case 'h': /* help */
      fprintf (stderr,
             "Usage: gfsview-batch [OPTION] FILE1 FILE2 ...\n"
	     "The Gerris flow solver visualisation tool (batch mode).\n"
	     "\n"
	     "  -h    --help                display this help and exit\n"
	     "  -V    --version             output version information and exit\n"
	     "\n"
	     "Reports bugs to %s\n",
	     FTT_MAINTAINER);
      return 0; /* success */
      break;
    case 'V': /* version */
      fprintf (stderr,
	       "gfsview-batch: using %dD libgfs version %s\n",
	       FTT_DIMENSION, VERSION);
      return 0; /* succes */
      break;
    case '?': /* wrong options */
      fprintf (stderr, "Try `gfsview-batch --help' for more information.\n");
      return 1; /* failure */
    }
  }

  gfs_gl_view_params_init (&view);
  
  /* read files on command line */
  for (c = optind; c < argc; c++) {
    FILE * fptr = open_pipe (argv[c]);
    GtsFile * fp;

    if (fptr == NULL) {
      fprintf (stderr, "gfsview-batch: cannot open file `%s'\n", argv[c]);
      return 1;
    }
    fp = gts_file_new (fptr);
    while (fp->type != GTS_ERROR && !feof (fptr)) {
      if (fp->type == '\n')
	gts_file_next_token (fp);
      else if (fp->type == GTS_INT) {
	GfsSimulation * sim1 = read_simulation (fp, list);
	if (sim1) {
	  if (sim)
	    gts_object_destroy (GTS_OBJECT (sim));
	  sim = sim1;
	}
      }
      else if (fp->type == GTS_STRING)
	read_commands (fp, &list, &view, sim);
      else
	gts_file_error (fp, "expecting an integer got %d", fp->type);
    }
    if (fp->type == GTS_ERROR) {
      fprintf (stderr, "gfsview-batch: %s:%d:%d: %s\n", argv[c], fp->line, fp->pos, fp->error);
      return 1;
    }
    gts_file_destroy (fp);
    fclose (fptr);
  }

  /* wait for parameters/simulations on standard input */
  while (1) {
    GtsFile * fp = gts_file_new (stdin);

    while (fp->type != GTS_ERROR)
      if (feof (stdin))
	return 0;
      else if (fp->type == '\n')
	gts_file_next_token (fp);
      else if (fp->type == GTS_INT) {
	GfsSimulation * sim1 = read_simulation (fp, list);
	if (sim1) {
	  if (sim)
	    gts_object_destroy (GTS_OBJECT (sim));
	  sim = sim1;	  
	}
      }
      else if (fp->type == GTS_STRING)
	read_commands (fp, &list, &view, sim);
      else
	gts_file_error (fp, "expecting an integer got %d", fp->type);

    fprintf (stderr, "gfsview-batch: <stdin>:%d:%d: %s\n", fp->line, fp->pos, fp->error);
    return 1;
  }
  
  return 0;
}
