/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001-2004 National Institute of Water and Atmospheric
 * Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#include "gfsgl.h"
#include "trackball.h"

#ifndef GL_CONSTANT
# define GL_CONSTANT GL_CONSTANT_EXT
#endif

#if FTT_2D
# include "gfsgl2D.h"
#else  /* 3D */
# include "gfsgl3D.h"
#endif /* 3D */

#include <GL/glu.h>

static GtsColor * color_new (gdouble r, gdouble g, gdouble b)
{
  GtsColor * c = g_malloc (sizeof (GtsColor));
  c->r = r; c->g = g; c->b = b;
  return c;
}

static void color_destroy (GtsColor * color)
{
  g_return_if_fail (color != NULL);

  g_free (color);
}

static void colormap_set_texture (GfsColormap * cmap)
{
  guint i;

  for (i = 0; i < GFS_COLORMAP_TEXTURE_SAMPLES; i++) {
    GtsColor c = gfs_colormap_color (cmap, i/(gdouble) (GFS_COLORMAP_TEXTURE_SAMPLES - 1));
    cmap->texture[3*i] = c.r;
    cmap->texture[3*i + 1] = c.g;
    cmap->texture[3*i + 2] = c.b;
  }
}

GfsColormap * gfs_colormap_jet (void)
{
  GfsColormap * cmap = g_malloc (sizeof (GfsColormap));
  gint i;

  cmap->reversed = FALSE;
  cmap->colors = g_ptr_array_new ();
  for (i = 0; i < 127; i++) {
    gdouble r = 
      i <= 46 ? 0. : 
      i >= 111 ? -0.03125*(i - 111) + 1. :
      i >= 78 ? 1. : 
      0.03125*(i - 46);
    gdouble g = 
      i <= 14 || i >= 111 ? 0. : 
      i >= 79 ? -0.03125*(i - 111) : 
      i <= 46 ? 0.03125*(i - 14) : 
      1.;
    gdouble b =
      i >= 79 ? 0. :
      i >= 47 ? -0.03125*(i - 79) :
      i <= 14 ? 0.03125*(i - 14) + 1.:
      1.;

    g_ptr_array_add (cmap->colors, color_new (r, g, b));
  }
  colormap_set_texture (cmap);
  return cmap;
}

void gfs_colormap_destroy (GfsColormap * colormap)
{
  guint i;

  g_return_if_fail (colormap != NULL);

  for (i = 0; i < colormap->colors->len; i++)
    color_destroy (colormap->colors->pdata[i]);
  g_ptr_array_free (colormap->colors, TRUE);
  g_free (colormap);
}

GtsColor gfs_colormap_color (GfsColormap * cmap, gdouble val)
{
  GtsColor c = {1., 1., 1.}, * c1, * c2;
  guint i, n;
  gdouble coef;

  g_return_val_if_fail (cmap != NULL, c);

  if (val > 1.0) val = 1.0;
  else if (val < 0.0) val = 0.0;
  if (cmap->reversed)
    val = 1.0 - val;

  n = cmap->colors->len;
  if (n == 0)
    return c;
  if (n == 1)
    return *((GtsColor *)cmap->colors->pdata[0]);

  i = floor ((gdouble)val*(gdouble)(n - 1));
  if (i == n - 1)
    return *((GtsColor *)cmap->colors->pdata[cmap->colors->len - 1]);
  coef = val*(gdouble)(n - 1) - (gdouble)i;
  c1 = cmap->colors->pdata[i];
  c2 = cmap->colors->pdata[i+1];
  c.r = c1->r + coef*(c2->r - c1->r);
  c.g = c1->g + coef*(c2->g - c1->g);
  c.b = c1->b + coef*(c2->b - c1->b);
  return c;
}

void gfs_colormap_texture (GfsColormap * cmap)
{
  g_return_if_fail (cmap != NULL);

  glTexImage1D (GL_TEXTURE_1D, 0, GL_RGB, GFS_COLORMAP_TEXTURE_SAMPLES, 0, GL_RGB, GL_FLOAT,
		cmap->texture);
  glTexParameteri (GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
  glTexParameteri (GL_TEXTURE_1D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
}

#define RC(r,c) m[(r)+(c)*4]
#define RCM(m,r,c) (m)[(r)+(c)*4]

static void matrix_multiply (float * m, float * n)
{
  float o[16];
  guint i;
  
  for (i = 0; i < 16; i++) o[i] = m[i];
  RC(0,0)=RCM(o,0,0)*RCM(n,0,0)+RCM(o,0,1)*RCM(n,1,0)+
          RCM(o,0,2)*RCM(n,2,0)+RCM(o,0,3)*RCM(n,3,0);
  RC(0,1)=RCM(o,0,0)*RCM(n,0,1)+RCM(o,0,1)*RCM(n,1,1)+
          RCM(o,0,2)*RCM(n,2,1)+RCM(o,0,3)*RCM(n,3,1);
  RC(0,2)=RCM(o,0,0)*RCM(n,0,2)+RCM(o,0,1)*RCM(n,1,2)+
          RCM(o,0,2)*RCM(n,2,2)+RCM(o,0,3)*RCM(n,3,2);
  RC(0,3)=RCM(o,0,0)*RCM(n,0,3)+RCM(o,0,1)*RCM(n,1,3)+
          RCM(o,0,2)*RCM(n,2,3)+RCM(o,0,3)*RCM(n,3,3);
  RC(1,0)=RCM(o,1,0)*RCM(n,0,0)+RCM(o,1,1)*RCM(n,1,0)+
          RCM(o,1,2)*RCM(n,2,0)+RCM(o,1,3)*RCM(n,3,0);
  RC(1,1)=RCM(o,1,0)*RCM(n,0,1)+RCM(o,1,1)*RCM(n,1,1)+
          RCM(o,1,2)*RCM(n,2,1)+RCM(o,1,3)*RCM(n,3,1);
  RC(1,2)=RCM(o,1,0)*RCM(n,0,2)+RCM(o,1,1)*RCM(n,1,2)+
          RCM(o,1,2)*RCM(n,2,2)+RCM(o,1,3)*RCM(n,3,2);
  RC(1,3)=RCM(o,1,0)*RCM(n,0,3)+RCM(o,1,1)*RCM(n,1,3)+
          RCM(o,1,2)*RCM(n,2,3)+RCM(o,1,3)*RCM(n,3,3);
  RC(2,0)=RCM(o,2,0)*RCM(n,0,0)+RCM(o,2,1)*RCM(n,1,0)+
          RCM(o,2,2)*RCM(n,2,0)+RCM(o,2,3)*RCM(n,3,0);
  RC(2,1)=RCM(o,2,0)*RCM(n,0,1)+RCM(o,2,1)*RCM(n,1,1)+
          RCM(o,2,2)*RCM(n,2,1)+RCM(o,2,3)*RCM(n,3,1);
  RC(2,2)=RCM(o,2,0)*RCM(n,0,2)+RCM(o,2,1)*RCM(n,1,2)+
          RCM(o,2,2)*RCM(n,2,2)+RCM(o,2,3)*RCM(n,3,2);
  RC(2,3)=RCM(o,2,0)*RCM(n,0,3)+RCM(o,2,1)*RCM(n,1,3)+
          RCM(o,2,2)*RCM(n,2,3)+RCM(o,2,3)*RCM(n,3,3);
  RC(3,0)=RCM(o,3,0)*RCM(n,0,0)+RCM(o,3,1)*RCM(n,1,0)+
          RCM(o,3,2)*RCM(n,2,0)+RCM(o,3,3)*RCM(n,3,0);
  RC(3,1)=RCM(o,3,0)*RCM(n,0,1)+RCM(o,3,1)*RCM(n,1,1)+
          RCM(o,3,2)*RCM(n,2,1)+RCM(o,3,3)*RCM(n,3,1);
  RC(3,2)=RCM(o,3,0)*RCM(n,0,2)+RCM(o,3,1)*RCM(n,1,2)+
          RCM(o,3,2)*RCM(n,2,2)+RCM(o,3,3)*RCM(n,3,2);
  RC(3,3)=RCM(o,3,0)*RCM(n,0,3)+RCM(o,3,1)*RCM(n,1,3)+
          RCM(o,3,2)*RCM(n,2,3)+RCM(o,3,3)*RCM(n,3,3);
}

static void vector_multiply (float * v, float * m)
{
  float o[4];
  guint i;
  
  for (i = 0; i < 4; i++) o[i] = v[i];
  
  v[0]=RC(0,0)*o[0]+RC(0,1)*o[1]+RC(0,2)*o[2]+RC(0,3)*o[3];
  v[1]=RC(1,0)*o[0]+RC(1,1)*o[1]+RC(1,2)*o[2]+RC(1,3)*o[3];
  v[2]=RC(2,0)*o[0]+RC(2,1)*o[1]+RC(2,2)*o[2]+RC(2,3)*o[3];
  v[3]=RC(3,0)*o[0]+RC(3,1)*o[1]+RC(3,2)*o[2]+RC(3,3)*o[3];
}

void gfs_gl_get_frustum (GfsGl * gl, GfsFrustum * f)
{
  GLint v[4];
  float p[16];
  int i;

  f->res = 2.*gl->p->res;
  glGetIntegerv (GL_VIEWPORT, v);
  f->width = v[2];
  glGetFloatv (GL_MODELVIEW_MATRIX, f->m);
  glGetFloatv (GL_PROJECTION_MATRIX, f->p);
  for (i = 0; i < 16; i++) p[i] = f->p[i];
  matrix_multiply (p, f->m);

  /* right */
  f->n[0][0] = p[3] - p[0];
  f->n[0][1] = p[7] - p[4];
  f->n[0][2] = p[11] - p[8];
  f->d[0]    = p[15] - p[12];
   
  /* left */
  f->n[1][0] = p[3] + p[0];
  f->n[1][1] = p[7] + p[4];
  f->n[1][2] = p[11] + p[8];
  f->d[1]    = p[15] + p[12];
  
  /* top */
  f->n[2][0] = p[3] - p[1];
  f->n[2][1] = p[7] - p[5];
  f->n[2][2] = p[11] - p[9];
  f->d[2]    = p[15] - p[13];

  /* bottom */
  f->n[3][0] = p[3] + p[1];
  f->n[3][1] = p[7] + p[5];
  f->n[3][2] = p[11] + p[9];
  f->d[3]    = p[15] + p[13];
  
  /* front */
  f->n[4][0] = p[3] + p[2];
  f->n[4][1] = p[7] + p[6];
  f->n[4][2] = p[11] + p[10];
  f->d[4]    = p[15] + p[14];
  
  /* back */
  f->n[5][0] = p[3] - p[2];
  f->n[5][1] = p[7] - p[6];
  f->n[5][2] = p[11] - p[10];
  f->d[5]    = p[15] - p[14];
  
  for (i = 0; i < 6; i++) {
    gdouble n = gts_vector_norm (f->n[i]);
    if (n > 0.) {
      f->n[i][0] /= n; f->n[i][1] /= n; f->n[i][2] /= n;
      f->d[i] /= n;
    }
  }
}

/**
 * gfs_sphere_in_frustum:
 * @p: the sphere center.
 * @r: the sphere radius.
 * @f: the view frustum.
 * 
 * Returns: GTS_OUT if the sphere is outside the view frustum, GTS_IN
 * if it is inside, GTS_ON if it is partly inside.
 */
GtsIntersect gfs_sphere_in_frustum (FttVector * p, gdouble r, GfsFrustum * f)
{
  guint i;

  g_return_val_if_fail (p != NULL, GTS_OUT);
  g_return_val_if_fail (f != NULL, GTS_OUT);

  for (i = 0; i < 6; i++) {
    gdouble d = f->n[i][0]*p->x + f->n[i][1]*p->y + f->n[i][2]*p->z + f->d[i];
    if (d < -r)
      return GTS_OUT;
    if (d < r)
      return GTS_ON;
  }
  return GTS_IN;
}

/**
 * gfs_sphere_size:
 * @c: the sphere center.
 * @r: the sphere radius.
 * @f: the view frustum.
 * 
 * Returns: the screen size (in pixels) of the projected sphere.
 */
gfloat gfs_sphere_size (FttVector * c, gdouble r, GfsFrustum * f)
{
  float v[4];

  g_return_val_if_fail (c != NULL, 0.);
  g_return_val_if_fail (f != NULL, 0.);

  v[0] = c->x; v[1] = c->y; v[2] = c->z; v[3] = 1.;
  vector_multiply (v, f->m);
  v[0] = r;
  vector_multiply (v, f->p);
  return v[3] == 0. ? 0 : v[0]*f->width/v[3];
}

static void cell_traverse_visible_no_check (FttCell * root,
					    GfsFrustum * f,
					    gint maxlevel,
					    FttCellTraverseFunc func,
					    gpointer data)
{
  if (FTT_CELL_IS_LEAF (root) || ftt_cell_level (root) == maxlevel)
    (* func) (root, data);
  else {
    gdouble r = ftt_cell_size (root)*DIAGONAL;
    FttVector p;
    
    ftt_cell_pos (root, &p);
    if (gfs_sphere_size (&p, r, f) < f->res)
      (* func) (root, data);
    else {
      struct _FttOct * children = root->children;
      guint n;
      
      for (n = 0; n < FTT_CELLS; n++) {
	FttCell * c = &(children->cell[n]);
	if (!FTT_CELL_IS_DESTROYED (c))
	  cell_traverse_visible_no_check (c, f, maxlevel, func, data);
      }
    }
  }
}

static void cell_traverse_visible (FttCell * root,
				   GfsFrustum * f,
				   gint maxlevel,
				   FttCellTraverseFunc func,
				   gpointer data)
{
  gdouble r = ftt_cell_size (root)*DIAGONAL;
  FttVector p;
  GtsIntersect i;

  ftt_cell_pos (root, &p);
  i = gfs_sphere_in_frustum (&p, r, f);
  if (i == GTS_OUT)
    return;
  if (FTT_CELL_IS_LEAF (root) ||
      ftt_cell_level (root) == maxlevel || 
      gfs_sphere_size (&p, r, f) < f->res)
    (* func) (root, data);
  else if (i == GTS_IN)
    cell_traverse_visible_no_check (root, f, maxlevel, func, data);
  else {
    struct _FttOct * children = root->children;
    guint n;

    for (n = 0; n < FTT_CELLS; n++) {
      FttCell * c = &(children->cell[n]);
      if (!FTT_CELL_IS_DESTROYED (c))
	cell_traverse_visible (c, f, maxlevel, func, data);
    }
  }
}

static void box_traverse_visible (GfsBox * b, gpointer * data)
{
  cell_traverse_visible (b->root, data[0], *((gint *)data[3]), data[1], data[2]);
}

/**
 * gfs_gl_cell_traverse_visible:
 * @gl: a #GfsGl.
 * @f: a view frustum.
 * @func: a used-defined function.
 * @data: user data to pass to @func.
 *
 * Traverse the cells of @gl which are visible.
 */
void gfs_gl_cell_traverse_visible (GfsGl * gl,
				   GfsFrustum * f,
				   FttCellTraverseFunc func,
				   gpointer data)
{
  gpointer dat[4];

  g_return_if_fail (gl != NULL);
  g_return_if_fail (f != NULL);
  g_return_if_fail (func != NULL);

  dat[0] = f;
  dat[1] = func;
  dat[2] = data;
  dat[3] = &gl->maxlevel;
  gts_container_foreach (GTS_CONTAINER (gl->sim), (GtsFunc) box_traverse_visible, dat);
}

static void cell_traverse_visible_mixed_no_check (FttCell * root,
						  GfsFrustum * f,
						  gint maxlevel,
						  FttCellTraverseFunc func,
						  gpointer data)
{
  if (!GFS_IS_MIXED (root))
    return;
  if (FTT_CELL_IS_LEAF (root) || ftt_cell_level (root) == maxlevel)
    (* func) (root, data);
  else {
    gdouble r = ftt_cell_size (root)*DIAGONAL;
    FttVector p;
    
    ftt_cell_pos (root, &p);
    if (gfs_sphere_size (&p, r, f) < f->res)
      (* func) (root, data);
    else {
      struct _FttOct * children = root->children;
      guint n;
      
      for (n = 0; n < FTT_CELLS; n++) {
	FttCell * c = &(children->cell[n]);
	if (!FTT_CELL_IS_DESTROYED (c))
	  cell_traverse_visible_mixed_no_check (c, f, maxlevel, func, data);
      }
    }
  }
}

static void cell_traverse_visible_mixed (FttCell * root,
					 GfsFrustum * f,
					 gint maxlevel,
					 FttCellTraverseFunc func,
					 gpointer data)
{
  gdouble r = ftt_cell_size (root)*DIAGONAL;
  FttVector p;
  GtsIntersect i;

  if (!GFS_IS_MIXED (root))
    return;
  ftt_cell_pos (root, &p);
  i = gfs_sphere_in_frustum (&p, r, f);
  if (i == GTS_OUT)
    return;
  if (FTT_CELL_IS_LEAF (root) ||
      ftt_cell_level (root) == maxlevel || 
      gfs_sphere_size (&p, r, f) < f->res)
    (* func) (root, data);
  else if (i == GTS_IN)
    cell_traverse_visible_mixed_no_check (root, f, maxlevel, func, data);
  else {
    struct _FttOct * children = root->children;
    guint n;

    for (n = 0; n < FTT_CELLS; n++) {
      FttCell * c = &(children->cell[n]);
      if (!FTT_CELL_IS_DESTROYED (c))
	cell_traverse_visible_mixed (c, f, maxlevel, func, data);
    }
  }
}

static void box_traverse_visible_mixed (GfsBox * b, gpointer * data)
{
  cell_traverse_visible_mixed (b->root, data[0], *((gint *)data[3]), data[1], data[2]);
}

/**
 * gfs_gl_cell_traverse_visible_mixed:
 * @gl: a #GfsGl.
 * @f: a view frustum.
 * @func: a used-defined function.
 * @data: user data to pass to @func.
 *
 * Traverse the cells of @gl which are visible and mixed.
 */
void gfs_gl_cell_traverse_visible_mixed (GfsGl * gl,
					 GfsFrustum * f,
					 FttCellTraverseFunc func,
					 gpointer data)
{
  gpointer dat[4];

  g_return_if_fail (gl != NULL);
  g_return_if_fail (f != NULL);
  g_return_if_fail (func != NULL);

  dat[0] = f;
  dat[1] = func;
  dat[2] = data;
  dat[3] = &gl->maxlevel;
  gts_container_foreach (GTS_CONTAINER (gl->sim), (GtsFunc) box_traverse_visible_mixed, dat);
}

static void cell_traverse_visible_boundary_no_check (FttCell * root,
						     GfsFrustum * f,
						     FttDirection d,
						     gint maxlevel,
						     FttCellTraverseFunc func,
						     gpointer data)
{
  if (FTT_CELL_IS_LEAF (root) || ftt_cell_level (root) == maxlevel)
    (* func) (root, data);
  else {
    gdouble r = ftt_cell_size (root)*DIAGONAL;
    FttVector p;
    
    ftt_cell_pos (root, &p);
    if (gfs_sphere_size (&p, r, f) < f->res)
      (* func) (root, data);
    else {
      FttCellChildren child;
      guint n;
      
      ftt_cell_children_direction (root, d, &child);
      for (n = 0; n < FTT_CELLS/2; n++)
	if (child.c[n])
	  cell_traverse_visible_boundary_no_check (child.c[n], f, d, maxlevel, func, data);
    }
  }
}

static void cell_traverse_visible_boundary (FttCell * root,
					    GfsFrustum * f,
					    FttDirection d,
					    gint maxlevel,
					    FttCellTraverseFunc func,
					    gpointer data)
{
  gdouble r = ftt_cell_size (root)*DIAGONAL;
  FttVector p;
  GtsIntersect i;

  ftt_cell_pos (root, &p);
  i = gfs_sphere_in_frustum (&p, r, f);
  if (i == GTS_OUT)
    return;
  if (FTT_CELL_IS_LEAF (root) ||
      ftt_cell_level (root) == maxlevel ||
      gfs_sphere_size (&p, r, f) < f->res)
    (* func) (root, data);
  else if (i == GTS_IN)
    cell_traverse_visible_boundary_no_check (root, f, d, maxlevel, func, data);
  else {
    FttCellChildren child;
    guint n;

    ftt_cell_children_direction (root, d, &child);
    for (n = 0; n < FTT_CELLS/2; n++)
      if (child.c[n])
	cell_traverse_visible_boundary (child.c[n], f, d, maxlevel, func, data);
  }
}

static void box_traverse_visible_boundary (GfsBox * b, gpointer * data)
{
  FttDirection d;

  for (d = 0; d < FTT_NEIGHBORS; d++)
    if (!b->neighbor[d] || GFS_IS_BOUNDARY (b->neighbor[d]))
      cell_traverse_visible_boundary (b->root, data[0], d, *((gint *)data[3]), data[1], data[2]);
}

/**
 * gfs_gl_cell_traverse_visible_boundary:
 * @gl: a #GfsGl.
 * @f: a view frustum.
 * @func: a used-defined function.
 * @data: user data to pass to @func.
 *
 * Traverse the boundary cells of @gl which are visible.
 */
void gfs_gl_cell_traverse_visible_boundary (GfsGl * gl,
					    GfsFrustum * f,
					    FttCellTraverseFunc func,
					    gpointer data)
{
  gpointer dat[4];

  g_return_if_fail (gl != NULL);
  g_return_if_fail (f != NULL);
  g_return_if_fail (func != NULL);

  dat[0] = f;
  dat[1] = func;
  dat[2] = data;
  dat[3] = &gl->maxlevel;
  gts_container_foreach (GTS_CONTAINER (gl->sim), (GtsFunc) box_traverse_visible_boundary, dat);
}

void gfs_gl_init (void)
{
  gfs_gl_cells_class ();
  gfs_gl_fractions_class ();
  gfs_gl_boundaries_class ();
  gfs_gl_squares_class ();
  gfs_gl_linear_class ();
  gfs_gl_isoline_class ();
  gfs_gl_solid_class ();
  gfs_gl_levels_class ();
  gfs_gl_vectors_class ();
    gfs_gl_streamlines_class ();
  gfs_gl_ellipses_class ();
  gfs_gl_location_class ();
#if (!FTT_2D)
  gfs_gl_isosurface_class ();
#endif /* 3D */
}

void gfs_gl_init_gl (void)
{
  GLfloat light0_pos[4]   = { 0.0, 0.0, 50.0, 0.0 };
  GLfloat light0_color[4] = { 1., 1., 1., 1.0 }; /* white light */

  glDisable (GL_CULL_FACE);
  glEnable (GL_DEPTH_TEST);

  /* speedups */
  glEnable (GL_DITHER);
  glShadeModel (GL_SMOOTH);
  glHint (GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
  glHint (GL_POLYGON_SMOOTH_HINT, GL_FASTEST);

  /* light */
  glLightfv (GL_LIGHT0, GL_POSITION, light0_pos);
  glLightfv (GL_LIGHT0, GL_DIFFUSE,  light0_color);
  glEnable (GL_LIGHT0);
  glEnable (GL_LIGHTING);

  glColorMaterial (GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);
  glEnable (GL_COLOR_MATERIAL);
}

/* GfsGl: Object */

static void gl_read (GtsObject ** o, GtsFile * fp)
{
  GfsGl * gl = GFS_GL (*o);
  gchar * shading;
  GtsFileVariable var[] = {
    {GTS_FLOAT,  "r",        TRUE},
    {GTS_FLOAT,  "g",        TRUE},
    {GTS_FLOAT,  "b",        TRUE},
    {GTS_STRING, "shading",  TRUE},
    {GTS_INT,    "maxlevel", TRUE},
    {GTS_NONE}
  };

  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (class)");
    return;
  }
  gts_file_next_token (fp);

  var[0].data = &gl->lc.r;
  var[1].data = &gl->lc.g;
  var[2].data = &gl->lc.b;
  var[3].data = &shading;
  var[4].data = &gl->maxlevel;

  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  if (var[3].set) {
    if (!strcmp (shading, "Constant"))
      gl->shading = GFS_GL_CONSTANT;
    else if (!strcmp (shading, "Flat"))
      gl->shading = GFS_GL_FLAT;
    else if (!strcmp (shading, "Smooth"))
      gl->shading = GFS_GL_SMOOTH;
    else if (!strcmp (shading, "CSmooth"))
      gl->shading = GFS_GL_CSMOOTH;
    else {
      gts_file_variable_error (fp, var, "shading", "unknown shading `%s'", shading);
      g_free (shading);
      return;
    }
    g_free (shading);
  }
}

static void gl_write (GtsObject * o, FILE * fp)
{
  GfsGl * gl = GFS_GL (o);

  g_assert (strlen (o->klass->info.name) > 5);
  fprintf (fp, "%s {\n"
	   "  r = %g g = %g b = %g\n"
	   "  shading = %s\n"
	   "  maxlevel = %d\n"
	   "}",
	   &o->klass->info.name[5],
	   gl->lc.r, gl->lc.g, gl->lc.b,
	   gl->shading == GFS_GL_CONSTANT ? "Constant" :
	   gl->shading == GFS_GL_FLAT ?     "Flat" :
	   gl->shading == GFS_GL_SMOOTH ?   "Smooth" :
	   gl->shading == GFS_GL_CSMOOTH ?  "CSmooth" : 
	                                    "Unknown",
	   gl->maxlevel);
}

static void gl_set_simulation (GfsGl * gl, GfsSimulation * sim)
{
  gl->sim = sim;
}

static void gl_class_init (GfsGlClass * klass)
{
  klass->set_simulation = gl_set_simulation;
  GTS_OBJECT_CLASS (klass)->read = gl_read;
  GTS_OBJECT_CLASS (klass)->write = gl_write;
}

static void gl_init (GfsGl * gl)
{
  GtsColor c = { 0., 0., 0. };

  gl->shading = GFS_GL_CONSTANT;
  gl->lc = c;
  gl->maxlevel = -1;
  gl->format = GFSGL_SCREEN;
}

GfsGlClass * gfs_gl_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_info = {
      "GfsGl",
      sizeof (GfsGl),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_class_init,
      (GtsObjectInitFunc) gl_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gts_object_class ()),
				  &gfs_gl_info);
  }

  return klass;
}

GfsGl * gfs_gl_new (GfsGlClass * klass)
{
  GfsGl * object;

  g_return_val_if_fail (klass != NULL, NULL);

  object = GFS_GL (gts_object_new (GTS_OBJECT_CLASS (klass)));

  return object;
}

/**
 * gfs_gl_new_from_file:
 * @fp: a #GtsFile pointer.
 *
 * Returns: a new #GfsGl object read from @fp.
 */
GfsGl * gfs_gl_new_from_file (GtsFile * fp)
{
  GtsObjectClass * klass;
  GfsGl * gl;
  GtsObject * o;

  g_return_val_if_fail (fp != NULL, NULL);

  klass = gts_object_class_from_name (fp->token->str);
  if (klass == NULL) {
    gchar * ename = g_strconcat ("GfsGl", fp->token->str, NULL);
    klass = gts_object_class_from_name (ename);
    g_free (ename);
  }
  if (klass == NULL || !gts_object_class_is_from_class (klass, gfs_gl_class ()))
    return NULL;
  gl = gfs_gl_new (GFS_GL_CLASS (klass));
  o = GTS_OBJECT (gl);
  (* klass->read) (&o, fp);
  if (fp->type == GTS_ERROR) {
    gts_object_destroy (o);
    return NULL;
  }
  else
    return gl;
}

void gfs_gl_set_simulation (GfsGl * gl, GfsSimulation * sim)
{
  g_return_if_fail (gl != NULL);
  g_return_if_fail (sim != NULL);

  (* GFS_GL_CLASS (GTS_OBJECT (gl)->klass)->set_simulation) (gl, sim);
}

void gfs_gl_draw (GfsGl * gl)
{
  g_return_if_fail (gl != NULL);

  if (gl->sim)
    (* GFS_GL_CLASS (GTS_OBJECT (gl)->klass)->draw) (gl);
}

/* GfsGl2D: Object */

static void gl2D_read (GtsObject ** o, GtsFile * fp)
{
  GfsGl2D * gl = GFS_GL2D (*o);
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "n.x", TRUE},
    {GTS_DOUBLE, "n.y", TRUE},
    {GTS_DOUBLE, "n.z", TRUE},
    {GTS_DOUBLE, "pos", TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl2D_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  var[0].data = &gl->n.x;
  var[1].data = &gl->n.y;
  var[2].data = &gl->n.z;
  var[3].data = &gl->pos;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  gfs_gl2D_update_plane (gl);
}

static void gl2D_write (GtsObject * o, FILE * fp)
{
  GfsGl2D * gl = GFS_GL2D (o);

  (* GTS_OBJECT_CLASS (gfs_gl2D_class ())->parent_class->write) (o, fp);

  fprintf (fp, " {\n"
	   "  n.x = %g n.y = %g n.z = %g\n"
	   "  pos = %g\n"
	   "}",
	   gl->n.x, gl->n.y, gl->n.z,
	   gl->pos);
}

void gfs_gl2D_update_plane (GfsGl2D * gl)
{
  gdouble n;
  GtsVector Q0 = {0., 0., 0.};
  GtsVector Q1 = {0., 0., 0.};
  gdouble max;
  guint d = 0;
  
  g_return_if_fail (gl != NULL);

  n = gts_vector_norm (&gl->n.x);
  g_assert (n > 0.);
  gts_vector_normalize (&gl->n.x);

  /* build a vector orthogonal to the constraint */
  max = gl->n.x*gl->n.x;
  if (gl->n.y*gl->n.y > max) { max = gl->n.y*gl->n.y; d = 1; }
  if (gl->n.z*gl->n.z > max) { max = gl->n.z*gl->n.z; d = 2; }
  switch (d) {
  case 0: Q0[0] = - gl->n.z/gl->n.x; Q0[2] = 1.0; break;
  case 1: Q0[1] = - gl->n.z/gl->n.y; Q0[2] = 1.0; break;
  case 2: Q0[2] = - gl->n.x/gl->n.z; Q0[0] = 1.0; break;
  }
  
  /* build a second vector orthogonal to the first and to the constraint */
  gts_vector_cross (Q1, &gl->n.x, Q0);
  
  gl->p[0].x = gl->pos*gl->n.x;
  gl->p[0].y = gl->pos*gl->n.y;
  gl->p[0].z = gl->pos*gl->n.z;
  
  gl->p[1].x = gl->p[0].x + Q0[0];
  gl->p[1].y = gl->p[0].y + Q0[1];
  gl->p[1].z = gl->p[0].z + Q0[2];
  
  gl->p[2].x = gl->p[0].x + Q1[0];
  gl->p[2].y = gl->p[0].y + Q1[1];
  gl->p[2].z = gl->p[0].z + Q1[2];
}

static gdouble gl2D_pick (GfsGl * gl, GfsGlRay * r)
{
  GfsGl2D * gl2 = GFS_GL2D (gl);
  GtsVector AB;
  gdouble ABn;

  gts_vector_init (AB, &r->a, &r->b);
  ABn = gts_vector_scalar (AB, &gl2->n.x);
  if (fabs (ABn) < 1e-6) {
    gl2->picked = NULL;
    return G_MAXDOUBLE;
  }
  else {
    GtsVector AP;
    gdouble a;

    AP[0] = gl2->n.x*gl2->pos - r->a.x;
    AP[1] = gl2->n.y*gl2->pos - r->a.y;
    AP[2] = gl2->n.z*gl2->pos - r->a.z;
    a = gts_vector_scalar (AP, &gl2->n.x)/ABn;
    gl2->pickedpos.x = r->a.x*(1. - a) + a*r->b.x;
    gl2->pickedpos.y = r->a.y*(1. - a) + a*r->b.y;
    gl2->pickedpos.z = r->a.z*(1. - a) + a*r->b.z;
    gl2->picked = gfs_domain_locate (GFS_DOMAIN (gl->sim), gl2->pickedpos, gl->maxlevel);
    return gl2->picked ? a : G_MAXDOUBLE;
  }
}

static void gl2D_class_init (GfsGlClass * klass)
{
  klass->pick = gl2D_pick;
  GTS_OBJECT_CLASS (klass)->read = gl2D_read;
  GTS_OBJECT_CLASS (klass)->write = gl2D_write;
  
}

static void gl2D_init (GfsGl2D * object)
{
  object->n.x = 0.; object->n.y = 0.; object->n.z = 1.;
  object->pos = 0.;

  object->p[0].x = object->p[0].y = object->p[0].z = 0.;
  object->p[1].x = 1.; object->p[1].y = object->p[1].z = 0.;
  object->p[2].y = 1.; object->p[2].x = object->p[2].z = 0.;
}

GfsGlClass * gfs_gl2D_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl2D_info = {
      "GfsGl2D",
      sizeof (GfsGl2D),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl2D_class_init,
      (GtsObjectInitFunc) gl2D_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_class ()), &gfs_gl2D_info);
  }

  return klass;
}

/* GfsGlCells: Object */

static void gl_cells_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_cell, gl);
  glPopMatrix ();
}

static void gl_cells_class_init (GfsGlClass * klass)
{
  klass->draw = gl_cells_draw;
}

GfsGlClass * gfs_gl_cells_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_cells_info = {
      "GfsGlCells",
      sizeof (GfsGl2D),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_cells_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl2D_class ()), &gfs_gl_cells_info);
  }

  return klass;
}

/* GfsGlFractions: Object */

static void gl_fractions_class_init (GfsGlClass * klass)
{
  klass->draw = gl_fractions_draw;
}

static void gl_fractions_init (GfsGl * gl)
{
  GtsColor c = { 0., 0., 1.};

  gl->lc = c;
}

GfsGlClass * gfs_gl_fractions_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_fractions_info = {
      "GfsGlFractions",
      sizeof (GfsGl),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_fractions_class_init,
      (GtsObjectInitFunc) gl_fractions_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_class ()), &gfs_gl_fractions_info);
  }

  return klass;
}

/* GfsGlBoundaries: Object */

static void gl_boundaries_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glBegin (GL_LINES);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_cell_traverse_visible_boundary (gl, &f, (FttCellTraverseFunc) gl_boundaries, gl);
  glEnd ();
  glPopMatrix ();
}

static void gl_boundaries_class_init (GfsGlClass * klass)
{
  klass->draw = gl_boundaries_draw;
}

static void gl_boundaries_init (GfsGl * gl)
{
  GtsColor c = { 0., 0., 0.};

  gl->lc = c;
}

GfsGlClass * gfs_gl_boundaries_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_boundaries_info = {
      "GfsGlBoundaries",
      sizeof (GfsGl),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_boundaries_class_init,
      (GtsObjectInitFunc) gl_boundaries_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_class ()), &gfs_gl_boundaries_info);
  }

  return klass;
}

/* GfsGlLevels: Object */

static void gl_levels_destroy (GtsObject * object)
{
  GfsGlLevels * gl = GFS_GL_LEVELS (object);

  if (gl->v)
    gts_object_destroy (GTS_OBJECT (gl->v));

  (* GTS_OBJECT_CLASS (gfs_gl_levels_class ())->parent_class->destroy) (object);
}

static void set_level (FttCell * cell, GfsVariable * v)
{
  GFS_VARIABLE (cell, v->i) = ftt_cell_level (cell);
}

static void gl_levels_set_simulation (GfsGl * object, GfsSimulation * sim)
{
  GfsDomain * domain = GFS_DOMAIN (sim);
  GfsGlLevels * gl = GFS_GL_LEVELS (object);

  (*GFS_GL_CLASS (GTS_OBJECT_CLASS (gfs_gl_levels_class ())->parent_class)->set_simulation)
    (object, sim);

  if (gl->v)
    gts_object_destroy (GTS_OBJECT (gl->v));
  gl->v = gfs_temporary_variable (domain);

  gfs_domain_cell_traverse (domain,
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) set_level, gl->v);
  gfs_domain_cell_traverse (domain,
			    FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			    (FttCellTraverseFunc) gfs_get_from_below_intensive, gl->v);
  gfs_domain_bc (domain, FTT_TRAVERSE_ALL, -1, gl->v);
}

static void gl_levels_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glBegin (GL_LINES);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_face, gl);
  glEnd ();
  glPopMatrix ();
}

static void gl_levels_class_init (GfsGlClass * klass)
{
  klass->set_simulation = gl_levels_set_simulation;
  klass->draw = gl_levels_draw;
  klass->pick = NULL;
  GTS_OBJECT_CLASS (klass)->destroy = gl_levels_destroy;
}

GfsGlClass * gfs_gl_levels_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_levels_info = {
      "GfsGlLevels",
      sizeof (GfsGlLevels),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_levels_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl2D_class ()),
				  &gfs_gl_levels_info);
  }

  return klass;
}

/* GfsGlVarFunc: object */

GfsGlVarFunc * gfs_gl_var_func_new (void)
{
  GfsGlVarFunc * vf = g_malloc0 (sizeof (GfsGlVarFunc));
  return vf;
}

void gfs_gl_var_func_destroy (GfsGlVarFunc * vf)
{
  g_return_if_fail (vf != NULL);

  if (vf->v && vf->v != gfs_function_get_variable (vf->f))
    gts_object_destroy (GTS_OBJECT (vf->v));
  if (vf->f)
    gts_object_destroy (GTS_OBJECT (vf->f));
  g_free (vf);
}

static void update_v (FttCell * cell, GfsGlVarFunc * vf)
{
  GFS_VARIABLE (cell, vf->v->i) = gfs_function_value (vf->f, cell);
}

GtsFile * gfs_gl_var_func_set (GfsGlVarFunc * vf, 
			       GfsDomain * domain, 
			       const gchar * func,
			       GString * expr)
{
  GfsFunction * f;
  GtsFile * fp;

  g_return_val_if_fail (vf != NULL, NULL);
  g_return_val_if_fail (domain != NULL, NULL);
  g_return_val_if_fail (func != NULL, NULL);

  fp = gts_file_new_from_string (func);
  f = gfs_function_new (gfs_function_class (), 0.);
  gfs_function_read (f, domain, fp);
  if (fp->type == GTS_ERROR) {
    gts_object_destroy (GTS_OBJECT (f));
    return fp;
  }
  gts_file_destroy (fp);

  if (vf->v && vf->v != gfs_function_get_variable (vf->f))
    gts_object_destroy (GTS_OBJECT (vf->v));
  if (vf->f)
    gts_object_destroy (GTS_OBJECT (vf->f));
  vf->f = f;

  if (!(vf->v = gfs_function_get_variable (vf->f))) {
    vf->v = gfs_temporary_variable (domain);
    gfs_domain_cell_traverse (domain,
			      FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			      (FttCellTraverseFunc) update_v, vf);
    gfs_domain_cell_traverse (domain,
			      FTT_POST_ORDER, FTT_TRAVERSE_NON_LEAFS, -1,
			      (FttCellTraverseFunc) vf->v->fine_coarse, vf->v);
    gfs_domain_bc (domain, FTT_TRAVERSE_ALL, -1, vf->v);
  }

  if (expr && expr->str != func) {
    g_free (expr->str);
    expr->str = g_strdup (func);
    expr->len = strlen (expr->str);
  }
  return NULL;
}

/* GfsGlScalar: Object */

static void gl_scalar_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlScalar * gl = GFS_GL_SCALAR (*o);
  gchar * cmap;
  GtsFileVariable var[] = {
    {GTS_INT,    "amin", TRUE},
    {GTS_DOUBLE, "min",  TRUE},
    {GTS_INT,    "amax", TRUE},
    {GTS_DOUBLE, "max",  TRUE},
    {GTS_STRING, "cmap", TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_scalar_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  g_string_free (gl->expr, TRUE);
  if (!(gl->expr = gfs_function_expression (fp, NULL)))
    return;
  gts_file_next_token (fp);

  var[0].data = &gl->amin;
  var[1].data = &gl->min;
  var[2].data = &gl->amax;
  var[3].data = &gl->max;
  var[4].data = &cmap;

  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  if (var[4].set)
    g_free (cmap);
}

static void gl_scalar_write (GtsObject * o, FILE * fp)
{
  GfsGlScalar * gl = GFS_GL_SCALAR (o);
  
  (* GTS_OBJECT_CLASS (gfs_gl_scalar_class ())->parent_class->write) (o, fp);

  fprintf (fp, " %s {\n  amin = %d", gl->expr->str, gl->amin);
  if (gl->amin)
    fputc ('\n', fp);
  else
    fprintf (fp, " min = %g\n", gl->min);
  fprintf (fp, "  amax = %d", gl->amax);
  if (gl->amax)
    fputc ('\n', fp);
  else
    fprintf (fp, " max = %g\n", gl->max);
  fprintf (fp, "  cmap = Jet\n}");
}

static void gl_scalar_destroy (GtsObject * object)
{
  GfsGlScalar * gl = GFS_GL_SCALAR (object);

  gfs_gl_var_func_destroy (gl->vf);
  g_string_free (gl->expr, TRUE);
  if (gl->cmap)
    gfs_colormap_destroy (gl->cmap);

  (* GTS_OBJECT_CLASS (gfs_gl_scalar_class ())->parent_class->destroy) (object);
}

static void min_max (FttCell * cell, GfsGlScalar * gl)
{
  gdouble v = GFS_VARIABLE (cell, gl->v->i);
  if (v > gl->amaxv)
    gl->amaxv = v;
  if (v < gl->aminv)
    gl->aminv = v;
}

GtsFile * gfs_gl_scalar_set (GfsGlScalar * gl, const gchar * func)
{
  GtsFile * fp;

  g_return_val_if_fail (gl != NULL, NULL);
  g_return_val_if_fail (func != NULL, NULL);

  if ((fp = gfs_gl_var_func_set (gl->vf, GFS_DOMAIN (GFS_GL (gl)->sim), func, gl->expr)))
    return fp;

  gl->v = gl->vf->v;
  gl->amaxv = -G_MAXDOUBLE;
  gl->aminv =  G_MAXDOUBLE;
  gfs_domain_cell_traverse (GFS_DOMAIN (GFS_GL (gl)->sim),
			    FTT_PRE_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) min_max, gl);
  if (gl->amax) gl->max = gl->amaxv;
  if (gl->amin) gl->min = gl->aminv;

  return NULL;
}

static void gl_scalar_set_simulation (GfsGl * object, GfsSimulation * sim)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (object);
  GtsFile * fp = NULL;

  (*GFS_GL_CLASS (GTS_OBJECT_CLASS (gfs_gl_scalar_class ())->parent_class)->set_simulation)
    (object, sim);

  if (gls->expr->str[0] == '\0' || (fp = gfs_gl_scalar_set (gls, gls->expr->str))) {
    GfsDomain * domain = GFS_DOMAIN (sim);
    
    if (domain->variables) {
      GfsVariable * v = domain->variables->data;
      gfs_gl_scalar_set (gls, v->name);
    }
    else
      gfs_gl_scalar_set (gls, "0");
  }
  if (fp)
    gts_file_destroy (fp);
}

static void gl_scalar_class_init (GfsGlClass * klass)
{
  klass->set_simulation = gl_scalar_set_simulation;
  GTS_OBJECT_CLASS (klass)->read = gl_scalar_read;
  GTS_OBJECT_CLASS (klass)->write = gl_scalar_write;
  GTS_OBJECT_CLASS (klass)->destroy = gl_scalar_destroy;
}

static void gl_scalar_init (GfsGlScalar * object)
{
  object->expr = g_string_new ("");
  object->vf = gfs_gl_var_func_new ();
  object->amax = object->amin = TRUE;
  object->min = 0.;
  object->max = 1.;
  object->cmap = gfs_colormap_jet ();
}

GfsGlClass * gfs_gl_scalar_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_scalar_info = {
      "GfsGlScalar",
      sizeof (GfsGlScalar),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_scalar_class_init,
      (GtsObjectInitFunc) gl_scalar_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl2D_class ()),
				  &gfs_gl_scalar_info);
  }

  return klass;
}

/* GfsGlSquares: Object */

static void gl_squares_class_init (GfsGlClass * klass)
{
  klass->draw = gl_squares_draw;
}

GfsGlClass * gfs_gl_squares_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_squares_info = {
      "GfsGlSquares",
      sizeof (GfsGlSquares),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_squares_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_scalar_class ()),
				  &gfs_gl_squares_info);
  }

  return klass;
}

/* GfsGlLinear: Object */

static void gl_linear_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlLinear * gl = GFS_GL_LINEAR (*o);
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "scale",  TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_linear_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  var[0].data = &gl->scale;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

static void gl_linear_write (GtsObject * o, FILE * fp)
{
  GfsGlLinear * gl = GFS_GL_LINEAR (o);
  
  (* GTS_OBJECT_CLASS (gfs_gl_linear_class ())->parent_class->write) (o, fp);
  fprintf (fp, " {\n  scale = %g\n}", gl->scale);
}

static void gl_linear_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read  = gl_linear_read;
  GTS_OBJECT_CLASS (klass)->write = gl_linear_write;
  klass->draw = gl_linear_draw;
}

GfsGlClass * gfs_gl_linear_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_linear_info = {
      "GfsGlLinear",
      sizeof (GfsGlLinear),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_linear_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_scalar_class ()),
				  &gfs_gl_linear_info);
  }

  return klass;
}

/* GfsGlIsoline: Object */

static void gl_isoline_destroy (GtsObject * o)
{
  g_array_free (GFS_GL_ISOLINE (o)->levels, TRUE);
  g_free (GFS_GL_ISOLINE (o)->ls);

  (* GTS_OBJECT_CLASS (gfs_gl_isoline_class ())->parent_class->destroy) (o);
}

static void gl_isoline_update_levels (GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  GfsGlIsoline * gli = GFS_GL_ISOLINE (gl);
  gchar * s, * list;

  g_array_set_size (gli->levels, 0);
  if (gli->n > 0.) {
    guint i;
    for (i = 0; i < gli->n; i++) {
      gdouble l = gls->min + (i + 1)*(gls->max - gls->min)/(gli->n + 1.);
      g_array_append_val (gli->levels, l);
    }
  }

  list = g_strdup (gli->ls);
  s = strtok (list, ",");
  while (s) {
    char * end;
    gdouble l = strtod (s, &end);
    if (*end == '\0')
      g_array_append_val (gli->levels, l);
    s = strtok (NULL, ",");
  }
  g_free (list);  
}

static void gl_isoline_read (GtsObject ** o, GtsFile * fp)
{
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "n",      TRUE},
    {GTS_STRING, "levels", TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_isoline_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  var[0].data = &GFS_GL_ISOLINE (*o)->n;
  var[1].data = &GFS_GL_ISOLINE (*o)->ls;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  gl_isoline_update_levels (GFS_GL (*o));
}

static void gl_isoline_write (GtsObject * o, FILE * fp)
{
  const gchar * levels = GFS_GL_ISOLINE (o)->ls;

  (* GTS_OBJECT_CLASS (gfs_gl_isoline_class ())->parent_class->write) (o, fp);

  fprintf (fp, " {\n  n = %g", GFS_GL_ISOLINE (o)->n);
  if (levels && levels[0] != '\0')
    fprintf (fp, " levels = %s", levels);
  fprintf (fp, "\n}");
}

static void gl_isoline_draw (GfsGl * gl)
{
  GfsFrustum f;

  gl_isoline_update_levels (gl);
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glBegin (GL_LINES);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_isoline, gl);
  glEnd ();
  glPopMatrix ();
}

static void gl_isoline_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->destroy = gl_isoline_destroy;
  GTS_OBJECT_CLASS (klass)->read = gl_isoline_read;
  GTS_OBJECT_CLASS (klass)->write = gl_isoline_write;
  klass->draw = gl_isoline_draw;
  klass->pick = NULL;
}

static void gl_isoline_init (GfsGlIsoline * gl)
{
  gl->levels = g_array_new (FALSE, FALSE, sizeof (gdouble));
  gl->n = 10.;
}

GfsGlClass * gfs_gl_isoline_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_isoline_info = {
      "GfsGlIsoline",
      sizeof (GfsGlIsoline),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_isoline_class_init,
      (GtsObjectInitFunc) gl_isoline_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_scalar_class ()),
				  &gfs_gl_isoline_info);
  }

  return klass;
}

void gfs_gl_isoline_set_levels (GfsGlIsoline * gl, const gchar * levels)
{
  g_return_if_fail (gl != NULL);
  g_return_if_fail (levels != NULL);

  g_free (gl->ls);
  gl->ls = g_strdup (levels);
}

/* GfsGlVectors: Object */

static void gl_vectors_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlVectors * gl = GFS_GL_VECTORS (*o);
  FttComponent c;
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "scale",      TRUE},
    {GTS_INT,    "use_scalar", TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_vectors_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  for (c = 0; c < FTT_DIMENSION; c++) {
    g_string_free (gl->expr[c], TRUE);
    if (!(gl->expr[c] = gfs_function_expression (fp, NULL)))
      return;
    gts_file_next_token (fp);
  }

  var[0].data = &gl->scale;
  var[1].data = &gl->use_scalar;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

static void gl_vectors_write (GtsObject * o, FILE * fp)
{
  GfsGlVectors * gl = GFS_GL_VECTORS (o);
  FttComponent c;

  (* GTS_OBJECT_CLASS (gfs_gl_vectors_class ())->parent_class->write) (o, fp);

  for (c = 0; c < FTT_DIMENSION; c++)
    fprintf (fp, " %s", gl->expr[c]->str);
  fprintf (fp, 
	   " {\n"
	   "  scale = %g\n"
	   "  use_scalar = %d\n"
	   "}",
	   gl->scale, gl->use_scalar);
}

static void gl_vectors_destroy (GtsObject * object)
{
  GfsGlVectors * gl = GFS_GL_VECTORS (object);
  FttComponent c;

  for (c = 0; c < FTT_DIMENSION; c++) {
    gfs_gl_var_func_destroy (gl->vf[c]);
    g_string_free (gl->expr[c], TRUE);
  }

  (* GTS_OBJECT_CLASS (gfs_gl_vectors_class ())->parent_class->destroy) (object);
}

static void maxv (FttCell * cell, GfsGlVectors * gl)
{
  FttComponent c;
  gdouble n2 = 0.;
  gdouble size = ftt_cell_size (cell);

  for (c = 0; c < FTT_DIMENSION; c++)
    n2 += GFS_VARIABLE (cell, gl->v[c]->i)*GFS_VARIABLE (cell, gl->v[c]->i);
  if (n2 > gl->max)
    gl->max = n2;
  if (size < gl->h)
    gl->h = size;
}

static void update_norm (GfsGlVectors * gl)
{
  gl->max = -G_MAXDOUBLE;
  gl->h = G_MAXDOUBLE;
  gfs_domain_cell_traverse (GFS_DOMAIN (GFS_GL (gl)->sim),
			    FTT_POST_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) maxv, gl);
  gl->max = gl->max >= 0. ? sqrt (gl->max) : 1.;
  if (!gl->already_set) {
    gl->scale = gl->max > 0. ? gl->h/gl->max : 1.;
    gl->already_set = TRUE;
  }
}

static void gl_vectors_set_simulation (GfsGl * object, GfsSimulation * sim)
{
  GfsGlVectors * gls = GFS_GL_VECTORS (object);
  GfsDomain * domain;
  FttComponent c;

  (*GFS_GL_CLASS (GTS_OBJECT_CLASS (gfs_gl_vectors_class ())->parent_class)->set_simulation)
    (object, sim);

  domain = GFS_DOMAIN (sim);
  for (c = 0; c < FTT_DIMENSION; c++) {
    GtsFile * fp = NULL;

    if (gls->expr[c]->str[0] == '\0' ||
	(fp = gfs_gl_var_func_set (gls->vf[c], domain, gls->expr[c]->str, NULL))) {
      GfsVariable ** u = gfs_domain_velocity (domain);

      if (u)
	gfs_gl_var_func_set (gls->vf[c], domain, u[c]->name, gls->expr[c]);
      else if (domain->variables) {
	GfsVariable * v = domain->variables->data;
	gfs_gl_var_func_set (gls->vf[c], domain, v->name, gls->expr[c]);
      }
      else
	gfs_gl_var_func_set (gls->vf[c], domain, "0", gls->expr[c]);
    }
    if (fp)
      gts_file_destroy (fp);
    gls->v[c] = gls->vf[c]->v;
  }
  update_norm (gls);
}

static void gl_vectors_draw (GfsGl * gl)
{
  GfsFrustum f;
  GfsGlVectors * gls = GFS_GL_VECTORS (gl);
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glBegin (GL_LINES);
  if (!gls->use_scalar)
    glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_vector, gl);
  glEnd ();
  glPopMatrix ();
}

static void gl_vectors_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gl_vectors_read;
  GTS_OBJECT_CLASS (klass)->write = gl_vectors_write;
  GTS_OBJECT_CLASS (klass)->destroy = gl_vectors_destroy;
  klass->set_simulation = gl_vectors_set_simulation;
  klass->draw = gl_vectors_draw;
  klass->pick = NULL;
}

static void gl_vectors_init (GfsGlVectors * object)
{
  FttComponent c;
  
  for (c = 0; c < FTT_DIMENSION; c++) {
    object->vf[c] = gfs_gl_var_func_new ();
    object->expr[c] = g_string_new ("");
  }
  object->scale = 1.;
}

GfsGlClass * gfs_gl_vectors_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_vectors_info = {
      "GfsGlVectors",
      sizeof (GfsGlVectors),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_vectors_class_init,
      (GtsObjectInitFunc) gl_vectors_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_scalar_class ()),
				  &gfs_gl_vectors_info);
  }

  return klass;
}

GtsFile * gfs_gl_vectors_set (GfsGlVectors * gl, FttComponent c, const gchar * func)
{
  GtsFile * fp;

  g_return_val_if_fail (gl != NULL, NULL);
  g_return_val_if_fail (c < FTT_DIMENSION, NULL);
  g_return_val_if_fail (func != NULL, NULL);

  if ((fp = gfs_gl_var_func_set (gl->vf[c], GFS_DOMAIN (GFS_GL (gl)->sim), func, gl->expr[c])))
    return fp;

  gl->v[c] = gl->vf[c]->v;
  update_norm (gl);

  return NULL;
}

/* GfsGlStreamline: Object */

static void gl_streamline_destroy (GtsObject * o)
{
  gfs_streamline_destroy (GFS_GL_STREAMLINE (o)->l);
  glDeleteLists (GFS_GL_STREAMLINE (o)->list, 1);

  (* GTS_OBJECT_CLASS (gfs_gl_streamline_class ())->parent_class->destroy) (o);
}

static void gl_streamline_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlStreamline * gl = GFS_GL_STREAMLINE (*o);
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "x", TRUE},
    {GTS_DOUBLE, "y", TRUE},
    {GTS_DOUBLE, "z", TRUE},
    {GTS_NONE}
  };

  var[0].data = &gl->c.x;
  var[1].data = &gl->c.y;
  var[2].data = &gl->c.z;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

static void gl_streamline_write (GtsObject * o, FILE * fp)
{
  GfsGlStreamline * gl = GFS_GL_STREAMLINE (o);

  fprintf (fp,
	   " {\n"
	   "    x = %.16f y = %.16f z = %.16f\n"
	   "  }",
	   gl->c.x, gl->c.y, gl->c.z);
}

static void gl_streamline_class_init (GtsObjectClass * klass)
{
  klass->destroy = gl_streamline_destroy;
  klass->read = gl_streamline_read;
  klass->write = gl_streamline_write;
}

GtsObjectClass * gfs_gl_streamline_class (void)
{
  static GtsObjectClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_streamline_info = {
      "GfsGlStreamline",
      sizeof (GfsGlStreamline),
      sizeof (GtsObjectClass),
      (GtsObjectClassInitFunc) gl_streamline_class_init,
      (GtsObjectInitFunc) NULL,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (gts_object_class (), &gfs_gl_streamline_info);
  }

  return klass;
}

static void gfs_gl_streamline_write (GfsGlStreamline * s, FILE * fp)
{
  (* GTS_OBJECT (s)->klass->write) (GTS_OBJECT (s), fp);
}

/* GfsGlStreamlines: Object */

static void gl_streamlines_destroy (GtsObject * o)
{
  GfsGlStreamlines * gl = GFS_GL_STREAMLINES (o);

  gfs_gl_streamlines_reset (gl);
  g_list_foreach (gl->stream, (GFunc) gts_object_destroy, NULL);
  g_list_free (gl->stream);

  if (gl->s)
    gts_object_destroy (GTS_OBJECT (gl->s));

  (* GTS_OBJECT_CLASS (gfs_gl_streamlines_class ())->parent_class->destroy) (o);
}

static void gl_streamlines_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlStreamlines * gl = GFS_GL_STREAMLINES (*o);
  GtsFileVariable var[] = {
    {GTS_INT,    "show_cells", TRUE},
    {GTS_DOUBLE, "dmin",       TRUE},
    {GTS_DOUBLE, "radius",     TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_streamlines_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;
  
  var[0].data = &gl->show_cells;
  var[1].data = &gl->dmin;
  var[2].data = &gl->radius;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  if (fp->type != '{') {
    gts_file_error (fp, "expecting an opening brace");
    return;
  }
  fp->scope_max++;
  gts_file_next_token (fp);
  while (fp->type == '\n')
    gts_file_next_token (fp);
  while (fp->type == '{') {
    GtsObject * o = gts_object_new (gfs_gl_streamline_class ());

    (* o->klass->read) (&o, fp);
    if (fp->type == GTS_ERROR) {
      gts_object_destroy (o);
      return;
    }
    gl->stream = g_list_append (gl->stream, o);
  }
  while (fp->type == '\n')
    gts_file_next_token (fp);
  if (fp->type != '}') {
    gts_file_error (fp, "expecting a closing brace");
    return;
  }
  fp->scope_max--;
  gts_file_next_token (fp);
}

static void gl_streamlines_write (GtsObject * o, FILE * fp)
{
  GfsGlStreamlines * gl = GFS_GL_STREAMLINES (o);

  (* GTS_OBJECT_CLASS (gfs_gl_streamlines_class ())->parent_class->write) (o, fp);

  fprintf (fp, " { show_cells = %d dmin = %g radius = %g } {\n ", 
	   gl->show_cells, gl->dmin, gl->radius);
  g_list_foreach (gl->stream, (GFunc) gfs_gl_streamline_write, fp);
  fputs ("\n}", fp);
}

static void gl_streamlines_set_simulation (GfsGl * object, GfsSimulation * sim)
{
  GfsGlStreamlines * gls = GFS_GL_STREAMLINES (object);
  
  gfs_gl_streamlines_reset (gls);

  (*GFS_GL_CLASS (GTS_OBJECT_CLASS (gfs_gl_streamlines_class ())->parent_class)->set_simulation)
    (object, sim);

  if (gls->s)
    gts_object_destroy (GTS_OBJECT (gls->s));
  gls->s = gfs_temporary_variable (GFS_DOMAIN (sim));
  gfs_domain_cell_traverse (GFS_DOMAIN (sim), FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			    (FttCellTraverseFunc) gfs_cell_reset, gls->s);
}

static gboolean cell_overlaps_segment (FttCell * cell, gpointer data)
{
  GList * l = data;
  GtsBBox bb;
  GtsSegment s;

  ftt_cell_bbox (cell, &bb);
  s.v1 = l->data; s.v2 = l->next->data;
  return gts_bbox_overlaps_segment (&bb, &s);
}

static void add_segment (FttCell * cell, gpointer * data)
{
  GfsVariable * v = data[0];
  GList * l = data[1];

  if (FTT_CELL_IS_LEAF (cell))
    GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, v->i)) = 
      g_slist_prepend (GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, v->i)), l);
  else
    GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, v->i)) = l;
}

static gdouble distance2 (GList * i, GtsPoint * p, GtsSegment * closest)
{
  gdouble d1 = G_MAXDOUBLE, d2 = G_MAXDOUBLE;
  GtsSegment s;

  s.v1 = i->data;
  if (i->next) {
    s.v2 = i->next->data;
    d1 = gts_point_segment_distance2 (p, &s);
  }
  if (i->prev) {
    s.v2 = i->prev->data;
    d2 = gts_point_segment_distance2 (p, &s);
  }
  if (closest) {
    closest->v1 = s.v1;
    closest->v2 = d1 < d2 ? i->next->data : i->prev->data;
  }
  return MIN (d1, d2);
}

static gboolean distance_is_larger (GList * a, GList * b, guint d)
{
  while (a && a != b && --d)
    a = a->next;
  return !d;
}

static gdouble cell_distance2 (FttCell * cell, GtsPoint * p, gpointer data)
{
  GfsGlStreamlines * gls = ((gpointer *) data)[0];
  GSList * i = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gls->s->i));

  if (i == NULL)
    return G_MAXDOUBLE;
  else if (!FTT_CELL_IS_LEAF (cell))
    return ftt_cell_point_distance2_min (cell, p);
  else {
    GfsGlStreamline * s = ((gpointer *) data)[1];
    GList * l = ((gpointer *) data)[2];
    gdouble dmin = G_MAXDOUBLE, h = ftt_cell_size (cell);

    while (i) {
      GList * j = i->data;
      gboolean same_stream = (GTS_OBJECT (j->data)->reserved == s);
      
      if (!same_stream || distance_is_larger (l, j, 16)) {
	gdouble d = distance2 (j, p, NULL);

	if (d < dmin)
	  dmin = d;
	if (same_stream && d < h*h/100.)
	  return 0.;
      }
      i = i->next;
    }
    return dmin;
  }
}

static gboolean stop (FttCell * cell, GList * l, gpointer data)
{
  GfsGlStreamlines * gls = ((gpointer *) data)[0];
  GfsGlStreamline * s = ((gpointer *) data)[1];
  gboolean stop = FALSE;

  ((gpointer *) data)[2] = l;
  stop = (gfs_domain_cell_point_distance2 (GFS_DOMAIN (GFS_GL (gls)->sim), l->data,
					   cell_distance2, data, NULL) 
	  <= gls->dmin*gls->dmin/4.);
  if (l->next) {
    gpointer datum[2];
    
    datum[0] = gls->s;
    datum[1] = l;
    gfs_domain_cell_traverse_condition (GFS_DOMAIN (GFS_GL (gls)->sim), 
					FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
					(FttCellTraverseFunc) add_segment, datum,
					cell_overlaps_segment, l);
  }
  GTS_OBJECT (l->data)->reserved = s;
  return stop;
}

static void gl_streamline_update_display_list (GfsGlStreamline * s,
					       GfsGlStreamlines * gls)
{
  if (s->list == 0) {
    s->list = glGenLists (1);
    if (s->list == 0)
      g_warning ("No available OpenGL display list!");
  }
  glNewList (s->list, GL_COMPILE);
  gl_streamline_draw (s, gls);
  glEndList ();
}

static void gl_streamline_update (GfsGlStreamline * s,
				  GfsGlStreamlines * gls)
{
  GtsPoint p;
  gpointer data[3];
  
  if (s->l)
    return;

  data[0] = gls;
  data[1] = s;
  p.x = s->c.x; p.y = s->c.y; p.z = s->c.z;
  if (gfs_domain_cell_point_distance2 (GFS_DOMAIN (GFS_GL (gls)->sim), &p,
				       cell_distance2, data, NULL) > gls->dmin*gls->dmin) {
    s->l = gfs_streamline_new (GFS_DOMAIN (GFS_GL (gls)->sim),
			       GFS_GL_VECTORS (gls)->v, s->c,
			       GFS_GL_VECTORS (gls)->use_scalar ? GFS_GL_SCALAR (gls)->v : NULL,
			       GFS_GL_SCALAR (gls)->min, GFS_GL_SCALAR (gls)->max, FALSE,
			       stop, data);
    if (s->l && !s->l->next) {
      gfs_streamline_destroy (s->l);
      s->l = NULL;
    }
    else
      gl_streamline_update_display_list (s, gls);
  }
}

static void gl_streamlines_draw (GfsGl * gl)
{
  GfsGlStreamlines * gls = GFS_GL_STREAMLINES (gl);
  GList * i = gls->stream;
  GfsFrustum f;

  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc/2.);
  gfs_gl_normal (gl);
  if (gls->show_cells) {
    glColor3f (0.3, 0.3, 0.3);
    gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_cell, gl);
  }
  glTranslatef (0., 0., gl->p->lc/2.);
  while (i) {
    GfsGlStreamline * s = i->data;

    if (i == gls->selected)
      glColor3f (1., 0., 0.);
    else if (!GFS_GL_VECTORS (gl)->use_scalar)
      glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
    gl_streamline_update (s, gls);
    if (s->l) {
      if (gl->format == GFSGL_PPM_OFFSCREEN)
	gl_streamline_draw (s, gls);
      else
	glCallList (s->list);
    }
    gl->size++;
    i = i->next;
  }
  glPopMatrix ();
}

static gdouble gl_streamlines_pick (GfsGl * gl, GfsGlRay * r)
{
  gint maxlevel = gl->maxlevel;
  gdouble a;

  gl->maxlevel = -1;
  a = gl2D_pick (gl, r);
  gl->maxlevel = maxlevel;
  if (a < G_MAXDOUBLE) {
    GfsGl2D * gl2 = GFS_GL2D (gl);
    GfsGlStreamlines * gls = GFS_GL_STREAMLINES (gl);
    GSList * i = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (gl2->picked, gls->s->i));
    gdouble dmin = G_MAXDOUBLE;
    GfsGlStreamline * closest = NULL;
    GtsPoint p;

    p.x = gl2->pickedpos.x; p.y = gl2->pickedpos.y; p.z = gl2->pickedpos.z;
    while (i) {
      GList * j = i->data;
      gdouble d = distance2 (j, &p, NULL);

      if (d < dmin) {
	dmin = d;
	closest = GTS_OBJECT (j->data)->reserved;
      }
      i = i->next;
    }
    gls->candidate = closest ? g_list_find (gls->stream, closest) : NULL;
  }
  return a;
}

static void gl_streamlines_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->destroy = gl_streamlines_destroy;
  GTS_OBJECT_CLASS (klass)->read = gl_streamlines_read;
  GTS_OBJECT_CLASS (klass)->write = gl_streamlines_write;
  klass->set_simulation = gl_streamlines_set_simulation;
  klass->draw = gl_streamlines_draw;
  klass->pick = gl_streamlines_pick;
}

static void gl_streamlines_init (GfsGlStreamlines * gl)
{
  GtsColor c = { 1., 1., 1. };

  GFS_GL (gl)->lc = c;
  gl->show_cells = TRUE;
}

GfsGlClass * gfs_gl_streamlines_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_streamlines_info = {
      "GfsGlStreamlines",
      sizeof (GfsGlStreamlines),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_streamlines_class_init,
      (GtsObjectInitFunc) gl_streamlines_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_vectors_class ()),
				  &gfs_gl_streamlines_info);
  }

  return klass;
}

static void reset_segments (FttCell * cell, GfsGlStreamlines * gl)
{
  if (FTT_CELL_IS_LEAF (cell)) {
    GSList * l = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gl->s->i));
    g_slist_free (l);
  }
  GFS_VARIABLE (cell, gl->s->i) = 0.;
}

void gfs_gl_streamlines_reset (GfsGlStreamlines * gl)
{
  GList * i;

  g_return_if_fail (gl != NULL);

  if (GFS_GL (gl)->sim)
    gfs_domain_cell_traverse (GFS_DOMAIN (GFS_GL (gl)->sim), 
			      FTT_PRE_ORDER, FTT_TRAVERSE_ALL, -1,
			      (FttCellTraverseFunc) reset_segments, gl);
  i = gl->stream;
  while (i) {
    GfsGlStreamline * s = i->data;
    gfs_streamline_destroy (s->l);
    s->l = NULL;    
    i = i->next;
  }
}

static void remove_segment (FttCell * cell, gpointer * data)
{
  GfsVariable * v = data[0];

  if (FTT_CELL_IS_LEAF (cell)) {
    GList * i = data[1];
    GSList * l = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, v->i));
    l = g_slist_remove (l, i);
    l = g_slist_remove (l, i->next);
    GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, v->i)) = l;
  }
  else {
    FttCellChildren child;
    gboolean empty = TRUE;
    guint i;
    
    ftt_cell_children (cell, &child);
    for (i = 0; i < FTT_CELLS && empty; i++)
      if (child.c[i] && GFS_VARIABLE (child.c[i], v->i) != 0.)
	empty = FALSE;
    if (empty)
      GFS_VARIABLE (cell, v->i) = 0.;
  }
}

void gfs_gl_streamlines_reset_selected (GfsGlStreamlines * gl)
{
  GfsGlStreamline * s;
  gpointer datum[2];
  GList * i;

  g_return_if_fail (gl != NULL);
  g_return_if_fail (gl->selected != NULL);

  s = gl->selected->data;
  datum[0] = gl->s;
  i = s->l;
  while (i && i->next) {
    datum[1] = i;
    gfs_domain_cell_traverse_condition (GFS_DOMAIN (GFS_GL (gl)->sim), 
					FTT_POST_ORDER, FTT_TRAVERSE_ALL, -1,
					(FttCellTraverseFunc) remove_segment, datum,
					cell_overlaps_segment, i);
    i = i->next;
  }
  gfs_streamline_destroy (s->l);
  s->l = NULL;
}

GfsGlStreamline * gfs_gl_streamlines_add (GfsGlStreamlines * gl, FttVector p)
{
  GfsGlStreamline * s;

  g_return_val_if_fail (gl != NULL, NULL);

  s = GFS_GL_STREAMLINE (gts_object_new (gfs_gl_streamline_class ()));
  s->c = p;
  gl_streamline_update (s, gl);
  if (s->l) {
    gl->stream = g_list_append (gl->stream, s);
    gl->selected = g_list_last (gl->stream);
    return s;
  }
  else {
    gts_object_destroy (GTS_OBJECT (s));
    return NULL;
  }
}

gboolean gfs_gl_streamlines_remove_selected (GfsGlStreamlines * gl)
{
  g_return_val_if_fail (gl != NULL, FALSE);

  if (gl->selected) {
    gfs_gl_streamlines_reset_selected (gl);
    gts_object_destroy (gl->selected->data);
    gl->stream = g_list_remove_link (gl->stream, gl->selected);
    g_list_free (gl->selected);
    gl->selected = NULL;
    return TRUE;
  }
  return FALSE;
}

void gfs_gl_streamlines_update_display_lists (GfsGlStreamlines * gl)
{
  g_return_if_fail (gl != NULL);

  g_list_foreach (gl->stream, (GFunc) gl_streamline_update_display_list, gl);
}

gdouble gfs_gl_streamlines_closest (GfsGlStreamlines * gl, 
				    FttVector * p,
				    GtsPoint * closest)
{
  FttCell * cell = NULL;
  GtsPoint p1;
  gdouble dmin;
  gpointer data[3];

  g_return_val_if_fail (gl != NULL, G_MAXDOUBLE);
  g_return_val_if_fail (p != NULL, G_MAXDOUBLE);
  g_return_val_if_fail (closest != NULL, G_MAXDOUBLE);

  data[0] = gl; data[1] = data[2] = NULL;
  p1.x = p->x; p1.y = p->y; p1.z = p->z;
  dmin = gfs_domain_cell_point_distance2 (GFS_DOMAIN (GFS_GL (gl)->sim), &p1,
					  cell_distance2, data, &cell);
  if (cell) {
    GSList * i = GFS_DOUBLE_TO_POINTER (GFS_VARIABLE (cell, gl->s->i));
    GtsSegment segment;
    gdouble dmin = G_MAXDOUBLE;
    
    while (i) {
      GList * j = i->data;
      GtsSegment s1;
      gdouble d = distance2 (j, &p1, &s1);
      
      if (d < dmin) {
	dmin = d;
	segment = s1;
      }
      i = i->next;
    }
    gts_point_segment_closest (&p1, &segment, closest);
  }
  return dmin;
}

static GList * streamline_step (GList * i, GtsPoint * p, gdouble ds)
{
  if (!i || !i->next)
    return NULL;
  if (p->x == G_MAXDOUBLE) {
    GtsPoint * p1 = i->data;
    p->x = p1->x; p->y = p1->y; p->z = p1->z;
    return i;
  }
  else {
    gdouble l = gts_point_distance (i->data, i->next->data);
    gdouble s;
  
    g_assert (l > 0.);
    s = (gts_point_distance (i->data, p) + ds)/l;
    while (s > 1. && i->next->next) {
      s = l*(s - 1.);
      i = i->next;
      l = gts_point_distance (i->data, i->next->data);
      g_assert (l > 0.);
      s /= l;
    }
    if (s > 1.)
      return NULL;
    else {
      GtsPoint * p1 = i->data;
      GtsPoint * p2 = i->next->data;
      p->x = p1->x + s*(p2->x - p1->x);
      p->y = p1->y + s*(p2->y - p1->y);
      p->z = p1->z + s*(p2->z - p1->z);
      return i;
    }
  }
}

static void push_stream (GfsGlStreamline * s, GtsFifo * fifo)
{
  gts_fifo_push (fifo, s);
}

void gfs_gl_streamlines_evenly_spaced (GfsGlStreamlines * gl,
				       gboolean (* callback) (GfsGlStreamlines *, gpointer),
				       gpointer data)
{
  GtsFifo * fifo;
  GfsGlStreamline * s;
  gboolean stop = FALSE;

  g_return_if_fail (gl != NULL);
  g_return_if_fail (gl->dmin > 0.);

  fifo = gts_fifo_new ();
  g_list_foreach (gl->stream, (GFunc) push_stream, fifo);
  while ((s = gts_fifo_pop (fifo)) && !stop) {
    GtsPoint p;
    GList * i = s->l;

    g_assert (i);
    p.x = G_MAXDOUBLE;
    while ((i = streamline_step (i, &p, gl->dmin/10.))) {
      GtsVector n;
      FttVector p1;
      GfsGlStreamline * s1;

      gts_vector_init (n, GTS_POINT (i->data), GTS_POINT (i->next->data));
      gts_vector_normalize (n);

      p1.x = p.x + n[1]*gl->dmin;
      p1.y = p.y - n[0]*gl->dmin;
      p1.z = 0.;
      if ((s1 = gfs_gl_streamlines_add (gl, p1))) {
	if (callback)
	  stop |= (* callback) (gl, data);
	gts_fifo_push (fifo, s1);
      }

      p1.x = p.x - n[1]*gl->dmin;
      p1.y = p.y + n[0]*gl->dmin;
      p1.z = 0.;
      if ((s1 = gfs_gl_streamlines_add (gl, p1))) {
	if (callback)
	  stop |= (* callback) (gl, data);	
	gts_fifo_push (fifo, s1);
      }
    }
  }
  gts_fifo_destroy (fifo);
  gl->selected = NULL;
}

/* GfsGlEllipses: Object */

static void gl_ellipses_read (GtsObject ** o, GtsFile * fp)
{
  GfsGlEllipses * gl = GFS_GL_ELLIPSES (*o);
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "scale",      TRUE},
    {GTS_INT,    "use_scalar", TRUE},
    {GTS_NONE}
  };
  guint i;

  (* GTS_OBJECT_CLASS (gfs_gl_ellipses_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;

  for (i = 0; i < 4; i++) {
    g_string_free (gl->expr[i], TRUE);
    if (!(gl->expr[i] = gfs_function_expression (fp, NULL)))
      return;
    gts_file_next_token (fp);
  }

  var[0].data = &gl->scale;
  var[1].data = &gl->use_scalar;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

static void gl_ellipses_write (GtsObject * o, FILE * fp)
{
  GfsGlEllipses * gl = GFS_GL_ELLIPSES (o);
  guint i;

  (* GTS_OBJECT_CLASS (gfs_gl_ellipses_class ())->parent_class->write) (o, fp);

  for (i = 0; i < 4; i++)
    fprintf (fp, " %s", gl->expr[i]->str);
  fprintf (fp, " {\n"
	   "  scale = %g\n"
	   "  use_scalar = %d\n"
	   "}",
	   gl->scale, gl->use_scalar);
}

static void gl_ellipses_destroy (GtsObject * o)
{
  GfsGlEllipses * gl = GFS_GL_ELLIPSES (o);
  guint i;

  for (i = 0; i < 4; i++) {
    gfs_gl_var_func_destroy (gl->vf[i]);
    g_string_free (gl->expr[i], TRUE);
  }

  (* GTS_OBJECT_CLASS (gfs_gl_ellipses_class ())->parent_class->destroy) (o);
}

static void maxe (FttCell * cell, GfsGlEllipses * gl)
{
  guint i;
  gdouble n2 = 0.;
  gdouble size = ftt_cell_size (cell);

  for (i = 0; i < 4; i++)
    n2 += GFS_VARIABLE (cell, gl->v[i]->i)*GFS_VARIABLE (cell, gl->v[i]->i);
  if (n2 > gl->max)
    gl->max = n2;
  if (size < gl->h)
    gl->h = size;
}

static void update_ellipses_norm (GfsGlEllipses * gl)
{
  gl->max = -G_MAXDOUBLE;
  gl->h = G_MAXDOUBLE;
  gfs_domain_cell_traverse (GFS_DOMAIN (GFS_GL (gl)->sim),
			    FTT_POST_ORDER, FTT_TRAVERSE_LEAFS, -1,
			    (FttCellTraverseFunc) maxe, gl);
  gl->max = gl->max >= 0. ? sqrt (gl->max) : 1.;
  if (!gl->already_set) {
    gl->scale = gl->max > 0. ? gl->h/gl->max : 1.;
    gl->already_set = TRUE;
  }
}

static void gl_ellipses_set_simulation (GfsGl * object, GfsSimulation * sim)
{
  GfsGlEllipses * gle = GFS_GL_ELLIPSES (object);
  GfsDomain * domain;
  FttComponent c;

  (*GFS_GL_CLASS (GTS_OBJECT_CLASS (gfs_gl_ellipses_class ())->parent_class)->set_simulation)
    (object, sim);

  domain = GFS_DOMAIN (sim);
  for (c = 0; c < 4; c++) {
    GtsFile * fp = NULL;

    if (gle->expr[c]->str[0] == '\0' ||
	(fp = gfs_gl_var_func_set (gle->vf[c], domain, gle->expr[c]->str, NULL))) {
      if (domain->variables) {
	GfsVariable * v = domain->variables->data;
	gfs_gl_var_func_set (gle->vf[c], domain, v->name, gle->expr[c]);
      }
      else
	gfs_gl_var_func_set (gle->vf[c], domain, "0", gle->expr[c]);
    }
    if (fp)
      gts_file_destroy (fp);
    gle->v[c] = gle->vf[c]->v;
  }
  update_ellipses_norm (gle);
}

static void gl_ellipses_draw (GfsGl * gl)
{
  GfsFrustum f;
  GfsGlEllipses * gls = GFS_GL_ELLIPSES (gl);
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  if (!gls->use_scalar)
    glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_ellipse, gl);
  glPopMatrix ();
}

static void gl_ellipses_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->read = gl_ellipses_read;
  GTS_OBJECT_CLASS (klass)->write = gl_ellipses_write;
  GTS_OBJECT_CLASS (klass)->destroy = gl_ellipses_destroy;
  klass->set_simulation = gl_ellipses_set_simulation;
  klass->draw = gl_ellipses_draw;
  klass->pick = NULL;
}

static void gl_ellipses_init (GfsGlEllipses * object)
{
  FttComponent c;
  
  for (c = 0; c < 4; c++) {
    object->vf[c] = gfs_gl_var_func_new ();
    object->expr[c] = g_string_new ("");
  }
  object->scale = 1.;
}

GfsGlClass * gfs_gl_ellipses_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_ellipses_info = {
      "GfsGlEllipses",
      sizeof (GfsGlEllipses),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_ellipses_class_init,
      (GtsObjectInitFunc) gl_ellipses_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_scalar_class ()),
				  &gfs_gl_ellipses_info);
  }

  return klass;
}

GtsFile * gfs_gl_ellipses_set (GfsGlEllipses * gl, guint i, const gchar * func)
{
  GtsFile * fp;
  
  g_return_val_if_fail (gl != NULL, NULL);
  g_return_val_if_fail (i < 4, NULL);
  g_return_val_if_fail (func != NULL, NULL);

  if ((fp = gfs_gl_var_func_set (gl->vf[i], GFS_DOMAIN (GFS_GL (gl)->sim), func, gl->expr[i])))
    return fp;

  gl->v[i] = gl->vf[i]->v;
  update_ellipses_norm (gl);

  return NULL;
}

/* GfsGlLocation: Object */

static void gl_location_destroy (GtsObject * o)
{
  gts_object_destroy (GTS_OBJECT (GFS_GL_LOCATION (o)->symbol));
  (* GTS_OBJECT_CLASS (gfs_gl_location_class ())->parent_class->destroy) (o);
}

static void gl_location_read (GtsObject ** o, GtsFile * fp)
{
  GtsFileVariable var[] = {
    {GTS_DOUBLE, "size", TRUE},
    {GTS_NONE}
  };

  (* GTS_OBJECT_CLASS (gfs_gl_location_class ())->parent_class->read) (o, fp);
  if (fp->type == GTS_ERROR)
    return;
  
  var[0].data = &GFS_GL_LOCATION (*o)->size;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

static void gl_location_write (GtsObject * o, FILE * fp)
{
  (* GTS_OBJECT_CLASS (gfs_gl_location_class ())->parent_class->write) (o, fp);

  fprintf (fp, " {\n  size = %g\n}", GFS_GL_LOCATION (o)->size);
}

static void gl_location (GtsTriangle * t, gpointer * data)
{
  gdouble * size = data[0];
  FttVector * p = data[1];
  GtsVertex * v1, * v2, * v3;

  gts_triangle_vertices (t, &v1, &v2, &v3);
  glNormal3d (GTS_POINT (v1)->x, GTS_POINT (v1)->y, GTS_POINT (v1)->z);
  glVertex3d (GTS_POINT (v1)->x**size + p->x, 
	      GTS_POINT (v1)->y**size + p->y, 
	      GTS_POINT (v1)->z**size + p->z);
  glNormal3d (GTS_POINT (v2)->x, GTS_POINT (v2)->y, GTS_POINT (v2)->z);
  glVertex3d (GTS_POINT (v2)->x**size + p->x, 
	      GTS_POINT (v2)->y**size + p->y, 
	      GTS_POINT (v2)->z**size + p->z);
  glNormal3d (GTS_POINT (v3)->x, GTS_POINT (v3)->y, GTS_POINT (v3)->z);
  glVertex3d (GTS_POINT (v3)->x**size + p->x, 
	      GTS_POINT (v3)->y**size + p->y, 
	      GTS_POINT (v3)->z**size + p->z);
}

static void gl_location_draw (GfsGl * gl)
{  
  GSList * i;
  gpointer data[2];
  gdouble size;

  glShadeModel (GL_SMOOTH);
  glBegin (GL_TRIANGLES);
  data[0] = &size;
  i = gl->sim->events->items;
  while (i) {
    if (GFS_IS_OUTPUT_LOCATION (i->data)) {
      GfsOutputLocation * l = i->data;
      guint j;

      for (j = 0; j < l->p->len; j++) {
	FttVector p = g_array_index (l->p, FttVector, j);
	FttCell * cell = gfs_domain_locate (GFS_DOMAIN (gl->sim), p, -1);

	if (cell == NULL) {
	  size = GFS_GL_LOCATION (gl)->size/64.;
	  glColor3f (1., 0., 0.);
	}
	else {
	  size = GFS_GL_LOCATION (gl)->size*ftt_cell_size (cell);
	  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
	}
	data[1] = &p;
	gts_surface_foreach_face (GFS_GL_LOCATION (gl)->symbol, (GtsFunc) gl_location, data);
      }
    }
    i = i->next;
  }
  glEnd ();
}

static void gl_location_init (GfsGl * gl)
{
  GtsColor c = { 0., 0., 1. };

  gl->lc = c;
  GFS_GL_LOCATION (gl)->size = 1.;
  GFS_GL_LOCATION (gl)->symbol = gts_surface_new (gts_surface_class (), gts_face_class (),
						  gts_edge_class (), gts_vertex_class ());
  gts_surface_generate_sphere (GFS_GL_LOCATION (gl)->symbol, 3);
}

static void gl_location_class_init (GfsGlClass * klass)
{
  GTS_OBJECT_CLASS (klass)->destroy = gl_location_destroy;
  GTS_OBJECT_CLASS (klass)->read = gl_location_read;
  GTS_OBJECT_CLASS (klass)->write = gl_location_write;
  klass->draw = gl_location_draw;
}

GfsGlClass * gfs_gl_location_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_location_info = {
      "GfsGlLocation",
      sizeof (GfsGlLocation),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_location_class_init,
      (GtsObjectInitFunc) gl_location_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_class ()), &gfs_gl_location_info);
  }

  return klass;
}

void gfs_gl_view_params_init (GfsGlViewParams * p)
{
  g_return_if_fail (p != NULL);

  p->do_init = TRUE;
  p->beginx = p->beginy = 0;
  p->dx = p->dy = 0;
  p->tx = p->ty = 0;
  p->quat[0] = p->quat[1] = p->quat[2] = 0; p->quat[3] = 1;
  p->dquat[0] = p->dquat[1] = p->dquat[2] = 0; p->dquat[3] = 1;
  p->fov = 30.;
  gfs_gl_trackball (p->quat , 0.0, 0.0, 0.0, 0.0);

  p->bg.r = 0.3; p->bg.g = 0.4; p->bg.b = 0.6;
  p->res = p->base_res = 1.;
  p->lc = 0.001;
  p->reactivity = 0.1;
  p->motion = FALSE;
}

GfsGlViewParams * gfs_gl_view_params_new (void)
{
  GfsGlViewParams * p = g_malloc (sizeof (GfsGlViewParams));
  gfs_gl_view_params_init (p);
  return p;
}

void gfs_gl_view_params_write (GfsGlViewParams * p, FILE * fp)
{
  g_return_if_fail (p != NULL);
  g_return_if_fail (fp != NULL);
  
  fprintf (fp, 
	   "View {\n"
	   "  tx = %g ty = %g\n"
	   "  q0 = %g q1 = %g q2 = %g q3 = %g\n"
	   "  fov = %g\n"
	   "  r = %g g = %g b = %g\n"
	   "  res = %g\n"
	   "  lc = %g\n"
	   "  reactivity = %g\n"
	   "}",
	   p->tx, p->ty, 
	   p->quat[0], p->quat[1], p->quat[2], p->quat[3], 
	   p->fov,
	   p->bg.r, p->bg.g, p->bg.b,
	   p->base_res,
	   p->lc,
	   p->reactivity);
}

void gfs_gl_view_params_read (GfsGlViewParams * p, GtsFile * fp)
{
  GtsFileVariable var[] = {
    {GTS_FLOAT, "tx",         TRUE},
    {GTS_FLOAT, "ty",         TRUE},
    {GTS_FLOAT, "q0",         TRUE},
    {GTS_FLOAT, "q1",         TRUE},
    {GTS_FLOAT, "q2",         TRUE},
    {GTS_FLOAT, "q3",         TRUE},
    {GTS_FLOAT, "fov",        TRUE},
    {GTS_FLOAT, "r",          TRUE},
    {GTS_FLOAT, "g",          TRUE},
    {GTS_FLOAT, "b",          TRUE},
    {GTS_FLOAT, "res",        TRUE},
    {GTS_FLOAT, "lc",         TRUE},
    {GTS_FLOAT, "reactivity", TRUE},
    {GTS_NONE}
  };

  g_return_if_fail (p != NULL);
  g_return_if_fail (fp != NULL);
  
  if (fp->type != GTS_STRING) {
    gts_file_error (fp, "expecting a string (\"View\")");
    return;
  }
  if (strcmp (fp->token->str, "View")) {
    gts_file_error (fp, "unknown keyword `%s'", fp->token->str);
    return;
  }
  gts_file_next_token (fp);

  var[0].data = &p->tx;
  var[1].data = &p->ty;
  var[2].data = &p->quat[0];
  var[3].data = &p->quat[1];
  var[4].data = &p->quat[2];
  var[5].data = &p->quat[3];
  var[6].data = &p->fov;
  var[7].data = &p->bg.r;
  var[8].data = &p->bg.g;
  var[9].data = &p->bg.b;
  var[10].data = &p->base_res;
  var[11].data = &p->lc;
  var[12].data = &p->reactivity;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;
}

void gfs_gl2ps_params_read (GfsGl2PSParams * p, GtsFile * fp)
{
  gchar * format, * orientation;
  GtsFileVariable var[] = {
    {GTS_STRING, "format",      TRUE},
    {GTS_STRING, "orientation", TRUE},
    {GTS_FLOAT,  "line_width",  TRUE},
    {GTS_UINT,   "width",       TRUE},
    {GTS_UINT,   "height",      TRUE},
    {GTS_NONE}
  };

  g_return_if_fail (p != NULL);
  g_return_if_fail (fp != NULL);

  p->format = GFSGL_PPM;
  p->options = (GL2PS_SIMPLE_LINE_OFFSET |
		GL2PS_SILENT |
		GL2PS_BEST_ROOT |
		GL2PS_OCCLUSION_CULL |
		GL2PS_USE_CURRENT_VIEWPORT |
		GL2PS_TIGHT_BOUNDING_BOX);
  p->width = p->height = 0;
  p->lw = 1.;
  p->sort = GL2PS_SIMPLE_SORT;
  var[0].data = &format;
  var[1].data = &orientation;
  var[2].data = &p->lw;
  var[3].data = &p->width;
  var[4].data = &p->height;
  gts_file_assign_variables (fp, var);
  if (fp->type == GTS_ERROR)
    return;

  if (var[0].set) {
    if (!strcmp (format, "PPM"))
      p->format = GFSGL_PPM;
    else if (!strcmp (format, "PS"))
      p->format = GL2PS_PS;
    else if (!strcmp (format, "EPS"))
      p->format = GL2PS_EPS;
    else if (!strcmp (format, "PDF"))
      p->format = GL2PS_PDF;
    else if (!strcmp (format, "Gnuplot"))
      p->format = GFSGL_GNUPLOT;
    else if (!strcmp (format, "Latex"))
      p->format = GL2PS_TEX;
    else {
      gts_file_variable_error (fp, var, "format", "unknown format `%s'", fp->token->str);
      g_free (format);
      return;
    }
    g_free (format);
  }

  if (var[1].set) {
    if (!strcmp (orientation, "Portrait"))
      p->options &= ~GL2PS_LANDSCAPE;
    else if (!strcmp (orientation, "Landscape"))
      p->options |= GL2PS_LANDSCAPE;
    else {
      gts_file_variable_error (fp, var, "orientation", "unknown orientation `%s'", fp->token->str);
      g_free (orientation);
      return;
    }
    g_free (orientation);
  }
  else
    p->options &= ~GL2PS_LANDSCAPE;  
}

void gfs_gl2ps_params_write (GfsGl2PSParams * p, FILE * fp)
{
  g_return_if_fail (p != NULL);
  g_return_if_fail (fp != NULL);

  fprintf (fp, " { format = %s orientation = %s line_width = %g width = %d height = %d }",
	   p->format == GFSGL_PPM ?     "PPM" :
	   p->format == GL2PS_PS ?      "PS" :
	   p->format == GL2PS_EPS ?     "EPS" :
	   p->format == GL2PS_PDF ?     "PDF" :
	   p->format == GFSGL_GNUPLOT ? "Gnuplot" :
	   p->format == GL2PS_TEX ?     "Latex" : 
	   "?",
	   p->options & GL2PS_LANDSCAPE ? "Landscape" : "Portrait",
	   p->lw, p->width, p->height);
}

void gfs_gl_write_image (FILE * fp, const GLubyte * buffer, guint width, guint height)
{
  gint i, x, y;
  const GLubyte *ptr = buffer;

  g_return_if_fail (fp != NULL);
  g_return_if_fail (buffer != NULL);

  fprintf (fp, "P6 %d %d 255\n", width, height);
  for (y = height - 1; y >= 0; y--) {
    for (x = 0; x < width; x++) {
      i = (y*width + x)*4;
      fputc (ptr[i], fp);   /* write red */
      fputc (ptr[i+1], fp); /* write green */
      fputc (ptr[i+2], fp); /* write blue */
    }
  }
}

static void extent (FttCell * cell, gdouble * max)
{
  gdouble h = ftt_cell_size (cell);
  FttVector p;
  FttComponent c;

  ftt_cell_pos (cell, &p);
  for (c = 0; c < 3; c++) {
    if (fabs ((&p.x)[c] + h) > *max) *max = fabs ((&p.x)[c] + h);
    if (fabs ((&p.x)[c] - h) > *max) *max = fabs ((&p.x)[c] - h);
  }
}

/**
 * gfs_gl_domain_extent:
 * @domain: a #GfsDomain.
 *
 * Returns: the maximum extent along any dimension of @domain.
 */
gdouble gfs_gl_domain_extent (GfsDomain * domain)
{
  gdouble max = 0.;

  if (domain == NULL)
    return 1.;
  
  gfs_domain_cell_traverse (domain, FTT_PRE_ORDER, FTT_TRAVERSE_LEVEL, 0,
			    (FttCellTraverseFunc) extent, &max);
  return max;
}

/**
 * gfs_gl_feedback_begin:
 * @buffersize: the size of the feedback buffer.
 *
 * Returns: a newly setup openGL feedback buffer.
 */
GfsGlFeedback * gfs_gl_feedback_begin (guint buffersize)
{
  GfsGlFeedback * f;

  g_return_val_if_fail (buffersize > 0, NULL);

  f = g_malloc (sizeof (GfsGlFeedback));
  f->feedback = g_malloc (sizeof (GLfloat)*buffersize);
  glFeedbackBuffer (buffersize, GL_3D_COLOR, f->feedback);
  glRenderMode (GL_FEEDBACK);
  return f;
}

/**
 * gfs_gl_feedback_end:
 * @f: the #GfsGlFeedback.
 * @fp: a file pointer.
 *
 * Writes to @fp the contents of @f.
 *
 * Returns: %FALSE if @f has overflowed, %TRUE otherwise.
 */
gboolean gfs_gl_feedback_end (GfsGlFeedback * f, FILE * fp)
{
  GLint size;

  g_return_val_if_fail (f != NULL, FALSE);
  g_return_val_if_fail (fp != NULL, FALSE);

  size = glRenderMode (GL_RENDER);
  if (size >= 0) {
    GLint used = size;
    GLfloat * current = f->feedback;
    GLdouble model[16], proj[16];
    GLint viewport[4];
    GLdouble x, y, z;

    glGetDoublev (GL_MODELVIEW_MATRIX, model);
    glGetDoublev (GL_PROJECTION_MATRIX, proj);
    glGetIntegerv (GL_VIEWPORT, viewport);
    while (used > 0)
      switch ((GLint) *current) {
      case GL_POINT_TOKEN :
	current++; used--;
	gluUnProject (current[0], current[1], current[2], model, proj, viewport, &x, &y, &z);
	fprintf (fp, "%g %g %g\n\n", x, y, z);
	current += 7; used -= 7;
	break;
      case GL_LINE_TOKEN :
      case GL_LINE_RESET_TOKEN :
	current++; used--;
	gluUnProject (current[0], current[1], current[2], model, proj, viewport, &x, &y, &z);
	fprintf (fp, "%g %g %g\n", x, y, z);
	current += 7; used -= 7;
	gluUnProject (current[0], current[1], current[2], model, proj, viewport, &x, &y, &z);
	fprintf (fp, "%g %g %g\n\n", x, y, z);
	current += 7; used -= 7;
	break;
      case GL_POLYGON_TOKEN : {
	GLint count = (GLint) current[1];
	current += 2;
	used -= 2;
	while (count > 0 && used > 0) {
	  gluUnProject (current[0], current[1], current[2], model, proj, viewport, &x, &y, &z);
	  fprintf (fp, "%g %g %g\n", x, y, z);
	  current += 7; used -= 7;
	  count--;
	}
	fputc ('\n', fp);
	break;
      }
      }
  }
  g_free (f->feedback);
  g_free (f);
  return (size >= 0);
}
