/* Gerris - The GNU Flow Solver
 * Copyright (C) 2001-2004 National Institute of Water and Atmospheric
 * Research
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.  
 */

#define DIAGONAL 0.707106781187

/* GfsGlCells: Object */

static void gl_cell (FttCell * cell, GfsGl * gl)
{
  FttVector p;
  gdouble size = ftt_cell_size (cell)/2.;
  
  ftt_cell_pos (cell, &p);
  glBegin (GL_LINE_LOOP);
  glVertex2d (p.x - size, p.y - size);
  glVertex2d (p.x + size, p.y - size);
  glVertex2d (p.x + size, p.y + size);
  glVertex2d (p.x - size, p.y + size);
  glEnd ();
  gl->size++;
}

/* GfsGlSquares: Object */

static void gl_square (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  FttVector p;
  gdouble size = ftt_cell_size (cell)/1.999;
  GtsColor c;

  gl->size++;
  ftt_cell_pos (cell, &p);
  c = gfs_colormap_color (gls->cmap, gls->max > gls->min ?
			  (GFS_VARIABLE (cell, gls->v->i) - gls->min)/(gls->max - gls->min) :
			  0.5);
  glColor3f (c.r, c.g, c.b);
  glVertex2d (p.x - size, p.y - size);
  glVertex2d (p.x + size, p.y - size);
  glVertex2d (p.x + size, p.y + size);
  glVertex2d (p.x - size, p.y + size);
}

static void gl_squares_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glBegin (GL_QUADS);
  gfs_gl_normal (gl);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_square, gl);
  glEnd ();
}

/* GfsGlSolid: Object */

#define CUT(d) (s->s[d] > 0. && s->s[d] < 1.)

static void gl_solid (FttCell * cell, GfsGl * gl)
{
  if (GFS_IS_MIXED (cell)) {
    GfsSolidVector * s = GFS_STATE (cell)->solid;
    guint n = 0;
    FttDirection d;
    FttVector c, p[4];
    gdouble h = ftt_cell_size (cell)/1.999;
    static guint etod[] = { 3, 0, 2, 1 };
    
    gl->size++;
    ftt_cell_pos (cell, &c);  
    p[0].x = c.x - h; p[0].y = c.y - h;
    p[1].x = c.x + h; p[1].y = c.y - h;
    p[2].x = c.x + h; p[2].y = c.y + h;
    p[3].x = c.x - h; p[3].y = c.y + h;
    
    for (d = 0; d < FTT_NEIGHBORS; d++)
      if (s->s[d] > 0. && s->s[d] < 1.)
	n++;
    switch (n) {
    case 2: {
      guint k, m, n1 = 0;
      FttVector r[2];
      gboolean ins;
      static guint ctoc[4][2] = {{0,2},{2,1},{1,3},{3,0}};
      
      for (m = 0; m < 4 && !CUT (etod[m]); m++);
      ins = CUT (ctoc[m][0]) ? s->s[ctoc[m][1]] : (s->s[ctoc[m][0]] != 1.);
      for (k = m; k < m + 4; k++) {
	guint i = k % 4, i1 = (i + 1) % 4;
	if (CUT (etod[i])) {
	  gdouble a = ins ? s->s[etod[i]] : 1. - s->s[etod[i]];
	  r[n1].x = (1. - a)*p[i].x + a*p[i1].x;
	  r[n1++].y = (1. - a)*p[i].y + a*p[i1].y;
	  ins = !ins;
	  if (n1 == 2) {
	    glVertex2d (r[0].x, r[0].y);
	    glVertex2d (r[1].x, r[1].y);
	    n1 = 0;
	  }
	}
      }
      break;
    }
    case 4:
      break;
    default:
#if 0
      g_assert_not_reached ();
#else
      {
	FttVector p;
	ftt_cell_pos (cell, &p);
	g_warning ("(%g,%g): %d", p.x, p.y, n);
      }
#endif
    }
  }
}

static void gl_solid_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glBegin (GL_LINES);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  glNormal3d (0., 0., 1.);
  gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_solid, gl);
  glEnd ();
  glPopMatrix ();
}

static void gl_solid_init (GfsGl * gl)
{
  GtsColor c = { 0., 1., 0. };

  gl->lc = c;
}

static void gl_solid_class_init (GfsGlClass * klass)
{
  klass->draw = gl_solid_draw;
}

GfsGlClass * gfs_gl_solid_class (void)
{
  static GfsGlClass * klass = NULL;

  if (klass == NULL) {
    GtsObjectClassInfo gfs_gl_solid_info = {
      "GfsGlSolid",
      sizeof (GfsGl),
      sizeof (GfsGlClass),
      (GtsObjectClassInitFunc) gl_solid_class_init,
      (GtsObjectInitFunc) gl_solid_init,
      (GtsArgSetFunc) NULL,
      (GtsArgGetFunc) NULL
    };
    klass = gts_object_class_new (GTS_OBJECT_CLASS (gfs_gl_class ()), &gfs_gl_solid_info);
  }

  return klass;
}

/* GfsGlFractions: Object */

static void gl_fractions (FttCell * cell, GfsGl * gl)
{
  GfsSolidVector * s = GFS_STATE (cell)->solid;
  FttVector c;
  gdouble h = ftt_cell_size (cell)*0.45;
  
  gl->size++;
  ftt_cell_pos (cell, &c);
  glVertex2d (c.x + h, c.y - h*s->s[0]);
  glVertex2d (c.x + h, c.y + h*s->s[0]);
  glVertex2d (c.x - h, c.y - h*s->s[1]);
  glVertex2d (c.x - h, c.y + h*s->s[1]);
  glVertex2d (c.x - h*s->s[2], c.y + h);
  glVertex2d (c.x + h*s->s[2], c.y + h);
  glVertex2d (c.x - h*s->s[3], c.y - h);
  glVertex2d (c.x + h*s->s[3], c.y - h);
}

static void gl_fractions_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glTranslatef (0., 0., gl->p->lc);
  glBegin (GL_LINES);
  glColor3f (gl->lc.r, gl->lc.g, gl->lc.b);
  gfs_gl_cell_traverse_visible_mixed (gl, &f, (FttCellTraverseFunc) gl_fractions, gl);
  glEnd ();
  glPopMatrix ();
}

/* GfsGlBoundaries: Object */

static void gl_boundaries (FttCell * cell, GfsGl * gl)
{
  if (!GFS_IS_MIXED (cell)) {
    FttDirection d;
    FttCellNeighbors n;
    gdouble h = ftt_cell_size (cell)/2.;
    FttVector p;

    ftt_cell_neighbors (cell, &n);
    ftt_cell_pos (cell, &p);
    for (d = 0; d < FTT_NEIGHBORS; d++)
      if (!n.c[d] || GFS_CELL_IS_BOUNDARY (n.c[d])) {
	static FttVector dp[FTT_NEIGHBORS][2] = {
	  {{1.,-1.,0.},{1.,1.,0.}},
	  {{-1.,1.,0.},{-1.,-1,0.}},
	  {{1.,1.,0.},{-1.,1.,0.}},
	  {{-1.,-1.,0.},{1.,-1.,0.}}
	};
	gl->size++;
	glVertex2d (p.x + dp[d][0].x*h, p.y + dp[d][0].y*h);
	glVertex2d (p.x + dp[d][1].x*h, p.y + dp[d][1].y*h);
      }
  }
}

/* GfsGlLevels: Object */

/**
 * ftt_face_gl:
 * @face: a #FttCellFace.
 *
 * Creates an OpenGL representation of @face.  
 */
static void ftt_face_gl (const FttCellFace * face)
{
  gdouble size;
  FttVector p;
#if FTT_2D
  static FttVector dp[FTT_NEIGHBORS][2] = {
    {{1.,-1.,0.},{1.,1.,0.}},
    {{-1.,1.,0.},{-1.,-1,0.}},
    {{1.,1.,0.},{-1.,1.,0.}},
    {{-1.,-1.,0.},{1.,-1.,0.}}
  };
#else  /* FTT_3D */
  static FttVector dp[FTT_NEIGHBORS][4] = {
    {{1.,-1.,1.},{1.,-1.,-1.},{1.,1.,-1.},{1.,1.,1.}},
    {{-1.,-1.,1.},{-1.,-1.,-1.},{-1.,1.,-1.},{-1.,1.,1.}},
    {{1.,1.,1.},{1.,1.,-1.},{-1,1.,-1.},{-1.,1.,1.}},
    {{1.,-1.,1.},{1.,-1.,-1.},{-1,-1.,-1.},{-1.,-1.,1.}},
    {{1.,-1.,1.},{1.,1.,1.},{-1.,1.,1.},{-1.,-1.,1.}},
    {{1.,-1.,-1.},{1.,1.,-1.},{-1.,1.,-1.},{-1.,-1.,-1.}},
  };
#endif /* FTT_3D */

  g_return_if_fail (face != NULL);

  size = ftt_cell_size (face->cell)/2.;
  ftt_cell_pos (face->cell, &p);
#if FTT_2D
  glVertex2d (p.x + dp[face->d][0].x*size, p.y + dp[face->d][0].y*size);
  glVertex2d (p.x + dp[face->d][1].x*size, p.y + dp[face->d][1].y*size);
#else  /* FTT_3D */
#if 0
  fprintf (fp, 
	   "OFF 4 1 4 "
	   "%g %g %g "
	   "%g %g %g "
	   "%g %g %g "
	   "%g %g %g "
	   "4 0 1 2 3\n",
	   p.x + dp[face->d][0].x*size,
	   p.y + dp[face->d][0].y*size,
	   p.z + dp[face->d][0].z*size,
	   p.x + dp[face->d][1].x*size,
	   p.y + dp[face->d][1].y*size,
	   p.z + dp[face->d][1].z*size,
	   p.x + dp[face->d][2].x*size,
	   p.y + dp[face->d][2].y*size,
	   p.z + dp[face->d][2].z*size,
	   p.x + dp[face->d][3].x*size,
	   p.y + dp[face->d][3].y*size,
	   p.z + dp[face->d][3].z*size);
#endif
#endif /* FTT_3D */
}

static void gl_face (FttCell * cell, GfsGlLevels * gl)
{
  FttCellNeighbors n;
  FttCellFace f;
  gboolean drawn = FALSE;

  ftt_cell_neighbors (cell, &n);
  f.cell = cell;
  for (f.d = 0; f.d < FTT_NEIGHBORS; f.d++) {
    f.neighbor = n.c[f.d];
    if (f.neighbor &&
	floor (GFS_VARIABLE (cell, gl->v->i)) != 
	floor (GFS_VARIABLE (f.neighbor, gl->v->i))) {
      ftt_face_gl (&f);
      drawn = TRUE;
    }
  }
  if (drawn)
    GFS_GL (gl)->size++;
}

/* GfsGlVectors: Object */

static void gl_vector (FttCell * cell, GfsGl * gl)
{
  GfsGlVectors * gls = GFS_GL_VECTORS (gl);
  FttComponent c;
  FttVector pos, f;

  gl->size++;
  if (gls->use_scalar) {
    GfsGlScalar * gla = GFS_GL_SCALAR (gl);
    GtsColor c = gfs_colormap_color (gla->cmap, gla->max > gla->min ?
				     (GFS_VARIABLE (cell, gla->v->i) - gla->min)/
				     (gla->max - gla->min) :
				     0.5);
    glColor3f (c.r, c.g, c.b);
  }
  gfs_cell_cm (cell, &pos);
  for (c = 0; c < FTT_DIMENSION; c++)
    (&f.x)[c] = GFS_VARIABLE (cell, gls->v[c]->i);
  f.x *= gls->scale;
  f.y *= gls->scale;
  glVertex2d (pos.x + f.x - (f.x - f.y/2.)/5., pos.y + f.y - (f.x/2. + f.y)/5.);
  glVertex2d (pos.x + f.x, pos.y + f.y);
  glVertex2d (pos.x + f.x, pos.y + f.y);
  glVertex2d (pos.x + f.x - (f.x + f.y/2.)/5., pos.y + f.y + (f.x/2. - f.y)/5.);
  glVertex2d (pos.x, pos.y);
  glVertex2d (pos.x + f.x, pos.y + f.y);
}

/* GfsGlEllipses: Object */

static void gl_ellipse (FttCell * cell, GfsGl * gl)
{
  GfsGlEllipses * gls = GFS_GL_ELLIPSES (gl);
  FttVector pos;
  gdouble t;

  gl->size++;
  if (gls->use_scalar) {
    GfsGlScalar * gla = GFS_GL_SCALAR (gl);
    GtsColor c = gfs_colormap_color (gla->cmap, gla->max > gla->min ?
				     (GFS_VARIABLE (cell, gla->v->i) - gla->min)/
				     (gla->max - gla->min) :
				     0.5);
    glColor3f (c.r, c.g, c.b);
  }
  gfs_cell_cm (cell, &pos);
  glBegin (GL_LINE_LOOP);
  for (t = 0.; t < 2.*M_PI; t += 2.*M_PI/20.) {
    gdouble cost = cos (t), sint = sin (t);

    glVertex2d (pos.x + gls->scale*(GFS_VARIABLE (cell, gls->v[0]->i)*cost + 
				    GFS_VARIABLE (cell, gls->v[2]->i)*sint),
		pos.y + gls->scale*(GFS_VARIABLE (cell, gls->v[1]->i)*cost + 
				    GFS_VARIABLE (cell, gls->v[3]->i)*sint));
  }
  glEnd ();
}

/* GfsGlLinear: Object */

#define param(v) (gls->max > gls->min ? ((v) - gls->min)/(gls->max - gls->min) : 0.5)
#define color(v) (gfs_colormap_color (gls->cmap, param (v)))

static void gl_linear_color (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  GfsGlLinear * gll = GFS_GL_LINEAR (gl);
  FttVector p;
  gdouble v, size = ftt_cell_size (cell)/1.999;
  GtsColor c;
  FttDirection d[2];

  gl->size++;
  ftt_cell_pos (cell, &p);
  d[0] = FTT_LEFT; d[1] = FTT_BOTTOM;
  v = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  c = color (v);
  glColor3f (c.r, c.g, c.b);
  glVertex3d (p.x - size, p.y - size, gll->scale*v);
  d[0] = FTT_RIGHT; d[1] = FTT_BOTTOM;
  v = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  c = color (v);
  glColor3f (c.r, c.g, c.b);
  glVertex3d (p.x + size, p.y - size, gll->scale*v);
  d[0] = FTT_RIGHT; d[1] = FTT_TOP;
  v = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  c = color (v);
  glColor3f (c.r, c.g, c.b);
  glVertex3d (p.x + size, p.y + size, gll->scale*v);
  d[0] = FTT_LEFT; d[1] = FTT_TOP;
  v = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  c = color (v);
  glColor3f (c.r, c.g, c.b);
  glVertex3d (p.x - size, p.y + size, gll->scale*v);
}

static void gl_linear_texture (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  GfsGlLinear * gll = GFS_GL_LINEAR (gl);
  FttVector p;
  gdouble v1, v2, v3, v4, p1, p2, p3, p4;
  gdouble size = ftt_cell_size (cell)/1.999;
  FttDirection d[2];

  gl->size++;
  ftt_cell_pos (cell, &p);

  d[0] = FTT_LEFT; d[1] = FTT_BOTTOM;
  v1 = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  p1 = param (v1);
  d[0] = FTT_RIGHT; d[1] = FTT_BOTTOM;
  v2 = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  p2 = param (v2);
  d[0] = FTT_RIGHT; d[1] = FTT_TOP;
  v3 = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  p3 = param (v3);
  d[0] = FTT_LEFT; d[1] = FTT_TOP;
  v4 = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  p4 = param (v4);

  glBegin (GL_TRIANGLE_FAN);
  glTexCoord1d ((p1 + p2 + p3 + p4)/4.);
  glVertex3d (p.x, p.y, gll->scale*(v1 + v2 + v3 + v4)/4.);
  glTexCoord1d (p1);
  glVertex3d (p.x - size, p.y - size, gll->scale*v1);
  glTexCoord1d (p2);
  glVertex3d (p.x + size, p.y - size, gll->scale*v2);
  glTexCoord1d (p3);
  glVertex3d (p.x + size, p.y + size, gll->scale*v3);
  glTexCoord1d (p4);
  glVertex3d (p.x - size, p.y + size, gll->scale*v4);
  glTexCoord1d (p1);
  glVertex3d (p.x - size, p.y - size, gll->scale*v1);
  glEnd ();
}

static void gl_linear_draw (GfsGl * gl)
{
  GfsFrustum f;
  
  gfs_gl_get_frustum (gl, &f);
  gl->size = 0;
  glShadeModel (GL_SMOOTH);
  gfs_gl_normal (gl);
  if (gl->format == GL2PS_PS || gl->format == GL2PS_EPS || gl->format == GL2PS_PDF) {
    glBegin (GL_QUADS);
    gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_linear_color, gl);
    glEnd ();
  }
  else {
    glEnable (GL_TEXTURE_1D);
    gfs_colormap_texture (GFS_GL_SCALAR (gl)->cmap);
    glColor3f (1., 1., 1.);
    gfs_gl_cell_traverse_visible_plane (gl, &f, (FttCellTraverseFunc) gl_linear_texture, gl);
    glDisable (GL_TEXTURE_1D);
  }
}

/* GfsGlIsoline: Object */

static guint inter (FttVector * p1, FttVector * p2, gdouble z)
{
  if ((p1->z > z && p2->z <= z) || (p1->z <= z && p2->z > z)) {
    gdouble a = (z - p1->z)/(p2->z - p1->z);
    glVertex2d (p1->x + a*(p2->x - p1->x), p1->y + a*(p2->y - p1->y));
    return 1;
  }
  return 0;
}

static void gl_isoline (FttCell * cell, GfsGl * gl)
{
  GfsGlScalar * gls = GFS_GL_SCALAR (gl);
  GArray * levels = GFS_GL_ISOLINE (gl)->levels;
  FttVector p[4];
  FttDirection d[2];
  guint i;

  gl->size++;
  d[0] = FTT_LEFT; d[1] = FTT_BOTTOM;
  ftt_corner_pos (cell, d, &p[0]);
  p[0].z = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  d[0] = FTT_RIGHT; d[1] = FTT_BOTTOM;
  ftt_corner_pos (cell, d, &p[1]);
  p[1].z = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  d[0] = FTT_RIGHT; d[1] = FTT_TOP;
  ftt_corner_pos (cell, d, &p[2]);
  p[2].z = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);
  d[0] = FTT_LEFT; d[1] = FTT_TOP;
  ftt_corner_pos (cell, d, &p[3]);
  p[3].z = gfs_cell_corner_value (cell, d, gls->v, gl->maxlevel);

  for (i = 0; i < levels->len; i++) {
    gdouble z = g_array_index (levels, gdouble, i);
    guint n = 0;

    n += inter (&p[0], &p[1], z);
    n += inter (&p[1], &p[2], z);
    n += inter (&p[2], &p[3], z);
    n += inter (&p[3], &p[0], z);
    g_assert (n % 2 == 0);
  }
}

/* GfsGlStreamline: Object */

static void draw_point (GtsPoint * p)
{
  glVertex3d (p->x, p->y, p->z);
}

static void gl_streamline_draw (GfsGlStreamline * s,
				GfsGlStreamlines * gls)
{
  glBegin (GL_LINE_STRIP);
  g_list_foreach (s->l, (GFunc) draw_point, NULL);
  glEnd ();
}
