!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeadm_taskloop
  use cubeadm_messaging
  use gkernel_interfaces
  use cube_types
  use cubeio_interfaces_public
  use cubedag_types
  !
  integer(kind=data_k), parameter :: nullplane = huge(0_data_k)  ! Flag for uninitialized plane value
  !
  type cubeadm_iterator_t
    ! Access mode
    integer(kind=code_k) :: access      ! Iterator access mode
    ! Data values
    integer(kind=data_k) :: nd          ! Total number of data values
    integer(kind=data_k) :: ndperentry  ! Number of data values per entry
    integer(kind=data_k) :: ndperplane  ! Number of data values per plane
    integer(kind=data_k) :: ndpertask   ! Number of data values per task
    integer(kind=data_k) :: ndperblock  ! Number of data values per block
    integer(kind=data_k) :: ndginentry  ! Number of data granularity in entry (1 entry is a multiple of this)
    integer(kind=data_k) :: ndginblock  ! Number of data granularity in block (1 block is a multiple of this)
    ! Entries
    integer(kind=entr_k) :: ne          ! Total number of entries
    integer(kind=entr_k) :: neperblock  ! Number of entries per IO block
    integer(kind=entr_k) :: nepertask   ! Number of entries per task
    integer(kind=entr_k) :: neperplane  ! Number of entries per plane (if <1, use npperentry)
    ! Planes
    integer(kind=entr_k) :: np          ! Total number of planes
    integer(kind=entr_k) :: nppertask   ! Number of planes per task
    integer(kind=entr_k) :: npperblock  ! Number of planes per block
    integer(kind=entr_k) :: npperentry  ! Number of planes per entry (if <1, use neperplane)
    integer(kind=entr_k) :: startplane  ! First plane to iterate
    integer(kind=entr_k) :: stopplane   ! Last plane to iterate
    integer(kind=entr_k) :: offsetplane ! Offset to be removed for put subcubes
    ! Tasks
    integer(kind=entr_k) :: nt          ! Total number of tasks
    real(kind=4)         :: ntperblock  ! Number of tasks per block
    ! Blocks
    real(kind=4)         :: nb          ! Number of blocks
    ! Lists for iterations
    integer(kind=entr_k), allocatable :: firstplanes(:)
    integer(kind=entr_k), allocatable :: lastplanes(:)
    integer(kind=entr_k), allocatable :: firstentries(:)
    integer(kind=entr_k), allocatable :: lastentries(:)
    ! Iterated components
    integer(kind=entr_k) :: ct          ! Current task
    integer(kind=data_k) :: firstdata   ! First data  to be processed
    integer(kind=data_k) :: lastdata    ! Last  data  to be processed
    integer(kind=entr_k) :: firstplane  ! First plane to be processed
    integer(kind=entr_k) :: lastplane   ! Last  plane to be processed
    integer(kind=entr_k) :: first       ! First entry to be processed (kept for backward compatibility)
    integer(kind=entr_k) :: last        ! Last  entry to be processed (kept for backward compatibility)
  end type cubeadm_iterator_t
  !
  interface cubeadm_taskloop_iterator_reallocate
    module procedure cubeadm_taskloop_iterator_reallocate4
    module procedure cubeadm_taskloop_iterator_reallocate8
  end interface cubeadm_taskloop_iterator_reallocate
  !
  public :: nullplane
  public :: cubeadm_iterator_t
  public :: cubeadm_taskloop_init,cubeadm_taskloop_iterate
  private
  !
contains
  !
  subroutine cubeadm_taskloop_init(incubes,oucubes,startplane,stopplane,iter,error)
    !----------------------------------------------------------------------
    ! Set up the iterator suited for the input and the output cubes
    !----------------------------------------------------------------------
    type(cubedag_link_t),     intent(in)    :: incubes     ! In cube list
    type(cubedag_link_t),     intent(in)    :: oucubes     ! Out cube list
    integer(kind=data_k),     intent(in)    :: startplane  ! Input range to be iterated
    integer(kind=data_k),     intent(in)    :: stopplane   ! Input range to be iterated
    type(cubeadm_iterator_t), intent(out)   :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>INIT'
    character(len=message_length) :: mess
    type(cubeadm_iterator_t), allocatable :: myiter(:)
    integer(kind=entr_k) :: icube,ncube,me
    type(cube_t), pointer :: cube
    integer(kind=4) :: ier
    !
    call cubeadm_message(admseve%trace,rname,'Welcome')
    !
    ! Compute one iterator per cube
    allocate(myiter(incubes%n+oucubes%n),stat=ier)
    if (failed_allocate(rname,'iterator',ier,error)) return
    ncube = 0
    !
    ! Input cubes
    do icube=1,incubes%n
      cube => cubetuple_cube_ptr(incubes%list(icube)%p,error)
      if (error)  return
      ! Skip cubes for which we won't iterate data
      if (cube%tuple%current%desc%action.eq.code_read_head)  cycle
      ! Set up an iterator suited for this cube
      ncube = ncube+1
      write(mess,'(A,I0)') '--- Computing iterator for cube #',ncube
      call cubeadm_message(admseve%others,rname,mess)
      call cubeadm_taskloop_iterator(cube,startplane,stopplane,myiter(ncube),error)
      if (error)  return
      myiter(ncube)%offsetplane = 0  ! By design for input cubes
    enddo
    !
    ! Output cubes
    do icube=1,oucubes%n
      cube => cubetuple_cube_ptr(oucubes%list(icube)%p,error)
      if (error)  return
      ! Set up an iterator suited for this cube
      ncube = ncube+1
      write(mess,'(A,I0)') '--- Computing iterator for cube #',ncube
      call cubeadm_message(admseve%others,rname,mess)
      call cubeadm_taskloop_iterator(cube,nullplane,nullplane,myiter(ncube),error)
      if (error)  return
      if (startplane.ne.nullplane) then
        myiter(ncube)%offsetplane = startplane-1  ! Offset to be removed for output subcubes
      else
        myiter(ncube)%offsetplane = 0
      endif
    enddo
    !
    ! Cubes consistency
    if (startplane.eq.0 .and. stopplane.eq.0) then
      ! Number of entries must be the same or 1
      me = maxval(myiter(1:ncube)%ne)
      do icube=1,ncube
        if (myiter(icube)%ne.ne.1 .and. myiter(icube)%ne.ne.me) then
          write(mess,'(A,I0,A,I0)')  &
            'Iterated cubes have different number of entries: ',myiter(icube)%ne,',',me
          call cubeadm_message(seve%e,rname,mess)
          error = .true.
          return
        endif
      enddo
    else
      ! Which consistency could we check?
    endif
    !
    ! Compute the global iterator by merging all the individual ones
    call cubeadm_taskloop_merge(myiter,ncube,iter,error)
    if (error)  return
    if (startplane.ne.nullplane) then
      iter%offsetplane = startplane-1  ! Offset to be removed for output subcubes
    else
      iter%offsetplane = 0
    endif
    !
    write(mess,'(7(A,I0))')   &
       'Split job using ',iter%nt,' tasks for ',iter%ne,' entries (',  &
       ceiling(iter%nb),' blocks, up to ',  &
       iter%neperblock,' entries per block, up to ',  &
       iter%nepertask,' entries per task, up to ',  &
       ceiling(iter%ntperblock),' tasks per block)'
    call cubeadm_message(admseve%others,rname,mess)
    !
    !$ if (iter%ntperblock.le.1.0) then
    !$   call cubeadm_message(seve%d,rname,'Parallelisation is ineffective here')
    !$ endif
    !
    ! Reset iteration before real use
    call cubeadm_taskloop_reset(iter,error)
    if (error)  return
  end subroutine cubeadm_taskloop_init
  !
  subroutine cubeadm_taskloop_iterator(cub,startplane,stopplane,iter,error)
    !----------------------------------------------------------------------
    ! Set up the iterator suited for a single cube.
    !----------------------------------------------------------------------
    type(cube_t),             intent(in)    :: cub
    integer(kind=data_k),     intent(in)    :: startplane  ! Input range to be iterated
    integer(kind=data_k),     intent(in)    :: stopplane   ! Input range to be iterated
    type(cubeadm_iterator_t), intent(out)   :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>ITERATOR'
    logical :: parallel
    integer(kind=chan_k) :: nchanperblock
    integer(kind=pixe_k) :: nyperblock
    integer(kind=data_k) :: nplaneperblock
    real(kind=4) :: totalsize,blocksize
    !
    call cubeadm_message(admseve%trace,rname,'Welcome')
    !
    ! Block size. 2 possibilities: MEMORY mode or DISK mode.
    ! What if AUTOMATIC mode? => falls into same division as DISK mode,
    ! which is acceptable
    totalsize = cub%tuple%current%size()  ! [Bytes]
    select case (cub%tuple%current%desc%buffered)
    case (code_buffer_memory)
      ! In MEMORY mode, block size is all the file
      blocksize = totalsize  ! [Bytes]
    case (code_buffer_disk)
      ! In DISK mode, block size is ruled by SET BUFFER BLOCK
      blocksize = cub%user%buff%block  ! [Bytes]
    case default
      call cubeadm_message(seve%e,rname,'Unexpected cube buffering')
      error = .true.
      return
    end select
    !
    ! Which access?
    select case (cub%access())
    case (code_access_subset,code_access_fullset)
      iter%access = cub%access()
    case (code_access_blobset)
      call cubeadm_message(seve%e,rname,'Blob access mode is not implemented')
      error = .true.
      return
    case default
      ! Other accesses based on actual data order
      select case (cub%order())
      case (code_access_imaset)
        iter%access = code_access_imaset
      case (code_access_speset)
        iter%access = code_access_speset
      case default
        call cubeadm_message(seve%e,rname,'Unsupported access mode')
        error = .true.
        return
      end select
    end select
    !
    ! Entries
    select case (iter%access)
    case (code_access_subset)
      call cubeio_max_any_block(cub%user,cub%tuple%current,blocksize,  &
        'SET\BUFFER /BLOCK',nplaneperblock,error)
      if (error)  return
      iter%ndginentry = cub%tuple%current%desc%n1*cub%tuple%current%desc%n2  ! ZZZ method ndata_per_plane
      iter%ndginblock = iter%ndginentry
      iter%ndperblock = nplaneperblock * iter%ndginblock
      iter%np         = cub%tuple%current%desc%n3
      iter%npperblock = nplaneperblock
      iter%ndperplane = cub%tuple%current%desc%n1 * cub%tuple%current%desc%n2
    case (code_access_fullset)
      iter%ndginentry = cub%tuple%current%desc%n1 *  &
                        cub%tuple%current%desc%n2 *  &
                        cub%tuple%current%desc%n3
      iter%ndginblock = iter%ndginentry
      iter%ndperblock = cub%tuple%current%desc%n3 * iter%ndginblock
      iter%np         = cub%tuple%current%desc%n3
      iter%npperblock = cub%tuple%current%desc%n3  ! The trick is here: all planes => whole cube
      iter%ndperplane = cub%tuple%current%desc%n1 * cub%tuple%current%desc%n2
    case (code_access_imaset)
      call cubeio_max_chan_block(cub%user,cub%tuple%current,blocksize,  &
        'SET\BUFFER /BLOCK',nchanperblock,error)
      if (error)  return
      iter%ndginentry = cub%tuple%current%desc%nx*cub%tuple%current%desc%ny  ! ZZZ method ndata_per_image
      iter%ndginblock = iter%ndginentry
      iter%ndperblock = nchanperblock * iter%ndginblock
      iter%np         = cub%tuple%current%desc%nc
      iter%npperblock = nchanperblock
      iter%ndperplane = cub%tuple%current%desc%nx * cub%tuple%current%desc%ny
    case (code_access_speset)
      call cubeio_max_y_block(cub%user,cub%tuple%current,blocksize,  &
        'SET\BUFFER /BLOCK',nyperblock,error)
      if (error)  return
      iter%ndginentry = cub%tuple%current%desc%nc
      iter%ndginblock = iter%ndginentry * cub%tuple%current%desc%nx
      iter%ndperblock = nyperblock * iter%ndginblock
      iter%np         = cub%tuple%current%desc%ny
      iter%npperblock = nyperblock
      iter%ndperplane = cub%tuple%current%desc%nc * cub%tuple%current%desc%nx
    end select
    blocksize = iter%ndperblock * cub%nbytes()  ! Re-rounding for correct granularity
    iter%nb = totalsize/blocksize  ! Floating point (feedback only)
    !
    ! Number of tasks per IO block
    parallel = .false.
    !$ parallel = .true.
    if (parallel) then
      ! Parallel mode: entries are divided in tasks of size SET BUFFER PARA
      iter%ntperblock = blocksize/cub%user%buff%para  ! Floating point, rounding comes after
      if (iter%ntperblock.lt.1.0) then
        ! This can happen if:
        !  1) buff%para > buff%block (wrong user settings?)
        !  2) buff%para > totalsize
        iter%ntperblock = 1.0  ! This means no parallel threads!
      endif
    else
      ! Serial mode: one task per block
      iter%ntperblock = 1.0
    endif
    !
    ! Tasking
    iter%nd = cub%ndata()
    select case (iter%access)
    case (code_access_subset)
      ! Entries are sized to maximum, tasks deal with 1 entry each
      iter%nepertask = 1
      ! Block is divided in ntperblock, each entry must be a multiple of ndginentry
      iter%nppertask  = ceiling(iter%npperblock/iter%ntperblock)
      iter%ndperentry = iter%nppertask*iter%ndginentry
      iter%neperblock = (iter%ndperblock-1)/iter%ndperentry+1  ! Should be equal to ntperblock or so
    case (code_access_fullset)
      ! One unique entry, one unique task
      iter%nepertask = 1
      iter%nppertask  = iter%npperblock
      iter%ndperentry = iter%ndginentry
      iter%neperblock = 1
    case (code_access_imaset,code_access_speset)
      ! Entries are sized to minimum, tasks deal with several entries each
      iter%nppertask  = ceiling(iter%npperblock/iter%ntperblock)
      iter%ndperentry = iter%ndginentry  ! No extra factor = minimum allowed size
      iter%neperblock = iter%ndperblock/iter%ndperentry  ! Should be exact integer division
      iter%nepertask = ceiling(iter%neperblock/iter%ntperblock)  ! ceiling() because task size is a minimum
    end select
    iter%ndpertask = iter%ndperentry * iter%nepertask
    iter%neperplane = iter%ndperplane/iter%ndperentry
    iter%npperentry = iter%ndperentry/iter%ndperplane
    !
    ! Apply subset to be iterated if relevant
    if (startplane.ne.nullplane) then
      iter%startplane = startplane
    else
      iter%startplane = 1
    endif
    iter%offsetplane = iter%startplane-1  ! Offset to be removed for put subcubes
    if (stopplane.ne.nullplane) then
      iter%stopplane = stopplane
    else
      iter%stopplane = iter%np
    endif
    !
    ! Total number of tasks and entries. Can not guess by simple
    ! divisions. Apply iterator for exact value.
    call cubeadm_taskloop_reset(iter,error)
    if (error)  return
    do while (cubeadm_taskloop_compute(iter,error))
      ! Reallocation in the loop because final size is unknown (that's the purpose here)
      call cubeadm_taskloop_iterator_reallocate(iter,iter%ct,error)
      if (error)  return
      iter%firstplanes(iter%ct)  = iter%firstplane
      iter%lastplanes(iter%ct)   = iter%lastplane
      iter%firstentries(iter%ct) = iter%first
      iter%lastentries(iter%ct)  = iter%last
    enddo
    iter%nt = iter%ct
    iter%ne = iter%last
    !
  end subroutine cubeadm_taskloop_iterator
  !
  subroutine cubeadm_taskloop_iterator_reallocate4(iter,rsize,error)
    use gkernel_interfaces
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    type(cubeadm_iterator_t), intent(inout) :: iter
    integer(kind=4),          intent(in)    :: rsize  ! Requested size
    logical,                  intent(inout) :: error
    call cubeadm_taskloop_iterator_reallocate8(iter,int(rsize,kind=8),error)
    if (error)  return
  end subroutine cubeadm_taskloop_iterator_reallocate4
  !
  subroutine cubeadm_taskloop_iterator_reallocate8(iter,rsize,error)
    use gkernel_interfaces
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    type(cubeadm_iterator_t), intent(inout) :: iter
    integer(kind=8),          intent(in)    :: rsize  ! Requested size
    logical,                  intent(inout) :: error
    !
    character(len=*), parameter :: rname='TASKLOOP>ITERATOR>REALLOCATE'
    integer(kind=4) :: osize,nsize,ier
    logical :: olddata
    integer(kind=4), parameter :: iter_min_alloc=10
    integer(kind=entr_k), allocatable :: fp(:),lp(:),fe(:),le(:)
    !
    olddata = allocated(iter%firstplanes)
    if (olddata) then
      osize = size(iter%firstplanes)
    else
      osize = 0
    endif
    if (osize.ge.rsize)  return
    !
    nsize = max(rsize,iter_min_alloc)
    nsize = max(nsize,2*osize)
    !
    if (olddata) then
      call move_alloc(from=iter%firstplanes,to=fp)
      call move_alloc(from=iter%lastplanes,to=lp)
      call move_alloc(from=iter%firstentries,to=fe)
      call move_alloc(from=iter%lastentries,to=le)
      ! => Arrays are now deallocated
    endif
    !
    allocate(iter%firstplanes(nsize),  &
             iter%lastplanes(nsize),  &
             iter%firstentries(nsize),  &
             iter%lastentries(nsize),  &
             stat=ier)
    if (failed_allocate(rname,'iterator lists',ier,error))  return
    !
    if (olddata) then
      iter%firstplanes(1:osize)  = fp(1:osize)
      iter%lastplanes(1:osize)   = lp(1:osize)
      iter%firstentries(1:osize) = fe(1:osize)
      iter%lastentries(1:osize)  = le(1:osize)
      deallocate(fp,lp,fe,le)
    endif
  end subroutine cubeadm_taskloop_iterator_reallocate8
  !
  subroutine cubeadm_taskloop_reset(iter,error)
    !-------------------------------------------------------------------
    ! Reset iterator before iterating with cubeadm_taskloop_iterate
    !-------------------------------------------------------------------
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    !
    iter%ct = 0
    iter%firstdata = 0
    iter%lastdata = 0
    iter%firstplane = 0
    iter%lastplane = iter%startplane-1  ! So that 1st iteration starts at correct position
    iter%first = 0
    iter%last = 0
  end subroutine cubeadm_taskloop_reset
  !
  function cubeadm_taskloop_compute(iter,error)
    !----------------------------------------------------------------------
    ! Iterate the iterator for next range of entries to be processed.
    ! Return .false. if no more data is to be processed.
    !----------------------------------------------------------------------
    logical :: cubeadm_taskloop_compute
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>COMPUTE'
    !
    if (.true.) then ! ZZZ select case (iter%access)
      ! Plane-iterating mode
      cubeadm_taskloop_compute = cubeadm_taskloop_compute_planes(iter,error)
      if (error)  return
    else
      ! Data-iterating mode
      cubeadm_taskloop_compute = cubeadm_taskloop_compute_data(iter,error)
      if (error)  return
    endif
  end function cubeadm_taskloop_compute
  !
  function cubeadm_taskloop_compute_planes(iter,error)
    !----------------------------------------------------------------------
    ! Iterate the iterator for next range of PLANES to be processed.
    ! This mode is mostly suited to iterate cubes of same 3rd dimension (2
    ! first dimensions might differ).
    !
    ! Return .false. if no more data is to be processed.
    !----------------------------------------------------------------------
    logical :: cubeadm_taskloop_compute_planes
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>COMPUTE>PLANES'
    integer(kind=4) :: iblock,nplane
    character(len=message_length) :: mess
    !
    ! No welcome on purpose here!
    if (iter%lastplane.ge.iter%stopplane) then
      ! Last entry iteration was processed on previous iteration: all done
      cubeadm_taskloop_compute_planes = .false.
      return
    endif
    !
    ! Current task
    iter%ct = iter%ct+1
    !
    ! Planes
    iter%firstplane = iter%lastplane+1
    if (iter%firstplane.ge.1) then
      ! Iblock 1, 2, 3, ... (in the data)
      iblock = (iter%firstplane-1)/iter%npperblock+1
    else
      ! Iblock 0, -1, -2, ... (off the data)
      iblock = -((-iter%firstplane)/iter%npperblock)
    endif
    iter%lastplane = iter%firstplane+iter%nppertask-1
    ! Range can not extend beyond current block:
    iter%lastplane = min(iter%lastplane,iblock*iter%npperblock)
    ! Last block can be smaller than the other ones, range can not extend
    ! beyond stop plane
    iter%lastplane = min(iter%lastplane,iter%stopplane)
    ! Can not overlap file boundary
    if (iter%firstplane.le.iter%np) then
      ! First and last planes are <= iter%np (i.e. in the cube)
      iter%lastplane = min(iter%lastplane,iter%np)
    else
      ! First and last planes are > iter%np (i.e. off the cube)
      continue
    endif
    !
    nplane = iter%lastplane-iter%firstplane+1
    !
    ! Entries.
    ! NB: the meaning of the entry numbering is unclear in the
    !     startplane/stopplane scheme
    ! ZZZ It is quite unsatisfying entries are recomputed by themselves.
    ! iter%first and iter%last should be removed at some point.
    iter%first = iter%last+1
    if (iter%neperplane.ge.1) then
      iter%last = iter%last + nplane*iter%neperplane
    else
      iter%last = iter%last + (nplane-1)/iter%npperentry+1
    endif
    !
    cubeadm_taskloop_compute_planes = .true.
    !
    write(mess,'(8(A,I0))')  'Computing block ',iblock,  &
      ', entries ',iter%first,' to ',iter%last,  &
      ' (',iter%last-iter%first+1,' entries)'//  &
      ', planes ',iter%firstplane,' to ',iter%lastplane,  &
      ' (',iter%lastplane-iter%firstplane+1,' planes)'
    call cubeadm_message(admseve%others,rname,mess)
    !
  end function cubeadm_taskloop_compute_planes
  !
  function cubeadm_taskloop_compute_data(iter,error)
    !----------------------------------------------------------------------
    ! Iterate the iterator for next range of DATA to be processed.
    ! This mode is mostly suited to iterate cubes of same total size (same
    ! dimensions) which have a bit-to-bit symmetry.
    !
    ! Return .false. if no more data is to be processed.
    !----------------------------------------------------------------------
    logical :: cubeadm_taskloop_compute_data
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>COMPUTE>DATA'
    integer(kind=4) :: iblock
!     integer(kind=data_k) :: ndata,ne
!     character(len=message_length) :: mess
    !
    ! No welcome on purpose here!
    if (iter%lastdata.ge.iter%nd) then
      ! Last data iteration was processed on previous iteration: all done
      cubeadm_taskloop_compute_data = .false.
      return
    endif
    !
    iter%firstdata = iter%lastdata  + 1
    iter%lastdata  = iter%firstdata + iter%ndpertask - 1
    ! Range can not extend beyond current block
    iblock = (iter%firstdata-1)/iter%ndperblock+1
    iter%lastdata = min(iter%lastdata,iblock*iter%ndperblock)
    ! Last block can be smaller than the other ones, range can not extend beyond file:
    iter%lastdata = min(iter%lastdata,iter%nd)
    !
    cubeadm_taskloop_compute_data = .true.
    !
    ! Not relevant is this context:
!   ! Convert firstdata and lastdata to entry numbers
!   ndata = iter%lastdata - iter%firstdata + 1
!   ne    = (ndata-1)/iter%ndperentry + 1 ! Exact integer value, except for last block
!   iter%firstplane = iter%lastplane + 1
!   iter%lastplane  = iter%lastplane + ne
!   iter%first = iter%firstplane
!   iter%last  = iter%lastplane
    !
    ! Debugging
!     write(mess,'(7(A,I0))')  'Processing block ',iblock,  &
!       ', data ',iter%firstdata,' to ',iter%lastdata,  &
!       ', entries ',iter%first,' to ',iter%last,  &
!       ' (',ne,' entries)'
!     call cubeadm_message(seve%i,rname,mess)
    !
  end function cubeadm_taskloop_compute_data
  !
  subroutine cubeadm_taskloop_merge(iter,niter,merged,error)
    use gkernel_interfaces
    !-------------------------------------------------------------------
    ! Merge all the iterators to produce the final iterator
    !-------------------------------------------------------------------
    integer(kind=entr_k),     intent(in)    :: niter
    type(cubeadm_iterator_t), intent(in)    :: iter(niter)
    type(cubeadm_iterator_t), intent(out)   :: merged
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>MERGE'
    integer(kind=entr_k) :: iiter,firstplane,currentplane,currententry
    integer(kind=4) :: mtasks,itask,jtask,jiter,stask,ier
    integer(kind=entr_k), allocatable :: firstplanes(:),tasknum(:)
    integer(kind=4), allocatable :: iternum(:)
    integer(kind=4), allocatable :: sort(:)
    logical :: renumber,disabled(niter)
    !
    ! First check if we are mixing several-planes with one-plane cubes.
    ! One-plane cubes are not involved in the merged iterator.
    do iiter=1,niter
      disabled(iiter) = iter(iiter)%np.eq.1
    enddo
    if (all(disabled))  disabled(:) = .false.  ! All are one-plane cubes, we
                                               ! are just mixing consistent data
    !
    ! Make a complete list of all plane ranges to be iterated
    ! 1) How many ranges?
    mtasks = 0
    do iiter=1,niter
      if (disabled(iiter))  cycle
      mtasks = mtasks + iter(iiter)%nt
    enddo
    !
    ! 2) Collect the first planes
    allocate(firstplanes(mtasks),iternum(mtasks),tasknum(mtasks),stat=ier)
    if (failed_allocate(rname,'merging buffers',ier,error)) return
    itask = 0
    do iiter=1,niter
      if (disabled(iiter))  cycle
      do jtask=1,iter(iiter)%nt
        itask = itask+1
        iternum(itask) = iiter  ! Iterator number
        tasknum(itask) = jtask  ! Task number in its iterator
        firstplanes(itask) = iter(iiter)%firstplanes(jtask)+iter(iiter)%offsetplane
      enddo
    enddo
    !
    ! Debug
!     print *,""
!     print *,"=== First planes (in the input cubes referential) ==="
!     do itask=1,mtasks
!       print *,itask,firstplanes(itask)
!     enddo
    !
    ! 3) Get sorting array by ordering first planes
    allocate(sort(mtasks),stat=ier)
    if (failed_allocate(rname,'sort',ier,error)) return
    call gi8_trie(firstplanes,sort,mtasks,error)
    if (error)  return
    !
    ! 4) Fill the output iterator using the sorting array
    !   and removing duplicates
    currentplane = huge(currentplane)
    renumber = iter(1)%access.eq.code_access_subset  .or.  &
               iter(1)%access.eq.code_access_fullset  ! Renumber entries?
    currententry = 0
    merged%ct = 0
    do itask=1,mtasks
      stask = sort(itask)
      jiter = iternum(stask)
      jtask = tasknum(stask)
      !
      firstplane = iter(jiter)%firstplanes(jtask)+iter(jiter)%offsetplane
      if (firstplane.eq.currentplane)  cycle  ! Remove duplicates
      currentplane = firstplane
      !
      merged%ct = merged%ct+1
      call cubeadm_taskloop_iterator_reallocate(merged,merged%ct,error)
      if (error)  return
      merged%firstplanes(merged%ct)  = iter(jiter)%firstplanes(jtask)+iter(jiter)%offsetplane
      if (renumber) then
        currententry = currententry+1  ! One per subcube
        merged%firstentries(merged%ct) = currententry
      else
        merged%firstentries(merged%ct) = iter(jiter)%firstentries(jtask)
      endif
      !
      if (merged%ct.gt.1) then
        merged%lastplanes(merged%ct-1)  = merged%firstplanes(merged%ct)-1
        merged%lastentries(merged%ct-1) = merged%firstentries(merged%ct)-1
      endif
    enddo
    merged%lastplanes(merged%ct)  = iter(jiter)%lastplanes(jtask)+iter(jiter)%offsetplane
    if (renumber) then
      merged%lastentries(merged%ct) = merged%firstentries(merged%ct)  ! One per subcube
    else
      merged%lastentries(merged%ct) = iter(jiter)%lastentries(jtask)
    endif
    merged%nt = merged%ct
    merged%ne = merged%lastentries(merged%ct)
    !
!     print *,"=== Sorted ==="
!     print *,merged%firstplanes(1:merged%nt)
!     print *,merged%lastplanes(1:merged%nt)
!     print *,merged%firstentries(1:merged%nt)
!     print *,merged%lastentries(1:merged%nt)
    !
  end subroutine cubeadm_taskloop_merge
  !
  function cubeadm_taskloop_iterate(iter,error)
    !----------------------------------------------------------------------
    ! Iterate the iterator for next range of entries to be processed.
    ! Return .false. if no more data is to be processed.
    !----------------------------------------------------------------------
    logical :: cubeadm_taskloop_iterate
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>ITERATE'
    !
    if (.true.) then
      ! Plane-iterating mode
      cubeadm_taskloop_iterate = cubeadm_taskloop_iterate_planes(iter,error)
      if (error)  return
    else
      ! Data-iterating mode
      ! cubeadm_taskloop_iterate = cubeadm_taskloop_iterate_data(iter,error)
      ! if (error)  return
    endif
  end function cubeadm_taskloop_iterate
  !
  function cubeadm_taskloop_iterate_planes(iter,error)
    !----------------------------------------------------------------------
    ! Iterate the iterator for next range of PLANES to be processed.
    ! This mode is mostly suited to iterate cubes of same 3rd dimension (2
    ! first dimensions might differ).
    !
    ! Return .false. if no more data is to be processed.
    !----------------------------------------------------------------------
    logical :: cubeadm_taskloop_iterate_planes
    type(cubeadm_iterator_t), intent(inout) :: iter
    logical,                  intent(inout) :: error
    ! Local
    character(len=*), parameter :: rname='TASKLOOP>ITERATE>PLANES'
    character(len=mess_l) :: mess
    !
    ! No welcome on purpose here!
    if (iter%ct.ge.iter%nt) then
      ! Last task was processed on previous iteration: all done
      cubeadm_taskloop_iterate_planes = .false.
      return
    endif
    !
    iter%ct = iter%ct+1
    iter%firstplane = iter%firstplanes(iter%ct)
    iter%lastplane  = iter%lastplanes(iter%ct)
    iter%first      = iter%firstentries(iter%ct)
    iter%last       = iter%lastentries(iter%ct)
    !
    cubeadm_taskloop_iterate_planes = .true.
    !
    write(mess,'(7(A,I0))')  'Iterating entries ', &
      iter%first,' to ',iter%last,  &
      ' (',iter%last-iter%first+1,' entries)'//  &
      ', planes ',iter%firstplane,' to ',iter%lastplane,  &
      ' (',iter%lastplane-iter%firstplane+1,' planes)'
    call cubeadm_message(admseve%others,rname,mess)
  end function cubeadm_taskloop_iterate_planes
  !
  function cubeadm_taskloop_idata2plane(iter,cube,pos1d)
    !-------------------------------------------------------------------
    ! Convert a data position (1D) into a plane number (3rd dimension)
    ! in the given cube
    !-------------------------------------------------------------------
    integer(kind=data_k) :: cubeadm_taskloop_idata2plane
    class(cubeadm_iterator_t), intent(in) :: iter  ! ZZZ Not needed here...
    type(cube_t),              intent(in) :: cube
    integer(kind=data_k),      intent(in) :: pos1d
    !
    integer(kind=data_k) :: n1,n2
    n1 = cube%tuple%current%desc%n1
    n2 = cube%tuple%current%desc%n2
    cubeadm_taskloop_idata2plane = (pos1d-1)/(n1*n2)+1
  end function cubeadm_taskloop_idata2plane
  !
  subroutine cubeadm_taskloop_iterator_debug(iter)
    type(cubeadm_iterator_t), intent(in) :: iter
    !
    print *,""
    print *,"nd         = ",iter%nd
    print *,"ndperentry = ",iter%ndperentry
    print *,"ndperplane = ",iter%ndperplane
    print *,"ndpertask  = ",iter%ndpertask
    print *,"ndperblock = ",iter%ndperblock
    print *,"ndginentry = ",iter%ndginentry
    print *,"ndginblock = ",iter%ndginblock

    print *,"ne         = ",iter%ne
    print *,"neperblock = ",iter%neperblock
    print *,"nepertask  = ",iter%nepertask
    print *,"neperplane = ",iter%neperplane

    print *,"np         = ",iter%np
    print *,"nppertask  = ",iter%nppertask
    print *,"npperblock = ",iter%npperblock
    print *,"npperentry = ",iter%npperentry

    print *,"nt         = ",iter%nt
    print *,"ntperblock = ",iter%ntperblock

    print *,"nb         = ",iter%nb
    !
  end subroutine cubeadm_taskloop_iterator_debug
  !
end module cubeadm_taskloop
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
