!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_extract
  use cubetools_structure
  use cube_types
  use cubeadm_cubeid_types
  use cubemain_messaging
  use cubemain_windowing
  use cubemain_speline_types
  use cubemain_sperange_types
  use cubemain_sparange_types
  use cubemain_spapos_types
  use cubemain_spasize_types
  !
  public :: extract
  public :: cubemain_extract_command
  private
  !
  type :: extract_comm_t
     type(option_t), pointer :: comm
     type(sperange_opt_t)    :: range
     type(speline_opt_t)     :: freq
     type(spapos_opt_t)      :: center
     type(spasize_opt_t)     :: size
   contains
     procedure, public  :: register => cubemain_extract_register
     procedure, private :: parse    => cubemain_extract_parse
     procedure, private :: main     => cubemain_extract_main
  end type extract_comm_t
  type(extract_comm_t) :: extract
  !
  integer(kind=4), parameter :: icube = 1
  type extract_user_t
     type(cubeid_user_t)  :: cubeids
     type(speline_user_t) :: line               ! Optional new line name and freq
     type(sperange_user_t):: range              ! Spectral range to be extracted
     type(spapos_user_t)  :: center             ! [absolute|relative] Center of the region to be extracted
     type(spasize_user_t) :: size               ! Size of the region to be extracted
   contains
     procedure, private :: toprog => cubemain_extract_user_toprog
  end type extract_user_t
  type extract_prog_t
     type(cube_t), pointer :: cube       ! Input cube
     type(cube_t), pointer :: extracted  ! Extracted cube
     logical               :: doline
     type(speline_prog_t)  :: line
     type(sperange_user_t) :: sperange
     integer(kind=ndim_k)  :: il,im,ic
     integer(kind=data_k)  :: range(3,2)      ! Extracted ranges
   contains
     procedure, private :: header          => cubemain_extract_prog_header
     procedure, private :: header_spectral => cubemain_extract_prog_header_spectral
     procedure, private :: header_spatial  => cubemain_extract_prog_header_spatial
     procedure, private :: data            => cubemain_extract_prog_data     
     procedure, private :: data_loop       => cubemain_extract_prog_data_loop
     procedure, private :: data_subcube    => cubemain_extract_prog_data_subcube
  end type extract_prog_t
  !
contains
  !
  subroutine cubemain_extract_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(extract_user_t) :: user
    character(len=*), parameter :: rname='EXTRACT>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    call extract%parse(line,user,error)
    if (error) return
    call extract%main(user,error)
    if (error) return
  end subroutine cubemain_extract_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_extract_register(extract,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extract_comm_t), intent(inout) :: extract
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    !
    character(len=*), parameter :: comm_abstract = &
         'Extract a subcube'
    character(len=*), parameter :: comm_help = &
         'The velocity range may be around the current reference&
         & frequency for the cube, or it can be around a new&
         & reference frequency (/FREQUENCY).'
    character(len=*), parameter :: rname='EXTRACT>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'EXTRACT','[cube]',&
         comm_abstract,&
         comm_help,&
         cubemain_extract_command,&
         extract%comm,error)
    if (error) return
    call cubearg%register( &
         'CUBE', &
         'Signal cube',  &
         strg_id,&
         code_arg_optional,  &
         [flag_cube], &
         error)
    if (error) return
    !
    call extract%range%register('RANGE',&
         'Spectral range of the extracted region',&
         error)
    if (error) return
    !
    call extract%freq%register(&
         'Line name & rest frequency of the extracted region',&
         error)
    if (error) return
    !
    call extract%center%register('CENTER',&
         'Spatial center of the extracted region',&
         error)
    if (error) return
    !
    call extract%size%register(&
         'Spatial size of the extracted region',&
         error)
    if (error) return
  end subroutine cubemain_extract_register
  !
  subroutine cubemain_extract_parse(extract,line,user,error)
    !----------------------------------------------------------------------
    ! EXTRACT cubname
    ! /RANGE vfirst vlast
    ! /FREQUENCY newname newrestfreq [unit]
    ! /CENTER xcen ycen ! can be relative[arcsec] or absolute[RA,DEC or LII,BII]
    ! /SIZE sx [sy]
    !----------------------------------------------------------------------
    class(extract_comm_t), intent(in)    :: extract
    character(len=*),      intent(in)    :: line
    type(extract_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='EXTRACT>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,extract%comm,user%cubeids,error)
    if (error) return
    call extract%range%parse(line,user%range,error)
    if (error) return
    call extract%freq%parse(line,user%line,error)
    if (error) return
    call extract%center%parse(line,user%center,error)
    if(error) return
    call extract%size%parse(line,user%size,error)
    if (error) return
    if (user%center%do.and..not.user%size%do) then
       call cubemain_message(seve%e,rname,'A size must be specified when giving a new center')
       error = .true.
       return
    endif
    !
    if (cubetools_nopt().eq.0) then
       call cubemain_message(seve%e,rname,'No options given, nothing to do')
       error = .true.
       return
    endif
  end subroutine cubemain_extract_parse
  !
  subroutine cubemain_extract_main(extract,user,error)
    use cubeadm_timing
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extract_comm_t), intent(in)    :: extract
    type(extract_user_t),  intent(in)    :: user
    logical,               intent(inout) :: error
    !
    type(extract_prog_t) :: prog
    character(len=*), parameter :: rname='EXTRACT>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_extract_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_extract_user_toprog(user,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extract_user_t), intent(in)    :: user
    type(extract_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    integer(kind=chan_k) :: stride
    type(sparange_prog_t) :: lrange,mrange
    type(spapos_prog_t) :: center
    integer(kind=ndim_k), parameter :: il=1,im=2
    character(len=*), parameter :: rname='EXTRACT>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(extract%comm,icube,user%cubeids,  &
         code_access_subset,code_read,prog%cube,error)
    if (error) return
    !
    select case (prog%cube%order())
    case (code_cube_imaset)
       prog%il = 1
       prog%im = 2
       prog%ic = 3
    case (code_cube_speset)
       prog%il = 2
       prog%im = 3
       prog%ic = 1
    case default
       call cubemain_message(seve%e,rname,'Order not supported')
       error = .true.
       return
    end select
    !
    prog%doline = user%line%do
    if (prog%doline) then
       call user%line%toprog(prog%cube,prog%line,error)
       if (error) return
    endif
    ! User range is copied because range resolution has to be done
    ! after frequency modification
    prog%sperange = user%range
    !
    call user%center%toprog(prog%cube,center,error)
    if (error) return
    call lrange%fromuser(prog%cube%head%set%il,prog%cube,center%rela(il),&
         user%size%x,user%size%unit,prog%cube%head%spa%l%inc,&
         prog%cube%head%spa%l%kind,error)
    if (error) return
    call lrange%to_pixe_k(prog%range(prog%il,1),prog%range(prog%il,2),stride,error)
    if (error) return
    call mrange%fromuser(prog%cube%head%set%im,prog%cube,center%rela(im),&
         user%size%y,user%size%unit,prog%cube%head%spa%m%inc,&
         prog%cube%head%spa%m%kind,error)
    if (error) return
    call mrange%to_pixe_k(prog%range(prog%im,1),prog%range(prog%im,2),stride,error)
    if (error) return
  end subroutine cubemain_extract_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_extract_prog_header(prog,error)
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extract_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='EXTRACT>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%cube,[flag_extract,flag_cube],  &
      prog%extracted,error)
    if (error) return
    call prog%header_spectral(error)
    if (error) return
    call prog%header_spatial(error)
    if (error) return
  end subroutine cubemain_extract_prog_header
  !
  subroutine cubemain_extract_prog_header_spectral(prog,error)
    use cubetools_unit
    use cubetools_axis_types
    use cubetools_header_methods
    !----------------------------------------------------------------------
    ! *** JP *** Think twice before modifying this one!
    !----------------------------------------------------------------------
    class(extract_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(axis_t) :: axis
    integer(kind=chan_k) :: stride
    type(sperange_prog_t) :: sperange
    character(len=*), parameter :: rname='EXTRACT>PROG>HEADER>SPECTRAL'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    ! Everything below is needed!
    if (prog%doline) then
       call cubetools_header_modify_rest_frequency(prog%line%freq,prog%extracted%head,error)
       if (error) return
       call cubetools_header_put_line(prog%line%name,prog%extracted%head,error)
       if (error) return
       call cubetools_header_get_axis_head_f(prog%extracted%head,axis,error)
       if (error) return
       call cubetools_header_update_frequency_from_axis(axis,prog%extracted%head,error)
       if (error) return
    endif
    !
    ! This part of the code has to be executed after the frequency
    ! modification otherwise the range of velocities is wrong!
    call prog%sperange%toprog(prog%extracted,sperange,error)
    if (error) return
    call sperange%to_chan_k(prog%range(prog%ic,1),prog%range(prog%ic,2),stride,error)
    if (error) return
    !
    call cubetools_header_get_axis_head_c(prog%extracted%head,axis,error)
    if (error) return
    axis%n = prog%range(prog%ic,2)-prog%range(prog%ic,1)+1
    axis%ref = axis%ref-prog%range(prog%ic,1)+1
    !
    ! This snippet of code was copied from: lib/edit/type-cube-buffer.f90:cubeedit_cube_buffer_resize
    ! This points to the necessity of factorizing this piece of code in lib/tools/header-methods.f90x
    if (axis%kind.eq.code_unit_freq) then
       call cubetools_header_update_frequency_from_axis(axis,prog%extracted%head,error)
       if (error) return
    else if (axis%kind.eq.code_unit_velo) then
       call cubetools_header_update_velocity_from_axis(axis,prog%extracted%head,error)
       if (error) return
    else
       call cubetools_header_update_axset_c(axis,prog%extracted%head,error)
       if (error) return
       call cubemain_message(seve%w,rname,'Unknown kind of Spectral axis')
    endif
  end subroutine cubemain_extract_prog_header_spectral
  !
  subroutine cubemain_extract_prog_header_spatial(prog,error)
    use cubetools_axis_types
    use cubetools_header_methods
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extract_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(axis_t) :: laxis, maxis
    character(len=*), parameter :: rname='EXTRACT>PROG>HEADER>SPATIAL'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_header_get_axis_head_l(prog%extracted%head,laxis,error)
    if (error) return
    laxis%n = prog%range(prog%il,2)-prog%range(prog%il,1)+1
    laxis%ref = laxis%ref-1d0*prog%range(prog%il,1)+1d0
    call cubetools_header_update_axset_l(laxis,prog%extracted%head,error)
    if (error) return
    !
    call cubetools_header_get_axis_head_m(prog%extracted%head,maxis,error)
    if (error) return
    maxis%n = prog%range(prog%im,2)-prog%range(prog%im,1)+1
    maxis%ref = maxis%ref-1d0*prog%range(prog%im,1)+1d0
    call cubetools_header_update_axset_m(maxis,prog%extracted%head,error)
    if (error) return
  end subroutine cubemain_extract_prog_header_spatial
  !
  subroutine cubemain_extract_prog_data(prog,error)
    use cubeadm_opened
    use cubetools_header_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extract_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='EXTRACT>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,prog%range(3,1),prog%range(3,2),error)
    if (error) return
    !
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) then
          call prog%data_loop(iter,error)
       endif
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_extract_prog_data
  !
  subroutine cubemain_extract_prog_data_loop(prog,iter,error)
    use cubeadm_entryloop
    use cubeadm_taskloop
    use cubemain_subcube_real
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    class(extract_prog_t),    intent(inout) :: prog
    type(cubeadm_iterator_t), intent(in)    :: iter
    logical,                  intent(inout) :: error
    !
    type(subcube_iterator_t) :: siter
    integer(kind=entr_k) :: isubcube
    character(len=*), parameter :: rname='EXTRACT>PROG>DATA_LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    do isubcube=iter%first,iter%last
      call cubeadm_entryloop_iterate(isubcube,error)
      if (error)  return
      call siter%init(iter,isubcube,error)
      if (error)  return
      call prog%data_subcube(siter,error)
      if (error)  return
    enddo
    !
  end subroutine cubemain_extract_prog_data_loop
  !
  subroutine cubemain_extract_prog_data_subcube(prog,iter,error)
    use cubetools_nan
    use cubeadm_taskloop
    use cubemain_subcube_real
    !-------------------------------------------------------------------
    ! Extract from 1 subcube
    !-------------------------------------------------------------------
    class(extract_prog_t),    intent(inout) :: prog
    type(subcube_iterator_t), intent(in)    :: iter
    logical,                  intent(inout) :: error
    !
    type(subcube_t) :: insub,ousub
    integer(kind=data_k) :: i1,j1,i1min,i1max,i1off
    integer(kind=data_k) :: i2,j2,i2min,i2max,i2off
    integer(kind=data_k) :: i3,j3,i3min,i3max
    logical :: overlap
    character(len=*), parameter :: rname='EXTRACT>PROG>DATA>SUBCUBE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    ! Subcubes are initialized here as their size (3rd dim) may change from
    ! from one subcube to another.
    call insub%init(prog%cube,error)
    if (error)  return
    call ousub%reallocate(prog%extracted,iter,error)
    if (error)  return
    !
    ! Sanity check
    !
    call insub%get(prog%cube,iter,error)
    if (error) return
    !
    overlap = .false.
    !
    i1off = prog%range(1,1)-1
    i1min = prog%range(1,1)
    if (i1min.lt.1) then
      overlap = .true.
      i1min = 1
    endif
    i1max = prog%range(1,2)
    if (i1max.gt.insub%n1) then
      overlap = .true.
      i1max = insub%n1
    endif
    !
    i2off = prog%range(2,1)-1
    i2min = prog%range(2,1)
    if (i2min.lt.1) then
      overlap = .true.
      i2min = 1
    endif
    i2max = prog%range(2,2)
    if (i2max.gt.insub%n2) then
      overlap = .true.
      i2max = insub%n2
    endif
    !
    ! 3rd dimension: by design, subcubes are completely in or completely
    ! off the input cube
    i3min = 1
    i3max = insub%n3
    if (i3min.gt.i3max)  overlap = .true.
    !
    ! In case of surset extraction, initialize the new bits to NaN.
    ! For simplicity, initialize everything. NaN will be overwritten by
    ! valid values afterwards.
    if (overlap) then
      call ousub%initval(gr4nan,error)
      if (error)  return
    else
       continue ! No overlap cube boundaries, no init"
    endif
    !
    do i3=i3min,i3max
      j3 = i3  ! For the 3rd dimension, the shift is operated by ousub%put()
      !
      do i2=i2min,i2max
        j2 = i2-i2off
        !
        do i1=i1min,i1max
          j1 = i1-i1off
          ousub%data(j1,j2,j3) = insub%data(i1,i2,i3)
        enddo
      enddo
    enddo
    !
    call ousub%put(prog%extracted,iter,error)
    if (error) return
    !
  end subroutine cubemain_extract_prog_data_subcube
  !
end module cubemain_extract
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
