module cubemain_extrema_types
  use cubemain_messaging
  use cube_types
  !
  public extrema_desc_t
  private
  !
  integer(kind=ndim_k), parameter :: ndim = 3
  integer(kind=ndim_k), parameter :: ix = 1
  integer(kind=ndim_k), parameter :: iy = 2
  integer(kind=ndim_k), parameter :: ic = 3 
  !
  type extrema_desc_t
     real(kind=sign_k)     :: min           ! Minimum inside region of interest
     real(kind=sign_k)     :: max           ! Maximum inside region of interest
     !
     integer(kind=data_k),private :: locmin(ndim)  ! Location of the minimum
     integer(kind=data_k),private :: locmax(ndim)  ! Location of the maximum
     integer(kind=data_k),private :: nnan          ! Number of NaNs
     integer(kind=data_k),private :: ndata         ! Number of data
     integer(kind=ndim_k),private :: iaxes(ndim) = [ix,iy,ic] ! Dimension pointers
     integer(kind=data_k),private :: range(ndim,2) ! Ranges
     type(cube_t),pointer,private :: cube          ! Pointer to cube
   contains
     procedure,public  :: compute       => cubemain_extrema_compute
     procedure,public  :: list          => cubemain_extrema_list
     procedure,public  :: def_substruct => cubemain_extrema_def_substruct
     !
     procedure,private :: init          => cubemain_extrema_init
     procedure,private :: getorder      => cubemain_extrema_getorder
     procedure,private :: test_range    => cubemain_extrema_test_range
     !
     procedure,private :: fetch         => cubemain_extrema_fetch
     !
     procedure,private :: data          => cubemain_extrema_data
     procedure,private :: data_loop     => cubemain_extrema_data_loop
     procedure,private :: data_subcube  => cubemain_extrema_data_subcube
     procedure,private :: merge_local   => cubemain_extrema_merge_local
  end type extrema_desc_t
  !
contains
  !
  subroutine cubemain_extrema_list(desc,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extrema_desc_t), intent(in)    :: desc
    logical,               intent(inout) :: error
    !
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='EXTREMA>LIST'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    write(mess,'(15x,a,4x,3(4x,a))') 'Value','Channel','X pixel','Y pixel'
    call cubemain_message(seve%r,rname,mess)
    write(mess,1000) 'Maximum:',desc%max,desc%locmax(desc%iaxes(ic)),&
         desc%locmax(desc%iaxes(ix)),desc%locmax(desc%iaxes(iy))
    call cubemain_message(seve%r,rname,mess)
    write(mess,1000) 'Minimum:',desc%min,desc%locmin(desc%iaxes(ic)),&
         desc%locmin(desc%iaxes(ix)),desc%locmin(desc%iaxes(iy))
    call cubemain_message(seve%r,rname,mess)
    call cubemain_message(seve%r,rname,'')
    write(mess,'(2(i0,a),x,1pg8.3,a)') desc%nnan,' NaNs out of ',desc%ndata,' elements, ',&
         100.0*desc%nnan/desc%ndata,'%'
    call cubemain_message(seve%r,rname,mess)
    call cubemain_message(seve%r,rname,'')
    !
1000 format(a,2x,1pg14.7,3(x,i10))
  end subroutine cubemain_extrema_list
  !
  subroutine cubemain_extrema_compute(desc,cube,xrange,yrange,crange,error)
    use cubemain_sperange_types
    use cubemain_sparange_types
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    class(extrema_desc_t), intent(out)   :: desc
    type(cube_t),target,   intent(inout) :: cube
    type(sparange_prog_t), intent(in)    :: xrange
    type(sparange_prog_t), intent(in)    :: yrange
    type(sperange_prog_t), intent(in)    :: crange
    logical,               intent(inout) :: error
    !
    logical :: dodata
    character(len=*), parameter :: rname='EXTREMA>COMPUTE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    desc%cube => cube
    call desc%test_range(xrange,yrange,crange,dodata,error)
    if (error) return
    !
    if (dodata) then
       call desc%data(error)
       if (error) return
    else
       call desc%fetch(error)
       if (error) return
    endif
  end subroutine cubemain_extrema_compute
  !
  subroutine cubemain_extrema_def_substruct(desc,name,struct,error)
    use cubetools_userstruct
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extrema_desc_t), intent(in)    :: desc
    character(len=*),      intent(in)    :: name
    type(userstruct_t),    intent(inout) :: struct
    logical,               intent(inout) :: error
    !
    type(userstruct_t) :: substruct,min,max
    character(len=*), parameter :: rname='EXTREMA>DEF>SUBSTRUCT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call struct%def_substruct(name,substruct,error)
    if (error) return
    call substruct%def_substruct('min',min,error)
    if (error) return
    call min%set_member('value',desc%min,error)
    if (error) return
    call min%set_member('location',desc%locmin,error)
    if (error) return
    call substruct%def_substruct('max',max,error)
    if (error) return
    call max%set_member('value',desc%max,error)
    if (error) return
    call max%set_member('location',desc%locmax,error)
    if (error) return
  end subroutine cubemain_extrema_def_substruct
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_extrema_init(desc,error)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    class(extrema_desc_t), intent(inout) :: desc
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname = 'EXTREMA>INIT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    desc%min =  huge(desc%min)
    desc%max = -huge(desc%max)
    desc%locmin(:) = 0
    desc%locmax(:) = 0
    desc%nnan = 0
  end subroutine cubemain_extrema_init
  !
  subroutine cubemain_extrema_getorder(desc,error)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    class(extrema_desc_t), intent(inout) :: desc
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='EXTREMA>GETORDER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    select case(desc%cube%order())
    case(code_cube_imaset)
       desc%iaxes = [1,2,3]
    case (code_cube_speset)
       desc%iaxes = [2,3,1]
    case default
       call cubemain_message(mainseve%trace,rname,'Internal error: Cube is neither in imaset or speset')
       error = .true.
       return
    end select
  end subroutine cubemain_extrema_getorder
  !
  subroutine cubemain_extrema_test_range(desc,xrange,yrange,crange,dodata,error)
    use cubetools_nan
    use cubemain_sperange_types
    use cubemain_sparange_types
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    class(extrema_desc_t), intent(inout) :: desc
    type(sparange_prog_t), intent(in)    :: xrange
    type(sparange_prog_t), intent(in)    :: yrange
    type(sperange_prog_t), intent(in)    :: crange
    logical,               intent(out)   :: dodata
    logical,               intent(inout) :: error
    !
    integer(kind=8) :: stride
    logical :: totalx,totaly,totalc,wholecube,bothnan,allnan
    character(len=*), parameter :: rname='EXTREMA>TEST>RANGE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call desc%getorder(error)
    if (error) return
    !
    call xrange%to_pixe_k(desc%range(desc%iaxes(ix),1),desc%range(desc%iaxes(ix),2),stride,error)
    if (error) return
    call yrange%to_pixe_k(desc%range(desc%iaxes(iy),1),desc%range(desc%iaxes(iy),2),stride,error)
    if (error) return
    call crange%to_chan_k(desc%range(desc%iaxes(ic),1),desc%range(desc%iaxes(ic),2),stride,error)
    if (error) return
    !
    ! If requested region goes beyond the cube resize it to the cube
    ! limits
    ! VVV should a warning be raised, I guess not
    if (desc%range(desc%iaxes(ic),1).lt.1) desc%range(desc%iaxes(ic),1) = 1
    if (desc%range(desc%iaxes(ic),2).gt.desc%cube%head%arr%n%c) &
         desc%range(desc%iaxes(ic),2) = desc%cube%head%arr%n%c
    if (desc%range(desc%iaxes(ix),1).lt.1) desc%range(desc%iaxes(ix),1) = 1
    if (desc%range(desc%iaxes(ix),2).gt.desc%cube%head%arr%n%l) &
         desc%range(desc%iaxes(ix),2) = desc%cube%head%arr%n%l
    if (desc%range(desc%iaxes(iy),1).lt.1) desc%range(desc%iaxes(iy),1) = 1
    if (desc%range(desc%iaxes(iy),2).gt.desc%cube%head%arr%n%m) &
         desc%range(desc%iaxes(iy),2) = desc%cube%head%arr%n%m
    !
    ! If we cover the whole x,y,c axes we only need to compute extrema
    ! if they are both NaN
    totalx = desc%range(desc%iaxes(ix),1).eq.1.and.desc%range(desc%iaxes(ix),2).eq.desc%cube%head%arr%n%l
    totaly = desc%range(desc%iaxes(iy),1).eq.1.and.desc%range(desc%iaxes(iy),2).eq.desc%cube%head%arr%n%m
    totalc = desc%range(desc%iaxes(ic),1).eq.1.and.desc%range(desc%iaxes(ic),2).eq.desc%cube%head%arr%n%c
    wholecube = totalc.and.totaly.and.totalx
    bothnan = ieee_is_nan(desc%cube%head%arr%min%val).and.ieee_is_nan(desc%cube%head%arr%max%val)
    allnan  = desc%cube%head%arr%n%nan.eq.desc%cube%head%arr%n%dat 
    !
    if (allnan) then
       dodata = .false.
    else
       if (wholecube.and..not.bothnan) then
          dodata = .false.
       else
          dodata = .true.
       endif
    endif
  end subroutine cubemain_extrema_test_range
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_extrema_fetch(desc,error)
    use cubetools_arrelt_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extrema_desc_t), intent(inout) :: desc
    logical,               intent(inout) :: error
    !
    type(arrelt_t), pointer :: min, max
    character(len=*), parameter :: rname='EXTREMA>FETCH'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    min => desc%cube%head%arr%min
    desc%min                    = min%val
    desc%locmin(desc%iaxes(ix)) = min%ix
    desc%locmin(desc%iaxes(iy)) = min%iy
    desc%locmin(desc%iaxes(ic)) = min%ic
    !
    max => desc%cube%head%arr%max
    desc%max                    = max%val
    desc%locmax(desc%iaxes(ix)) = max%ix
    desc%locmax(desc%iaxes(iy)) = max%iy
    desc%locmax(desc%iaxes(ic)) = max%ic
    !
    desc%nnan  = desc%cube%head%arr%n%nan
    desc%ndata = desc%cube%head%arr%n%dat
  end subroutine cubemain_extrema_fetch
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_extrema_data(global,error)
    use cubeadm_opened
    use cubetools_nan
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(extrema_desc_t), intent(inout) :: global
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    integer(kind=ndim_k) :: idim
    character(len=*), parameter :: rname='EXTREMA>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call global%init(error)
    if (error) return
    global%ndata = 1
    do idim=1,ndim
       global%ndata = global%ndata*(global%range(idim,2)-global%range(idim,1)+1)
    enddo
    !
    call cubeadm_datainit_all(iter,global%range(3,1),global%range(3,2),error)
    if (error) return
    !
    !$OMP PARALLEL DEFAULT(none) SHARED(global,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(global) FIRSTPRIVATE(iter,error)
       if (.not.error) then
          call global%data_loop(iter,error)
       endif
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
    !
    if (global%ndata.eq.global%nnan) then
       call global%init(error)
       if (error) return
       global%min = gr4nan
       global%max = gr4nan
    endif
  end subroutine cubemain_extrema_data
  !
  subroutine cubemain_extrema_data_loop(global,iter,error)
    use cubeadm_opened
    use cubeadm_entryloop
    use cubemain_subcube_real
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    class(extrema_desc_t),    intent(inout) :: global
    type(cubeadm_iterator_t), intent(in)    :: iter
    logical,                  intent(inout) :: error
    !
    type(extrema_desc_t) :: local
    type(subcube_iterator_t) :: siter
    integer(kind=entr_k) :: isubcube
    character(len=*), parameter :: rname='EXTREMA>DATA>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    do isubcube=iter%first,iter%last
      call cubeadm_entryloop_iterate(isubcube,error)
      if (error)  return
      call siter%init(iter,isubcube,error)
      if (error)  return
      call global%data_subcube(siter,local,error)
      if (error)  return
      !
      ! This part has to be critical to avoid conflict betwen
      ! threads. Third dimension correction has to be applied to
      ! account for i3 being the index inside de subcube.
      !
      ! !$OMP CRITICAL
      call global%merge_local(isubcube,local,error)
      if (error) return
      ! !$OMP END CRITICAL
    enddo
    !
  end subroutine cubemain_extrema_data_loop
  !
  subroutine cubemain_extrema_data_subcube(global,siter,local,error)
    use cubemain_subcube_real
    use cubetools_nan
    !-------------------------------------------------------------------
    ! 
    !-------------------------------------------------------------------
    class(extrema_desc_t),    intent(inout) :: global
    type(subcube_iterator_t), intent(in)    :: siter
    type(extrema_desc_t),     intent(out)   :: local
    logical,                  intent(inout) :: error
    !
    type(subcube_t) :: insub
    integer(kind=data_k) :: i1,i2,i3
    character(len=*), parameter :: rname='EXTREMA>DATA>SUBCUBE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call local%init(error)
    if (error) return
    call insub%init(global%cube,error)
    if (error)  return
    call insub%get(global%cube,siter,error)
    if (error) return
    !
    ! i3 goes from 1 to insub%n3 as the cut in the third dimension has
    ! been done when preparing the iterator
    do i3=1,insub%n3
       do i2=global%range(2,1),global%range(2,2)
          do i1=global%range(1,1),global%range(1,2)
             if (ieee_is_nan(insub%data(i1,i2,i3))) then
                local%nnan = local%nnan+1
             else
                if (insub%data(i1,i2,i3).gt.local%max) then
                   local%max       = insub%data(i1,i2,i3)
                   local%locmax(:) = [i1,i2,i3]
                else if (insub%data(i1,i2,i3).lt.local%min) then
                   local%min       = insub%data(i1,i2,i3)
                   local%locmin(:) = [i1,i2,i3]
                else
                   ! Nothing to do
                endif
             end if
          enddo
       enddo
    enddo
  end subroutine cubemain_extrema_data_subcube
  !
  subroutine cubemain_extrema_merge_local(global,isubcube,local,error)
    !------------------------------------------------------------------------
    !
    !------------------------------------------------------------------------
    class(extrema_desc_t), intent(inout) :: global
    integer(kind=entr_k),  intent(in)    :: isubcube
    type(extrema_desc_t),  intent(in)    :: local
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname = 'EXTREMA>MERGE>LOCAL'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (local%max.gt.global%max) then
       global%max         = local%max
       global%locmax(1:2) = local%locmax(1:2)
       global%locmax(3)   = local%locmax(3)+isubcube+global%range(3,1)-2
    endif
    if (local%min.lt.global%min) then
       global%min         = local%min
       global%locmin(1:2) = local%locmin(1:2)
       global%locmin(3)   = local%locmin(3)+isubcube+global%range(3,1)-2
    endif
    global%nnan = global%nnan+local%nnan
  end subroutine cubemain_extrema_merge_local
  !
end module cubemain_extrema_types
