subroutine init_clean
  use gkernel_interfaces
  use imager_interfaces, except_this=>init_clean
  !----------------------------------------------------------------------
  ! @ public
  !
  ! IMAGER
  !   Initialize Mapping Languages
  !----------------------------------------------------------------------
  use clean_def
  use clean_arrays
  use clean_default
  use clean_types
  !
  ! vocabulaire de CLEAN
  integer mclean
  parameter (mclean=120+1+2) ! +16 obsolete ones  
  character(len=16) vocab(mclean)
  character(len=40) version
  data vocab /' DUMP',   &
       ' ALMA', '/FLUX', '/PLOT', '/QUERY', '/NOISE', '/METHOD', &
       ' BUFFERS', &
       ' CLEAN', '/FLUX', '/PLOT', '/QUERY', '/NITER', '/ARES', &  !No conflict with ALMA ?
       '*COLOR', &
!       ' CLARK', '/FLUX', '/PLOT', '/QUERY', &
       ' FIT', &
!       ' HOGBOM', '/FLUX', &
       ' INSPECT_3D', &
       ' MAP_COMPRESS', ' MAP_RESAMPLE', ' MAP_INTEGRATE', &
       ' MOSAIC', &
!       ' MRC', '/FLUX', '/PLOT', '/QUERY', &
!       ' MULTI', '/FLUX', &
       ' MX', '/FLUX', '/PLOT', '/QUERY', &
       ' PRIMARY', '/TRUNCATE', &
       ' READ', '/COMPACT', '/FREQUENCY', '/PLANE', '/RANGE',  '/NOTRAIL', '/TELESCOPE', &
       ' SHOW', &
!       ' SDI', '/FLUX', '/PLOT', '/QUERY', &
       ' SPECIFY', &
       ' STATISTIC', '/NOISE', '/WHOLE', &
       ' SUPPORT', '/CURSOR', '/MASK', '/PLOT', '/RESET', '/THRESHOLD', '/VARIABLE', &
       ' UV_BASELINE', '/CHANNELS', '/FILE', '/FREQUENCY', '/RANGE', '/VELOCITY', '/WIDTH', &
       ' UV_CHECK', ' UV_COMPRESS', '/FILE', ' UV_CONTINUUM','/INDEX', '/RANGE', &
       ' UV_EXTRACT', '/FIELDS', '/FILE', '/RANGE', &
       ' UV_FLAG', '/ANTENNA', '/DATE', '/RESET', &
       ' UV_FIELDS', '/CENTER', '/FILE', &
       ' UV_FILTER', '/CHANNELS', '/FILE', '/FREQUENCY', '/RANGE', '/VELOCITY', '/WIDTH', '/ZERO', &
       ' UV_HANNING', '/FILE', &
       ' UV_MAP', '/TRUNCATE',  '/FIELDS', '/SELF', '/RANGE', '/CONTINUUM', '/INDEX', &
       ' UV_RESAMPLE', '/FILE', ' UV_RESIDUAL', '/FIELDS', ' UV_RESTORE', '/COPY', '/SLOW', '/SELF', &
       ' UV_REWEIGHT', '/FLAG', '/FILE', '/RANGE', &
       ' UV_SHIFT', ' UV_SORT',  '/FILE', & 
       ' UV_SPLIT', '/CHANNELS', '/FILE', '/FREQUENCY', '/RANGE', '/VELOCITY', '/WIDTH', &
       ' UV_STAT',   &
       ' UV_TIME', '/FILE', '/WEIGHT', &
       ' UV_TRIM', '/FILE', ' UV_TRUNCATE', &
       ' VIEW',   '/NOPAUSE',  &
       ' WRITE', '/APPEND ', '/RANGE', '/REPLACE', '/TRIM'/ 
  logical error
  integer, parameter :: nself=10
  character(len=16) vself(nself)
  character(len=40) verself
  data vself/  & 
       ' APPLY', '/FLAG',     & ! Apply Self calibration
       ' MODEL', '/MINVAL',   & ! Create a UV model from CCT components
       ' SCALE_FLUX',         & ! Verify day-to-day Flux consistency
       ' SOLVE', '/MODE',     & ! Solve for Amplitude and/or Phase
       ' UV_SELF', '/RANGE', '/RESTORE' /
  !
  integer, parameter :: news=43
  character(len=16) vnews(news)
  character(len=40) vernews
  data vnews/' CATALOG', ' EXTRACT',                            & 
    & ' FEATHER', '/FILE', '/REPROJECT',                        &
    & '*FLUX', '/CCT', ' HOW_TO',                               &
    & ' MAP_POLAR',                                             &
    & ' MAP_CONTINUUM', ' MASK', ' MFS',                        & 
    & ' MOMENTS', '/MASK', '/METHOD', '/RANGE', '/THRESHOLD',   & 
    & '*SELFCAL', '/WIDGET', ' SLICE',  ' STOKES', '/FILE',     & 
    & ' UV_ADD', '/FILE',                                       &
    & ' UV_DEPROJECT', '*UV_FIT', '/QUIET', '/SAVE', '/WIDGET', & 
    & ' UV_MERGE', '/FILES', '/MODE', '/SCALES', '/WEIGHTS',    &
    & '*UV_PREVIEW', '/FILE',                                   &
    & ' UV_RADIAL', '/SAMPLING', '/U_ONLY', '/ZERO',            & 
    & ' UV_SHORT', '/REMOVE', ' XY_SHORT' /
  character(len=20) :: v
  ! 
  integer :: iout
  !
  v = '4.0-01 08-Mar-2018' ! First IMAGER release with new Variable names (may still change)
  v = '4.1-01 08-Mar-2018' ! IMAGER release with Self Calibration commands
  v = '4.1-02 15-Mar-2018' ! Bug correction in Self Calibration commands
  v = '4.1-02 02-May-2018' ! Split with CALIBRATE\ language 
  v = '4.2-01 03-Jul-2018' ! Add /RANGE options to UV_FILTER and UV_BASELINE
  v = '4.3-01 05-Sep-2018' ! Minor change in SHOW NOISE to identify properly Dirty and Clean data
  v = '4.3-02 05-Nov-2018' ! Debug MX, improve MULTI and cleanup messages  
  v = '4.3-03 14-Nov-2018' ! Add the UV_TRIM command, support READ CCT /RANGE
  v = '4.4-01 14-Jan-2019' ! Add the UV_HANNING command
  v = '4.4-02 17-Jan-2019' ! Add the UV_EXTRACT command
  v = '4.4-03 04-Feb-2019' ! Add the /FILE option to UV_BASELINE,UV_FILTER, UV_SPLIT and UV_SORT
  v = '4.4-04 07-Feb-2019' ! Bug correction in UV_MAP for Phase offset mosaics 
  v = '4.4-05 08-Feb-2019' ! UV_TRIM /FILE syntax
  v = '4.4-06 12-Feb-2019' ! Protect against incomplete UV data (Nvisi < dim(2))
  v = '4.4-07 13-Feb-2019' ! Operational version of UV_SORT /FILE (still slow)
  v = '4.4-08 14-Feb-2019' ! Operational version of UV_SORT /FILE (reasonable now) 
  v = '4.4-09 15-Feb-2019' ! UV_TIME /FILE syntax
  v = '4.4-10 25-Feb-2019' ! UV_RESAMPLE, UV_COMPRESS, UV_HANNING /FILE syntax
  v = '4.5-01 27-Feb-2019' ! Move STOKES command to ADVANCED\ language
  v = '4.5-02 14-Mar-2019' ! Improve the STATISTIC and SUPPORT commands behaviour 
  v = '4.5-03 29-Mar-2019' ! Support for SHOW SOURCES
  v = '5.0-01 05-Apr-2019' ! Remove CLARK, ..., SDI commands, leave only CLEAN
  v = '5.0-02 05-Apr-2019' ! Move COLOR to CLEAN\
  v = '5.0-03 29-Apr-2019' ! Option /CONT for UV_MAP
  v = '5.0-04 02-May-2019' ! BUFFERS command
  v = '5.0-05 06-May-2019' ! /NOPAUSE option to VIEW command
  v = '5.0-06 07-May-2019' ! UV_STAT DEFAULT and associated variables
  v = '5.0-07 25-May-2019' ! /INDEX option for UV_MAP
  v = '5.1-01 05-Jun-2019' ! Command UV_REWEIGHT re-written
  v = '5.1-02 15-Jun-2019' ! UV_EXTRACT /FIELDS
  v = '5.1-03 25-Jun-2019' ! UV_FIELDS command 
  version = v//'S.Guilloteau'
  !
  ! ADVANCED\ Language
  v = '1.0-01 01-Dec-2017' ! Command UV_PREVIEW 
  v = '1.0-02 08-Mar-2018' ! Commands UV_DEPROJECT  and UV_RADIAL
  v = '1.1-02 08-Mar-2018' ! Commands MODEL, SOLVE /MODE, CALIBRATE and UV_MAP /SELF 
  v = '1.2-01 10-Apr-2018' ! Template for Command UV_SHORT
  v = '1.2-02 26-Apr-2018' ! Operational Command UV_SHORT - Still problems with repetitions.
  v = '1.3-01 03-Jul-2018' ! Rename CALIBRATE into APPLY -- Split with CALIBRATE\ language
  v = '1.3-02 26-Sep-2018' ! Add SLICE and EXTRACT commands, and allow support 
  !         for VIEW (or SHOW) AnySicImage and UV_PREVIEW bug correction
  v = '1.4-01 23-Nov-2018' ! Rename language to ADVANCED, and insert SELFCAL command
  v = '1.4-02 26-Nov-2018' ! Add UV_FIT command 
  v = '1.4-03 28-Nov-2018' ! Add MOMENTS command 
  v = '1.4-04 28-Nov-2018' ! Add CATALOG command 
  v = '1.4-05 15-Jan-2019' ! Better version of the CATALOG command - Drives a script
  v = '1.4-06 23-Jan-2019' ! Correction of a bug in UV_PREVIEW
  v = '1.5-01 24-Jan-2019' ! Add the ADVANCED\MASK command
  v = '1.5-02 04-Feb-2019' ! Debug the UV_FIT command -- Add the /FILE option for UV_PREVIEW
  v = '1.6-01 26-Feb-2019' ! Add the UV_ADD command (similar to UV_DOPPLER in ASTRO)
  v = '1.6-02 27-Feb-2019' ! Move the STOKES command from the Clean\ language
  v = '1.6-03 19-Mar-2019' ! Add the UV_MERGE command (simpler than the task...)
  v = '1.6-04 20-Mar-2019' ! Add the COLOR and FLUX commands  
  v = '1.6-05 28-Mar-2019' ! /SAVE option of UV_FIT 
  v = '1.6-06 14-May-2019' ! HOW_TO command
  v = '1.7-01 19-May-2019' ! FEATHER command
  v = '1.7-02 24-May-2019' ! XY_SHORT command
  vernews = v//'S.Guilloteau'
  !
  ! CALIBRATE\ Language
  v = '1.0-01 03-Jul-2018' ! Split with CALIBRATE\ language 
  v = '1.1-01 07-Jul-2018' ! /MINVAL option instead of /MINFLUX for MODEL 
  v = '1.2-01 20-Mar-2019' ! Rename command FLUX_SCALE into SCALE_FLUX 
  verself = v//'S.Guilloteau'
  !
  call sic_begin('CLEAN','gag_doc:hlp/imager-help-clean.hlp',mclean,vocab, version, run_clean, gr_error)
  call sic_begin('CALIBRATE','gag_doc:hlp/imager-help-calib.hlp',nself,vself, verself, run_self, gr_error)
  call sic_begin('ADVANCED','gag_doc:hlp/imager-help-news.hlp',news,vnews, vernews, run_advanced, gr_error)
  !
  call gagout('I-IMAGER,  CLEAN\     Version '//version)
  call gagout('I-IMAGER,  ADVANCED\  Version '//vernews)
  call gagout('I-IMAGER,  CALIBRATE\ Version '//verself)
  !
  call sic_defstructure('OMP_MAP',.true.,error)
  omp_compiled = .false.
  !$ omp_compiled = .true.
  call sic_def_logi('OMP_MAP%COMPILED',omp_compiled,.false.,error)
  call sic_def_logi('OMP_MAP%DEBUG',omp_debug,.false.,error)
  call sic_def_logi('OMP_MAP%UV_MAP',omp_uvmap,.false.,error)
  call sic_def_inte('OMP_MAP%GRID',omp_grid,0,0,.false.,error)
  call sic_def_logi('OMP_MAP%RESTORE_FAST',omp_restore_fast,.false.,error)
  call sic_def_logi('OMP_MAP%UV_MAP_LOCK',omp_uvmap_lock,.false.,error)
  call sic_def_inte('OMP_MAP%SINGLE_BEAM_CODE',omp_single_beam,0,0,.false.,error)
  !
  user_method%do_mask = .true.  ! Unclear why ...
  call sic_def_logi('METHOD_DOMASK',user_method%do_mask,.false.,error) ! To check
  nullify(duvr)
  nullify(duvs)
  call define_var(error)
  MappingError = .false.
  call sic_def_logi('MAPPING_ERROR',MappingError,.false.,error)
  !
  ! Re-define the DO_LOOP variable for VIEW
  call sic_delvariable('DO_LOOP',.false.,error)
  call sic_def_logi('DO_LOOP',view_do_loop,.false.,error)
  view_do_loop = .true.
  !
  call sic_def_inte('MAP_VERSION',map_version,0,0,.false.,error)
  call sic_def_real('MAP_ROUNDING',map_rounding,0,0,.false.,error)
  call sic_def_inte('MAP_POWER   ',map_power,0,0,.false.,error)
  call gi4_round_forfft(64,iout,error,map_rounding,map_power)
  !
  call sic_def_login('SAVE_DATA',save_data,1,mtype,.false.,error)
  call gprompt_set('IMAGER')
  !
  call sic_get_inte('SIC%RAMSIZE',sys_ramsize,error)
  call feather_comm(' ',error) ! Initialize
end subroutine init_clean
!
subroutine run_self (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_self
  use gkernel_interfaces, only : sic_present, sic_mapgildas, sic_delvariable
  !----------------------------------------------------------------------
  ! @ public
  !
  ! IMAGER   Main routine
  !   Call appropriate subroutine according to COMM
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  character(len=*), intent(in)    :: comm  ! Command name
  logical,          intent(out)   :: error ! Logical error flag
  !
  ! Local variables
  logical use_self
  character(len=1) :: code
  integer, save :: icall=0
  integer :: i
  !
  if (icall.ne.0) then
     Print *,'Rentrant call to RUN_CLEAN ',comm
     read(5,*) i
  endif
  icall = icall+1
  !
  MappingError = .false.
  call map_message(seve%c,'CLEAN',line)
  !
  select case (comm)
  case ('UV_SELF')
    call uv_map_comm(line,comm,error)
  case ('APPLY')
    call uv_calibrate(line,error)
  case ('MODEL') 
    call cct_fast_uvmodel(line,error)
  case ('SOLVE') 
    call solve_gain(line,error)
  case ('SCALE_FLUX')
    call flux_dispatch (line,error)
  case default
    call map_message(seve%e,'SELF','No code for '//comm)
    error = .true.
  end select
  icall = icall-1
end subroutine run_self
!
subroutine run_clean (line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_clean
  use gkernel_types
  use gkernel_interfaces, only : sic_present, sic_mapgildas, sic_delvariable
  !----------------------------------------------------------------------
  ! @ public
  !
  ! IMAGER   Main routine
  !   Call appropriate subroutine according to COMM
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  character(len=*), intent(in)    :: comm  ! Command name
  logical,          intent(out)   :: error ! Logical error flag
  !
  ! Local variables
  type(sic_descriptor_t) :: desc   ! Descriptor
  logical use_self
  character(len=1) :: code
  character(len=64) :: ctest
  integer, save :: icall=0
  integer :: i
  logical :: err, found, quiet
  !
  if (icall.ne.0) then
     Print *,'Rentrant call to RUN_CLEAN ',comm
     read(5,*) i
  endif
  icall = icall+1
  !
  MappingError = .false.
  if (sic_present(0,1)) then
    ctest = ' '    
    call sic_ch(line,0,1,ctest,i,.false.,error)
    quiet = (ctest(1:1).eq.'?')
  else
    quiet = .false.
  endif
  call map_message(seve%c,'CLEAN',line)
  !
  ! Analyze command
  ! First cases do not require to update variables
  select case (comm)
    case ('FIT')
    call clean_beam(line,error)
  case ('COLOR')
    call color_comm(line,error)
  case ('WRITE')
    call write_image(line,error)
  case ('SUPPORT')
    call com_support(line,error)
  case ('STATISTIC')
    call statistic(line,error)
  case ('SHOW','VIEW','INSPECT_3D')
    call display_buffer(comm,line,error)
    quiet = .true.
  case ('UV_STAT')
    call uv_stat_comm(line,error)
  case ('DUMP')
    call debug_all(line,error)
    quiet = .true.
  case ('BUFFERS')
    call buffers_comm(line,error)
    quiet = .true.
  !
  ! Other cases should update their own variables
  case default
    select case (comm)
  !  case ('LOAD')
  !    call load_buffer(line,error)
    case ('READ')
      call read_image(line,error)
    case ('CLEAN')
      call dispatch_clean(line,error)
    case ('MX')
      if (themap%nfields.ne.0) then
        call map_message(seve%e,comm,'UV data is a Mosaic, not supported')
        error = .true.
      else
        call uvmap('MX',line,error)
      endif
      case ('ALMA')
        call alma_clean(line,error)
      case ('UV_FLAG')
        call uv_flag_comm(line,error)
      case ('UV_MAP')
        call uv_map_comm(line,comm,error)
      case ('UV_SORT')
        if (sic_present(1,0)) then
          call uv_sort_file(line,error)
        else
          call uv_sort_comm(line,error)
        endif
    case ('UV_RESAMPLE','UV_COMPRESS','UV_HANNING')
      call uv_resample_comm(line,comm,error)
    case ('UV_RESTORE')
      call uv_map_comm(line,comm,error)
    case ('UV_SHIFT')
      call uv_shift_mosaic(line,comm,error)
    case ('UV_CONTINUUM')
      call uv_line2cont (line,error)
    case ('UV_FILTER')
      call uv_filter (line,error)
    case ('UV_BASELINE')
      call uv_baseline (line,error)
    case ('UV_TIME')
      if (themap%nfields.ne.0) call map_message(seve%w,comm,'UV data is a Mosaic')
      call uv_time_comm (line,error)
    case ('UV_TRUNCATE')
      call uv_truncate_comm (line,error)
    case ('UV_TRIM')
      call uv_trim_comm (line,error)
    case ('UV_REWEIGHT')
      call uv_reweight_comm (line,comm,error)
    case ('UV_CHECK')
      call uv_check_comm (line,error)
    case ('UV_FIELDS')
      call uv_fields_comm (line,comm,error)
    case ('PRIMARY')
      call primary_comm(line,error)
    case ('SPECIFY')
      call com_modify (line,error)
!  case ('SG_DEBUG')
!    call sg_debug(line,error)
    case ('UV_RESIDUAL')
      if (last_resid.eq.'CCT') then
        if (themap%nfields.ne.0) then
          call map_message(seve%w,comm,'UV data is a Mosaic - UNDER TESTS !!!')
        endif
        call uv_residual(line,error)
      else if (last_resid.eq.'UV_FIT') then
        call uvfit_residual(line,error)
      else
        call map_message(seve%w,comm,'No CLEAN or UV_FIT performed')
      endif
    case ('MAP_RESAMPLE','MAP_COMPRESS','MAP_INTEGRATE') 
      call map_resample_comm(line,comm,error)
    case ('UV_SPLIT')
      call uv_split_comm(line,error)
    case ('UV_EXTRACT')
      call uv_extract_comm(line,error)
! No longer used...
    case ('CLARK')
      call clark_clean(line,error)
    case ('HOGBOM')
      call hogbom_clean(line,error)
    case ('MOSAIC')
      call mode_mosaic(line,error)
    case ('MRC')
      call mrc_clean(line,error)
    case ('MULTI')
      call multi_clean(line,error)
    case ('SDI')
      call sdi_clean(line,error)
    case default
      call map_message(seve%i,'CLEAN',comm//' not yet implemented')
      icall = icall-1
      error = .true.
      return
    end select
    ! Delete variables that may have been affected
    call sic_descriptor('SPECTRA',desc,found)  
    err = .false.
    if (found) call sic_delvariable('SPECTRA',.false.,err)
  end select
  icall = icall-1
  if (MappingError) error = .true.
  !
  if (error) then
    call message_colour(1)
    call map_message(seve%e,comm,'failed to complete')  
  else if (.not.quiet) then
    call message_colour(2)
    call map_message(seve%i,comm,'Successful completion')
  endif
  call message_colour(-1)
  !
end subroutine run_clean
!
subroutine run_advanced(line,comm,error)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, except_this=>run_advanced
  !----------------------------------------------------------------------
  ! @ public
  !
  ! IMAGER   Main routine
  !   Call appropriate subroutine according to COMM
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  character(len=*), intent(in)    :: comm  ! Command name
  logical,          intent(out)   :: error ! Logical error flag
  !
  ! Local variables
  integer, save :: icall=0
  integer :: i
  !
  if (icall.ne.0) then
    Print *,'Rentrant call to RUN_ADVANCED ',comm
    read(5,*) i
  endif
  icall = icall+1
  !
  MappingError = .false.
  call map_message(seve%c,'ADVANCED',line)
  !
  ! Analyze command
  select case (comm)
  case ('SELFCAL') 
    call selfcal(line,comm,error)
  case ('UV_PREVIEW')
    call uv_preview(line,error)
  case ('MAP_CONTINUUM')
    call map_continuum(line,error)
!  case ('MAP_POLAR')   ! Not yet
!    call map_polar(line,comm,error)
  case ('MFS')
    call clean_mfs (line,error)
  case ('UV_DEPROJECT','UV_RADIAL') 
    call uv_radial(line,comm,error)
  case ('UV_SHORT')
    call uvshort_com(line,comm,error)
  case ('XY_SHORT')
    call uvshort_com(line,comm,error)
  case ('UV_FIT')
    call uvfit_comm(line,error)
  case ('SLICE') 
    call slice_comm(line,error)
  case ('STOKES')
    call stokes_comm(line,error)
  case ('EXTRACT') 
    call extract_comm(line,error)
  case ('MOMENTS')
    call moments_comm(line,error)
  case ('CATALOG')
    call catalog_comm(line,error)
  case ('MASK') 
    call mask_comm(line,error)
  case ('UV_ADD') 
    call uv_add_comm(line,error)
  case ('UV_MERGE') 
    call uv_merge_many(line,error)
  case ('FLUX')
    call flux_comm(line,error)
  case ('HOW_TO')
    call howto_comm(line,error)
  case ('FEATHER')
    call feather_comm(line,error)
  case default
    call map_message(seve%i,'ADVANCED',comm//' not yet implemented')
  end select
  icall = icall-1
  if (MappingError) error = .true.
end subroutine run_advanced
!
subroutine save_result(error)
  use gkernel_interfaces
  use gildas_def
  use clean_types
  use gbl_message
  logical error
  character(len=filename_length) :: ans
  character(len=filename_length) :: autosave='autosave'  ! Default name when saving
  character(len=commandline_length) line
  character(len=message_length) :: chain
  character(len=*), parameter :: rname = 'IMAGER'
  integer :: n, icode
  logical :: do_write,noprompt
  !
  ans = ' '
  error = .false.
  do_write = .false.
  call gmaster_get_exitnoprompt(noprompt)  ! Check for SIC\EXIT /NOPROMPT
  !
  if (any(save_data) .and. .not.noprompt) then
    call map_message(seve%w,rname,'There are unsaved data, should we save them?')
    do while (.true.)
      call sic_wprn('Y)es, N)o, C)ancel exit (default No): ',ans,n)
      call sic_upper(ans)
      if (ans(1:1).eq.'C') then
        error = .true.
        return
      elseif (ans(1:1).eq.'N' .or. ans(1:1).eq.' ') then
        do_write = .false.
      elseif (ans(1:1).ne.'Y') then
        call map_message(seve%e,rname,'Unrecognized answer '//ans)
        cycle
      else 
        do_write = .true.
      endif
      exit
    enddo
  endif
  !
  call sic_delvariable('DIRTY',.false.,error)
  call sic_delvariable('CLEAN',.false.,error)
  call sic_delvariable('RESIDUAL',.false.,error)
  call sic_delvariable('PRIMARY',.false.,error)
  call sic_delvariable('SKY',.false.,error)
  if (.not.do_write) return
  !
  do icode=1,mtype
    ans = ' '
    error = .false.
    do while (save_data(icode))
       write(chain,100) vtype(icode)
       call sic_wprn(trim(chain)//' ',ans,n)
       if (ans.eq.' ') ans = autosave
       line = 'CLEAN'//char(92)//'WRITE '//vtype(icode)//' "'//trim(ans)//'"'
       call exec_command (line,error)
    enddo
  enddo
  !
100 format('Enter a filename for ',a,' (default ''autosave''): ')
end subroutine save_result
!
subroutine debug_all(line,error)
  use gkernel_interfaces
  use imager_interfaces, except_this => debug_all
  use clean_def
  use clean_arrays
  use clean_types
  use clean_default
  use gbl_message
  !----------------------------------------------------------------------
  ! @  private
  !
  ! IMAGER     Print debug information
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line
  logical, intent(out) :: error
  !
  character(len=4) meth
  integer :: n, nc, ier
  integer, allocatable :: mic(:)
  !
  call sic_ke(line,0,1,meth,n,.false.,error)
  if (error) return
  if (meth.eq.'USER') then
     call mapping_print_debug(user_method)
  else if (meth.eq.'METH') then
     call mapping_print_debug(method)
  else if (meth.eq.'BUFF') then
    call uv_dump_buffers('DUMP')
  else if (meth.eq.'SG') then
    !
    nc = huv%gil%nchan
    if (nc.ne.hcct%gil%dim(2)) then
      Print *,'Mismatch number of channels between HUV ',nc,' and CCT ',hcct%gil%dim(2)
    endif
    !
    allocate(mic(nc),stat=ier)
    call uv_clean_sizes(hcct,dcct,mic,1,nc)
    Print *,'SG_DEBUG -- MIC ',mic
    deallocate(mic)
  else
    Print *,' DEBUG  BUFFers|METHod|SG|USER'
  endif
end subroutine debug_all
!
subroutine mode_mosaic (line,error)
  use gkernel_interfaces
  use imager_interfaces, only : sub_mosaic
  !----------------------------------------------------------------------
  ! @  private
  !
  ! IMAGER      MOSAIC ON|OFF
  !             Activates or desactivates the mosaic mode
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line
  logical, intent(out) :: error
  !
  integer na,iv
  character(len=8) name,argum,voc1(2)
  data voc1/'OFF','ON'/
  !
  ! Mosaic mode. Default = .true.
  argum = 'ON'
  call sic_ke (line,0,1,argum,na,.false.,error)
  if (error) return
  call sic_ambigs ('MOSAIC',argum,name,iv,voc1,2,error)
  if (error) return
  call sub_mosaic(name,error)
end subroutine mode_mosaic
!
subroutine sic_insert_log(line)
  use gkernel_interfaces, only : sic_insert, sic_log
  !----------------------------------------------------------------------
  ! @ private
  !   Insert command line in Stack and Log file
  !----------------------------------------------------------------------
  character(len=*), intent(in) :: line
  !
  integer :: nl
  !
  nl = len_trim(line)
  call sic_insert(line(1:nl))
  call sic_log(line,nl,0)
end subroutine sic_insert_log
