module clic_interfaces_none
  interface
    subroutine check_input_file(error)
      !---------------------------------------------------------------------
      ! Verify if any input file
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine check_input_file
  end interface
  !
  interface
    subroutine chopper (ip, iant, freqs, freqi, tcals, tcali, error)
      !---------------------------------------------------------------------
      ! CLIC
      !	compute TCAL, TSYS, TREC, from the chop and atm values observed
      !	according to the usual chopper method (Ulich and Haas)
      !
      !	Double side band operation
      !
      !	If r.cmode(ir) is 'AU' and a cold load is available, or if r.cmode(ir)
      !	is 'TR', derive the sky emission TEMI from the receiver
      !	temperature TREC, and then H2OMM from the best model fitting.
      !	Else, if r.cmode(ir) is 'AU', the sky emission is computed from
      !	the given H2OMM, and if r.cmode(ir) is 'MA', the sky emission is
      !	computed from the given opacities and atmospheric temperatures.
      !
      !  	Uses a curved atmosphere, equivalent height 5.5 KM
      !
      ! Input:
      !       ip              i       calibration number (almost logical)
      !       iant            i       antenna number (logical)
      !	FREQS, FREQI	R	Frequencies signal image in GHz (!!!)
      !	Calibration section (-25)
      ! Output:
      !	R.TSYSS 	Signal SB system temperature (K)
      !	R.TSYSI		Image SB system temperature (K)
      !	R.TREC		receiver dsb temperature (K)
      !
      ! Call Tree
      !			READ_DATA,
      !	READ_SPEC,	SUB_READ_SPEC,	SUB_SPECTRUM,	DO_CALIB
      !			CLIC_SPECTRUM,
      !---------------------------------------------------------------------
      integer :: ip                   !
      integer :: iant                   !
      real*4 :: freqs                   !
      real*4 :: freqi                   !
      real*4 :: tcals                   !
      real*4 :: tcali                   !
      logical :: error                  !
    end subroutine chopper
  end interface
  !
  interface
    function air_mass(el)
      !---------------------------------------------------------------------
      ! clic
      !	Computes airmass content corresponding to elevation
      !	EL
      !---------------------------------------------------------------------
      real*8 :: air_mass                !
      real :: el                        !
    end function air_mass
  end interface
  !
  interface
    subroutine run_clic(line,comm,error)
      !---------------------------------------------------------------------
      ! CLIC CLIC
      !	Main routine
      ! 	Call appropriate subroutine according to COMM
      ! Arguments :
      !	LINE	C*(*)	Command line		Input/Output
      !	COMM	C*(*)	Command name		Input
      !	ERROR	L	Logical error flag	Output
      ! (2-Jan-1985)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      character(len=*) :: comm          !
      logical :: error                  !
    end subroutine run_clic
  end interface
  !
  interface
    subroutine clic_atmos(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command ATMOS [DETECTOR|CORRELATOR]
      ! This command  quasi real time amplitude and IF passband calibration.
      ! It processes CAL, AUTO, IFPB and CWVR scans as they are encountered.
      ! Option 1 /COMPRESS time uvmax
      ! Option 2 /RESET {CAL|IFPB:AUTO:ALL}
      ! Option 3 /NOWRITE
      ! Option 4 /NOMONITOR
      ! Option 5 /CMODE  wvrmode wvrcpol
      !
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_atmos
  end interface
  !
  interface
    subroutine load_atmos
      !---------------------------------------------------------------------
      ! CLIC
      !	Load Atmospheric Parameters
      !---------------------------------------------------------------------
      ! Global
    end subroutine load_atmos
  end interface
  !
  interface
    subroutine cal_levels(arg,mes)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Show total power levels
      ! Input
      !	ARG C*(*)	'Detector' or 'Correlator'
      !---------------------------------------------------------------------
      character(len=*) :: arg           !
      integer :: mes                    !
    end subroutine cal_levels
  end interface
  !
  interface
    subroutine check_atmos (error)
      !---------------------------------------------------------------------
      ! Check that spectrum to be reduced is compatible with available
      ! calibration buffers.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine check_atmos
  end interface
  !
  interface
    subroutine real_to_complex(rel,cmpl,n)
      !---------------------------------------------------------------------
      ! Reverse storage order from:
      !	N real parts, N imaginary parts
      ! 	to:
      !	N complex numbers
      !---------------------------------------------------------------------
      integer :: n                      !
      real*4 :: rel(n,2)                !
      complex :: cmpl(n)                !
    end subroutine real_to_complex
  end interface
  !
  interface
    subroutine vel_scale
      !---------------------------------------------------------------------
      ! CLIC
      !     Updates frequencies in header according to last
      !     'SET FREQUENCY' command
      !---------------------------------------------------------------------
      ! Global
    end subroutine vel_scale
  end interface
  !
  interface
    function prec_water(tamb,humidity)
      !---------------------------------------------------------------------
      ! CLIC
      !	Compute the precipitable water content
      !	Assumes scale height 2km
      ! Input
      !     TAMB       REAL	Ambiant Temperature in Kelvin
      !     HUMIDITY   REAL   Humidity in %
      ! Output
      !     PREC_WATER REAL   Water vapor in mm
      !---------------------------------------------------------------------
      real :: prec_water                !
      real :: tamb                      !
      real :: humidity                  !
    end function prec_water
  end interface
  !
  interface
    subroutine compare_sec(n1,n2,nn,ok)
      !---------------------------------------------------------------------
      ! CLIC: compare two memory areas
      !---------------------------------------------------------------------
      integer :: nn                     !
      integer :: n1(nn)                 !
      integer :: n2(nn)                 !
      logical :: ok                     !
    end subroutine compare_sec
  end interface
  !
  interface
    subroutine check_flux(ndata, data, fluxout, errflux, phcorok,   &
        lowfac, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Check the flux on a phase calibrator.
      ! Assume the efficiencies in r_jykel are correct.
      ! The best of corrected and uncorrected data is used.
      !
      ! Arguments
      !     LDATA_IN      Integer Length of data array
      !     DATA(*)       (*)     Data array
      !     Fluxout       R       Measured flux if procedure is flux measurement
      !     Errflux       R       Error on flux "  "             "      "
      !     phcorOK       L(mnant) true if flux with phcor is best
      !     floss         R(mnant) degradation factor for each ant. (0 to 1)
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      integer :: ndata                  !
      real :: data(ndata)               !
      real :: fluxout                   !
      real :: errflux                   !
      logical :: phcorok(mnant)         !
      real :: lowfac(mnant)             !
      logical :: error                  !
    end subroutine check_flux
  end interface
  !
  interface
    real function effmax(f)
      real :: f                         !
    end function effmax
  end interface
  !
  interface
    subroutine sub_atmos (ldata_in,data,arg1,monitor,error,scale)
      !---------------------------------------------------------------------
      ! Process a scan of data depending on procedure
      !
      ! Arguments
      !     LDATA_IN      Integer Length of data array
      !     DATA(*)       (*)     Data array
      !     ARG1          C*(*)   'C' or 'D' for correlator or detector
      !                           'W'for 22 GHz water vapor radiometer
      !
      ! Call Tree
      !			READ_DATA,
      !	READ_SPEC,	SUB_READ_SPEC,
      !			CLIC_ATMOS,
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      integer :: ldata_in               !
      real :: data(ldata_in)            !
      character :: arg1                 !
      logical :: monitor                !
      logical :: error                  !
      real :: scale(mnant,matmcal)      !
    end subroutine sub_atmos
  end interface
  !
  interface
    subroutine do_tsys (monitor, error)
      !---------------------------------------------------------------------
      ! Compute TSYS from a sky measurement in r_csky.
      !---------------------------------------------------------------------
      logical :: monitor                !
      logical :: error                  !
    end subroutine do_tsys
  end interface
  !
  interface
    subroutine do_calib (error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Processes a CAL scan. Autocorrelations on ambient choppper and
      ! Sky have been read in by READ_AUTO.
      ! The chopper standard program is called for each antenna,
      ! in the mode selected in OBS.
      ! The autocorrelation gains are stored in C_C and C_L arrays.
      ! do_test : test the linearity correlator versus total power.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine do_calib
  end interface
  !
  interface
    subroutine do_test(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! check subbands for low gain after a sky measurement.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine do_test
  end interface
  !
  interface
    subroutine do_atmosc (data, error, ido)
      !---------------------------------------------------------------------
      ! CLIC
      !     Process temporal data: apply atmospheric calibration
      ! Input
      !     DATA    COMPLEX(*)
      !     ERROR   LOGICAL      error return
      !     IDO     INTEGER     1 to update the datamodifier section
      !                     (do it only once for the cont and once for the line).
      !---------------------------------------------------------------------
      complex :: data(*)                !
      logical :: error                  !
      integer :: ido                    !
    end subroutine do_atmosc
  end interface
  !
  interface
    subroutine do_atmosl (data, error, ido)
      !---------------------------------------------------------------------
      ! CLIC
      !     Process spectral data, after unclipping and fourier transform:
      !     apply atmospheric calibration.
      ! Input
      !     DATA     COMPLEX(*)
      !     ERROR   OGICAL      error return
      !     IDO     INTEGER     1 to update the datamodifier section
      !                     (do it only once for the cont and once for the line
      !---------------------------------------------------------------------
      complex :: data(*)                !
      logical :: error                  !
      integer :: ido                    !
    end subroutine do_atmosl
  end interface
  !
  interface
    subroutine do_auto_atmosc (data, error, ido)
      !---------------------------------------------------------------------
      ! CLIC
      !     Process temporal data from a sky autocorrelation spectrum
      ! Input
      !     DATA    REAL(*)
      !     ERROR   LOGICAL      error return
      !     IDO     INTEGER     1 to update the datamodifier section
      !                     (do it only once for the cont and once for the line).
      !---------------------------------------------------------------------
      real :: data(*)                   !
      logical :: error                  !
      integer :: ido                    !
    end subroutine do_auto_atmosc
  end interface
  !
  interface
    subroutine do_auto_atmosl (data, error, ido)
      !---------------------------------------------------------------------
      ! CLIC
      !     Process a sky autocorrelation spectrum, after unclipping and fourier
      !     transform: apply atmospheric calibration.
      ! Input
      !     DATA     REAL(*)
      !     ERROR   LOGICAL      error return
      !     IDO     INTEGER     1 to update the datamodifier section
      !                     (do it only once for the cont and once for the line).
      !---------------------------------------------------------------------
      real :: data(*)                   !
      logical :: error                  !
      integer :: ido                    !
    end subroutine do_auto_atmosl
  end interface
  !
  interface
    subroutine do_autol  (mes, data, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Read in auto correlation function
      !     Increment the C_L and C_C buffers for the specified cal phase "mes".
      ! Input
      !     mes      I           Calibration phase
      !     DATA     REAL(*)     The data.
      !---------------------------------------------------------------------
      integer :: mes                    !
      real :: data(*)                   !
      logical :: error                  !
    end subroutine do_autol
  end interface
  !
  interface
    subroutine protect_double(ldata_in,data)
      !---------------------------------------------------------------------
      ! --- protect total power from occasional doubled values
      !---------------------------------------------------------------------
      integer :: ldata_in               !
      real :: data(ldata_in)            !
    end subroutine protect_double
  end interface
  !
  interface
    function n_input(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Determines the number of correlator entries connected to any unit
      !---------------------------------------------------------------------
      integer :: n_input                !
      logical :: error                  !
    end function n_input
  end interface
  !
  interface
    subroutine clic_compress(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_compress
  end interface
  !
  interface
    subroutine triangles (in,out,error)
      !---------------------------------------------------------------------
      ! Call Tree
      !	CLIC_SPECTRUM
      !	CLIC_COMPRESS
      !---------------------------------------------------------------------
      real :: in(*)                     !
      real :: out(*)                    !
      logical :: error                  !
    end subroutine triangles
  end interface
  !
  interface
    subroutine compress(time_max, uv_max, in, nout, error)
      !---------------------------------------------------------------------
      !	Compress data records (temporal data only)
      ! IN	Input and output data
      ! NOUT  Output number of records
      !       time_max = -1. -> compress beam switch...
      !
      ! Call Tree
      !	CLIC_SPECTRUM
      !	CLIC_COMPRESS
      !	CLIC_TABLE
      !---------------------------------------------------------------------
      real :: time_max                  !
      real :: uv_max                    !
      real :: in(*)                     !
      integer :: nout                   !
      logical :: error                  !
    end subroutine compress
  end interface
  !
  interface
    subroutine check_data (data, qdata)
      !---------------------------------------------------------------------
      ! Check data and flag subbands if absolute value greater than vmax
      !---------------------------------------------------------------------
      integer :: qdata                  !
      real :: data(qdata)               !
    end subroutine check_data
  end interface
  !
  interface
    subroutine add_record (qdata, data, buffer)
      !---------------------------------------------------------------------
      ! Call Tree
      !	CLIC_SPECTRUM
      !	CLIC_COMPRESS	COMPRESS
      !	CLIC_TABLE
      !---------------------------------------------------------------------
      integer :: qdata                  !
      real :: data(qdata)               !
      real :: buffer(qdata)             !
    end subroutine add_record
  end interface
  !
  interface
    subroutine zero_record (qdata,buffer)
      !---------------------------------------------------------------------
      ! Call Tree
      !	CLIC_SPECTRUM
      !	CLIC_COMPRESS	COMPRESS
      !	CLIC_TABLE
      !---------------------------------------------------------------------
      integer :: qdata                  !
      real :: buffer(qdata)             !
    end subroutine zero_record
  end interface
  !
  interface
    subroutine output_record (qdata, data, buffer, nout)
      !---------------------------------------------------------------------
      ! Call Tree
      !	CLIC_SPECTRUM
      !	CLIC_COMPRESS	COMPRESS
      !	CLIC_TABLE
      !---------------------------------------------------------------------
      integer :: qdata                  !
      real :: data(qdata)               !
      real :: buffer(qdata)             !
      integer :: nout                   !
    end subroutine output_record
  end interface
  !
  interface
    function phrms(v)
      real :: phrms                     ! rms in phase (radians)
      real :: v                         ! visibility modulus, in terms of sigma
    end function phrms
  end interface
  !
  interface
    function amrms(v)
      real :: amrms                     ! rms in amplitude, in terms of sigma
      real :: v                         ! visibility modulus, in terms of sigma
    end function amrms
  end interface
  !
  interface
    subroutine clic_copy(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command COPY = MODIFY NOTHING
      !     well not so simple
      ! COPY DATA|HEADERS [ANT | NOANT] [BASE | NOBASE]
      ! 1:
      !     DATA     data and headers are copied
      !     HEADERS  headers only
      ! 2 and/or 3:
      !     ANT      Antenna calibration section created
      !     BASE     Baseline calibration section created
      !     NOANT    Antenna calibration sections  NOT created
      !     NOBASE   Baseline calibration sections NOT created
      ! otherwise sections are copied if and only if present.
      ! default is "ANT"
      ! Bure copy Job does "NOANT NOBASE"
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_copy
  end interface
  !
  interface
    subroutine clic_cursor(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Returns x_data and y_data at cursor position
      ! Il importe d'appeler gt_curs avec une chaine de plus de 1 caractere car
      ! elle deborde ...
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_cursor
  end interface
  !
  interface
    subroutine sub_cursor(argum,ib,nd,xd,yd,wd,ud,rd,sd,   &
        code,x4,y4,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Returns parameters of the observation displayed closest to the
      ! cursor position.
      !
      !     ARGUM C*(*)    Proximity mode ( X, Y, Distance )
      !     IB    INTEGER  Box number
      !     ND    INTEGER  Number of points in box
      !     XD    REAL     X coordinates
      !     YD    REAL     Y values
      !     WD    REAL     Weights
      !     UD    REAL     Observation number
      !     RD    REAL     Record number
      !     sD    REAL     work area
      !     CODE  C*(*)    Code of operation
      !     X4,Y4 REAL     cursor coordinates
      !---------------------------------------------------------------------
      character(len=1) :: argum         !
      integer :: ib                     !
      integer :: nd                     !
      real :: xd(nd)                    !
      real :: yd(nd)                    !
      real :: wd(nd)                    !
      integer :: ud(nd)                 !
      integer :: rd(nd)                 !
      logical :: sd(nd)                 !
      character(len=1) :: code          !
      real :: x4                        !
      real :: y4                        !
      logical :: error                  !
    end subroutine sub_cursor
  end interface
  !
  interface
    subroutine do_flags(imin, imax, ibase, fl, reset, ud, rd, sd, nd,   &
        error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Loop to set or reset flags in current index
      !---------------------------------------------------------------------
      integer :: imin                   !
      integer :: imax                   !
      integer :: ibase                  !
      integer :: fl                     !
      logical :: reset                  !
      integer :: nd                     !
      integer :: ud(nd)                 !
      integer :: rd(nd)                 !
      logical :: sd(nd)                 !
      logical :: error                  !
    end subroutine do_flags
  end interface
  !
  interface
    subroutine clic_drop(line,error)
      !---------------------------------------------------------------------
      ! CLIC 	Support routine for command
      !	DROP Number [Version]
      !	Deletes an observation from current index
      ! Arguments :
      !	LINE	C*(*)	Command line		Input
      ! (5-mar-1985)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_drop
  end interface
  !
  interface
    subroutine clic_flag(line,error)
      !---------------------------------------------------------------------
      ! CLIC	Command CLIC\FLAG f1 f2 ... [/ANTENNA a1 a2 ... ]
      !		[/BASELINE b1 b2 ...] [/RESET]
      ! changes the flags in the observation and record currently in memory
      ! (to be used in a MODIFY DATA_HEADER procedure)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_flag
  end interface
  !
  interface
    subroutine get_flags(line,iarg,iao,ibo,iro,afl,bfl,saf,sbf, &
        reset, error)
      use gildas_def
      !---------------------------------------------------------------------
      !
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      character(len=*) :: line          !
      integer :: iarg                   !
      integer :: iao                    !
      integer :: ibo                    !
      integer :: iro                    !
      integer :: afl(mnant)             !
      integer :: bfl(mnbas)             !
      integer :: saf(mnant)             !
      integer :: sbf(mnbas)             ! 
      logical :: reset                  !
      logical :: error                  !
    end subroutine get_flags
  end interface
  !
  interface
    subroutine get_ant_mask(chain,mask,error)
      character(len=*) :: chain         !
      integer :: mask                   !
      logical :: error                  !
    end subroutine get_ant_mask
  end interface
  !
  interface
    subroutine get_base_mask(chain,mask,error)
      character(len=*) :: chain         !
      integer :: mask                   !
      logical :: error                  !
    end subroutine get_base_mask
  end interface
  !
  interface
    function down_baseline (ib)
      logical :: down_baseline          !
      integer :: ib                     !
    end function down_baseline
  end interface
  !
  interface
    function down_antenna (iant)
      !---------------------------------------------------------------------
      ! Is ANTENNA Iant Flagged ?
      !---------------------------------------------------------------------
      logical :: down_antenna           !
      integer :: iant                   !
    end function down_antenna
  end interface
  !
  interface
    function down_channel (ib,ic)
      !---------------------------------------------------------------------
      ! Is ANTENNA Iant Flagged ?
      !---------------------------------------------------------------------
      logical :: down_channel           !
      integer :: ib                     !
      integer :: ic                     !
    end function down_channel
  end interface
  !
  interface
    subroutine clic_get(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! SAS	Support routine for command
      !	GET Number [Version] /DUMP idump
      !	    NEXT
      !	    FIRST
      ! Arguments :
      !	LINE	C*(*)	Command line
      !	ERROR	L	Logical error flag
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_get
  end interface
  !
  interface
    subroutine get_sub(nnn,mmm,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     GET Entry Version
      !---------------------------------------------------------------------
      integer :: nnn                    !
      integer :: mmm                    !
      logical :: error                  !
    end subroutine get_sub
  end interface
  !
  interface
    subroutine get_it (kx,error)
      use gildas_def
      integer :: kx                     !
      logical :: error                  !
    end subroutine get_it
  end interface
  !
  interface
    subroutine fill_rec2(kx,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! fill in atmospheric data from receiver 2
      !---------------------------------------------------------------------
      integer :: kx                     !
      logical :: error                  !
    end subroutine fill_rec2
  end interface
  !
  interface
    subroutine check_eff
      !---------------------------------------------------------------------
      ! CLIC
      !	Define Efficiencies if not known.
      !---------------------------------------------------------------------
      ! Global
    end subroutine check_eff
  end interface
  !
  interface
    subroutine get_memory(length,address,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Get some virtual memory, new or used.
      !---------------------------------------------------------------------
      integer :: length                        !
      integer(kind=address_length) :: address  !
      logical :: error                         !
    end subroutine get_memory
  end interface
  !
  interface
    subroutine new_file
      use gildas_def
      !---------------------------------------------------------------------
      ! A New File : deassign IO channel, unmap all sections, open again
      !---------------------------------------------------------------------
      ! Global
    end subroutine new_file
  end interface
  !
  interface
    subroutine clic_fits(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !  Writes the available line subbands as a FITS spectrum.
      !  FITS file_name
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_fits
  end interface
  !
  interface
    subroutine sum_base(md, mb , x_data, y_data, w_data,   &
        nd, ns, p_data, weight)
      use gildas_def
      !---------------------------------------------------------------------
      ! Average real and imaginary data in all boxes (baselines)
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: nd                     !
      integer :: ns                     !
      real :: p_data(nd,ns, 3)          !
      real :: weight                    !
    end subroutine sum_base
  end interface
  !
  interface
    subroutine clic_tofits(nd,ydata,is,isb, check,error)
      use gildas_def
      use gbl_constant
      !---------------------------------------------------------------------
      ! FITS	Internal routine.
      !	Write current spectrum on tape.
      ! Arguments :
      !	ERROR		L	Logical error flag	Output
      !
      ! Current limitations:
      !	- Horizontal offsets incorrectly handled.
      !---------------------------------------------------------------------
      integer :: nd                     !
      real :: ydata(nd)                 !
      integer :: is                     !
      integer :: isb                    !
      logical :: check                  !
      logical :: error                  !
    end subroutine clic_tofits
  end interface
  !
  interface
    subroutine minmax (rmin,rmax,a,na,rbad)
      real*4 :: rmin                    !
      real*4 :: rmax                    !
      real*4 :: a(*)                    !
      integer :: na                     !
      real*4 :: rbad                    !
    end subroutine minmax
  end interface
  !
  interface
    subroutine clic_mark(line,error)
      !---------------------------------------------------------------------
      ! CLIC	Command CLIC\MARK f1 f2 ... [/ANTENNA] [/BASELINE] [/RESET]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_mark
  end interface
  !
  interface
    subroutine list_flags(nant,nbas,afl,bfl,saf,sbf,out)
      use gildas_def
      integer :: nant                   !
      integer :: nbas                   !
      integer :: afl(nant)              !
      integer :: bfl(nbas)              !
      integer :: saf(nant)              !
      integer :: sbf(nbas)              !
      character(len=*) :: out           !
    end subroutine list_flags
  end interface
  !
  interface
    subroutine clic_mask(line,error)
      !---------------------------------------------------------------------
      ! CLIC	Command CLIC\MASK f1 f2 ... [/ANTENNA] [/BASELINE] [/RESET]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_mask
  end interface
  !
  interface
    subroutine clic_modify(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY args
      ! options :
      ! 1 	/BEFORE
      ! 2 	/AFTER
      ! 3	[ANTENNA|BASELINE] /OFFSET
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_modify
  end interface
  !
  interface
    subroutine ini_header(line,iopt,error)
      !---------------------------------------------------------------------
      ! 	MODIFY HEADER [procedure]
      !	Execute a SIC procedure (involving variables) for
      !	each header read in the index, and rewrite.
      !	default is "HEADER.CLIC"
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_header
  end interface
  !
  interface
    subroutine ini_data(line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	MODIFY DATA [procedure] /ATM
      !	Execute a SIC command procedure (involving variables) for
      !	each data average record read in the index, and rewrite.
      !	default is "DATA.CLIC"
      ! 	MODIFY RECORD [procedure]
      !       same thing but every data record ...
      ! 24-oct-01, RM: add possibility to modify record by record :
      !       MODIFY RECORD [procedure]  /record idump
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_data
  end interface
  !
  interface
    subroutine ini_reference(line,iopt,error)
      !---------------------------------------------------------------------
      ! 	MODIFY REFERENCE UPPER|LOWER r1 r2 r3 r4
      ! 	       VELOCITY  UPPER|LOWER v1 v2 v3 v4
      !	       NOTHING
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_reference
  end interface
  !
  interface
    subroutine ini_velocity(line,iopt,error)
      !---------------------------------------------------------------------
      ! 	MODIFY VELOCITY value
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_velocity
  end interface
  !
  interface
    subroutine ini_flux(line,iopt,error)
      !---------------------------------------------------------------------
      ! 	MODIFY FLUX new_value
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_flux
  end interface
  !
  interface
    subroutine ini_freq(line,iopt,error)
      !---------------------------------------------------------------------
      ! MODIFY FREQUENCIES
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_freq
  end interface
  !
  interface
    subroutine ini_scale(line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY SCALE amplitude phase
      ! Multiply all data by the given scale factor (complex)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_scale
  end interface
  !
  interface
    subroutine sub_scale(arg,n,f,c)
      !---------------------------------------------------------------------
      ! Multiply complex array C of length N by factor F, or take its complex
      !  conjugate, according to ARG.
      !---------------------------------------------------------------------
      character(len=*) :: arg           !
      integer :: n                      !
      complex :: f                      !
      complex :: c(n)                   !
    end subroutine sub_scale
  end interface
  !
  interface
    subroutine ini_doppler(line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY DOPPLER
      ! Recompute Doppler correction
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_doppler
  end interface
  !
  interface
    subroutine exec_modify (buffer,end,write)
      character(len=*) :: buffer        !
      logical :: end                    !
      logical :: write                  !
    end subroutine exec_modify
  end interface
  !
  interface
    subroutine load_modify
    end subroutine load_modify
  end interface
  !
  interface
    subroutine clic_minmax(error)
      logical :: error                  !
    end subroutine clic_minmax
  end interface
  !
  interface
    subroutine sub_minmax (qsb, qband, qbas, qntch,   &
        data_c, data_l, pass_c, pass_l, error)
      integer :: qsb                       !
      integer :: qband                     !
      integer :: qbas                      !
      integer :: qntch                     !
      complex :: data_c(qsb, qband, qbas)  !
      complex :: data_l(qntch, qsb, qbas)  !
      complex :: pass_c(qsb, qband, qbas)  !
      complex :: pass_l(qntch, qsb, qbas)  !
      logical :: error                     !
    end subroutine sub_minmax
  end interface
  !
  interface
    subroutine ini_cable (line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	MODIFY Cable a1 a2 a3 a4 :
      !        correct for a permutation of the cables to the phasemeters
      ! 	MODIFY Cable CONTINUUOUS :
      !        correct the cable phases to be continuous
      !        (should use only the correlations, without LOCK flag,
      !          for which the phases have a meaning)
      ! 	MODIFY Cable FACTOR f:
      !        Assume the phase correction applied  has to be multiplied by f
      !        (replaces the old MODIFY CABL2)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_cable
  end interface
  !
  interface
    subroutine ini_bug (line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	MODIFY BUG number
      ! NUMBER   DATE         Description
      !
      ! 1        11/Nov 95    C02 to C06 badly calibrated.
      !                       (counter CC not init.d properly in DO_AUTOL)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_bug
  end interface
  !
  interface
    subroutine redo_autol  (mes, data, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Read in auto correlation function
      !     Increment the C_L and C_C buffers for the specified cal phase "mes".
      ! Input
      !     mes      I           Calibration phase
      !     DATA     REAL(*)     The data.
      !---------------------------------------------------------------------
      integer :: mes                    !
      real :: data(*)                   !
      logical :: error                  !
    end subroutine redo_autol
  end interface
  !
  interface
    subroutine redo_atmosc (data, error, ido)
      !---------------------------------------------------------------------
      ! CLIC
      !     REProcess temporal data: after a BUG # 1
      !     The right calibration factor is in CHOP, the wrong in COLD.
      ! Input
      !     DATA    COMPLEX(*)
      !---------------------------------------------------------------------
      complex :: data(*)                !
      logical :: error                  !
      integer :: ido                    !
    end subroutine redo_atmosc
  end interface
  !
  interface
    subroutine pol_chan(ipola,inbc,ichan,error)
      integer :: ipola                  ! Output: polarisation
      integer :: inbc                   ! Output: correlator entry
      integer :: ichan                  ! Input: channel
      logical :: error                  !
    end subroutine pol_chan
  end interface
  !
  interface
    subroutine clic_plot(line,error)
      use gildas_def
      use gkernel_types
      !---------------------------------------------------------------------
      !	PLOT [SAME] [obs_number] [NEXT] [FIRST] [CLOURE]
      !		/NOFRAME [pen]		option 1
      !		/RECORD			option 2
      !		/IDENTIFY		option 3
      !		/RESET			option 4
      !		/NODRAW			option 5
      !		/APPEND                 option 6
      !               /PHYSICAL               option 7
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_plot
  end interface
  !
  interface
    subroutine sub_plot(argum, ident, colour, pen_plot, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! this intermediate entry point needed for solve ... /plot
      !---------------------------------------------------------------------
      character(len=*) :: argum         !
      logical :: ident                  !
      logical :: colour                 !
      integer :: pen_plot               !
      logical :: error                  !
    end subroutine sub_plot
  end interface
  !
  interface
    subroutine sub_clos_pha(nant,nd,nb,x,y,w,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! same comment as for amplitudes.
      !---------------------------------------------------------------------
      integer :: nant                   !
      integer :: nd                     !
      integer :: nb                     !
      real :: x(nd,nb)                  !
      real :: y(nd,nb)                  !
      real :: w(nd,nb)                  !
      logical :: error                  !
    end subroutine sub_clos_pha
  end interface
  !
  interface
    subroutine sub_clos_amp(nant,nd,nb,x,y,w,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! take only 2 closures for each quadrangle (13.24/12.34 and 14.23/12.34)
      ! this would produce n*(n-1)*(n-2)*(n-3)|(4*3) closure
      ! relations of which only n*(n-3)|2 are independent. For 5 antennas these
      ! numbers are 10 and 5. For more than 5 antennas one has to sort out
      ! dependent relations because there will be more clos. relations produced
      ! than available baselines.
      ! Hint : start with 2 independent relations in a given quadrangle, and add
      ! relations with additional baselines; using only once each new baseline
      ! should give independent relations:
      !  (n-1)(n-4)|2 relations between n-1 antennas
      ! there are n-1 new baselines with a nth antenna
      ! add (n-2) relations (the first has 2 new baselines)
      ! one gets (n-1)(n-4)|2 + n-2 = (n^2-5n+4 +2n-4)|2 = n*(n-3)|2 relations.
      !---------------------------------------------------------------------
      integer :: nant                   !
      integer :: nd                     !
      integer :: nb                     !
      real :: x(nd,nb)                  !
      real :: y(nd,nb)                  !
      real :: w(nd,nb)                  !
      logical :: error                  !
    end subroutine sub_clos_amp
  end interface
  !
  interface
    subroutine sub_sub_plot(md,mb,x_data,y_data,w_data,   &
        i_data,b_data,j_data,argum,ident, colour, pen_plot, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! This version for GREG1 Version 7.2 and later : call list of
      ! GR4_HISTO was incorrect previously.
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      real :: i_data(md,mb)             !
      real :: b_data(md,mb)             !
      real :: j_data(mb)                !
      character(len=*) :: argum         !
      logical :: ident                  !
      logical :: colour                 !
      integer :: pen_plot               !
      logical :: error                  !
    end subroutine sub_sub_plot
  end interface
  !
  interface
    subroutine save_limits(ibox,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Save current GREG plot limits into display common, for box # IBOX
      !---------------------------------------------------------------------
      integer :: ibox                   !
      logical :: error                  !
    end subroutine save_limits
  end interface
  !
  interface
    subroutine gr4_ident (np,x,y,id,ind,colour,bars,b,error)
      !---------------------------------------------------------------------
      ! Plot data with identifying markers
      !---------------------------------------------------------------------
      integer :: np                     ! number of points
      real :: x(np)                     ! coordinates
      real :: y(np)                     ! coordinates
      integer :: id(np)                 ! identifying numbers
      integer :: ind(np)                ! identifying numbers
      logical :: colour                 !
      logical :: bars                   !
      real :: b(np)                     ! Error bars
      logical :: error                  !
    end subroutine gr4_ident
  end interface
  !
  interface
    subroutine user_info(name)
      character(len=*) :: name          !
    end subroutine user_info
  end interface
  !
  interface
    subroutine getvar(md,mb,x_data,y_data,w_data)
      use gildas_def
      use gkernel_types
      !---------------------------------------------------------------------
      ! Load the plot buffers with the contents of related variables
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
    end subroutine getvar
  end interface
  !
  interface
    subroutine reset_time
      use gildas_def
      ! Global
    end subroutine reset_time
  end interface
  !
  interface
    subroutine clic_print(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command PRINT [ BASELINE | DELAY | FLUX ]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_print
  end interface
  !
  interface
    subroutine show_luns
      ! Local
    end subroutine show_luns
  end interface
  !
  interface
    subroutine print_delay (date,error)
      use gildas_def
      logical :: date                   !
      logical :: error                  !
    end subroutine print_delay
  end interface
  !
  interface
    subroutine clic_set (line,error)
      use gildas_def
      use gkernel_types
      !---------------------------------------------------------------------
      ! CLIC Support routine for command
      !      SET Something [value1, value2, ...]
      ! Arguments :
      !	LINE	C*(*)	Command line		Input
      !	ERROR	L	Logical error flag	Output
      !----------------------------------------------------------------------
      !
      ! Options:
      !   1 	/LIMITS
      !   2 	/WINDOW
      !   3 	/ANTENNA
      !   4 	/DEFAULT
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_set
  end interface
  !
  interface
    subroutine get_antennas(line,iopt,nant,iant,needed,error)
      !---------------------------------------------------------------------
      ! Get list of antennas from /ANTENNA option
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      integer :: nant                   !
      integer :: iant(*)                !
      logical :: needed                 !
      logical :: error                  !
    end subroutine get_antennas
  end interface
  !
  interface
    subroutine clic_show(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC 	Support routine for command
      !	SHOW Something
      ! Arguments :
      !	LINE	C*(*)	Command line		Input
      !	ERROR	L	Logical error flag	Output
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_show
  end interface
  !
  interface
    subroutine show_criteria(argum,criteria,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     Display all or part of selection criteria
      !---------------------------------------------------------------------
      character(len=*) :: argum         !
      logical :: criteria               !
      logical :: error                  !
    end subroutine show_criteria
  end interface
  !
  interface
    subroutine show_general(argum,general,line,error)
      use gildas_def
      character(len=*) :: argum         !
      logical :: general                !
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine show_general
  end interface
  !
  interface
    subroutine show_display(argum,display,error)
      use gildas_def
      character(len=*) :: argum         !
      logical :: display                !
      logical :: error                  !
    end subroutine show_display
  end interface
  !
  interface
    subroutine clic_table(line,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! CLIC 	Internal routine.
      !	Prepare a UV Table (GILDAS Format)
      ! Command :
      ! 	TABLE file_name OLD|NEW
      ! 1		/RESAMPLE nchan rchan res VELOCITY|FREQUENCY
      ! 2		/DROP n1 n2
      ! 3		/COMPRESS tmax uvmax
      ! 4		/FREQUENCY name frest
      ! 5		/NOCHECK [SOURCE|POINTING|EPOCH|PHASE|SCAN]
      ! 6		/SCAN Replace w coordinate by scan number.  (set by default  !...)
      ! 7   /FFT  Fourier Transform resampling algorithm
      ! 8   /POSITIONS
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_table
  end interface
  !
  interface
    subroutine check_table(nc, nv, vis, nn, ndobs, ndscan, error)
      !---------------------------------------------------------------------
      ! CLIC TABLE command :
      ! Check that new data is really new
      !---------------------------------------------------------------------
      integer :: nc                     !
      integer :: nv                     !
      real :: vis(nc,nv)                !
      integer :: nn                     !
      integer :: ndobs(nn)              !
      integer :: ndscan(nn)             !
      logical :: error                  !
    end subroutine check_table
  end interface
  !
  interface
    subroutine dummy_visi(visi,n)
      !---------------------------------------------------------------------
      ! Add a dummy (zero weight) visibility in the table to defne the
      ! field of view for automatic imaging
      !---------------------------------------------------------------------
      integer :: n                      !
      real :: visi(n)                   !
    end subroutine dummy_visi
  end interface
  !
  interface
    subroutine check_window(fmin,fmax,imin,imax,inbc,nwin)
      ! 
      ! Checks windows overlap, optionally, reduce window number
      !
      ! Global
      include 'clic_parameter.inc'
      include 'clic_clic.inc'
      !
      integer :: inbc, nwin, imin(matmcal,2*(mwin_out+mrlband))
      integer :: imax(matmcal,2*(mwin_out+mrlband))
      real :: fmin(matmcal,2*(mwin_out+mrlband))
      real :: fmax(matmcal,2*(mwin_out+mrlband)) 
    end subroutine check_window
  end interface
  !
  interface
    subroutine print_window(fmin,fmax,nbc,nwin)
      ! 
      ! Print windows
      !
      ! Global
      include 'clic_parameter.inc'
      include 'clic_clic.inc'
      !
      integer :: nwin, nbc
      real :: fmin(matmcal,2*(mwin_out+mrlband))
      real :: fmax(matmcal,2*(mwin_out+mrlband))
    end subroutine print_window
  end interface
  !
  interface
    subroutine check_units(first,error)
      use gildas_def
      logical :: first      
      logical :: error
    end subroutine check_units
  end interface
  !
  interface
    subroutine vel_scale_2 (name,freq)
      !---------------------------------------------------------------------
      ! Updates frequencies in header according to last 'SET FREQUENCY' command
      !---------------------------------------------------------------------
      character(len=*) :: name          !
      real*8 :: freq                    !
    end subroutine vel_scale_2
  end interface
  !
  interface
    subroutine interpolate_2(x,xdim,xinc,xref,xval,w,   &
        y,ydim,yinc,yref,yval)
      !---------------------------------------------------------------------
      ! LAS	Internal routine
      !	Performs the linear interpolation/integration
      !
      ! 1) if actual interpolation (XINC <= YINC)
      !     this is a 2-point linear interpolation formula.
      !     the data is untouched if the resolution is unchanged and the shift
      !     is an integer number of channels.
      !
      ! 2) if not (XINC > YINC)
      !     boxcar convolution (width xinc-yinc) followed by linear interpolation
      !
      ! Arguments:
      !	X	R*4(*)	Output spectrum
      !	XDIM	I*4	Output pixel number
      !	XINC	R*8	Output first axis increment
      !	XREF	R*4	Output first axis reference pixel
      !	XVAL	R*4	Output first axis value at reference pixel
      ! 	Y	R*4(*)	Input spectrum
      !---------------------------------------------------------------------
      integer :: xdim                   !
      complex :: x(xdim)                !
      real*8 :: xinc                    !
      real :: xref                      !
      real*8 :: xval                    !
      real :: w(xdim)                   !
      integer :: ydim                   !
      complex :: y(ydim)                !
      real :: yinc                      !
      real :: yref                      !
      real*8 :: yval                    !
    end subroutine interpolate_2
  end interface
  !
  interface
    subroutine fft_interpolate(x,xdim,xinc,xref,xval,xwid,xshape,   &
        w,y,ydim,yinc,yref,yval,ywid,yshape,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC	Internal routine
      ! (from CLASS, but complex input/ouput)
      !	Performs the  interpolation/integration, using FFT.
      ! Arguments:
      !	X	C*8(*)	Output spectrum
      !	XDIM	I*4	Output pixel number
      !	XINC	R*8	Output first axis increment
      !	XREF	R*4	Output first axis reference pixel
      !	XVAL	R*4	Output first axis value at reference pixel
      !	XWID	R*4	Output channel width, in terms of channel separation.
      !           (e.g. oversampling parameter)
      !	XSHAPE  C*      Channel shape (TBox, TParabola, FBox, Ftriangle)
      !           TBox means a box in delay space (unsmoothed correlator, w=1)
      !           Ppar means a parabola in delay space (smoothed "      , w=1)
      !           FBox means a box in frequency space (square filter,     w=0.5)
      !           FTri means a triangle in frequency space
      !                       (Hanning smoothed square filter,            w=1)
      ! 	Y	C*8(*)	Input spectrum, same parameters ...
      !---------------------------------------------------------------------
      integer :: xdim                   !
      complex :: x(xdim)                !
      real*8 :: xinc                    !
      real :: xref                      !
      real*8 :: xval                    !
      real :: xwid                      !
      character(len=*) :: xshape        !
      real :: w(xdim)                   !
      integer :: ydim                   !
      complex :: y(ydim)                !
      real :: yinc                      !
      real :: yref                      !
      real*8 :: yval                    !
      real :: ywid                      !
      character(len=*) :: yshape        !
      logical :: error                  !
    end subroutine fft_interpolate
  end interface
  !
  interface
    subroutine reverse(n,r)
      !---------------------------------------------------------------------
      ! Reverses complex array R
      !---------------------------------------------------------------------
      integer :: n                      !
      complex :: r(n)                   !
    end subroutine reverse
  end interface
  !
  interface
    subroutine fft_offset(n,z,x)
      !---------------------------------------------------------------------
      ! offset by x channels
      !---------------------------------------------------------------------
      integer :: n                      !
      complex :: z(n)                   !
      real :: x                         !
    end subroutine fft_offset
  end interface
  !
  interface
    subroutine fft_factor(n,z,x)
      !---------------------------------------------------------------------
      ! Multiply Z complex array by X
      !---------------------------------------------------------------------
      integer :: n                      !
      complex :: z(n)                   !
      real :: x                         !
    end subroutine fft_factor
  end interface
  !
  interface
    subroutine fft_reconv(n,y,w,shape)
      !---------------------------------------------------------------------
      ! Reconvolution according to shape
      ! 'TB'    Time Box
      ! 'TP'    Time Parabola
      ! 'FB'    Frequency Box
      ! 'FT'    Frequency Triangle
      !---------------------------------------------------------------------
      integer :: n                      !
      complex :: y(n)                   !
      real :: w                         !
      character(len=*) :: shape         !
    end subroutine fft_reconv
  end interface
  !
  interface
    subroutine fft_deconv(n,y,w,shape)
      !---------------------------------------------------------------------
      ! Deconvolution according to shape
      ! 'TB'    Time Box
      ! 'TP'    Time Parabola
      ! 'FB'    Frequency Box
      ! 'FT'    Frequency Triangle
      !---------------------------------------------------------------------
      integer :: n                      !
      complex :: y(n)                   !
      real :: w                         !
      character(len=*) :: shape         !
    end subroutine fft_deconv
  end interface
  !
  interface
    subroutine fft_extend(c,n,m)
      !---------------------------------------------------------------------
      ! complete FFT with zeroes in high frequencies
      !---------------------------------------------------------------------
      integer :: m                      !
      complex :: c(m)                   !
      integer :: n                      !
    end subroutine fft_extend
  end interface
  !
  interface
    subroutine fft_cutoff(c,m,n)
      !---------------------------------------------------------------------
      ! cutoff FFT in high frequencies
      !---------------------------------------------------------------------
      integer :: m                      !
      complex :: c(m)                   !
      integer :: n                      !
    end subroutine fft_cutoff
  end interface
  !
  interface
    subroutine pfactor(n,p)
      !---------------------------------------------------------------------
      ! Returns in p the largest prime factor in n
      !---------------------------------------------------------------------
      integer :: n                      !
      integer :: p                      !
    end subroutine pfactor
  end interface
  !
  interface
    function prime(n,p)
      !---------------------------------------------------------------------
      ! Find if n is prime, returns the smallest integer divisor p.
      !---------------------------------------------------------------------
      logical :: prime                  !
      integer :: n                      !
      integer :: p                      !
    end function prime
  end interface
  !
  interface
    subroutine clic_variable(line,error)
      !---------------------------------------------------------------------
      ! CLIC command VARIABLE group ON|OFF
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_variable
  end interface
  !
  interface
    subroutine index_variable
      use gildas_def
      ! Global
    end subroutine index_variable
  end interface
  !
  interface
    subroutine rw_variables(off)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC 	Initialisation routine
      !	defines all RW variables
      !---------------------------------------------------------------------
      logical :: off                    !
    end subroutine rw_variables
  end interface
  !
  interface
    subroutine general_variables(off)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC 	Initialisation routine
      !	defines all SIC variables
      !---------------------------------------------------------------------
      logical :: off                    !
    end subroutine general_variables
  end interface
  !
  interface
    subroutine functions
      ! Global
    end subroutine functions
  end interface
  !
  interface
    function s_bessj1(x)
      real :: s_bessj1                  !
      real :: x                         !
    end function s_bessj1
  end interface
  !
  interface
    function s_bessj0(x)
      real :: s_bessj0                  !
      real :: x                         !
    end function s_bessj0
  end interface
  !
  interface
    subroutine ipb_write(chain,check,error)
      !---------------------------------------------------------------------
      !  Write a scan
      ! Argument:
      !	CHAIN	C*(*)	Write status ('NEW', 'EXTEND', 'UPDATE'
      !       CHECK	L	Check for duplicate write
      !	ERROR	L	Error return
      !---------------------------------------------------------------------
      character(len=*) :: chain         !
      logical :: check                  !
      logical :: error                  !
    end subroutine ipb_write
  end interface
  !
  interface
    subroutine ipb_close(error)
      use gbl_format
      !---------------------------------------------------------------------
      ! 	Close an observation. Reopens the output file.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine ipb_close
  end interface
  !
  interface
    subroutine ipb_done (error)
      !---------------------------------------------------------------------
      !	Close an observation, but not the file
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine ipb_done
  end interface
  !
  interface
    subroutine ipb_open (error)
      use gbl_format
      !---------------------------------------------------------------------
      !	Reopen the output file.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine ipb_open
  end interface
  !
  interface
    function ipb_size(dummy)
      real :: ipb_size                  !
      integer :: dummy                  !
    end function ipb_size
  end interface
  !
  interface
    function done(w)
      !---------------------------------------------------------------------
      !	Simple-minded message for IPB_WRITE
      !---------------------------------------------------------------------
      character(len=7) :: done          !
      character(len=*) :: w             !
    end function done
  end interface
  !
  interface
    subroutine write_scan (do_data,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Call Tree
      !	STORE_FLUX
      !	CLIC_MODIFY
      !	STORE_CAL
      !	STORE_FLAG
      !	STORE_PASS
      !---------------------------------------------------------------------
      logical :: do_data                !
      logical :: error                  !
    end subroutine write_scan
  end interface
  !
  interface
    subroutine clic_check (line,error)
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_check
  end interface
  !
  interface
    subroutine clic_check_contin (step,error)
      real :: step                      !
      logical :: error                  !
    end subroutine clic_check_contin
  end interface
  !
  interface
    subroutine sub_continuum (qsb, qband, qbas, qntch,   &
        data_c, data_l, pass_c, pass_l, step,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Check continuum correlator stability on current scan
      !
      ! Call Tree
      !	CLIC_CHECK
      !---------------------------------------------------------------------
      integer :: qsb                     !
      integer :: qband                   !
      integer :: qbas                    !
      integer :: qntch                   !
      complex :: data_c(qsb,qband,qbas)  !
      complex :: data_l(qntch,qsb,qbas)  !
      complex :: pass_c(qsb,qband,qbas)  !
      complex :: pass_l(qntch,qsb,qbas)  !
      real :: step                       !
      logical :: error                   !
    end subroutine sub_continuum
  end interface
  !
  interface
    subroutine clic_code
      ! Global
    end subroutine clic_code
  end interface
  !
  interface
    subroutine clic_convcod (gcode,icode,conver)
      use gbl_convert
      !---------------------------------------------------------------------
      !	GCODE system code
      !	ICODE file code
      !---------------------------------------------------------------------
      integer :: gcode                  !
      integer :: icode                  !
      integer :: conver                 !
    end subroutine clic_convcod
  end interface
  !
  interface
    subroutine convert_dh (iwin,nw,conve,nant,nbas,nband,npol)
      use gbl_convert
      !---------------------------------------------------------------------
      !	Convert data header IWIN using conversion code CONVE,
      !	assuming NW is the length of the data header.
      ! Arguments
      !	IWIN	I(*)	Array containing data header to be converted
      !	NW	I	Size of IWIN
      !	CONVE	I	Conversion code
      !	NANT	I	Number of antennas
      !	NBAS	I	Number of baselines
      !---------------------------------------------------------------------
      integer :: iwin(*)                !
      integer :: nw                     !
      integer :: conve                  !
      integer :: nant                   !
      integer :: nbas                   !
      integer :: nband                  !
      integer :: npol                   !
    end subroutine convert_dh
  end interface
  !
  interface
    subroutine convert_dhsub(iwin,nw,nant,nbas,nband,npol,r4,r8,i4,cc)
      !---------------------------------------------------------------------
      !	Convert data header IWIN using conversion code CONVE,
      !	assuming NW is the length of the data header.
      ! Arguments
      !	IWIN	I(*)	Array containing data header to be converted
      !	NW	I	Size of IWIN
      !	CONVE	I	Conversion code
      !	NANT	I	Number of antennas
      !	NBAS	I	Number of baselines
      !---------------------------------------------------------------------
      integer :: iwin(*)                !
      integer :: nw                     !
      integer :: nant                   !
      integer :: nbas                   !
      integer :: nband                  !
      integer :: npol                   !
      external :: r4                    !
      external :: r8                    !
      external :: i4                    !
      external :: cc                    !
    end subroutine convert_dhsub
  end interface
  !
  interface
    subroutine convert_data (ndata,data,jconve)
      use gbl_convert
      integer :: ndata                  !
      integer :: data(*)                !
      integer :: jconve                 !
    end subroutine convert_data
  end interface
  !
  interface
    subroutine copyw (n,in,out)
      integer :: n                      !
      integer :: in(n)                  !
      integer :: out(n)                 !
    end subroutine copyw
  end interface
  !
  interface
    subroutine crsec (scode,len,error)
      use gbl_constant
      use gbl_convert
      !---------------------------------------------------------------------
      !	Convert section from File data type to System data type
      !---------------------------------------------------------------------
      integer :: scode                  !
      integer :: len                    !
      logical :: error                  !
    end subroutine crsec
  end interface
  !
  interface
    subroutine scrsec (scode,len,error,r4,r8,i4,cc)
      use gbl_constant
      !---------------------------------------------------------------------
      !	Convert section from File data type to System data type
      !---------------------------------------------------------------------
      integer :: scode                  !
      integer :: len                    !
      logical :: error                  !
      external :: r4                    !
      external :: r8                    !
      external :: i4                    !
      external :: cc                    !
    end subroutine scrsec
  end interface
  !
  interface
    subroutine cubspl (n,tau,c1,c2,c3,c4,ibcbeg,ibcend,error)
      !---------------------------------------------------------------------
      ! CLIC Internal routine
      !
      !	Piecewise cubic spline interpolants computation; adapted from
      !	'A practical guide to SPLINES' , CARL DE BOOR ,
      !	Applied Mathematical Sciences, SPRINGER-VERLAG, VOL.27, P57-59 (1978).
      !
      ! Arguments :
      !     	N	I	Number of data points. assumed to be .ge. 2.	Input
      !	TAU	R*4(N)	Abscissae of data points, strictly monotonous	Input
      !	C1	R*4(N)	Ordinates of data points			Input
      !     	IBCBEG	I							Input
      !	IBCEND  I	Boundary condition indicators, and		Input
      !	C1	R*4(N)	First polynomial coefficient of spline	Output
      !	C2	R*4(N)	Second   --         --           --	Output/Input
      !	C3	R*4(N)	Third    --         --           --	Output
      !	C4	R*4(N)	Fourth   --         --           --	Output
      !
      ! 	C2(1), C2(N)  are boundary condition information.
      !	Specifically,
      !	IBCBEG = 0  means no boundary condition at tau(1) is given.
      !		In this case, the not-a-knot condition is used, i.e. the
      !		jump in the third derivative across TAU(2) is forced to
      !		zero, thus the first and the second cubic polynomial pieces
      !		are made to coincide.
      !        IBCBEG = 1 means that the slope at TAU(1) is made to equal
      !		C2(1), supplied by input.
      !        IBCBEG = 2 means that the second derivative at TAU(1) is
      !		made to equal C2(1), supplied by input.
      !        IBCEND = 0, 1, or 2 has analogous meaning concerning the
      !		boundary condition at TAU(N), with the additional
      !		information taken from C2(N).
      !
      !	CJ(I), J=1,...,4; I=1,...,L (= N-1) are the polynomial
      !	coefficients of the cubic interpolating spline. Precisely,
      !	in the interval    [ TAU(I), TAU(I+1) ]    the spline F
      !	is given by
      !		F(X) = C1(I)+H*(C2(I)+H*(C3(I)+H*C4(I)/3.)/2.)
      !	where H = X - TAU(I).
      !
      !	In other words, for I=1,...,N, C2(I) and C3(I) are respectively
      !	equal to the values of the first and second derivatives of
      !	the interpolating spline, and C4(I) is equal to the third
      !	derivative of the interpolating spline in the interval
      !	[ TAU(I), TAU(I+1) ]. C4(N) is meaningless and is set to 0.
      !---------------------------------------------------------------------
      integer :: n                      !
      real :: tau(n)                    !
      real :: c1(n)                     !
      real :: c2(n)                     !
      real :: c3(n)                     !
      real :: c4(n)                     !
      integer :: ibcbeg                 !
      integer :: ibcend                 !
      logical :: error                  !
    end subroutine cubspl
  end interface
  !
  interface
    subroutine cubint (n,x,y,cm,ci)
      !---------------------------------------------------------------------
      ! Calculates the array of integrals required in SPLIF and SPLIFK
      !
      !     N      Number of data points (N is assumed .GE. 2).
      !     (X(I), Y(I), I=1,...,N) abcissae and ordinates of the points
      !            (X is assumed to be strictly increasing or decreasing).
      !     CM(I)  array of values of the second derivative of the spline-
      !            function at abscissae (X(I), I=1,...,N).
      !     CI(I)  array of the integrals of the cubic spline interpolation
      !            function from X(1) TO X(I).
      !---------------------------------------------------------------------
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: ci(n)                     !
    end subroutine cubint
  end interface
  !
  interface
    subroutine cuberr (n,x,y,ycub,err,iw,nw)
      !---------------------------------------------------------------------
      !       Crude estimate of spline-interpolation errors
      !       N       :number of pivots
      !       X       :array of pivots
      !       Y       :array of values of the function
      !       YCUB    :array of spline second-derivatives
      !       ERR     :output array for error estimates
      !       ( ERR(K): error in the interval [X(K-1)...X(K)] for K=3...N-1,
      !         ERR(1), ERR(2) and ERR(N) are set to 0.)
      !       IW      :printing device. set iw=0 to cancel the print.
      !                the estimated interpolation errors are listed for
      !                the worst intervals.
      !       NW      :approximate number of errors printed.
      !---------------------------------------------------------------------
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: ycub(n)                   !
      real :: err(n)                    !
      integer :: iw                     !
      integer :: nw                     !
    end subroutine cuberr
  end interface
  !
  interface
    function kspl (n,x,t,k)
      !---------------------------------------------------------------------
      !     The integer function KSPL returns the index KSPL such that the
      !     argument T lies within the interval [X(KSPL-1)...X(KSPL].
      !     In case of extrapolation, KSPL is forced to the value 2 or N.
      !
      !     N      number of data points (N is assumed .GE. 2).
      !     (X(I), I=1,...,N) abcissae of the points
      !            (X is assumed to be strictly increasing or decreasing).
      !     T      Argument for which the spline function is to be determined.
      !     K      Initial guess for KSPL.
      !---------------------------------------------------------------------
      integer :: kspl                   !
      integer :: n                      !
      real :: x(n)                      !
      real :: t                         !
      integer :: k                      !
    end function kspl
  end interface
  !
  interface
    function splf (n,x,y,cm,t,k)
      !---------------------------------------------------------------------
      !     *** Calculation of a cubic spline interpolation function. ***
      !     SPLF computes the value of a cubic polynomial piece within an
      !     interval given the abscissae, ordinates and second derivatives
      !     of the piece at both ends.
      !
      !     N      number of data points (N is assumed .ge. 2).
      !     (X(I), Y(I), I=1,...,N) abcissae and ordinates of the points
      !            (X is assumed to be strictly increasing or decreasing).
      !     CM(I)  array of values of the second derivative of the spline-
      !            function at abscissae (X(I), I=1,...,N).
      !     T      argument for which the spline function is to be determined.
      !     K      the user-supplied index K selects the cubic piece
      !            associated with the interval [X(K-1), X(K)]. K is assumed
      !            to be .GE. 2 and .LE. N.  The spline interpolation is
      !            obtained when T lies in this interval.
      !---------------------------------------------------------------------
      real :: splf                      !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splf
  end interface
  !
  interface
    function splf1 (n,x,y,cm,t,k)
      !---------------------------------------------------------------------
      !     Calculation of the first derivative of a cubic spline function.
      !     Arguments are the same as for SPLF.
      !---------------------------------------------------------------------
      real :: splf1                     !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splf1
  end interface
  !
  interface
    function splf2 (n,x,y,cm,t,k)
      !---------------------------------------------------------------------
      !     Calculation of the second derivative of a cubic spline function.
      !     Arguments are the same as for SPLF.
      !---------------------------------------------------------------------
      real :: splf2                     !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splf2
  end interface
  !
  interface
    function splfi (n,x,y,cm,ci,t,k)
      !---------------------------------------------------------------------
      !     *** calculates the integral of a cubic spline interpolation
      !         function from X(1) to X(T).
      !
      !     *** N,X,Y,CM,T,K are the same as in SPLF.
      !     *** CI(I): array of integrals of the spline function from X(1)
      !                to X(I). The array CI can be computed with CUBINT.
      !---------------------------------------------------------------------
      real :: splfi                     !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: ci(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splfi
  end interface
  !
  interface
    function splfk (n,x,y,cm,t,k)
      !---------------------------------------------------------------------
      !     *** Calculation of a cubic spline interpolation function. ***
      !     SPLFK computes the value of a cubic polynomial piece within an
      !     interval given the abscissae, ordinates and second derivatives
      !     of the piece at both ends.
      !
      !     *** The arguments are the same as those of SPLF, excepted for K.
      !     *** K:  computed as K=KSPL(N,X,T,K)
      !---------------------------------------------------------------------
      real :: splfk                     !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splfk
  end interface
  !
  interface
    function splf1k (n,x,y,cm,t,k)
      !---------------------------------------------------------------------
      !     Calculation of the first derivative of a cubic spline function.
      !     *** The arguments are the same as those of SPLF1, excepted for K.
      !     *** K:  computed as K=KSPL(N,X,T,K)
      !---------------------------------------------------------------------
      real :: splf1k                    !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splf1k
  end interface
  !
  interface
    function splf2k (n,x,y,cm,t,k)
      !---------------------------------------------------------------------
      !     Calculation of the second derivative of a cubic spline function.
      !     *** The arguments are the same as those of SPLF2, excepted for K.
      !     *** K:  computed as K=KSPL(N,X,T,K)
      !---------------------------------------------------------------------
      real :: splf2k                    !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splf2k
  end interface
  !
  interface
    function splfik (n,x,y,cm,ci,t,k)
      !---------------------------------------------------------------------
      !     Calculation of the integral of a cubic spline function.
      !     *** The arguments are the same as those of SPLFI, excepted for K.
      !     *** K:  computed as K=KSPL(N,X,T,K)
      !---------------------------------------------------------------------
      real :: splfik                    !
      integer :: n                      !
      real :: x(n)                      !
      real :: y(n)                      !
      real :: cm(n)                     !
      real :: ci(n)                     !
      real :: t                         !
      integer :: k                      !
    end function splfik
  end interface
  !
  interface
    subroutine cut_header (ndrop,newfreq,positions,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Prepare header for shortened records
      ! Call Tree
      !	CLIC_TABLE
      !---------------------------------------------------------------------
      integer, intent(in) :: ndrop(2)               !
      real(8), intent(in) :: newfreq                 !
      logical, intent(out) :: error                  !
      logical, intent(in) :: positions
    end subroutine cut_header
  end interface
  !
  interface
    subroutine r_to_y
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Updates table header parameters (Y) from CLIC Header parameters (R)
      !---------------------------------------------------------------------
      ! Global
    end subroutine r_to_y
  end interface
  !
  interface
    subroutine cwsec (scode,len,error)
      use gbl_constant
      use gbl_convert
      !---------------------------------------------------------------------
      !	Convert section from System data type to File data type
      !---------------------------------------------------------------------
      integer :: scode                  !
      integer :: len                    !
      logical :: error                  !
    end subroutine cwsec
  end interface
  !
  interface
    subroutine scwsec (scode,len,error,r4,r8,i4,cc)
      use gbl_constant
      !---------------------------------------------------------------------
      !	Convert section from System data type to File data type
      !---------------------------------------------------------------------
      integer :: scode                  !
      integer :: len                    !
      logical :: error                  !
      external :: r4                    !
      external :: r8                    !
      external :: i4                    !
      external :: cc                    !
    end subroutine scwsec
  end interface
  !
  interface
    subroutine cdate(date,idate,error)
      !---------------------------------------------------------------------
      ! ASTRO Utility routine
      !	Converts the date '01-JAN-1984' in internal code
      ! Arguments
      !	DATE 	C*(*)	String date			Input
      !	IDATE 	I	Internal coded date value	Output
      !	ERROR 	L	Logical error flag		Output
      ! Subroutines :
      !	(SIC)
      !	DATJ,JDAT
      ! No commons
      !---------------------------------------------------------------------
      character(len=*) :: date          !
      integer :: idate                  !
      logical :: error                  !
    end subroutine cdate
  end interface
  !
  interface
    subroutine datj(id,im,iy,jour)
      !---------------------------------------------------------------------
      ! ASTRO	Utility routine
      !	Convert a date (id/im/iy) in number of days till the
      !	01/01/1984
      ! Arguments :
      !	ID	I	Day number		Input
      !	IM	I	Month number		Input
      !	IY	I	Year number		Input
      !	JOUR	I	Elapsed day count	Output
      ! Subroutine
      !	JULDA
      !---------------------------------------------------------------------
      integer :: id                     !
      integer :: im                     !
      integer :: iy                     !
      integer :: jour                   !
    end subroutine datj
  end interface
  !
  interface
    function julda(nyr)
      !---------------------------------------------------------------------
      ! ASTRO	Utility routine
      !	Returns Julian date of 1 day of Year
      ! Arguments
      !	NYR	I	Year				Input
      !	JULDA	I	Date returned			Output
      !---------------------------------------------------------------------
      integer :: julda                  !
      integer :: nyr                    !
    end function julda
  end interface
  !
  interface
    subroutine adate(chain,id,im,iy,error)
      character(len=*) :: chain         !
      integer :: id                     !
      integer :: im                     !
      integer :: iy                     !
      logical :: error                  !
    end subroutine adate
  end interface
  !
  interface
    subroutine newdata (line,error)
      !---------------------------------------------------------------------
      ! SAS	Support routine for command
      !	NEW_DATA
      !	Find a set of new observations of any kind.
      ! Arguments :
      !	LINE	C*(*)	Command line			Input
      !	ERROR	L	Logical error flag		Output
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine newdata
  end interface
  !
  interface
    subroutine eix(first_new,last_new,change,error)
      use gbl_format
      !---------------------------------------------------------------------
      ! LAS
      !	Examine input index, until new data is written.
      !	FIRST_NEW = -1 to initialise
      !---------------------------------------------------------------------
      integer :: first_new              ! First new index entry
      integer :: last_new               ! Last new index entry
      logical :: change                 ! Change type of observations
      logical :: error                  ! Error return
    end subroutine eix
  end interface
  !
  interface
    subroutine file(line,error)
      !---------------------------------------------------------------------
      ! SAS	Support routine for command
      !	FILE 	I[N] 	File_name
      !		O[UT]	File_name	[NEW]
      !               RAW	[PROJ] [DATE] [/WINDOW] [/DIRECTORY]
      ! Arguments :
      !	LINE	C*(*)	Command line			Input
      !	ERROR	L	Logical error flag		Output
      ! (5-mar-1985)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine file
  end interface
  !
  interface
    subroutine check_extension (ftype, flen, error)
      character(len=*) :: ftype         !
      integer :: flen                   !
      logical :: error                  !
    end subroutine check_extension
  end interface
  !
  interface
    subroutine file_raw(window,proj,date,name,error)
      !---------------------------------------------------------------------
      ! Get the name of data file with stardard name
      !     - corresponding with optional PROJ and DATE
      !     - found in the IPB_DATAi: logical directory list
      ! if not WINDOW, the first file is selected to be opened;
      ! if WINDOW, a window is opened to select the name in the available
      ! choices in the list.
      !---------------------------------------------------------------------
      logical :: window                 !
      character(len=*) :: proj          !
      character(len=*) :: date          !
      character(len=*) :: name          !
      logical :: error                  !
    end subroutine file_raw
  end interface
  !
  interface
    subroutine check_raw_name(name,ext,tproj,tdate,long,err)
      character(len=*) :: name          !
      character(len=*) :: ext           !
      character(len=*) :: tproj         !
      character(len=*) :: tdate         !
      character(len=*) :: long          !
      logical :: err                    !
    end subroutine check_raw_name
  end interface
  !
  interface
    subroutine file_in(name,error)
      character(len=*) :: name          !
      logical :: error                  !
    end subroutine file_in
  end interface
  !
  interface
    subroutine decrypt(short,long,error)
      !---------------------------------------------------------------------
      ! Decrypt short name into long one.  Only correct if the
      ! long name has the format of dd-mm-yyyy-proj
      !---------------------------------------------------------------------
      character(len=*) :: short         !
      character(len=*) :: long          !
      logical :: error                  !
    end subroutine decrypt
  end interface
  !
  interface
    subroutine encrypt(long,short,error)
      !---------------------------------------------------------------------
      ! Encrypt long name into short one.  Only correct if the
      ! long name has the format of dd-mm-yyyy-proj
      !---------------------------------------------------------------------
      character(len=*) :: long          !
      character(len=*) :: short         !
      logical :: error                  !
    end subroutine encrypt
  end interface
  !
  interface
    subroutine file_directory(line,error)
      !---------------------------------------------------------------------
      ! Interactively edit the default raw data file directories.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine file_directory
  end interface
  !
  interface
    subroutine find (line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! SAS	Support routine for command
      !	FIND
      !	Find a set of observations according to setup searh rules
      !	and command options.
      ! Arguments :
      !	LINE	C*(*)	Command line			Input
      !	ERROR	L	Logical error flag		Output
      ! (5-mar-1985)
      ! #1	Modified	13-Nov-1985	S.Guilloteau
      !	Use of "...*"	for source, line and telescope names
      ! 	and FIND/OFFSET as a restriction over the range defined in
      !	SET RANGE or SET OFFSET
      ! #2	Modified	20-Mar-1986	S.Guilloteau
      !	No modification indeed but FIX behaviour has changed,
      !	and IFIND is now dummy
      ! #3	Modified	11-Nov-1986	S.Guilloteau
      !	Support for continuum
      ! #4	Number of observations in index
      ! #5	Quality flag /REDUCED option suppressed.
      ! #6	Modified 	05-Oct-1987	R.Lucas
      !	FIND NEWDATA Option
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine find
  end interface
  !
  interface
    subroutine nix(error)
      use gbl_format
      use gbl_constant
      !---------------------------------------------------------------------
      ! LAS
      !	Examine input index, to see if any change occured.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine nix
  end interface
  !
  interface
    subroutine clic_derive(gg,gg2,fcn)
      !---------------------------------------------------------------------
      ! SAS	Internal routine
      !	Calculates the first derivatives of FCN (GG),
      !	either by finite differences or by transforming the user-
      !	supplied derivatives to internal coordinates,
      !	according to whether ISW(3) is zero or one.
      !	if ISW(3) = 0, an error estimate GG2  is available
      !
      !	External routine FCN added to allow use of various
      !	minimization functions.
      !
      ! Arguments :
      !	GG	R*8 (*)	First derivatives of function FCN	Output
      !	GG2	R*8 (*)	Error estimate for variables		Output
      !	FCN	Ext	Function to minimise			Input
      !---------------------------------------------------------------------
      real*8 ::   gg                    !
      real*8 ::   gg2                   !
      external :: fcn                   !
    end subroutine clic_derive
  end interface
  !
  interface
    subroutine clic_extoin(pint)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Transforms the external parameter values X  to internal
      !	values in the dense array PINT.   Function PINTF is used.
      ! Arguments :
      !	PINT	R*8 (2)	Dense array
      !---------------------------------------------------------------------
      real*8 ::   pint                  !
    end subroutine clic_extoin
  end interface
  !
  interface
    subroutine clic_fixpar(i2,kode,ilax)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Removes parameter I2 from the internal (variable) parameter
      !	list, and arranges the rest of the list to fill the hole.
      !	If KODE=0, I2 is an external number, otherwise internal.
      !	ILAX is returned as the external number of the parameter.
      ! Arguments :
      !	I2	I	Internal or external parameter number	Input
      !	KODE	I	Indicates Internal (1) or External (0)	Input
      !	ILAX	I	External number				Output
      !---------------------------------------------------------------------
      integer ::  i2                    !
      integer ::  kode                  !
      integer ::  ilax                  !
    end subroutine clic_fixpar
  end interface
  !
  interface
    subroutine clic_hesse(fcn)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Calculates the full second-derivative matrix of FCN
      !	by taking finite differences.   Includes some safeguards
      !	against non-positive-definite matrices, and it may set
      !	 off-diagonal elements to zero in attempt to force
      !	positiveness.
      ! Arguments :
      !	FCN	Ext	Function to minimise
      !---------------------------------------------------------------------
      external :: fcn                   !
    end subroutine clic_hesse
  end interface
  !
  interface
    subroutine clic_intoex(pint)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Transforms from internal coordinates (PINT) to external
      !	parameters (U).   The minimizing routines which work in
      !	internal coordinates call this routine before calling FCN.
      ! Arguments :
      !	PINT	R*8 (2)	Dense array			Input
      !---------------------------------------------------------------------
      real*8 ::  pint                   !
    end subroutine clic_intoex
  end interface
  !
  interface
    subroutine clic_migrad(fcn,ier)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Performs a local function minimization using basically the
      !	method of DAVIDON-FLETCHER-POWELL as modified by FLETCHER
      !	Ref. -- FLETCHER, COMP.J. 13,317 (1970)   "Switching method"
      ! Arguments :
      !	FCN	Ext	Function to minimise			Input
      !	IER 	I	Error condition flag			Output
      !---------------------------------------------------------------------
      external :: fcn                   !
      integer :: ier                    !
    end subroutine clic_migrad
  end interface
  !
  interface
    function clic_pintf(pexti,i)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Calculates the internal parameter value PINTF corresponding
      !	to the external value PEXTI for parameter I.
      ! Arguments :
      !	PEXTI	R*8	External value			Input
      !	I	I	Parameter number		Input
      !	PINTF	R*8	Internal value			Output
      !---------------------------------------------------------------------
      real*8 ::  clic_pintf                  !
      real*8 ::  pexti                  !
      integer :: i                      !
    end function clic_pintf
  end interface
  !
  interface
    subroutine clic_razzia (ynew,pnew,ier)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Called only by SIMPLEX (and IMPROV) to add a new point
      !	and remove an old one from the current simplex, and get the
      !	estimated distance to minimum.
      ! Arguments :
      !	YNEW	R*8
      !	PNEW	R*8 (*)
      !	ERROR	L	Logical error flag
      !---------------------------------------------------------------------
      real*8 ::  ynew                   !
      real*8 ::  pnew                   !
      integer :: ier                    !
    end subroutine clic_razzia
  end interface
  !
  interface
    subroutine clic_restor(k)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Restores a fixed parameter to variable status
      !	by inserting it into the internal parameter list at the
      !	appropriate place.
      !
      !       K = 0 means restore all parameters
      !       K = 1 means restore the last parameter fixed
      !       K = -I means restore external parameter I (if possible)
      !       IQ = fix-location where internal parameters were stored
      !       IR = external number of parameter being restored
      !       IS = internal number of parameter being restored
      ! Arguments :
      !	K	I	Code for operation			Input
      !----------------------------------------------------------------------
      ! S. Guilloteau Patch
      !---------------------------------------------------------------------
      integer ::  k                     !
    end subroutine clic_restor
  end interface
  !
  interface
    subroutine clic_simplx(fcn,ier)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Performs a minimization using the SIMPLEX method of NELDER
      !	and MEAD (REF. -- COMP. J. 7,308 (1965)).
      ! Arguments :
      !	FCN	Ext	Function to minimize		Input
      !	IER	I	Error condition flag		Output
      !----------------------------------------------------------------------
      ! S. Guilloteau Patch
      !---------------------------------------------------------------------
      external :: fcn                   !
      integer ::  ier                   !
    end subroutine clic_simplx
  end interface
  !
  interface
    subroutine ucopy(a,b,n)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Copies N words from A to B
      !---------------------------------------------------------------------
      real*8 ::  a                      !
      real*8 ::  b                      !
      integer :: n                      !
    end subroutine ucopy
  end interface
  !
  interface
    subroutine clic_vermin(a,l,m,n,ifail)
      !---------------------------------------------------------------------
      ! MINUIT	Internal routine
      !	Inverts a symmetric matrix.   Matrix is first scaled to
      !	have all ones on the diagonal (equivalent to change of units)
      !	but no pivoting is done since matrix is positive-definite.
      !
      ! Arguments :
      !	A	R*8(L,M)	Matrix to be inverted	Input/Output
      !	L	I		First dimension		Input
      !	M	I		Second dimension	Input
      !	N	I
      !	IFAIL	I		Error code		Output
      !----------------------------------------------------------------------
      ! S. Guilloteau Patch
      !---------------------------------------------------------------------
      real*8 ::  a                      !
      integer ::  l                     !
      integer ::  m                     !
      integer ::  n                     !
      integer ::  ifail                 !
    end subroutine clic_vermin
  end interface
  !
  interface
    subroutine fix (nfound,ifound,update,new_wanted,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	search the index of the input data file
      !	for parameters in the common rindex.inc
      !	entry FOX does the same thing for the output data file
      !
      !	IFOUND is unused when UPDATE = .TRUE.
      !	NEW_WANTED = .TRUE. : wait for new data to be present in input file.
      !	(RL 05-oct-1987)
      !---------------------------------------------------------------------
      integer :: nfound                 !
      integer :: ifound(1)              !
      logical :: update                 !
      logical :: new_wanted             !
      logical :: error                  !
    end subroutine fix
  end interface
  !
  interface
    subroutine fox(nfound,ifound,error)
      !---------------------------------------------------------------------
      ! RW	Same as FIX for output file
      !---------------------------------------------------------------------
      integer :: nfound                 !
      integer :: ifound(1)              !
      logical :: error                  !
    end subroutine fox
  end interface
  !
  interface
    subroutine fitgauss (fcn,liter,ier)
      !---------------------------------------------------------------------
      ! SAS	Internal routine
      !	Setup and starts a GAUSS fit minimisation using MINUIT
      ! Arguments :
      !	FCN	Ext.	Function to be mininized	Input
      !	LITER	L	Logical of iterate fit		Input
      !	IER	I	error code 			Output
      ! (5-mar-1985)
      !---------------------------------------------------------------------
      external :: fcn                   !
      logical :: liter                  !
      integer :: ier                    !
    end subroutine fitgauss
  end interface
  !
  interface
    subroutine itegauss (ier)
      !---------------------------------------------------------------------
      ! LAS	Support routine for command
      !	ITERATE
      !	Case of gaussian profiles
      !---------------------------------------------------------------------
      integer :: ier                    !
    end subroutine itegauss
  end interface
  !
  interface
    subroutine midgauss (ifatal,liter)
      !---------------------------------------------------------------------
      ! ANALYSE	Internal routine
      !	Start a gaussian fit by building the PAR array and internal
      !	variable used by Minuit
      ! Arguments
      !	IFATAL	I	Number of fatal errors		Output
      !	LITER	L	Iterate a fit			Input
      !---------------------------------------------------------------------
      integer :: ifatal                 !
      logical :: liter                  !
    end subroutine midgauss
  end interface
  !
  interface
    subroutine mingauss (npar,g,f,x,iflag)
      !---------------------------------------------------------------------
      ! LAS	Internal routine
      !	Function to be minimized in the gaussian fit.
      !	By using 3 hidden parameters, the method allows dependent
      !	and independent gaussians to be fitted. The computation is
      !	highly optimised, but take care of R*4 and R*8 values !...
      !	Basic parameters are Area, Position and Width.
      ! Arguments :
      !	NPAR	I	Number of parameters		Input
      !	G	R*8(1)	Array of derivatives		Output
      !	F	R*8	Function value			Output
      !	X	R*8(1)	Parameter values		Input
      !	IFLAG	I	Code operation			Input
      ! (5-mar-1985)
      !---------------------------------------------------------------------
      integer :: npar                   !
      real*8 :: g(*)                    !
      real*8 :: f                       !
      real*8 :: x(*)                    !
      integer :: iflag                  !
    end subroutine mingauss
  end interface
  !
  interface
    function progauss (a,m)
      !---------------------------------------------------------------------
      ! ANALYSE	(Internal routine)
      !	M = 0
      ! 	Computes the composite profile from sum of all gaussians.
      !	M # 0
      !	Computes the profile from gaussian number M
      !
      ! S. Guilloteau 19-Dec-1984
      !---------------------------------------------------------------------
      real*4 :: progauss                !
      real*4 :: a                       !
      integer :: m                      !
    end function progauss
  end interface
  !
  interface
    subroutine inigauss
      !---------------------------------------------------------------------
      ! CLIC 	Internal routine
      !	Start a gaussian fit by building the PAR array
      !---------------------------------------------------------------------
      ! Global
    end subroutine inigauss
  end interface
  !
  interface
    subroutine get_first(forward,error)
      use gildas_def
      logical :: forward                !
      logical :: error                  !
    end subroutine get_first
  end interface
  !
  interface
    subroutine get_data(ndata,d,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Read data
      !---------------------------------------------------------------------
      integer :: ndata                   !
      integer(kind=address_length) :: d  !
      logical :: error                   !
    end subroutine get_data
  end interface
  !
  interface
    subroutine update_header
      use gildas_def
      !---------------------------------------------------------------------
      ! Read data
      !---------------------------------------------------------------------
      ! Global
    end subroutine update_header
  end interface
  !
  interface
    subroutine clic_header(line,error)
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_header
  end interface
  !
  interface
    subroutine tagout(line,error)
      use gkernel_types
      !---------------------------------------------------------------------
      ! CLASS	Support routine for command
      !	TAG Quality List_of_scans
      ! Argument
      !	LINE	C*(*)	Command line
      !	ERROR	L	Error flag
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine tagout
  end interface
  !
  interface
    subroutine init_clic
      use gildas_def
      use gbl_message
      ! Global
    end subroutine init_clic
  end interface
  !
  interface
    subroutine check_commons
      use gildas_def
      ! Global
    end subroutine check_commons
  end interface
  !
  interface
    subroutine check_iaddress(error,name,int)
      use gildas_def
      logical :: error                  !
      character(len=*) :: name          !
      integer :: int                    !
    end subroutine check_iaddress
  end interface
  !
  interface
    subroutine check_caddress(error,name,int)
      use gildas_def
      logical :: error                  !
      character(len=*) :: name          !
      character(len=*) :: int           !
    end subroutine check_caddress
  end interface
  !
  interface
    subroutine input(spec,nspec,error)
      use gbl_format
      use gbl_convert
      !---------------------------------------------------------------------
      ! CLASS	Internal routine
      !	Close any input file, set the file specification for input,
      !	open this file.
      ! Arguments :
      ! 	SPEC	C*(*)	File name including extension	Input
      !	NSPEC	I	Length of SPEC			Input
      !	ERROR	L	Logical error flag		Output
      !---------------------------------------------------------------------
      character(len=*) :: spec          !
      integer :: nspec                  !
      logical :: error                  !
    end subroutine input
  end interface
  !
  interface
    subroutine liste(line,error)
      use gildas_def
      use gkernel_types
      !---------------------------------------------------------------------
      ! CLIC 	Internal routine
      !	Liste breve,normale ou longue d'un ensemble d'observations
      !	Construit par find (LIST)
      !	Tout le fichier d'entree (LIST IN[file])
      !	Tout le fichier de sortie (LIST OU[tfile])
      !	LIST OUT supported except for LONG
      !	/OUTPUT File	Send to a file
      !	/FLAGS  (4)
      !	/SHORT	(5)
      !	/PROJECT (6)
      !       /SOURCE   (7)
      !       /VARIABLE (8) [/FORMAT (9)]
      ! Arguments :
      !	LINE	C*(*)	Command line			Input
      !	ERROR	L	Logical error flag		Output
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine liste
  end interface
  !
  interface
    subroutine list_all_flags(lun,flagged,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! List flagged data from scan in R on unit LUN
      ! set FLAGGED logical if flags were found.
      !---------------------------------------------------------------------
      integer :: lun                    !
      logical :: flagged                !
      logical :: error                  !
    end subroutine list_all_flags
  end interface
  !
  interface
    subroutine message(prio,sever,progra,line)
      use gbl_message
      !---------------------------------------------------------------------
      ! CLIC  Internal routine
      !
      !       Output message according to current priority levels, and
      !       set the severity information
      !---------------------------------------------------------------------
      integer :: prio                   !
      integer :: sever                  !
      character(len=*) :: progra        !
      character(len=*) :: line          !
    end subroutine message
  end interface
  !
  interface
    subroutine message_level(ip)
      !---------------------------------------------------------------------
      ! entry point to set priority level returns the previous value.
      !---------------------------------------------------------------------
      integer :: ip                     !
    end subroutine message_level
  end interface
  !
  interface
    subroutine message_init(line,it,ip)
      !---------------------------------------------------------------------
      ! Init of priorities. Obsolescent (SB, 16-jun-2008)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: it                     !
      integer :: ip                     !
    end subroutine message_init
  end interface
  !
  interface
    subroutine noir (cin,cout,n)
      !---------------------------------------------------------------------
      ! ASTRO	Utility routine
      ! 	Reduce all separators to one.
      ! Arguments :
      !	CIN	C*(*)	Input character string
      !	COUT	C*(*)	Output character string
      !	N	I	Length of COUT
      !---------------------------------------------------------------------
      character(len=*) :: cin           !
      character(len=*) :: cout          !
      integer :: n                      !
    end subroutine noir
  end interface
  !
  interface
    subroutine newformat(nd,d)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     convert the data section to new format (spectral data averaged).
      ! Arguments:
      !     ND      INTEGER    Data length (Input/Output)
      !     D       INTEGER    Data address
      !---------------------------------------------------------------------
      integer :: nd                      !
      integer(kind=address_length) :: d  !
    end subroutine newformat
  end interface
  !
  interface
    subroutine sub_newformat(inbuf,outbuf,nd,d)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     convert the data section to new format (spectral data averaged).
      ! Arguments:
      !     ND      INTEGER    Data length (Input/Output)
      !     D       INTEGER    Data address
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      real :: inbuf(mdatac+mdatal)       !
      real :: outbuf(mdatac+mdatal)      !
      integer :: nd                      !
      integer(kind=address_length) :: d  !
    end subroutine sub_newformat
  end interface
  !
  interface
    subroutine amodec(nl,nb,cin,na,rout)
      !---------------------------------------------------------------------
      ! convert from old data order for continuum, autocorrelations
      !
      !	NL       INTEGER     Number of channels
      !       NB       INTEGER     Number of baselines
      !       NA       INTEGER     Number of antennas
      !       CIN(2,NL,NB) COMPLEX Input pseudo-complex autocorrelation data
      !       ROUT(NL,NA)  REAL    Output autocorrelation data
      !---------------------------------------------------------------------
      integer :: nl                     !
      integer :: nb                     !
      complex :: cin(2, nl, nb)         !
      integer :: na                     !
      real :: rout(nl, na)              !
    end subroutine amodec
  end interface
  !
  interface
    subroutine amodel(nl,nb,cin,na,rout)
      !---------------------------------------------------------------------
      ! convert from old data order for line, autocorrelations
      !
      !	NL       INTEGER     Number of channels
      !       NB       INTEGER     Number of baselines
      !       NA       INTEGER     Number of antennas
      !       CIN(2,NL,NB) COMPLEX Input pseudo-complex autocorrelation data
      !       ROUT(NL,NA)  REAL    Output autocorrelation data
      !---------------------------------------------------------------------
      integer :: nl                     !
      integer :: nb                     !
      complex :: cin(nl, 2, nb)         !
      integer :: na                     !
      real :: rout(nl, na)              !
    end subroutine amodel
  end interface
  !
  interface
    subroutine cmodec(nl,nb,cin,cout)
      !---------------------------------------------------------------------
      ! convert from old data order for continuum, correlations
      !
      !	NL       INTEGER     Number of channels
      !       NB       INTEGER     Number of baselines
      !       CIN(2,NL,NB)  COMPLEX Input complex correlation data
      !       COUT(NL,2,NB) COMPLEX Output correlation data
      !---------------------------------------------------------------------
      integer :: nl                     !
      integer :: nb                     !
      complex :: cin(2,nl,nb)           !
      complex :: cout(nl,2,nb)          !
    end subroutine cmodec
  end interface
  !
  interface
    function base_number(i,j)
      !---------------------------------------------------------------------
      !  Number of base line from antenna I to antenna J
      !---------------------------------------------------------------------
      integer :: base_number            !
      integer :: i                      !
      integer :: j                      !
    end function base_number
  end interface
  !
  interface
    subroutine base_to_n(c,n,error)
      !---------------------------------------------------------------------
      !  Number of base line from antenna C
      !---------------------------------------------------------------------
      character(len=2) :: c             !
      integer :: n                      !
      logical :: error                  !
    end subroutine base_to_n
  end interface
  !
  interface
    function base(i,j)
      !---------------------------------------------------------------------
      ! Returns the number of baseline I,J (not oriented)
      !---------------------------------------------------------------------
      integer :: base                   !
      integer :: i                      !
      integer :: j                      !
    end function base
  end interface
  !
  interface
    subroutine out0(type,x,y,error)
      !---------------------------------------------------------------------
      ! SAS	Internal routine
      !	Sortie terminal ou graphique d'un texte tabule
      !	'G',x,y : en graphique en x,y
      !	'T',0.,0. : ecriture normale sur le terminal
      !	'F',0.,0. : fichier imprime par le call out2
      !
      ! (2-Jan-1985)
      !---------------------------------------------------------------------
      character(len=*) :: type          !
      real*4 :: x                       !
      real*4 :: y                       !
      logical :: error                  !
    end subroutine out0
  end interface
  !
  interface
    subroutine clic_popup (line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Pop-up the box pointed out by the cursor, and
      !     then call the cursor again to get "local" user coordinates.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_popup
  end interface
  !
  interface
    subroutine sub_popup(ib)
      integer :: ib                     !
    end subroutine sub_popup
  end interface
  !
  interface
    subroutine planet_flux(name,date,ut,fnu,flux,fbeam,size,error)
      !---------------------------------------------------------------------
      !	FLUX Command :
      !		character*(*) name 	input
      !		integer date 	        input	CLASS date (DH_OBS)
      !		real*8  ut 	        input	UT time (DH_UTC)
      !		real*8 	FNU		input
      !		real*8	flux		output
      !		REAL*8	SIZE(3)		output
      !		logical error 		in/output
      !---------------------------------------------------------------------
      character(len=12) :: name         !
      integer :: date                   !
      real*8 :: ut                      !
      real*8 :: fnu                     !
      real*8 :: flux                    !
      real*8 :: fbeam                   !
      real*8 :: size(3)                 !
      logical :: error                  !
    end subroutine planet_flux
  end interface
  !
  interface
    subroutine apparent_flux (fnu,flux,size,type,uvm)
      !---------------------------------------------------------------------
      ! The Fourier Transform of the elliptical disk is an Elliptical Bessel
      ! function of same position angle (Caution: positive towards East).
      ! Using the similarity theorem, an appropriate scaling of the U,V vector
      ! gives the answer:
      !     U1 =  U*COS(PA) - V*SIN(PA)
      !     V1 =  U*SIN(PA) + V*COS(PA)
      !
      !     U2 = U1
      !     V2 = V1*MAJOR/MINOR
      !
      !     UV = SQRT(U2^2+V2^2)/C*FNU     ! in wavelengths
      !
      ! For type = -1 (Disk)
      !     visi = minor/major * Bessel(2*pi*UV,MAJOR)
      ! For type # 0 (Gaussian)
      !     visi = minor/major * Gauss (2*pi*UV,MAJOR)
      !---------------------------------------------------------------------
      real*8 :: fnu                     ! Observing frequency
      real*8 :: flux                    ! Zero spacing flux
      real*8 :: size(3)                 ! Major, Minor, PA in radians
      integer :: type                   ! Type of source (1 Disk, 2 Gauss, etc...)
      real*4 :: uvm(2)                  ! U,V in meters
    end subroutine apparent_flux
  end interface
  !
  interface
    function known_flux (name,le_flux)
      !---------------------------------------------------------------------
      ! CLIC
      !     Check if a source has a known flux and structure.
      !---------------------------------------------------------------------
      integer :: known_flux             !
      character(len=*) :: name          !
      real*8 :: le_flux                 !
    end function known_flux
  end interface
  !
  interface
    subroutine get_flux(fnu,isou,le_flux,size)
      real*8 :: fnu                     !
      integer :: isou                   !
      real*8 :: le_flux                 !
      real*8 :: size(3)                 !
    end subroutine get_flux
  end interface
  !
  interface
    subroutine set_fluxspl(fich,planet_model)
      character(len=*) :: fich          !
      integer :: planet_model           !
    end subroutine set_fluxspl
  end interface
  !
  interface
    subroutine rgen (error)
      use gildas_def
      logical :: error                  !
    end subroutine rgen
  end interface
  !
  interface
    subroutine wgen (error)
      logical :: error                  !
    end subroutine wgen
  end interface
  !
  interface
    subroutine udescr (error)
      logical :: error                  !
    end subroutine udescr
  end interface
  !
  interface
    subroutine rcom(com,ncom,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Read a comment section
      ! Arguments :
      !	COM	I*4 (*)	Array to receive the comment	Output
      !	NCOM	I	Number of words read		Output
      !			Size of COM			Input
      !	ERROR	L	Logical error flag		Output
      !---------------------------------------------------------------------
      integer*4 :: com(*)               !
      integer :: ncom                   !
      logical :: error                  !
    end subroutine rcom
  end interface
  !
  interface
    subroutine rdata (ndata,data,error)
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	read the data section
      ! Arguments :
      !	NDATA 	I	Number of data words to read	Input/Output
      !	DATA 	I*4 (*)	Array into which the data is transferred Output
      !	ERROR	L	Logical error flag			Output
      !---------------------------------------------------------------------
      integer :: ndata                  !
      integer :: data(*)                !
      logical :: error                  !
    end subroutine rdata
  end interface
  !
  interface
    subroutine wdata (ndata,data,copy_data,error)
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	write the data section
      !	NDATA 	I	Number of data words to write 	Input
      !	DATA 	I(*)    Array from which the data is transferred Input
      !	ERROR	L	Logical error flag		Output
      !       copy_data L     copy mode: Use do_write_data as command to
      !                       actually write the data  section        Input
      !       (otherwise the data is written only if R/W file)
      !---------------------------------------------------------------------
      integer :: ndata                  !
      integer :: data(*)                !
      logical :: copy_data              !
      logical :: error                  !
    end subroutine wdata
  end interface
  !
  interface
    subroutine rdata_ext (ndata,data,error)
      use gbl_format
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	read the data section in external file
      ! Arguments :
      !	NDATA 	I	Number of data words to read	Output
      !	DATA 	I*4 (*)	Array into which the data is transferred Output
      !       DCONVE  I       The conversion code needed for DATA.
      !	ERROR	L	Logical error flag			Output
      !---------------------------------------------------------------------
      integer :: ndata                  !
      integer :: data(*)                !
      logical :: error                  !
    end subroutine rdata_ext
  end interface
  !
  interface
    subroutine modify_datac(data)
      !---------------------------------------------------------------------
      ! Apply the modifying factors in the data modifier section,
      ! to the continuum data.
      !---------------------------------------------------------------------
      real :: data(*)                   !
    end subroutine modify_datac
  end interface
  !
  interface
    subroutine modify_datal(data)
      !---------------------------------------------------------------------
      ! Apply the modifying factors in the data modifier section,
      ! to the line data.
      !---------------------------------------------------------------------
      real :: data(*)                   !
    end subroutine modify_datal
  end interface
  !
  interface
    subroutine modify_dh(data)
      !---------------------------------------------------------------------
      ! Modify the data header according to the current modifier section.
      ! We only change the antenna and baseline flags.
      !---------------------------------------------------------------------
      integer :: data(*)                !
    end subroutine modify_dh
  end interface
  !
  interface
    subroutine rbuf(add,len,vec,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Read the vector vec in the current observation,
      ! Arguments :
      !	ADD 	I	starting address (words)	Input
      !	LEN 	I	length in words
      !	VEC	I*2 (*)	Vector to be transferred	Output
      !	ERROR	L	Logical error flag		Output
      !---------------------------------------------------------------------
      integer*4 :: add                  !
      integer*4 :: len                  !
      integer*4 :: vec(*)               !
      logical :: error                  !
    end subroutine rbuf
  end interface
  !
  interface
    subroutine wbuf(add,len,vec,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Write the vector VEC in the current observation,
      ! Arguments :
      !	ADD 	I	starting address (words)
      !	LEN 	I	length in words
      !	VEC	I*2 (*)	Vector to write
      !	ERROR	L	Logical error flag
      !---------------------------------------------------------------------
      integer*4 :: add                  !
      integer*4 :: len                  !
      integer*4 :: vec(1)               !
      logical :: error                  !
    end subroutine wbuf
  end interface
  !
  interface
    subroutine cbuf(error)
      !---------------------------------------------------------------------
      !	empties the buffer for the current observation
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine cbuf
  end interface
  !
  interface
    subroutine dldump(line,error)
      use gildas_def
      use gkernel_types
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine dldump
  end interface
  !
  interface
    subroutine sub_dldump(line,berkx,berky,berka,berkp,berki,ndatal,error)
      use gildas_def
      character(len=*) :: line          !
      integer :: ndatal                 !
      real :: berkx(ndatal)             !
      real :: berky(ndatal)             !
      real :: berka(ndatal)             !
      real :: berkp(ndatal)             !
      real :: berki(ndatal)             !
      logical :: error                  !
    end subroutine sub_dldump
  end interface
  !
  interface
    subroutine sdump(error)
      use gildas_def
      logical :: error                  !
    end subroutine sdump
  end interface
  !
  interface
    subroutine rdump(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC	Debug routine
      ! 	Types the contents of an obs.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine rdump
  end interface
  !
  interface
    subroutine dhdump(error)
      use gildas_def
      logical :: error                  !
    end subroutine dhdump
  end interface
  !
  interface
    subroutine dcdump(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine dcdump
  end interface
  !
  interface
    subroutine idump(error)
      use gildas_def
      logical :: error                  !
    end subroutine idump
  end interface
  !
  interface
    subroutine filedump(error)
      use gildas_def
      logical :: error                  !
    end subroutine filedump
  end interface
  !
  interface
    subroutine vdump(error)
      use gildas_def
      logical :: error                  !
    end subroutine vdump
  end interface
  !
  interface
    subroutine iidump(chain,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! A simple debugging routine, to help debug
      ! Call it when needed to make a pause in the program...
      !---------------------------------------------------------------------
      character(len=*) :: chain         !
      logical :: error                  !
    end subroutine iidump
  end interface
  !
  interface
    subroutine ini_base(line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY BASELINE  b1 x1 y1 x1 b2 x2 y2 z2 ...[/OFFSET]
      !                          [FITTED]
      ! Command MODIFY ANTENNA a1 x1 y1 x1 a2 x2 y2 z2 .. [/OFFSET]
      !                          [FITTED]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_base
  end interface
  !
  interface
    subroutine ini_delay(line,iopt,error,kpol)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY DELAY a1 d1 ... a2 x2 ... [/OFFSET]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
      integer :: kpol
    end subroutine ini_delay
  end interface
  !
  interface
    subroutine ini_position (line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	MODIFY POSITION Ra Dec
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_position
  end interface
  !
  interface
    subroutine ini_time (line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	MODIFY TIME Dut1
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_time
  end interface
  !
  interface
    subroutine ini_axes(line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY AXES_OFFSET  a1 x1 a2 x2 a3 x3 ...
      ! correct the phases of antenna ai for the offset xi (in mm) between
      ! the qazimuth and elevation axes.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_axes
  end interface
  !
  interface
    subroutine ini_focus(line,iopt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MODIFY FOCUS ...
      ! correct the phases of antenna ai for the offset in focus
      ! if not (correctly) applied in real time:
      !
      ! if  DATE_OBS is before 29-may-1996 :
      !        focus offset was actually logged
      ! if  DATE_OBS is before 29-nov-1996 :
      !        no correction applied to phase from focus offset
      ! if  DATE_OBS is before 02-apr-1998 and if HIGH LOCK:
      !       the focus correction was wrong sign
      ! ->
      ! MODIFY FOCUS:
      !   - between 29-may-1996 and 29-nov-1996: apply correction in header
      !   - between 29-nov-1996 and 02-apr-1998: change sign of corr.in head.
      ! MODIFY FOCUS  a1 x1 a2 x2 a3 x3 :
      !   - any date: modify phases for the focus change and log value in head.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_focus
  end interface
  !
  interface
    subroutine modify_ph(data,x,iunit)
      !---------------------------------------------------------------------
      ! Actually modify the data and keep track of the modify factors.
      ! X is a baseline dependent phase to frequency ratio (in radians/MHz)
      ! Add the possibility only one spectral unit
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      real :: data(*)                   !
      real :: x(mnbas)                  !
      integer :: iunit                  ! <= 0 means all, else unit to modify
    end subroutine modify_ph
  end interface
  !
  interface
    subroutine rotate(c,phi)
      !---------------------------------------------------------------------
      ! Rotate Phase of complex number C by PHI radians
      !---------------------------------------------------------------------
      complex :: c                      !
      real*8 :: phi                     !
    end subroutine rotate
  end interface
  !
  interface
    subroutine clic_kw(line,iopt,iarg,kw,nkey,voc,mvoc,present,error,verbose)
      character(len=*) :: line          !
      integer :: iopt                   !
      integer :: iarg                   !
      character(len=*) :: kw            !
      integer :: nkey                   !
      integer :: mvoc                   !
      character(len=*) :: voc(mvoc)     !
      logical :: present                !
      logical :: error, verbose         !
    end subroutine clic_kw
  end interface
  !
  interface
    subroutine read_data(arg,sort,init,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Fill in the display data buffers from current index.
      !	arg : 'A' all scans in current index
      !	      'N' next scan in current index
      !---------------------------------------------------------------------
      character(len=*) :: arg           !
      logical :: sort                   !
      logical :: init                   !
      logical :: error                  !
    end subroutine read_data
  end interface
  !
  interface
    subroutine init_data(n,x,v)
      !---------------------------------------------------------------------
      ! CLIC
      !     Initialise array X(N) with variable V
      !---------------------------------------------------------------------
      integer :: n                      !
      real :: x(n)                      !
      real :: v                         !
    end subroutine init_data
  end interface
  !
  interface
    subroutine brecord (qsb, qband, qbt, qntch,   &
        datac, datal, passc, passl,   &
        ibase,iband,ipol,nch,ich,jw1,jw2,ix,iy,vt,wt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get baseline-based parameters to be plotted as a function of
      !     a time-like variable
      !     QSB     INTEGER   Number of sidebands (2)
      !     QBAND   INTEGER   Number of temporal data
      !     QBT     INTEGER   Number of baseline+triangles
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     DATAC(QBAND,QSB,QBT)  COMPLEX  Temporal data
      !     DATAL(QNTCH,QSB,QBT)  COMPLEX  Spectral data
      !     PASSC(QBAND,QSB,QBT)  COMPLEX  PassBand of temporal data
      !     PASSL(QNTCH,QSB,QBT)  COMPLEX  PassBand of spectral data
      !     IBASE   INTEGER   Baseline number
      !     IBAND   INTEGER   Subband number
      !     ICH1    INTEGER   First subband
      !     ICH2    INTEGER   Last subband
      !     IX      INTEGER   X variable type
      !     IY      INTEGER   Y variable type
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !---------------------------------------------------------------------
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbt                    !
      integer :: qntch                  !
      complex :: datac(qband,qsb,qbt)   !
      complex :: datal(qntch,qsb,qbt)   !
      complex :: passc(qband,qsb,qbt)   !
      complex :: passl(qntch,qsb,qbt)   !
      integer :: ibase                  !
      integer :: iband                  !
      integer :: ipol                   ! Polarization
      integer :: nch                    !
      integer :: ich(nch)               !
      integer :: jw1                    !
      integer :: jw2                    !
      integer :: ix                     !
      integer :: iy                     !
      real*4 :: vt(4,3)                 !
      real*4 :: wt(4,3)                 !
      logical :: error                  !
    end subroutine brecord
  end interface
  !
  interface
    subroutine arecord (qsb, qband, qbas, qntch,   &
        datac, datal, passc, passl,   &
        iant, iband, ipol, nch, ich, jw1, jw2, ix, iy,   &
        vt, wt, phcorr,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get antenna-based parameters to be plotted from a record,
      !     as a function of a time-like variable.
      !     QSB     INTEGER   Number of sidebands (2)
      !     QBAND   INTEGER   Number of temporal data
      !     QBAS    INTEGER   Number of baselines
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     DATAC(QBAND,QSB,QBT)  COMPLEX  Temporal data
      !     DATAL(QNTCH,QSB,QBT)  COMPLEX  Spectral data
      !     PASSC(QBAND,QSB,QBT)  COMPLEX  PassBand of temporal data
      !     PASSL(QNTCH,QSB,QBT)  COMPLEX  PassBand of spectral data
      !     IANT    INTEGER   Antenna number
      !     IBAND   INTEGER   Subband number
      !     ICH1    INTEGER   First subband
      !     ICH2    INTEGER   Last subband
      !     IX      INTEGER   X variable type
      !     IY      INTEGER   Y variable type
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !     WPT     REAL      Positive weight
      !     WT      REAL      Total weight
      !     XT      REAL      X value
      !     YT      REAL      Y value
      !     TT      REAL      Signed integration time
      !     ZT      COMPLEX   Complex output
      !     PHCORR  LOGICAL   Correct amplitude for rms phase ?
      ! Call Tree
      !	SOLVE_POINT	...	FILL_VISI_POSI
      !	SOLVE_FOCUS	...	FILL_VISI_POSI
      !	...		READ_DATA
      !	SOLVE_FLUX
      !---------------------------------------------------------------------
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbas                   !
      integer :: qntch                  !
      complex :: datac(qband,qsb,qbas)  !
      complex :: datal(qntch,qsb,qbas)  !
      complex :: passc(qband,qsb,qbas)  !
      complex :: passl(qntch,qsb,qbas)  !
      integer :: iant                   !
      integer :: iband                  !
      integer :: ipol                   ! Polarization
      integer :: nch                    !
      integer :: ich(nch)               !
      integer :: jw1                    !
      integer :: jw2                    !
      integer :: ix                     !
      integer :: iy                     !
      real*4 :: vt(4,3)                 !
      real*4 :: wt(4,3)                 !
      logical :: phcorr                 !
      logical :: error                  !
    end subroutine arecord
  end interface
  !
  interface
    subroutine check_cal(error)
      logical :: error                  !
    end subroutine check_cal
  end interface
  !
  interface
    subroutine spectral_dump(kr,iant,ibase)
      !---------------------------------------------------------------------
      ! CLIC
      !     Get the dump of spectral data. It depends on phase correction.
      !     iant.gt.0  -> antenna mode
      !     ibase.gt.0 -> baseline mode
      !---------------------------------------------------------------------
      integer :: kr                     !
      integer :: iant                   !
      integer :: ibase                  !
    end subroutine spectral_dump
  end interface
  !
  interface
    subroutine aver(ib,vt,wt,it,ut,rt,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Put average value of a visibility-like information into display
      !     buffers
      !     IB    INTEGER    Box number
      !     UT    INTEGER    Observation number
      !     IT    INTEGER    Identification number (source number)
      !     RT    INTEGER    Record number
      !     ZT    COMPLEX    Visibility
      !     XT    REAL       X value
      !     YT    REAL       Y value
      !     WT    REAL       Weight
      !     T     REAL       Integration time
      !     WPT   REAL       Positive weight
      !---------------------------------------------------------------------
      integer :: ib                     !
      real*4 :: vt(4,3)                 !
      real*4 :: wt(4,3)                 !
      integer :: it                     !
      integer :: ut                     !
      integer :: rt                     !
      logical :: error                  !
    end subroutine aver
  end interface
  !
  interface
    subroutine sub_aver(ib,vt,wt,it,ut,rt,   &
        md, mb, x_data,y_data,z_data,w_data,   &
        i_data,u_data,r_data,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     Put average value of a visibility-like information into display
      !     buffers
      !     IB    INTEGER    Box number
      !     UT    INTEGER    Observation number
      !     IT    INTEGER    Identification number (source number)
      !     RT    INTEGER    Record number
      !     ZT    COMPLEX    Visibility
      !     XT    REAL       X value
      !     YT    REAL       Y value
      !     WT    REAL       Weight
      !     T     REAL       Integration time
      !     WPT   REAL       Positive weight
      !     MD      INTEGER        Number of points per box
      !     MB      INTEGER        Number of boxes
      !     X_DATA  REAL(MB,MD)    X coordinates
      !     Y_DATA  REAL(MB,MD)    Y values
      !     W_DATA  REAL(MB,MD)    Weight values (for error bars)
      !     I_DATA  REAL(MB,MD)    Identification number
      !     U_DATA  REAL(MB,MD)    Observation number of point
      !     Z_DATA  COMPLEX(MB,MD) Visibility of point used to compute X and Y
      !     A_DATA  REAL(MB,MD)    "Amplitude" of point
      !     ARG     C*(*)
      !     ERROR   L              Logical error flag
      !---------------------------------------------------------------------
      integer :: ib                     !
      real*4 :: vt(4,3)                 !
      real*4 :: wt(4,3)                 !
      integer :: it                     !
      integer :: ut                     !
      integer :: rt                     !
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      complex :: z_data(md,mb)          !
      real :: w_data(md,mb)             !
      integer :: i_data(md,mb)          !
      integer :: u_data(md,mb)          !
      integer :: r_data(md,mb)          !
      logical :: error                  !
    end subroutine sub_aver
  end interface
  !
  interface
    subroutine resetvar(error)
      use gildas_def
      logical :: error                  !
    end subroutine resetvar
  end interface
  !
  interface
    subroutine sub_resetvar(md,mb,x_data,y_data,w_data,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Redefine SIC X_DATA,Y_DATA and W_DATA variables when number of
      !     boxes/points as changed. Get their content from the X_DATA
      !     Y_DATA and W_DATA arrays.
      !
      !     MD   INTEGER    Number of points per box
      !     MB   INTEGER    Number of boxes
      !     X_DATA  REAL    X_DATA array
      !     Y_DATA  REAL    Y_DATA array
      !     W_DATA  REAL    W_DATA array
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      logical :: error                  !
    end subroutine sub_resetvar
  end interface
  !
  interface
    subroutine copyvar(in,nout1,nin1,n2,out)
      !---------------------------------------------------------------------
      ! CLIC
      !     Utility routine: copy a 2-d array in another of different
      !     dimensions
      !---------------------------------------------------------------------
      integer :: nin1                   !
      integer :: n2                     !
      real*4 :: in(nin1,n2)             !
      integer :: nout1                  !
      real*4 :: out(nout1,n2)           !
    end subroutine copyvar
  end interface
  !
  interface
    subroutine zrecord (qsb, qband, qbt, qntch,   &
        datac, datal, passc, passl, ibase, iband,   &
        nch, ich, jw1, jw2,  z, fw, fwp,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get a visibility like information from a record
      !
      !     QSB     INTEGER   Number of sidebands (2)
      !     QBAND   INTEGER   Number of temporal data
      !     QBT     INTEGER   Number of baseline+triangles
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     DATAC(QBAND,QSB,QBT)  COMPLEX  Temporal data
      !     DATAL(QNTCH,QSB,QBT)  COMPLEX  Spectral data
      !     PASSC(QBAND,QSB,QBT)  COMPLEX  PassBand of temporal data
      !     PASSL(QNTCH,QSB,QBT)  COMPLEX  PassBand of spectral data
      !     IBASE   INTEGER   Baseline number
      !     IBAND   INTEGER   Subband number
      !     ICH1    INTEGER   First subband
      !     ICH2    INTEGER   Last subband
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !     JW3     INTEGER   NUMBER of GIBSS Channels
      !     WP      REAL      Positive weight
      !     FW      REAL      (full) Total weight
      !     Z       COMPLEX   Complex output
      ! Call Tree
      !	...		...		ARECORD
      !	...		READ_DATA	BRECORD
      !---------------------------------------------------------------------
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbt                    !
      integer :: qntch                  !
      complex :: datac(qband,qsb,qbt)   !
      complex :: datal(qntch,qsb,qbt)   !
      complex :: passc(qband,qsb,qbt)   !
      complex :: passl(qntch,qsb,qbt)   !
      integer :: ibase                  !
      integer :: iband                  !
      integer :: nch                    !
      integer :: ich(nch)               !
      integer :: jw1                    !
      integer :: jw2                    !
      complex :: z                      !
      real*4 :: fw                      !
      real*4 :: fwp                     !
      logical :: error                  !
    end subroutine zrecord
  end interface
  !
  interface
    subroutine irecord (qband, qant, qntch,   &
        datac, datal, iant, iband, nch, ich, jw1, jw2, z, fw, wp)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get an auto-correlation like information from a record
      !
      !     QANT    INTEGER   Number of antennas
      !     QBAND   INTEGER   Number of temporal data
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     DATAC(QBAND,QANT) REAL     Temporal data
      !     DATAL(QNTCH,QANT) REAL     Spectral data
      !     IANT    INTEGER   Antenna number
      !     IBAND   INTEGER   Subband number
      !     ICH1    INTEGER   First subband
      !     ICH2    INTEGER   Last subband
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !     WP      REAL      Positive weight
      !     FW      REAL      (full) Total weight
      !     Z       COMPLEX   Complex output (yes, complex for compatibility)
      ! Call Tree
      !	...		...		ARECORD
      !	...		READ_DATA	BRECORD
      !---------------------------------------------------------------------
      integer :: qband                  !
      integer :: qant                   !
      integer :: qntch                  !
      real :: datac(qband,qant)         !
      real :: datal(qntch,qant)         !
      integer :: iant                   !
      integer :: iband                  !
      integer :: nch                    !
      integer :: ich(nch)               !
      integer :: jw1                    !
      integer :: jw2                    !
      complex :: z                      !
      real*4 :: fw                      !
      real*4 :: wp                      !
    end subroutine irecord
  end interface
  !
  interface
    subroutine value(ix,z,ibase,iband,kpol,x,w,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Returns value of the specified variable
      !     IX    INTEGER   Type of desired value
      !     IBASE INTEGER   Baseline/Antenna number
      !     IBAND INTEGER   Sideband number
      !     Z     COMPLEX   Complex visibility
      !     X     REAL      Returned value
      !     W     REAL      Returned weight (0 or 1)
      !---------------------------------------------------------------------
      integer :: ix                     !
      complex :: z                      !
      integer :: ibase                  !
      integer :: iband                  !
      integer :: kpol                   ! Polarization
      real*4 :: x                       !
      real*4 :: w                       !
      logical :: error                  !
    end subroutine value
  end interface
  !
  interface
    subroutine bin(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Bin data in plot arrays
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine bin
  end interface
  !
  interface
    subroutine get_nbins(x,nx,bs,nb)
      integer :: nx                     !
      real :: x(nx)                     !
      real :: bs                        !
      integer :: nb                     !
    end subroutine get_nbins
  end interface
  !
  interface
    subroutine binning(iy,ja,x,y,z,w,nin,nh,nout,bval,eval,degrees,   &
        hmin,hstep,work)
      !---------------------------------------------------------------------
      ! CLIC	Do data binning in X for data values in Y
      ! Arguments
      ! Input/Output:
      !       IY      Integer Type of value
      !	X,Y,W	R*4(*)	Input arrays (size>max(np,nh))
      !	Z	C*8(*)	...
      !				Output data
      ! Input:
      !	NIN 	I	Number of input points
      !	NH	I	Number of bins (<NIN)
      !	HMIN	R*4	Minimum of first bin
      !	HSTEP	R*4	Bin size
      !	BVAL	R*4	Blanking values
      !	EVAL	R*4	Tolerance on blanking
      ! 	WORK	R*4	Space of size 4*NH
      !---------------------------------------------------------------------
      integer :: iy                     !
      integer :: ja                     !
      real :: x(*)                      !
      real :: y(*)                      !
      complex :: z(*)                   !
      real :: w(*)                      !
      integer :: nin                    !
      integer :: nh                     !
      integer :: nout                   !
      real :: bval                      !
      real :: eval                      !
      logical :: degrees                !
      real :: hmin                      !
      real :: hstep                     !
      real :: work(4,nin)               !
    end subroutine binning
  end interface
  !
  interface
    subroutine jlimits(nch,jw1,jw2,jw3,jok)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC: compute actual channel limits of subband being displayed
      !---------------------------------------------------------------------
      ! number of channels, limits from SET SUBB /WINDOW, SET GIBBS:
      integer :: nch                    !
      integer :: jw1                    !
      integer :: jw2                    !
      integer :: jw3                    !
      integer :: jok(nch)               !
    end subroutine jlimits
  end interface
  !
  interface
    subroutine set_corr(error)
      !---------------------------------------------------------------------
      !     compute the phase corrections factors in radians/unit of 
      !     tot pow monitor or WVR
      !     these are used in the scaling routine
      !---------------------------------------------------------------------
      logical :: error                  
    end subroutine set_corr
  end interface
  !
  interface
    function h_offset(idump)
      !---------------------------------------------------------------------
      ! gives the offset of the data header in the data section in words ,
      ! for dump idump (1 to r_ndump+2)
      !---------------------------------------------------------------------
      integer :: h_offset               !
      integer :: idump                  !
    end function h_offset
  end interface
  !
  interface
    function c_offset(idump)
      !---------------------------------------------------------------------
      ! gives the offset of the continuum data in the data section in words ,
      ! for dump idump (1 to r_ndump+2)
      !---------------------------------------------------------------------
      integer :: c_offset               !
      integer :: idump                  !
    end function c_offset
  end interface
  !
  interface
    function l_offset(idump)
      !---------------------------------------------------------------------
      ! gives the offset of the data header in the data section in words ,
      ! for dump idump (1 to r_ndump+2)
      !---------------------------------------------------------------------
      integer :: l_offset               !
      integer :: idump                  !
    end function l_offset
  end interface
  !
  interface
    real function my_total(dopolar,ia)
      !---------------------------------------------------------------------
      ! average total powers according to dopolar (1,2,3=average)
      !---------------------------------------------------------------------
      integer :: dopolar                !
      integer :: ia                     !
    end function my_total
  end interface
  !
  interface
    real function my_total2(jinput,ia,error)
      !---------------------------------------------------------------------
      ! average total powers according to jinput (1,2,0=average)
      !---------------------------------------------------------------------
      integer :: jinput                 !
      integer :: ia                     !
      logical :: error                  !
    end function my_total2
  end interface
  !
  interface
    subroutine gvalue (qsb, qband, qant, qntch,   &
        gainc, gainl, wgainc, wgainl,   &
        iant, iband, isub, jw1, jw2, jw3, w, zz, aa, ww, ny)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get an antenna gain-like value
      !     QSB     INTEGER   Number of sidebands (2)
      !     QBAND   INTEGER   Number of temporal data
      !     QANT    INTEGER   Number of antennas
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     GAINC(QBAND,QSB,QANT)   COMPLEX  Antenna gains (temporal data)
      !     GAINL(QNTCH,QSB,QBAND)  COMPLEX  Antenna gains (spectral data)
      !     WGAINC(QBAND,QSB,QANT)  REAL  Weight of antenna gain
      !     WGAINL(QNTCH,QSB,QBAND) REAL  Weight of antenna gain
      !     IANT    INTEGER   Antenna number
      !     IBAND   INTEGER   Subband number
      !     ISUB    INTEGER   Sideband number
      !     NY      INTEGER   Number of Y values (output)
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !     W       REAL      Signed integration time
      !     AA(*)   REAL      Positive weights
      !     WW(*)   REAL      Total weights
      !     ZZ(*)   COMPLEX   Complex output
      ! Call Tree
      !	...	SUB_READ_SPEC
      !---------------------------------------------------------------------
      integer :: qsb                     !
      integer :: qband                   !
      integer :: qant                    !
      integer :: qntch                   !
      complex :: gainc(qband,qsb,qant)   !
      complex :: gainl(qntch,qsb,qband)  !
      real :: wgainc(qband,qsb,qant)     !
      real :: wgainl(qntch,qsb,qband)    !
      integer :: iant                    !
      integer :: iband                   !
      integer :: isub                    !
      integer :: jw1                     !
      integer :: jw2                     !
      integer :: jw3                     !
      real*4 :: w                        !
      complex :: zz(*)                   !
      real*4 :: aa(*)                    !
      real*4 :: ww(*)                    !
      integer :: ny                      !
    end subroutine gvalue
  end interface
  !
  interface
    subroutine mixband(iband,z,w,av,zm,wm)
      !---------------------------------------------------------------------
      !     IBAND    INTEGER   Sideband code (u/l/aver/diff/ratio)
      !     Z(2)     COMPLEX   Complex antenna gain for both sidebands
      !     W(2)     REAL      Weight
      !     AV(2)    COMPLEX   Complex sideband ratio
      !     ZM       COMPLEX   Output complex gain
      !     WM       REAL      Output weight
      !---------------------------------------------------------------------
      integer :: iband                  !
      complex :: z(2)                   !
      real :: w(2)                      !
      complex :: av(2)                  !
      complex :: zm                     !
      real :: wm                        !
    end subroutine mixband
  end interface
  !
  interface
    subroutine gain_cont (qsb,qband,qbas,qant,   &
        data_in, passc,gainc,wgainc,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Compute the antenna gains from temporal data
      ! Arguments
      !     QSB     INTEGER   Number of sidebands (2)
      !     QBAND   INTEGER   Number of temporal data
      !     QANT    INTEGER   Number of antennas
      !     QBAS    INTEGER   Number of baselines
      !     DATC(QBAND,QSB,QBAS)   COMPLEX  Temporal data
      !     PASSC(QBAND,QSB,QBAS)  COMPLEX  PassBand of temporal data
      !     GAINC(QBAND,QSB,QANT)  COMPLEX  Antenna gains (temporal data)
      !     WGAINC(QBAND,QSB,QANT) REAL  Weight of antenna gain
      ! Call Tree
      !	...	SUB_READ_SPEC
      !---------------------------------------------------------------------
      integer :: qsb                           !
      integer :: qband                         !
      integer :: qbas                          !
      integer :: qant                          !
      integer(kind=address_length) :: data_in  !
      complex :: passc(qband,qsb,qbas)         !
      complex :: gainc(qband,qsb,qant)         !
      real :: wgainc(qband,qsb,qant)           !
      logical :: error                         !
    end subroutine gain_cont
  end interface
  !
  interface
    subroutine gain_line (qntch,qsb,qbas,qant,   &
        data_in,passl,gainl,wgainl,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Compute the antenna gains from temporal data
      ! Arguments
      !     QSB     INTEGER   Number of sidebands (2)
      !     QNTCH   INTEGER   Number of spectral channels
      !     QANT    INTEGER   Number of antennas
      !     QBAS    INTEGER   Number of baselines
      !     DATL(QNTCH,QSB,QBAS)   COMPLEX  Spectral data
      !     PASSL(QNTCH,QSB,QBAS)  COMPLEX  PassBand of spectral data
      !     GAINL(QNTCH,QSB,QANT)  COMPLEX  Antenna gains (spectral data)
      !     WGAINL(QNTCH,QSB,QANT) REAL     Weight of antenna gain
      ! Call Tree
      !	...	SUB_READ_SPEC
      !---------------------------------------------------------------------
      integer :: qntch                         !
      integer :: qsb                           !
      integer :: qbas                          !
      integer :: qant                          !
      integer(kind=address_length) :: data_in  !
      complex :: passl(qntch,qsb,qbas)         !
      complex :: gainl(qntch,qsb,qant)         !
      real :: wgainl(qntch,qsb,qant)           !
      logical :: error                         !
    end subroutine gain_line
  end interface
  !
  interface
    subroutine antgain(z,w,zant,want)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Derive antenna "gains" from baseline visibilities
      ! Arguments:
      !     Z(MNBAS)    COMPLEX   Visibility
      !     W(MNBAS)    REAL      Weight
      !     ZANT(MNANT) COMPLEX   Complex antenna gain
      !     WANT(MNANT) REAL      Weight
      ! Call Tree
      !	...	GAIN_CONT
      !	...	GAIN_LINE
      !	...	ARECORD
      !	...	CONT_AVERAGE
      !	...	SUB_SOLVE_PASS
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      complex :: z(mnbas)               !
      real*4 :: w(mnbas)                !
      complex :: zant(mnant)            !
      real*4 :: want(mnant)             !
    end subroutine antgain
  end interface
  !
  interface
    subroutine new_antgain (z,w,zant,want)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Derive antenna "gains" from baseline visibilities
      ! Arguments:
      !     Z(MNBAS)    COMPLEX   Visibility
      !     W(MNBAS)    REAL      Weight
      !     ZANT(MNANT) COMPLEX   Complex antenna gain
      !     WANT(MNANT) REAL      Weight
      ! Call Tree
      !	...	GAIN_CONT
      !	...	GAIN_LINE
      !	...	ARECORD
      !	...	CONT_AVERAGE
      !	...	SUB_SOLVE_PASS
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      complex :: z(mnbas)               !
      real*4 :: w(mnbas)                !
      complex :: zant(mnant)            !
      real*4 :: want(mnant)             !
    end subroutine new_antgain
  end interface
  !
  interface
    subroutine old_antgain (z,w,zant,want)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Derive antenna "gains" from baseline visibilities
      ! Arguments:
      !     Z(MNBAS)    COMPLEX   Visibility
      !     W(MNBAS)    REAL      Weight
      !     ZANT(MNANT) COMPLEX   Complex antenna gain
      !     WANT(MNANT) REAL      Weight
      ! Call Tree
      !	...	GAIN_CONT
      !	...	GAIN_LINE
      !	...	ARECORD
      !	...	CONT_AVERAGE
      !	...	SUB_SOLVE_PASS
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      complex :: z(mnbas)               !
      real*4 :: w(mnbas)                !
      complex :: zant(mnant)            !
      real*4 :: want(mnant)             !
    end subroutine old_antgain
  end interface
  !
  interface
    subroutine gain_ant(iy, nbas, iant, jant, iref,   &
        nant, y, w, wk2, wk3, ss, c, wc, error)
      !---------------------------------------------------------------------
      !     gain_ant computes a weighted least-squares approximation
      !     to the antenna amplitudes or phases
      !     for an arbitrary set of baseline data points.
      ! parameters:
      !     iy             I   Input    1 for log(amplitude), 2 for phase
      !     nbas           I   Input    the number of baselines
      !     iant(nbas)     I   Input    start antenna for each baseline
      !     jant(nbas)     I   Input    end  antenna for each baseline
      !     iref           i   Input    Reference antenna for phases
      !     nant           I   Input    the number of antennas
      !     y(m,nbas)      R8  Input    the data values
      !     w(m,nbas)      R8  Input    weights
      !     wk2(nant**2)   R8  Output   work space
      !     wk3(nant)      R8  Output   work space
      !     ss(nbas)       R8  Output   rms of fit for each baseline
      !     c(nant)        R8  Output   the gains
      !     wc(nant)        R8  Output   the output weights
      !---------------------------------------------------------------------
      integer :: iy                     !
      integer :: nbas                   !
      integer :: iant(nbas)             !
      integer :: jant(nbas)             !
      integer :: iref                   !
      integer :: nant                   !
      real*8 :: y(nbas)                 !
      real*8 :: w(nbas)                 !
      real*8 :: wk2(nant,nant)          !
      real*8 :: wk3(nant)               !
      real*8 :: ss(nbas)                !
      real*8 :: c(nant)                 !
      real*8 :: wc(nant)                !
      logical :: error                  !
    end subroutine gain_ant
  end interface
  !
  interface
    function cmpl2(amp, pha)
      !---------------------------------------------------------------------
      ! Returns a complex of given amplitude and phase
      ! Test blanking value BLANK4
      !---------------------------------------------------------------------
      complex :: cmpl2                  !
      real :: amp                       !
      real :: pha                       !
    end function cmpl2
  end interface
  !
  interface
    subroutine retrieve_datac(qband,qsb,qbas,datac,ic,isb,ib,z)
      integer :: qband                  !
      integer :: qsb                    !
      integer :: qbas                   !
      complex :: datac(qband,qsb,qbas)  !
      integer :: ic                     !
      integer :: isb                    !
      integer :: ib                     !
      complex :: z                      !
    end subroutine retrieve_datac
  end interface
  !
  interface
    subroutine retrieve_datal(qntch,qsb,qbas,datal,k,isb,ib,z)
      integer :: qntch                  !
      integer :: qsb                    !
      integer :: qbas                   !
      complex :: datal(qntch,qsb,qbas)  !
      integer :: k                      !
      integer :: isb                    !
      integer :: ib                     !
      complex :: z                      !
    end subroutine retrieve_datal
  end interface
  !
  interface
    subroutine read_spec(arg,init,error)
      use gildas_def
      character(len=*) :: arg           !
      logical :: init                   !
      logical :: error                  !
    end subroutine read_spec
  end interface
  !
  interface
    subroutine sub_read_spec(md,mb,x_data,y_data,w_data,   &
        i_data,u_data,z_data,a_data,arg,init,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Fill in the display data buffers from current index.
      !	arg : 'A' all scans in current index are averaged
      !	      'N' next scan in current index is averaged
      !	data is the spectrum og selected subbands, bands, baselines.
      !  MD      INTEGER        Number of points per box
      !  MB      INTEGER        Number of boxes
      !  X_DATA  REAL(MB,MD)    X coordinates
      !  Y_DATA  REAL(MB,MD)    Y values
      !  W_DATA  REAL(MB,MD)    Weight values (for error bars)
      !  I_DATA  REAL(MB,MD)    Identification number
      !  U_DATA  REAL(MB,MD)    UT time of point
      !  Z_DATA  COMPLEX(MB,MD) Visibility of point used to compute X and Y
      !  A_DATA  REAL(MB,MD)    "Amplitude" of point
      !  ARG     C*(*)
      !  INIT    L              Initialize the plot buffers.
      !  ERROR   L              Logical error flag
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: i_data(md,mb)          !
      integer :: u_data(md,mb)          !
      complex :: z_data(md,mb)          !
      real :: a_data(md,mb)             !
      character(len=*) :: arg           !
      logical :: init                   !
      logical :: error                  !
    end subroutine sub_read_spec
  end interface
  !
  interface
    subroutine yvalue(iy,zz,aa,ww,nn,yy)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get spectral values
      !     IY      INTEGER    Type of Y value desired
      !     ZZ(NN)  COMPLEX    Visibilities of the NN points
      !     AA(NN)  REAL       Amplitude (??) of the NN points
      !     WW(NN)  REAL       Weight of the NN points
      !     NN      INTEGER    Number of points
      !     YY(NN)  REAL       Returned Y values
      !---------------------------------------------------------------------
      integer :: iy                     !
      integer :: nn                     !
      complex :: zz(nn)                 !
      real :: aa(*)                     !
      real :: ww(nn)                    !
      real :: yy(nn)                    !
    end subroutine yvalue
  end interface
  !
  interface
    function faz(z)
      !---------------------------------------------------------------------
      ! 	Compute the phase of a complex number Z
      !---------------------------------------------------------------------
      real :: faz                       !
      complex :: z                      !
    end function faz
  end interface
  !
  interface
    function expim(p)
      !---------------------------------------------------------------------
      !     Returns the complex number of phase P and modulus 1
      !---------------------------------------------------------------------
      complex :: expim                  !
      real :: p                         !
    end function expim
  end interface
  !
  interface
    subroutine xvalue(ix,ibase,iband,isub,x,nx,four)
      !---------------------------------------------------------------------
      ! CLIC
      !	Get spectral abscissa of subband ISUB of continuum or line
      !     IX     INTEGER    Type of X value
      !     IBASE  INTEGER    Baseline number
      !     IBAND  INTEGER    Sideband (U or L)
      !     ISUB   INTEGER    Subbband of correlator
      !     X      REAL(*)    Returned X values
      !     NX     INTEGER    Number of X values
      !     FOUR   LOGICAL    Indicate "lag data"
      !---------------------------------------------------------------------
      integer :: ix                     !
      integer :: ibase                  !
      integer :: iband                  !
      integer :: isub                   !
      real*4 :: x(*)                    !
      integer :: nx                     !
      logical :: four                   !
    end subroutine xvalue
  end interface
  !
  interface
    subroutine zvalue (qsb, qband, qbt, qntch,   &
        datac, datal, passc, passl,   &
        ibase,iband,isub, jw1, jw2, jw3, w,zz,aa,ww,ny,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     QSB     INTEGER   Number of sidebands (2)
      !     QBAND   INTEGER   Number of temporal data
      !     QBT     INTEGER   Number of baseline+triangles
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     DATAC(QBAND,QSB,QBT)  COMPLEX  Temporal data
      !     DATAL(QNTCH,QSB,QBT)  COMPLEX  Spectral data
      !     PASSC(QBAND,QSB,QBT)  COMPLEX  PassBand of temporal data
      !     PASSL(QNTCH,QSB,QBT)  COMPLEX  PassBand of spectral data
      !     IBASE   INTEGER   Baseline number
      !     IBAND   INTEGER   Subband number
      !     ISUB    INTEGER   Sideband number
      !     NY      INTEGER   Number of Y values (output)
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !     W       REAL      Signed integration time
      !     AA(*)   REAL      Positive weights
      !     WW(*)   REAL      Total weights
      !     ZZ(*)   COMPLEX   Complex output
      ! Call Tree
      !	...	SUB_READ_SPEC	AVALUE
      !	...	SUB_READ_SPEC
      !---------------------------------------------------------------------
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbt                    !
      integer :: qntch                  !
      complex :: datac(qband,qsb,qbt)   !
      complex :: datal(qntch,qsb,qbt)   !
      complex :: passc(qband,qsb,qbt)   !
      complex :: passl(qntch,qsb,qbt)   !
      integer :: ibase                  !
      integer :: iband                  !
      integer :: isub                   !
      integer :: jw1                    !
      integer :: jw2                    !
      integer :: jw3                    !
      real*4 :: w                       !
      complex :: zz(*)                  !
      real*4 :: aa(*)                   !
      real*4 :: ww(*)                   !
      integer :: ny                     !
      logical :: error                  !
    end subroutine zvalue
  end interface
  !
  interface
    subroutine avalue (qband, qant, qntch, datac, datal,   &
        iant, iband, isub, jw1, jw2, jw3, w, zz, aa, ww, ny)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Get antenna based values for display
      ! Input:
      !     QBAND   INTEGER   Number of temporal data
      !     QANT    INTEGER   Number of antennas
      !     QNTCH   INTEGER   Number of channels (in total per baseline)
      !     DATAC(QBAND,QANT)   REAL     Temporal data
      !     DATAL(QNTCH,QANT)   REAL     Spectral data
      !     IANT    INTEGER   Antenna number
      !     IBAND   INTEGER   Subband number
      !     ISUB    INTEGER   Sideband number
      !     JW1,JW2 INTEGER   Lowest and Highest channel in window subband
      !     W       REAL      Signed integration time
      ! Output:
      !     ZZ(*)   COMPLEX   Complex output
      !     AA(*)   REAL      Positive weights
      !     WW(*)   REAL      Total weights
      !     NY      INTEGER   Number of Y values (output)
      ! Call Tree
      !	...	SUB_READ_SPEC	AVALUE
      !	...	SUB_READ_SPEC
      !---------------------------------------------------------------------
      integer :: qband                  !
      integer :: qant                   !
      integer :: qntch                  !
      real :: datac(qband,qant)         !
      real :: datal(qntch,qant)         !
      integer :: iant                   !
      integer :: iband                  !
      integer :: isub                   !
      integer :: jw1                    !
      integer :: jw2                    !
      integer :: jw3                    !
      real*4 :: w                       !
      complex :: zz(*)                  !
      real*4 :: aa(*)                   !
      real*4 :: ww(*)                   !
      integer :: ny                     !
    end subroutine avalue
  end interface
  !
  interface
    subroutine zswap (qntch, qsb, qbas, wk1, wk2, wk4, datal)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLC
      !     Swap Berkeley correlator unfouriered data into place
      !     (negative delay channels in LSB, Positive in USB)
      !     QNTCH    INTEGER   Number of channels
      !     QSB      INTEGER   Number of sidebands
      !     QBAS     INTEGER   Number of baselines
      !     DATAL(QNTCH,QSB,QBAS)  COMPLEX   Berkeley correlator data
      !     WK1(QNTCH,2,1,QBAS)    COMPLEX
      !     WK2(QNTCH/2,2,2,QBAS)  COMPLEX   Work spaces
      !     WK4(QNTCH/4,2,4,QBAS)  COMPLEX
      ! Call Tree
      !	...	SUB_READ_SPEC
      !---------------------------------------------------------------------
      integer :: qntch                  !
      integer :: qsb                    !
      integer :: qbas                   !
      complex :: wk1(qntch,2,1,qbas)    !
      complex :: wk2(qntch/2,2,2,qbas)  !
      complex :: wk4(qntch/4,2,4,qbas)  !
      complex :: datal(qntch,qsb,qbas)  !
    end subroutine zswap
  end interface
  !
  interface
    subroutine cont_average (qband,qbas,data_in,zero,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !      Get the average of continuum channels in each subband
      !      This is used for the scaling of further bandpass calibrations
      !      These averages are stored in ZSBA(isb,ib) (weights in WSBA(isb,ib))
      !      (negative ib for antenna based quantities).
      ! Arguments
      !      INTEGER QBAND	! Number of bands
      !      INTEGER QBAS	! Number of baselines
      !      COMPLEX CONT(QBAND,2,QBAS)	! Usb/Lsb, Band, Base
      !      logical zero     ! initialize average
      !---------------------------------------------------------------------
      integer :: qband                         ! Number of bands
      integer :: qbas                          ! Number of baselines
      integer(kind=address_length) :: data_in  !
      logical :: zero                          !
      logical :: error                         !
    end subroutine cont_average
  end interface
  !
  interface
    subroutine get_cont_average (av,ibase)
      !---------------------------------------------------------------------
      ! Get the continuum average phases used to average sidebands.
      !---------------------------------------------------------------------
      complex :: av(2)                  !
      integer :: ibase                  !
    end subroutine get_cont_average
  end interface
  !
  interface
    real function tsys_b(isub,iband,ibase)
      !---------------------------------------------------------------------
      !     Compute TSYS**2 for a given subband, band, baseline, using the
      !     polarization information
      !---------------------------------------------------------------------
      integer :: isub                   !
      integer :: iband                  !
      integer :: ibase                  !
    end function tsys_b
  end interface
  !
  interface
    real function tsys_a(isub,iband,iant)
      !---------------------------------------------------------------------
      !     Compute TSYS**2 for a given subband, band, antenna, using the
      !     polarization information
      !---------------------------------------------------------------------
      integer :: isub                   !
      integer :: iband                  !
      integer :: iant                   !
    end function tsys_a
  end interface
  !
  interface
    subroutine rix(ix,error)
      use gbl_constant
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	read an entry in the input file index
      !---------------------------------------------------------------------
      integer :: ix                     !
      logical :: error                  !
    end subroutine rix
  end interface
  !
  interface
    subroutine rox(ix,error)
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	read an entry in the output file index
      !---------------------------------------------------------------------
      integer :: ix                     !
      logical :: error                  !
    end subroutine rox
  end interface
  !
  interface
    subroutine wox(error)
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	write an entry in the output file index
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine wox
  end interface
  !
  interface
    subroutine mox(ix,error)
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW 	Internal routine
      !	modifies an entry in the output file index
      !---------------------------------------------------------------------
      integer :: ix                     !
      logical :: error                  !
    end subroutine mox
  end interface
  !
  interface
    subroutine cox(error)
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Close an observation in the output file index
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine cox
  end interface
  !
  interface
    subroutine rdx(ix,error)
      use gbl_constant
      use gbl_convert
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	read an entry in the external data file index
      ! results in lbloc, ....
      !---------------------------------------------------------------------
      integer :: ix                     !
      logical :: error                  !
    end subroutine rdx
  end interface
  !
  interface
    subroutine robs(ix,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Read the record parameters of an observation
      ! Arguments :
      !	IX 	I	index number of this observation	Input
      !	ERROR	L	Logical error flag			Output
      !
      ! 26-Feb-1988	RL	Added CLIC paramerters to index
      !---------------------------------------------------------------------
      integer :: ix                     !
      logical :: error                  !
    end subroutine robs
  end interface
  !
  interface
    subroutine xobs (error)
      !---------------------------------------------------------------------
      !	Opens an output observation in the output file to extend it.
      !	This works only if this obs. is the last in the file
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine xobs
  end interface
  !
  interface
    subroutine mobs (error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	open an observation of the output file to modify it
      !	the length of each section modified must be smaller
      !	than the length of the corresponding section already present.
      !
      !	Find the latest version of this obs. in this file
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine mobs
  end interface
  !
  interface
    subroutine cobs(ltype,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	ends the writing of the output observation.
      !
      !	find the latest version of this obs. in this file
      !	and decrease its version number by 32768
      !	(and modify its pointers) supprime 15 11 84 rl
      !---------------------------------------------------------------------
      integer :: ltype                  !
      logical :: error                  !
    end subroutine cobs
  end interface
  !
  interface
    subroutine iobs(n,ix,check,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Starts the writing of an observation.
      ! Arguments :
      !	N 	I	maximum # of sections to be written
      !			(in addition of the data section)
      !	IX 	I	returns the index number of the observation
      !			in the output file
      !      CHECK    L       .true., check that observation is not a diplicate.
      !---------------------------------------------------------------------
      integer :: n                      !
      integer :: ix                     !
      logical :: check                  !
      logical :: error                  !
    end subroutine iobs
  end interface
  !
  interface
    subroutine decode_header (iw)
      integer :: iw(*)                  !
    end subroutine decode_header
  end interface
  !
  interface
    subroutine encode_header(iw)
      integer :: iw(*)                  !
    end subroutine encode_header
  end interface
  !
  interface
    subroutine rsec(ksec,lsec,sec,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Read a section in the input observation
      ! Arguments :
      !	KSEC 	I	Code of the section		Input
      !	LSEC 	I	Maximum length allowed (words)	Input
      !			Actual length (words), 		Output
      !	SEC	I*4	Section buffer			Output
      !			part not found (max - actual) is
      !			reset to zero
      !	ERROR	L	Logical error flag		Output
      !
      ! Commons : COMMON
      !---------------------------------------------------------------------
      integer :: ksec                   !
      integer :: lsec                   !
      integer*4 :: sec(*)               !
      logical :: error                  !
    end subroutine rsec
  end interface
  !
  interface
    subroutine wsec(ksec,lsec,sec,error)
      !---------------------------------------------------------------------
      ! RW	Internal routine
      !	Write a section in the current output observation
      !	KSEC 	I	Code for this section		Input
      !	LSEC 	I 	length of this section (words)	Input
      !	SEC 	I*2	contents of the section		Input
      !---------------------------------------------------------------------
      integer :: ksec                   !
      integer :: lsec                   !
      integer*4 :: sec(*)               !
      logical :: error                  !
    end subroutine wsec
  end interface
  !
  interface
    function fsec(ksec)
      !---------------------------------------------------------------------
      ! RW	Internal function
      !	Finds out if a section is present in a given observation
      !	KSEC 	I	Code for this section		Input
      !	FSEC 	L 	True if section KSEC is present Output
      !---------------------------------------------------------------------
      logical :: fsec                   !
      integer :: ksec                   !
    end function fsec
  end interface
  !
  interface
    function lensec(ksec)
      !---------------------------------------------------------------------
      ! RW	Internal function
      !	Returns the length of a section, 0 if not present
      !	KSEC 	I	Code for this section		Input
      !---------------------------------------------------------------------
      integer :: lensec                 !
      integer :: ksec                   !
    end function lensec
  end interface
  !
  interface
    subroutine rzero
      !---------------------------------------------------------------------
      ! CLASS	Internal routine
      !	Initialize the current observation (R)
      ! No subroutine
      !---------------------------------------------------------------------
      ! Global
    end subroutine rzero
  end interface
  !
  interface
    subroutine set_pass (qsb,qband,qbt,qntch,pc,pl,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Computes the passband calibration factors
      ! Baseline or Antenna based.
      !---------------------------------------------------------------------
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbt                    !
      integer :: qntch                  !
      complex :: pc(qband,qsb,qbt)      !
      complex :: pl(qntch,qsb,qbt)      !
      logical :: error                  !
    end subroutine set_pass
  end interface
  !
  interface
    subroutine set_scaling (error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Compute the scaling factors for a given scan.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_scaling
  end interface
  !
  interface
    subroutine scaling (is,isb,iba,z,a,error)
      integer :: is                     ! subband
      integer :: isb                    ! side band
      integer :: iba                    ! baseline
      complex :: z                      ! complex amplitude
      real :: a                         ! amplitude factor
      logical :: error                  !
    end subroutine scaling
  end interface
  !
  interface
    subroutine ascaling (ia,is,az,a,error)
      !---------------------------------------------------------------------
      ! This is used to rescale autocorrelation
      ! (actually only AMPLITUDE RAW = unapply atmospheric calibration)
      !
      ! created 2000-10-20 RL
      !---------------------------------------------------------------------
      integer :: ia                     ! antenna
      integer :: is                     ! subband
      real :: az                        ! amplitude factor
      real :: a                         ! amplitude factor
      logical :: error                  !
    end subroutine ascaling
  end interface
  !
  interface
    subroutine solve_baseline(line,error)
      use gildas_def
      use gkernel_types
      !---------------------------------------------------------------------
      ! CLIC
      !     Interface routine for SOLVE BASELINE/SEARCH
      ! This has been modified to allocate the outpput variables arrays dynamically
      ! What remains to be done: move alloptions/arguments recovery
      ! in the main subroutine before calls to sic_libr,
      ! so that it is not necessary to reanalyse the command line by a call to
      ! sic_analyse.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_baseline
  end interface
  !
  interface
    subroutine sub_solve_baseline(md,mb,x_data,y_data,   &
        w_data,line,error,b_dim,b_solution, b_start, b_rmss, b_dist)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Interface routine for SOLVE BASELINE/SEARCH
      !     Solves for baseline components errors.
      !     SOLVE BASELINE [DIFF] /OFFSET base1 dx dy dz ...
      ! Arguments
      !     MD       I     Number of data
      !     MB       I     Number of boxes
      !     X_DATA   R*4   X coordinates of data in boxes
      !     Y_DATA   R*4   Y coordinates of data in boxes
      !     W_DATA   R*4   Weights of data in boxes
      !     LINE     C*(*) Command line
      !     ERROR    L     Logical error flag
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      character(len=*) :: line          !
      logical :: error                  !
      integer :: b_dim                  !
      real :: b_solution(3,b_dim)       ! x, y, z
      real :: b_start(3,b_dim)          ! x0, y0, z0
      real :: b_rmss(b_dim)             ! rms
      real :: b_dist(b_dim)             ! rms
    end subroutine sub_solve_baseline
  end interface
  !
  interface
    subroutine show_errors(nd,a,kw,err)
      use gildas_def
      integer :: nd                     !
      real*8 :: a(nd,8)                 !
      real*8 :: kw                      !
      real :: err(4)                    !
    end subroutine show_errors
  end interface
  !
  interface
    subroutine resid_baseline(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine resid_baseline
  end interface
  !
  interface
    subroutine sub_resid_baseline(md,mb,x_data,y_data,line,error)
      use gildas_def
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine sub_resid_baseline
  end interface
  !
  interface
    subroutine check_baseline_closure
      use gildas_def
      ! Global
    end subroutine check_baseline_closure
  end interface
  !
  interface
    subroutine solve_bas_sub (nd,kw,x_dec,   &
        x_ha, x_tim, x_elev, y_pha,   &
        w, boff, bmes, sig, a, b, error, diff)
      use gildas_def
      integer, parameter :: mvar=8
      integer :: nd                     !
      real*8 :: kw                      !
      real :: x_dec(nd)                 !
      real :: x_ha(nd)                  !
      real :: x_tim(nd)                 !
      real :: x_elev(nd)                !
      real :: y_pha(nd)                 !
      real :: w(nd)                     !
      real :: boff(3)                   !
      real :: bmes(mvar)                !
      real :: sig                       !
      real*8 :: a(nd,mvar)              !
      real*8 :: b(nd)                   !
      logical :: error                  !
      logical :: diff                   !
    end subroutine solve_bas_sub
  end interface
  !
  interface
    subroutine solve_cal(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_cal
  end interface
  !
  interface
    subroutine sub_solve_cal(md,mb,x_data,y_data,w_data,   &
        ind,wk1,xxx,yyy,www,line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Computes a calibration curve, based on the phase calibrators,
      !	assumed to be in the current index.
      !	Amplitudes should be also checked.
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: ind(md)                !
      real*8 :: wk1(3*md)               !
      real*8 :: xxx(md)                 !
      real*8 :: yyy(md)                 !
      real*8 :: www(md)                 !
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine sub_solve_cal
  end interface
  !
  interface
    subroutine nag_fail (fac,prog,ifail,error)
      character(len=*) :: fac           !
      character(len=*) :: prog          !
      integer :: ifail                  !
      logical :: error                  !
    end subroutine nag_fail
  end interface
  !
  interface
    function gr8_random (x,n)
      logical :: gr8_random             !
      integer :: n                      !
      real*8 :: x(n)                    !
    end function gr8_random
  end interface
  !
  interface
    subroutine solve_cal_ant(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Computes a calibration curve, based on the phase calibrators,
      !	assumed to be in the current index.
      !	Amplitudes should be also checked.
      ! Antenna based.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_cal_ant
  end interface
  !
  interface
    subroutine sub_solve_cal_ant(ly,nd,nd_work,nbas,nant,xxx,yyy,www,   &
        ind,wk1,   &
        md,mb,x_data,y_data,w_data,nkn,tkn,weight,poly,ipol,plot,error)
      use gildas_def
      integer :: ly                     !
      integer :: nd                     !
      integer :: nd_work                    !
      integer :: nbas                   !
      integer :: nant                   !
      real*8 :: xxx(nd_work)                 !
      real*8 :: yyy(nd_work,nbas)            !
      real*8 :: www(nd_work,nbas)            !
      integer :: ind(nd_work)                !
      real*8 :: wk1(nd_work,nbas,4)          !
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: nkn                    !
      real :: tkn(nkn)                  !
      logical :: weight                 !
      logical :: poly                   !
      integer :: ipol                   !
      logical :: plot                   !
      logical :: error                  !
    end subroutine sub_solve_cal_ant
  end interface
  !
  interface
    subroutine check_amp_clos(np,nant,y,w)
      !---------------------------------------------------------------------
      ! Check available amplitude closures
      !---------------------------------------------------------------------
      integer :: np                     !
      integer :: nant                   !
      real*8 :: y(np,*)                 !
      real*8 :: w(np,*)                 !
    end subroutine check_amp_clos
  end interface
  !
  interface
    subroutine check_pha_clos(np,nant,y,w)
      use gildas_def
      !---------------------------------------------------------------------
      ! Check available amplitude closures,
      !---------------------------------------------------------------------
      integer :: np                     !
      integer :: nant                   !
      real*8 :: y(np,*)                 !
      real*8 :: w(np,*)                 !
    end subroutine check_pha_clos
  end interface
  !
  interface
    subroutine solve_delay(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Interface routine for SOLVE DELAY/SEARCH/PRINT
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_delay
  end interface
  !
  interface
    subroutine sub_solve_delay(md,mb,x_data,y_data,line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     CLIC
      !     Interface routine for SOLVE DELAY/SEARCH
      !     Solves for DELAY components errors.
      !     SOLVE DELAY [DIFF] /OFFSET base1 dx dy dz ...
      !     Arguments
      !     MD       I     Number of data
      !     MB       I     Number of boxes
      !     X_DATA   R*4   X coordinates of data in boxes
      !     Y_DATA   R*4   Y coordinates of data in boxes
      !     LINE     C*(*) Command line
      !     ERROR    L     Logical error flag
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine sub_solve_delay
  end interface
  !
  interface
    subroutine plot_delay(ib,iband,bmin,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Plot the result from delay determination
      !     IB      INTEGER    Box number
      !     BMIN(2) REAL       Limits for plot
      !---------------------------------------------------------------------
      integer :: ib                     !
      integer :: iband                  !
      real :: bmin(2)                   !
      logical :: error                  !
    end subroutine plot_delay
  end interface
  !
  interface
    subroutine show_del_errors(a)
      !---------------------------------------------------------------------
      ! CLIC
      !     Print result from delay determination
      !     A       REAL*8    Square of the errors
      !---------------------------------------------------------------------
      real*8 :: a                       !
    end subroutine show_del_errors
  end interface
  !
  interface
    subroutine solve_del_sub (nd,   &
        x_if, y_pha, iband, boff, bmes, sig, a, b, error)
      use gildas_def
      !---------------------------------------------------------------------
      !     ND        INTEGER    Number of points
      !     X_IF(ND)  REAL       I_F frequency of points
      !     Y_PHA(ND) REAL       Phases of points
      !     BOFF      REAL       Original delay offset
      !     BMES(2)   REAL       Returned delay (?) or phase range (?)
      !     SIG       REAL       Rms error on phase (?) or delay (?)
      !     A(ND,MVAR)   REAL*8  Work space for minimization routine
      !                          on return, covariance matrix ?
      !     B(ND)        REAL*8  Ibid
      !     ERROR     LOGICAL    Error flag
      !---------------------------------------------------------------------
      integer :: nd                     !
      real :: x_if(nd)                  !
      real :: y_pha(nd)                 !
      integer :: iband                  !
      real :: boff                      !
      real :: bmes(2)                   !
      real :: sig                       !
      real*8 :: a(nd,*)                 !
      real*8 :: b(nd)                   !
      logical :: error                  !
    end subroutine solve_del_sub
  end interface
  !
  interface
    subroutine solve_five(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for command
      !	SOLVE FIVE [/PLOT] [/OUTPUT File [NEW]]
      !	[/Store /Reset /Offset /Closure /Search] Not relevant here
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_five
  end interface
  !
  interface
    subroutine load_five (data,nant,wpos,vpos,offset,woffset,   &
        error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE FIVE
      !	Measure antenna pointing position from a specialised
      !	routine in the real-time acquisition system
      ! Arguments:
      !	JADDR	I	Array address
      !	NADDR	I	Array length
      !	LO1	R*4 	LO1 frequency
      !	NANT	I	Number of antennas
      !---------------------------------------------------------------------
      real :: data(*)                   !
      integer :: nant                   !
      real :: wpos(5,nant)              !
      real :: vpos(5,nant)              !
      real :: offset                    !
      real :: woffset                   !
      logical :: error                  !
    end subroutine load_five
  end interface
  !
  interface
    subroutine sub_five(plot,qlun,iout,out,flux,   &
        wpos,vpos,offset,woffset,bsize,error,end)
      include 'clic_parameter.inc'
      logical :: plot                   !
      integer :: qlun                   !
      integer :: iout                   !
      logical :: out                    !
      logical :: flux                   !
      real :: wpos(5,mnant)             !
      real :: vpos(5,mnant)             !
      real :: offset                    !
      real :: woffset                   !
      real :: bsize                     !
      logical :: error                  !
      logical :: end                    !
    end subroutine sub_five
  end interface
  !
  interface
    subroutine gaufiv (z,dz,aoff,boff,i0,daoff,dboff,di0,bsize,offset)
      !---------------------------------------------------------------------
      !	Find the gaussian of known width going through five points
      !			5
      !		1	3	2
      !			4
      !	Z	R*4(5)	The measurements
      !	DZ	R*4	The typical error on Z
      !	OFFSET	R*4	Point separation
      !       bsize   R*4     Beam size
      !	AOFF	R*4	First axis offset		Output
      !	BOFF	R*4	First axis offset		Output
      !	I0	R*4	Peak intensity                  Output
      !	DAOFF	R*4	Error on AOFF                   Output
      !	DBAOFF	R*4	Error on BOFF                   Output
      !       DI0
      !---------------------------------------------------------------------
      real :: z(5)                      !
      real :: dz                        !
      real :: aoff                      !
      real :: boff                      !
      real :: i0                        !
      real :: daoff                     !
      real :: dboff                     !
      real :: di0                       !
      real :: bsize                     !
      real :: offset                    !
    end subroutine gaufiv
  end interface
  !
  interface
    subroutine check_subb(n,first,error)
      use gildas_def
      integer :: n                      ! number of subband groups to be checke
      logical :: first                  !
      logical :: error                  !
    end subroutine check_subb
  end interface
  !
  interface
    subroutine out_point(iout,flux,iant,coo,i0,di0,off,doff,b,db,   &
        error)
      !---------------------------------------------------------------------
      ! Output a line in plot result files
      !---------------------------------------------------------------------
      integer :: iout                   !
      logical :: flux                   !
      integer :: iant                   !
      character(len=*) :: coo           !
      real :: i0                        !
      real :: di0                       !
      real :: off                       !
      real :: doff                      !
      real :: b                         !
      real :: db                        !
      logical :: error                  !
    end subroutine out_point
  end interface
  !
  interface
    subroutine file_point(line,output,p_lun,flux,q_lun,tpoint,error)
      !---------------------------------------------------------------------
      ! Open an output file for the PLOT session
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: output                 !
      integer :: p_lun                  !
      logical :: flux                   !
      integer :: q_lun                  !
      logical :: tpoint                 !
      logical :: error                  !
    end subroutine file_point
  end interface
  !
  interface
    subroutine solve_flux(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	Calculate the flux of sources in the current index,
      !	or the efficiency (JY_TO_KEL) if the flux is known
      !       /RESET : reset all unknown fluxes to zero.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_flux
  end interface
  !
  interface
    subroutine solve_focus(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      ! CLIC
      !	Support routine for command
      !	SOLVE FOCUS [/PLOT] [/OUTPUT File [NEW]] [/COMPRESS time_max]
      !	[/Store /Reset /Offset /Search] Not relevant here
      !       [/PRINT]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_focus
  end interface
  !
  interface
    subroutine check_focus(error)
      !---------------------------------------------------------------------
      ! CLIC
      !	Check if data is a pointing scan
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine check_focus
  end interface
  !
  interface
    subroutine output_file(caller,line,iopt,lun,ext,header,error)
      !---------------------------------------------------------------------
      ! Open output file for logging
      !---------------------------------------------------------------------
      character(len=*) :: caller        !
      character(len=*) :: line          !
      integer :: iopt                   !
      integer :: lun                    !
      character(len=*) :: ext           !
      character(len=*) :: header        !
      logical :: error                  !
    end subroutine output_file
  end interface
  !
  interface
    subroutine load_focw (beam, bsize, lo1, nant)
      use gildas_def
      real :: beam                      !
      real :: bsize                     !
      real :: lo1                       !
      integer :: nant                   !
    end subroutine load_focw
  end interface
  !
  interface
    subroutine solve_gain(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     CLIC
      !     Support routine for command
      !     SOLVE GAIN [SCAN]
      !     [/output /plot /Store /Reset /Offset /Closure /Search] :
      !     Not relevant here
      !     SOLVE GAIN SCAN = reduction scan by scan; otherwise average of scans.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_gain
  end interface
  !
  interface
    subroutine sub_gain (m,mean_visi,mean_wisi,inbc,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Measure the receiver gain ratios
      !       M number of antennas
      !---------------------------------------------------------------------
      integer :: m                      !
      complex :: mean_visi(2,m)         !
      real ::    mean_wisi(2,m)         !
      integer :: inbc                   !
      logical :: error                  !
    end subroutine sub_gain
  end interface
  !
  interface
    subroutine sub_solve_pha(qlun, ir, mean_visi, mean_wisi, error)
      use gildas_def
      include 'clic_parameter.inc'
      integer :: qlun                      !
      integer :: ir                        !
      complex :: mean_visi(2,mnant,mnrec)  !
      real    :: mean_wisi(2,mnant,mnrec)  !
      logical :: error                     !
    end subroutine sub_solve_pha
  end interface
  !
  interface
    subroutine sub_solve_gain(qlun, ir, mean_visi, mean_wisi, error)
      use gildas_def
      include 'clic_parameter.inc'
      integer :: qlun                      !
      integer :: ir                        !
      complex :: mean_visi(2,mnant,mnrec)  !
      real    :: mean_wisi(2,mnant,mnrec)  !
      logical :: error                     !
    end subroutine sub_solve_gain
  end interface
  !
  interface
    subroutine solve_pass(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     +	' SOLVE',	'/PLOT',	'/WEIGHT',	'/OUTPUT',
      !     +	'/RESET',	'/OFFSET',	'/COMPRESS',	'/SEARCH',
      !     +	'/BREAK',	'/POLYNOMIAL',
      !	Computes a passband based on the passband calibrator
      !	assumed to be in the current index.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_pass
  end interface
  !
  interface
    subroutine channel_cont(md, mb, ib, inbc, nd,   &
        x_data, y_data, w_data, z_data)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC: passband calibration.
      ! store values as channel-by channel for continuum
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      integer :: ib                     !
      integer :: inbc                   !
      integer :: nd                     !
      real :: x_data(md, mb)            !
      real :: y_data(md, mb)            !
      real :: w_data(md, mb)            !
      complex :: z_data(md, mb)         !
    end subroutine channel_cont
  end interface
  !
  interface
    subroutine fit_polyc (md,mb,ib,inbc,nd,x_data,y_data,w_data,   &
        z_data,work1,xx,yy,ww,ind,plot,ndeg,rss,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC routine: Passband calibration
      !     Fit polynomial as a function of I_F frequency
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      integer :: ib                     !
      integer :: inbc                   !
      integer :: nd                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      complex :: z_data(md,mb)          !
      real*8 :: work1(3*nd)             !
      real*8 :: xx(nd)                  !
      real*8 :: yy(nd)                  !
      real*8 :: ww(nd)                  !
      integer :: ind(nd)                !
      logical :: plot                   !
      integer :: ndeg                   !
      real :: rss                       !
      logical :: error                  !
    end subroutine fit_polyc
  end interface
  !
  interface
    subroutine fit_poly (md,mb,ib,inbc,nd,x_data,y_data,w_data,z_data,   &
        work1,xx,yy,ww,plot,ndeg,rss,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC routine: Passband calibration
      !     Fit polynomial as a function frequency-like variable in an individual
      !     line subband.
      !---------------------------------------------------------------------
      integer :: md                     !
      integer :: mb                     !
      integer :: ib                     !
      integer :: inbc                   !
      integer :: nd                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      complex :: z_data(md,mb)          !
      real*8 :: work1(3*nd)             !
      real*8 :: xx(nd)                  !
      real*8 :: yy(nd)                  !
      real*8 :: ww(nd)                  !
      logical :: plot                   !
      integer :: ndeg                   !
      real :: rss                       !
      logical :: error                  !
    end subroutine fit_poly
  end interface
  !
  interface
    subroutine getxvall(ix,isb,ic,xr,xv,xd)
      !---------------------------------------------------------------------
      ! CLIC
      !	Return conversion formula for the X axis of type IX
      !	Valid for spectral correlator
      ! Input:
      !	IX	Integer	Type of data (Chann, Veloc, I_fre, Sky_f)
      !	ISB	Integer	Sideband
      !	IC	Integer SubBand
      ! Output:
      !	XR	Real	Reference point
      !	XV	Real	Value at reference point
      !	XD	Real	Increment per point
      !---------------------------------------------------------------------
      integer :: ix                     !
      integer :: isb                    !
      integer :: ic                     !
      real :: xr                        !
      real :: xv                        !
      real :: xd                        !
    end subroutine getxvall
  end interface
  !
  interface
    subroutine getxvalc(ix,isb,xr,xv,xd)
      !---------------------------------------------------------------------
      ! CLIC
      !	Return conversion formula for the X axis of type IX
      !	Valid for continuum correlator (old model)
      ! Input:
      !	IX	Integer	Type of data (Chann, Veloc, I_fre, Sky_f)
      !	ISB	Integer	Sideband
      ! Output:
      !	XR	Real	Reference point
      !	XV	Real	Value at reference point
      !	XD	Real	Increment per point
      !---------------------------------------------------------------------
      integer :: ix                     !
      integer :: isb                    !
      real :: xr                        !
      real :: xv                        !
      real :: xd                        !
    end subroutine getxvalc
  end interface
  !
  interface
    function fillin(r,ival,imin,imax,bad)
      !---------------------------------------------------------------------
      ! SAS	Internal routine
      !	Interpolate bad channel (if possible)
      ! Arguments
      !	R	R*4(*)	Array to be interpolated
      !	IVAL	I	Pixel to be interpolated
      !	IMIN	I	First pixel in array
      !	IMAX	I	Last pixel in array
      !	BAD	R*4	Blanking value
      ! Created	21-Mar-1986	S.Guilloteau
      !---------------------------------------------------------------------
      real :: fillin                    !
      integer :: imax                   !
      real :: r(imax)                   !
      integer :: ival                   !
      integer :: imin                   !
      real :: bad                       !
    end function fillin
  end interface
  !
  interface
    subroutine solve_pass_ant(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     +	' SOLVE',	'/PLOT',	'/WEIGHT',	'/OUTPUT',
      !     +	'/RESET',	'/OFFSET',	'/COMPRESS',	'/SEARCH',
      !     +	'/BREAK',	'/POLYNOMIAL',
      !	Computes a passband based on the passband calibrator
      !	assumed to be in the current index.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_pass_ant
  end interface
  !
  interface
    subroutine save_averages(sbav,error)
      use gildas_def
      include 'clic_parameter.inc'
      complex :: sbav(2,-mnant:mnbas,mnrec)  !
      logical :: error                       !
    end subroutine save_averages
  end interface
  !
  interface
    subroutine channel_cont_ant(ly, md, mb, x_data, y_data, w_data,   &
        error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC: passband calibration.
      !     store values as channel-by channel passband for continuum
      !     Antenna-based version: LY = 1 (Amp) or 2 (Phases)
      !---------------------------------------------------------------------
      integer :: ly                     !
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md, mb)            !
      real :: y_data(md, mb)            !
      real :: w_data(md, mb)            !
      logical :: error                  !
    end subroutine channel_cont_ant
  end interface
  !
  interface
    subroutine phase_ant(nant,nbas,bd,wbd,ad,wad,error)
      integer :: nant                   !
      integer :: nbas                   !
      real :: bd(nbas)                  !
      real :: wbd(nbas)                 !
      real :: ad(nant)                  !
      real :: wad(nant)                 !
      logical :: error                  !
    end subroutine phase_ant
  end interface
  !
  interface
    subroutine ampli_ant(nant,nbas,bd,wbd,ad,wad,error)
      integer :: nant                   !
      integer :: nbas                   !
      real :: bd(nbas)                  !
      real :: wbd(nbas)                 !
      real :: ad(nant)                  !
      real :: wad(nant)                 !
      logical :: error                  !
    end subroutine ampli_ant
  end interface
  !
  interface
    subroutine fit_poly_fr_ant(ly,md,mb,x_data,y_data,w_data,   &
        nd, nbas,nant,xx,yy,ww,ind,wk,plot,ndeg,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC routine: Passband calibration
      !     Fit polynomial as a function of I_F frequency
      !     Antenna-based version
      !---------------------------------------------------------------------
      integer :: ly                     !
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: nd                     !
      integer :: nbas                   !
      integer :: nant                   !
      real*8 :: xx(nd)                  !
      real*8 :: yy(nd,nbas)             !
      real*8 :: ww(nd,nbas)             !
      integer :: ind(nd)                !
      real*8 :: wk(nd)                  !
      logical :: plot                   !
      integer :: ndeg                   !
      logical :: error                  !
    end subroutine fit_poly_fr_ant
  end interface
  !
  interface
    subroutine phi_plot(g1,g2,nd,xx,yy,turn)
      real :: g1                        !
      real :: g2                        !
      integer :: nd                     !
      real*8 :: xx(nd)                  !
      real*8 :: yy(nd)                  !
      real :: turn                      !
    end subroutine phi_plot
  end interface
  !
  interface
    subroutine fit_poly_ch_ant(ly,md,mb,x_data,y_data,w_data,   &
        nd, nbas,nant,xx,yy,ww,ind,wk,plot,ndeg,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC routine: Passband calibration
      !     Fit polynomial as a function frequency-like variable in an individual
      !     line subband.
      !     Antenna-based version
      !---------------------------------------------------------------------
      integer :: ly                     !
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: nd                     !
      integer :: nbas                   !
      integer :: nant                   !
      real*8 :: xx(nd)                  !
      real*8 :: yy(nd,nbas)             !
      real*8 :: ww(nd,nbas)             !
      integer :: ind(nd)                !
      real*8 :: wk(nd)                  !
      logical :: plot                   !
      integer :: ndeg                   !
      logical :: error                  !
    end subroutine fit_poly_ch_ant
  end interface
  !
  interface
    subroutine antvisi (z,w,zant,want,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Derive antenna "gains" from baseline visibilities
      ! Arguments:
      !     Z(MNBAS)    COMPLEX   Visibility
      !     W(MNBAS)    REAL      Weight
      !     ZANT(MNANT) COMPLEX   Complex antenna gain
      !     WANT(MNANT) REAL      Weight
      ! Call Tree
      !	...	GAIN_CONT
      !	...	GAIN_LINE
      !	...	ARECORD
      !	...	CONT_AVERAGE
      !	...	SUB_SOLVE_PASS
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      complex :: z(mnbas)               !
      real*4 :: w(mnbas)                !
      complex :: zant(mnant)            !
      real*4 :: want(mnant)             !
      logical :: error                  !
    end subroutine antvisi
  end interface
  !
  interface
    subroutine solve_pseudo (line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for command
      !	SOLVE PSEUDO [BEAM] [/PLOT] [/OUTPUT File [OLD|NEW|PROC]]
      !              [/COMPRESS TIME_MAX]
      !	[/Store /Reset /Offset /Search] Not relevant here
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_pseudo
  end interface
  !
  interface
    subroutine check_pseudo(argum,error)
      !---------------------------------------------------------------------
      ! CLIC
      !	Check if data is a PSEUDOing scan
      !---------------------------------------------------------------------
      character(len=*) :: argum         !
      logical :: error                  !
    end subroutine check_pseudo
  end interface
  !
  interface
    subroutine load_pseudo (icount,jaddr,naddr,error,lo1,nant,   &
        time_max)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE PSEUDO
      !	Measure antenna pointing position from a specialised
      !	routine in the real-time acquisition system
      ! Arguments:
      !	ICOUNT	I	Subscan number 0 : not loaded
      !	JADDR	I	Array address
      !	NADDR	I	Array length
      !	LO1	R*4 	LO1 frequency
      !	NANT	I	Number of antennas
      !       TIME_MAX R*4    Max. integration time (compression)
      !---------------------------------------------------------------------
      integer :: icount                      !
      integer(kind=address_length) :: jaddr  !
      integer :: naddr                       !
      logical :: error                       !
      real :: lo1                            !
      integer :: nant                        !
      real :: time_max                       !
    end subroutine load_pseudo
  end interface
  !
  interface
    subroutine fill_pseu_posi (qdump,qbas,qant,visi,posi,weig,   &
        time_max,data_in,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Read the data section and extract the visibilities for all baselines
      ! and the positions of all antennas
      ! Arguments:
      !	R 	Struc.	Header
      !	VISI	R*4(*)	Visibilities to be returned
      !	POSI	R*4(*)	"Position" of antenna. The "position"
      !			is actually a parameter defined by R.SCATY
      !	WEIG	R*4(*)	Weights to be returned
      !---------------------------------------------------------------------
      integer :: qdump                         !
      integer :: qbas                          !
      integer :: qant                          !
      real :: visi(qdump,qant)                 !
      real :: posi(qdump,qant)                 !
      real :: weig(qdump,qant)                 !
      real :: time_max                         !
      integer(kind=address_length) :: data_in  !
      logical :: error                         !
    end subroutine fill_pseu_posi
  end interface
  !
  interface
    subroutine load_holo(iant,nvis,ivis,vis,all_bases,   &
        nbasu,basu,fudge,pcorr,acheat,pcheat,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE HOLOGRAPHY
      ! Arguments:
      !	IANT	I	antenna used
      !       TIME_MAX R*4    Max. integration time (compression)
      !---------------------------------------------------------------------
      integer :: iant                   !
      integer :: nvis                   !
      integer :: ivis                   !
      real :: vis(5,nvis)               !
      logical :: all_bases              !
      integer :: nbasu                  !
      integer :: basu(nbasu)            !
      real :: fudge                     !
      real :: pcorr(2)                  !
      real :: acheat                    !
      real :: pcheat                    !
      logical :: error                  !
    end subroutine load_holo
  end interface
  !
  interface
    subroutine solve_holo(line,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! CLIC
      ! SOLVE HOLO [MODES nmode] [NP np] [MASK npm pm1 ... pmn] [ITER niter gain]
      ! /PLOT [NUMBER] [AMP min max step] [PHA min max step]
      ! defaults
      !         MODES 0
      !         NP    64
      !         MASK  0
      !         ITER  0
      !         NONUMBER
      !         AMP -15  0    3  (decibels)
      !         PHA -pi pi pi|8  (radians)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_holo
  end interface
  !
  interface
    subroutine fit_illum(v, nx, ny, a,b, ames, nxy, npm,pm, docheat)
      use gildas_def
      use clic_xypar
      integer, parameter :: mvar=5      !
      integer :: nx                     !
      integer :: ny                     !
      real*4 :: v(nx,ny,4)              ! the amplitude is in V(3)
      integer :: nxy                    !
      real*8 :: a(nxy,*)                !
      real*8 :: b(*)                    !
      real*8 :: ames(mvar)              !
      integer :: npm                    !
      character(len=5) :: pm(200)       !
      logical :: docheat                !
    end subroutine fit_illum
  end interface
  !
  interface
    subroutine amp_residuals(v, nx, ny)
      use gildas_def
      integer :: nx                     !
      integer :: ny                     !
      real*4 :: v(nx,ny,4)              !
    end subroutine amp_residuals
  end interface
  !
  interface
    subroutine amp_restore(v, nx, ny)
      use gildas_def
      integer :: nx                     !
      integer :: ny                     !
      real*4 :: v(nx,ny,4)              !
    end subroutine amp_restore
  end interface
  !
  interface
    subroutine fit_holo(v,nx,ny,kw,a,b,boff,bmes,sig,nxy,   &
        nfr,ifr,nfs,ifs,mvar,defocus)
      use gildas_def
      use clic_xypar
      integer :: nx                     !
      integer :: ny                     !
      real*4 :: v(nx,ny,4)              !
      real*8 :: kw                      !
      integer :: nxy                    !
      real*8 :: a(nxy,*)                !
      real*8 :: b(*)
      integer, parameter :: maxvar = 8!
      real*8 :: boff(maxvar)                 !
      real*8 :: bmes(maxvar)                 !
      real*8 :: sig                     !
      integer :: nfr                    !
      integer :: ifr(nfr)               !
      integer :: nfs                    !
      integer :: ifs(nfs)               !
      integer :: mvar                   !
      real*4 :: defocus                 !
    end subroutine fit_holo
  end interface
  !
  interface
    subroutine range_pi8(b)
      real*8 :: b                       !
    end subroutine range_pi8
  end interface
  !
  interface
    subroutine res_holo(v,nx,ny,bmes,ames,kw,   &
         rmsu,rmsw,rmsast,&
         sigu,sigw,eff,rms_ring,s_ring,z_ring,mvar,defocus,taper, &
         rigging, elevation, riggingElevation, tempBias, biasTemperature, &
         temperature, error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Compute in V(2) the residuals from the fit in V(4)
      !---------------------------------------------------------------------
      include 'clic_panels.inc'
      integer :: nx                     !
      integer :: ny                     !
      integer, parameter :: Maxvar=8
      real :: v(nx,ny,5)                !
      real*8 :: bmes(maxvar)                 !
      real*8 :: ames(5)                 !
      real*8 :: kw                      !
      real :: rmsu                      !
      real :: rmsw  
      real :: rmsast                     !
      real :: sigu                      !
      real :: sigw                      !
      real :: eff(8)                    !
      real :: rms_ring(mring)           !
      real :: s_ring(mring)             !
      real :: z_ring(mring)             !
      integer :: mvar                   !
      real :: defocus                   !
      real :: taper                     !
      real :: xref                      !
      real :: xval                      !
      real :: xinc                      !
      real :: rlam                     !
      ! Global
      include 'gbl_pi.inc'
      ! Local
      real :: xx, yy
      real :: a(maxvar)
      real*8 :: k230, k345, df
      real :: x, y, x2, y2, r2, f, cx, cz, w, sw, su, def
      real :: meansu, meansw, meanw, meanu, d
      integer :: i, j, l, npanel, iring, ipanel
      complex :: e, e230, e345
      real :: e1, e2, twopi, pf, spf, radius
      character(len=5) :: pch
      !
      real :: elevation, riggingElevation(2) 
      real :: temperature, biasTemperature
      logical :: error, rigging, tempBias
    end subroutine res_holo
  end interface
  !
  interface
    subroutine zero_holo(v,n)
      integer :: n                      !
      real :: v(n)                      !
    end subroutine zero_holo
  end interface
  !
  interface
    subroutine solve_panels(niter,gain,amp,phas,phfit,phres,nxap,   &
        rlam,xref,xval,xinc,nmode,kant,h1,h2,work1,work2,nxbeam,nybeam,   &
        plot,scale,lprint,rwork1,rwork2, npm, pm, numscrew, screwname,   &
        screwmotion, screwmotionerror, phasrms, error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      !	from map of aperture phase distribution ,calculates 5 modes for
      !	each panel of 15m telescopes,then gives the screw adjustements.
      !
      !	return fitted aperture plane phase and residuals
      !
      !     niter   nombre d'iterations
      !     gain    gain
      !     rlam    wavelength (mm)
      !     xref    index of reference point in x
      !     xval    x coord of ref point (mm)
      !     xinc    sampling interval (mm)
      !     ( assumed the same for y coord)
      !---------------------------------------------------------------------
      integer :: niter                          !
      real :: gain                              !
      integer :: nxap                           !
      real :: amp(nxap,nxap)                    !
      real :: phas(nxap,nxap)                   !
      real :: phfit(nxap,nxap)                  !
      real :: phres(nxap,nxap)                  !
      real :: rlam                              !
      real :: xref                              !
      real :: xval                              !
      real :: xinc                              !
      integer :: nmode                          !
      integer :: kant                           !
      character(len=*) :: h1                    !
      character(len=*) :: h2                    !
      complex :: work1(nxap,nxap)               !
      complex :: work2(nxap,nxap)               !
      integer :: nxbeam                         !
      integer :: nybeam                         !
      ! logical :: plot, rigging, error                  !
      logical :: plot, error                  !
      real :: scale(3)                          !
      integer :: lprint                         !
      real :: rwork1(nxap,nxap)                 !
      real :: rwork2(nxap,nxap)                 !
      integer :: npm                            !
      character(len=5) :: pm(npm)               !
      integer :: numscrew                       !
      character(len=12) :: screwname(numscrew)  !
      real :: screwmotion(numscrew)             !
      real :: screwmotionerror(numscrew)        !
      real(kind=4), intent(out) :: phasrms      !
    end subroutine solve_panels
  end interface
  !
  interface
    subroutine mode (amp,phas,nxap,xref,xval,xinc,rlam,   &
        am, ae, nmode, r0)
      !---------------------------------------------------------------------
      !	takes aperture phase distribution phas,interpolates values
      !	onto fine grid on actual aperture.those values falling on a
      !	particular panel are used in the cal of least squares sums
      !	for that panel.
      !	output are amplitudes of 5 modes per panel
      !
      !	phas	"map" of phase of aperture distribution
      !	nxap	nxap x nxap is dimension of phas
      !       xref    index of reference point in x
      !       xval    x coord of ref point (mm)
      !       xinc    sampling interval (mm)
      !       ( assumed the same for y coord)
      !	rlam	wavelength in mm
      !	ng2	ng2 x nxap is number of interpolated values across antenna
      !	mpan	max no of panels per ring (32 for 15m tel.)
      !	nr	no of rings (6 for 15m tel.)
      !	aa	array of "tilt" modal amplitudes in x direction
      !	bb	array of "tilt" modal amplitudes in y direction
      !	dd	array of "piston"  modal amplitudes
      !	tt	array of "torsion" modal amplitudes
      !	cc	array of "boss"    modal amplitudes (assumed spherical)
      !---------------------------------------------------------------------
      include 'clic_panels.inc'
      integer :: nxap                   !
      real :: amp(nxap,nxap)            !
      real :: phas(nxap,nxap)           !
      real :: xref                      !
      real :: xval                      !
      real :: xinc                      !
      real :: rlam                      !
      real :: am(mm,mring,mpan)         !
      real :: ae(mm,mring,mpan)         !
      integer :: nmode                  !
      real :: r0                        !
    end subroutine mode
  end interface
  !
  interface
    function fmode(jm,ir,xx,yy)
      real :: fmode                     !
      integer :: jm, im                     !
      integer :: ir                     !
      real :: xx                        !
      real :: yy                        !
    end function fmode
  end interface
  !
  interface
    subroutine screws(am,ae,nmode,lprint, numscrew, screwname,   &
         screwmotion, screwmotionerror,  error)                          
         !    rigging, elevation, riggingElevation, error)
      !---------------------------------------------------------------------
      !     calculates screw adjustments for 15m telescope given 5 modes for
      !     each panel-- aa,bb,dd,tt,cc.
      !
      !     z=dd + aa*(x-xxm) + bb*y + tt*y*(x-xx2)
      !     + cc*[ c2*(x-xc)**2 + s2*y**2 - c2*xxm2 - s2*yy2 ]
      !
      !     calculation as for 30m telescope but screw nos changed for output
      !
      !     x local panel coord is directed radially outwards
      !     y local panel coord is directed in azimuth in direction of
      !     panel counting in each ring.
      !     xxm  mean x coord of panel
      !     xx2  rms  x coord of panel
      !     xc       x coord for "boss" calculation
      !     xxm2 mean square (x-xc)
      !     yy2  mean square y
      !     c2   cos(phi)**2 (0.5 as first approx.)
      !     s2   sin(phi)**2 (0.5 as first approx.)
      !     phi  angle specifying ratio of curvature of "boss" in x and y
      !     =45 degrees as first approximation
      !---------------------------------------------------------------------
      include 'clic_panels.inc'
      real :: am(mm,mring,mpan)                 !
      real :: ae(mm,mring,mpan)                 !
      integer :: nmode                          !
      integer :: lprint                         !
      integer :: numscrew                       ! screw results for CalDM
      character(len=12) :: screwname(numscrew)  !
      real :: screwmotion(numscrew)           !
      ! real :: screwmotion(numscrew), elevation, riggingElevation(2)             !
      real :: screwmotionerror(numscrew)
      logical error
    end subroutine screws
  end interface
  !
  interface
    subroutine modemap(phfit,nxap,rlamda,   &
        xref, xval, xinc,am,ae,nmode)
      !---------------------------------------------------------------------
      !	makes an aperture plane map of 15m telescope using up to 5 modes
      !	makes residal map also,using phas map
      !
      !	fills a complex aperture array psi from data given by the modes
      !	am
      !	takes measured phases and makes a residual complex map
      !
      !	number of modes used defined by nmode 1,3,4,5
      !	wavelength rlamda,relative sampling interval relsampi and power
      !	taper is taperdb (+ve)
      !	returns rms deflection normally,and axially,and no points used
      !---------------------------------------------------------------------
      include 'clic_panels.inc'
      integer :: nxap                   !
      real :: phfit(nxap,nxap)          !
      real :: rlamda                    !
      real :: xref                      !
      real :: xval                      !
      real :: xinc                      !
      real :: am(5,mring,mpan)          !
      real :: ae(5,mring,mpan)          !
      integer :: nmode                  !
    end subroutine modemap
  end interface
  !
  interface
    subroutine resmap(phas,phfit,phres,nxap,rlamda,   &
        xref, xval, xinc, znrms, zarms, rmsres, phasrms,   &
        amp,work1,work2,nxbeam,nybeam, npm, pm)
      !---------------------------------------------------------------------
      ! Using the phase of Fitted panels PHFIT, compute the residuals
      ! from the input phase map PHAS
      !---------------------------------------------------------------------
      integer :: nxap                   !
      real :: phas(nxap,nxap)           !
      real :: phfit(nxap,nxap)          !
      real :: phres(nxap,nxap)          !
      real :: rlamda                    !
      real :: xref                      !
      real :: xval                      !
      real :: xinc                      !
      real :: znrms                     !
      real :: zarms                     !
      real :: rmsres                    !
      real :: phasrms                   !
      real :: amp(nxap,nxap)            !
      complex :: work1(nxap,nxap)       !
      complex :: work2(nxap,nxap)       !
      integer :: nxbeam                 !
      integer :: nybeam                 !
      integer :: npm                    !
      character(len=5) :: pm(200)       !
    end subroutine resmap
  end interface
  !
  interface
    subroutine xypanel(x,y,ir,ip,xpanel,ypanel)
      !---------------------------------------------------------------------
      !     takes x,y,coords in aperture (mm)  for each call
      !     identifies corresponding panel(i3,i1) and calculates
      !     local coords xpanel,ypanel
      !
      !     outside radius of rings (inner section being 0),and average
      !---------------------------------------------------------------------
      real :: x                         !
      real :: y                         !
      integer :: ir                     !
      integer :: ip                     !
      real :: xpanel                    !
      real :: ypanel                    !
    end subroutine xypanel
  end interface
  !
  interface
    function cpanel(ir,ip)
      character(len=5) :: cpanel        !
      integer :: ir                     !
      integer :: ip                     !
    end function cpanel
  end interface
  !
  interface
    subroutine zpanel(x,y,ir,ip,am,ae,zpan,epan,nmode)
      include 'clic_panels.inc'
      real :: x                         !
      real :: y                         !
      integer :: ir                     !
      integer :: ip                     !
      real :: am(mm,mring,mpan)         !
      real :: ae(mm,mring,mpan)         !
      real :: zpan                      !
      real :: epan                      !
      integer :: nmode                  !
    end subroutine zpanel
  end interface
  !
  interface
    subroutine fitres(phres,am_obs,phfit,work,zap,   &
        nxbeam,nybeam,nxap,nyap)
      !---------------------------------------------------------------------
      ! compute the effect of fitted panels (phfit) on the observed phase
      ! phres is in fact the response to phfit by the observing process.
      ! one uses the observed amplitude as a weight.
      !---------------------------------------------------------------------
      integer :: nxap                   ! dimensions in aperture plane
      integer :: nyap                   ! dimensions in aperture plane
      real :: phres(nxap,nyap)          !
      real :: am_obs(nxap,nyap)         !
      real :: phfit(nxap,nyap)          !
      complex :: work(nxap,nyap)        !
      complex :: zap(nxap,nyap)         !
      integer :: nxbeam                 ! dimensions in beam map plane
      integer :: nybeam                 ! dimensions in beam map plane
    end subroutine fitres
  end interface
  !
  interface
    subroutine set_panels(type)
      !---------------------------------------------------------------------
      ! Set panels info according to antenna type
      !---------------------------------------------------------------------
      integer :: type                   !
    end subroutine set_panels
  end interface
  !
  interface
    subroutine get_rigging(elevation, riggingElevation, correction, error)
      !---------------------------------------------------------------------
      !
      !---------------------------------------------------------------------
      include 'clic_panels.inc'
      real :: elevation, riggingElevation(2), correction(mring, mpan, mm)
      logical error
    end subroutine get_rigging
  end interface
  !
  interface
    subroutine rigging_mode(phas,nxap,xref,xval,xinc,rlam,   &
         elevation, riggingElevation, error)
      include 'clic_panels.inc'
      logical :: error
      integer :: nxap                   !
      real :: phas(nxap,nxap)           !
      real, allocatable :: correction(:,:,:)
      real :: xref                      !
      real :: xval                      !
      real :: xinc                      !
      real :: rlam, xx, yy, fmode                      !
      real :: am(mm,mring,mpan)         !
      real :: ae(mm,mring,mpan)         !
      real :: elevation, riggingElevation(2)             !  
    end subroutine rigging_mode
  end interface
  !
  interface
    subroutine temperature_mode(phTemp, nxap, xref, xval, xinc, rlam,   &
         temperature, biasTemperature, error)
      ! should we fit panbel shapes in the modeled temp deformation or use it
      !  directly...
      ! Global
      include 'gbl_pi.inc'
      include 'clic_panels.inc'
      logical :: error
      integer :: nxap                   !
      real :: phTemp(nxap,nxap)           !
      real :: xref, xval, xinc                      !
      real :: rlam, temperature, biasTemperature
    end subroutine temperature_mode
  end interface
  !
  interface
    subroutine zernike_poly ( m, n, rho, z )
      !*****************************************************************************80
      !
      !! ZERNIKE_POLY evaluates a Zernike polynomial at RHO.
      !
      !  Discussion:
      !
      !    This routine uses the facts that:
      !
      !    *) R^M_N = 0 if M < 0, or N < 0, or N < M.
      !    *) R^M_M = RHO^M
      !    *) R^M_N = 0 if mod ( N - M ) = 1.
      !
      !    and the recursion:
      !
      !    R^M_(N+2) = A * [ ( B * RHO**2 - C ) * R^M_N - D * R^M_(N-2) ]
      !
      !    where 
      !
      !    A = ( N + 2 ) / ( ( N + 2 )**2 - M**2 )
      !    B = 4 * ( N + 1 )
      !    C = ( N + M )**2 / N + ( N - M + 2 )**2 / ( N + 2 )
      !    D = ( N**2 - M**2 ) / N
      !
      !    I wish I could clean up the recursion in the code, but for
      !    now, I have to treat the case M = 0 specially.
      !
      !  Licensing:
      !
      !    This code is distributed under the GNU LGPL license. 
      !
      !  Modified:
      !
      !    11 November 2005
      !
      !  Author:
      !
      !    John Burkardt
      !
      !  Reference:
      !
      !    Eric Weisstein,
      !    CRC Concise Encyclopedia of Mathematics,
      !    CRC Press, 2002,
      !    Second edition,
      !    ISBN: 1584883472,
      !    LC: QA5.W45
      !
      !  Parameters:
      !
      !    Input, integer ( kind = 4 ) M, the upper index.
      !
      !    Input, integer ( kind = 4 ) N, the lower index.
      !
      !    Input, real ( kind = 8 ) RHO, the radial coordinate.
      !
      !    Output, real ( kind = 8 ) Z, the value of the Zernike 
      !    polynomial R^M_N at the point RHO.
      !
      ! 
    end subroutine zernike_poly
  end interface
  !
  interface
    subroutine solve_point(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for command
      !	SOLVE POINT [BEAM] [/PLOT] [/OUTPUT File [OLD|NEW|PROC]]
      !              [/COMPRESS TIME_MAX]
      !	[/Store /Reset /Offset /Search] Not relevant here
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_point
  end interface
  !
  interface
    subroutine check_point(error)
      !---------------------------------------------------------------------
      ! CLIC
      !	Check if data is a pointing scan
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine check_point
  end interface
  !
  interface
    subroutine load_point (data,   &
        icount,jaddr,naddr,error,lo1,nant,time_max)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE POINTING
      !	Measure antenna pointing position from a specialised
      !	routine in the real-time acquisition system
      ! Arguments:
      !     DATA      *      Data array
      !	ICOUNT	I	Subscan number 0 : not loaded
      !	JADDR	I	Array address
      !	NADDR	I	Array length
      !	LO1	R*4 	LO1 frequency
      !	NANT	I	Number of antennas
      !       TIME_MAX R*4    Max. integration time (compression)
      !---------------------------------------------------------------------
      real :: data(*)                        !
      integer :: icount                      !
      integer(kind=address_length) :: jaddr  !
      integer :: naddr                       !
      logical :: error                       !
      real :: lo1                            !
      integer :: nant                        !
      real :: time_max                       !
    end subroutine load_point
  end interface
  !
  interface
    subroutine sub_point (plot,out,iout,jaddr,naddr,qlun,   &
        error,beam,flu,end)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE POINTING/PLOT
      !	Measure antenna pointing position from a specialised
      !	routine in the real-time acquisition system
      ! Arguments:
      !	PLOT	L	Plot data and fit
      !	OUT	L	Write fit on output file
      !	IOUT	I	Output logical unit
      !	JADDR	I	Work space address
      !	NADDR	I	Work space length
      !	ERROR	L	Logical error flag
      !	BEAM	R*4	Beam size (radian)
      !     flu      L       Special format for flux determination.
      !     end      L       Means this is the last scan in the index
      !---------------------------------------------------------------------
      logical :: plot                        !
      logical :: out                         !
      integer :: iout                        !
      integer(kind=address_length) :: jaddr  !
      integer :: naddr                       !
      integer :: qlun                        !
      logical :: error                       !
      real :: beam                           !
      logical :: flu                         !
      logical :: end                         !
    end subroutine sub_point
  end interface
  !
  interface
    subroutine plot_point(nmob,imob,iant,icode,clea,chain,   &
        np,gain,posi,weig,limit)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE POINTING/PLOT
      ! 	Plot gain data
      ! Arguments
      !	NMOB	I 	Number of mobile antennas
      !	IMOB	I 	Antenna
      !	ICODE	I	1,2,3 = Focus, Az, El
      !	CLEA	L	Reset plot
      !	CHAIN	C	Header
      !	NP	I 	Number of data points
      !	GAIN	R*4(NP)	Gain values
      !	POSI	R*4(NP) Position values
      !	WEIG 	R*4(NP) Weights
      !	CHAIN	C*(*)	X Axis Label
      !       LIMIT   C*(*)   Limits command
      !---------------------------------------------------------------------
      integer :: nmob                   !
      integer :: imob                   !
      integer :: iant                   !
      integer :: icode                  !
      logical :: clea                   !
      character(len=*) :: chain         !
      integer :: np                     !
      real :: gain(np)                  !
      real :: posi(np)                  !
      real :: weig(np)                  !
      character(len=*) :: limit         !
    end subroutine plot_point
  end interface
  !
  interface
    subroutine  pro_point(error)
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE POINTING/PLOT
      !	Plots fit result
      ! Arguments
      !	ERROR 	L 	Logical error flag
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine pro_point
  end interface
  !
  interface
    subroutine load_beam (beam, bsize, lo1, nant)
      use gildas_def
      real :: beam                      !
      real :: bsize                     !
      real :: lo1                       !
      integer :: nant                   !
    end subroutine load_beam
  end interface
  !
  interface
    subroutine check_scan(lun,iscan,measure)
      integer :: lun                    !
      integer :: iscan                  !
      character(len=*) :: measure       !
    end subroutine check_scan
  end interface
  !
  interface
    subroutine fit_point (npt,gain,posi,weig,pars,errs,beam,error)
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for SOLVE POINTING
      ! 	Determine pointing parameters from a drift
      ! Arguments:
      !	NPT	I	Number of points in drift
      !	GAIN	R*4(Np)	Antenna gains
      !	POSI	R*4(Np)	Antenna positions
      !	WEIG	R*4(Np)	Minimization weights
      !	PARS	R*4(3)	Fitted parameters
      !	ERRS	R*4(3)	Fitted errors
      !	BEAM	R*4	Beam size (Radians)
      !---------------------------------------------------------------------
      integer :: npt                    !
      real :: gain(npt)                 !
      real :: posi(npt)                 !
      real :: weig(npt)                 !
      real :: pars(3)                   !
      real :: errs(3)                   !
      real :: beam                      !
      logical :: error                  !
    end subroutine fit_point
  end interface
  !
  interface
    subroutine fill_visi_posi (qdump,qbas,qant,visi,posi,weig,   &
        time_max,data,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Read the data section and extract the visibilities for all baselines
      ! and the positions of all antennas
      ! Arguments:
      !	R 	Struc.	Header
      !	VISI	R*4(*)	Visibilities to be returned
      !	POSI	R*4(*)	"Position" of antenna. The "position"
      !			is actually a parameter defined by R.SCATY
      !	WEIG	R*4(*)	Weights to be returned
      !---------------------------------------------------------------------
      integer :: qdump                  !
      integer :: qbas                   !
      integer :: qant                   !
      real :: visi(qdump,qant)          !
      real :: posi(qdump,qant)          !
      real :: weig(qdump,qant)          !
      real :: time_max                  !
      real :: data(*)                   !
      logical :: error                  !
    end subroutine fill_visi_posi
  end interface
  !
  interface
          subroutine refrac(refraction)
    !----------------------------------------------------------------------
    !       cette subroutine calcule les coefficients de la refraction
    !       atmospherique (copiee de obs.)
    !
    !       les parametres d'entree sont :
    !
    !               tambia  = temperature ambiante en k ( 223 < tambia <  323   )
    !               humidi  = humidite relative (%)     (   0 < humidi <  100   )
    !               pambia  = pression in mbar          ( 690 < pambia <  790   )
    !
    !       les parametres de sortie sont :
    !               refrc1,refrc2,refrc3
    !               la formule utilisee est
    !               refr = refrc1/tan(el.)+refrc2/tan(el.)**3+refrc3/tan(el.)**5
    !
    !       pour la refraction radio on utilise la formule suivante
    !       waters, j.w. 1976, in methods of experimental physics
    !       vol 12, part b, pag. 186.
    !
    !       n = (77.6 pambia/tambia+ 3.73e05 e/tambia**2)*10e-06 radians
    !
    !       ou e est la pression partielle de la vapeur d'eau donnee par
    !       e=es*humidi/100./(1-(1-humidi/100.)*es/pambia) mbar
    !       ou es est la pression de saturation de la vapeur d'eau donnee par
    !       es=6.105*exp(25.22*(tambia-273)/tambia-5.31*ln(tambia/273)) mbar
    !
    !       par example, pambia=1013, tambia=273, humidi=90
    !       es=6.105 mbar
    !       e=5.491 mbar
    !       n=59.39" + 5.67" (contribution de la vapeur d'eau)
    !---------------------------------------------------------------------- 
          real refraction(3)
        end subroutine refrac
  end interface
  !
  interface
    subroutine solve_skydip(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for command
      !	SOLVE SKYDIP [TREC|EFF] [/PLOT] [/OUTPUT File [NEW]]
      !	[/Store /Reset /Offset /Closure /Search] Not relevant here
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_skydip
  end interface
  !
  interface
    subroutine load_auto (data, auto_data, iinput, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC 	read a data record for SKYDIP
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      real :: data(*)                   !
      real :: auto_data(mnant)          !
      integer :: iinput                 !
      logical :: error                  !
    end subroutine load_auto
  end interface
  !
  interface
    subroutine sub_skydip (plot,out,iout,nel,c_sky,c_el,iinput,error)
      !---------------------------------------------------------------------
      ! Call Tree
      !	SOLVE_SKYDIP
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      logical :: plot                   !
      logical :: out                    !
      integer :: iout                   !
      integer :: nel                    !
      real :: c_sky(mnant,nel)          !
      real :: c_el(nel)                 !
      integer :: iinput
      logical :: error                  !
    end subroutine sub_skydip
  end interface
  !
  interface
    subroutine plot_skydip(iant,trec,h2o,f_eff)
      integer :: iant                   !
      real :: trec                      !
      real :: h2o                       !
      real :: f_eff                     !
    end subroutine plot_skydip
  end interface
  !
  interface
    subroutine fitsky (fcn,liter,ier,meancho)
      !---------------------------------------------------------------------
      ! RED	Internal routine
      !	Setup and starts a SKYDIP fit minimisation using MINUIT
      ! Arguments :
      !	FCN	Ext.	Function to be mininized	Input
      !	LITER	L	Logical of iterate fit		Input
      !	IER	L	Logical error flag		Output
      ! Created	S.Guilloteau 	5-Nov-1986
      !---------------------------------------------------------------------
      external :: fcn                   !
      logical :: liter                  !
      integer :: ier                    !
      real :: meancho                   !
    end subroutine fitsky
  end interface
  !
  interface
    subroutine midsky (ier,liter)
      !---------------------------------------------------------------------
      ! RED	Internal routine
      !	Start a SKYDIP fit by building the PAR array and internal
      !	variable used by Minuit.
      ! 	Parameters are
      !	PAR(1)	= RFOEFF	Forward efficiency
      !	or	  RTREC		Receiver temperature
      !	PAR(2)	= RH2OMM	Water vapor amount (mm)
      !
      ! Arguments
      !	IER	I	Error code 			Output
      !	LITER	L	Iterate a fit			Input
      !
      ! S.Guilloteau 16-Feb-1987
      !---------------------------------------------------------------------
      integer :: ier                    !
      logical :: liter                  !
    end subroutine midsky
  end interface
  !
  interface
    subroutine minsky (npar,g,f,x,iflag)
      !---------------------------------------------------------------------
      ! RED	Internal routine
      !	Function to be minimized in the SKYDIP.
      ! Arguments :
      !	NPAR	I	Number of parameters		Input (2)
      !	G	R*8(1)	Array of derivatives		Output
      !	F	R*8	Function value			Output
      !	X	R*8(1)	Parameter values		Input
      !	IFLAG	I	Code operation			Input
      ! (5-mar-1985)
      !---------------------------------------------------------------------
      integer :: npar                   !
      real*8 :: g(*)                    !
      real*8 :: f                       !
      real*8 :: x(*)                    !
      integer :: iflag                  !
    end subroutine minsky
  end interface
  !
  interface
    subroutine skydip_boxes(n_boxes,c_setbox,error)
      integer :: n_boxes                     !
      character(len=*) :: c_setbox(n_boxes)  !
      logical :: error                       !
    end subroutine skydip_boxes
  end interface
  !
  interface
    subroutine solve_corr(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Interface routine for SOLVE corr/SEARCH
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_corr
  end interface
  !
  interface
    subroutine solve_corr_sub (nd,   &
        x_iatm, x_jatm, x_time,   &
        y_pha, boff, bmes, emes, sig, a, b, error)
      use gildas_def
      !---------------------------------------------------------------------
      !     ND        INTEGER    Number of points
      !     X_IF(ND)  REAL       I_F frequency of points
      !     Y_PHA(ND) REAL       Phases of points
      !     BOFF      REAL       Original delay offset
      !     BMES(2)   REAL       Returned delay (?) or phase range (?)
      !     SIG       REAL       Rms error on phase (?) or delay (?)
      !     A(ND,MVAR)   REAL*8  Work space for minimization routine
      !                          on return, covariance matrix ?
      !     B(ND)        REAL*8  Ibid
      !     ERROR     LOGICAL    Error flag
      !---------------------------------------------------------------------
      integer :: nd                     !
      real :: x_iatm(nd)                !
      real :: x_jatm(nd)                !
      real :: x_time(nd)                !
      real :: y_pha(nd)                 !
      real :: boff(2)                   !
      real :: bmes(2)                   !
      real :: emes(2)                   !
      real :: sig                       !
      real*8 :: a(nd,*)                 !
      real*8 :: b(nd)                   !
      logical :: error                  !
    end subroutine solve_corr_sub
  end interface
  !
  interface
    subroutine sort_index (error)
      logical :: error                  !
    end subroutine sort_index
  end interface
  !
  interface
    function gtt_i  (m,l)
      logical :: gtt_i                  !
      integer :: m                      !
      integer :: l                      !
    end function gtt_i
  end interface
  !
  interface
    function gte_i (m,l)
      logical :: gte_i                  !
      integer :: m                      !
      integer :: l                      !
    end function gte_i
  end interface
  !
  interface
    subroutine tri_index (x,n,gtt,gte,error)
      !---------------------------------------------------------------------
      ! 	Sorting program that uses a quicksort algorithm.
      !	Applies for an input array of integer values, which are SORTED
      !
      !	X	I(*) 	Indexes					Input
      !	N	I	Length of arrays			Input
      !	*	Return	Error return
      !---------------------------------------------------------------------
      integer :: x(*)                   !
      integer*4 :: n                    !
      external :: gtt                   !
      logical :: gtt                    !
      external :: gte                   !
      logical :: gte                    !
      logical :: error                  !
    end subroutine tri_index
  end interface
  !
  interface
    subroutine sort_fluxes(error)
      logical :: error                  !
    end subroutine sort_fluxes
  end interface
  !
  interface
    function gtt_flux (m,l)
      !---------------------------------------------------------------------
      ! Flux record comparison (1-source then 2-freq then 3-date)
      !---------------------------------------------------------------------
      logical :: gtt_flux               !
      integer :: m                      !
      integer :: l                      !
    end function gtt_flux
  end interface
  !
  interface
    function gte_flux (m,l)
      !---------------------------------------------------------------------
      ! Flux record comparison (1-source then 2-freq then 3-date)
      !---------------------------------------------------------------------
      logical :: gte_flux               !
      integer :: m                      !
      integer :: l                      !
    end function gte_flux
  end interface
  !
  interface
    subroutine store_cal(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores a calibration curve, in the source observations,
      !	assumed to be in the current index.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_cal
  end interface
  !
  interface
    subroutine get_cal(iy,ir,isb,ib,ispline,ipol,pola,x,y,oscan,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Returns in Y the value and derivatives of the calibration curve
      ! at time X,
      ! for baseline IB, side band ISB.
      ! IX = 1 for amplitude and 2 for phase
      !---------------------------------------------------------------------
      integer :: iy                     !
      integer :: ir                     !
      integer :: isb                    !
      integer :: ib                     !
      integer :: ispline                !
      integer :: ipol                   !
      integer :: pola                   !
      real :: x                         !
      real :: y(4)                      !
      integer :: oscan                  !
      logical :: error                  !
    end subroutine get_cal
  end interface
  !
  interface
    subroutine band_factor(ib,isb,ir,factor,error)
      !---------------------------------------------------------------------
      ! Compute the phase conversion factor (Complex) from band ISB to
      ! Average band, for receiver IR, baseline IB (or antenna -IB)
      ! _ if IR is the current one (IR=R_NREC) then one takes the bandpass data
      !   of the observation;
      ! _ if not, then the bandpass for receiver IR must be in program memory,
      !   or an error occurs.
      !---------------------------------------------------------------------
      integer :: ib                     !
      integer :: isb                    !
      integer :: ir                     !
      real :: factor                    !
      logical :: error                  !
    end subroutine band_factor
  end interface
  !
  interface
    subroutine get_self_cal(krec,iy1,iy2,ibmin,ibmax,jsb,y2,w2,   &
        error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Get self calibration values (from current receiver or receiver krec)
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      integer :: krec                   !
      integer :: iy1                    !
      integer :: iy2                    !
      integer :: ibmin(2)               !
      integer :: ibmax(2)               !
      integer :: jsb                    !
      real :: y2(2,2,-mnant:mnbas)      !
      real :: w2(2,2,-mnant:mnbas)      !
      logical :: error                  !
    end subroutine get_self_cal
  end interface
  !
  interface
    subroutine store_flag(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores flags for current index
      ! CLIC	Command CLIC\STORE FLAG f1 f2 ... [/ANTENNA a1 a2 ... ]
      !		[/BASELINE b1 b2 ...] [/RESET]
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_flag
  end interface
  !
  interface
    subroutine doflags(nant,nbas,af,bf,afl,bfl,reset)
      integer :: nant                   !
      integer :: nbas                   !
      integer :: af(nant)               !
      integer :: bf(nbas)               !
      integer :: afl(nant)              !
      integer :: bfl(nbas)              !
      logical :: reset                  !
    end subroutine doflags
  end interface
  !
  interface
    subroutine store_quality(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores data quality for all scans in current index
      ! CLIC	Command CLIC\STORE QUALITY quality
      ! where quality is a keyword on an integer in range 0-9
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_quality
  end interface
  !
  interface
    subroutine get_quality(argum, iqual, qual, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! decode quality value from argument argum
      !---------------------------------------------------------------------
      character(len=*) :: argum         !
      integer :: iqual                  !
      character(len=*) :: qual          !
      logical :: error                  !
    end subroutine get_quality
  end interface
  !
  interface
    subroutine store_correction(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores validity of radiometric phase correction
      ! CLIC	Command CLIC\STORE CORRECTION GOOD|BAD|AUTO|SELF
      !      GOOD or BAD : force to value
      !      AUTO : use calibrator amplitude and spread result to neighbouring
      !             source observations
      !      SELF : use source amplitude, for all observations
      !      (will produce random result if no continuum, useful for point sources)
      !      /RECEIVER ir
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_correction
  end interface
  !
  interface
    subroutine check_correction(na, ia, ncal, source, tmin, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! for phase calibrators, compute the recorrelation factor due to
      ! the phase correction
      ! for other sources,interpolate between nearest calibrators.
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      integer :: na                     !
      integer :: ia(mnant)              !
      integer :: ncal                   !
      logical :: source                 !
      real*8 :: tmin                    !
      logical :: error                  !
    end subroutine check_correction
  end interface
  !
  interface
    subroutine store_flux(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores internal fluxes and efficiencies
      !	into observations from current index
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_flux
  end interface
  !
  interface
    subroutine store_pass(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores a passband calibration in the source observations,
      !	assumed to be in the current index.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_pass
  end interface
  !
  interface
    subroutine sub_store_pass_ant(first,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores a passband calibration in the observations currently in
      !	memory
      !       Antenna based mode.
      !---------------------------------------------------------------------
      logical :: first                  !
      logical :: error                  !
    end subroutine sub_store_pass_ant
  end interface
  !
  interface
    subroutine sub_store_pass(first,error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Stores a passband calibration in the observations currently in
      !	memory
      !---------------------------------------------------------------------
      logical :: first                  !
      logical :: error                  !
    end subroutine sub_store_pass
  end interface
  !
  interface
    subroutine store_monitor(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_monitor
  end interface
  !
  interface
    subroutine get_mon(ia,ispline,ipol,x,y,oscan,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Returns in Y the value and derivatives of the calibration curve
      ! at time X,
      ! for baseline IB, side band ISB.
      !---------------------------------------------------------------------
      integer :: ia                     !
      integer :: ispline                !
      integer :: ipol                   !
      real :: x                         !
      real :: y                         !
      integer :: oscan                  !
      logical :: error                  !
    end subroutine get_mon
  end interface
  !
  interface
    subroutine set_display (error)
      use gildas_def
      !---------------------------------------------------------------------
      !	Set up the display page according to previous SET commands
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_display
  end interface
  !
  interface
    subroutine position_boxes(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Position the boxes according to the Greg plot page if it has changed
      ! or if the number of boxes, or aspect ratio has changed.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine position_boxes
  end interface
  !
  interface
    subroutine switch_spectral
      use gildas_def
      ! Global
    end subroutine switch_spectral
  end interface
  !
  interface
    subroutine switch_time
      use gildas_def
      ! Global
    end subroutine switch_time
  end interface
  !
  interface
    subroutine set_format(l,h,error)
      use gildas_def
      character(len=*) :: l             !
      character(len=*) :: h             !
      logical :: error                  !
    end subroutine set_format
  end interface
  !
  interface
    subroutine setdef (line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC	Internal routine
      !	Setup default parameters
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine setdef
  end interface
  !
  interface
    subroutine set_data(np, nb)
      use gildas_def
      integer :: np                     !
      integer :: nb                     !
    end subroutine set_data
  end interface
  !
  interface
    subroutine coffse(line,offset,error)
      !---------------------------------------------------------------------
      ! SAS	Internal routine
      !	Converts the formatted offset angle value in current units to
      !	internal value.
      ! Arguments :
      !	LINE	C*(*)	Formatted value			Input
      !	OFFSET	R*4	Offset value in radian		Output
      !	ERROR	L	Logical error flag		Output
      ! (5-mar-1985)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      real*4 :: offset                  !
      logical :: error                  !
    end subroutine coffse
  end interface
  !
  interface
    subroutine setfor(line,error)
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine setfor
  end interface
  !
  interface
    subroutine reset_phases
      use gildas_def
      !---------------------------------------------------------------------
      ! Process phases in plot buffer so that they are between -pi and pi,
      ! and optionnally insure continuity for plot.
      !---------------------------------------------------------------------
      ! Global
    end subroutine reset_phases
  end interface
  !
  interface
    subroutine prophase(x,p,n,w,degrees,continuous,nc)
      real*4 :: x(*)                    !
      real*4 :: p(*)                    !
      integer :: n                      !
      real*4 :: w(*)                    !
      logical :: degrees                !
      logical :: continuous             !
      integer :: nc                     !
    end subroutine prophase
  end interface
  !
  interface
    subroutine cvphase(n,p,w,degrees,old_degrees)
      !---------------------------------------------------------------------
      ! Convert phases between degrees and radians
      !---------------------------------------------------------------------
      integer :: n                      !
      real*4 :: p(n)                    !
      real*4 :: w(n)                    !
      logical :: degrees                !
      logical :: old_degrees            !
    end subroutine cvphase
  end interface
  !
  interface
    subroutine switch_antenna
      use gildas_def
      ! Global
    end subroutine switch_antenna
  end interface
  !
  interface
    subroutine switch_baseline
      use gildas_def
      ! Global
    end subroutine switch_baseline
  end interface
  !
  interface
    subroutine set_all_baselines
      use gildas_def
      !---------------------------------------------------------------------
      ! take all currect baselines
      !---------------------------------------------------------------------
      ! Global
    end subroutine set_all_baselines
  end interface
  !
  interface
    subroutine set_all_subbands
      use gildas_def
      !---------------------------------------------------------------------
      ! take all current subbands
      !---------------------------------------------------------------------
      ! Global
    end subroutine set_all_subbands
  end interface
  !
  interface
    subroutine set_each_subband
      use gildas_def
      !---------------------------------------------------------------------
      ! take all current subbands
      !---------------------------------------------------------------------
      ! Global
    end subroutine set_each_subband
  end interface
  !
  interface
    subroutine set_pol_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! select current subbands according to their polarization
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_pol_subbands
  end interface
  !
  interface
    subroutine set_each_pol_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! select current subbands according to their polarization
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_each_pol_subbands
  end interface
  !
  interface
    subroutine set_quar_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! select current subbands according to the quarter they are connected to
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_quar_subbands
  end interface
  !
  interface
    subroutine set_nbc_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! select current subbands according to the Narrow Band Correlator entry
      ! they are connected to
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_nbc_subbands
  end interface
  !
  interface
    subroutine set_widex_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! select a widex subband
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine set_widex_subbands
  end interface
  !
  interface
    subroutine check_pol_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! check the polarization of all subbands and update DO_POLAR accordingly
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine check_pol_subbands
  end interface
  !
  interface
    subroutine check_nbc_subbands(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! check the NBC entry of all subbands and update NARROW_INPUT accordingly
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine check_nbc_subbands
  end interface
  !
  interface
    function quarter(flo2,flo2bis)
      !---------------------------------------------------------------------
      ! This function returns the quarter to which an unit is connected
      ! It depends on the values of FLO2 and FLO2BIS
      !---------------------------------------------------------------------
      integer :: quarter                !
      real*8 :: flo2                    !
      real*8 :: flo2bis                 !
    end function quarter
  end interface
  !
  interface
    function nbc_entry(lunit,punit,flo2,flo2bis,lpolmode,lpolentry)
      use gildas_def
      !---------------------------------------------------------------------
      ! This function returns the correlator entry to which an unit is
      ! connected (for narrow-band correlator) or the "logical" widex unit 
      ! number. 
      !
      ! First correlator entry can only be  Q1H, Q2H, Q3V, or Q4V
      ! Second correlator entry can only be Q3H, Q4H, Q1V, or Q2V
      !---------------------------------------------------------------------
      integer :: nbc_entry              !
      integer :: lunit                  !
      integer :: punit                  !
      real*8 :: flo2                    !
      real*8 :: flo2bis                 !
      integer :: lpolmode               !
      integer :: lpolentry              !
    end function nbc_entry
  end interface
  !
  interface
    subroutine set_numbers
      use gildas_def
      !---------------------------------------------------------------------
      ! sets up numbering system for baselines, triangles, quadrangles.
      !---------------------------------------------------------------------
      ! Global
    end subroutine set_numbers
  end interface
  !
  interface
    subroutine mesname (file)
      character(len=*) :: file          !
    end subroutine mesname
  end interface
  !
  interface
    subroutine messios (prio,seve,name,ier)
      !---------------------------------------------------------------------
      ! Output system message corresponding to IO error code IER, and
      ! using the MESSAGE routine, using the portable GAG_IOSTAT routine
      !---------------------------------------------------------------------
      integer :: prio                   !
      integer :: seve                   !
      character(len=*) :: name          !
      integer :: ier                    !
    end subroutine messios
  end interface
  !
  interface
    function get_file_type ( fspec, ftype, ftype_len, new)
      integer :: get_file_type          !
      character(len=*) :: fspec         ! file spec to parse
      character(len=*) :: ftype         ! type field
      integer*4 :: ftype_len            ! length of type string
      logical :: new                    !
    end function get_file_type
  end interface
  !
  interface
    function get_file_name ( fspec, fname, fname_len)
      integer :: get_file_name          !
      character(len=*) :: fspec         ! file spec to parse
      character(len=*) :: fname         ! type field
      integer*4 :: fname_len            ! length of type string
    end function get_file_name
  end interface
  !
  interface
    function no_number(f)
      !---------------------------------------------------------------------
      !  Return codes of fpclassifyf:
      !
      !  FP_PLUS_NORM      0	/* Positive normalized */
      !  FP_MINUS_NORM     1	/* Negative normalized */
      !  FP_PLUS_ZERO      2	/* Positive zero */
      !  FP_MINUS_ZERO     3	/* Negative zero */
      !  FP_PLUS_INF	     4	/* Positive infinity */
      !  FP_MINUS_INF      5	/* Negative infinity */
      !  FP_PLUS_DENORM    6	/* Positive denormalized */
      !  FP_MINUS_DENORM   7	/* Negative denormalized */
      !  FP_SNAN	     8	/* Signalling NaN */
      !  FP_QNAN	     9	/* Quiet NaN */
      ! shanged to use gag_isreal
      !---------------------------------------------------------------------
      logical :: no_number              !
      real :: f                         !
    end function no_number
  end interface
  !
  interface
    subroutine clic_directory(line,error)
      !---------------------------------------------------------------------
      ! INPUT
      !     line       char    command line
      !     error      LOGICAL error parameter
      ! OUTPUT
      !     define  DIR_NFILE and DIR_FILE variables
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_directory
  end interface
  !
  interface
    subroutine title(long)
      use gildas_def
      !---------------------------------------------------------------------
      ! SAS	Internal routine
      !	draw a header above the box
      ! Arguments
      !	LONG	C*(*)	Header type (Long, Brief or Full)
      !---------------------------------------------------------------------
      character(len=*) :: long          !
    end subroutine title
  end interface
  !
  interface
    subroutine titout(type,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC	Internal routine
      !	Write a title
      ! Arguments :
      !	TYPE	C*(*)	Type of title to be displayed:
      !
      !      FULL = all informations - used by HEADER command
      !      LONG = part/all informations, depending on SET FORMAT  - used by
      !             LIST /LONG command
      !      SHORT/ZHORT = first/last scans on same line - used by LIST /SHORT
      !             command; NB: SHORT = first pass, ZHORT = 2nd pass
      !      NORMAL = summary information on one line - used by LIST, PLOT,
      !             CURSOR commands
      !      PLOT/BLOT = special for PLOT command (NGRx); NB: PLOT = first
      !             pass, BLOT = 2nd pass
      !---------------------------------------------------------------------
      character(len=*) :: type          !
      logical :: error                  !
    end subroutine titout
  end interface
  !
  interface
    function procname (nproc)
      character(len=4) :: procname      !
      integer :: nproc                  !
    end function procname
  end interface
  !
  interface
    subroutine vartitout(short,scan,nv,type,addr,form,end,error)
      use gildas_def
      use gbl_format
      logical :: short                          !
      integer :: scan                           !
      integer :: nv                             !
      integer(kind=address_length) :: type(nv)  !
      integer(kind=address_length) :: addr(nv)  !
      character(len=12) :: form(nv)             !
      logical :: end                            !
      logical :: error                          !
    end subroutine vartitout
  end interface
  !
  interface
    subroutine iform(iv,ch,lch,form,error)
      integer :: iv                     !
      character(len=*) :: ch            !
      integer :: lch                    !
      character(len=12) :: form         !
      logical :: error                  !
    end subroutine iform
  end interface
  !
  interface
    subroutine decode_backend(narrow)
      character(len=36) :: narrow(2)    !
    end subroutine decode_backend
  end interface
  !
  interface
    subroutine zero_index
      !--------------------------------------------------------------
      ! Zeroes the current index
      !--------------------------------------------------------------
      ! Global
    end subroutine zero_index
  end interface
  !
  interface
    function header_data(ix)
      logical :: header_data            !
      integer :: ix                     !
    end function header_data
  end interface
  !
  interface
    function angle_data(xy)
      logical :: angle_data             !
      integer :: xy                     !
    end function angle_data
  end interface
  !
  interface
    function corr_data(x,y)
      logical :: corr_data              !
      integer :: x                      !
      integer :: y                      !
    end function corr_data
  end interface
  !
  interface
    function auto_data(x,y)
      logical :: auto_data              !
      integer :: x                      !
      integer :: y                      !
    end function auto_data
  end interface
  !
  interface
    function ante_data(x)
      logical :: ante_data              !
      integer :: x                      !
    end function ante_data
  end interface
  !
  interface
    function base_data(x)
      logical :: base_data              !
      integer :: x                      !
    end function base_data
  end interface
  !
  interface
    subroutine polyant(iy, m, nbas, iant, jant, iref,   &
        kplus1, nant, x, y, w,   &
        wk1, wk2, wk3, ss, wss, c, error)
      !---------------------------------------------------------------------
      !     polyant computes a weighted least-squares polynimial approximation
      !     to the antenna amplitudes or phases
      !     for an arbitrary set of baseline data points.
      ! parameters:
      !     iy             I   Input    1 for log(amplitude), 2 for phase
      !     m              I   Input    the number of data points for each baseline
      !     nbas           I   Input    the number of baselines
      !     iant(nbas)     I   Input    start antenna for each baseline
      !     jant(nbas)     I   Input    end  antenna for each baseline
      !     iref           i   Input    Reference antenna for phases
      !     kplus1         I   Input    degree of polynomials + 1
      !     nant           I   Input    the number of antennas
      !     x(m)           R8  Input    the data abscissae
      !     y(m,nbas)      R8  Input    the data values
      !     w(m,mbas)      R8  Input    weights
      !     wk1(kplus1)    R8  Output   work space
      !     wk2(kplus1**2*nant**2)
      !                    R8  Output   work space
      !     wk3(kplus1*nant)
      !                    R8  Output   work space
      !     ss(nbas)       R8  Output   rms of fit for each baseline
      !     c(nant,kplus1) R8  Output   the polynomial coefficients
      !---------------------------------------------------------------------
      integer :: iy                           !
      integer :: m                            !
      integer :: nbas                         !
      integer :: iant(nbas)                   !
      integer :: jant(nbas)                   !
      integer :: iref                         !
      integer :: kplus1                       !
      integer :: nant                         !
      real*8 :: x(m)                          !
      real*8 :: y(m,nbas)                     !
      real*8 :: w(m,nbas)                     !
      real*8 :: wk1(kplus1)                   !
      real*8 :: wk2(kplus1*nant,kplus1*nant)  !
      real*8 :: wk3(kplus1*nant)              !
      real*8 :: ss(nbas)                      !
      real*8 :: wss(nbas)                     !
      real*8 :: c(nant,kplus1)                !
      logical :: error                        !
    end subroutine polyant
  end interface
  !
  interface
    subroutine cheb(nplus1, xcap, p, error)
      !---------------------------------------------------------------------
      !  Compute nplus1 Chebishev Polynomials at x = xcap
      !---------------------------------------------------------------------
      integer :: nplus1                 !
      real*8 :: xcap                    !
      real*8 :: p(nplus1)               !
      logical :: error                  !
    end subroutine cheb
  end interface
  !
  interface
    subroutine splinant(iy, m, nbas, iant, jant, iref,   &
        ncap7, nant, x, y, w,   &
        k, wk1, wk2, wk3, ss, c, error)
      !---------------------------------------------------------------------
      !     splinant computes a weighted least-squares approximation
      !     to an arbitrary set of data points, either amplitude or phase.
      !     with knots prescribed by the user.
      ! parameters:
      !     iy            I   Input    1 for log(amplitude), 2 for phase
      !     m             I   Input    the number of data points for each baseline
      !     nbas          I   Input    the number of baselines
      !     iant(nbas)    I   Input    start antenna for each baseline
      !     jant(nbas)    I   Input    end  antenna for each baseline
      !     iref          i   Input    Reference antenna for phases
      !     ncap7         I   Input    number of knots for splines (see e02baf)
      !     nant          I   Input    the number of antennas
      !     X(m)          R8  Input    the data abscissae
      !     Y(m,nbas)     R8  Input    the data values
      !     W(m,mbas)     R8  Input    weights
      !     k(ncap7)      R8  Input    knots for the splines (inners + 4 at each end)
      !     wk1(4,m)    R8  Output   wk space
      !     wk2(4*nant,ncap7*nant)
      !                   R8  Output   work space
      !     wk3(ncap7*nant)
      !                   R8  Output   work space
      !     ss(nbas)      R8  Output   rms of fit for each baseline
      !     c(nant,ncap7) R8  Output   the spline coefficients (ncap3 values)
      !---------------------------------------------------------------------
      integer :: iy                     !
      integer :: m                      !
      integer :: nbas                   !
      integer :: iant(nbas)             !
      integer :: jant(nbas)             !
      integer :: iref                   !
      integer :: ncap7                  !
      integer :: nant                   !
      real*8 :: x(m)                    !
      real*8 :: y(m,nbas)               !
      real*8 :: w(m,nbas)               !
      real*8 :: k(ncap7)                !
      real*8 :: wk1(4,m)                !
      real*8 :: wk2(4*nant,ncap7*nant)  !
      real*8 :: wk3(ncap7*nant)         !
      real*8 :: ss(nbas)                !
      real*8 :: c(nant,ncap7)           !
      logical :: error                  !
    end subroutine splinant
  end interface
  !
  interface
    subroutine spline_check(m, ncap7, x, k, wk, error)
      !---------------------------------------------------------------------
      !     splinant computes a weighted least-squares approximation
      !     to an arbitrary set of data points by a cubic spline
      !     with knots prescribed by the user.
      !---------------------------------------------------------------------
      integer :: m                      !
      integer :: ncap7                  !
      real*8 :: x(m)                    !
      real*8 :: k(ncap7)                !
      real*8 :: wk(m)                   !
      logical :: error                  !
    end subroutine spline_check
  end interface
  !
  interface
    function zant(i,r)
      integer :: zant                   !
      integer :: i                      !
      integer :: r                      !
    end function zant
  end interface
  !
  interface
    subroutine clic_monitor(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command MONITOR
      ! It processes CAL scans as encountered, to deduce the atmospheric
      ! parameters from the monitor at 230 GHz (receiver 2, in fact).
      ! Monitor time 0|1
      ! time defaults to 1 h ; 0 means each interval is one scan
      ! 0 for constant, 1 for linear fit to the atm power data.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_monitor
  end interface
  !
  interface
    subroutine do_monitor (again,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! The chopper standard program is called for each antenna,
      ! in the mode selected in OBS.
      !---------------------------------------------------------------------
      logical :: again                  !
      logical :: error                  !
    end subroutine do_monitor
  end interface
  !
  interface
    subroutine chopper_mon (cmode, el, frs, fri, tamb, pamb, alti,   &
        feff, tchop, tcold, trec, gim, csky, cchop, ccold,   &
        tsysd, tem, h2o, tatms, tatmi, taus, taui,   &
        again, fro, path, dpath, error)
      !---------------------------------------------------------------------
      ! CLIC
      !	compute TCAL, TSYS, TREC, from the chop and atm values observed
      !	according to the usual chopper method (Ulich and Haas)
      !
      !	Double side band operation
      !
      !	_ If cmode is 'AU' and a cold load is available, or if cmode
      !	is 'TR', derive the sky emission TEMI from the receiver
      !	temperature TREC, and then H2O from the best model fitting.
      !	_ Else, if cmode is 'AU', the sky emission is computed from
      !	the given H2O,
      !       _ if cmode is 'MA', the sky emission is computed from the
      !       given opacities and atmospheric temperatures.
      !
      !  	Uses a curved atmosphere, equivalent height 5.5 KM
      !
      !	FRS, FRI   I	R Frequencies signal image in GHz (!!!)
      !       tamb       I    R Ambient temperature (K)
      !       pamb       I    R Ambient pressure (mbar)
      !       alti       I    R altitude in km
      !       FEFF       I    R Forward Eff
      !       BEFF       I    R Beam Eff
      !       TCHOP      I    R Ambient Load Temperature (K)
      !       TCOLD      I    R Cold  Load Temperature (K)
      !       GIM        I    R Image/Signal Gain ratio
      !       CCHOP      I    R Ambient Load Counts
      !       CSKY       I    R Sky Counts
      !       CCOLD      I    R Cold  Load Counts
      !	TREC       I/O  R Receiver dsb temperature (K)
      !	TSYSD 	   O    R DSB system temperature (K)
      !       H2O        O    R Water vapor content (mm)
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      include 'clic_clic.inc'
      !
      character(len=2) :: cmode         !
      real :: el                        !
      real :: frs                       !
      real :: fri                       !
      real :: tamb                      !
      real :: pamb                      !
      real :: alti                      !
      real :: feff                      !
      real :: tchop                     !
      real :: tcold                     !
      real :: trec                      !
      real :: gim                       !
      real :: csky                      !
      real :: cchop                     !
      real :: ccold                     !
      real :: tsysd                     !
      real :: tem                       !
      real :: h2o                       !
      real :: tatms                     !
      real :: tatmi                     !
      real :: taus                      !
      real :: taui                      !
      logical :: again                  !
      real :: fro                       !
      real :: path                      !
      real :: dpath                     !
      logical :: error                  !
    end subroutine chopper_mon
  end interface
  !
  interface
    subroutine feedcorrection(nx,ny,p,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Perform the Focal Feed phase correction
      ! The co-polar complex gain of the feed in direction theta, phi is:
      ! - Circular symmetry of the feed is assumed.
      ! G_co(th,ph)= sin^2(ph) G_E(th) + cos^2(ph) G_H(th)
      !              [Kilda, Foundations of Antennas - A unified approach, p 56]
      ! - G_E(th) and G_H(th) are either modelled or measured.
      !   and developped as:
      !   G_E(th) = sigma { ge(k) * cos(k*th) } k=1, nk
      !   G_H(th) = sigma { gh(k) * cos(k*th) } k=1, nk
      !
      !
      ! Input
      !     nx      integer          number of X points
      !     ny      integer          number of Y points
      !     p       real    nx,ny    raw phases
      ! Output
      !     p       real    nx,ny    corrected phases
      !---------------------------------------------------------------------
      integer :: nx                     !
      integer :: ny                     !
      real :: p(nx,ny)                  !
      logical :: error                  !
    end subroutine feedcorrection
  end interface
  !
  interface
    subroutine fft(x,kx,ky,y,nx,ny,w)
      integer :: kx                     !
      integer :: ky                     !
      complex :: x(kx,ky)               !
      integer :: nx                     !
      integer :: ny                     !
      complex :: y(nx,ny)               !
      complex :: w(nx,ny)               !
    end subroutine fft
  end interface
  !
  interface
    subroutine field_extend(x,n1,n2,y,m1,m2)
      !---------------------------------------------------------------------
      ! extend input map into output map by zeroes
      !---------------------------------------------------------------------
      integer :: n1                     !
      integer :: n2                     !
      complex :: x(n1,n2)               !
      integer :: m1                     !
      integer :: m2                     !
      complex :: y(m1,m2)               !
    end subroutine field_extend
  end interface
  !
  interface
    subroutine field_extract(x,n1,n2,y,m1,m2)
      !---------------------------------------------------------------------
      ! extract input map from extended map
      !---------------------------------------------------------------------
      integer :: n1                     !
      integer :: n2                     !
      complex :: x(n1,n2)               !
      integer :: m1                     !
      integer :: m2                     !
      complex :: y(m1,m2)               !
    end subroutine field_extract
  end interface
  !
  interface
    subroutine do_apodize(x,n1,n2)
      !---------------------------------------------------------------------
      ! extend input map into output map by zeroes
      !---------------------------------------------------------------------
      integer :: n1                     !
      integer :: n2                     !
      complex :: x(n1,n2)               !
    end subroutine do_apodize
  end interface
  !
  interface
    subroutine do_test1(x,n1,n2)
      !---------------------------------------------------------------------
      ! extend input map into output map by zeroes
      !---------------------------------------------------------------------
      integer :: n1                     !
      integer :: n2                     !
      complex :: x(n1,n2)               !
    end subroutine do_test1
  end interface
  !
  interface
    subroutine field_to_aperture(n1,n2,x,dx,y,dy,w,distance,fresnel,   &
        dfocus,cheat)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Include non fresnel terms.
      !     X: input field map,
      !     Y: input aperture  map,
      !     W: work space (two complex planes)
      !     d: distance of field
      !     fresnel: use fresnel approximation (otherwise use add. terms)
      !---------------------------------------------------------------------
      integer :: n1                     !
      integer :: n2                     !
      complex :: x(n1,n2)               !
      real *8 :: dx                     !
      complex :: y(n1,n2)               !
      real *8 :: dy                     !
      complex :: w(n1,n2,2)             !
      real :: distance                  !
      logical :: fresnel                !
      real :: dfocus                    !
      complex :: cheat                  !
    end subroutine field_to_aperture
  end interface
  !
  interface
    subroutine corr_aperture(n1, n2,   &
        amp, pha, aperture)
      integer :: n1                     !
      integer :: n2                     !
      real :: amp(n1, n2)               !
      real :: pha(n1, n2)               !
      complex :: aperture(n1, n2)       !
    end subroutine corr_aperture
  end interface
  !
  interface
    subroutine corr_aperture_perfect(n1, n2,   &
        amp, pha, aperture)
      integer :: n1                     !
      integer :: n2                     !
      real :: amp(n1, n2)               !
      real :: pha(n1, n2)               !
      complex :: aperture(n1, n2)       !
    end subroutine corr_aperture_perfect
  end interface
  !
  interface
    subroutine aperture_to_field(n1,n2,x,dx,y,dy,w,distance,fresnel,   &
        dfocus)
      !---------------------------------------------------------------------
      ! Include non fresnel terms.
      !     X: output field map
      !     Y: input  aperture map.
      !     d: distance of field
      !     fresnel: use fresnel approximation (otherwise use add. terms)
      !     DIAMETER       antenna diameter
      !     Focus          focus distance
      !     DFocus         focus offset
      !---------------------------------------------------------------------
      integer :: n1                     !
      integer :: n2                     !
      complex :: x(n1,n2)               !
      real*8 :: dx                      !
      complex :: y(n1,n2)               !
      real*8 :: dy                      !
      complex :: w(n1,n2,2)             !
      real :: distance                  !
      logical :: fresnel                !
      real :: dfocus                    !
    end subroutine aperture_to_field
  end interface
  !
  interface
    subroutine phases(x,nx,ny,y,npm,pm,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Compute the phase aperture distribution.
      !     X     Input Complex array (X slot)
      !     NX    Input complex array X dimension
      !     NY    Input complex array Y dimension
      !     Y     output array (NX,NY,2) :
      !           in 1 = amplitudes in dB (max 0dB)
      !           in 2 = phases in radians
      !     NPM   number of masked panels
      !     PM    masked panels
      !     ERROR error return
      !---------------------------------------------------------------------
      integer :: nx                     !
      integer :: ny                     !
      complex :: x(nx,ny)               !
      real :: y(nx,ny,2)                !
      integer :: npm                    !
      character(len=5) :: pm(200)       !
      logical :: error                  !
    end subroutine phases
  end interface
  !
  interface
    function mask(x,y,d,npm,pm)
      !---------------------------------------------------------------------
      ! returns .true. if point (x,y) is within d of blockage area
      !---------------------------------------------------------------------
      logical :: mask                   !
      real :: x                         !
      real :: y                         !
      real :: d                         !
      integer :: npm                    !
      character(len=5) :: pm(npm)       !
    end function mask
  end interface
  !
  interface
    subroutine check_holo(iant, beta1, beta2, betam,   &
        lambda1, lambda2, lambdam, error)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !	Check if data is a Holo scan
      !       Check if demanded antenna was moved.
      !---------------------------------------------------------------------
      integer :: iant                   !
      real :: beta1                     !
      real :: beta2                     !
      real :: betam                     !
      real :: lambda1                   !
      real :: lambda2                   !
      real :: lambdam                   !
      logical :: error                  !
    end subroutine check_holo
  end interface
  !
  interface
    subroutine open_map(nx,ny,dx,dy,rx,ry,iband,iant,   &
        nxmap,nymap,distance,error)
      use gildas_def
      use gbl_format
      use clic_xypar
      !---------------------------------------------------------------------
      ! Open map GDF files
      ! input:
      !     NX, NY number of pixel on each axis for beam map
      !     DX, DY coordinate increments, in radians!
      !     RX, RY pixel numbers for center (0,0)
      !     IBAND  side band used 1=USB, 2=LSB, 3=DSB (for freq.)
      !     IANT   physical number of antenna
      !     NXMAP, NYMAP number of pixel on each axis for antenna surface map
      !     ERROR  error return
      !---------------------------------------------------------------------
      integer :: nx                     !
      integer :: ny                     !
      real :: dx                        !
      real :: dy                        !
      real :: rx                        !
      real :: ry                        !
      integer :: iband                  !
      integer :: iant                   !
      integer :: nxmap                  !
      integer :: nymap                  !
      real :: distance                  !
      logical :: error                  !
    end subroutine open_map
  end interface
  !
  interface
    subroutine create_aperture_map(nxmap,nymap,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Create an aperture map data file
      !---------------------------------------------------------------------
      integer :: nxmap                  !
      integer :: nymap                  !
      logical :: error                  !
    end subroutine create_aperture_map
  end interface
  !
  interface
    subroutine close_map(error)
      use gildas_def
      use clic_xypar
      logical :: error                  !
    end subroutine close_map
  end interface
  !
  interface
    subroutine plot_holo(iant,kw,sigu,sigw,rmsu,rmsw,eff,   &
        rms_ring,s_ring,z_ring,bmes,rmsast,taperx,tapery,x0, y0, h1, h2,   &
        basu,nbasu, distance, rigging, riggingElevation, & 
        tempBias, biasTemperature, temperature)
      use gildas_def
      include 'clic_parameter.inc'
      include 'clic_panels.inc'
      integer :: iant                   !
      real*8 :: kw                      !
      real :: sigu                      !
      real :: sigw                      !
      real :: rmsu                      !
      real :: rmsw, rmsast                      !
      real :: eff(8)                    !
      real :: rms_ring(mring)           !
      real :: s_ring(mring)             !
      real :: z_ring(mring)             !
      real*8 :: bmes(8)                 !
      real :: taperx                    !
      real :: tapery                    !
      real :: x0                        !
      real :: y0 , distance , riggingElevation(2), redeg(2), biasTemperature, temperature
      logical :: rigging, tempBias
      character(len=*) :: h1            !
      character(len=*) :: h2            !
      integer :: basu(mnbas)            !
      integer :: nbasu                  !
    end subroutine plot_holo
  end interface
  !
  interface
    subroutine plot_maps(v1,v2,nx,scale1,scale2)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Plot the two maps V1 and V2
      !---------------------------------------------------------------------
      integer :: nx                     !
      real :: v1(nx,nx)                 !
      real :: v2(nx,nx)                 !
      real :: scale1(3)                 !
      real :: scale2(3)                 !
    end subroutine plot_maps
  end interface
  !
  interface
    subroutine plot_ctrs(v1,cs1,v2,cs2,nx,scale1,scale2,beam,number   &
        ,iant,pen_col)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! Plot the contour maps of V1 and V2, as well as the Boxes and labels.
      !---------------------------------------------------------------------
      integer :: nx                     !
      real :: v1(nx,nx)                 !
      character(len=*) :: cs1           !
      real :: v2(nx,nx)                 !
      character(len=*) :: cs2           !
      real :: scale1(3)                 !
      real :: scale2(3)                 !
      real*8 :: beam                    !
      logical :: number                 !
      integer :: iant                   !
      integer :: pen_col                !
    end subroutine plot_ctrs
  end interface
  !
  interface
    subroutine draw_panels(beam,number,pen_col)
      real*8 :: beam                    !
      logical :: number                 !
      integer :: pen_col                !
    end subroutine draw_panels
  end interface
  !
  interface
    subroutine draw_new_bure_panels(iant)
      use gildas_def
      !---------------------------------------------------------------------
      ! draw darker the new panels set in clic_new_bure_panels.inc
      !---------------------------------------------------------------------
      integer :: iant                   !
    end subroutine draw_new_bure_panels
  end interface
  !
  interface
    subroutine sub_grid(nvis,vis,sd_beam,nd,thres,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      !     Cloned from SUB_GRID (author S.G.)
      !     Creation of a "well behaved" map from the spectra.
      !
      !     The "well behaved" map is obtained by linear  operations  (convolutions)
      !     from  the original spectra, in an optimum way from signal to noise point
      !     of view.
      !     nd = first dimension of VIS,
      !     x, y, and weights in columns 1 2 3,
      !     the remaining (nd-3) columns are data to be gridded.
      !
      !     thres is the weight threshold under which pixels are blanked
      !
      !     iscan is the column used for scanning (1 for X, 2 for Y)
      !---------------------------------------------------------------------
      integer :: nvis                   !
      integer :: nd                     !
      real :: vis(nd,nvis)              !
      real :: sd_beam                   !
      real :: thres                     !
      logical :: error                  !
    end subroutine sub_grid
  end interface
  !
  interface
    subroutine docoor (n,xref,xval,xinc,x)
      integer :: n                      !
      real*8 :: xref                    !
      real*8 :: xval                    !
      real*8 :: xinc                    !
      real :: x(n)                      !
    end subroutine docoor
  end interface
  !
  interface
    subroutine doconv (nd,np,visi,jx,jy,jo,we,gwe,   &
        nc,nx,ny,map,mapx,mapy,sup,cell,maxw)
      integer :: nd                     ! Number of "visibilities"
      integer :: np                     ! number of values
      real :: visi(nd,np)               ! values
      integer :: jx                     ! X coord, Y coord location in VISI
      integer :: jy                     ! X coord, Y coord location in VISI
      integer :: jo                     ! offset for data in VISI
      real :: we(np)                    ! Weights
      integer :: nx                     ! map size
      integer :: ny                     ! map size
      real :: gwe(nx,ny)                ! gridded weights
      integer :: nc                     ! number of channels
      real :: map(nc,nx,ny)             ! gridded values
      real :: mapx(nx)                  ! Coordinates of grid
      real :: mapy(ny)                  ! Coordinates of grid
      real :: sup(2)                    ! Supp.of convolving fct in User Units
      real :: cell(2)                   ! cell size in User Units
      real :: maxw                      ! Maximum weight
    end subroutine doconv
  end interface
  !
  interface
    subroutine dowei (visi,nd,np,we,iw)
      integer :: nd                     !
      integer :: np                     !
      real :: visi(nd,np)               !
      real :: we(np)                    !
      integer :: iw                     !
    end subroutine dowei
  end interface
  !
  interface
    subroutine findr (nv,nc,ic,xx,xlim,nlim)
      !---------------------------------------------------------------------
      ! GILDAS	Internal routine
      !	Find NLIM such as
      !	 	XX(IC,NLIM-1) < XLIM < XX(IC,NLIM)
      !	for input data ordered, retrieved from memory
      !	Assumes NLIM already preset so that XX(IC,NLIM-1) < XLIM
      !---------------------------------------------------------------------
      integer :: nv                     !
      integer :: nc                     !
      integer :: ic                     !
      real :: xx(nc,nv)                 !
      real :: xlim                      !
      integer :: nlim                   !
    end subroutine findr
  end interface
  !
  interface
    subroutine convol (du,dv,resu)
      real :: du                        !
      real :: dv                        !
      real :: resu                      !
    end subroutine convol
  end interface
  !
  interface
    subroutine convfn (type, parm, buffer, bias)
      use gildas_def
      !---------------------------------------------------------------------
      ! 	CONVFN computes the convolving functions and stores them in
      !   	the supplied buffer. Values are tabulated every 1/100 cell.
      ! Arguments :
      !	TYPE		I*4	Convolving function type
      ! 	PARM(10)        R*4  	Convolving function parameters.
      !				PARM(1) = radius of support in cells
      !	BUFFER(4096)	R*4  	Work buffer.
      !	BIAS		R*4	Center of convolution
      !---------------------------------------------------------------------
      integer :: type                   !
      real*4 ::  parm(10)               !
      real*4 ::  buffer(4096)           !
      real*4 ::  bias                   !
    end subroutine convfn
  end interface
  !
  interface
    subroutine sphfn (ialf, im, iflag, eta, psi, ier)
      !---------------------------------------------------------------------
      !     SPHFN is a subroutine to evaluate rational approximations to se-
      !  lected zero-order spheroidal functions, psi(c,eta), which are, in a
      !  sense defined in VLA Scientific Memorandum No. 132, optimal for
      !  gridding interferometer data.  The approximations are taken from
      !  VLA Computer Memorandum No. 156.  The parameter c is related to the
      !  support width, m, of the convoluting function according to c=
      !  pi*m/2.  The parameter alpha determines a weight function in the
      !  definition of the criterion by which the function is optimal.
      !  SPHFN incorporates approximations to 25 of the spheroidal func-
      !  tions, corresponding to 5 choices of m (4, 5, 6, 7, or 8 cells)
      !  and 5 choices of the weighting exponent (0, 1/2, 1, 3/2, or 2).
      !
      !  Input:
      !    IALF    I*4   Selects the weighting exponent, alpha.  IALF =
      !                  1, 2, 3, 4, and 5 correspond, respectively, to
      !                  alpha = 0, 1/2, 1, 3/2, and 2.
      !    IM      I*4   Selects the support width m, (=IM) and, correspond-
      !                  ingly, the parameter c of the spheroidal function.
      !                  Only the choices 4, 5, 6, 7, and 8 are allowed.
      !    IFLAG   I*4   Chooses whether the spheroidal function itself, or
      !                  its Fourier transform, is to be approximated.  The
      !                  latter is appropriate for gridding, and the former
      !                  for the u-v plane convolution.  The two differ on-
      !                  by a factor (1-eta**2)**alpha.  IFLAG less than or
      !                  equal to zero chooses the function appropriate for
      !                  gridding, and IFLAG positive chooses its F.T.
      !    ETA     R*4   Eta, as the argument of the spheroidal function, is
      !                  a variable which ranges from 0 at the center of the
      !                  convoluting function to 1 at its edge (also from 0
      !                  at the center of the gridding correction function
      !                  to unity at the edge of the map).
      !
      !  Output:
      !    PSI      R*4  The function value which, on entry to the subrou-
      !                  tine, was to have been computed.
      !    IER      I*4  An error flag whose meaning is as follows:
      !                     IER = 0  =>  No evident problem.
      !                           1  =>  IALF is outside the allowed range.
      !                           2  =>  IM is outside of the allowed range.
      !                           3  =>  ETA is larger than 1 in absolute
      !                                     value.
      !                          12  =>  IALF and IM are out of bounds.
      !                          13  =>  IALF and ETA are both illegal.
      !                          23  =>  IM and ETA are both illegal.
      !                         123  =>  IALF, IM, and ETA all are illegal.
      !---------------------------------------------------------------------
      integer*4 :: ialf                 !
      integer*4 :: im                   !
      integer*4 :: iflag                !
      real*4 :: eta                     !
      real*4 :: psi                     !
      integer*4 :: ier                  !
    end subroutine sphfn
  end interface
  !
  interface
    subroutine grdflt (ctypx, ctypy, xparm, yparm)
      !---------------------------------------------------------------------
      !     GRDFLT determines default parameters for the convolution functions
      !     If no convolving type is chosen, an Spheroidal is picked.
      !     Otherwise any unspecified values ( = 0.0) will be set to some
      !     value.
      ! Arguments:
      !     CTYPX,CTYPY           I  Convolution types for X and Y direction
      !                                1 = pill box
      !                                2 = exponential
      !                                3 = sinc
      !                                4 = expontntial * sinc
      !                                5 = spheroidal function
      !     XPARM(10),YPARM(10)   R*4  Parameters for the convolution fns.
      !                                (1) = support radius (cells)
      !---------------------------------------------------------------------
      integer :: ctypx                  !
      integer :: ctypy                  !
      real*4 ::    xparm(10)            !
      real*4 ::    yparm(10)            !
    end subroutine grdflt
  end interface
  !
  interface
    subroutine dosor  (visi,nd,np,we,iy)
      use gildas_def
      integer :: nd                     !
      integer :: np                     !
      real :: visi(nd,np)               ! Corrected 22-Mar-1995
      real :: we(nd)                    ! Corrected 22-Mar-1995
      integer :: iy                     !
    end subroutine dosor
  end interface
  !
  interface
    function trione (x,nd,n,ix,work)
      !---------------------------------------------------------------------
      ! 	Sorting program that uses a quicksort algorithm.
      !	Sort on one row
      !	X	R*4(*)	Unsorted array				Input
      !	ND	I	First dimension of X 			Input
      !	N	I	Second dimension of X			Input
      !	IX	I	X(IX,*) is the key for sorting		Input
      !	WORK	R*4(ND)	Work space for exchange			Input
      !---------------------------------------------------------------------
      integer :: trione                 !
      integer :: nd                     !
      integer :: n                      !
      real*4 :: x(nd,n)                 !
      integer :: ix                     !
      real*4 :: work(nd)                !
    end function trione
  end interface
  !
  interface
    subroutine domask (nc, n, x, w, wm, bval)
      !---------------------------------------------------------------------
      !     mask points with weights lower than wm
      !---------------------------------------------------------------------
      integer :: nc                     !
      integer :: n                      !
      real :: x(nc, n)                  !
      real :: w(n)                      !
      real :: wm                        !
      real :: bval                      !
    end subroutine domask
  end interface
  !
  interface
    subroutine  nearfield(nx,ny,p,kw,df,distance,diameter,f)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      !     Correct phase map for near field effects.
      !     This means geometrical pathlength variation due to
      !     1)- finite distance of transmitter (DISTANCE in m)
      !       dp1 = r^2/(2*R) - r^4 / (8*R^3)
      !     2)- compensating displacement of receiver DF (in m) from prime focus.
      !       dp2 = DF* (1- DF * cos(alpha) + DF^2 * sin(alpha)^2 /(2*rho)
      ! where
      !     r = radius in aperture plane                     r   = 2*f*t
      !     alpha = angle between ray from focus and  axis:  t   = tan(alpha/2)
      !     rho = ray length between focus and parabola:     rho = f*(1+t^2)
      !     kw = wave vector (2*pi/lambda) in rad / m
      !---------------------------------------------------------------------
      integer :: nx                     !
      integer :: ny                     !
      real :: p(nx,ny)                  !
      real*8 :: kw                      !
      real :: df                        !
      real :: distance                  !
      real :: diameter                  !
      real :: f                         !
    end subroutine nearfield
  end interface
  !
  interface
    subroutine load_x_image(name,type,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      !     open and read X image header
      !---------------------------------------------------------------------
      character(len=*) :: name          !
      character(len=*) :: type          !
      logical :: error                  !
    end subroutine load_x_image
  end interface
  !
  interface
    subroutine reference_feed(nx,ny,x,kw, center)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      !     do reference feed correction
      !     correct for the phase effect due to the motion of the reference feed
      !---------------------------------------------------------------------
      integer :: nx                     !
      integer :: ny                     !
      complex :: x(nx,ny)               !
      real :: kw                        !
      real :: center                    !
    end subroutine reference_feed
  end interface
  !
  interface
    subroutine grid_corr(nx, ny, data)
      ! dummy
      integer :: nx, ny
      real ::  data(2, nx, ny)
    end subroutine grid_corr
  end interface
  !
  interface
    subroutine grdtab (n, buff, bias, corr)
      !---------------------------------------------------------------------
      ! GILDAS	UVMAP
      ! Compute fourier transform of gridding function
      !	N	I*4	Number of pixels, assuming center on N/2+1
      !	BUFF	R*4(*)	Gridding function, tabulated every 1/100 cell
      !	BIAS	R*4	Center of gridding function
      !	CORR	R*4(N)	Gridding correction FT
      !---------------------------------------------------------------------
      integer :: n                      !
      real :: buff(*)                   !
      real :: bias                      !
      real :: corr(n)                   !
    end subroutine grdtab
  end interface
  !
  interface
    subroutine solve_monitor(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_monitor
  end interface
  !
  interface
    subroutine sub_solve_mon(md,mb,x_data,y_data,w_data,   &
        ind,wk1,xxx,yyy,www,line,error)
      use gildas_def
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      integer :: ind(md)                !
      real*8 :: wk1(3*md)               !
      real*8 :: xxx(md)                 !
      real*8 :: yyy(md)                 !
      real*8 :: www(md)                 !
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine sub_solve_mon
  end interface
  !
  interface
    subroutine plot_power(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Plot the power spectra of the variables in Y (X must be TIME)
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine plot_power
  end interface
  !
  interface
    subroutine sub_power(md,nd,x,y,w,z,work,bin,xbin,error)
      use gildas_def
      integer :: md                     !
      integer :: nd                     !
      real*4 :: x(md)                   !
      real*4 :: y(md)                   !
      real*4 :: w(md)                   !
      complex :: z(md)                  !
      real :: work(md,2)                !
      logical :: bin                    !
      real*4 :: xbin                    !
      logical :: error                  !
    end subroutine sub_power
  end interface
  !
  interface
    subroutine clic_wvr(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command WVR
      ! It processes all scans as encountered, except cwvr scans,
      ! to deduce the atmospheric parameter from the wvr.
      ! options /CMODE (1) /NOWRITE (2)
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_wvr
  end interface
  !
  interface
    subroutine sub_wvr(ldata_in,data,wvrmode,wvrcpol,wvrprev,error)
      !---------------------------------------------------------------------
      ! Process a WVR data
      !
      ! Arguments
      !     LDATA_IN      Integer Length of data array
      !     DATA(*)       (*)     Data array
      !
      ! oct-01, RM: add calibration of wvr
      !---------------------------------------------------------------------
      integer :: ldata_in               !
      real :: data(ldata_in)            !
      integer :: wvrmode                !
      integer :: wvrcpol                !
      integer :: wvrprev                !
      logical :: error                  !
    end subroutine sub_wvr
  end interface
  !
  interface
    subroutine wvr_cali(iant, wvr, wvrstat, wvrmode,   &
        wvrcpol, wvrprev, message, status)
      !---------------------------------------------------------------------
      ! Author: 18-Oct 2001 Michael Bremer, IRAM
      ! Routine to calibrate the WVR data. Each calibration phase is stored
      ! individually, and the difference hot/ambient can be corrected with a
      ! polynomial of degree wvrcpol (fit with Bevington routines in real*8
      ! attached at the end of this file, independent of NAG or LAPACK).
      !
      ! INPUT: wvr       real*4   raw radiometer counts
      !        wvrstat   integer  instrumental state flag
      !        n_dumps      integer  number of subscans
      !        wvrmode      integer  calibration mode. 1=trec+gain external,
      !                                                2=trec+gain laboratory,
      !                                                3=trec lab, gain calibrated
      !                                                4=trec+gain calibrated
      !        wvrcpol      integer  degree+1 of the polynomial fitted through
      !                              the ambient load counts to improve the Hot
      !                              load counts background subtraction.
      !        wvrprev      integer  if 0: re-calculate wvrtcal, 1: don't re-calculate
      !        wvrtamb      real*4   load temperature in K
      !        wvrtpel      real*4   Peltier temperature in K
      !        wvrlab_tcal  real*4   laboratory value for wvrtcal
      !        wvrlab_trec  real*4   laboratory value for wvrtrec
      !        wvrlab_tdio  real*4   laboratory value for noise diode temperature
      !
      ! OUTPUT:  wvramb     real*4    counts on ambient load for each channel
      !          wvrtcal    real*4    gain K/counts for each channel
      !          wvrtrec    real*4    receiver temperature for each channel
      !          message    character*160   detailed accumulated errors
      !          status     integer   0=ok, 1=warning, 2=error
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      integer, parameter :: mcwvrdump=128
      integer :: iant                      !
      real :: wvr(mcwvrdump,mwvrch,mnant)  !
      integer :: wvrstat(mcwvrdump,mnant)  !
      integer :: wvrmode                   !
      integer :: wvrcpol                   !
      integer :: wvrprev                   !
      character(len=160) :: message        !
      integer :: status                    !
    end subroutine wvr_cali
  end interface
  !
  interface
    subroutine do_wvr (again,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! The WVR MODEL program is called for each antenna.
      !---------------------------------------------------------------------
      logical :: again                  !
      logical :: error                  !
    end subroutine do_wvr
  end interface
  !
  interface
    subroutine wvr_model(ia,airmass, status)
      !---------------------------------------------------------------------
      ! M. Bremer, 17-Oct-2001
      ! Tested with small "frame" program using WVR test data sets.
      ! Calculates an atmospherical model for phase correction for the
      ! triple-band radiometer, assumes operational radiometers, i.e.
      ! antennas not yet equipped with WVRs must NOT call this routine.
      !
      ! LINK LIBRARY:
      !     libatm.a
      !
      ! BEFORE calling wvr_model in an antenna-specific loop:
      !        call atm_atmosp(r_tamb, r_pamb, r_alti)
      !
      ! INPUT:
      !
      !     real wvrfreq(mwvrch)  I  frequencies
      !     real wvrbw(mwvrch)    I  bandwidth
      !     real r_pamb           I  atm. press
      !     real r_tamb           I  atm. Temp
      !     real r_alti           I  observatory alititude
      !     real airmass          I  airmass
      !     real wvrfeff(mwvrch)  I  Forward efficiency for each channel
      !     real wvrtcal(mwvrch)  I  k/count channel gains
      !     real wvrtrec(mwvrch)  I  receiver temperatures (per channel)
      !     real wvraver(mwvrch)  I  average counts of current obs, which
      !                              must be on the sky
      !     real wvramb(mwvrch)   I  average counts on last ambient measurement
      !     real wvrtamb          I  last load temperature
      !
      ! OUTPUT:
      !     real wvrtatm          O  WVR central channel "clear sky"
      !                              atmospheric temperature
      !     real wvrtsys (mwvrch) O  WVR channel system temperature
      !     real wvrdpath(mwvrch) O  factors for channels for
      !                              Tsky to path conversion [microns/K]
      !     real wvrh2o           O  Precipitable amount of water vapour [mm]
      !     real wvrliq(mwvrch)   O  Tsky weight factors for each channel
      !                              for combination into cloud emission []
      !     real wvrdcloud(mwvrch)O  (experimental:) factors to convert
      !                              the cloud emission into liquid phase shifts.
      !                              Units: [microns/K]
      !     integer status        O  atm standard error codes, +10 superimposed
      !                              if iteration for water does not reproduce
      !                              t_triple to 1e-4 (relative error)
      !
      ! MEMO:
      !
      ! vapor_path =sum(ic=1,mwvrch){wvrdpath(ic)*(tsky(ic)}
      ! cloud_path =sum(ic=1,mwvrch){wvrdcloud(ic)*wvrliq(ic)*tsky(ic)}
      !
      ! where tsky(ic) = atmospheric emission in Kelvin along the line of sight
      !---------------------------------------------------------------------
      integer :: ia                     !
      real :: airmass                   !
      integer :: status                 !
    end subroutine wvr_model
  end interface
  !
  interface
    subroutine atm_tri(ia,t_triple,b,c1,c2,airmass,   &
        dpdtri, temib, wvrtaut, ier)
      !---------------------------------------------------------------------
      ! Purpose: determines the amount of water vapor for the triple frequency
      !          radiometer.
      ! INPUT:  real*4 t_triple: weighted sum of the 3 monitoring bands in Kelvin
      !        wvrfreq(mwvrch) : effective central monitoring frequencies in GHz
      !          wvrbw(mwvrch) : channel band widths
      !                b,c1,c2 : weighting factors between the bands
      !                airmass : Air mass, "one" in zenith
      ! OUTPUT: real*4 wvrh2o   : precipitable amount of water vapor in mm (Zenith)
      !                dpdtri  : gradient dpath/dt_triple in microns/K
      !                wvrtatm : atmospherical temperature (center water line)
      !           temib(mwvrch): "blue sky" emission at the given elevation,
      !                          i.e. modelled monitor band signals without
      !                          clouds in Kelvin
      !                path_out: "blue sky" optical path in microns
      !         wvrtaut(mwvrch): total clear sky opacity of the wvr channels
      !         integer*4 ier  : error flag.
      !
      ! NOTE: Routine needs a previous call to  atm_atmosp(t0, p0, h0), by
      !       preference OUTSIDE the antenna specific loop.
      !
      ! Author: Michael Bremer, IRAM 10-Jun-1999
      !---------------------------------------------------------------------
      include 'clic_parameter.inc'
      integer :: ia                     !
      real :: t_triple                  !
      real :: b                         !
      real :: c1                        !
      real :: c2                        !
      real :: airmass                   !
      real :: dpdtri                    !
      real :: temib(mwvrch)             !
      real :: wvrtaut(mwvrch)           !
      integer :: ier                    !
    end subroutine atm_tri
  end interface
  !
  interface
    function air_mass_m (elevation, altitude)
      !---------------------------------------------------------------------
      ! NIC	SOLVE support routine
      !	Compute air masses
      ! 10-JUL-1995 modified by M. Bremer
      !
      ! Arguments :
      !	ELEV		R*4	elevation in degrees		Input
      !	AIR_MASS	R*4	number of airmasses		Output
      !       ALTITUDE        R*4     altitude of observatory above
      !                               sea level                       Input
      !---------------------------------------------------------------------
      real*4 ::  air_mass_m             !
      real*4 ::  elevation              !
      real*4 ::  altitude               !
    end function air_mass_m
  end interface
  !
  interface
    subroutine poly_3(x,y,para)
      !---------------------------------------------------------------------
      ! Calculate the 3rd degree polynomial parameters going through 4 points:
      ! y = d + x*(c + x*(b + x*a) )
      ! para(1,2,3,4) = (a,b,c,d)
      ! Caution: all x must be different from each other. No rezoning to [-1,1] is
      !          done, values of x>100 can result in dubious results, especially
      !          for "d" (the other parameters depend on differences)
      !
      !  Input: real*8 x(4), y(4)
      ! Output: real*8 para(4)
      !
      !  Speed: about 30 times faster than a Gauss-Jordan matrix inversion
      !
      ! Author: M. Bremer, 10-Jun-1999 (tested, OK)
      !---------------------------------------------------------------------
      real*4 :: x(4)                    !
      real*4 :: y(4)                    !
      real*4 :: para(4)                 !
    end subroutine poly_3
  end interface
  !
  interface
    subroutine atm_tri_cl(ia,tcloud)
      !---------------------------------------------------------------------
      ! MB 16-Aug-2001
      ! Simple cloud model after Staelin (1966), J. Geophys. Res. 71, 2875
      ! Take the difference between the extreme lowfreq/highfreq channels to
      ! get a good result (central channel might be contaminated by residual
      ! water vapor emission). Taking the difference should remove
      ! background fluctuations.
      ! No airmass is needed here.
      !---------------------------------------------------------------------
      integer :: ia                     !
      real :: tcloud                    !
    end subroutine atm_tri_cl
  end interface
  !
  interface
    subroutine bev_polfit (x,y,sigmay,npts,nterms,mode,a,chisqr)
      integer :: npts                   !
      real*8 :: x(npts)                 !
      real*8 :: y(npts)                 !
      real*8 :: sigmay(npts)            !
      integer :: nterms                 !
      integer :: mode                   !
      real*8 :: a(*)                    !
      real*8 :: chisqr                  !
    end subroutine bev_polfit
  end interface
  !
  interface
    subroutine solve_wvr(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_wvr
  end interface
  !
  interface
    subroutine get_wvr_jumps(md,mb,x_data,y_data,w_data,   &
        airmin, dtmax,error)
      use gildas_def
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      real :: airmin                    !
      real :: dtmax                     !
      logical :: error                  !
    end subroutine get_wvr_jumps
  end interface
  !
  interface
    subroutine  sub_solve_wvr(md,mb, x_data, y_data, w_data,   &
        wk1, xxx, yyy, www, line, plot, ipol, nkey, error)
      use gildas_def
      integer :: md                     !
      integer :: mb                     !
      real :: x_data(md,mb)             !
      real :: y_data(md,mb)             !
      real :: w_data(md,mb)             !
      real*8 :: wk1(3*md)               !
      real*8 :: xxx(md)                 !
      real*8 :: yyy(md)                 !
      real*8 :: www(md)                 !
      character(len=*) :: line          !
      logical :: plot                   !
      integer :: ipol                   !
      integer :: nkey                   !
      logical :: error                  !
    end subroutine sub_solve_wvr
  end interface
  !
  interface
    subroutine store_wvr(line,error)
      use gildas_def
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine store_wvr
  end interface
  !
  interface
    subroutine get_wvr(ia,ich,x,y,oscan,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Returns in Y the value and derivatives of the calibration curve
      ! at time X,
      ! for baseline IB, side band ISB.
      !---------------------------------------------------------------------
      integer :: ia                     !
      integer :: ich                    !
      real :: x                         !
      real :: y                         !
      integer :: oscan                  !
      logical :: error                  !
    end subroutine get_wvr
  end interface
  !
  interface
    subroutine clic_clone(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Command COPY = MODIFY NOTHING
      !     well not so simple
      ! COPY DATA|HEADERS [ANT | NOANT] [BASE | NOBASE]
      ! 1:
      !     DATA     data and headers are copied
      !     HEADERS  headers only
      ! 2 and/or 3:
      !     ANT      Antenna calibration section created
      !     BASE     Baseline calibration section created
      !     NOANT    Antenna calibration sections  NOT created
      !     NOBASE   Baseline calibration sections NOT created
      ! otherwise sections are copied if and only if present.
      ! default is "ANT"
      ! Bure copy Job does "NOANT NOBASE"
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_clone
  end interface
  !
  interface
    subroutine fill_data(ndata, data,   &
        t_nant, t_nbas, t_ldpar, t_ldatc, t_ldump,   &
        nnewdata, newdata)
      use gildas_def
      !---------------------------------------------------------------------
      ! create a data section with nant antennas (nant > r_nant)
      !---------------------------------------------------------------------
      integer :: ndata                  !
      real :: data(ndata)               !
      integer :: t_nant                 !
      integer :: t_nbas                 !
      integer :: t_ldpar                !
      integer :: t_ldatc                !
      integer :: t_ldump                !
      integer :: nnewdata               !
      real :: newdata(nnewdata)         !
    end subroutine fill_data
  end interface
  !
  interface
    subroutine clic_grid_tp(line,error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! CLIC
      ! GRID_TOTAL computes a total power map.
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine clic_grid_tp
  end interface
  !
  interface
    subroutine load_tp(iant,nvis,ivis,vis,   &
        lambda1, lambda2, beta1, beta2, msize, fudge, error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      ! CLIC
      !	Support routine for GRID_TP
      ! Arguments:
      !	IANT	I	antenna used
      !       TIME_MAX R*4    Max. integration time (compression)
      !---------------------------------------------------------------------
      integer :: iant                   !
      integer :: nvis                   !
      integer :: ivis                   !
      real :: vis(4,nvis)               !
      real :: lambda1                   !
      real :: lambda2                   !
      real :: beta1                     !
      real :: beta2                     !
      real :: msize                     !
      real :: fudge                     !
      logical :: error                  !
    end subroutine load_tp
  end interface
  !
  interface
    subroutine do_baseline(pvis, vis, msize, a, b, error)
      use gildas_def
      integer :: pvis                   !
      real :: vis(4,pvis)               !
      real :: msize                     !
      real*8 :: a(pvis,2)               !
      real*8 :: b(pvis)                 !
      logical :: error                  !
    end subroutine do_baseline
  end interface
  !
  interface
    subroutine open_tpmap(nx,ny,rx,ry,vx,vy,dx,dy,iant,error)
      use gildas_def
      use gbl_format
      use clic_xypar
      !---------------------------------------------------------------------
      ! Open map GDF files
      ! input:
      !     NX, NY number of pixel on each axis for beam map
      !     DX, DY coordinate increments, in radians!
      !     RX, RY pixel numbers for center (0,0)
      !     IANT   physical number of antenna
      !     ERROR  error return
      !---------------------------------------------------------------------
      integer :: nx                     !
      integer :: ny                     !
      real :: rx                        !
      real :: ry                        !
      real :: vx                        !
      real :: vy                        !
      real :: dx                        !
      real :: dy                        !
      integer :: iant                   !
      logical :: error                  !
    end subroutine open_tpmap
  end interface
  !
  interface
    subroutine plot_tpmap(tmin, tmax, error)
      use gildas_def
      use clic_xypar
      !---------------------------------------------------------------------
      !
      !---------------------------------------------------------------------
      real :: tmin                      !
      real :: tmax                      !
      logical :: error                  !
    end subroutine plot_tpmap
  end interface
  !
  interface
    subroutine t_extr(n, y, bval, eval, ymin, ymax)
      integer :: n                      !
      real :: y(n)                      !
      real :: bval                      !
      real :: eval                      !
      real :: ymin                      !
      real :: ymax                      !
    end subroutine t_extr
  end interface
  !
  interface
    subroutine mask_tpmap(nx,ny, tp, size)
      use gildas_def
      use clic_xypar
      integer :: nx                     !
      integer :: ny                     !
      real :: tp(nx,ny)                 !
      real :: size                      !
    end subroutine mask_tpmap
  end interface
  !
  interface
    subroutine check_limits(error)
      logical :: error                  !
    end subroutine check_limits
  end interface
  !
  interface
    subroutine solve_tp(line,error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Antenna based.
      ! SOLVE arg /TOTAL /FIX /PLOT /COMPRESS
      ! option numbers
      !      1  /PLOT
      !                      (plot data and fit)
      !      6  /COMPRESS
      !                      (used to improve SNR and combine beam switch states)
      !     11  /FIX parameter1 value1 parameter2 value2 ...
      !                      ( to fix parameters)
      !     12  /TOTAL       (is there)
      !         /OUT
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine solve_tp
  end interface
  !
  interface
    subroutine fitfcn(iflag,m,nvpar,x,f,fjac,ljc)
      use gildas_def
      !---------------------------------------------------------------------
      ! FCN is called by DNLS1E
      !
      !     IFLAG   Integer    Code for print out                Input
      !     M       Integer    Number of data points             Input
      !     NVPAR   Integer    Number of variable parameters     Input
      !     X       Real*8     Variable Parameters               Input
      !     F       Real*8     Function value at each point      Output
      !     FJAC    Real*8     Partial derivatives of F          Output
      !     LJC     Integer    First dimension of FJAC           Input
      !
      ! replacing     SUBROUTINE LSFUN2(M, NVPAR, X, F, FJAC, LJC)
      !---------------------------------------------------------------------
      integer :: iflag                  !
      integer :: m                      !
      integer :: nvpar                  !
      real*8 :: x(nvpar)                !
      real*8 :: f(m)                    !
      integer :: ljc                    !
      real*8 :: fjac(ljc,nvpar)         !
    end subroutine fitfcn
  end interface
  !
  interface
    subroutine gettotal(mp, mb, xd, yd, wd,   &
        i, ll, tt, ww)
      use gildas_def
      !---------------------------------------------------------------------
      !     Obtain U, V, and the visibility from the array UVRIW.
      !
      !     Np     Integer     Number data points        Input
      !     Nb     Integer     Number of boxes            Input
      !     xd     real        X data plot array
      !     yd     real        Y data plot array
      !     ibl    integer     box number of lambda       Input
      !     ibb    integer     box number of beta         Input
      !     i      integer     point nmber                Input
      !
      !     ll     Real*8      Lambda coord.              Output
      !     bb     Real*8      Beta   coord.              Output
      !     tt     Real*8      Total Power                Output
      !     ww     Real*8      Weight                     Output
      !---------------------------------------------------------------------
      include 'clic_tpfit.inc'
      integer :: mp                     !
      integer :: mb                     !
      real :: xd(mp, mb)                !
      real :: yd(mp, mb)                !
      real :: wd(mp, mb)                !
      integer :: i                      !
      real*8 ::  ll(mdim)               !
      real*8 ::  tt                     !
      real*8 ::  ww                     !
    end subroutine gettotal
  end interface
  !
  interface
    subroutine diagonal(m,n,r,c)
      integer :: m                      !
      integer :: n                      !
      real*8 :: r(m,n)                  !
      real*8 :: c(n)                    !
    end subroutine diagonal
  end interface
  !
  interface
    subroutine guess_tp(x0, t0, t1, error)
      use gildas_def
      include 'clic_tpfit.inc'
      real :: x0(mdim)                  !
      real :: t0                        !
      real :: t1                        !
      logical :: error                  !
    end subroutine guess_tp
  end interface
  !
  interface
    subroutine gmodel(xx, par, y, dy)
      use gildas_def
      !---------------------------------------------------------------------
      ! Function is
      !     t0 + t1 * exp (-4.*log(2.)*((xx-x0)**2+(yy-y0)**2)*ast**2)/beam**2
      ! pars are:
      !
      !     1 x0       x position
      !     2 y0       y position [optional]
      !     3 t1       amplitude (peak-offset)
      !     4 beam     half-power beam
      !     5 t0       amplitude offset
      !     6 ast      beam ratio y/x
      !---------------------------------------------------------------------
      include 'clic_tpfit.inc'
      real*8 :: xx(mdim)                !
      real*8 :: par(6)                  !
      real*8 :: y                       !
      real*8 :: dy(6)                   !
    end subroutine gmodel
  end interface
  !
  interface
    subroutine get_diameter(telescope,diam)
      character(len=12) :: telescope    !
      real *4 :: diam                   !
    end subroutine get_diameter
  end interface
  !
  interface
    function z_exp(x)
      real*8 :: z_exp                   !
      real*8 :: x                       !
    end function z_exp
  end interface
  !
  interface
    subroutine plot_fittp(error)
      use gildas_def
      !---------------------------------------------------------------------
      ! Plot total power fit.
      !---------------------------------------------------------------------
      logical :: error                  !
    end subroutine plot_fittp
  end interface
  !
  interface
    subroutine ini_telescope(line,iopt,error)
      !---------------------------------------------------------------------
      ! 	MODIFY TELESCOPE
      ! compute the telescope name according to new denomination
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      integer :: iopt                   !
      logical :: error                  !
    end subroutine ini_telescope
  end interface
  !
  interface
    subroutine clic_pack_set(pack)
      use gpack_def
      !---------------------------------------------------------------------
      ! Public
      !---------------------------------------------------------------------
      type(gpack_info_t), intent(out) :: pack  !
    end subroutine clic_pack_set
  end interface
  !
  interface
    subroutine clic_pack_init(gpack_id,error)
      use sic_def
      !---------------------------------------------------------------------
      ! Private
      !---------------------------------------------------------------------
      integer, intent(in) :: gpack_id   !
      logical, intent(inout) :: error   !
    end subroutine clic_pack_init
  end interface
  !
  interface
    subroutine clic_message_set_id(id)
      use gbl_message
      !---------------------------------------------------------------------
      ! Alter library id into input id. Should be called by the library
      ! which wants to share its id with the current one.
      !---------------------------------------------------------------------
      integer, intent(in) :: id         !
    end subroutine clic_message_set_id
  end interface
  !
  interface
    subroutine clic_message(mkind,procname,message)
      !---------------------------------------------------------------------
      ! Messaging facility for the current library. Calls the low-level
      ! (internal) messaging routine with its own identifier.
      !---------------------------------------------------------------------
      integer,          intent(in) :: mkind     ! Message kind
      character(len=*), intent(in) :: procname  ! Name of calling procedure
      character(len=*), intent(in) :: message   ! Message string
    end subroutine clic_message
  end interface
  !
  interface
    subroutine newuvt_close (xima) 
      use image_def
      !---------------------------------------------------------------------
      ! Closes the output table.
      !---------------------------------------------------------------------
      type (gildas), intent(inout) :: xima
    end subroutine newuvt_close
  end interface
  !
  interface
    subroutine newuvt_init (xima,name,nvis,scan,positions,error)
      use image_def
      use gbl_format
      !---------------------------------------------------------------------
      ! Creates the output table header in X (the significant header parameters
      ! have already been defned in clic_table)
      !---------------------------------------------------------------------
      type (gildas), intent(inout) :: xima  ! UV data file
      character(len=*), intent(in) :: name  ! desired file name
      integer, intent(in) :: nvis           ! Number of visibilities
      logical, intent(in) :: scan           ! Use SCAN number 
      logical, intent(in) :: positions      ! Add position information
      logical, intent(out) :: error         ! Error flag
    end subroutine newuvt_init
  end interface
  !
  interface
    subroutine newuvt_extend (xima,name,nvis,nn,ndobs,ndscan,error,lc)
      use image_def
      !---------------------------------------------------------------------
      ! Open a Table to extend it.
      !---------------------------------------------------------------------
      type (gildas), intent(inout) :: xima  ! UV data file
      character(len=*), intent(in) :: name  ! desired file name
      integer, intent(in) :: nvis           ! Number of visibilities
      logical, intent(out) :: error         ! Error flag
      integer, intent(in) :: nn             !
      integer, intent(in) :: ndobs(nn)      !
      integer, intent(in) :: ndscan(nn)     !
      logical, intent(in) :: lc             ! Check table before
    end subroutine newuvt_extend
  end interface
  !
  interface
    subroutine newuvt_cut (xima,nvis,error)
      use image_def
      !---------------------------------------------------------------------
      ! Update the number of visibilities
      !---------------------------------------------------------------------
      type(gildas), intent(inout) :: xima !
      integer, intent(in) :: nvis         ! True number of visibilities
      logical :: error                    !
    end subroutine newuvt_cut
  end interface
  !
  interface
    subroutine newuvt_open (xima,name,error)
      use image_def
      use gbl_format
      !---------------------------------------------------------------------
      ! Open an existing table, to get the header information 
      !---------------------------------------------------------------------
      type(gildas), intent(inout) :: xima  ! Image header
      character(len=*), intent(in) :: name !
      logical, intent(out) :: error        !
    end subroutine newuvt_open
  end interface
  !
  interface
    subroutine newuvt_addvisi (xima,visi,qsb,qband,qbas,qntch,qnchan,   &
        ibase,isb,inbc,frout,fres,datac,datal,passc,passl,   &
        ndrop, good, scan, width, shape, resample, fft, &
        positions, error)
      use image_def
      use phys_const
      !---------------------------------------------------------------------
      ! Called by CLIC_TABLE
      ! creates one visibility point (i.e. one baseline and one band for a
      ! given scan), but many channels if in line mode.
      !---------------------------------------------------------------------
      type (gildas), intent(inout) :: xima  ! 
      real, intent(out) :: visi(*)      ! Current visibility
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbas                   !
      integer :: qntch                  !
      integer :: qnchan                 !
      integer :: ibase                  !
      integer :: isb                    !
      integer :: inbc                   !
      real*8 :: frout                   !
      real*4 :: fres
      complex :: datac(qband,qsb,qbas)  !
      complex :: datal(qntch,qsb,qbas)  !
      complex :: passc(qband,qsb,qbas)  !
      complex :: passl(qntch,qsb,qbas)  !
    !  real*4 :: daps(7)                 !
    !  real*4 :: visi(3, qnchan)         !
    !  real*4 :: offsets(2)              !
      integer :: ndrop(2)               !
      logical :: good                   !
      logical :: scan                   !
      real*4 :: width                   !
      character(len=*) :: shape         !
      logical :: resample               !
      logical :: fft                    !
      logical :: positions              !
      logical :: error                  !
    end subroutine newuvt_addvisi
  end interface
  !
  interface
    subroutine sg_table(line,error)
      use image_def
      use gio_params
      use clic_xypar
      !---------------------------------------------------------------------
      ! CLIC 	Internal routine.
      !	Prepare a UV Table (GILDAS Format)
      ! Command :
      ! 	TABLE file_name OLD|NEW
      ! 1   /ADD Item
      ! 2		/DROP n1 n2
      ! 3   /FFT  Fourier Transform resampling algorithm
      ! 4		/FREQUENCY name frest
      ! 5		/NOCHECK [SOURCE|POINTING|EPOCH|PHASE|SCAN]
      ! 6		/RESAMPLE nchan rchan res VELOCITY|FREQUENCY
      ! 7   /POLARIZATION SPLIT|AVERAGE|JOINT
      !
      ! Code currently handles dual-polarization in 3 different ways
      !		SPLIT:	 A different visibility for each polarization
      !     AVERAGE: averaged polarizations before writing
      !     JOINT:   All channels Polar 1, then All channels polar 2 in visibility
      !---------------------------------------------------------------------
      character(len=*) :: line          !
      logical :: error                  !
    end subroutine sg_table
  end interface
  !
  interface
    subroutine uvt_extra_cols(line,nextra,extra,error)
      use gio_params
      use gbl_message
      !
      character(len=*), intent(in) :: line
      integer, intent(inout) :: nextra
      integer, intent(out) :: extra(*)
      logical, intent(out) :: error
    end subroutine uvt_extra_cols
  end interface
  !
  interface
    subroutine newuvt_create (xima,name,nvis,polcode,error)
      use image_def
      use gbl_format
      !---------------------------------------------------------------------
      ! Creates the output table header in X (the significant header parameters
      ! have already been defned in clic_table)
      !---------------------------------------------------------------------
      type (gildas), intent(inout) :: xima  ! UV data file
      character(len=*), intent(in) :: name  ! desired file name
      integer, intent(in) :: nvis           ! Number of visibilities
      integer, intent(in) :: polcode        ! Polarization treatment 
      logical, intent(out) :: error         ! Error flag
    end subroutine newuvt_create
  end interface
  !
  interface
    subroutine check_consistency(lcheck,xima,yima,tolerance,r_num,r_scan,error)
      use image_def
      include 'gbl_pi.inc'
      logical, intent(in) :: lcheck(4)
      real(8),  intent(in) :: tolerance(4)
      type(gildas), intent(in) :: xima
      type(gildas), intent(in) :: yima
      integer, intent(in) :: r_num, r_scan
      logical, intent(out) :: error
    end subroutine check_consistency
  end interface
  !
  interface
    subroutine add_visiline (xima,npola,visi,qsb,qband,qbas,qntch,   &
        ibase,isb,frout,datal,passl,   &
        ndrop, width, shape, resample, fft, error, dvis, polcode, xang)
      use image_def
      use phys_const
      !---------------------------------------------------------------------
      ! Called by SG_TABLE
      !
      ! creates one visibility point (i.e. one baseline and one band for a
      ! given scan), but many channels if in line mode.
      !
      ! Average Polarisations if needed, or split them, or concatenate them
      !---------------------------------------------------------------------
      type (gildas), intent(inout) :: xima  ! 
      integer, intent(in) :: npola      ! Number of requested polarization
      real, intent(out) :: visi(xima%gil%dim(1),npola)      ! Current visibilities
      integer :: qsb                    !
      integer :: qband                  !
      integer :: qbas                   !
      integer :: qntch                  !
      integer :: ibase                  ! Baseline number
      integer :: isb                    ! Sideband indicator
      real(8) :: frout                  !
      complex :: datal(qntch,qsb,qbas)  !
      complex :: passl(qntch,qsb,qbas)  !
      integer :: ndrop(2)               !
      real(4) :: width                  !
      character(len=*) :: shape         ! 
      logical :: resample               !
      logical :: fft                    !
      logical :: error                  !
      integer, intent(out) :: dvis      ! Number of visibilities added
      integer, intent(in) :: polcode    ! Code to handle polarizations
      real(8), intent(in) :: xang(5)    ! HA, Dec, Az, El, Parallactic Angle
    end subroutine add_visiline
  end interface
  !
  interface
    subroutine get_angles(x)
      use gildas_def
      !---------------------------------------------------------------------
      ! CLIC
      !     Return values of the descriptive angles
      !  Hour, Dec, Az, El, Parallactic
      !---------------------------------------------------------------------
      real(8), intent(out) :: x(5)
    end subroutine get_angles
  end interface
  !
  interface
    subroutine vel_scale_3 (name,freq)
      !---------------------------------------------------------------------
      ! Updates frequencies in header according to last 'SET FREQUENCY' command
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: name          !
      real(8), intent(in) :: freq                    !
    end subroutine vel_scale_3
  end interface
  !
  interface
    subroutine sg_zpassband(line, error)
      use gildas_def
      character*(*), intent(inout) :: line
      logical, intent(inout) :: error
    end subroutine sg_zpassband
  end interface
  !
  interface
    subroutine table_passband(file, iband, nsubant, subant, error)
      use image_def
      !---------------------------------------------------------------------
      ! Write the optional passband calibration fits table.
      ! assumes NO_POL=1
      !---------------------------------------------------------------------
      integer :: iband
      character(len=*) :: file
      integer :: nsubant                ! Number of antennas
      integer :: subant(nsubant)        ! selected antennas in table
      logical :: error                  !
    end subroutine table_passband
  end interface
  !
  interface
    subroutine fcheb  (xx,a,np1)
      !---------------------------------------------------------------------
      ! Fitting routine for a Chebychev polynomial of degree NP-1.
      ! Compute all coefficients of the polynomials, from the usual recurrence
      ! Beware of first point for "true" Chebychev polynomials
      !---------------------------------------------------------------------
      real*8 :: xx                      !
      integer :: np1                    !
      real*8 :: a(np1)                  !
    end subroutine fcheb
  end interface
  !
  interface
    subroutine svdput(f,np,a,nrows)
      !---------------------------------------------------------------------
      ! Move solution of least-square fit of Chebychev polynomial
      ! found by Singular Value Decomposition or Linear Fit into
      ! appropriate place for compatibility with NAG.
      !
      ! A bit crazy, but...
      !---------------------------------------------------------------------
      integer :: np                     !
      real*8 :: f(np)                   !
      integer :: nrows                  !
      real*8 :: a(nrows,np)             !
    end subroutine svdput
  end interface
  !
  interface
    subroutine svdfit(x,y,weight,ndata,a,ma,u,v,w,mp,np,chisq,   &
        funcs,b,afunc,error)
      !---------------------------------------------------------------------
      ! Singular value decomposition
      ! Given a set of NDATA points X(I), Y(I) with individual weights
      ! WEIGHT(I), use Chi2 minimization to determine the MA coefficients A of the
      ! fitting function. Here we solve the fitting
      ! equations using singular value decomposition of the NDATA by MA matrix.
      ! Arrays U,V,W provide workspace on input. On output they define the singular
      ! value decomposition and can be used to obtain the covariance matrix. MP, NP
      ! are the physical dimensions of the matrices U, V, W, as indicated below. It
      ! is necessary that MP>=NDATA, NP>=MA. The programs returns values for the MA
      ! fit parameters and Chi2, CHISQ. The user supplies a subroutine
      ! FUNCS(X,AFUNC,MA) that returns the MA basis functions evaluated at x=X in the
      ! array AFUNC.
      !
      ! As compared to the basic algorithm in "Numerical recipes",
      !     work arrays AFUNC and B are passed as argument. This suppress
      !     the need for size dependent storage.
      ! WEIGHT is the inverse of the sigma.
      !     Written by T.Forveille sometime ago
      !     Modifications by S.Guilloteau 28-feb-1995
      !---------------------------------------------------------------------
      integer :: ndata                  !
      real*8 :: x(ndata)                !
      real*8 :: y(ndata)                !
      real*8 :: weight(ndata)           !
      integer :: ma                     !
      real*8 :: a(ma)                   !
      integer :: mp                     !
      integer :: np                     !
      real*8 :: u(mp,np)                !
      real*8 :: v(np,np)                !
      real*8 :: w(np)                   !
      real*8 :: chisq                   !
      external :: funcs                 !
      real*8 :: b(ndata)                !
      real*8 :: afunc(ma)               !
      logical :: error                  !
    end subroutine svdfit
  end interface
  !
  interface
    subroutine svdcmp(a,m,n,mp,np,w,v,error)
      !---------------------------------------------------------------------
      ! Singular value decomposition
      ! Given a matrix A with logical dimensions M by N and physical dimensions MP
      ! by NP, this routine computes its singular value decomposition
      !	A = U.W.Vt
      ! The matrix U replaces A on output. The diagonal matrix of singular values W
      ! is a output as a vector W. The matrix V (not the transpose Vt) is output as
      ! V. M must be greater or equal to N. If it is smaller, then A should be filled
      ! up to square with zero rows.
      ! Typed from "numerical recipes".  TF
      !---------------------------------------------------------------------
      integer :: mp                     !
      integer :: np                     !
      real*8 :: a(mp,np)                !
      integer :: m                      !
      integer :: n                      !
      real*8 :: w(np)                   !
      real*8 :: v(np,np)                !
      logical :: error                  !
    end subroutine svdcmp
  end interface
  !
  interface
    subroutine svbksb(u,w,v,m,n,mp,np,b,x,error)
      !---------------------------------------------------------------------
      ! Singular value decomposition
      ! Solves A.X = B for a vector X, where A is specified by the arrays U,W,V as
      ! returned by SVDCMP. M and N are the logical dimensions of A, and will be
      ! equal for square matrices. MP and NP are the physical dimensions of A. B
      ! is the input right hand side. X is the output soolution vector. No input
      ! quantities are destroyed, so the routine may be called sequentially with
      ! different B's.
      ! Typed from "numerical recipes".  TF
      !---------------------------------------------------------------------
      integer :: mp                     !
      integer :: np                     !
      real*8 :: u(mp,np)                !
      real*8 :: w(np)                   !
      real*8 :: v(np,np)                !
      integer :: m                      !
      integer :: n                      !
      real*8 :: b(mp)                   !
      real*8 :: x(np)                   !
      logical :: error                  !
    end subroutine svbksb
  end interface
  !
  interface
    subroutine fpoly(x,p,np)
      !---------------------------------------------------------------------
      ! Fitting routine for a polynomial of degree NP-1.
      !---------------------------------------------------------------------
      real*8 :: x                       !
      integer :: np                     !
      real*8 :: p(np)                   !
    end subroutine fpoly
  end interface
  !
  interface
    subroutine lfit(x,y,weight,ndata,a,ma,covar,ncvm,chisq,   &
        funcs,error)
      !---------------------------------------------------------------------
      ! Given a set of NDATA points X(I),Y(I) with individual weights WEIGHT(I),
      ! use Chi2 minimization to determine the MA coefficients A of a function
      ! that depends linearly on A.
      !
      ! WEIGHT is the inverse of the Sigmas.
      !---------------------------------------------------------------------
      integer :: ndata                  !
      real*8 :: x(ndata)                !
      real*8 :: y(ndata)                !
      real*8 :: weight(ndata)           !
      integer :: ma                     !
      real*8 :: a(ma)                   !
      integer :: ncvm                   !
      real*8 :: covar(ncvm,ncvm)        !
      real*8 :: chisq                   !
      external :: funcs                 !
      logical :: error                  !
    end subroutine lfit
  end interface
  !
  interface
    subroutine gaussj(a,n,np,b,m,mp)
      integer :: np                     !
      real*8 :: a(np,np)                !
      integer :: n                      !
      integer :: mp                     !
      real*8 :: b(np,mp)                !
      integer :: m                      !
    end subroutine gaussj
  end interface
  !
  interface
    subroutine mindeg(a,n)
      integer :: n                      !
      real*8 :: a(n)                    !
    end subroutine mindeg
  end interface
  !
  interface
    subroutine svdvar(v,ma,np,w,cvm,ncvm,wti,sigma)
      !---------------------------------------------------------------------
      ! Compute covariance matrix
      ! Note NP > MA
      !---------------------------------------------------------------------
      integer :: np                     ! Dimension of V matrix
      real*8 :: v(np,np)                !
      integer :: ma                     ! Number of variables
      real*8 :: w(np)                   !
      integer :: ncvm                   ! Dimension of covariance matrix
      real*8 :: cvm(ncvm,ncvm)          !
      real*8 :: wti(ma)                 !
      real*8 :: sigma                   !
    end subroutine svdvar
  end interface
  !
  interface
    subroutine mth_init
    end subroutine mth_init
  end interface
  !
  interface
    subroutine mth_fitspl (name, nd, nk8, x, y, w, ks,   &
        wk1, wk2, cs, ss, error)
      character(len=*) :: name          !
      integer :: nd                     !
      integer :: nk8                    !
      real*8 :: x(nd)                   !
      real*8 :: y(nd)                   !
      real*8 :: w(nd)                   !
      real*8 :: ks(nk8)                 !
      real*8 :: wk1(nd)                 !
      real*8 :: wk2(4,nk8)              !
      real*8 :: cs(nk8)                 !
      real*8 :: ss                      !
      logical :: error                  !
    end subroutine mth_fitspl
  end interface
  !
  interface
    subroutine mth_getspl(name,nk8, ks, cs, xx, yy, error)
      character(len=*) :: name          !
      integer :: nk8                    !
      real*8 :: ks(nk8)                 !
      real*8 :: cs(nk8)                 !
      real*8 :: xx                      !
      real*8 :: yy                      !
      logical :: error                  !
    end subroutine mth_getspl
  end interface
  !
  interface
    subroutine mth_getspd(name,nk8, ks, cs, xx, left, yy, error)
      character(len=*) :: name          !
      integer :: nk8                    !
      real*8 :: ks(nk8)                 !
      real*8 :: cs(nk8)                 !
      real*8 :: xx                      !
      integer :: left                   !
      real*8 :: yy(4)                   !
      logical :: error                  !
    end subroutine mth_getspd
  end interface
  !
  interface
    subroutine fitbspl(m,ncap7,x,y,w,k,work1,work2,c,ss,ifail)
      !---------------------------------------------------------------------
      ! Computes a weighted least-squares approximation to an arbitrary set
      ! of data points by a cubic spline with knots prescribed by the user.
      ! Cubic spline interpolation can also be carried out.
      !
      ! COX-DE BOOR method for evaluating B-splines with adaptation of
      ! GENTLEMAN*s plane rotation scheme for solving over-determined linear systems.
      !
      ! Redesigned to use classical GIVENS rotations in order to
      ! avoid the occasional underflow (and hence overflow) problems
      ! experienced by GENTLEMAN*s 3-multiplication plane rotation scheme
      !
      ! WORK1 and WORK2 are workspace areas.
      !
      !     WORK1(R)  contains the value of the  R th  distinct data
      !     abscissa and, subsequently, for  R = 1, 2, 3, 4,  the
      !     values of the non-zero B-splines for each successive
      !     abscissa value.
      !
      !     WORK2(L, J)  contains, for  L = 1, 2, 3, 4,  the value of
      !     the  J th  element in the  L th  diagonal of the
      !     upper triangular matrix of bandwidth  4  in the
      !     triangular system defining the B-spline coefficients.
      !---------------------------------------------------------------------
      integer :: m                      !
      integer :: ncap7                  !
      real*8 :: x(m)                    !
      real*8 :: y(m)                    !
      real*8 :: w(m)                    !
      real*8 :: k(ncap7)                !
      real*8 :: work1(m)                !
      real*8 :: work2(4,ncap7)          !
      real*8 :: c(ncap7)                !
      real*8 :: ss                      !
      integer :: ifail                  !
    end subroutine fitbspl
  end interface
  !
  interface
    subroutine getbspl(ncap7, k, c, x, s, ifail)
      !---------------------------------------------------------------------
      ! Evaluates a cubic spline from its B-spline representation.
      !
      ! Uses DE BOOR*s method of convex combinations.
      !---------------------------------------------------------------------
      integer :: ncap7                  !
      real*8 :: k(ncap7)                !
      real*8 :: c(ncap7)                !
      real*8 :: x                       !
      real*8 :: s                       !
      integer :: ifail                  !
    end subroutine getbspl
  end interface
  !
  interface
    subroutine getbspd (ncap7, k, c, x, left, s, ifail)
      !---------------------------------------------------------------------
      !  Evaluation of cubic spline and its derivatives from its
      !  B-spline representation
      !---------------------------------------------------------------------
      integer :: ncap7                  !
      real*8 :: k(ncap7)                !
      real*8 :: c(ncap7)                !
      real*8 :: x                       !
      integer :: left                   !
      real*8 :: s(4)                    !
      integer :: ifail                  !
    end subroutine getbspd
  end interface
  !
  interface
    subroutine mth_fitpol(name,nd,ipol,m_pol,x,y,w,   &
        wk1,wk2,apol,spol,error)
      use gildas_def
      !---------------------------------------------------------------------
      !     ND Number of data points
      !     X(ND),Y(ND),W(ND) data points and weights
      !                       X must be non-decreasing
      !     WK1(3*ND)         work space
      !     WK2(2*IPOL)       work space
      !     M_POL             First size of matrix (>= IPOL)
      !     IPOL              Maximum degree of polynomials
      !     APOL(M_POL,IPOL)  coefficients of polynomials for degrees 0 to IPOL-1
      !     SPOL(IPOL)        Rms of fit for corresponding degree
      !---------------------------------------------------------------------
      character(len=*) :: name          !
      integer :: nd                     !
      integer :: ipol                   !
      integer :: m_pol                  !
      real*8 :: x(nd)                   !
      real*8 :: y(nd)                   !
      real*8 :: w(nd)                   !
      real*8 :: wk1(3*nd)               !
      real*8 :: wk2(2*ipol)             !
      real*8 :: apol(m_pol,ipol)        !
      real*8 :: spol(ipol)              !
      logical :: error                  !
    end subroutine mth_fitpol
  end interface
  !
  interface
    subroutine mth_getpol(name,ipol,aa,xx,yy,error)
      !---------------------------------------------------------------------
      ! Evaluate polynomial YY of order IPOL-1 at position XX
      ! from its Chebychev coefficients AA(IPOL)
      !---------------------------------------------------------------------
      character(len=*) :: name          !
      integer :: ipol                   ! Degree of polynomial + 1
      real*8 :: aa(ipol)                !
      real*8 :: xx                      !
      real*8 :: yy                      !
      logical :: error                  !
    end subroutine mth_getpol
  end interface
  !
  interface
    subroutine mth_fitlin (name,ndd,nvar,a,b,nd,sigma)
      use gildas_def
      character(len=*) :: name          !
      integer :: ndd                    ! Number of data points
      integer :: nvar                   ! Number of variables
      integer :: nd                     ! Size of original problem
      real*8 :: a(nd,nvar)              ! Input matrix/Output covariance
      real*8 :: b(nd)                   ! Input data/Output solutions
      real*8 :: sigma                   ! Sigma
    end subroutine mth_fitlin
  end interface
  !
  interface
    subroutine mth_dpotrf (name, uplo, n, a, lda, error)
      character(len=*) :: name           !
      character(len=*) :: uplo           !
      integer :: n                       !
      integer :: lda                     !
      real*8 :: a(lda,*)                 !
      logical :: error                   !
    end subroutine mth_dpotrf
  end interface
  !
  interface
    subroutine mth_dpotrs (name,   &
        uplo, n, nrhs, a, lda, b, ldb, info )
      character(len=*) :: name           !
      character(len=*) :: uplo           !
      integer :: n                       !
      integer :: nrhs                    !
      integer :: lda                     !
      real*8 :: a(lda,*)                 !
      integer :: ldb                     !
      real*8 :: b(ldb,*)                 !
      ! Local
      integer :: info
      logical :: error
      !
      !  Purpose
      !  =======
      !
      !  DPOTRS solves a system of linear equations A*X = B with a symmetric
      !  positive definite matrix A using the Cholesky factorization
      !  A = U**T*U or A = L*L**T computed by DPOTRF.
      !
      !  Arguments
      !  =========
      !
      !  UPLO    (input) CHARACTER*1
      !          = 'U':  Upper triangle of A is stored;
      !          = 'L':  Lower triangle of A is stored.
      !
      !  N       (input) INTEGER
      !          The order of the matrix A.  N >= 0.
      !
      !  NRHS    (input) INTEGER
      !          The number of right hand sides, i.e., the number of columns
      !          of the matrix B.  NRHS >= 0.
      !
      !  A       (input) DOUBLE PRECISION array, dimension (LDA,N)
      !          The triangular factor U or L from the Cholesky factorization
      !          A = U**T*U or A = L*L**T, as computed by DPOTRF.
      !
      !  LDA     (input) INTEGER
      !          The leading dimension of the array A.  LDA >= max(1,N).
      !
      !  B       (input/output) DOUBLE PRECISION array, dimension (LDB,NRHS)
      !          On entry, the right hand side matrix B.
      !          On exit, the solution matrix X.
      !
      !  LDB     (input) INTEGER
      !          The leading dimension of the array B.  LDB >= max(1,N).
      !
      !  INFO    (output) INTEGER
      !          = 0:  successful exit
      !          < 0:  if INFO = -i, the i-th argument had an illegal value
      !
      ! Call LAPACK routine
    end subroutine mth_dpotrs
  end interface
  !
  interface
    subroutine mth_dpbtrf (name, uplo, n, kd, ab, ldab, error)
      character(len=*) :: name             !
      character(len=*) :: uplo             !
      integer :: n                         !
      integer :: kd                        !
      integer :: ldab                      !
      real*8 :: ab(ldab,*)                 !
      logical :: error                     !
    end subroutine mth_dpbtrf
  end interface
  !
  interface
    subroutine mth_dpbtrs (name,   &
        uplo, n, kd, nrhs, ab, ldab, b, ldb, error)
      character(len=*) :: name             !
      character(len=*) :: uplo             !
      integer :: n                         !
      integer :: kd                        !
      integer :: nrhs                      !
      integer :: ldab                      !
      real*8 :: ab(ldab,*)                 !
      integer :: ldb                       !
      real*8 :: b(ldb,*)                   !
      logical :: error                     !
    end subroutine mth_dpbtrs
  end interface
  !
  interface
    function mth_bessj0 (x)
      !---------------------------------------------------------------------
      ! Compute Bessel function J0
      !---------------------------------------------------------------------
      real*8 :: mth_bessj0              !
      real*8 :: x                       !
    end function mth_bessj0
  end interface
  !
  interface
    function mth_bessj1 (x)
      !---------------------------------------------------------------------
      ! Compute Bessel function J1
      !---------------------------------------------------------------------
      real*8 :: mth_bessj1              !
      real*8 :: x                       !
    end function mth_bessj1
  end interface
  !
  interface
    subroutine mth_bessjn (x,j,n)
      !---------------------------------------------------------------------
      ! Compute Bessels functions from J1 to Jn
      !---------------------------------------------------------------------
      real*8 :: x                       !
      integer :: n                      !
      real*8 :: j(n)                    !
    end subroutine mth_bessjn
  end interface
  !
  interface
    subroutine mth_fail (fac,prog,ifail,error)
      !---------------------------------------------------------------------
      ! Error handling routine
      !---------------------------------------------------------------------
      character(len=*) :: fac           !
      character(len=*) :: prog          !
      integer :: ifail                  !
      logical :: error                  !
    end subroutine mth_fail
  end interface
  !
  interface
    function mth_bessj(n,x)
      !---------------------------------------------------------------------
      ! Compute Bessel function Jn, with n integer
      !---------------------------------------------------------------------
      real*8 :: mth_bessj               !
      integer :: n                      !
      real*8 :: x                       !
    end function mth_bessj
  end interface
  !
end module clic_interfaces_none
