module cubedag_index
  use gkernel_interfaces
  use cubetools_parameters
  use cubedag_parameters
  use cubedag_messaging
  use cubedag_types
  !
  integer(kind=entr_k), parameter :: cubedag_optimize_minalloc=100
  !
  public :: cubedag_optimize_to_optimize_next,cubedag_index_reallocate_expo,  &
            cubedag_index_reallocate
  private
  !
contains
  !
  subroutine cubedag_index_reallocate_expo(optx,mnodes,error)
    !---------------------------------------------------------------------
    !  Reallocate the 'optimize' type arrays. If current allocation is not
    ! enough, double the size of allocation. Since this reallocation
    ! routine is used in a context of adding more and more data, data
    ! is always preserved after reallocation.
    !---------------------------------------------------------------------
    type(cubedag_optimize_t), intent(inout) :: optx    !
    integer(kind=entr_k),     intent(in)    :: mnodes  ! Requested size
    logical,                  intent(inout) :: error   ! Logical error flag
    ! Local
    integer(kind=entr_k) :: nnodes
    !
    if (allocated(optx%object)) then
      nnodes = size(optx%object,kind=8)
      if (nnodes.ge.mnodes)  return           ! Enough size yet
      nnodes = 2_8*nnodes                     ! Request twice more place than before
      if (nnodes.lt.mnodes)  nnodes = mnodes  ! Twice is not enough, use mnodes
    else
      nnodes = max(mnodes,cubedag_optimize_minalloc)  ! No allocation yet, use mnodes
    endif
    !
    call cubedag_index_reallocate(optx,nnodes,.true.,error)
    if (error)  return
  end subroutine cubedag_index_reallocate_expo
  !
  subroutine cubedag_index_reallocate(optx,mnodes,keep,error)
    !---------------------------------------------------------------------
    !  Allocate the 'optimize' type arrays. Enlarge the arrays to the
    ! requested size, if needed. No shrink possible. Keep data if
    ! requested.
    !---------------------------------------------------------------------
    type(cubedag_optimize_t), intent(inout) :: optx    !
    integer(kind=entr_k),     intent(in)    :: mnodes  ! Requested size
    logical,                  intent(in)    :: keep    ! Keep previous data?
    logical,                  intent(inout) :: error   ! Logical error flag
    ! Local
    character(len=*), parameter :: rname='INDEX>REALLOCATE'
    integer(kind=4) :: ier
    integer(kind=entr_k) :: nnodes,inode
    integer(kind=4), allocatable :: bufi4(:)
    type(cubedag_node_pobject_t), allocatable :: bufno(:)
    !
    if (allocated(optx%object)) then
      nnodes = size(optx%object,kind=8)  ! Size of allocation
      if (nnodes.ge.mnodes) then
        ! Index is already allocated with a larger size. Keep it like this.
        ! Shouldn't we deallocate huge allocations if user requests a small one?
        return
      endif
    elseif (mnodes.eq.0) then
      ! No problem: can occur when dealing with empty files (e.g. nothing
      ! was written in an output file)
      return
    elseif (mnodes.lt.0) then
      call cubedag_message(seve%e,rname,'Can not allocate empty indexes')
      error = .true.
      return
    endif
    !
    nnodes = min(nnodes,optx%next-1)  ! Used part of the arrays
    if (keep) then
      allocate(bufi4(nnodes),stat=ier)
      allocate(bufno(nnodes),stat=ier)
      if (failed_allocate(rname,'buf arrays',ier,error)) then
        error = .true.
        return
      endif
    endif
    !
    call reallocate_optimize_no(rname,'object array',optx%object,mnodes,keep,bufno,error)
    if (error)  return
    call reallocate_optimize_i4(rname,'topomarker array',optx%topomarker,mnodes,keep,bufi4,error)
    if (error)  return
    !
    if (keep) then
      if (allocated(bufi4))  deallocate(bufi4)
      if (allocated(bufno))  deallocate(bufno)
    endif
    !
    ! Initialize the new components
    do inode=nnodes+1,mnodes
      call cubedag_optimize_init(optx,inode,error)
      if (error)  return
    enddo
  end subroutine cubedag_index_reallocate
  !
  subroutine reallocate_optimize_i4(rname,name,val,mnodes,keep,buf,error)
    character(len=*),     intent(in)    :: rname
    character(len=*),     intent(in)    :: name
    integer(kind=4),      allocatable   :: val(:)
    integer(kind=entr_k), intent(in)    :: mnodes
    logical,              intent(in)    :: keep
    integer(kind=4),      allocatable   :: buf(:)  ! Not allocated if keep is .false.
    logical,              intent(inout) :: error
    ! Local
    integer(kind=4) :: ier
    integer(kind=entr_k) :: nnodes
    !
    if (keep) then
      nnodes = size(buf)
      buf(:) = val(1:nnodes)
    endif
    if (allocated(val)) deallocate(val)
    allocate(val(mnodes),stat=ier)
    if (failed_allocate(rname,name,ier,error))  return
    if (keep) val(1:nnodes) = buf(:)
  end subroutine reallocate_optimize_i4
  !
  subroutine reallocate_optimize_no(rname,name,val,mnodes,keep,buf,error)
    character(len=*),             intent(in)    :: rname
    character(len=*),             intent(in)    :: name
    type(cubedag_node_pobject_t), allocatable   :: val(:)
    integer(kind=entr_k),         intent(in)    :: mnodes
    logical,                      intent(in)    :: keep
    type(cubedag_node_pobject_t), allocatable   :: buf(:)  ! Not allocated if keep is .false.
    logical,                      intent(inout) :: error
    ! Local
    integer(kind=4) :: ier
    integer(kind=entr_k) :: inode,nnodes
    !
    if (keep) then
      nnodes = size(buf)
      do inode=1,nnodes
        buf(inode)%p => val(inode)%p
      enddo
    endif
    if (allocated(val)) deallocate(val)
    allocate(val(mnodes),stat=ier)
    if (failed_allocate(rname,name,ier,error))  return
    if (keep) then
      do inode=1,nnodes
        val(inode)%p => buf(inode)%p
      enddo
    endif
  end subroutine reallocate_optimize_no
  !
  subroutine cubedag_optimize_to_optimize_next(in,i,out,error)
    !---------------------------------------------------------------------
    ! Copy the i-th element of the input optimize to the next available
    ! position in output optimize. It does not check if the optimize
    ! index is correctly allocated.
    !---------------------------------------------------------------------
    type(cubedag_optimize_t), intent(in)    :: in
    integer(kind=entr_k),     intent(in)    :: i
    type(cubedag_optimize_t), intent(inout) :: out
    logical,                  intent(inout) :: error
    !
    call cubedag_optimize_to_optimize_inplace(in,i,out,out%next,error)
    if (error)  return
    out%next = out%next+1
  end subroutine cubedag_optimize_to_optimize_next
  !
  subroutine cubedag_optimize_to_optimize_inplace(in,i,out,j,error)
    !---------------------------------------------------------------------
    ! Copy the i-th element of the input optimize to the j-th position
    ! in output optimize. It does not check if the optimize index is
    ! correctly allocated.
    !---------------------------------------------------------------------
    type(cubedag_optimize_t), intent(in)    :: in
    integer(kind=entr_k),     intent(in)    :: i
    type(cubedag_optimize_t), intent(inout) :: out
    integer(kind=entr_k),     intent(in)    :: j
    logical,                  intent(inout) :: error
    !
    out%object(j)%p   => in%object(i)%p
    out%topomarker(j) =  in%topomarker(i)
  end subroutine cubedag_optimize_to_optimize_inplace
  !
  subroutine cubedag_optimize_init(optx,i,error)
    !---------------------------------------------------------------------
    ! Initialize the i-th component in the index
    !---------------------------------------------------------------------
    type(cubedag_optimize_t), intent(inout) :: optx
    integer(kind=entr_k),     intent(in)    :: i
    logical,                  intent(inout) :: error
    !
    optx%object(i)%p  => null()
    optx%topomarker(i) = 0
  end subroutine cubedag_optimize_init

end module cubedag_index
