!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeio_types
  use image_def
  use gkernel_types
  use gkernel_interfaces
  use cubetools_parameters
  use cubetools_setup_types
  use cubetools_header_types
  use cubefitsio_header
  use cubeio_messaging
  use cubeio_cube_define
  use cubeio_block
  use cubeio_desc
  use cubeio_data
  !
  !--- Main CUBE type --------------------------------------------------
  !
  type cubeio_time_t
    type(cputime_t) :: total  ! Total execution time
    type(cputime_t) :: read   ! Reading data from disk
    type(cputime_t) :: writ   ! Writing data to disk
  end type cubeio_time_t
  !
  type :: cubeio_cube_t
    type(cubeio_desc_t)   :: desc   ! [public]  User friendly cube description
    type(gildas)          :: hgdf   ! [private] GDF header
    type(fitsio_header_t) :: hfits  ! [private] FITS header descriptor
    type(cubeio_block_t)  :: block  ! [private] Read bookkeeping
    type(cubeio_data_t)   :: data   ! [private] LMV or VLM placeholder
    type(cubeio_time_t)   :: time   ! [private] time measurements
  contains
    procedure :: order     => cubeio_get_order
    procedure :: access    => cubeio_get_access
    procedure :: iscplx    => cubeio_get_iscplx
    procedure :: haskind   => cubeio_has_filekind
    procedure :: nbytes    => cubeio_get_nbytes
    procedure :: nentry    => cubeio_get_nentry
    procedure :: ndata     => cubeio_get_ndata
    procedure :: size      => cubeio_get_size
    procedure :: planesize => cubeio_get_planesize
    procedure :: chansize  => cubeio_get_chansize
    procedure :: ysize     => cubeio_get_ysize
    procedure :: memsize   => cubeio_get_memsize
    procedure :: ready     => cubeio_data_ready
    procedure :: open      => cubeio_cube_open
    procedure :: close     => cubeio_cube_close
    procedure :: free      => cubeio_cube_free
  ! final     :: cubeio_cube_final  => crash if implicit (order matters?)
  end type cubeio_cube_t
  !
  ! --- Utilities ------------------------------------------------------
  !
  type cubeio_range_t
    integer(kind=index_length) :: blc(gdf_maxdims)
    integer(kind=index_length) :: trc(gdf_maxdims)
  end type cubeio_range_t
  !
contains
  !
  function cubeio_get_order(cub)
    integer(kind=code_k) :: cubeio_get_order
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_order = cub%desc%order
  end function cubeio_get_order
  !
  function cubeio_get_access(cub)
    integer(kind=code_k) :: cubeio_get_access
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_access = cub%desc%access
  end function cubeio_get_access
  !
  function cubeio_get_iscplx(cub)
    logical :: cubeio_get_iscplx
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_iscplx = cub%desc%iscplx
  end function cubeio_get_iscplx
  !
  function cubeio_has_filekind(cub,code_filekind)
    !-------------------------------------------------------------------
    ! Return .true. if the cube provides the given kind description
    !-------------------------------------------------------------------
    logical :: cubeio_has_filekind
    class(cubeio_cube_t), intent(in) :: cub
    integer(kind=code_k), intent(in) :: code_filekind
    select case (code_filekind)
    case (code_filekind_fits)
      ! Test if the hfits structure is used to describe some FITS header
      cubeio_has_filekind = cub%hfits%type.ne.code_null
    case (code_filekind_gdf)
      ! Test if the hgdf structure is used to describe some GDF header
      ! The GDF header provides both GDF description and some technical
      ! components around the file. Try to return .true. if the GDF
      ! description is filled independently of any file on disk.
      cubeio_has_filekind = cub%hgdf%gil%ndim.gt.0
    case default
      cubeio_has_filekind = .false.
    end select
  end function cubeio_has_filekind
  !
  function cubeio_get_nbytes(cub)
    integer(kind=4) :: cubeio_get_nbytes
    class(cubeio_cube_t), intent(in) :: cub
    if (cub%iscplx()) then
      cubeio_get_nbytes = 8
    else
      cubeio_get_nbytes = 4
    endif
  end function cubeio_get_nbytes
  !
  function cubeio_get_nentry(cub)
    integer(kind=entr_k) :: cubeio_get_nentry
    class(cubeio_cube_t), intent(in) :: cub
    select case (cub%desc%access)
    case (code_access_imaset)
      cubeio_get_nentry = cub%desc%nc
    case (code_access_speset)
      cubeio_get_nentry = cub%desc%nx*cub%desc%ny
    case (code_access_subset)
      cubeio_get_nentry = cub%desc%n3
    case (code_access_fullset)
      cubeio_get_nentry = cub%desc%n3
    case default
      cubeio_get_nentry = 0
    end select
  end function cubeio_get_nentry
  !
  function cubeio_get_ndata(cub)
    !-------------------------------------------------------------------
    ! Return the number of data values
    !-------------------------------------------------------------------
    integer(kind=data_k) :: cubeio_get_ndata
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_ndata = cub%desc%n1 * cub%desc%n2 * cub%desc%n3
  end function cubeio_get_ndata
  !
  function cubeio_get_size(cub)
    !-------------------------------------------------------------------
    ! Return the cube data size in bytes (allocated or not)
    !-------------------------------------------------------------------
    integer(kind=data_k) :: cubeio_get_size
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_size = cub%ndata()
    if (cub%desc%iscplx) then
      cubeio_get_size = cubeio_get_size*8
    else
      cubeio_get_size = cubeio_get_size*4
    endif
  end function cubeio_get_size
  !
  function cubeio_get_planesize(cub)
    !-------------------------------------------------------------------
    ! Return the one plane (1st x 2nd dimension) size in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_planesize
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_planesize = cub%desc%n1 * cub%desc%n2
    if (cub%desc%iscplx) then
      cubeio_get_planesize = cubeio_get_planesize*8
    else
      cubeio_get_planesize = cubeio_get_planesize*4
    endif
  end function cubeio_get_planesize
  !
  function cubeio_get_chansize(cub)
    !-------------------------------------------------------------------
    ! Return the one channel (2D plane) size in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_chansize
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_chansize = cub%desc%nx * cub%desc%ny
    if (cub%desc%iscplx) then
      cubeio_get_chansize = cubeio_get_chansize*8
    else
      cubeio_get_chansize = cubeio_get_chansize*4
    endif
  end function cubeio_get_chansize
  !
  function cubeio_get_ysize(cub)
    !-------------------------------------------------------------------
    ! Return the one Y row (2D plane) size in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_ysize
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_ysize = cub%desc%nx * cub%desc%nc
    if (cub%desc%iscplx) then
      cubeio_get_ysize = cubeio_get_ysize*8
    else
      cubeio_get_ysize = cubeio_get_ysize*4
    endif
  end function cubeio_get_ysize
  !
  function cubeio_get_memsize(cub)
    !-------------------------------------------------------------------
    ! Return the memory footprint in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_memsize
    class(cubeio_cube_t), intent(in) :: cub
    !
    cubeio_get_memsize =  &
      cub%block%memsize() +  &
      cub%data%memsize()
    !
  end function cubeio_get_memsize
  !
  function cubeio_data_ready(cub)
    !-------------------------------------------------------------------
    ! Return .true. if the data has been prepared for access (memory or
    ! disk). Data is prepared when invoking one of the cubeio_iterate_*
    ! subroutines.
    !-------------------------------------------------------------------
    logical :: cubeio_data_ready
    class(cubeio_cube_t), intent(in) :: cub
    !
    cubeio_data_ready = cub%data%ready.eq.cub%desc%buffered
  end function cubeio_data_ready
  !
  subroutine cubeio_cube_open(cub,error)
    use gkernel_interfaces
    !---------------------------------------------------------------------
    ! Open (or reopen) a 'cubeio_cube_t' instance for READ-ONLY
    ! access, if not yet opened.
    !---------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub    !
    logical,              intent(inout) :: error  !
    !
    character(len=*), parameter :: rname='CUBE>OPEN'
    !
    select case (cub%desc%filekind)
    case (code_filekind_fits)
      if (cub%hfits%unit.gt.0)  return  ! Already opened
      call cub%hfits%open(cub%desc%filename,'???',error)
      if (error)  return
    case (code_filekind_gdf)
      if (cub%hgdf%loca%islo.gt.0)  return  ! Already opened
      ! ZZZ Unfortunately, there is no symetric to gdf_close_image. For the
      ! time being, this re-reads the whole header...
      cub%hgdf%file = cub%desc%filename
      cub%hgdf%blc = 0
      cub%hgdf%trc = 0
      call gdf_read_header(cub%hgdf,error)
      if (error) return
    case default
      call cubeio_message(seve%e,rname,'No associated file on disk')
      error = .true.
      return
    end select
  end subroutine cubeio_cube_open
  !
  subroutine cubeio_cube_close(cub,error)
    use cubefitsio_image_utils
    !---------------------------------------------------------------------
    ! GIO-close a 'cubeio_cube_t' instance. This is worth calling this
    ! subroutine as GIO slots are a limited ressource
    !---------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub    !
    logical,              intent(inout) :: error  !
    ! Local
    character(len=*), parameter :: rname='CUBE>CLOSE'
    logical :: myerror
    !
    ! Free image slot
    myerror = .false.
    select case (cub%desc%filekind)
    case (code_filekind_fits)
      call cubefitsio_image_close(cub%hfits,myerror)
      if (myerror)  continue
    case (code_filekind_gdf)
      if (cub%hgdf%loca%islo.gt.0) then
        call gdf_close_image(cub%hgdf,myerror)
        if (myerror)  continue
      endif
    case default
      ! For a generic use, it is valid to "close" a cube with no
      ! associated file on disk.
      continue
    end select
    if (myerror)  error = .true.
  end subroutine cubeio_cube_close
  !
  subroutine cubeio_cube_free(cub,error)
    !---------------------------------------------------------------------
    ! Free the memory-consuming components of a 'cubeio_cube_t' instance.
    ! The cubeio_cube_t remains useable after this free. It is the
    ! responsibility of the caller to ensure the data remains available
    ! elsewhere (most likely on disk).
    ! Use cubeio_cube_final to free consistently all the object.
    !---------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub
    logical,              intent(inout) :: error
    !
    ! Beware!
    ! If this cube is memory-only, it should be completely discarded.
    ! If it has a file associated on disk, on the data in memory is deleted.
    call cubeio_block_free(cub%block,error)
    if (error)  return
    call cubeio_data_free(cub%data,error)
    if (error)  return
    if (cub%desc%buffered.eq.code_buffer_memory) then
      call cubeio_desc_reset(cub%desc,error)
      if (error)  return
    endif
  end subroutine cubeio_cube_free
  !
  subroutine cubeio_cube_final(cub)
    !---------------------------------------------------------------------
    ! Finalize a 'cubeio_cube_t' instance
    !---------------------------------------------------------------------
    type(cubeio_cube_t), intent(inout) :: cub    !
    ! Local
    logical :: error
    !
    error = .false.
    call cubeio_block_free(cub%block,error)
    if (error)  continue
    call cubeio_data_free(cub%data,error)
    if (error)  continue
    call cub%close(error)  ! Before cubeio_desc_final
    if (error)  continue
    call cubeio_desc_final(cub%desc) ! Explicit because above component rely on it
  end subroutine cubeio_cube_final
end module cubeio_types
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
