!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubefield_divergence
  use cube_types
  use cubetools_parameters
  use cubetools_structure
  use cubeadm_cubeid_types
  use cubefield_messaging
  ! 
  public :: cubefield_divergence_register,cubefield_divergence_command
  private
  !
  type :: divergence_comm_t
     type(option_t), pointer :: divergence  
   contains
     procedure, private :: parse => cubefield_divergence_parse
     procedure, private :: main  => cubefield_divergence_main
  end type divergence_comm_t
  type(divergence_comm_t) :: comm
  !
  integer(kind=4), parameter :: i_df_dx = 1
  integer(kind=4), parameter :: i_df_dy = 2
  type divergence_user_t
     type(cubeid_user_t) :: df_dx
     type(cubeid_user_t) :: df_dy
   contains
     procedure, private :: toprog => cubefield_divergence_user_toprog
  end type divergence_user_t
  !
  type divergence_prog_t
     type(cube_t), pointer :: df_dx
     type(cube_t), pointer :: df_dy
     type(cube_t), pointer :: divergence
     integer(kind=pixe_k)  :: nx = 0
     integer(kind=pixe_k)  :: idx = 0
     real(kind=coor_k)     :: dx = 0d0
     integer(kind=pixe_k)  :: ny = 0
     integer(kind=pixe_k)  :: idy = 0
     real(kind=coor_k)     :: dy = 0d0
   contains
     procedure, private :: header     => cubefield_divergence_prog_header
     procedure, private :: data       => cubefield_divergence_prog_data
     procedure, private :: loop       => cubefield_divergence_prog_loop
     procedure, private :: act        => cubefield_divergence_prog_act
  end type divergence_prog_t
  !
contains
  !
  subroutine cubefield_divergence_register(error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    logical, intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    character(len=*), parameter :: comm_abstract = 'Compute the divergence of a 2D spatial gradient'
    character(len=*), parameter :: comm_help = strg_id
    character(len=*), parameter :: rname='DIVERGENCE>REGISTER'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'DIVERGENCE','[df_dx df_dy]',&
         comm_abstract,&
         comm_help,&
         cubefield_divergence_command,&
         comm%divergence,error)
    if (error) return
    call cubearg%register(&
         'DF_DX',&
         'Gradient x component',&
         strg_id,&
         code_arg_optional,&
         [flag_dx,flag_gradient],&
         error)
    call cubearg%register(&
         'DF_DY',&
         'Gradient y component',&
         strg_id,&
         code_arg_optional,&
         [flag_dy,flag_gradient],&
         error)
    if (error) return
  end subroutine cubefield_divergence_register
  !
  subroutine cubefield_divergence_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(divergence_user_t) :: user
    character(len=*), parameter :: rname='DIVERGENCE>COMMAND'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call comm%parse(line,user,error)
    if (error) return
    call comm%main(user,error)
    if (error) continue
  end subroutine cubefield_divergence_command
  !
  subroutine cubefield_divergence_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    ! DIVERGENCE df_dx_id df_dy_id
    !----------------------------------------------------------------------
    class(divergence_comm_t), intent(in)    :: comm
    character(len=*),         intent(in)    :: line
    type(divergence_user_t),  intent(out)   :: user
    logical,                  intent(inout) :: error
    !
    character(len=*), parameter :: rname='DIVERGENCE>PARSE'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,comm%divergence,user%df_dx,error)
    if (error) return
    call cubeadm_cubeid_parse(line,comm%divergence,user%df_dy,error)
    if (error) return
  end subroutine cubefield_divergence_parse
  !
  subroutine cubefield_divergence_main(comm,user,error)    
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(divergence_comm_t), intent(in)    :: comm
    type(divergence_user_t),  intent(inout) :: user
    logical,                  intent(inout) :: error
    !
    type(divergence_prog_t) :: prog
    character(len=*), parameter :: rname='DIVERGENCE>MAIN'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubefield_divergence_main
  !
  !------------------------------------------------------------------------
  !
  subroutine cubefield_divergence_user_toprog(user,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(divergence_user_t), intent(in)    :: user
    type(divergence_prog_t),  intent(out)   :: prog
    logical,                  intent(inout) :: error
    !
    character(len=*), parameter :: rname='DIVERGENCE>USER>TOPROG'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(comm%divergence,i_df_dx,user%df_dx,&
         code_access_imaset,code_read,prog%df_dx,error)
    if (error) return
    call cubeadm_cubeid_get_header(comm%divergence,i_df_dy,user%df_dy,&
         code_access_imaset,code_read,prog%df_dy,error)
    if (error) return
  end subroutine cubefield_divergence_user_toprog
  !
  !------------------------------------------------------------------------
  !
  subroutine cubefield_divergence_prog_header(prog,error)
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(divergence_prog_t), intent(inout) :: prog
    logical,                  intent(inout) :: error
    !
    character(len=*), parameter :: rname='DIVERGENCE>PROG>HEADER'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    ! *** JP: there should be a check that the input cubes are consistent
    call cubeadm_clone_header(prog%df_dx,[flag_divergence],&
         prog%divergence,error)
    if (error) return
    call cubefield_divergence_prog_header_axis(prog%divergence,&
         prog%divergence%head%set%il,prog%nx,prog%idx,prog%dx,error)
    if (error) return
    call cubefield_divergence_prog_header_axis(prog%divergence,&
         prog%divergence%head%set%im,prog%ny,prog%idy,prog%dy,error)
    if (error) return
  end subroutine cubefield_divergence_prog_header
  !
  subroutine cubefield_divergence_prog_header_axis(cube,iaxis,nx,idx,dx,error)
    use phys_const
    use cubetools_axis_types
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    type(cube_t), pointer, intent(inout) :: cube
    integer(kind=ndim_k),  intent(in)    :: iaxis
    integer(kind=pixe_k),  intent(inout) :: nx
    integer(kind=pixe_k),  intent(inout) :: idx
    real(kind=coor_k),     intent(inout) :: dx
    logical,               intent(inout) :: error
    !
    type(axis_t) :: axis
    character(len=*), parameter :: rname='DIVERGENCE>PROG>HEADER>AXIS'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubetools_header_get_axis_head(iaxis,cube%head,axis,error)
    if (error) return
    nx = axis%n
    idx = 1 ! *** JP: should be customizable depending on the angular resolution
    dx = (2*idx)*axis%inc*sec_per_rad ! *** JP: should use the current angle unit
  end subroutine cubefield_divergence_prog_header_axis
  !
  subroutine cubefield_divergence_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(divergence_prog_t), intent(inout) :: prog
    logical,                  intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='DIVERGENCE>PROG>DATA'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error)  &
         call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubefield_divergence_prog_data
  !   
  subroutine cubefield_divergence_prog_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(divergence_prog_t), intent(inout) :: prog
    integer(kind=entr_k),     intent(in)    :: first
    integer(kind=entr_k),     intent(in)    :: last
    logical,                  intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: df_dx,df_dy,divergence
    character(len=*), parameter :: rname='DIVERGENCE>PROG>LOOP'
    !
    call df_dx%init(prog%df_dx,error)
    if (error) return
    call df_dy%init(prog%df_dy,error)
    if (error) return
    call divergence%reallocate('divergence',prog%divergence%head%arr%n%l,prog%divergence%head%arr%n%m,error)
    if (error) return
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act(ie,df_dx,df_dy,divergence,error)
      if (error) return
    enddo
  end subroutine cubefield_divergence_prog_loop
  !   
  subroutine cubefield_divergence_prog_act(prog,ie,df_dx,df_dy,divergence,error)
    use cubetools_nan
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(divergence_prog_t), intent(inout) :: prog
    integer(kind=entr_k),     intent(in)    :: ie
    type(image_t),            intent(inout) :: df_dx
    type(image_t),            intent(inout) :: df_dy
    type(image_t),            intent(inout) :: divergence
    logical,                  intent(inout) :: error
    !
    real(kind=sign_k) :: ddf_dxdx,ddf_dydy
    integer(kind=pixe_k) :: ix,iy
    character(len=*), parameter :: rname='DIVERGENCE>PROG>ACT'
    !
    call df_dx%get(prog%df_dx,ie,error)
    if (error)  return
    call df_dy%get(prog%df_dy,ie,error)
    if (error)  return
    divergence%z(:,:) = gr4nan
    do iy=prog%idy+1,prog%nx-prog%idy
       do ix=prog%idx+1,prog%ny-prog%idx
          ddf_dxdx = (df_dx%z(ix+prog%idx,iy)-df_dx%z(ix-prog%idx,iy))/prog%dx
          ddf_dydy = (df_dy%z(ix,iy+prog%idy)-df_dy%z(ix,iy-prog%idy))/prog%dy
          divergence%z(ix,iy) = ddf_dxdx+ddf_dydy
       enddo ! ix
    enddo ! iy
    call divergence%put(prog%divergence,ie,error)
    if (error)  return
  end subroutine cubefield_divergence_prog_act
end module cubefield_divergence
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
