!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_hyperdetect
  use cube_types
  use cubetools_structure
  use cubeadm_cubeid_types
  use cubemain_sperange_types
  use cubemain_messaging
  !
  public :: hyperdetect
  public :: cubemain_hyperdetect_command
  private
  !
  type :: hyperdetect_comm_t
     type(option_t), pointer :: comm  
     type(option_t), pointer :: islands
     type(sperange_opt_t)    :: sperange
   contains
     procedure, public  :: register => cubemain_hyperdetect_register
     procedure, private :: parse    => cubemain_hyperdetect_parse
     procedure, private :: main     => cubemain_hyperdetect_main
  end type hyperdetect_comm_t
  type(hyperdetect_comm_t) :: hyperdetect
  !
  integer(kind=4), parameter :: icube = 1
  integer(kind=4), parameter :: ilabeled = 2
  type hyperdetect_user_t
     type(cubeid_user_t)   :: cubeids
     type(sperange_user_t) :: sperange
     character(len=argu_l) :: islrange(2)
   contains
     procedure, private :: toprog => cubemain_hyperdetect_user_toprog
  end type hyperdetect_user_t
  !
  type hyperdetect_prog_t
     type(sperange_prog_t) :: sperange
     integer(kind=data_k)  :: islrange(2)
     type(cube_t), pointer :: cube
     type(cube_t), pointer :: labeled
     type(cube_t), pointer :: detected
     type(cube_t), pointer :: residuals
   contains
     procedure, private :: header => cubemain_hyperdetect_prog_header
     procedure, private :: data   => cubemain_hyperdetect_prog_data
     procedure, private :: loop   => cubemain_hyperdetect_prog_loop
     procedure, private :: act    => cubemain_hyperdetect_prog_act
  end type hyperdetect_prog_t
  !
contains
  !
  subroutine cubemain_hyperdetect_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(hyperdetect_user_t) :: user
    character(len=*), parameter :: rname='HYPERDETECT>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call hyperdetect%parse(line,user,error)
    if (error) return
    call hyperdetect%main(user,error)
    if (error) continue
  end subroutine cubemain_hyperdetect_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_hyperdetect_register(hyperdetect,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(hyperdetect_comm_t), intent(inout) :: hyperdetect
    logical,                   intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract='Spatially split a cube into detected signal, and residuals'
    character(len=*), parameter :: comm_help = &
         'Select from a segmented SNR cube a segment range and create a &
         &cube of detected signal inside the corresponding island mask. &
         &The remainder is output into a residual cube.'
    character(len=*), parameter :: rname='HYPERDETECT>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'DETECT','[cube]',&
         comm_abstract,&
         comm_help,&
         cubemain_hyperdetect_command,&
         hyperdetect%comm,error)
    if (error) return
    call cubearg%register(&
         'SIGNAL',&
         'Signal cube',&
         strg_id,&
         code_arg_optional,&
         [flag_cube],&
         error)
    if (error) return
    call cubearg%register(&
         'SEGMENTS',&
         'Segment labels',&
         strg_id,&
         code_arg_optional,&
         [flag_segments],&
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'SEGMENTS','first last',&
         'First and last segments to form the detection mask',&
         strg_id,&
         hyperdetect%islands,error)
    if (error) return
    call stdarg%register(&
         'first',&
         'first',&
         'Default to 1',&
         code_arg_mandatory, error)
    if (error) return
    call stdarg%register(&
         'last',&
         'last',&
         'Default to the number of segments minus 1',&
         code_arg_mandatory, error)
    if (error) return
    !
    call hyperdetect%sperange%register(&
         'RANGE',&
         'Set the velocity range of the spectral region of interest',&
         error)
    if (error) return
  end subroutine cubemain_hyperdetect_register
  !
  subroutine cubemain_hyperdetect_parse(hyperdetect,line,user,error)
    !----------------------------------------------------------------------
    ! HYPERDETECT cubeid /ISLANDS first last
    !----------------------------------------------------------------------
    class(hyperdetect_comm_t), intent(in)    :: hyperdetect
    character(len=*),     intent(in)    :: line
    type(hyperdetect_user_t),  intent(out)   :: user
    logical,              intent(inout) :: error
    !
    character(len=*), parameter :: rname='HYPERDETECT>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,hyperdetect%comm,user%cubeids,error)
    if (error) return
    call cubemain_hyperdetect_parse_twoarg(line,hyperdetect%islands,user%islrange,error)
    if (error) return
    call hyperdetect%sperange%parse(line,user%sperange,error)
    if (error) return
  end subroutine cubemain_hyperdetect_parse
  !
  subroutine cubemain_hyperdetect_parse_twoarg(line,opt,args,error)
    !----------------------------------------------------------------------
    ! /OPT first last
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    type(option_t),   intent(in)    :: opt
    character(len=*), intent(out)   :: args(2)
    logical,          intent(inout) :: error
    !
    logical :: present
    integer(kind=4) :: iarg
    character(len=*), parameter :: rname='HYPERDETECT>PARSE>TWOARG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call opt%present(line,present,error)
    if (error) return
    if (present) then
       do iarg=1,2
          call cubetools_getarg(line,opt,iarg,args(iarg),mandatory,error)
          if (error) return
       enddo ! iarg
    else
       do iarg=1,2
          args(iarg) = strg_star
       enddo ! iarg
    endif
  end subroutine cubemain_hyperdetect_parse_twoarg
  !
  subroutine cubemain_hyperdetect_main(hyperdetect,user,error)    
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(hyperdetect_comm_t), intent(in)    :: hyperdetect
    type(hyperdetect_user_t),  intent(inout) :: user
    logical,              intent(inout) :: error
    !
    type(hyperdetect_prog_t) :: prog
    character(len=*), parameter :: rname='HYPERDETECT>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_hyperdetect_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_hyperdetect_user_toprog(user,prog,error)
    use cubetools_arrelt_types
    use cubetools_header_methods    
    use cubetools_user2prog
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(hyperdetect_user_t), intent(in)    :: user
    type(hyperdetect_prog_t),  intent(out)   :: prog
    logical,              intent(inout) :: error
    !
    integer(kind=4) :: iarg
    type(arrelt_t) :: min,max
    integer(kind=data_k) :: default(2)
    character(len=*), parameter :: rname='HYPERDETECT>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(hyperdetect%comm,icube,user%cubeids,&
         code_access_imaset,code_read,prog%cube,error)
    if (error) return
    call cubeadm_cubeid_get_header(hyperdetect%comm,ilabeled,user%cubeids,&
         code_access_imaset,code_read,prog%labeled,error)
    if (error) return
    !
    call cubetools_header_get_array_minmax(prog%labeled%head,min,max,error)
    if (error) return
    default(1) = 1
    default(2) = max%val-1
    do iarg=1,2
       call cubetools_user2prog_resolve_star(user%islrange(iarg),default(iarg),default(iarg),error)
       if (error) return
    enddo ! iarg
    prog%islrange(1) = max%val-default(2)+1
    prog%islrange(2) = max%val-default(1)+1
    !
    call user%sperange%toprog(prog%cube,prog%sperange,error)
    if (error) return
  end subroutine cubemain_hyperdetect_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_hyperdetect_prog_header(prog,error)
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(hyperdetect_prog_t), intent(inout) :: prog
    logical,              intent(inout) :: error
    !
    character(len=*), parameter :: rname='HYPERDETECT>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%cube,flag_detected,prog%detected,error)
    if (error) return
    call cubeadm_clone_header(prog%cube,flag_residuals,prog%residuals,error)
    if (error) return
  end subroutine cubemain_hyperdetect_prog_header
  !
  subroutine cubemain_hyperdetect_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(hyperdetect_prog_t), intent(inout) :: prog
    logical,              intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iterentry
    character(len=*), parameter :: rname='HYPERDETECT>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iterentry,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iterentry)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iterentry,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iterentry,error)
       if (.not.error)  &
         call prog%loop(iterentry%first,iterentry%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_hyperdetect_prog_data
  !   
  subroutine cubemain_hyperdetect_prog_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(hyperdetect_prog_t), intent(inout) :: prog
    integer(kind=entr_k),      intent(in)    :: first
    integer(kind=entr_k),      intent(in)    :: last
    logical,                   intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(image_t) :: cube,labeled,detected,residuals
    character(len=*), parameter :: rname='HYPERDETECT>PROG>LOOP'
    !
    call cube%init(prog%cube,error)
    if (error) return
    call labeled%init(prog%labeled,error)
    if (error) return
    call detected%reallocate('detected',prog%detected%head%arr%n%l,prog%detected%head%arr%n%m,error)
    if (error) return
    call residuals%reallocate('residuals',prog%residuals%head%arr%n%l,prog%residuals%head%arr%n%m,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act(ie,cube,labeled,detected,residuals,error)
      if (error) return
    enddo
  end subroutine cubemain_hyperdetect_prog_loop
  !   
  subroutine cubemain_hyperdetect_prog_act(prog,ie,cube,labeled,detected,residuals,error)
    use cubetools_nan
    use cubemain_image_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(hyperdetect_prog_t), intent(inout) :: prog
    integer(kind=entr_k),      intent(in)    :: ie
    type(image_t),             intent(inout) :: cube
    type(image_t),             intent(inout) :: labeled
    type(image_t),             intent(inout) :: detected
    type(image_t),             intent(inout) :: residuals
    logical,                   intent(inout) :: error
    !
    integer(kind=data_k) :: ix,nx
    integer(kind=data_k) :: iy,ny
    character(len=*), parameter :: rname='HYPERDETECT>PROG>ACT'
    !
    call cube%get(prog%cube,ie,error)
    if (error)  return
    call labeled%get(prog%labeled,ie,error)
    if (error)  return
    !
    nx = cube%nx
    ny = cube%ny
    if ((prog%sperange%p(1).le.ie).and.(ie.le.prog%sperange%p(2))) then
       do iy=1,ny
          do ix=1,nx
             if (ieee_is_nan(cube%z(ix,iy))) then
                detected%z(ix,iy)  = gr4nan
                residuals%z(ix,iy) = gr4nan
             else
                if ((prog%islrange(1).le.labeled%z(ix,iy)).and.&
                     (labeled%z(ix,iy).le.prog%islrange(2))) then
                   detected%z(ix,iy)  = cube%z(ix,iy)
                   residuals%z(ix,iy) = 0
                else
                   detected%z(ix,iy)  = gr4nan
                   residuals%z(ix,iy) = cube%z(ix,iy)
                endif
             endif
          enddo ! ix
       enddo ! iy
    else
       do iy=1,ny
          do ix=1,nx
             detected%z(ix,iy)  = 0
             residuals%z(ix,iy) = cube%z(ix,iy)
          enddo ! ix
       enddo ! iy
    endif
    call detected%put(prog%detected,ie,error)
    if (error)  return
    call residuals%put(prog%residuals,ie,error)
    if (error)  return
  end subroutine cubemain_hyperdetect_prog_act
end module cubemain_hyperdetect
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
