/*
 * G-ToDo Gkrellm Plugin
 * Copyright 2003-2004 Dimitar Haralanov
 *
 */
/* This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <quick_view.h>
#include <support.h>
#include <config.h>
#include <cellrendererspin.h>
#include <gtodo.h>
#include <debug.h>

static GtkWidget     *detail_window;
static GtkWidget     *action_button_comment;
static GtkWidget     *task_treeview;
static GtkTextBuffer *text_buffer;

/* Quick View Window Callback Prototypes */
static void on_tree_view_selection_changed   (GtkTreeSelection *, gpointer);
static void on_action_button_add_clicked     (GtkButton *, gpointer);
static void on_action_button_remove_clicked  (GtkButton *, gpointer);
static void on_action_button_comment_clicked (GtkButton *, gpointer);
static void on_action_button_ok_clicked      (GtkButton *, gpointer);
static void on_action_button_cancel_clicked  (GtkButton *, gpointer);
static void on_active_toggled                (GtkCellRendererToggle *, gchar *, gpointer);
static void quick_view_deadline_edited       (GtkCellRendererText *, gchar *, gchar *, gpointer);
static void quick_view_complete_edited       (GtkCellRendererText *, gchar *, gchar *, gpointer);
static void quick_view_comment_edited        (GtkCellRendererText *, gchar *, gchar *, gpointer);

/* Function prototypes for the comment window */
static void create_comment_window            (GtkListStore *, GtkTreeIter);
static void on_comment_ok_button_clicked     (GtkButton *, gpointer);
static void on_comment_cancel_button_clicked (GtkButton *, gpointer);

GtkWidget * gtodo_create_quick_view (GkrellmMonitor *mon, GtkTreeIter *iter) {
     GtkWidget *quick_view_window;
     GtkWidget *quick_view_vbox;
     GtkWidget *quick_view_task_frame;
     GtkWidget *task_scrolledwindow;
     GtkWidget *quick_view_hbutton_box;
     GtkWidget *action_button_add;
     GtkWidget *action_button_remove;
     GtkWidget *action_button_comment_image;
     GtkWidget *action_button_comment_label;
     GtkWidget *button_alignment;
     GtkWidget *button_hbox;
     GtkWidget *action_button_ok;
     GtkWidget *action_button_cancel;
     GtkCellRenderer *renderer;
     GtkTreeViewColumn *column;
     GtkTreeSelection *selection;
     GtkTreeSortable  *sortmodel;

     quick_view_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title (GTK_WINDOW (quick_view_window), _("G-ToDo Quick View"));
     gtk_window_resize (GTK_WINDOW (quick_view_window), 670, 270);
     
     quick_view_vbox = gtk_vbox_new (FALSE, 3);
     gtk_container_add (GTK_CONTAINER (quick_view_window), quick_view_vbox);

     /* we can pack everything into a frame and that way it will look
      * much nicer */
     quick_view_task_frame = gtk_frame_new (_("Current Tasks"));
     gtk_box_pack_start (GTK_BOX (quick_view_vbox), quick_view_task_frame, TRUE, TRUE, 0);
     gtk_frame_set_label_align (GTK_FRAME (quick_view_task_frame), 0.03, 0.5);
     gtk_container_border_width (GTK_CONTAINER (quick_view_task_frame), 3);
     
     /* now that the tree model is create, we can go ahead and create the
      * tree view */
     task_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
     gtk_container_set_border_width (GTK_CONTAINER (task_scrolledwindow), 3);
     gtk_container_add (GTK_CONTAINER (quick_view_task_frame), task_scrolledwindow);
     gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (task_scrolledwindow),
				     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     
     /* setup to sorting stuff... */
     sortmodel = GTK_TREE_SORTABLE (task_list_store);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_TASK,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_TASK),
				      NULL);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_COMPLETE,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_COMPLETE),
				      NULL);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_DEADLINE,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_DEADLINE),
				      NULL);
     gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (sortmodel), SORT_ID_TIME,
				      gtodo_list_store_sort, GINT_TO_POINTER (SORT_ID_TIME),
				      NULL);

     task_treeview = gtk_tree_view_new_with_model ((GtkTreeModel *)task_list_store);
     gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (task_treeview), TRUE);
     gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (task_treeview), TRUE);
     gtk_container_add (GTK_CONTAINER (task_scrolledwindow), task_treeview);

     /* this button has to be declared here because we need it for one of the
      * renderer callback, however, it gets filled out later... */
     action_button_comment = gtk_button_new ();

     /* ok... lets add the columns... */
     
     /* define all the cell renderers - they will take the value from the
      * sortmodel and convert it to the appropriate widget with its value
      * set */
     renderer = gtk_cell_renderer_toggle_new ();
     gtk_cell_renderer_toggle_set_radio (GTK_CELL_RENDERER_TOGGLE (renderer), TRUE);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Active"),
						  renderer, "active", ACTIVE_COLUMN, NULL);
     g_signal_connect (G_OBJECT (renderer), "toggled", G_CALLBACK (on_active_toggled), action_button_comment);
     
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Task"),
						  renderer, "text", TITLE_COLUMN, NULL);

     renderer = gui_cell_renderer_spin_new (0, 100, 1, 10, 10, 1, 0);
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Complete"),
						  renderer, "text", COMPLETE_COLUMN,
						  "editable", ACTIVE_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 2);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_complete_column, NULL, NULL);
     g_signal_connect (G_OBJECT (renderer), "edited", G_CALLBACK (quick_view_complete_edited), NULL);
     
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Deadline"),
						  renderer, "text", DEADLINE_COLUMN,
						  "editable", ACTIVE_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 3);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_date_column, NULL, NULL);
     g_signal_connect (G_OBJECT (renderer), "edited", G_CALLBACK (quick_view_deadline_edited),
		       quick_view_window);

     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Remind At"),
						  renderer, "text", REMINDER_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 4);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_date_column, NULL, NULL);
     
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Time Spent"),
						  renderer, "text", TIME_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 5);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_time_column, NULL, NULL);
     
     renderer = gtk_cell_renderer_text_new ();
     gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (task_treeview), -1, _("Comments"),
						  renderer, "text", COMMENT_COLUMN, NULL);
						  //"editable", ACTIVE_COLUMN, NULL);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 6);
     gtk_tree_view_column_set_cell_data_func (column, renderer, display_comment_column, NULL, NULL);
     g_signal_connect (G_OBJECT (renderer), "edited", G_CALLBACK (quick_view_comment_edited), NULL);

     /* setup column sort ids */
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 1);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_TASK);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 2);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_COMPLETE);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 3);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_DEADLINE);
     column = gtk_tree_view_get_column (GTK_TREE_VIEW (task_treeview), 5);
     gtk_tree_view_column_set_sort_column_id (column, SORT_ID_TIME);
     
     /* if the passed itertation is not NULL, select the row that it
      * corresponds to
      * this happens when the "Reschedule" button of the Reminder window
      * is clicked */
     selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (task_treeview));
     if (iter != NULL) {
	  /* we have to show the widget first, otherwise the rows won't
	   * be filled and the selection will not happen */
	  gtk_widget_show (GTK_WIDGET (quick_view_window));
	  gtk_tree_selection_select_iter (GTK_TREE_SELECTION (selection), iter);
     }
     
     /* create the bottom action button box */
     quick_view_hbutton_box = gtk_hbutton_box_new ();
     gtk_box_pack_start (GTK_BOX (quick_view_vbox), quick_view_hbutton_box, FALSE, TRUE, 0);
     gtk_container_set_border_width (GTK_CONTAINER (quick_view_hbutton_box), 3);
     gtk_button_box_set_layout (GTK_BUTTON_BOX (quick_view_hbutton_box), GTK_BUTTONBOX_END);
     gtk_box_set_spacing (GTK_BOX (quick_view_hbutton_box), 5);
     
     action_button_add = gtk_button_new_from_stock ("gtk-add");
     gtk_container_add (GTK_CONTAINER (quick_view_hbutton_box), action_button_add);

     action_button_remove = gtk_button_new_from_stock ("gtk-remove");
     gtk_container_add (GTK_CONTAINER (quick_view_hbutton_box), action_button_remove);
     gtk_widget_set_sensitive (GTK_WIDGET (action_button_remove),
			       gtk_tree_selection_get_selected (GTK_TREE_SELECTION (selection),
								NULL, NULL));
     /* This belongs to the comment_button which was declared above... */
     button_alignment = gtk_alignment_new (0.5, 0.5, 0, 0);
     gtk_container_add (GTK_CONTAINER (action_button_comment), button_alignment);
     
     button_hbox = gtk_hbox_new (FALSE, 2);
     gtk_container_add (GTK_CONTAINER (button_alignment), button_hbox);
     
     action_button_comment_image = gtk_image_new_from_stock ("gtk-justify-fill", GTK_ICON_SIZE_BUTTON);
     gtk_box_pack_start (GTK_BOX (button_hbox), action_button_comment_image, FALSE, FALSE, 0);
     
     action_button_comment_label = gtk_label_new_with_mnemonic (_("View"));
     gtk_box_pack_start (GTK_BOX (button_hbox), action_button_comment_label, FALSE, FALSE, 0);
     gtk_label_set_justify (GTK_LABEL (action_button_comment_label), GTK_JUSTIFY_LEFT);
     gtk_container_add (GTK_CONTAINER (quick_view_hbutton_box), action_button_comment);
     gtk_widget_set_sensitive (GTK_WIDGET (action_button_comment),
			       gtk_tree_selection_get_selected (GTK_TREE_SELECTION (selection),
								NULL, NULL));
     /* we are done with the comment button */
     
     action_button_ok = gtk_button_new_from_stock ("gtk-ok");
     gtk_container_add (GTK_CONTAINER (quick_view_hbutton_box), action_button_ok);
     GTK_WIDGET_SET_FLAGS (action_button_ok, GTK_CAN_DEFAULT);
     
     action_button_cancel = gtk_button_new_from_stock ("gtk-cancel");
     gtk_container_add (GTK_CONTAINER (quick_view_hbutton_box), action_button_cancel);

     g_signal_connect (G_OBJECT (selection), "changed",
		       G_CALLBACK (on_tree_view_selection_changed),
		       action_button_remove);
     
     /* connect the buttons to the callbacks
      * here we connect only the buttons from the hbuttonbox
      * the column renderers were connected above */
     g_signal_connect (G_OBJECT (quick_view_window), "destroy",
		       G_CALLBACK (gtk_widget_destroy),
		       NULL);
     g_signal_connect (G_OBJECT (task_treeview), "scroll-event",
		       G_CALLBACK (gtodo_mouse_wheel_scrolled),
		       NULL);
     g_signal_connect (G_OBJECT (action_button_add), "clicked",
		       G_CALLBACK (on_action_button_add_clicked),
		       mon);
     g_signal_connect (G_OBJECT (action_button_remove), "clicked",
		       G_CALLBACK (on_action_button_remove_clicked),
		       NULL);
     g_signal_connect (G_OBJECT (action_button_comment), "clicked",
		       G_CALLBACK (on_action_button_comment_clicked),
		       task_treeview);
     g_signal_connect (G_OBJECT (action_button_ok), "clicked",
		       G_CALLBACK (on_action_button_ok_clicked),
		       quick_view_window);
     g_signal_connect (G_OBJECT (action_button_cancel), "clicked",
		       G_CALLBACK (on_action_button_cancel_clicked),
		       quick_view_window);
     
     return quick_view_window;
}

void gtodo_quick_view_set_comment_button (GtkTreeIter *iter) {
     GtkTreeSelection *selection;
     gboolean active;

     /* check whether task_treeview points to a GtkTreeView
      * because once we close the quick view window, it could
      * point to anything...
      * This only matters when the quick view window is not open when
      * this function is called (which happens often). Otherwise, the
      * create_quick_view_window function would have assigned a GtkTreeView
      * to the variable. */
     if (GTK_IS_TREE_VIEW (task_treeview)) {
	  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (task_treeview));
	  DPRINTF ("i am here");
	  gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), iter, ACTIVE_COLUMN, &active, -1);
	  DPRINTF ("got this far");
	  if (active &&
	      gtk_tree_selection_iter_is_selected (GTK_TREE_SELECTION (selection), iter)) {
	       gtk_widget_set_sensitive (GTK_WIDGET (action_button_comment), TRUE);
	  } else
	       gtk_widget_set_sensitive (GTK_WIDGET (action_button_comment), FALSE);
     }
}

/* if there is a valid row selected, make the comment-view button
 * active */
static void on_tree_view_selection_changed (GtkTreeSelection *selection, gpointer data) {
     GtkTreeIter iter;

     if (gtk_tree_selection_get_selected (GTK_TREE_SELECTION (selection), NULL, &iter)) {
	  gtodo_quick_view_set_comment_button (&iter);
	  gtk_widget_set_sensitive (GTK_WIDGET (data), TRUE);
     } else
	  gtk_widget_set_sensitive (GTK_WIDGET (data), FALSE);
}

static void on_action_button_add_clicked (GtkButton *button, gpointer data) {
     gkrellm_open_config_window ((GkrellmMonitor *)data);
}

static void on_action_button_remove_clicked (GtkButton *button, gpointer data) {
     GtkTreeIter iter;
     GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (task_treeview));

     if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
	  GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (task_treeview));
	  gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
	  global_config.clean = FALSE;
     }
}

static void on_action_button_comment_clicked (GtkButton *button, gpointer data) {
     GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (data));
     GtkTreeSelection *selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (data));
     GtkTreeIter iter;

     if (gtk_tree_selection_get_selected (selection, NULL, &iter)) {
	  if (detail_window)
	       gtk_widget_destroy (GTK_WIDGET (detail_window));
	  create_comment_window (GTK_LIST_STORE (model), iter);
     }
}

static void on_action_button_ok_clicked (GtkButton *button, gpointer data) {
     if (detail_window) 
	  gtk_widget_destroy (GTK_WIDGET (detail_window));
     gtk_widget_destroy (GTK_WIDGET (data));
     global_config.window = FALSE;
}

static void on_action_button_cancel_clicked (GtkButton *button, gpointer data) {
     if (detail_window)
	  gtk_widget_destroy (GTK_WIDGET (detail_window));
     gtk_widget_destroy (GTK_WIDGET (data));
     global_config.window = FALSE;
}


/* This callback is called when any one of the toggle buttons in the Active
 * column gets clicked. The interesting thing is that the toggle buttons don't have
 * the usual toggled signals and radio group like normal radio buttons.
 * That is why here we have to manually switch the values of all the toggle buttons
 * so we select the correct one
 */
static void on_active_toggled (GtkCellRendererToggle *renderer, gchar *arg, gpointer data) {
     GtkTreePath *path;
     GtkTreeIter iter;
     GtkTreeIter walker;
     gboolean valid;
     gboolean active;
     
     //path = gtk_tree_model_sort_convert_path_to_child_path (GTK_TREE_MODEL_SORT (task_list_store),
     //gtk_tree_path_new_from_string (arg));
     path = gtk_tree_path_new_from_string (arg);
     if (path) DPRINTF ("path %s", arg);
     gtk_tree_model_get_iter (GTK_TREE_MODEL (task_list_store), &iter, path);

     valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (task_list_store), &walker);
     while (valid) {
	  if (!memcmp (&walker, &iter, sizeof (GtkTreeIter))) {
	       gchar *t;
	       glong d;
	       gint c;
	       gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), &walker, ACTIVE_COLUMN, &active,
				   TITLE_COLUMN, &t, DEADLINE_COLUMN, &d, COMPLETE_COLUMN, &c, -1);
	       if (active) {
		    gtodo_set_panel_tooltip (NULL, -1, -1);
		    gtk_list_store_set (GTK_LIST_STORE (task_list_store), &walker, ACTIVE_COLUMN, FALSE, -1);
		    global_config.have_active = FALSE;
	       } else {
		    gtodo_set_panel_tooltip (t, d, c);
		    gtk_list_store_set (GTK_LIST_STORE (task_list_store), &walker, ACTIVE_COLUMN, TRUE, -1);
		    global_config.have_active = TRUE;
	       }
	       g_free (t);
	  } else {
	       gtk_list_store_set (GTK_LIST_STORE (task_list_store), &walker, ACTIVE_COLUMN, FALSE, -1);
	  }
	  valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (task_list_store), &walker);
     }
     //gtk_tree_model_sort_convert_child_iter_to_iter (GTK_TREE_MODEL_SORT (sortmodel), &child_iter, &iter);
     gtodo_quick_view_set_comment_button (&iter);
}

static void quick_view_deadline_edited (GtkCellRendererText *renderer, gchar *path, gchar *value, gpointer data) {
     GtkTreePath *p = gtk_tree_path_new_from_string (path);
     GtkTreeIter iter;
     glong reminder;
     glong deadline = gtodo_get_time_from_date_str (value);

     if (deadline == -1) {
	  alert (GTK_WIDGET (data),
		 _("Invalid date/time format while editing\nQuick View task entry!\n"));
	  return;
     }
     reminder = (deadline - global_config.start_interval * 60);
     
     gtk_tree_model_get_iter (GTK_TREE_MODEL (task_list_store), &iter, p);
     gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter, DEADLINE_COLUMN, deadline,
			 REMINDER_COLUMN, reminder, -1);
     global_config.clean = FALSE;
}

static void quick_view_complete_edited (GtkCellRendererText *renderer, gchar *path, gchar *value, gpointer data) {
     GtkTreePath *p = gtk_tree_path_new_from_string (path);
     GtkTreeIter iter;
     gint perc = atoi (value);
     
     if (perc < 0) perc = 0;
     if (perc > 100) perc = 100;
     
     gtk_tree_model_get_iter (GTK_TREE_MODEL (task_list_store), &iter, p);
     gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter, COMPLETE_COLUMN, perc, -1);
     global_config.clean = FALSE;
}

static void quick_view_comment_edited (GtkCellRendererText *renderer, gchar *path, gchar *value, gpointer dada) {
     GtkTreePath *p = gtk_tree_path_new_from_string (path);
     GtkTreeIter iter;
     
     gtk_tree_model_get_iter (GTK_TREE_MODEL (task_list_store), &iter, p);
     gtk_list_store_set (GTK_LIST_STORE (task_list_store), &iter, COMMENT_COLUMN, value, -1);
}


static void create_comment_window (GtkListStore *store, GtkTreeIter iter) {
     GtkWidget *frame;
     GtkWidget *table;
     GtkWidget *title_label;
     GtkWidget *title;
     GtkWidget *deadline_label;
     GtkWidget *deadline;
     GtkWidget *scrolledview;
     GtkWidget *textview;
     GtkWidget *button_box;
     GtkWidget *ok_button;
     GtkWidget *cancel_button;
     gchar *task;
     glong time;
     gchar *comment;
     
     gtk_tree_model_get (GTK_TREE_MODEL (store), &iter, TITLE_COLUMN, &task,
			 DEADLINE_COLUMN, &time, COMMENT_COLUMN, &comment, -1);

     detail_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
     gtk_window_set_default_size (GTK_WINDOW (detail_window), 400, 250);
     gtk_window_set_title (GTK_WINDOW (detail_window), _("Task Comments"));

     frame = gtk_frame_new (_("Comment"));
     gtk_container_set_border_width (GTK_CONTAINER (frame), 3);
     gtk_container_add (GTK_CONTAINER (detail_window), frame);
     
     table = gtk_table_new (4, 2, FALSE);
     gtk_container_set_border_width (GTK_CONTAINER (table), 3);
     gtk_table_set_row_spacings (GTK_TABLE (table), 3);
     gtk_table_set_col_spacings (GTK_TABLE (table), 3);
     gtk_container_add (GTK_CONTAINER (frame), table);

     /* add the static task information here */
     title_label = gtk_label_new (_("Task:"));
     gtk_label_set_justify (GTK_LABEL (title_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (title_label), 0, 0.5);
     gtk_table_attach (GTK_TABLE (table), title_label, 0, 1, 0, 1,
		       (GtkAttachOptions) (0), (GtkAttachOptions) (0),
		       0, 0);

     title = gtk_label_new (task);
     gtk_label_set_justify (GTK_LABEL (title), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (title), 0.015, 0.5);
     gtk_table_attach (GTK_TABLE (table), title, 1, 2, 0, 1,
		       (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), (GtkAttachOptions) (0),
		       0, 0);
     
     deadline_label = gtk_label_new (_("Due on:"));
     gtk_label_set_justify (GTK_LABEL (deadline_label), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (deadline_label), 0, 0.5);
     gtk_table_attach (GTK_TABLE (table), deadline_label, 0, 1, 1, 2,
		       (GtkAttachOptions) (0), (GtkAttachOptions) (0),
		       0, 0);

     deadline = gtk_label_new (gtodo_get_date_from_time_str (time, NULL));
     gtk_label_set_justify (GTK_LABEL (deadline), GTK_JUSTIFY_LEFT);
     gtk_misc_set_alignment (GTK_MISC (deadline), 0.015, 0.5);
     gtk_table_attach (GTK_TABLE (table), deadline, 1, 2, 1, 2,
		       (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), (GtkAttachOptions) (0),
		       0, 0);
     /* end static information */
     
     scrolledview = gtk_scrolled_window_new (NULL, NULL);
     gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledview),
				     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
     gtk_table_attach (GTK_TABLE (table), scrolledview, 0, 2, 2, 3,
		       (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
		       (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);

     textview = gtk_text_view_new ();
     gtk_text_view_set_editable (GTK_TEXT_VIEW (textview), TRUE);
     gtk_container_add (GTK_CONTAINER (scrolledview), textview);

     text_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview));
     gtk_text_buffer_set_text (GTK_TEXT_BUFFER (text_buffer), comment, -1);
     
     button_box = gtk_hbutton_box_new ();
     gtk_hbutton_box_set_layout_default (GTK_BUTTONBOX_END);
     gtk_box_set_spacing (GTK_BOX (button_box), 5);
     gtk_table_attach (GTK_TABLE (table), button_box, 0, 2, 3, 4,
		       (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
		       (GtkAttachOptions) (0), 0, 0);

     ok_button = gtk_button_new_from_stock ("gtk-ok");
     gtk_container_add (GTK_CONTAINER (button_box), ok_button);

     cancel_button = gtk_button_new_from_stock ("gtk-cancel");
     gtk_container_add (GTK_CONTAINER (button_box), cancel_button);

     g_signal_connect (G_OBJECT (detail_window), "destroy",
		       G_CALLBACK (gtk_widget_destroyed),
		       &detail_window);
     g_signal_connect (G_OBJECT (ok_button), "clicked",
		       G_CALLBACK (on_comment_ok_button_clicked),
		       g_memdup (&iter, sizeof (GtkTreeIter)));
     g_signal_connect (G_OBJECT (cancel_button), "clicked",
		       G_CALLBACK (on_comment_cancel_button_clicked),
		       detail_window);
     
     gtk_widget_show_all (detail_window);
     g_free (task);
     g_free (comment);
}

static void on_comment_ok_button_clicked (GtkButton *button, gpointer data) {
     GtkTextIter start, end;
     gchar *comment;
     //GtkTreeIter iter;
     
     gtk_text_buffer_get_start_iter (GTK_TEXT_BUFFER (text_buffer), &start);
     gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (text_buffer), &end);
     comment = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (text_buffer), &start, &end, FALSE);

     //gtk_tree_model_sort_convert_iter_to_child_iter (GTK_TREE_MODEL_SORT (sortmodel), &iter,
     //					     (GtkTreeIter *)data);     
     gtk_list_store_set (GTK_LIST_STORE (task_list_store), (GtkTreeIter *)data,
			 COMMENT_COLUMN, comment, -1);

     gtk_widget_destroy (GTK_WIDGET (gtk_widget_get_toplevel (GTK_WIDGET (button))));
     global_config.clean = FALSE;
}

static void on_comment_cancel_button_clicked (GtkButton *button, gpointer data) {
     gtk_widget_destroy (GTK_WIDGET (data));
}

/* Callback for the mouse wheel events */
gboolean gtodo_mouse_wheel_scrolled (GtkWidget *widget, GdkEventScroll *event, gpointer data) {
     GtkTreeIter old_iter, new_iter;
     GtkTreePath *path;
     gboolean valid;
     gboolean active = FALSE;
     
     /* we only do this if there is an active task
      * otherwise, the loop will go through without setting
      * set anyway */
     if (global_config.have_active) {
	  /* get the current task */
	  valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (task_list_store), &old_iter);
	  while (valid) {
	       gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), &old_iter, ACTIVE_COLUMN, &active, -1);
	       if (active) break;
	       valid = gtk_tree_model_iter_next (GTK_TREE_MODEL (task_list_store), &old_iter);
	  }
	  /* get the path of the active task so we can roll to next one */
	  path = gtk_tree_model_get_path (GTK_TREE_MODEL (task_list_store), &old_iter);
	  
	  if (event->direction == GDK_SCROLL_UP) /* go to previous task */
	       gtk_tree_path_prev (path);
	  else if (event->direction == GDK_SCROLL_DOWN) /* go to next task */
	       gtk_tree_path_next (path);
	  
	  if (gtk_tree_model_get_iter (GTK_TREE_MODEL (task_list_store), &new_iter, path)) {
	       gchar *title;
	       glong time;
	       gint comp;
	       
	       gtk_tree_model_get (GTK_TREE_MODEL (task_list_store), &new_iter, TITLE_COLUMN, &title,
				   DEADLINE_COLUMN, &time, COMPLETE_COLUMN, &comp, -1);
	       gtk_list_store_set (GTK_LIST_STORE (task_list_store), &old_iter, ACTIVE_COLUMN, FALSE, -1);
	       gtk_list_store_set (GTK_LIST_STORE (task_list_store), &new_iter, ACTIVE_COLUMN, TRUE, -1);
	       gtodo_set_panel_tooltip (title, time, comp);
	       gtodo_quick_view_set_comment_button (&new_iter);
	  }
     } else {
	  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (task_list_store), &new_iter)) {
	       gtk_list_store_set (GTK_LIST_STORE (task_list_store), &new_iter, ACTIVE_COLUMN, TRUE, -1);
	       gtodo_quick_view_set_comment_button (&new_iter);
	       global_config.have_active = TRUE;
	  }
     }
     return TRUE;
}
