/**
 * Copyright 2007, Michael Bertolacci
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <fcntl.h>

#define POOL_FILE ".jobpool.pool"
#define SLOT_FILE ".jobpool.slot"
#define WAIT_INTERVAL 1000
static char *DEFAULT_DIRECTORY = "./";

static char *directory = NULL;
static int max_jobs = 2;

#define FALSE 0
#define TRUE 1

typedef struct _command_t {
    char *command;
    char **arguments;
    int num_args;
} command_t;

/**
 * Returns a FILE pointer to the pool file. The process has exclusive
 * access to the pool file while this FILE pointer remains open.
 */
static FILE *open_pool_file() {
    FILE *pool_fp = NULL;
    int pool_fd;
    char *pool_file;

    asprintf(&pool_file, "%s%s", directory, POOL_FILE);
    assert(pool_file != NULL);

    pool_fd = open(pool_file, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
    if (pool_fd == -1) {
	perror("Error opening pool file");
	exit(EXIT_FAILURE);
    }
    free(pool_file);

    /* Block to acquire an exclusive lock */
    if (flock(pool_fd, LOCK_EX) == -1) {
	perror("Error locking pool file");
	exit(EXIT_FAILURE);
    }

    pool_fp = fdopen(pool_fd, "r+");

    return pool_fp;
}

/**
 * Adds a command to the end of the pool queue.
 *
 * @param command Command to add. Must not be NULL.
 */
static void enqueue(command_t *command) {
    FILE *pool_fp;
    int i;

    assert(command != NULL);

    pool_fp = open_pool_file();
    fseek(pool_fp, 0, SEEK_END);
    fprintf(pool_fp, "%d %s %d", (int)strlen(command->command), command->command,
	    command->num_args);
    for (i = 0; i < command->num_args; ++i) {
	fprintf(pool_fp, " %d %s", (int)strlen(command->arguments[i]),
		command->arguments[i]);
    }
    fprintf(pool_fp, "\n");
    flock(fileno(pool_fp), LOCK_UN);
    fclose(pool_fp);
}

/**
 * Reads a "token" from a file pointer. A token is an integer d
 * followed by d bytes. The bytes are returned in a buffer that the
 * caller is responsible for freeing.
 *
 * @param fp File pointer to read from.
 * @return the token, or NULL if none found.
 */
static char *read_token(FILE *fp) {
    char *token;
    int length;

    if (fscanf(fp, "%d", &length) == 0) {
	return NULL;
    }
    fgetc(fp);
    token = malloc(sizeof(char) * (length + 1));
    assert(token != NULL);
    fgets(token, length + 1, fp);

    return token;
}

/**
 * Reads a command from a file pointer.
 *
 * @param fp File pointer to read from.
 * @return command read from file, or NULL.
 */
static command_t *read_command(FILE *fp) {
    int i;
    command_t *command = malloc(sizeof(command));
    assert(command != NULL);

    command->command = read_token(fp);
    if (command->command == NULL) {
	free(command);
	return NULL;
    }
    fscanf(fp, "%d", &(command->num_args));

    command->arguments = malloc(sizeof(char *) * (command->num_args + 1));
    assert(command->arguments != NULL);
    for (i = 0; i < command->num_args; ++i) {
	command->arguments[i] = read_token(fp);
    }
    command->arguments[command->num_args] = NULL;

    return command;
}

/**
 * Frees the provided command structure.
 *
 * @param command Command structure to free.
 */
static void free_command(command_t *command) {
    int i;

    free(command->command);

    for (i = 0; i < command->num_args; ++i) {
	free(command->arguments[i]);
    }
    free(command->arguments);
}

/**
 * Returns the frontmost job in the pool, or NULL if the pool is
 * empty.
 *
 * @param pool_fp FILE pointer to pool.
 * @return allocated command struct, or NULL.
 */
static command_t *get_front_job(FILE *pool_fp) {
    struct stat sbuf;
    fstat(fileno(pool_fp), &sbuf);
    if (sbuf.st_size == 0) {
	return NULL;
    }
    fseek(pool_fp, 0, SEEK_SET);
    return read_command(pool_fp);
}

/**
 * Removes the front job in the pool queue.
 *
 * @param pool_fp FILE pointer to pool.
 */
static void dequeue(FILE *pool_fp) {
    command_t *temp_command;
    char *buf;
    struct stat sbuf;
    int length;

    fseek(pool_fp, 0, SEEK_SET);

    temp_command = read_command(pool_fp);
    if (temp_command == NULL) {
	return;
    }
    free_command(temp_command);

    length = ftell(pool_fp) + 1;

    fstat(fileno(pool_fp), &sbuf);
    if (length != sbuf.st_size) {
	/* Likely mmap was not intended for this task, but it works. */
	buf = mmap(0, sbuf.st_size, PROT_READ | PROT_WRITE, MAP_SHARED,
		   fileno(pool_fp), 0);
	if (buf == MAP_FAILED) {
	    perror("Error reading next job");
	    exit(EXIT_FAILURE);
	}
	memmove(buf, buf + length, sbuf.st_size - length);
	munmap(buf, sbuf.st_size);
    }
    fseek(pool_fp, 0, SEEK_SET);
    ftruncate(fileno(pool_fp), sbuf.st_size - length);
}

/**
 * Attempts to acquire one of the max_jobs running slots. Returns -1
 * if it fails, otherwise a file description corresponding to that
 * slot.
 *
 * @return Slot file description, or -1.
 */
static int attempt_slot_acquisition(void) {
    int i;

    for (i = 0; i < max_jobs; ++i) {
	char *slot_file;
	int fd;

	asprintf(&slot_file, "%s%s%d", directory, SLOT_FILE, i);
	assert(slot_file != NULL);
	fd = open(slot_file, O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);
	free(slot_file);

	if (flock(fd, LOCK_EX | LOCK_NB) == -1) {
	    close(fd);
	} else {
	    return fd;
	}
    }

    return -1;
}

/**
 * Run the command given, waiting until it completes.
 *
 * @param command Description of the job.
 */
static void run_job(command_t *command) {
    int pid;
    int status;

    if ((pid = fork()) < 0) {
	perror("fork");
	exit(1);
    }

    if (pid == 0) {
	execvp(command->command, command->arguments);
	perror("execvp");
	return;
    }

    while (wait(&status) != pid) {
	;
    }
}

/**
 * Runs jobs from the job pool until it cannot get a slot or no
 * jobs remain.
 */
static void work_job_pool(void) {
    FILE *pool_fp;
    command_t *job;

    /* Begin pool reading critical section */
    pool_fp = open_pool_file();
    job = get_front_job(pool_fp);

    while (job != NULL) {
	int slot_fd;

	/* This is a critical region: a job must only be run while a
	   slot file is open and locked. */
	slot_fd = attempt_slot_acquisition();
	if (slot_fd == -1) {
	    free_command(job);
	    fclose(pool_fp);
	    return;
	}
	dequeue(pool_fp);
	fclose(pool_fp);
	/* End pool reading critical section. */

	/* Here, the job is no longer in the queue, and we read and
	   removed the job while holding a lock, so this is the only
	   process that will ever this job. */
	run_job(job);

	free_command(job);
	close(slot_fd);
	/* End job running critical section. */

	pool_fp = open_pool_file();
	job = get_front_job(pool_fp);
    }

    fclose(pool_fp);
}

static void usage(const char *name) {
    printf("%s: [-h] [-w] [-d control_directory] [-j jobs] command "
	   "[arg1] [arg2] [...]\n", name);
}

int main(int argc, char **argv) {
    command_t *command = malloc(sizeof(command));
    int i;
    int wait = FALSE;
    int c;

    assert(command != NULL);

    opterr = 0;
    directory = DEFAULT_DIRECTORY;

    while ((c = getopt(argc, argv, "+j:wd:h")) != -1) {
	if (c == 'j') {
	    max_jobs = atoi(optarg);
	} else if (c == 'w') {
	    wait = TRUE;
	} else if (c == 'd') {
	    directory = strdup(optarg);
	    assert(directory != NULL);
	} else if (c == 'h' || c == '?') {
	    usage(argv[0]);
	    return 0;
	} else {
	    abort();
	}
    }
    
    if (argc - optind == 0) {
	usage(argv[0]);
	return 1;
    }

    command->command = strdup(argv[optind]);
    assert(command->command != NULL);
    command->num_args = argc - optind;
    if (argc > 0) {
	command->arguments = malloc(sizeof(char *) * (command->num_args + 1));
	for (i = optind; i < argc; ++i) {
	    command->arguments[i - optind] = strdup(argv[i]);
	    assert(command->arguments[i - optind] != NULL);
	}
	command->arguments[i - optind] = NULL;
    }

    if (wait) {
	/* We attempt to run the job before doing anything else */
	int slot_fd = attempt_slot_acquisition();
	while (slot_fd == -1) {
	    usleep(WAIT_INTERVAL);
	    slot_fd = attempt_slot_acquisition();
	}
	run_job(command);
	close(slot_fd);
    } else {
	enqueue(command);
    }
    free_command(command);
    /* Even if given the -w option we should try to run jobs */
    work_job_pool();

    return 0;
}
