'use strict';

const MetricsAggregationBase = require('./metrics-aggregation-base');

/**
 * A multi-value metrics aggregation that computes stats over numeric values
 * extracted from the aggregated documents. These values can be extracted either
 * from specific numeric fields in the documents, or be generated by a provided
 * script.
 *
 * The stats that are returned consist of: min, max, sum, count and avg.
 *
 * [Elasticsearch reference](https://www.elastic.co/guide/en/elasticsearch/reference/current/search-aggregations-metrics-stats-aggregation.html)
 *
 * Aggregation that computes stats over numeric values extracted from the
 * aggregated documents.
 *
 * @example
 * const agg = esb.statsAggregation('grades_stats', 'grade');
 *
 *
 * @example
 * // Use a file script
 * const agg = esb.statsAggregation('grades_stats').script(
 *     esb.script('file', 'my_script').params({ field: 'price' })
 * );
 *
 * @example
 * // Value script to apply the conversion rate to every value
 * // before it is aggregated
 * const agg = esb.statsAggregation('grades_stats').script(
 *     esb.script('inline', '_value * params.conversion_rate').params({
 *         conversion_rate: 1.2
 *     })
 * );
 *
 * @param {string} name The name which will be used to refer to this aggregation.
 * @param {string=} field The field to aggregate on
 *
 * @extends MetricsAggregationBase
 */
class StatsAggregation extends MetricsAggregationBase {
    // eslint-disable-next-line require-jsdoc
    constructor(name, field) {
        super(name, 'stats', field);
    }
}

module.exports = StatsAggregation;
