/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.protocol;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import java.net.SocketAddress;
import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.traccar.BaseProtocolDecoder;
import org.traccar.NetworkMessage;
import org.traccar.Protocol;
import org.traccar.config.Keys;
import org.traccar.helper.BitUtil;
import org.traccar.helper.DataConverter;
import org.traccar.helper.Parser;
import org.traccar.helper.PatternBuilder;
import org.traccar.helper.UnitsConverter;
import org.traccar.model.CellTower;
import org.traccar.model.Network;
import org.traccar.model.Position;
import org.traccar.model.WifiAccessPoint;
import org.traccar.session.DeviceSession;

public class Gl200TextProtocolDecoder
extends BaseProtocolDecoder {
    private static final Map<String, String> PROTOCOL_MODELS = Map.ofEntries(Map.entry("02", "GL200"), Map.entry("04", "GV200"), Map.entry("06", "GV300"), Map.entry("08", "GMT100"), Map.entry("09", "GV50P"), Map.entry("0F", "GV55"), Map.entry("10", "GV55 LITE"), Map.entry("11", "GL500"), Map.entry("1A", "GL300"), Map.entry("1F", "GV500"), Map.entry("21", "GL200"), Map.entry("25", "GV300"), Map.entry("27", "GV300W"), Map.entry("28", "GL300VC"), Map.entry("2C", "GL300W"), Map.entry("2D", "GV500VC"), Map.entry("2F", "GV55"), Map.entry("30", "GL300"), Map.entry("31", "GV65"), Map.entry("35", "GV200"), Map.entry("36", "GV500"), Map.entry("3F", "GMT100"), Map.entry("40", "GL500"), Map.entry("41", "GV75W"), Map.entry("42", "GT501"), Map.entry("44", "GL530"), Map.entry("45", "GB100"), Map.entry("50", "GV55W"), Map.entry("52", "GL50"), Map.entry("55", "GL50B"), Map.entry("5E", "GV500MAP"), Map.entry("6E", "GV310LAU"), Map.entry("BD", "CV200"), Map.entry("C2", "GV600M"), Map.entry("DC", "GV600MG"), Map.entry("DE", "GL500M"), Map.entry("F1", "GV350M"), Map.entry("F8", "GV800W"), Map.entry("FC", "GV600W"), Map.entry("802004", "GV58LAU"), Map.entry("802005", "GV355CEU"));
    private boolean ignoreFixTime;
    private final DateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
    private static final Pattern PATTERN_VER = new PatternBuilder().text("+").expression("(?:RESP|BUFF):GTVER,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").expression("([^,]*),").number("(xxxx),").number("(xxxx),").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd),").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_LOCATION = new PatternBuilder().number("(d{1,2}.?d?)?,").number("(d{1,3}.d)?,").number("(d{1,3}.?d?)?,").number("(-?d{1,5}.d)?,").number("(-?d{1,3}.d{6})?,").number("(-?d{1,2}.d{6})?,").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).groupBegin().number(",d+").number("((?:,x{12},-d+,,,)+)").groupEnd("?").text(",").number("(d+)?,").number("(d+)?,").groupBegin().number("(d+),").number("(d+),").or().number("(x+)?,").number("(x+)?,").groupEnd().number("(?:d+|(d+.d))?,").compile();
    private static final Pattern PATTERN_OBD = new PatternBuilder().text("+RESP:GTOBD,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("(?:[0-9A-Z]{17})?,").expression("[^,]{0,20},").expression("[01],").number("x{1,8},").expression("(?:[0-9A-Z]{17})?,").number("[01],").number("(?:d{1,5})?,").number("(?:x{8})?,").number("(d{1,5})?,").number("(d{1,3})?,").number("(-?d{1,3})?,").number("(d+.?d*|Inf|NaN)?,").number("(d{1,5})?,").number("(?:d{1,5})?,").expression("([01])?,").number("(d{1,3})?,").number("(x*),").number("(d{1,3})?,").number("(?:d{1,3})?,").number("(d{1,3})?,").expression("(?:[0-9A],)?").number("(d+),").expression(PATTERN_LOCATION.pattern()).number("(d{1,7}.d)?,").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_FRI = new PatternBuilder().text("+").expression("(?:RESP|BUFF):GT...,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("(?:([0-9A-Z]{17}),)?").expression("[^,]*,").number("(d+)?,").number("(d{1,2}),").optional().number("d{1,2},").optional().number("d*,").optional().number("(d+),").optional().expression("((?:").expression(PATTERN_LOCATION.pattern()).expression(")+)").groupBegin().number("d{1,2},").number("(d{1,5})?,").number("(d{1,3}),").number("[01],").number("(?:[01])?,").number("(-?d{1,2}.d)?,").or().number("(d{1,7}.d)?,").number("(d{5}:dd:dd)?,").number("(x+)?,").number("(x+)?,").number("(d{1,3})?,").number("(x{6})?,").number("(d+)?,").number("(?:d+.?d*|Inf|NaN)?,").number("(d+)?,").or().number("(-?d),").number("(d{1,3}),").or().number("(d{1,7}.d)?,").optional().number("(d{1,3})?,").groupEnd().any().number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_LSW = new PatternBuilder().text("+RESP:").expression("GT[LT]SW,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("[01],").number("([01]),").expression(PATTERN_LOCATION.pattern()).number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_IDA = new PatternBuilder().text("+RESP:GTIDA,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,,").number("([^,]+),").expression("[01],").number("1,").expression(PATTERN_LOCATION.pattern()).number("(d+.d),").text(",,,,").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_WIF = new PatternBuilder().text("+RESP:GTWIF,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("(d+),").number("((?:x{12},-?d+,,,,)+),,,,").number("(d{1,3}),").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_GSM = new PatternBuilder().text("+RESP:GTGSM,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("(?:STR|CTN|NMR|RTL),").expression("(.*)").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_PNA = new PatternBuilder().text("+RESP:GT").expression("P[NF]A,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_DAR = new PatternBuilder().text("+RESP:GTDAR,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("(d),").number("(d{1,2}),,,").expression(PATTERN_LOCATION.pattern()).any().number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_DTT = new PatternBuilder().text("+RESP:GTDTT,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,,,").number("d,").number("d+,").number("(x+),").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_BAA = new PatternBuilder().text("+RESP:GTBAA,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("x+,").number("d,").number("d,").number("x+,").number("(x{4}),").expression("((?:[^,]+,){0,6})").expression(PATTERN_LOCATION.pattern()).any().number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_BID = new PatternBuilder().text("+RESP:GTBID,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("d,").number("d,").number("(x{4}),").expression("((?:[^,]+,){0,2})").expression(PATTERN_LOCATION.pattern()).any().number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_LSA = new PatternBuilder().text("+RESP:GTLSA,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("d,").number("d,").number("d+,").expression(PATTERN_LOCATION.pattern()).number("d+,").number("(d),").number("(d+),").number("[01],").number("[01]?,").number("(-?d+.d)?,").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN = new PatternBuilder().text("+").expression("(?:RESP|BUFF):GT...,").expression("(?:.{6}|.{10})?,").number("(d{15}|x{14}),").expression("[^,]*,").number("d*,").number("(x{1,2}),").number("d{1,2},").number("d*,").optional().expression(PATTERN_LOCATION.pattern()).groupBegin().number("(?:(d{1,7}.d)|0)?,").optional().number("(d{1,3})?,").or().number("(d{1,7}.d)?,").groupEnd().number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").text(",").number("(xxxx)").text("$").optional().compile();
    private static final Pattern PATTERN_BASIC = new PatternBuilder().text("+").expression("(?:RESP|BUFF)").text(":").expression("GT...,").expression("[^,]+,").optional().number("(d{15}|x{14}),").any().text(",").number("(d{1,2}),").groupBegin().number("(d{1,3}.d),").number("(d{1,3}),").number("(-?d{1,5}.d),").number("(-?d{1,3}.d{6}),").number("(-?d{1,2}.d{6}),").number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").text(",").or().text(",,,,,,").groupEnd().number("(d+),").number("(d+),").number("(x+),").number("(x+),").optional(4).any().number("(dddd)(dd)(dd)").number("(dd)(dd)(dd)").optional(2).text(",").number("(xxxx)").text("$").optional().compile();

    public Gl200TextProtocolDecoder(Protocol protocol) {
        super(protocol);
        this.dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
    }

    @Override
    protected void init() {
        this.ignoreFixTime = this.getConfig().getBoolean(Keys.PROTOCOL_IGNORE_FIX_TIME.withPrefix(this.getProtocolName()));
    }

    private String getDeviceModel(DeviceSession deviceSession, String protocolVersion) {
        String declaredModel = this.getDeviceModel(deviceSession);
        if (declaredModel != null) {
            return declaredModel.toUpperCase();
        }
        String versionPrefix = protocolVersion.length() > 6 ? protocolVersion.substring(0, 6) : protocolVersion.substring(0, 2);
        String model = PROTOCOL_MODELS.get(versionPrefix);
        return model != null ? model : "";
    }

    private Position initPosition(Parser parser, Channel channel, SocketAddress remoteAddress) {
        DeviceSession deviceSession;
        if (parser.matches() && (deviceSession = this.getDeviceSession(channel, remoteAddress, parser.next())) != null) {
            Position position = new Position(this.getProtocolName());
            position.setDeviceId(deviceSession.getDeviceId());
            return position;
        }
        return null;
    }

    private void decodeDeviceTime(Position position, Parser parser) {
        if (parser.hasNext(6)) {
            if (this.ignoreFixTime) {
                position.setTime(parser.nextDateTime());
            } else {
                position.setDeviceTime(parser.nextDateTime());
            }
        }
    }

    private Long parseHours(String hoursString) {
        if (hoursString != null && !hoursString.isEmpty()) {
            String[] hours = hoursString.split(":");
            return ((long)Integer.parseInt(hours[0]) * 3600L + (hours.length > 1 ? (long)Integer.parseInt(hours[1]) * 60L : 0L) + (long)(hours.length > 2 ? Integer.parseInt(hours[2]) : 0)) * 1000L;
        }
        return null;
    }

    private Position decodeAck(Channel channel, SocketAddress remoteAddress, String[] values) throws ParseException {
        DeviceSession deviceSession = this.getDeviceSession(channel, remoteAddress, values[2]);
        if (deviceSession == null) {
            return null;
        }
        if (values[0].equals("+ACK:GTHBD")) {
            if (channel != null) {
                channel.writeAndFlush((Object)new NetworkMessage("+SACK:GTHBD," + values[1] + "," + values[values.length - 1] + "$", remoteAddress));
            }
        } else {
            Position position = new Position(this.getProtocolName());
            position.setDeviceId(deviceSession.getDeviceId());
            this.getLastLocation(position, this.dateFormat.parse(values[values.length - 2]));
            position.setValid(false);
            position.set("result", values[0]);
            return position;
        }
        return null;
    }

    private Object decodeInf(Channel channel, SocketAddress remoteAddress, String[] v) throws ParseException {
        DeviceSession deviceSession;
        int index = 0;
        int n = ++index;
        ++index;
        String protocolVersion = v[n];
        if (protocolVersion.length() > 10) {
            return null;
        }
        if ((deviceSession = this.getDeviceSession(channel, remoteAddress, v[index++])) == null) {
            return null;
        }
        Position position = new Position(this.getProtocolName());
        position.setDeviceId(deviceSession.getDeviceId());
        String model = this.getDeviceModel(deviceSession, protocolVersion);
        int n2 = ++index;
        ++index;
        if (!v[n2].isEmpty()) {
            int state = Integer.parseInt(v[index - 1], 16);
            switch (state) {
                case 18: 
                case 22: 
                case 26: {
                    position.set("ignition", false);
                    position.set("motion", true);
                    break;
                }
                case 17: {
                    position.set("ignition", false);
                    position.set("motion", false);
                    break;
                }
                case 33: {
                    position.set("ignition", true);
                    position.set("motion", false);
                    break;
                }
                case 34: {
                    position.set("ignition", true);
                    position.set("motion", true);
                    break;
                }
                case 65: {
                    position.set("motion", false);
                    break;
                }
                case 66: {
                    position.set("motion", true);
                }
            }
        }
        position.set("iccid", v[index++]);
        if (!v[index++].isEmpty()) {
            position.set("rssi", Integer.parseInt(v[index - 1]));
        }
        ++index;
        if (v[++index + 1].length() >= 12) {
            int n3 = ++index;
            position.set("deviceTemp", Integer.parseInt(v[n3]));
            int n4 = ++index;
            position.set("humidity", Integer.parseInt(v[n4]));
        }
        int n5 = ++index;
        ++index;
        if (!v[n5].isEmpty()) {
            String value = v[index - 1];
            if (value.contains(".")) {
                position.set("odometer", Double.parseDouble(value) * 1000.0);
            } else {
                position.set("power", (double)Integer.parseInt(value) / 1000.0);
            }
        }
        if (!(model.equals("GV500VC") || model.equals("GV350M") || model.equals("GV310LAU") || v[index++].isEmpty())) {
            position.set("power2", (double)Integer.parseInt(v[index - 1]) / 1000.0);
        }
        int n6 = ++index;
        ++index;
        if (!v[n6].isEmpty()) {
            position.set("battery", Double.parseDouble(v[index - 1]));
        }
        if (!v[index++].isEmpty()) {
            position.set("charge", Integer.parseInt(v[index++]) == 1 ? Boolean.valueOf(true) : null);
        }
        if (model.equals("GV310LAU")) {
            ++index;
            ++index;
            ++index;
            ++index;
            int n7 = ++index;
            position.set("adc1", Integer.parseInt(v[n7]));
            int n8 = ++index;
            position.set("adc2", Integer.parseInt(v[n8]));
            int n9 = ++index;
            ++index;
            position.set("adc3", Integer.parseInt(v[n9]));
        }
        Date time = this.dateFormat.parse(v[v.length - 2]);
        if (this.ignoreFixTime) {
            position.setTime(time);
        } else {
            position.setDeviceTime(time);
        }
        this.getLastLocation(position, position.getDeviceTime());
        return position;
    }

    private Object decodeVer(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_VER, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        position.set("deviceType", parser.next());
        position.set("versionFw", parser.nextHexInt());
        position.set("versionHw", parser.nextHexInt());
        this.getLastLocation(position, parser.nextDateTime());
        return position;
    }

    private void skipLocation(Parser parser) {
        parser.skip(20);
    }

    private void decodeLocation(Position position, Parser parser) {
        Double hdop = parser.nextDouble();
        position.setValid(hdop == null || hdop > 0.0);
        position.set("hdop", hdop);
        position.setSpeed(UnitsConverter.knotsFromKph(parser.nextDouble(0.0)));
        position.setCourse(parser.nextDouble(0.0));
        position.setAltitude(parser.nextDouble(0.0));
        if (parser.hasNext(8)) {
            position.setValid(true);
            position.setLongitude(parser.nextDouble());
            position.setLatitude(parser.nextDouble());
            position.setTime(parser.nextDateTime());
        } else {
            this.getLastLocation(position, null);
        }
        Network network = new Network();
        if (parser.hasNext()) {
            String[] values = parser.next().split(",");
            for (int i = 0; i < values.length; i += 5) {
                String mac = values[i + 1].replaceAll("(..)", "$1:");
                network.addWifiAccessPoint(WifiAccessPoint.from(mac.substring(0, mac.length() - 1), Integer.parseInt(values[i + 2])));
            }
        }
        if (parser.hasNext(6)) {
            int mcc = parser.nextInt();
            int mnc = parser.nextInt();
            if (parser.hasNext(2)) {
                network.addCellTower(CellTower.from(mcc, mnc, parser.nextInt(), parser.nextInt().intValue()));
            }
            if (parser.hasNext(2)) {
                network.addCellTower(CellTower.from(mcc, mnc, parser.nextHexInt(), parser.nextHexInt().intValue()));
            }
        }
        if (network.getWifiAccessPoints() != null || network.getCellTowers() != null) {
            position.setNetwork(network);
        }
        if (parser.hasNext()) {
            position.set("odometer", parser.nextDouble() * 1000.0);
        }
    }

    private int decodeLocation(Position position, String model, String[] values, int index) throws ParseException {
        double hdop = values[index++].isEmpty() ? 0.0 : Double.parseDouble(values[index - 1]);
        position.set("hdop", hdop);
        position.setSpeed(UnitsConverter.knotsFromKph(values[index++].isEmpty() ? 0.0 : Double.parseDouble(values[index - 1])));
        position.setCourse(values[index++].isEmpty() ? 0.0 : (double)Integer.parseInt(values[index - 1]));
        position.setAltitude(values[index++].isEmpty() ? 0.0 : Double.parseDouble(values[index - 1]));
        if (!values[index].isEmpty()) {
            position.setValid(true);
            position.setLongitude(values[index++].isEmpty() ? 0.0 : Double.parseDouble(values[index - 1]));
            position.setLatitude(values[index++].isEmpty() ? 0.0 : Double.parseDouble(values[index - 1]));
            position.setTime(this.dateFormat.parse(values[index++]));
        } else {
            index += 3;
            this.getLastLocation(position, null);
        }
        Network network = new Network();
        if (!values[index].isEmpty()) {
            network.addCellTower(CellTower.from(Integer.parseInt(values[index++]), Integer.parseInt(values[index++]), Integer.parseInt(values[index++], 16), Long.parseLong(values[index++], 16)));
        } else {
            index += 4;
        }
        if (network.getWifiAccessPoints() != null || network.getCellTowers() != null) {
            position.setNetwork(network);
        }
        if (model.startsWith("GL5")) {
            ++index;
        }
        int n = ++index;
        ++index;
        if (!values[n].isEmpty()) {
            int appendMask = Integer.parseInt(values[index - 1]);
            if (BitUtil.check(appendMask, 0)) {
                position.set("sat", Integer.parseInt(values[index++]));
            }
            if (BitUtil.check(appendMask, 1)) {
                ++index;
            }
        }
        return index;
    }

    private Object decodeObd(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_OBD, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        position.set("rpm", parser.nextInt());
        position.set("obdSpeed", parser.nextInt());
        position.set("temp1", parser.nextInt());
        position.set("fuelConsumption", parser.next());
        position.set("dtcsClearedDistance", parser.nextInt());
        if (parser.hasNext()) {
            position.set("odbConnect", parser.nextInt() == 1);
        }
        position.set("dtcsNumber", parser.nextInt());
        position.set("dtcsCodes", parser.next());
        position.set("throttle", parser.nextInt());
        position.set("fuel", parser.nextInt());
        if (parser.hasNext()) {
            position.set("obdOdometer", parser.nextInt() * 1000);
        }
        this.decodeLocation(position, parser);
        if (parser.hasNext()) {
            position.set("obdOdometer", (int)(parser.nextDouble() * 1000.0));
        }
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeCan(Channel channel, SocketAddress remoteAddress, String[] v) throws ParseException {
        String value;
        int index = 0;
        int n = ++index;
        String protocolVersion = v[n];
        String[] stringArray = new String[1];
        int n2 = ++index;
        ++index;
        stringArray[0] = v[n2];
        DeviceSession deviceSession = this.getDeviceSession(channel, remoteAddress, stringArray);
        if (deviceSession == null) {
            return null;
        }
        Position position = new Position(this.getProtocolName());
        position.setDeviceId(deviceSession.getDeviceId());
        String model = this.getDeviceModel(deviceSession, protocolVersion);
        ++index;
        ++index;
        int n3 = ++index;
        ++index;
        long reportMask = Long.parseLong(v[n3], 16);
        if (BitUtil.check(reportMask, 0)) {
            position.set("vin", v[index++]);
        }
        if (BitUtil.check(reportMask, 1) && !v[index++].isEmpty()) {
            position.set("ignition", Integer.parseInt(v[index - 1]) > 0);
        }
        if (BitUtil.check(reportMask, 2) && !v[index++].isEmpty()) {
            position.set("obdOdometer", Integer.parseInt(v[index - 1].substring(1)));
        }
        if (BitUtil.check(reportMask, 3) && !v[index++].isEmpty()) {
            position.set("fuelUsed", Double.parseDouble(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 5) && !v[index++].isEmpty()) {
            position.set("rpm", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 4) && !v[index++].isEmpty()) {
            position.set("obdSpeed", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 6) && !v[index++].isEmpty()) {
            position.set("coolantTemp", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 7) && !v[index++].isEmpty() && (value = v[index - 1]).startsWith("L/H")) {
            position.set("fuelConsumption", Double.parseDouble(value.substring(3)));
        }
        if (BitUtil.check(reportMask, 8) && !v[index++].isEmpty()) {
            position.set("fuel", Double.parseDouble(v[index - 1].substring(1)));
        }
        if (BitUtil.check(reportMask, 9) && !v[index++].isEmpty()) {
            position.set("range", Long.parseLong(v[index - 1]) * 100L);
        }
        if (BitUtil.check(reportMask, 10) && !v[index++].isEmpty()) {
            position.set("throttle", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 11) && !v[index++].isEmpty()) {
            position.set("hours", UnitsConverter.msFromHours(Double.parseDouble(v[index - 1])));
        }
        if (BitUtil.check(reportMask, 12) && !v[index++].isEmpty()) {
            position.set("drivingTime", Double.parseDouble(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 13) && !v[index++].isEmpty()) {
            position.set("idleHours", Double.parseDouble(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 14) && !v[index++].isEmpty()) {
            position.set("idleFuelConsumption", Double.parseDouble(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 15) && !v[index++].isEmpty()) {
            position.set("axleWeight", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 16) && !v[index++].isEmpty()) {
            position.set("tachographInfo", Integer.parseInt(v[index - 1], 16));
        }
        if (BitUtil.check(reportMask, 17) && !v[index++].isEmpty()) {
            position.set("indicators", Integer.parseInt(v[index - 1], 16));
        }
        if (BitUtil.check(reportMask, 18) && !v[index++].isEmpty()) {
            position.set("lights", Integer.parseInt(v[index - 1], 16));
        }
        if (BitUtil.check(reportMask, 19) && !v[index++].isEmpty()) {
            position.set("doors", Integer.parseInt(v[index - 1], 16));
        }
        if (BitUtil.check(reportMask, 20) && !v[index++].isEmpty()) {
            position.set("vehicleOverspeed", Double.parseDouble(v[index - 1]));
        }
        if (BitUtil.check(reportMask, 21) && !v[index++].isEmpty()) {
            position.set("engineOverspeed", Double.parseDouble(v[index - 1]));
        }
        if ("GV350M".equals(model)) {
            if (BitUtil.check(reportMask, 22)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 23)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 24)) {
                ++index;
            }
        } else if ("GV355CEU".equals(model)) {
            if (BitUtil.check(reportMask, 22)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 23)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 24)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 25)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 26)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 27)) {
                ++index;
            }
            if (BitUtil.check(reportMask, 28)) {
                ++index;
            }
        }
        long reportMaskExt = 0L;
        if (BitUtil.check(reportMask, 29) && !v[index++].isEmpty()) {
            reportMaskExt = Long.parseLong(v[index - 1], 16);
        }
        if (BitUtil.check(reportMaskExt, 0) && !v[index++].isEmpty()) {
            position.set("adBlueLevel", Double.parseDouble(v[index - 1].substring(1)));
        }
        if (BitUtil.check(reportMaskExt, 1) && !v[index++].isEmpty()) {
            position.set("axleWeight1", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMaskExt, 2) && !v[index++].isEmpty()) {
            position.set("axleWeight3", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMaskExt, 3) && !v[index++].isEmpty()) {
            position.set("axleWeight4", Integer.parseInt(v[index - 1]));
        }
        if (BitUtil.check(reportMaskExt, 4)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 5)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 6)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 7) && !v[index++].isEmpty()) {
            position.set("adc1", (double)Integer.parseInt(v[index - 1]) * 0.001);
        }
        if (BitUtil.check(reportMaskExt, 8)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 9)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 10)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 11)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 12)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 13)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 14)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 15) && !v[index++].isEmpty()) {
            position.set("driver1Card", v[index - 1]);
        }
        if (BitUtil.check(reportMaskExt, 16) && !v[index++].isEmpty()) {
            position.set("driver2Card", v[index - 1]);
        }
        if (BitUtil.check(reportMaskExt, 17) && !v[index++].isEmpty()) {
            position.set("driver1Name", v[index - 1]);
        }
        if (BitUtil.check(reportMaskExt, 18) && !v[index++].isEmpty()) {
            position.set("driver2Name", v[index - 1]);
        }
        if (BitUtil.check(reportMaskExt, 19) && !v[index++].isEmpty()) {
            position.set("registration", v[index - 1]);
        }
        if (BitUtil.check(reportMaskExt, 20)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 21)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 22)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 23)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 24)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 25)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 26)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 27)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 28)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 29)) {
            ++index;
        }
        if (BitUtil.check(reportMaskExt, 30)) {
            ++index;
        }
        long reportMaskCan = 0L;
        if (BitUtil.check(reportMaskExt, 31) && !v[index++].isEmpty()) {
            reportMaskCan = Long.parseLong(v[index - 1], 16);
        }
        if (BitUtil.check(reportMaskCan, 0)) {
            ++index;
        }
        if (BitUtil.check(reportMaskCan, 1)) {
            ++index;
        }
        if (BitUtil.check(reportMaskCan, 2)) {
            ++index;
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        if (!"GV355CEU".equals(model) && BitUtil.check(reportMask, 30)) {
            while (v[index].isEmpty()) {
                ++index;
            }
            position.setValid(Integer.parseInt(v[index++]) > 0);
            if (!v[index].isEmpty()) {
                position.setSpeed(UnitsConverter.knotsFromKph(Double.parseDouble(v[index++])));
                position.setCourse(Integer.parseInt(v[index++]));
                position.setAltitude(Double.parseDouble(v[index++]));
                position.setLongitude(Double.parseDouble(v[index++]));
                position.setLatitude(Double.parseDouble(v[index++]));
                position.setTime(dateFormat.parse(v[index++]));
            } else {
                index += 6;
                this.getLastLocation(position, null);
            }
        } else {
            this.getLastLocation(position, null);
        }
        if (BitUtil.check(reportMask, 31)) {
            index += 4;
            ++index;
        }
        index = v.length - 2;
        if (this.ignoreFixTime) {
            position.setTime(dateFormat.parse(v[index]));
        } else {
            position.setDeviceTime(dateFormat.parse(v[index]));
        }
        return position;
    }

    private void decodeStatus(Position position, long value) {
        long ignition = BitUtil.between(value, 16, 24);
        if (BitUtil.check(ignition, 4)) {
            position.set("ignition", false);
        } else if (BitUtil.check(ignition, 5)) {
            position.set("ignition", true);
        }
        long input = BitUtil.between(value, 8, 16);
        long output = BitUtil.to(value, 8);
        position.set("input", input);
        position.set("in1", BitUtil.check(input, 1));
        position.set("in2", BitUtil.check(input, 2));
        position.set("output", output);
        position.set("out1", BitUtil.check(output, 0));
        position.set("out2", BitUtil.check(output, 1));
    }

    private Object decodeFri(Channel channel, SocketAddress remoteAddress, String sentence) {
        Position position;
        Parser parser = new Parser(PATTERN_FRI, sentence);
        if (!parser.matches()) {
            return null;
        }
        DeviceSession deviceSession = this.getDeviceSession(channel, remoteAddress, parser.next());
        if (deviceSession == null) {
            return null;
        }
        LinkedList<Position> positions = new LinkedList<Position>();
        String vin = parser.next();
        Integer power = parser.nextInt();
        Integer reportType = parser.nextInt();
        Integer battery = parser.nextInt();
        Parser itemParser = new Parser(PATTERN_LOCATION, parser.next());
        while (itemParser.find()) {
            position = new Position(this.getProtocolName());
            position.setDeviceId(deviceSession.getDeviceId());
            position.set("vin", vin);
            this.decodeLocation(position, itemParser);
            positions.add(position);
        }
        position = (Position)positions.getLast();
        this.skipLocation(parser);
        if (power != null && power > 10) {
            position.set("power", (double)power.intValue() * 0.001);
        }
        if (battery != null) {
            position.set("batteryLevel", battery);
        }
        if (parser.hasNext()) {
            position.set("battery", (double)parser.nextInt().intValue() * 0.001);
        }
        position.set("batteryLevel", parser.nextInt());
        position.set("temp1", parser.nextDouble());
        if (parser.hasNext()) {
            position.set("odometer", parser.nextDouble() * 1000.0);
        }
        position.set("hours", this.parseHours(parser.next()));
        position.set("adc1", parser.next());
        position.set("adc2", parser.next());
        position.set("batteryLevel", parser.nextInt());
        if (parser.hasNext()) {
            this.decodeStatus(position, parser.nextHexLong());
        }
        position.set("rpm", parser.nextInt());
        position.set("fuel", parser.nextInt());
        if (parser.hasNext(2)) {
            if (reportType != null) {
                position.set("motion", BitUtil.check(reportType.intValue(), 0));
                position.set("charge", BitUtil.check(reportType.intValue(), 1));
            }
            position.set("rssi", parser.nextInt());
            position.set("batteryLevel", parser.nextInt());
        }
        if (parser.hasNext()) {
            position.set("odometer", parser.nextDouble() * 1000.0);
        }
        position.set("batteryLevel", parser.nextInt());
        this.decodeDeviceTime(position, parser);
        if (this.ignoreFixTime) {
            positions.clear();
            positions.add(position);
        }
        return positions;
    }

    private Object decodeEri(Channel channel, SocketAddress remoteAddress, String[] v) throws ParseException {
        int i;
        int index = 0;
        int n = ++index;
        String protocolVersion = v[n];
        String[] stringArray = new String[1];
        int n2 = ++index;
        ++index;
        stringArray[0] = v[n2];
        DeviceSession deviceSession = this.getDeviceSession(channel, remoteAddress, stringArray);
        if (deviceSession == null) {
            return null;
        }
        String model = this.getDeviceModel(deviceSession, protocolVersion);
        int n3 = ++index;
        long mask = Long.parseLong(v[n3], 16);
        Double power = v[++index].isEmpty() ? null : Double.valueOf((double)Integer.parseInt(v[++index - 1]) * 0.001);
        int n4 = ++index;
        ++index;
        int count = Integer.parseInt(v[n4]);
        LinkedList<Position> positions = new LinkedList<Position>();
        for (int i2 = 0; i2 < count; ++i2) {
            Position position = new Position(this.getProtocolName());
            position.setDeviceId(deviceSession.getDeviceId());
            index = this.decodeLocation(position, model, v, index);
            positions.add(position);
        }
        Position position = (Position)positions.getLast();
        position.set("power", power);
        if (!model.startsWith("GL5")) {
            position.set("odometer", v[index++].isEmpty() ? null : Double.valueOf(Double.parseDouble(v[index - 1]) * 1000.0));
            position.set("hours", this.parseHours(v[index++]));
            position.set("adc1", v[index++].isEmpty() ? null : Double.valueOf((double)Integer.parseInt(v[index - 1]) * 0.001));
        }
        if (model.startsWith("GV") && !model.startsWith("GV6") && !model.equals("GV350M")) {
            position.set("adc2", v[index++].isEmpty() ? null : Double.valueOf((double)Integer.parseInt(v[index - 1]) * 0.001));
        }
        if (model.equals("GV200") || model.equals("GV310LAU")) {
            position.set("adc3", v[index++].isEmpty() ? null : Double.valueOf((double)Integer.parseInt(v[index - 1]) * 0.001));
        }
        if (model.startsWith("GV355CEU")) {
            ++index;
        }
        if (model.startsWith("GL5")) {
            position.set("batteryLevel", v[index++].isEmpty() ? null : Integer.valueOf(Integer.parseInt(v[index - 1])));
            position.set("motion", v[++index].isEmpty() ? null : Boolean.valueOf(Integer.parseInt(v[++index - 1]) > 0));
        } else if (model.equals("GV200")) {
            position.set("input", v[index++].isEmpty() ? null : Integer.valueOf(Integer.parseInt(v[index - 1], 16)));
            position.set("output", v[index++].isEmpty() ? null : Integer.valueOf(Integer.parseInt(v[index - 1], 16)));
            ++index;
        } else {
            position.set("batteryLevel", v[index++].isEmpty() ? null : Integer.valueOf(Integer.parseInt(v[index - 1])));
            if (!v[index++].isEmpty()) {
                this.decodeStatus(position, Long.parseLong(v[index - 1]));
            }
            ++index;
        }
        if (BitUtil.check(mask, 0)) {
            position.set("fuel", v[index++].isEmpty() ? null : Integer.valueOf(Integer.parseInt(v[index - 1], 16)));
        }
        if (BitUtil.check(mask, 1)) {
            int deviceCount = Integer.parseInt(v[index++]);
            for (i = 1; i <= deviceCount; ++i) {
                ++index;
                int n5 = ++index;
                ++index;
                if (v[n5].isEmpty()) continue;
                position.set("temp" + i, (double)((short)Integer.parseInt(v[index - 1], 16)) * 0.0625);
            }
        }
        if (BitUtil.check(mask, 2)) {
            ++index;
        }
        if (BitUtil.check(mask, 3) || BitUtil.check(mask, 4)) {
            int deviceCount = Integer.parseInt(v[index++]);
            for (i = 1; i <= deviceCount; ++i) {
                ++index;
                if (BitUtil.check(mask, 3)) {
                    position.set("fuel", Double.parseDouble(v[index++]));
                }
                if (!BitUtil.check(mask, 4)) continue;
                ++index;
            }
        }
        Date time = this.dateFormat.parse(v[v.length - 2]);
        if (this.ignoreFixTime) {
            position.setTime(time);
            positions.clear();
            positions.add(position);
        } else {
            position.setDeviceTime(time);
        }
        return positions;
    }

    private Object decodeIgn(Channel channel, SocketAddress remoteAddress, String[] v, String type) throws ParseException {
        int index = 0;
        int n = ++index;
        String protocolVersion = v[n];
        String[] stringArray = new String[1];
        int n2 = ++index;
        ++index;
        stringArray[0] = v[n2];
        DeviceSession deviceSession = this.getDeviceSession(channel, remoteAddress, stringArray);
        if (deviceSession == null) {
            return null;
        }
        Position position = new Position(this.getProtocolName());
        position.setDeviceId(deviceSession.getDeviceId());
        String model = this.getDeviceModel(deviceSession, protocolVersion);
        ++index;
        ++index;
        index = this.decodeLocation(position, model, v, index);
        position.set("ignition", type.contains("GN"));
        position.set("hours", this.parseHours(v[index++]));
        position.set("odometer", Double.parseDouble(v[index]) * 1000.0);
        Date time = this.dateFormat.parse(v[v.length - 2]);
        if (this.ignoreFixTime) {
            position.setTime(time);
        } else {
            position.setDeviceTime(time);
        }
        return position;
    }

    private Object decodeLsw(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_LSW, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        position.set("in" + (sentence.contains("LSW") ? 1 : 2), parser.nextInt() == 1);
        this.decodeLocation(position, parser);
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeIda(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_IDA, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        position.set("driverUniqueId", parser.next());
        this.decodeLocation(position, parser);
        position.set("odometer", parser.nextDouble() * 1000.0);
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeWif(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_WIF, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        this.getLastLocation(position, null);
        Network network = new Network();
        parser.nextInt();
        Matcher matcher = Pattern.compile("([0-9a-fA-F]{12}),(-?\\d+),,,,").matcher(parser.next());
        while (matcher.find()) {
            String mac = matcher.group(1).replaceAll("(..)", "$1:");
            network.addWifiAccessPoint(WifiAccessPoint.from(mac.substring(0, mac.length() - 1), Integer.parseInt(matcher.group(2))));
        }
        position.setNetwork(network);
        position.set("batteryLevel", parser.nextInt());
        return position;
    }

    private Object decodeGsm(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_GSM, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        this.getLastLocation(position, null);
        Network network = new Network();
        String[] data = parser.next().split(",");
        for (int i = 0; i < 6; ++i) {
            if (data[i * 6].isEmpty()) continue;
            network.addCellTower(CellTower.from(Integer.parseInt(data[i * 6]), Integer.parseInt(data[i * 6 + 1]), Integer.parseInt(data[i * 6 + 2], 16), Integer.parseInt(data[i * 6 + 3], 16), Integer.parseInt(data[i * 6 + 4])));
        }
        position.setNetwork(network);
        return position;
    }

    private Object decodePna(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_PNA, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        this.getLastLocation(position, null);
        position.addAlarm(sentence.contains("PNA") ? "powerOn" : "powerOff");
        return position;
    }

    private Object decodeDar(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_DAR, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        int warningType = parser.nextInt();
        int fatigueDegree = parser.nextInt();
        if (warningType == 1) {
            position.addAlarm("fatigueDriving");
            position.set("fatigueDegree", fatigueDegree);
        } else {
            position.set("warningType", warningType);
        }
        this.decodeLocation(position, parser);
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeDtt(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_DTT, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        this.getLastLocation(position, null);
        String data = Unpooled.wrappedBuffer((byte[])DataConverter.parseHex(parser.next())).toString(StandardCharsets.US_ASCII);
        if (data.contains("COMB")) {
            position.set("fuel", Double.parseDouble(data.split(",")[2]));
        } else {
            position.set("result", data);
        }
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeBaa(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_BAA, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        int mask = parser.nextHexInt();
        String[] values = parser.next().split(",");
        int index = 0;
        if (BitUtil.check(mask, 0)) {
            position.set("accessoryName", values[index++]);
        }
        if (BitUtil.check(mask, 1)) {
            position.set("accessoryMac", values[index++]);
        }
        if (BitUtil.check(mask, 2)) {
            position.set("accessoryStatus", Integer.parseInt(values[index++]));
        }
        if (BitUtil.check(mask, 3)) {
            position.set("accessoryVoltage", (double)Integer.parseInt(values[index++]) * 0.001);
        }
        if (BitUtil.check(mask, 4)) {
            position.set("accessoryTemp", Integer.parseInt(values[index++]));
        }
        if (BitUtil.check(mask, 5)) {
            position.set("accessoryHumidity", Integer.parseInt(values[index]));
        }
        this.decodeLocation(position, parser);
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeBid(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_BID, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        int mask = parser.nextHexInt();
        String[] values = parser.next().split(",");
        int index = 0;
        if (BitUtil.check(mask, 1)) {
            position.set("accessoryMac", values[index++]);
        }
        if (BitUtil.check(mask, 3)) {
            position.set("accessoryVoltage", (double)Integer.parseInt(values[index]) * 0.001);
        }
        this.decodeLocation(position, parser);
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeLsa(Channel channel, SocketAddress remoteAddress, String sentence) {
        Parser parser = new Parser(PATTERN_LSA, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        this.decodeLocation(position, parser);
        position.set("lightLevel", parser.nextInt());
        position.set("batteryLevel", parser.nextInt());
        position.set("temp1", parser.nextDouble());
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeLbs(Channel channel, SocketAddress remoteAddress, String[] v) throws ParseException {
        int index = 0;
        ++index;
        String[] stringArray = new String[1];
        int n = ++index;
        ++index;
        stringArray[0] = v[n];
        DeviceSession deviceSession = this.getDeviceSession(channel, remoteAddress, stringArray);
        if (deviceSession == null) {
            return null;
        }
        Position position = new Position(this.getProtocolName());
        position.setDeviceId(deviceSession.getDeviceId());
        ++index;
        ++index;
        int n2 = ++index;
        ++index;
        position.set("batteryLevel", Integer.parseInt(v[n2]));
        ++index;
        ++index;
        ++index;
        ++index;
        ++index;
        Network network = new Network();
        for (int i = 0; i <= 6; ++i) {
            if (!v[index + 1].isEmpty()) {
                network.addCellTower(CellTower.from(Integer.parseInt(v[index++]), Integer.parseInt(v[index++]), Integer.parseInt(v[index++], 16), Integer.parseInt(v[index++], 16), Integer.parseInt(v[index++])));
            } else {
                index += 5;
            }
            ++index;
        }
        position.setNetwork(network);
        Date time = this.dateFormat.parse(v[v.length - 2]);
        this.getLastLocation(position, time);
        return position;
    }

    private Object decodeOther(Channel channel, SocketAddress remoteAddress, String sentence, String type) {
        Parser parser = new Parser(PATTERN, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        int reportType = parser.nextHexInt();
        if (type.equals("NMR")) {
            position.set("motion", reportType == 1);
        } else if (type.equals("SOS")) {
            position.addAlarm("sos");
        } else if (type.equals("DIS")) {
            position.set("in" + reportType / 16, reportType % 16 == 1);
        } else if (type.equals("IGL")) {
            position.set("ignition", reportType % 16 == 1);
        } else if (type.equals("HBM")) {
            switch (reportType % 16) {
                case 0: 
                case 3: {
                    position.addAlarm("hardBraking");
                    break;
                }
                case 1: 
                case 4: {
                    position.addAlarm("hardAcceleration");
                    break;
                }
                case 2: {
                    position.addAlarm("hardCornering");
                }
            }
        }
        this.decodeLocation(position, parser);
        if (parser.hasNext()) {
            position.set("odometer", parser.nextDouble() * 1000.0);
        }
        position.set("batteryLevel", parser.nextInt());
        if (parser.hasNext()) {
            position.set("odometer", parser.nextDouble() * 1000.0);
        }
        this.decodeDeviceTime(position, parser);
        return position;
    }

    private Object decodeBasic(Channel channel, SocketAddress remoteAddress, String sentence, String type) {
        Parser parser = new Parser(PATTERN_BASIC, sentence);
        Position position = this.initPosition(parser, channel, remoteAddress);
        if (position == null) {
            return null;
        }
        if (parser.hasNext()) {
            int hdop = parser.nextInt();
            position.setValid(hdop > 0);
            position.set("hdop", hdop);
        }
        position.setSpeed(UnitsConverter.knotsFromKph(parser.nextDouble(0.0)));
        position.setCourse(parser.nextDouble(0.0));
        position.setAltitude(parser.nextDouble(0.0));
        if (parser.hasNext(2)) {
            position.setLongitude(parser.nextDouble());
            position.setLatitude(parser.nextDouble());
        } else {
            this.getLastLocation(position, null);
        }
        if (parser.hasNext(6)) {
            position.setTime(parser.nextDateTime());
        }
        if (parser.hasNext(4)) {
            position.setNetwork(new Network(CellTower.from(parser.nextInt(), parser.nextInt(), parser.nextHexInt(), parser.nextHexInt().intValue())));
        }
        this.decodeDeviceTime(position, parser);
        switch (type) {
            case "TOW": {
                position.addAlarm("tow");
                break;
            }
            case "IDL": {
                position.addAlarm("idle");
                break;
            }
            case "PNA": {
                position.addAlarm("powerOn");
                break;
            }
            case "PFA": {
                position.addAlarm("powerOff");
                break;
            }
            case "EPN": 
            case "MPN": {
                position.addAlarm("powerRestored");
                break;
            }
            case "EPF": 
            case "MPF": {
                position.addAlarm("powerCut");
                break;
            }
            case "BPL": {
                position.addAlarm("lowBattery");
                break;
            }
            case "STT": {
                position.addAlarm("movement");
                break;
            }
            case "SWG": {
                position.addAlarm("geofence");
                break;
            }
            case "TMP": 
            case "TEM": {
                position.addAlarm("temperature");
                break;
            }
            case "JDR": 
            case "JDS": {
                position.addAlarm("jamming");
            }
        }
        return position;
    }

    @Override
    protected Object decode(Channel channel, SocketAddress remoteAddress, Object msg) throws Exception {
        Object result;
        String sentence = ((ByteBuf)msg).toString(StandardCharsets.US_ASCII).replaceAll("\\$$", "");
        int typeIndex = sentence.indexOf(":GT");
        if (typeIndex < 0) {
            return null;
        }
        String[] values = sentence.split(",");
        String type = sentence.substring(typeIndex + 3, typeIndex + 6);
        if (sentence.startsWith("+ACK")) {
            result = this.decodeAck(channel, remoteAddress, values);
        } else {
            switch (type) {
                case "INF": {
                    Object object = this.decodeInf(channel, remoteAddress, values);
                    break;
                }
                case "OBD": {
                    Object object = this.decodeObd(channel, remoteAddress, sentence);
                    break;
                }
                case "CAN": {
                    Object object = this.decodeCan(channel, remoteAddress, values);
                    break;
                }
                case "CTN": 
                case "FRI": 
                case "GEO": 
                case "RTL": 
                case "DOG": 
                case "STR": {
                    Object object = this.decodeFri(channel, remoteAddress, sentence);
                    break;
                }
                case "ERI": {
                    Object object = this.decodeEri(channel, remoteAddress, values);
                    break;
                }
                case "IGN": 
                case "IGF": 
                case "VGN": 
                case "VGF": {
                    Object object = this.decodeIgn(channel, remoteAddress, values, type);
                    break;
                }
                case "LSW": 
                case "TSW": {
                    Object object = this.decodeLsw(channel, remoteAddress, sentence);
                    break;
                }
                case "IDA": {
                    Object object = this.decodeIda(channel, remoteAddress, sentence);
                    break;
                }
                case "WIF": {
                    Object object = this.decodeWif(channel, remoteAddress, sentence);
                    break;
                }
                case "GSM": {
                    Object object = this.decodeGsm(channel, remoteAddress, sentence);
                    break;
                }
                case "VER": {
                    Object object = this.decodeVer(channel, remoteAddress, sentence);
                    break;
                }
                case "PNA": 
                case "PFA": {
                    Object object = this.decodePna(channel, remoteAddress, sentence);
                    break;
                }
                case "DAR": {
                    Object object = this.decodeDar(channel, remoteAddress, sentence);
                    break;
                }
                case "DTT": {
                    Object object = this.decodeDtt(channel, remoteAddress, sentence);
                    break;
                }
                case "BAA": {
                    Object object = this.decodeBaa(channel, remoteAddress, sentence);
                    break;
                }
                case "BID": {
                    Object object = this.decodeBid(channel, remoteAddress, sentence);
                    break;
                }
                case "LSA": {
                    Object object = this.decodeLsa(channel, remoteAddress, sentence);
                    break;
                }
                case "LBS": {
                    Object object = this.decodeLbs(channel, remoteAddress, values);
                    break;
                }
                default: {
                    Object object = result = this.decodeOther(channel, remoteAddress, sentence, type);
                }
            }
            if (result == null) {
                result = this.decodeBasic(channel, remoteAddress, sentence, type);
            }
            if (result != null) {
                if (result instanceof Position) {
                    Position position = (Position)result;
                    position.set("type", type);
                } else {
                    for (Position p : (List)result) {
                        p.set("type", type);
                    }
                }
            }
        }
        if (channel != null && this.getConfig().getBoolean(Keys.PROTOCOL_ACK.withPrefix(this.getProtocolName()))) {
            channel.writeAndFlush((Object)new NetworkMessage("+SACK:" + values[values.length - 1] + "$", remoteAddress));
        }
        return result;
    }
}

