"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.isIBranchKeyStoreNode = exports.BranchKeyStoreNode = void 0;
const kms_config_1 = require("./kms_config");
const client_kms_1 = require("@aws-sdk/client-kms");
const client_dynamodb_1 = require("@aws-sdk/client-dynamodb");
const material_management_1 = require("@aws-crypto/material-management");
const uuid_1 = require("uuid");
const branch_keystore_helpers_1 = require("./branch_keystore_helpers");
const constants_1 = require("./constants");
const types_1 = require("./types");
const dynamodb_key_storage_1 = require("./dynamodb_key_storage");
class BranchKeyStoreNode {
    constructor({ logicalKeyStoreName, storage, keyManagement, kmsConfiguration, keyStoreId, }) {
        /* Precondition: Logical keystore name must be a string */
        (0, material_management_1.needs)(typeof logicalKeyStoreName === 'string', 'Logical keystore name must be a string');
        /* Precondition: KMS Configuration must be provided. */
        (0, material_management_1.readOnlyProperty)(this, 'kmsConfiguration', new kms_config_1.KmsKeyConfig(kmsConfiguration));
        /* Precondition: KMS client must be a KMSClient */
        if (keyManagement?.kmsClient) {
            (0, material_management_1.needs)(keyManagement.kmsClient instanceof client_kms_1.KMSClient, 'KMS client must be a KMSClient');
        }
        if ('getEncryptedActiveBranchKey' in storage &&
            'getEncryptedBranchKeyVersion' in storage) {
            // JS does structural typing.
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If [Storage](#storage) is configured with [KeyStorage](#keystorage)
            //# then this MUST be the configured [KeyStorage interface](./key-store/key-storage.md#interface).
            this.storage = storage;
        }
        else {
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
            //# If [Storage](#storage) is not configured with [KeyStorage](#keystorage)
            //# a [default key storage](./key-store/default-key-storage.md#initialization) MUST be created.
            (0, material_management_1.needs)(!storage.ddbClient ||
                storage.ddbClient instanceof client_dynamodb_1.DynamoDBClient, 'DDB client must be a DynamoDBClient');
            this.storage = new dynamodb_key_storage_1.DynamoDBKeyStorage({
                //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
                //# This constructed [default key storage](./key-store/default-key-storage.md#initialization)
                //# MUST be configured with either the [Table Name](#table-name) or the [DynamoDBTable](#dynamodbtable) table name
                //# depending on which one is configured.
                ddbTableName: storage.ddbTableName,
                //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
                //# This constructed [default key storage](./key-store/default-key-storage.md#overview)
                //# MUST be configured with the provided [logical keystore name](#logical-keystore-name).
                logicalKeyStoreName,
                ddbClient: 
                //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
                //# This constructed [default key storage](./key-store/default-key-storage.md#initialization)
                //# MUST be configured with either the [DynamoDb Client](#dynamodb-client), the DDB client in the [DynamoDBTable](#dynamodbtable)
                //# or a constructed DDB client depending on what is configured.
                storage.ddbClient instanceof client_dynamodb_1.DynamoDBClient
                    ? storage.ddbClient
                    : new client_dynamodb_1.DynamoDBClient({
                        //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
                        //# If a DDB client needs to be constructed and the AWS KMS Configuration is KMS Key ARN or KMS MRKey ARN,
                        //# a new DynamoDb client MUST be created with the region of the supplied KMS ARN.
                        //#
                        //# If a DDB client needs to be constructed and the AWS KMS Configuration is Discovery,
                        //# a new DynamoDb client MUST be created with the default configuration.
                        //#
                        //# If a DDB client needs to be constructed and the AWS KMS Configuration is MRDiscovery,
                        //# a new DynamoDb client MUST be created with the region configured in the MRDiscovery.
                        region: this.kmsConfiguration.getRegion(),
                    }),
            });
        }
        (0, material_management_1.readOnlyProperty)(this, 'storage', this.storage);
        (0, material_management_1.needs)(logicalKeyStoreName == this.storage.getKeyStorageInfo().logicalName, 'Configured logicalKeyStoreName does not match configured storage interface.');
        /* Precondition: Keystore id must be a string */
        if (keyStoreId) {
            (0, material_management_1.needs)(typeof keyStoreId === 'string', 'Keystore id must be a string');
        }
        else {
            // ensure it's strictly undefined and not some other falsey value
            keyStoreId = undefined;
        }
        /* Precondition: Grant tokens must be a string array */
        if (keyManagement?.grantTokens) {
            (0, material_management_1.needs)(Array.isArray(keyManagement.grantTokens) &&
                keyManagement.grantTokens.every((grantToken) => typeof grantToken === 'string'), 'Grant tokens must be a string array');
        }
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#keystore-id
        //# The Identifier for this KeyStore.
        //# If one is not supplied, then a [version 4 UUID](https://www.ietf.org/rfc/rfc4122.txt) MUST be used.
        (0, material_management_1.readOnlyProperty)(this, 'keyStoreId', keyStoreId ? keyStoreId : (0, uuid_1.v4)());
        /* Postcondition: If unprovided, the keystore id is a generated valid uuidv4 */
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#aws-kms-grant-tokens
        //# A list of AWS KMS [grant tokens](https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token).
        (0, material_management_1.readOnlyProperty)(this, 'grantTokens', keyManagement?.grantTokens || undefined);
        /* Postcondition: If unprovided, the grant tokens are undefined */
        // TODO: when other KMS configuration types/classes are supported for the keystore,
        // verify the configuration object type to determine how we instantiate the
        // KMS client. This will ensure safe type casting.
        (0, material_management_1.readOnlyProperty)(this, 'kmsClient', 
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
        //# If no AWS KMS client is provided one MUST be constructed.
        keyManagement?.kmsClient ||
            new client_kms_1.KMSClient({
                //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
                //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is KMS Key ARN or KMS MRKey ARN,
                //# a new AWS KMS client MUST be created with the region of the supplied KMS ARN.
                //#
                //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is Discovery,
                //# a new AWS KMS client MUST be created with the default configuration.
                //#
                //# If AWS KMS client needs to be constructed and the AWS KMS Configuration is MRDiscovery,
                //# a new AWS KMS client MUST be created with the region configured in the MRDiscovery.
                region: this.kmsConfiguration.getRegion(),
                //= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
                //# On initialization the KeyStore SHOULD
                //# append a user agent string to the AWS KMS SDK Client with
                //# the value `aws-kms-hierarchy`.
                customUserAgent: constants_1.KMS_CLIENT_USER_AGENT,
            }));
        /* Postcondition: If unprovided, the KMS client is configured */
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#logical-keystore-name
        //# This name is cryptographically bound to all data stored in this table,
        //# and logically separates data between different tables.
        //#
        //# The logical keystore name MUST be bound to every created key.
        //#
        //# There needs to be a one to one mapping between DynamoDB Table Names and the Logical KeyStore Name.
        //# This value can be set to the DynamoDB table name itself, but does not need to.
        //#
        //# Controlling this value independently enables restoring from DDB table backups
        //# even when the table name after restoration is not exactly the same.
        (0, material_management_1.needs)(logicalKeyStoreName, 'Logical Keystore name required');
        (0, material_management_1.readOnlyProperty)(this, 'logicalKeyStoreName', logicalKeyStoreName);
        // make this instance immutable
        Object.freeze(this);
    }
    async getActiveBranchKey(branchKeyId) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# On invocation, the caller:
        //#
        //# - MUST supply a `branch-key-id`
        (0, material_management_1.needs)(branchKeyId && typeof branchKeyId === 'string', 'MUST supply a string branch key id');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# GetActiveBranchKey MUST get the active version for the branch key id from the keystore
        //# by calling the configured [KeyStorage interface's](./key-store/key-storage.md#interface)
        //# [GetEncryptedActiveBranchKey](./key-store/key-storage.md#getencryptedactivebranchkey)
        //# using the supplied `branch-key-id`.
        const activeEncryptedBranchKey = await this.storage.getEncryptedActiveBranchKey(branchKeyId);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# Because the storage interface can be a custom implementation the key store needs to verify correctness.
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# GetActiveBranchKey MUST verify that the returned EncryptedHierarchicalKey MUST have the requested `branch-key-id`.
        (0, material_management_1.needs)(activeEncryptedBranchKey.branchKeyId == branchKeyId, 'Unexpected branch key id.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# GetActiveBranchKey MUST verify that the returned EncryptedHierarchicalKey is an ActiveHierarchicalSymmetricVersion.
        (0, material_management_1.needs)(activeEncryptedBranchKey.type instanceof
            types_1.ActiveHierarchicalSymmetricVersion, 'Unexpected type. Not a version record.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# GetActiveBranchKey MUST verify that the returned EncryptedHierarchicalKey MUST have a logical table name equal to the configured logical table name.
        (0, material_management_1.needs)(activeEncryptedBranchKey.encryptionContext[constants_1.TABLE_FIELD] ==
            this.logicalKeyStoreName, 'Unexpected logical table name. Expected ${this.logicalKeyStoreName}, found ${activeEncryptedBranchKey.encryptionContext[TABLE_FIELD]}.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# If the branch key fails to decrypt, GetActiveBranchKey MUST fail.
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# The operation MUST decrypt the EncryptedHierarchicalKey according to the [AWS KMS Branch Key Decryption](#aws-kms-branch-key-decryption) section.
        const branchKey = await (0, branch_keystore_helpers_1.decryptBranchKey)(this, activeEncryptedBranchKey);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# This GetActiveBranchKey MUST construct [branch key materials](./structures.md#branch-key-materials)
        //# according to [Branch Key Materials From Authenticated Encryption Context](#branch-key-materials-from-authenticated-encryption-context).
        const branchKeyMaterials = (0, branch_keystore_helpers_1.constructBranchKeyMaterials)(branchKey, activeEncryptedBranchKey);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getactivebranchkey
        //# This operation MUST return the constructed [branch key materials](./structures.md#branch-key-materials).
        return branchKeyMaterials;
    }
    async getBranchKeyVersion(branchKeyId, branchKeyVersion) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# On invocation, the caller:
        //#
        //# - MUST supply a `branch-key-id`
        //# - MUST supply a `branchKeyVersion`
        (0, material_management_1.needs)(branchKeyId && typeof branchKeyId === 'string', 'MUST supply a string branch key id');
        (0, material_management_1.needs)(branchKeyVersion && typeof branchKeyVersion === 'string', 'MUST supply a string branch key version');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //= type=implication
        //# GetBranchKeyVersion MUST get the requested version for the branch key id from the keystore
        //# by calling the configured [KeyStorage interface's](./key-store/key-storage.md#interface)
        //# [GetEncryptedActiveBranchKey](./key-store/key-storage.md#getencryptedbranchkeyversion)
        //# using the supplied `branch-key-id`.
        const encryptedBranchKey = await this.storage.getEncryptedBranchKeyVersion(branchKeyId, branchKeyVersion);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# GetBranchKeyVersion MUST verify that the returned EncryptedHierarchicalKey MUST have the requested `branch-key-id`.
        (0, material_management_1.needs)(encryptedBranchKey.branchKeyId == branchKeyId, 'Unexpected branch key id.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# GetBranchKeyVersion MUST verify that the returned EncryptedHierarchicalKey MUST have the requested `branchKeyVersion`.
        (0, material_management_1.needs)(encryptedBranchKey.type.version == branchKeyVersion, 'Unexpected branch key id.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# GetActiveBranchKey MUST verify that the returned EncryptedHierarchicalKey is an HierarchicalSymmetricVersion.
        (0, material_management_1.needs)(encryptedBranchKey.type instanceof types_1.HierarchicalSymmetricVersion, 'Unexpected type. Not a version record.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# GetBranchKeyVersion MUST verify that the returned EncryptedHierarchicalKey MUST have a logical table name equal to the configured logical table name.
        (0, material_management_1.needs)(encryptedBranchKey.encryptionContext[constants_1.TABLE_FIELD] ==
            this.logicalKeyStoreName, 'Unexpected logical table name. Expected ${this.logicalKeyStoreName}, found ${encryptedBranchKey.encryptionContext[TABLE_FIELD}.');
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# If the branch key fails to decrypt, this operation MUST fail.
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# The operation MUST decrypt the branch key according to the [AWS KMS Branch Key Decryption](#aws-kms-branch-key-decryption) section.
        const branchKey = await (0, branch_keystore_helpers_1.decryptBranchKey)(this, encryptedBranchKey);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# This GetBranchKeyVersion MUST construct [branch key materials](./structures.md#branch-key-materials)
        //# according to [Branch Key Materials From Authenticated Encryption Context](#branch-key-materials-from-authenticated-encryption-context).
        const branchKeyMaterials = (0, branch_keystore_helpers_1.constructBranchKeyMaterials)(branchKey, encryptedBranchKey);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#getbranchkeyversion
        //# This operation MUST return the constructed [branch key materials](./structures.md#branch-key-materials).
        return branchKeyMaterials;
    }
    //= aws-encryption-sdk-specification/framework/branch-key-store.md#getkeystoreinfo
    //= type=implication
    //# This operation MUST return the keystore information in this keystore configuration.
    getKeyStoreInfo() {
        return {
            keystoreId: this.keyStoreId,
            //= aws-encryption-sdk-specification/framework/branch-key-store.md#getkeystoreinfo
            //= type=implication
            //# The [keystore name](#table-name) MUST be obtained
            //# from the configured [KeyStorage](./key-store/key-storage.md#interface)
            //# by calling [GetKeyStorageInfo](./key-store/key-storage.md#getkeystorageinfo).
            keystoreTableName: this.storage.getKeyStorageInfo().name,
            logicalKeyStoreName: this.logicalKeyStoreName,
            grantTokens: this.grantTokens ? this.grantTokens.slice() : [],
            kmsConfiguration: this.kmsConfiguration._config,
        };
    }
}
exports.BranchKeyStoreNode = BranchKeyStoreNode;
(0, material_management_1.immutableClass)(BranchKeyStoreNode);
// type guard
function isIBranchKeyStoreNode(keyStore) {
    return keyStore instanceof BranchKeyStoreNode;
}
exports.isIBranchKeyStoreNode = isIBranchKeyStoreNode;
// The JS implementation is not encumbered with the legacy construction
// by passing DDB clients et al.
// So it can be simplified.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
//= type=exception
//# - [AWS KMS Grant Tokens](#aws-kms-grant-tokens)
//= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
//= type=exception
//# - [DynamoDb Client](#dynamodb-client)
//= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
//= type=exception
//# - [Table Name](#table-name)
//# - [KMS Client](#kms-client)
//= aws-encryption-sdk-specification/framework/branch-key-store.md#operations
//= type=exception
//# - [CreateKeyStore](#createkeystore)
//# - [CreateKey](#createkey)
//# - [VersionKey](#versionkey)
//= aws-encryption-sdk-specification/framework/branch-key-store.md#operations
//= type=exception
//# - [GetBeaconKey](#getbeaconkey)
// Only `Storage` is defined as as input
// because JS was only released with this option.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
//= type=exception
//# If neither [Storage](#storage) nor [Table Name](#table-name) is configured initialization MUST fail.
//# If both [Storage](#storage) and [Table Name](#table-name) are configured initialization MUST fail.
//# If both [Storage](#storage) and [DynamoDb Client](#dynamodb-client) are configured initialization MUST fail.
// Only `KeyManagement` is defined as as input
// because JS was only released with this option.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#initialization
//= type=exception
//# If both [KeyManagement](#keymanagement) and [KMS Client](#kms-client) are configured initialization MUST fail.
//# If both [KeyManagement](#keymanagement) and [Grant Tokens](#aws-kms-grant-tokens) are configured initialization MUST fail.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#createkeystore
//= type=exception
//# If a [table Name](#table-name) was not configured then CreateKeyStore MUST fail.
//#
//# This operation MUST first calls the DDB::DescribeTable API with the configured `tableName`.
//#
//# If the response is successful, this operation validates that the table has the expected
//# [KeySchema](#keyschema) as defined below.
//# If the [KeySchema](#keyschema) does not match
//# this operation MUST yield an error.
//# The table MAY have additional information,
//# like GlobalSecondaryIndex defined.
//#
//# If the client responds with a `ResourceNotFoundException`,
//# then this operation MUST continue and
//# MUST call [AWS DDB CreateTable](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html)
//# with the following specifics:
//#
//# - TableName is the configured tableName.
//# - [KeySchema](#keyschema) as defined below.
//#
//# If the operation fails to create table, the operation MUST fail.
//#
//# If the operation successfully creates a table, the operation MUST return the AWS DDB Table Arn
//# back to the caller.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#createkey
//= type=exception
//# The CreateKey caller MUST provide:
//#
//# - An optional branch key id
//# - An optional encryption context
//#
//# If an optional branch key id is provided
//# and no encryption context is provided this operation MUST fail.
//#
//# If the Keystore's KMS Configuration is `Discovery` or `MRDiscovery`,
//# this operation MUST fail.
//#
//# If no branch key id is provided,
//# then this operation MUST create a [version 4 UUID](https://www.ietf.org/rfc/rfc4122.txt)
//# to be used as the branch key id.
//#
//# This operation MUST create a [branch key](structures.md#branch-key) and a [beacon key](structures.md#beacon-key) according to
//# the [Branch Key and Beacon Key Creation](#branch-key-and-beacon-key-creation) section.
//#
//# If creation of the keys are successful,
//# then the key store MUST call the configured [KeyStorage interface's](./key-store/key-storage.md#interface)
//# [WriteNewEncryptedBranchKey](./key-store/key-storage.md#writenewencryptedbranchkey) with these 3 [EncryptedHierarchicalKeys](./key-store/key-storage.md#encryptedhierarchicalkey).
//#
//# If writing to the keystore succeeds,
//# the operation MUST return the branch-key-id that maps to both
//# the branch key and the beacon key.
//#
//# Otherwise, this operation MUST yield an error.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-and-beacon-key-creation
//= type=exception
//# To create a branch key, this operation MUST take the following:
//#
//# - `branchKeyId`: The identifier
//# - `encryptionContext`: Additional encryption context to bind to the created keys
//#
//# This operation needs to generate the following:
//#
//# - `version`: a new guid. This guid MUST be [version 4 UUID](https://www.ietf.org/rfc/rfc4122.txt)
//# - `timestamp`: a timestamp for the current time.
//#   This timestamp MUST be in ISO 8601 format in UTC, to microsecond precision (e.g. “YYYY-MM-DDTHH:mm:ss.ssssssZ“)
//#
//# The wrapped Branch Keys, DECRYPT_ONLY and ACTIVE, MUST be created according to [Wrapped Branch Key Creation](#wrapped-branch-key-creation).
//#
//# To create a beacon key, this operation will continue to use the `branchKeyId` and `timestamp` as the [Branch Key](structures.md#branch-key).
//#
//# The operation MUST call [AWS KMS API GenerateDataKeyWithoutPlaintext](https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html).
//# The call to AWS KMS GenerateDataKeyWithoutPlaintext MUST use the configured AWS KMS client to make the call.
//# The operation MUST call AWS KMS GenerateDataKeyWithoutPlaintext with a request constructed as follows:
//#
//# - `KeyId` MUST be [compatible with](#aws-key-arn-compatibility) the configured KMS Key in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//# - `NumberOfBytes` MUST be 32.
//# - `EncryptionContext` MUST be the [encryption context for beacon keys](#beacon-key-encryption-context).
//# - `GrantTokens` MUST be this keystore's [grant tokens](https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token).
//#
//# If the call to AWS KMS GenerateDataKeyWithoutPlaintext succeeds,
//# the operation MUST use the `CiphertextBlob` as the wrapped Beacon Key.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#wrapped-branch-key-creation
//= type=exception
//# The operation MUST call [AWS KMS API GenerateDataKeyWithoutPlaintext](https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html).
//# The call to AWS KMS GenerateDataKeyWithoutPlaintext MUST use the configured AWS KMS client to make the call.
//# The operation MUST call AWS KMS GenerateDataKeyWithoutPlaintext with a request constructed as follows:
//#
//# - `KeyId` MUST be [compatible with](#aws-key-arn-compatibility) the configured KMS Key in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//# - `NumberOfBytes` MUST be 32.
//# - `EncryptionContext` MUST be the [DECRYPT_ONLY encryption context for branch keys](#decrypt_only-encryption-context).
//# - GenerateDataKeyWithoutPlaintext `GrantTokens` MUST be this keystore's [grant tokens](https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token).
//#
//# If the call to AWS KMS GenerateDataKeyWithoutPlaintext succeeds,
//# the operation MUST use the GenerateDataKeyWithoutPlaintext result `CiphertextBlob`
//# as the wrapped DECRYPT_ONLY Branch Key.
//#
//# The operation MUST call [AWS KMS API ReEncrypt](https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html)
//# with a request constructed as follows:
//#
//# - `SourceEncryptionContext` MUST be the [DECRYPT_ONLY encryption context for branch keys](#decrypt_only-encryption-context).
//# - `SourceKeyId` MUST be [compatible with](#aws-key-arn-compatibility) the configured KMS Key in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//# - `CiphertextBlob` MUST be the wrapped DECRYPT_ONLY Branch Key.
//# - ReEncrypt `GrantTokens` MUST be this keystore's [grant tokens](https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token).
//# - `DestinationKeyId` MUST be [compatible with](#aws-key-arn-compatibility) the configured KMS Key in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//# - `DestinationEncryptionContext` MUST be the [ACTIVE encryption context for branch keys](#active-encryption-context).
//#
//# If the call to AWS KMS ReEncrypt succeeds,
//# the operation MUST use the ReEncrypt result `CiphertextBlob`
//# as the wrapped ACTIVE Branch Key.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#active-encryption-context
//= type=exception
//# The ACTIVE branch key is a copy of the DECRYPT_ONLY with the same `version`.
//# It is structured slightly differently so that the active version can be accessed quickly.
//#
//# In addition to the [encryption context](#encryption-context):
//#
//# The ACTIVE encryption context value of the `type` attribute MUST equal to `"branch:ACTIVE"`.
//# The ACTIVE encryption context MUST have a `version` attribute.
//# The `version` attribute MUST store the branch key version formatted like `"branch:version:"` + `version`.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#decrypt-only-encryption-context
//= type=exception
//# In addition to the [encryption context](#encryption-context):
//#
//# The DECRYPT_ONLY encryption context MUST NOT have a `version` attribute.
//# The `type` attribute MUST stores the branch key version formatted like `"branch:version:"` + `version`.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#beacon-key-encryption-context
//= type=exception
//# In addition to the [encryption context](#encryption-context):
//#
//# The Beacon key encryption context value of the `type` attribute MUST equal to `"beacon:ACTIVE"`.
//# The Beacon key encryption context MUST NOT have a `version` attribute.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#versionkey
//= type=exception
//# - MUST supply a `branch-key-id`
//#
//# If the Keystore's KMS Configuration is `Discovery` or `MRDiscovery`,
//# this operation MUST immediately fail.
//#
//# VersionKey MUST first get the active version for the branch key from the keystore
//# by calling the configured [KeyStorage interface's](./key-store/key-storage.md#interface)
//# [GetEncryptedActiveBranchKey](./key-store/key-storage.md##getencryptedactivebranchkey)
//# using the `branch-key-id`.
//#
//# The `KmsArn` of the [EncryptedHierarchicalKey](./key-store/key-storage.md##encryptedhierarchicalkey)
//# MUST be [compatible with](#aws-key-arn-compatibility)
//# the configured `KMS ARN` in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//#
//# Because the storage interface can be a custom implementation the key store needs to verify correctness.
//#
//# VersionKey MUST verify that the returned EncryptedHierarchicalKey MUST have the requested `branch-key-id`.
//# VersionKey MUST verify that the returned EncryptedHierarchicalKey is an ActiveHierarchicalSymmetricVersion.
//# VersionKey MUST verify that the returned EncryptedHierarchicalKey MUST have a logical table name equal to the configured logical table name.
//#
//# The `kms-arn` stored in the table MUST NOT change as a result of this operation,
//# even if the KeyStore is configured with a `KMS MRKey ARN` that does not exactly match the stored ARN.
//# If such were allowed, clients using non-MRK KeyStores might suddenly stop working.
//#
//# The [EncryptedHierarchicalKey](./key-store/key-storage.md##encryptedhierarchicalkey)
//# MUST be authenticated according to [authenticating a keystore item](#authenticating-an-encryptedhierarchicalkey).
//# If the item fails to authenticate this operation MUST fail.
//#
//# The wrapped Branch Keys, DECRYPT_ONLY and ACTIVE, MUST be created according to [Wrapped Branch Key Creation](#wrapped-branch-key-creation).
//#
//# If creation of the keys are successful,
//# then the key store MUST call the configured [KeyStorage interface's](./key-store/key-storage.md#interface)
//# [WriteNewEncryptedBranchKeyVersion](./key-store/key-storage.md##writenewencryptedbranchkeyversion)
//# with these 2 [EncryptedHierarchicalKeys](./key-store/key-storage.md##encryptedhierarchicalkey).
//#
//# If the [WriteNewEncryptedBranchKeyVersion](./key-store/key-storage.md##writenewencryptedbranchkeyversion) is successful,
//# this operation MUST return a successful response containing no additional data.
//# Otherwise, this operation MUST yield an error.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#authenticating-an-encryptedhierarchicalkey
//= type=exception
//# The operation MUST use the configured `KMS SDK Client` to authenticate the value of the keystore item.
//#
//# The operation MUST call [AWS KMS API ReEncrypt](https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html)
//# with a request constructed as follows:
//#
//# - `SourceEncryptionContext` MUST be the [encryption context](#encryption-context) of the EncryptedHierarchicalKey to be authenticated
//# - `SourceKeyId` MUST be [compatible with](#aws-key-arn-compatibility) the configured KMS Key in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//# - `CiphertextBlob` MUST be the `CiphertextBlob` attribute value on the EncryptedHierarchicalKey to be authenticated
//# - `GrantTokens` MUST be the configured [grant tokens](https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#grant_token).
//# - `DestinationKeyId` MUST be [compatible with](#aws-key-arn-compatibility) the configured KMS Key in the [AWS KMS Configuration](#aws-kms-configuration) for this keystore.
//# - `DestinationEncryptionContext` MUST be the [encryption context](#encryption-context) of the EncryptedHierarchicalKey to be authenticated
// Custom EC is only _added_ during construction.
// in all other cases, the EC will be associated with the existing records.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#custom-encryption-context
//= type=exception
//# If custom [encryption context](./structures.md#encryption-context-3)
//# is associated with the branch key these values MUST be added to the AWS KMS encryption context.
//# To avoid name collisions each added attribute from the custom [encryption context](./structures.md#encryption-context-3)
//# MUST be prefixed with `aws-crypto-ec:`.
//# Across all versions of a Branch Key, the custom encryption context MUST be equal.
//= aws-encryption-sdk-specification/framework/branch-key-store.md#keyschema
//= type=exception
//# The following KeySchema MUST be configured on the table:
//#
//# | AttributeName | KeyType   | Type |
//# | ------------- | --------- | ---- |
//# | branch-key-id | Partition | S    |
//# | type          | Sort      | S    |
//= aws-encryption-sdk-specification/framework/branch-key-store.md#getbeaconkey
//= type=exception
//# On invocation, the caller:
//#
//# - MUST supply a `branch-key-id`
//#
//# GetBeaconKey MUST get the requested beacon key from the keystore
//# by calling the configured [KeyStorage interface's](./key-store/key-storage.md#interface)
//# [GetEncryptedBeaconKey](./key-store/key-storage.md#getencryptedbeaconkey)
//# using the supplied `branch-key-id`.
//#
//# Because the storage interface can be a custom implementation the key store needs to verify correctness.
//#
//# GetBeaconKey MUST verify that the returned EncryptedHierarchicalKey MUST have the requested `branch-key-id`.
//# GetBeaconKey MUST verify that the returned EncryptedHierarchicalKey is an ActiveHierarchicalSymmetricBeacon.
//# GetBeaconKey MUST verify that the returned EncryptedHierarchicalKey MUST have a logical table name equal to the configured logical table name.
//#
//# The operation MUST decrypt the beacon key according to the [AWS KMS Branch Key Decryption](#aws-kms-branch-key-decryption) section.
//#
//# If the beacon key fails to decrypt, this operation MUST fail.
//#
//# This GetBeaconKey MUST construct [beacon key materials](./structures.md#beacon-key-materials) from the decrypted branch key material
//# and the `branchKeyId` from the returned `branch-key-id` field.
//#
//# This operation MUST return the constructed [beacon key materials](./structures.md#beacon-key-materials).
//# sourceMappingURL=data:application/json;base64,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